<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

if ( ! class_exists( 'PSU_Url_Suffix' ) ) :

class PSU_Url_Suffix {

    /**
     * PSU_Url_Suffix instance
     */
    protected static $_instance = null;

    /**
     * The product url suffix
     */
    protected $product_suffix = '';

    /**
     * The category url suffix
     */
    protected $category_suffix = '';

    /**
     * Main PSU_Url_Suffix instance
     * Ensures only one instance of PSU_Url_Suffix is loaded or can be loaded.
     *
     * @static
     */
    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Cloning is forbidden.
     *
     * @since 2.4
     */
    public function __clone() {
        _doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'psu' ), '2.4' );
    }

    /**
     * Unserializing instances of this class is forbidden.
     *
     * @since 2.4
     */
    public function __wakeup() {
        _doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'psu' ), '2.4' );
    }

    /**
     * Constructor
     */
    public function __construct() {
        if ( PSU()->is_activated( true ) ) {
            add_filter( 'psu_request_filter_url_before', array( $this, 'delete_suffix' ) );
            add_filter( 'redirect_canonical', array( $this, 'redirect_canonical' ), 10, 2 );

            if ( get_option( 'psu_url_suffix_enabled' ) === 'yes' ) {
                $this->set_product_suffix();

                add_filter( 'psu_post_type_link_after', array( $this, 'add_product_suffix' ) );
                add_filter( 'psu_rewrite_url_suffix', array( $this, 'get_product_suffix' ) );
                add_filter( 'post_comments_feed_link', array( $this, 'post_comments_feed_link' ) );
                add_filter( 'psu_product_canonical_after', array( $this, 'psu_product_canonical_after' ) );
            }

            if ( get_option( 'psu_category_url_suffix_enabled' ) === 'yes' ) {
                $this->set_category_suffix();

                add_filter( 'psu_term_link_after', array( $this, 'add_category_suffix' ) );

                add_filter( 'psu_product_cat_canonical_page_link', array( $this, 'page_link' ), 10, 3 );
                add_filter( 'taxonomy_feed_link', array( $this, 'taxonomy_feed_link' ) );
                add_filter( 'paginate_links', array( $this, 'paginate_links' ) );
                add_filter( 'get_pagenum_link', array( $this, 'get_pagenum_link' ) );

                add_filter( 'wpseo_next_rel_link', array( $this, 'wpseo_next_rel_link' ) );
                add_filter( 'wpseo_prev_rel_link', array( $this, 'wpseo_prev_rel_link' ) );

                add_filter( 'psu_rewrite_category_url_suffix', array( $this, 'get_category_suffix' ) );
            }
        }
    }

    /**
     * Set product suffix
     */
    protected function set_product_suffix() {
        $this->product_suffix = get_option( 'psu_url_suffix' );
    }

    /**
     * Set category suffix
     */
    protected function set_category_suffix() {
        $this->category_suffix = get_option( 'psu_category_url_suffix' );
    }

    /**
     * Return product suffix
     */
    public function get_product_suffix() {
        return $this->product_suffix;
    }

    /**
     * Return category suffix
     */
    public function get_category_suffix() {
        return $this->category_suffix;
    }

    /**
     * Add product suffix to link
     */
    public function add_product_suffix( $link ) {
        $link = rtrim( $link, '/' );
        $link = $link . $this->get_product_suffix();

        return $link;
    }

    /**
     * Add category suffix to link
     */
    public function add_category_suffix( $link ) {
        $link = rtrim( $link, '/' );
        $link = $link . $this->get_category_suffix();

        return $link;
    }

    /**
     * Delete product suffix from link
     */
    public function delete_product_suffix( $link ) {
        if ( ! is_wp_error( $link ) ) {
            $link = preg_replace( '/' . $this->get_product_suffix() . '/', '', $link );
        }

        return $link;
    }

    /**
     * Delete category suffix from link
     */
    public function delete_category_suffix( $link ) {
        if ( ! is_wp_error( $link ) ) {
            $link = preg_replace( '/' . $this->get_category_suffix() . '/', '', $link );
        }

        return $link;
    }

    /**
     * Delete suffix from link
     */
    public function delete_suffix( $link ) {
        if ( ! is_wp_error( $link ) ) {
            $link = preg_replace( '/' . $this->get_product_suffix() . '/', '', $link );
            $link = preg_replace( '/' . $this->get_category_suffix() . '/', '', $link );
        }

        return $link;
    }

    /**
     * Set product feed link
     */
    public function post_comments_feed_link( $link ) {
        return $this->delete_product_suffix( $link );
    }

    /**
     * Set product canonical link
     */
    public function psu_product_canonical_after( $link ) {
        $link = $this->delete_product_suffix( $link );
        $link = $this->add_product_suffix( $link );

        return $link;
    }

    /**
     * Set category feed link
     */
    public function taxonomy_feed_link( $link ) {
        return $this->delete_category_suffix( $link );
    }

    /**
     * Set pagination links
     */
    public function paginate_links( $link ) {
        if ( is_admin() || is_search() || ! is_tax( PSU()->woocommerce_taxonomy ) ) {
            return $link;
        }

        $page = explode( '?', $link );

        $parameters = '';
        if (isset($page[1])) {
            $parameters = '?' . $this->delete_category_suffix( $page[1] );
        }

        $page = explode( '/', trim( $page[0], '/' ) );
        $page = end( $page );
        $page = (int) filter_var($page, FILTER_SANITIZE_NUMBER_INT);

        $term = PSU()->get_current_category();

        if (!is_object($term)) {
            return $link;
        }

        $link = $this->page_link( $link, $page, $term->term_id, $parameters );

        return $link;
    }

    /**
     * Set pagination links
     */
    public function get_pagenum_link( $result ) {
        if ( is_search() || ! is_tax( PSU()->woocommerce_taxonomy ) ) {
            return $result;
        }

        $result = $this->delete_category_suffix( $result );
        $result = $this->add_category_suffix( $result );

        return $result;
    }

    /**
     * Set page link for category archive
     */
    public function page_link( $term_link, $page, $term_id, $parameters = '' ) {
        if ( is_search() || ! is_tax( PSU()->woocommerce_taxonomy ) ) {
            return $term_link;
        }

        global $wp_rewrite;

        $link = get_term_link( (int) $term_id, PSU()->woocommerce_taxonomy );

        if ( is_wp_error( $link ) ) {
            return $term_link;
        }

        $link = $this->delete_category_suffix( $link );

        if ( $page == 1 ) {
            return $link . $this->get_category_suffix() . $parameters;
        }

        $link = trailingslashit( $link ) . $wp_rewrite->pagination_base . '/' . $page . $this->get_category_suffix() . $parameters;

        return $link;
    }

    /**
     * Prevent redirecting url
     *
     * @param string $redirect_url The redirect URL.
     * @param string $requested_url The requested URL.
     *
     * @return string|boolean
     */
    public function redirect_canonical( $redirect_url, $requested_url ) {
        return false;
    }

    public function wpseo_next_rel_link( $output ) {
        if ( is_search() || ! is_tax( PSU()->woocommerce_taxonomy ) ) {
            return $output;
        }

        global $wp_rewrite;

        $term = PSU()->get_current_category();
        $page = get_query_var('paged') + 1;
        if ( $page === 1 ) {
            $page = 2;
        }

        $link = get_term_link( (int) $term->term_id, PSU()->woocommerce_taxonomy );
        $link = $this->delete_category_suffix( $link );
        $link = trailingslashit( $link ) . $wp_rewrite->pagination_base . '/' . $page . $this->get_category_suffix();

        $output = '<link rel="next" href="' . esc_url( $link ) . "\" />\n";

        return $output;
    }

    public function wpseo_prev_rel_link( $output ) {
        if ( is_search() || ! is_tax( PSU()->woocommerce_taxonomy ) ) {
            return $output;
        }

        global $wp_rewrite;

        $term = PSU()->get_current_category();
        $page = get_query_var('paged') - 1;

        $link = get_term_link( (int) $term->term_id, PSU()->woocommerce_taxonomy );
        $link = $this->delete_category_suffix( $link );

        if ( $page === 1 ) {
            $link = $link . $this->get_category_suffix();
        } else {
            $link = trailingslashit( $link ) . $wp_rewrite->pagination_base . '/' . $page . $this->get_category_suffix();
        }

        $output = '<link rel="prev" href="' . esc_url( $link ) . "\" />\n";

        return $output;
    }

}

endif;
