<?php
/*
Custom List Table Example
http://www.mattvanandel.com/
A highly documented plugin that demonstrates how to create custom List Tables using official WordPress APIs.
Version 1.1
Author Matt Van Andel
Author URI http://www.mattvanandel.com
License GPL2
*/
/*  Copyright 2011  Matthew Van Andel  (email : matt@mattvanandel.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2, as
    published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/



/* == NOTICE ===================================================================
 * Please do not alter this file. Instead: make a copy of the entire plugin, 
 * rename it, and work inside the copy. If you modify this plugin directly and 
 * an update is released, your changes will be lost!
 * ========================================================================== */



/*************************** LOAD THE BASE CLASS *******************************
 *******************************************************************************
 * The WP_List_Table class isn't automatically available to plugins, so we need
 * to check if it's available and load it if necessary.
 */
if(!class_exists('WP_List_Table')){
    require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
}




/************************** CREATE A PACKAGE CLASS *****************************
 *******************************************************************************
 * Create a new list table package that extends the core WP_List_Table class.
 * WP_List_Table contains most of the framework for generating the table, but we
 * need to define and override some methods so that our data can be displayed
 * exactly the way we need it to be.
 * 
 * To display this example on a page, you will first need to instantiate the class,
 * then call $yourInstance->prepare_items() to handle any data manipulation, then
 * finally call $yourInstance->display() to render the table to the page.
 * 
 * Our theme for this list table is going to be movies.
 */
class WPRS_All_List_Table extends WP_List_Table {
    
    /** ************************************************************************
     * Normally we would be querying data from a database and manipulating that
     * for use in your list table. For this example, we're going to simplify it
     * slightly and create a pre-built array. Think of this as the data that might
     * be returned by $wpdb->query().
     * 
     * @var array 
     **************************************************************************/
  

    /** ************************************************************************
     * REQUIRED. Set up a constructor that references the parent constructor. We 
     * use the parent reference to set some default configs.
     ***************************************************************************/
    function __construct(){
        global $status, $page, $post;
                
        //Set parent defaults
        parent::__construct( array(
            'singular'  => 'post',     //singular name of the listed records
            'plural'    => 'posts',    //plural name of the listed records
            'ajax'      => false        //does this table support ajax?
        ) );
        
    }
    
    
    /** ************************************************************************
     * Recommended. This method is called when the parent class can't find a method
     * specifically build for a given column. Generally, it's recommended to include
     * one method for each column you want to render, keeping your package class
     * neat and organized. For example, if the class needs to process a column
     * named 'title', it would first see if a method named $this->column_title() 
     * exists - if it does, that method will be used. If it doesn't, this one will
     * be used. Generally, you should try to use custom column methods as much as 
     * possible. 
     * 
     * Since we have defined a column_title() method later on, this method doesn't
     * need to concern itself with any column with a name of 'title'. Instead, it
     * needs to handle everything else.
     * 
     * For more detailed insight into how columns are handled, take a look at 
     * WP_List_Table::single_row_columns()
     * 
     * @param array $item A singular item (one full row's worth of data)
     * @param array $column_name The name/slug of the column to be processed
     * @return string Text or HTML to be placed inside the column <td>
     **************************************************************************/
    function column_default($item, $column_name){
		
		global $wprs_template, $wprs_prefix;

		$rating_retuls = '';
		
		switch($column_name){
            case 'post_title':
				
				$permalink = get_permalink( $item['ID'] ); // get post id
				$summary = get_post_meta($item['ID'], $wprs_prefix.'item_description', true); // get review summary
				$title = '<strong><a href="'.$permalink.'" title="'.$summary.'"target="_blank">'.$item['post_title'].'</a></strong>';
				$title .= '<div class="row-actions">';
				$title .= '<span class="view">';
				$title .= '<a href="'.get_edit_post_link( $item['ID'] ).'">'.__('Edit','wprs').'</a>';
				$title .= ' | ';
				$title .= '<a href="'.$permalink.'" title="'.__('View','wprs').'" target="_blank">'.__('View','wprs').'</a>';
				$title .= ' | ';
				$title .= '<a href="https://developers.google.com/structured-data/testing-tool/?url='.$permalink.'" title="'.__('Structured Data Testing Tool','wprs').'" target="_blank">'.__('Test','wprs').' →</a>';
				$title .= '</span>';
				$title .= '</div>';
				return $title;
				//return $item['post_title'];  // return post title
            
			case 'post_author':
				
				// Check for author name field
				// @since 1.2.8
				$author_name = get_post_meta($item['ID'], $wprs_prefix.'author_name', true);
				$author = empty($author_name) ? get_the_author_meta( 'display_name', $item['post_author'] ) : $author_name;
				
				//$author = get_the_author_meta( 'display_name', $item['post_author'] );
				return $author;
				
			case 'wprs_rating':
				
				//if ( !wprs_is_enabled($item['ID']) ) return '';
					
				$type = get_post_meta($item['ID'], $wprs_prefix.'review_type', true);
					
				switch($type){
					
						case 'rating':
							$rating = get_post_meta($item['ID'], $wprs_prefix.'star_rating', true);
							$rating_retuls = wprs_get_star_rating($rating);
							break;
							
						case 'percentage':
							$percentage = get_post_meta($item['ID'], $wprs_prefix.'slider_percentage', true);
							$rating_retuls = wprs_get_progressbar($percentage, 'editor');
							break;
							
						case 'votes':
							$rating = get_post_meta($item['ID'], $wprs_prefix.'user_rating', true);
							$rating_retuls = wprs_get_user_star_rating($item['ID']);
							break;
							
						case 'aggregate':
							//$rating = get_post_meta($item['ID'], "wprs_post_user_rating", true);
							$rating_retuls = wprs_get_userrating_stars($item['ID']);
							break;

				}
					
				return $rating_retuls;
					
				
			
			case 'post_type':
				
				$da_type = get_post_type( $item['ID'] );
				$da_type = ucfirst($da_type);
				return $da_type;
			
			case 'wprs_schema_types':
				
				$snippets_types 	= get_post_meta($item['ID'], $wprs_prefix.'snippets_types', true);
				//
				// @since 1.4.1
				// Check for sub
				$snippets_organization_type = get_post_meta($item['ID'], $wprs_prefix.'organization_type', true);
				if ( isset($snippets_organization_type) && $snippets_organization_type !='' ) $snippets_types = $snippets_organization_type;
				// Check for sub of sub
				$snippets_organization_type_sub = get_post_meta($item['ID'], $wprs_prefix.'organization_type_sub', true);
				if ( isset($snippets_organization_type_sub) && $snippets_organization_type_sub !='' ) $snippets_types = $snippets_organization_type_sub;
				
				$snippets_types = ucfirst($snippets_types);
				return $snippets_types;
			
			case 'wprs_review_type':
				
				$review_type = get_post_meta($item['ID'], $wprs_prefix.'review_type', true);
				$review_type = ucfirst($review_type);
				return $review_type;
			
			case 'post_date':
				
				$postid = $item['ID'];								// get post id
				$date = get_the_time( 'F j, Y g:i a', $postid ); 	// set time formate (this CAN NOT be changed with general settings)
				
				//return $item['post_date'];
				return $date;	
                
				//return $item[$column_name];
				
            default:
                return print_r($item,true); //Show the whole array for troubleshooting purposes
        }
    }
    
        
    /** ************************************************************************
     * Recommended. This is a custom column method and is responsible for what
     * is rendered in any column with a name/slug of 'title'. Every time the class
     * needs to render a column, it first looks for a method named 
     * column_{$column_title} - if it exists, that method is run. If it doesn't
     * exist, column_default() is called instead.
     * 
     * This example also illustrates how to implement rollover actions. Actions
     * should be an associative array formatted as 'slug'=>'link html' - and you
     * will need to generate the URLs yourself. You could even ensure the links
     * 
     * 
     * @see WP_List_Table::::single_row_columns()
     * @param array $item A singular item (one full row's worth of data)
     * @return string Text to be placed inside the column <td> (movie title only)
     **************************************************************************/
    function column_title($item){
        
        //Build row actions
        $actions = array(
            'edit'      => sprintf('<a href="?post=%s&action=%s&post=%s">Edit</a>',$_REQUEST['page'],'edit',$item['ID']),
            'delete'    => sprintf('<a href="?post=%s&action=%s&post=%s">Delete</a>',$_REQUEST['page'],'delete',$item['ID']),
        );
        
        //Return the title contents
        return sprintf('%1$s <span style="color:silver">(id:%2$s)</span>%3$s',
            /*$1%s*/ $item['title'],
            /*$2%s*/ $item['ID'],
            /*$3%s*/ $this->row_actions($actions)
        );
    }
    
    /** ************************************************************************
     * REQUIRED if displaying checkboxes or using bulk actions! The 'cb' column
     * is given special treatment when columns are processed. It ALWAYS needs to
     * have it's own method.
     * 
     * @see WP_List_Table::::single_row_columns()
     * @param array $item A singular item (one full row's worth of data)
     * @return string Text to be placed inside the column <td> (movie title only)
     **************************************************************************/
    function column_cb($item){
        
		// disabled
		// @since 1.4.7
		return;
		
		return sprintf(
            '<input type="checkbox" name="%1$s[]" value="%2$s" />',
            /*$1%s*/ $this->_args['singular'],  //Let's simply repurpose the table's singular label ("movie")
            /*$2%s*/ $item['ID']                //The value of the checkbox should be the record's id
        );
    }
    
    
    /** ************************************************************************
     * REQUIRED! This method dictates the table's columns and titles. This should
     * return an array where the key is the column slug (and class) and the value 
     * is the column's title text. If you need a checkbox for bulk actions, refer
     * to the $columns array below.
     * 
     * The 'cb' column is treated differently than the rest. If including a checkbox
     * column in your table you must create a column_cb() method. If you don't need
     * bulk actions or checkboxes, simply leave the 'cb' entry out of your array.
     * 
     * @see WP_List_Table::::single_row_columns()
     * @return array An associative array containing column information: 'slugs'=>'Visible Titles'
     **************************************************************************/
    function get_columns(){
        $columns = array(
            //'cb'        			=> '<input type="checkbox" />', //Render a checkbox instead of text
            'post_title'			=> 'Title',
            'post_author'			=> 'Author',
			'post_type'				=> 'Post Type',
			'wprs_schema_types'		=> 'Schema Type',
			'wprs_review_type'		=> 'Review Type',
			'post_date'				=> 'Date',
			'wprs_rating'			=> 'Rating',
        );
        return $columns;
    }
    
    /** ************************************************************************
     * Optional. If you want one or more columns to be sortable (ASC/DESC toggle), 
     * you will need to register it here. This should return an array where the 
     * key is the column that needs to be sortable, and the value is db column to 
     * sort by. Often, the key and value will be the same, but this is not always
     * the case (as the value is a column name from the database, not the list table).
     * 
     * This method merely defines which columns should be sortable and makes them
     * clickable - it does not handle the actual sorting. You still need to detect
     * the ORDERBY and ORDER querystring variables within prepare_items() and sort
     * your data accordingly (usually by modifying your query).
     * 
     * @return array An associative array containing all the columns that should be sortable: 'slugs'=>array('data_values',bool)
     **************************************************************************/
    function get_sortable_columns() {
        $sortable_columns = array(
            'post_title'     		=> array('post_title',false),     
            'post_author'  			=> array('post_author',false),
			'post_type'  			=> array('post_type',false),
			'post_date'  			=> array('post_date',true)					//true means its already sorted
			
			// sorting based on rating require some work, so it's not included here yet!
			//'ta_post_review_rating'	=> array('ta_post_review_rating',false)
        );
        return $sortable_columns;
    }
    
    
    /** ************************************************************************
     * Optional. If you need to include bulk actions in your list table, this is
     * the place to define them. Bulk actions are an associative array in the format
     * 'slug'=>'Visible Title'
     * 
     * If this method returns an empty value, no bulk action will be rendered. If
     * you specify any bulk actions, the bulk actions box will be rendered with
     * the table automatically on display().
     * 
     * Also note that list tables are not automatically wrapped in <form> elements,
     * so you will need to create those manually in order for bulk actions to function.
     * 
     * @return array An associative array containing all the bulk actions: 'slugs'=>'Visible Titles'
     **************************************************************************/
    function get_bulk_actions() {
        $actions = array(
            'delete'    => 'Delete'
        );
		
		// disable bulk actions by passing an empty array
		// @since 1.4.7
		$actions = array();
        
		return $actions;
    }
    
    
    /** ************************************************************************
     * Optional. You can handle your bulk actions anywhere or anyhow you prefer.
     * For this example package, we will handle it in the class to keep things
     * clean and organized.
     * 
     * @see $this->prepare_items()
     **************************************************************************/
    function process_bulk_action() {
        
        //Detect when a bulk action is being triggered...
        if( 'delete'===$this->current_action() ) {
            wp_die('Items deleted (or they would be if we had items to delete)!');
        }
        
    }
    
    
    /** ************************************************************************
     * REQUIRED! This is where you prepare your data for display. This method will
     * usually be used to query the database, sort and filter the data, and generally
     * get it ready to be displayed. At a minimum, we should set $this->items and
     * $this->set_pagination_args(), although the following properties and methods
     * are frequently interacted with here...
     * 
     * @uses $this->_column_headers
     * @uses $this->items
     * @uses $this->get_columns()
     * @uses $this->get_sortable_columns()
     * @uses $this->get_pagenum()
     * @uses $this->set_pagination_args()
     **************************************************************************/
    function prepare_items() {
        
        /**
         * First, lets decide how many records per page to show
         */
        $per_page = 20;
        
        
        /**
         * REQUIRED. Now we need to define our column headers. This includes a complete
         * array of columns to be displayed (slugs & titles), a list of columns
         * to keep hidden, and a list of columns that are sortable. Each of these
         * can be defined in another method (as we've done here) before being
         * used to build the value for our _column_headers property.
         */
        $columns = $this->get_columns();
        $hidden = array();
        $sortable = $this->get_sortable_columns();
        
        
        /**
         * REQUIRED. Finally, we build an array to be used by the class for column 
         * headers. The $this->_column_headers property takes an array which contains
         * 3 other arrays. One for all columns, one for hidden columns, and one
         * for sortable columns.
         */
        $this->_column_headers = array($columns, $hidden, $sortable);
        
        
        /**
         * Optional. You can handle your bulk actions however you see fit. In this
         * case, we'll handle them within our package just to keep things clean.
         */
        $this->process_bulk_action();
        
		
        /**
         * Instead of querying a database, we're going to fetch the example data
         * property we created for use in this plugin. This makes this example 
         * package slightly different than one you might build on your own. In 
         * this example, we'll be using array manipulation to sort and paginate 
         * our data. In a real-world implementation, you will probably want to 
         * use sort and pagination data to build a custom query instead, as you'll
         * be able to use your precisely-queried data immediately.
         */

			// get review posts
			global $wpdb;
			
			$orderby = (!empty($_REQUEST['orderby'])) ? $_REQUEST['orderby'] : 'post_date'; //If no sort, default to title
			//$sql = "SELECT * FROM wp_nc_location ORDER BY " . $orderby;
		
			$sql_r = "SELECT wposts.*
   					FROM $wpdb->posts wposts, $wpdb->postmeta wpostmeta
   					WHERE wposts.ID = wpostmeta.post_id
   					AND wpostmeta.meta_key = '_wprs_post_snippets_types'
   					AND wposts.post_status = 'publish'
   					ORDER BY wposts.post_date ASC";
			
			// the sql returns an opject array
			// so we transfort it to a normal array
			// and store in $myreviews
			//$myreviews = $wpdb->get_results($sql, ARRAY_A);
			
			$myreviews = $wpdb->get_results($sql_r, ARRAY_A);
			
			// unset the unwanted values
			/*foreach($myreviews as $key => $value) {

				unset($myreviews[$key]['post_date_gmt']);
				unset($myreviews[$key]['post_content']);
				unset($myreviews[$key]['post_excerpt']);
				unset($myreviews[$key]['post_status']);
				unset($myreviews[$key]['comment_status']);
				unset($myreviews[$key]['ping_status']);
				unset($myreviews[$key]['post_password']);
				unset($myreviews[$key]['post_name']);
				unset($myreviews[$key]['to_ping']);
				unset($myreviews[$key]['pinged']);
				unset($myreviews[$key]['post_modified']);
				unset($myreviews[$key]['post_modified_gmt']);
				unset($myreviews[$key]['post_content_filtered']);
				unset($myreviews[$key]['post_parent']);
				unset($myreviews[$key]['guid']);
				unset($myreviews[$key]['menu_order']);
				unset($myreviews[$key]['post_type']);
				unset($myreviews[$key]['post_mime_type']);
				unset($myreviews[$key]['comment_count']);
    		
			}*/
			
			// I use this to test the array values
			//print '<pre>' . print_r( $myreviews, true ) . '</pre>';
			
			// set data to $myreviews array
			$data = $myreviews;
		  
			
        
        /**
         * This checks for sorting input and sorts the data in our array accordingly.
         * 
         * In a real-world situation involving a database, you would probably want 
         * to handle sorting by passing the 'orderby' and 'order' values directly 
         * to a custom query. The returned data will be pre-sorted, and this array
         * sorting technique would be unnecessary.
         */
        function usort_reorder($a,$b){
            $orderby = (!empty($_REQUEST['orderby'])) ? $_REQUEST['orderby'] : 'post_date'; //If no sort, default to date
            $order = (!empty($_REQUEST['order'])) ? $_REQUEST['order'] : 'desc'; //If no order, default to asc
            $result = strcmp($a[$orderby], $b[$orderby]); //Determine sort order
            return ($order==='asc') ? $result : -$result; //Send final sort direction to usort
        }
		

        usort($data, 'usort_reorder');
        
        
        /***********************************************************************
         * ---------------------------------------------------------------------
         * vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
         * 
         * In a real-world situation, this is where you would place your query.
         * 
         * ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
         * ---------------------------------------------------------------------
         **********************************************************************/
        
                
        /**
         * REQUIRED for pagination. Let's figure out what page the user is currently 
         * looking at. We'll need this later, so you should always include it in 
         * your own package classes.
         */
        $current_page = $this->get_pagenum();
        
        /**
         * REQUIRED for pagination. Let's check how many items are in our data array. 
         * In real-world use, this would be the total number of items in your database, 
         * without filtering. We'll need this later, so you should always include it 
         * in your own package classes.
         */
        $total_items = count($data);
        
        
        /**
         * The WP_List_Table class does not handle pagination for us, so we need
         * to ensure that the data is trimmed to only the current page. We can use
         * array_slice() to 
         */
        $data = array_slice($data,(($current_page-1)*$per_page),$per_page);
        
        
        
        /**
         * REQUIRED. Now we can add our *sorted* data to the items property, where 
         * it can be used by the rest of the class.
         */
        $this->items = $data;
        
        
        /**
         * REQUIRED. We also have to register our pagination options & calculations.
         */
        $this->set_pagination_args( array(
            'total_items' => $total_items,                  //WE have to calculate the total number of items
            'per_page'    => $per_page,                     //WE have to determine how many items to show on a page
            'total_pages' => ceil($total_items/$per_page)   //WE have to calculate the total number of pages
        ) );
    }
    
}


/** ************************ REGISTER THE TEST PAGE ****************************
 *******************************************************************************
 * Now we just need to define an admin page. For this example, we'll add a top-level
 * menu item to the bottom of the admin menus.
 */
function wprs_reviews_list_add_menu_items(){
    //add_menu_page('Reviews', 'Reviews', 'activate_plugins', 'tt_list_test', 'wpsr_render_list_page');
}
//add_action('admin_menu', 'wprs_reviews_list_add_menu_items');


/***************************** RENDER TEST PAGE ********************************
 *******************************************************************************
 * This function renders the admin page and the example list table. Although it's
 * possible to call prepare_items() and display() from the constructor, there
 * are often times where you may need to include logic here between those steps,
 * so we've instead called those methods explicitly. It keeps things flexible, and
 * it's the way the list tables are used in the WordPress core.
 */
function wprs_render_list_page(){
    
    //Create an instance of our package class...
    $testListTable = new WPRS_All_List_Table();
    //Fetch, prepare, sort, and filter our data...
    $testListTable->prepare_items();
    
    ?>
    <div class="wrap">
        
        <div class="icon32 wprs_icon32" id=""></div>
        <h2>List All</h2>
        
        <!-- Forms are NOT created automatically, so you need to wrap the table in one to use features like bulk actions -->
        <form id="movies-filter" method="get">
            <!-- For plugins, we also need to ensure that the form posts back to our current page -->
            <input type="hidden" name="page" value="<?php echo $_REQUEST['page'] ?>" />
            <!-- Now we can render the completed list table -->
            <?php $testListTable->display() ?>
        </form>
        
    </div>
    <?php
}