import { __ } from '@wordpress/i18n'
import { addAction, addFilter, doAction, applyFilters } from '@wordpress/hooks'
import { has, get, isEmpty, forEach } from 'lodash';
import WithForm from '../../vendor/barn2/setup-wizard/resources/js/steps/with-form';
import checkConditions from 'json-conditions'
import axios from "axios";
import qs from 'qs';

import { SetupWizardSettings, getParsedAjaxError } from '../../vendor/barn2/setup-wizard/resources/js/utilities'

/**
 * Make sure at least one option is selected.
 */
class AccessTypes extends WithForm {
	constructor( props ) {
		super( props );
	}

	/**
	 * Automatically toggle steps on mount if required.
	 */
	componentDidMount() {

		const step = this.props.step
		const fields = step.fields

		const anyone = get( fields.anyone, 'value', false );
		const logged = get( fields.logged, 'value', false );

		if ( anyone === true && logged === true ) {
			this.props.showStep( 'password-protection' )
			this.props.showStep( 'user-protection' )
		} else if ( anyone === true ) {
			this.props.showStep( 'password-protection' )
			this.props.hideStep( 'user-protection' )
		} else if ( logged === true ) {
			this.props.hideStep( 'password-protection' )
			this.props.showStep( 'user-protection' )
		}

	}

	/**
	 * Submit the form via ajax
	 */
	onSubmit( values ) {
		
        // Reset the steps so we make sure the hidden ones stay hidden.
		this.props.resetSteps()

		this.setErrorMessage( false ); // reset the error message.

        const anyone = get( values, 'anyone', false );
		const logged = get( values, 'logged', false );

        // Make sure at least one option is clicked.
		if ( anyone === false && logged === false ) {
			this.setErrorMessage( 'Please select at least one option.' );
			return;
		}

		if ( anyone === true && logged === true ) {
			this.props.showStep( 'password-protection' )
			this.props.showStep( 'user-protection' )
		} else if ( anyone === true ) {
			this.props.showStep( 'password-protection' )
			this.props.hideStep( 'user-protection' )
		} else if ( logged === true ) {
			this.props.hideStep( 'password-protection' )
			this.props.showStep( 'user-protection' )
		}

        super.onSubmit(values)

	}
}

/**
 * Custom class for the user protection page.
 * 
 * This page is a temporary placeholder until a proper way
 * to detect the selected value of a dropdown field is found
 * for the main setup-wizard library in order to avoid compatibility
 * issues with products that already implement the library.
 * 
 * The class will be left here until we study integration
 * in other products.
 */
class UserProtection extends WithForm {
	constructor( props ) {
		super( props );
	}

	/**
	 * Determine if the field should be visible.
	 *
	 * @param {object} field contains the details about the field being processed.
	 * @param {string} id id of the field being processed.
	 * @param {object} values list of all values submitted throughout the whole wizard.
	 * @returns bool
	 */
	 isFieldVisible( field, id, values ) {

		if ( has( field, 'conditions' ) ) {
			const rules = []
			const conditions = field.conditions

			forEach( conditions, ( settings, condition_field_id ) => {
				rules.push(
					{
						property: condition_field_id,
						op: settings.op,
						value: settings.value
					}
				)
			} )

			let formattedValues = { ...values };

			forEach( values, ( the_value, the_id ) => {
				if ( has( the_value[0], 'key' ) ) {
					formattedValues[ the_id ] = the_value[0].key
				}
			} )

			if ( ! isEmpty( rules ) ) {
				const shouldLog = has( field, 'debug' ) ? console.log : false

				return checkConditions({
					rules: rules,
					satisfy: 'ALL',
					log: shouldLog,
				}, formattedValues )
			}
		}

		return true

	}
}

/**
 * Override the components of the pages.
 */
addFilter( 'barn2_setup_wizard_steps', 'ppc-wizard', ( steps ) => {
	steps[ 2 ].container = AccessTypes
	steps[ 4 ].container = UserProtection
	return steps
} );

/**
 * Handle the wizard restart request.
 */
addAction( 'barn2_wizard_on_restart', 'ppc-wizard', ( wizard ) => {

	wizard.setState( { wizard_loading: true, wizard_complete: true } );

	axios.post( SetupWizardSettings.ajax, qs.stringify(
		{
			action: `barn2_wizard_${ SetupWizardSettings.plugin_slug }_on_restart`,
			nonce: SetupWizardSettings.nonce,
		}
	 ) )
	.then(function (response) {

		if ( has( response, 'data.data.toggle' ) ) {
			const toToggle = response.data.data.toggle

			const anyone = toToggle.includes( 'anyone' )
			const logged = toToggle.includes( 'logged' )

			if ( anyone === false && logged === false ) {
				wizard.setState( { wizard_loading: false } );
				return;
			}

			if ( anyone === true && logged === true ) {
				wizard.showStep( 'password-protection' )
				wizard.showStep( 'user-protection' )
			} else if ( anyone === true ) {
				wizard.showStep( 'password-protection' )
				wizard.hideStep( 'user-protection' )
			} else if ( logged === true ) {
				wizard.hideStep( 'password-protection' )
				wizard.showStep( 'user-protection' )
			}

			wizard.setState( { wizard_loading: false } );

			wizard.setStepsCompleted( true )

		}
	})
	.catch(function (error) {
		console.log(error)
		if ( error.response ) {
			if ( ! isEmpty( getParsedAjaxError( error.response ) ) ) {
				wizard.setErrorMessage( getParsedAjaxError( error.response ) );
			} else {
				wizard.setErrorMessage( error.response.statusText );
			}
		} else if (error.request) {
			wizard.setErrorMessage( __( 'The request was made but no response was received.' ) );
		} else {
			wizard.setErrorMessage( __( 'Something went wrong while making the request.' ) );
		}
	});

} )