<?php

namespace Barn2\Plugin\Password_Protected_Categories;

use Barn2\Plugin\Password_Protected_Categories\Dependencies\Lib\Registerable;

/**
 * Plugin ajax handler.
 *
 * @package   Barn2\password-protected-categories
 * @author    Barn2 Plugins <support@barn2.co.uk>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
class Ajax implements Registerable {

	/**
	 * Hook into WP.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'admin_enqueue_scripts', [ $this, 'scripts' ] );
		add_action( 'wp_ajax_ppc_get_users', [ $this, 'get_users' ] );
		add_action( 'wp_ajax_ppc_preload_users', [ $this, 'get_preloaded_users' ] );
	}

	/**
	 * Inline script required for ajax.
	 *
	 * @return void
	 */
	public function scripts() {

		wp_add_inline_script(
			'ppc-admin',
			'const PPC_Admin = ' . wp_json_encode(
				[
					'ajaxUrl'     => admin_url( 'admin-ajax.php' ),
					'users_nonce' => wp_create_nonce( 'ppc_get_users_nonce' ),
				]
			),
			'before'
		);

	}

	/**
	 * Load users via ajax.
	 * 
	 * We need 2 queries because WP_User_Query can't search 2 things at the same time.
	 * 1st query looks for ID, username, display name and email.
	 * 2nd query looks for first name and last name.
	 * 
	 * @return void
	 */
	public function get_users() {

		check_ajax_referer( 'ppc_get_users_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			$this->send_error( esc_html__( 'You are not authorized', 'password-protected-categories' ) );
		}

		$search_term = isset( $_POST['search_term'] ) ? sanitize_text_field( $_POST['search_term'] ) : '';

		if ( empty( $search_term ) ) {
			wp_send_json_error( __( 'The search term was empty.', 'password-protected-categories' ) );
		}

		$args = [
			'blog_id'        => get_current_blog_id(),
			'orderby'        => 'display_name',
			'order'          => 'ASC',
			'fields'         => [ 'ID', 'user_login', 'display_name', 'user_email' ],
			'search'         => '*' . esc_attr( $search_term ) . '*',
			'search_columns' => [
				'user_login',
				'user_nicename',
				'user_email',
			]
		];

		$query = ( new \WP_User_Query( $args ) )->get_results();

		$meta_args = [
			'blog_id'    => get_current_blog_id(),
			'orderby'    => 'display_name',
			'order'      => 'ASC',
			'fields'     => [ 'ID', 'user_login', 'display_name' ],
			'meta_query' => [
				'relation' => 'OR',
				[
					'key'     => 'first_name',
					'value'   => $search_term,
					'compare' => 'LIKE'
				],
				[
					'key'     => 'last_name',
					'value'   => $search_term,
					'compare' => 'LIKE'
				],
			]
		];

		$meta_query = ( new \WP_User_Query( $meta_args ) )->get_results();

		$results = array_merge( $query, $meta_query );
		$results = array_unique( $results, SORT_REGULAR );
		
		$list_of_ids = [];
		$users = [];

		if ( ! empty( $results ) ) {
			foreach ( $results as $user ) {
				if ( ! in_array( $user->ID, $list_of_ids, true ) ) {
					$list_of_ids[] = $user->ID;
					$users[] = [
						'id'   => $user->ID,
						'text' => $user->user_login
					];
				}
			}
		}

		wp_send_json( $users );

	}

	/**
	 * Get details about the pre-selected users.
	 * This is used on the edit screen of a taxonomy term.
	 *
	 * @return void
	 */
	public function get_preloaded_users() {
		
		check_ajax_referer( 'ppc_get_users_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			$this->send_error( esc_html__( 'You are not authorized', 'password-protected-categories' ) );
		}

		$users = isset( $_POST['users'] ) && ! empty( $_POST['users'] ) && is_array( $_POST['users'] ) ? array_map( 'absint', $_POST['users'] ) : [];

		if ( empty( $users ) ) {
			wp_send_json_error( __( 'No user was preselected', 'password-protected-categories' ) );
		}

		$args = [
			'include' => $users,
			'fields'     => [ 'ID', 'user_login', 'display_name' ],
		];

		$query = ( new \WP_User_Query( $args ) )->get_results();

		wp_send_json( $query );

	}

	/**
	 * Send a json error back to the browser.
	 *
	 * @param string $message
	 * @return void
	 */
	public function send_error( string $message ) {
		wp_send_json_error( [ 'error_message' => $message ], 403 );
	}

}
