<?php
namespace Barn2\Plugin\Password_Protected_Categories;

use Barn2\Plugin\Password_Protected_Categories\Dependencies\Lib\Registerable;

use function Barn2\Plugin\Password_Protected_Categories\ppc;

/**
 * Theme and plugin integrations for Password Protected Categories.
 *
 * @package   Barn2\password-protected-categories
 * @author    Barn2 Plugins <support@barn2.co.uk>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
class Integrations implements Registerable {

	public function register() {
		if ( function_exists( 'tribe' ) ) {
			add_action( 'ppc_protected_category_template', [ $this, 'the_events_calendar' ] );
		}

		if ( function_exists( '\Barn2\Plugin\Document_Library_Pro\document_library_pro' ) ) {
			add_filter( 'document_library_pro_search_filter_get_terms_args', [ $this, 'dlp_filter_get_terms_args' ], 10, 3 );
			add_filter( 'document_library_pro_preview_button_should_display', [ $this, 'dlp_determine_button_display' ], 10, 5 );
			add_filter( 'document_library_pro_download_button_should_display', [ $this, 'dlp_determine_button_display' ], 10, 5 );
			add_filter( 'document_library_pro_search_filter_terms', [ $this, 'dlp_filter_terms' ], 10, 3 );
		}

		// If the ultimate member plugin is active, add a filter.
		if ( defined( 'UM_PATH' ) && function_exists( 'um_is_core_page' ) ) {
			add_filter( 'ppc_should_bail_get_terms_args', [ $this, 'um_filter_get_terms_args' ], 10, 4 );
		}

	}

	public function the_events_calendar() {

		// Prevent The Events Calendar running its template customisations.
		\remove_filter( 'template_include', [ 'Tribe__Events__Templates', 'templateChooser' ] );
		\remove_action( 'template_redirect', 'tribe_initialize_view' );
		\remove_action( 'wp_head', [ 'Tribe__Events__Templates', 'maybeSpoofQuery' ], 100 );

		if ( class_exists( '\Tribe\Events\Views\V2\Hooks' ) ) {
			remove_filter( 'template_include', [ tribe( \Tribe\Events\Views\V2\Hooks::class ), 'filter_template_include' ], 50 );
		}
	}

	public function dlp_filter_get_terms_args( $term_args, $taxonomy, $config_args ) {
		$term_args['ppc_check'] = true;

		return $term_args;
	}

	/**
	 * Hide the DLP Grid template "preview" button when the post is locked.
	 *
	 * @param boolean $display
	 * @param object $document
	 * @param string $link_text
	 * @param string $link_style
	 * @param string $view
	 * @return boolean
	 */
	public function dlp_determine_button_display( $display, $document, $link_text, $link_style, $view ) {
		if ( ! PPC_Util::is_unlocked_post( $document->post_object ) ) {
			return false;
		}

		return $display;
	}

	/**
	 * Filter the terms returned by the DLP Search Filter widget.
	 *
	 * @param array $terms
	 * @param string $taxonomy
	 * @param array $args
	 * @return array
	 */
	public function dlp_filter_terms( $terms, $taxonomy, $args ) {
		$protector = ppc()->get_service( 'term_protector' );

		if ( ! $protector instanceof Term_Protector ) {
			return $terms;
		}
		
		$unlocked_categories = $protector->unlocked_categories();

		foreach ( $terms as $key => $term ) {
			$visibility = new Term_Visibility( $term );

			if ( $visibility->is_protected( true ) && ! in_array( $term->term_id, $unlocked_categories, true ) ) {
				unset( $terms[ $key ] );
			}
		}

		return $terms;
	}

	/**
	 * Prevent the Ultimate Member plugin from filtering out protected terms.
	 *
	 * @param boolean $bail Whether to bail early.
	 * @param array $args The get_terms() arguments.
	 * @param array $taxonomies The taxonomies to get terms from.
	 * @param object $protector The term protector.
	 * @return boolean Whether to bail early.
	 */
	public function um_filter_get_terms_args( $bail, $args, $taxonomies, $protector ) {
		if ( isset( $args['um_ignore_exclude'] ) ) {
			return true;
		}

		return $bail;
	}

}
