<?php
namespace Barn2\Plugin\Password_Protected_Categories;

use Barn2\Plugin\Password_Protected_Categories\Admin\Plugin_Setup;
use Barn2\Plugin\Password_Protected_Categories\Dependencies\Lib\Registerable;
use Barn2\Plugin\Password_Protected_Categories\Dependencies\Lib\Translatable;
use Barn2\Plugin\Password_Protected_Categories\Dependencies\Lib\Service_Provider;
use Barn2\Plugin\Password_Protected_Categories\Dependencies\Lib\Plugin\Premium_Plugin;
use Barn2\Plugin\Password_Protected_Categories\Dependencies\Lib\Plugin\Licensed_Plugin;
use Barn2\Plugin\Password_Protected_Categories\Dependencies\Lib\Util;
use Barn2\Plugin\Password_Protected_Categories\Dependencies\Lib\Admin\Notices;

/**
 * The main plugin class. Responsible for setting up to core plugin services.
 *
 * @package   Barn2\password-protected-categories
 * @author    Barn2 Plugins <support@barn2.co.uk>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
class Plugin extends Premium_Plugin implements Licensed_Plugin, Registerable, Translatable, Service_Provider {

	const NAME    = 'Password Protected Categories';
	const ITEM_ID = 16321;

	/**
	 * Constructs and initalizes the main plugin class.
	 *
	 * @param string $file The root plugin __FILE__
	 * @param string $version The current plugin version
	 */
	public function __construct( $file = null, $version = '1.0' ) {
		parent::__construct(
			[
				'name'               => self::NAME,
				'item_id'            => self::ITEM_ID,
				'version'            => $version,
				'file'               => $file,
				'settings_path'      => 'options-general.php?page=' . Admin\Settings_Page::MENU_SLUG,
				'documentation_path' => 'kb-categories/ppc-kb/',
				'legacy_db_prefix'   => 'ppc',
			]
		);

		// Add core service to handle plugin setup.
		$this->add_service( 'plugin_setup', new Plugin_Setup( $this->get_file(), $this ), true );
	}

	/**
	 * Registers the plugin with WordPress.
	 */
	public function register() {
		parent::register();

		add_action( 'init', [ $this, 'load_textdomain' ], 5 );
		add_action( 'init', [ $this, 'create_services' ], 10 );
		add_action( 'init', [ $this, 'register_services' ], 20 );

		// Register styles and scripts
		add_action( 'wp_enqueue_scripts', [ $this, 'front_end_scripts' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'admin_scripts' ] );
	}

	public function create_services() {
		if ( ! $this->check_min_requirements() ) {
			return;
		}

		$services = [];

		if ( \is_admin() ) {
			$services['settings_page'] = new Admin\Settings_Page( $this );
		}

		$services['setup_wizard'] = new Admin\Wizard\Service( $this );

		if ( $this->get_license()->is_valid() ) {
			// Protection for sitemaps.
			$services['sitemaps'] = new XML_Sitemaps();

			// Ajax handler for the admin panel.
			$services['ajax'] = new Ajax();

			if ( \is_admin() ) {
				// Adds a visibility field to all categories/taxonomies.
				$services['term_visibility'] = new Admin\Term_Visibility_Field();
			}
			if ( Util::is_front_end() ) {
				// Front-end protection, including login form display and template handling.
				$services['term_protector']   = new Term_Protector();
				$services['template_handler'] = new Template_Handler();
				$services['login_shortcode']  = new Login_Shortcode();
				$services['password_form']    = new Password_Form();
				$services['integrations']     = new Integrations();
			}
		}

		foreach ( $services as $name => $service ) {
			$this->add_service( $name, $service );
		}
	}

	public function load_textdomain() {
		\load_plugin_textdomain( 'password-protected-categories', false, $this->get_slug() . '/languages' );
	}

	public function front_end_scripts() {
		\wp_enqueue_style( 'ppc-style', \plugins_url( 'assets/css/ppc.css', $this->get_file() ), false, $this->get_version() );
	}

	public function admin_scripts( $hook ) {

		\wp_register_script( 'ppc-admin', \plugins_url( 'assets/js/admin/ppc-admin.js', $this->get_file() ), [ 'jquery' ], $this->get_version(), true );

		if ( \in_array( $hook, [ 'edit-tags.php', 'term.php' ] ) ) {
			\wp_enqueue_style( 'ppc-select2', \plugins_url( 'assets/select2/css/selectWoo.min.css', $this->get_file() ), false, $this->get_version() );
			\wp_enqueue_style( 'ppc-admin', \plugins_url( 'assets/css/admin/ppc-admin.css', $this->get_file() ), false, $this->get_version() );

			\wp_enqueue_script(
				'ppc-tips',
				$this->get_dir_url() . '/lib/assets/js/jquery-tiptip/jquery.tipTip.min.js',
				false,
				$this->get_version()
			);
			\wp_enqueue_script( 'ppc-select2', \plugins_url( 'assets/select2/js/selectWoo.full.min.js', $this->get_file() ), [ 'jquery' ], $this->get_version(), true );
			\wp_enqueue_script( 'ppc-admin' );

			\wp_localize_script(
				'ppc-admin',
				'ppc_params',
				[
					'confirm_delete' => __( 'Are you sure you want to remove this password?', 'password-protected-categories' ),
				]
			);

			// Specific inline styling required when Avada theme & yoast seo are enabled.
			if ( defined( 'WPSEO_FILE' ) && defined( 'AVADA_VERSION' ) ) {
				$custom_css = '.select2-container--default .select2-selection--multiple .select2-selection__choice__remove {position:static !important; padding:0 4px 0 0 !important;}';

				wp_add_inline_style( 'ppc-admin', $custom_css );
			}
		}

		if ( $hook === 'settings_page_protected_categories' ) {
			\wp_enqueue_script( 'ppc-admin' );
		}
	}

	private function check_min_requirements() {
		global $wp_version;

		if ( \is_admin() && \version_compare( $wp_version, '4.4', '<' ) ) {
			$can_update_core = current_user_can( 'update_core' );

			$notices = new Notices();
			$notices->add(
				'ppc_invalid_wp_version',
				'',
				\sprintf(
					__( 'The %1$s plugin requires WordPress 4.4 or greater. Please %2$supdate%3$s your WordPress installation.', 'password-protected-categories' ),
					'<strong>' . self::NAME . '</strong>',
					( $can_update_core ? \sprintf( '<a href="%s">', \esc_url( \self_admin_url( 'update-core.php' ) ) ) : '' ),
					( $can_update_core ? '</a>' : '' )
				),
				[
					'type'       => 'error',
					'capability' => 'install_plugins',
					'screens'    => [ 'plugins', 'settings_page_' . Admin\Settings_Page::MENU_SLUG ],
				]
			);
			$notices->boot();
			return false;
		}
		return true;
	}
}
