<?php
/**
 * WC_Boxpack_Box file.
 *
 * @package woocommerce-shipping-canada-post
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WC_Boxpack_Box class.
 */
class WC_Boxpack_Box {

	/**
	 * ID of the box - given to packages.
	 *
	 * @var string
	 */
	private $id = '';

	/**
	 * Weight of the box itself.
	 *
	 * @var float
	 */
	private $weight;

	/**
	 * Max allowed weight of box + contents.
	 *
	 * @var float
	 */
	private $max_weight = 0;

	/**
	 * Outer dimension of box sent to shipper.
	 *
	 * @var float
	 */
	private $outer_height;

	/**
	 * Outer dimension of box sent to shipper.
	 *
	 * @var float
	 */
	private $outer_width;

	/**
	 * Outer dimension of box sent to shipper.
	 *
	 * @var float
	 */
	private $outer_length;

	/**
	 * Inner dimension of box used when packing.
	 *
	 * @var float
	 */
	private $height;

	/**
	 * Inner dimension of box used when packing.
	 *
	 * @var float
	 */
	private $width;

	/**
	 * Inner dimension of box used when packing.
	 *
	 * @var float
	 */
	private $length;

	/**
	 * Dimension is stored here if adjusted during packing.
	 *
	 * @var float
	 */
	private $packed_height;

	/**
	 * Box's height that maybe can be packed.
	 *
	 * @var float
	 */
	private $maybe_packed_height = null;

	/**
	 * Dimension is stored here if adjusted during packing.
	 *
	 * @var float
	 */
	private $packed_width;

	/**
	 * Box's width that maybe can be packed.
	 *
	 * @var float
	 */
	private $maybe_packed_width = null;

	/**
	 * Dimension is stored here if adjusted during packing.
	 *
	 * @var float
	 */
	private $packed_length;

	/**
	 * Box's length that maybe can be packed.
	 *
	 * @var float
	 */
	private $maybe_packed_length = null;

	/**
	 * Volume of the box.
	 *
	 * @var float
	 */
	private $volume;

	/**
	 * Valid box types which affect packing.
	 *
	 * @var array
	 */
	private $valid_types = array( 'box', 'tube', 'envelope', 'packet' );

	/**
	 * This box type.
	 *
	 * @var string
	 */
	private $type = 'box';

	/**
	 * Class constructor.
	 *
	 * @param float $length Box length.
	 * @param float $width Box width.
	 * @param float $height Box height.
	 * @param float $weight Box weight.
	 *
	 * @return void
	 */
	public function __construct( $length, $width, $height, $weight = 0 ) {
		$dimensions = array( $length, $width, $height );

		sort( $dimensions );

		$this->outer_length = floatval( $dimensions[2] );
		$this->length       = floatval( $dimensions[2] );
		$this->outer_width  = floatval( $dimensions[1] );
		$this->width        = floatval( $dimensions[1] );
		$this->outer_height = floatval( $dimensions[0] );
		$this->height       = floatval( $dimensions[0] );
		$this->weight       = floatval( $weight );
	}

	/**
	 * Set ID.
	 *
	 * @param string $id Box ID.
	 *
	 * @return void
	 */
	public function set_id( $id ) {
		$this->id = $id;
	}

	/**
	 * Set the volume to a specific value, instead of calculating it.
	 *
	 * @param float $volume Box volume.
	 *
	 * @return void
	 */
	public function set_volume( $volume ) {
		$this->volume = floatval( $volume );
	}

	/**
	 * Set the type of box.
	 *
	 * @param string $type Type of box.
	 *
	 * @return void
	 */
	public function set_type( $type ) {
		if ( in_array( $type, $this->valid_types, true ) ) {
			$this->type = $type;
		}
	}

	/**
	 * Get max weight.
	 *
	 * @return float
	 */
	public function get_max_weight() {
		return floatval( $this->max_weight );
	}

	/**
	 * Set maximum weight.
	 *
	 * @param mixed $weight Maximum weight.
	 *
	 * @return void
	 */
	public function set_max_weight( $weight ) {
		$this->max_weight = $weight;
	}

	/**
	 * Set inner dimension of box.
	 *
	 * @param mixed $length Inner length.
	 * @param mixed $width Inner width.
	 * @param mixed $height Inner height.
	 *
	 * @return void
	 */
	public function set_inner_dimensions( $length, $width, $height ) {
		$dimensions = array( $length, $width, $height );

		sort( $dimensions );

		$this->length = $dimensions[2];
		$this->width  = $dimensions[1];
		$this->height = $dimensions[0];
	}

	/**
	 * See if an item fits into the box.
	 *
	 * @param object $item Cart item.
	 *
	 * @return bool
	 */
	public function can_fit( $item ) {
		switch ( $this->type ) {
			// Tubes are designed for long thin items so see if the item meets that criteria here.
			case 'tube':
				$can_fit = ( $this->get_length() >= $item->get_length() && $this->get_width() >= $item->get_width() && $this->get_height() >= $item->get_height() && $item->get_volume() <= $this->get_volume() ) ? true : false;
				$can_fit = $can_fit && $item->get_length() >= ( ( $item->get_width() + $this->get_height() ) * 2 );
				break;
			// Packets are flexible.
			case 'packet':
				$can_fit = ( $this->get_packed_length() >= $item->get_length() && $this->get_packed_width() >= $item->get_width() && $item->get_volume() <= $this->get_volume() ) ? true : false;

				if ( $can_fit && $item->get_height() > $this->get_packed_height() ) {
					$this->maybe_packed_height = $item->get_height();
					$this->maybe_packed_length = $this->get_packed_length() - ( $this->maybe_packed_height - $this->get_height() );
					$this->maybe_packed_width  = $this->get_packed_width() - ( $this->maybe_packed_height - $this->get_height() );

					$can_fit = ( $this->maybe_packed_height < $this->maybe_packed_width && $this->maybe_packed_length >= $item->get_length() && $this->maybe_packed_width >= $item->get_width() ) ? true : false;
				}
				break;
			// Boxes are easy.
			default:
				$can_fit = ( $this->get_length() >= $item->get_length() && $this->get_width() >= $item->get_width() && $this->get_height() >= $item->get_height() && $item->get_volume() <= $this->get_volume() ) ? true : false;
				break;
		}

		return $can_fit;
	}

	/**
	 * Reset packed dimensions to originals
	 */
	private function reset_packed_dimensions() {
		$this->packed_length = $this->length;
		$this->packed_width  = $this->width;
		$this->packed_height = $this->height;
	}

	/**
	 * Pack the items.
	 *
	 * @param object $items List of items.
	 *
	 * @return object Package
	 */
	public function pack( $items ) {
		$packed        = array();
		$unpacked      = array();
		$packed_weight = $this->get_weight();
		$packed_volume = 0;
		$packed_value  = 0;

		$this->reset_packed_dimensions();

		$number_of_items = count( $items );
		while ( $number_of_items > 0 ) {
			$item            = array_shift( $items );
			$number_of_items = count( $items );
			// Check dimensions.
			if ( ! $this->can_fit( $item ) ) {
				$unpacked[] = $item;
				continue;
			}

			// Check max weight.
			if ( ( $packed_weight + $item->get_weight() ) > $this->get_max_weight() && $this->get_max_weight() > 0 ) {
				$unpacked[] = $item;
				continue;
			}

			// Check volume.
			if ( ( $packed_volume + $item->get_volume() ) > $this->get_volume() ) {
				$unpacked[] = $item;
				continue;
			}

			// Pack.
			$packed[]       = $item;
			$packed_volume += $item->get_volume();
			$packed_weight += $item->get_weight();
			$packed_value  += $item->get_value();

			// Adjust dimensions if needed, after this item has been packed inside.
			if ( ! is_null( $this->maybe_packed_height ) ) {
				$this->packed_height       = $this->maybe_packed_height;
				$this->packed_length       = $this->maybe_packed_length;
				$this->packed_width        = $this->maybe_packed_width;
				$this->maybe_packed_height = null;
				$this->maybe_packed_length = null;
				$this->maybe_packed_width  = null;
			}
		}

		// Get weight of unpacked items.
		$unpacked_weight = 0;
		$unpacked_volume = 0;
		foreach ( $unpacked as $item ) {
			$unpacked_weight += $item->get_weight();
			$unpacked_volume += $item->get_volume();
		}

		$package           = new stdClass();
		$package->id       = $this->id;
		$package->packed   = $packed;
		$package->unpacked = $unpacked;
		$package->weight   = $packed_weight;
		$package->volume   = $packed_volume;
		$package->length   = $this->get_outer_length();
		$package->width    = $this->get_outer_width();
		$package->height   = $this->get_outer_height();
		$package->value    = $packed_value;

		// Calculate packing success % based on % of weight and volume of all items packed.
		$packed_weight_ratio      = null;
		$packed_volume_ratio      = null;
		$packed_weight_to_compare = $packed_weight - $this->get_weight();

		if ( $packed_weight_to_compare + $unpacked_weight > 0 ) {
			$packed_weight_ratio = $packed_weight_to_compare / ( $packed_weight_to_compare + $unpacked_weight );
		}
		if ( $packed_volume + $unpacked_volume ) {
			$packed_volume_ratio = $packed_volume / ( $packed_volume + $unpacked_volume );
		}

		if ( is_null( $packed_weight_ratio ) && is_null( $packed_volume_ratio ) ) {
			// Fallback to amount packed.
			$package->percent = ( count( $packed ) / ( count( $unpacked ) + count( $packed ) ) ) * 100;
		} elseif ( is_null( $packed_weight_ratio ) ) {
			// Volume only.
			$package->percent = $packed_volume_ratio * 100;
		} elseif ( is_null( $packed_volume_ratio ) ) {
			// Weight only.
			$package->percent = $packed_weight_ratio * 100;
		} else {
			$package->percent = $packed_weight_ratio * $packed_volume_ratio * 100;
		}

		return $package;
	}

	/**
	 * Get box volume.
	 *
	 * @return float
	 */
	public function get_volume() {
		if ( $this->volume ) {
			return $this->volume;
		} else {
			return floatval( $this->get_height() * $this->get_width() * $this->get_length() );
		}
	}

	/**
	 * Get box height.
	 *
	 * @return float
	 */
	public function get_height() {
		return $this->height;
	}

	/**
	 * Get box width.
	 *
	 * @return float
	 */
	public function get_width() {
		return $this->width;
	}

	/**
	 * Get box length.
	 *
	 * @return float
	 */
	public function get_length() {
		return $this->length;
	}

	/**
	 * Get box weight.
	 *
	 * @return float
	 */
	public function get_weight() {
		return $this->weight;
	}

	/**
	 * Get box outher height.
	 *
	 * @return float
	 */
	public function get_outer_height() {
		return $this->outer_height;
	}

	/**
	 * Get box outer width.
	 *
	 * @return float
	 */
	public function get_outer_width() {
		return $this->outer_width;
	}

	/**
	 * Get box outer length.
	 *
	 * @return float
	 */
	public function get_outer_length() {
		return $this->outer_length;
	}

	/**
	 * Get packed box height.
	 *
	 * @return float
	 */
	public function get_packed_height() {
		return $this->packed_height;
	}

	/**
	 * Get packed box width.
	 *
	 * @return float
	 */
	public function get_packed_width() {
		return $this->packed_width;
	}

	/**
	 * Get packed box length.
	 *
	 * @return float
	 */
	public function get_packed_length() {
		return $this->packed_length;
	}
}
