<?php
/**
 * WooCommerce Box Packer file.
 *
 * @package woocommerce-shipping-canada-post
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WooCommerce Box Packer
 *
 * @version 2.0.1
 */
class WC_Boxpack {

	/**
	 * List of WC_Boxpack_Box.
	 *
	 * @var array
	 */
	private $boxes;

	/**
	 * List of WC_Boxpack_Item.
	 *
	 * @var array
	 */
	private $items;

	/**
	 * List of best package object.
	 *
	 * @var array
	 */
	private $packages;

	/**
	 * List of items that cannot be packed.
	 *
	 * @var array
	 */
	private $cannot_pack;

	/**
	 * __construct function.
	 *
	 * @return void
	 */
	public function __construct() {
		include_once WC_CANADA_POST_ABSPATH . 'includes/box-packer/class-wc-boxpack-box.php';
		include_once WC_CANADA_POST_ABSPATH . 'includes/box-packer/class-wc-boxpack-item.php';
	}

	/**
	 * Clear items list.
	 *
	 * @return void
	 */
	public function clear_items() {
		$this->items = array();
	}

	/**
	 * Clear boxes list.
	 *
	 * @return void
	 */
	public function clear_boxes() {
		$this->boxes = array();
	}

	/**
	 * Add item into items list.
	 *
	 * @param float  $length Item length.
	 * @param float  $width Item width.
	 * @param float  $height Item height.
	 * @param float  $weight Item weight.
	 * @param string $value Item value.
	 * @param array  $meta Item meta.
	 *
	 * @return void
	 */
	public function add_item( $length, $width, $height, $weight, $value = '', $meta = array() ) {
		$this->items[] = new WC_Boxpack_Item( $length, $width, $height, $weight, $value, $meta );
	}

	/**
	 * Add box into Boxes list.
	 *
	 * @param float $length Box length.
	 * @param float $width Box width.
	 * @param float $height Box height.
	 * @param float $weight Box weight.
	 *
	 * @return WC_Boxpack_Box
	 */
	public function add_box( $length, $width, $height, $weight = 0 ) {
		$new_box       = new WC_Boxpack_Box( $length, $width, $height, $weight );
		$this->boxes[] = $new_box;

		return $new_box;
	}

	/**
	 * Get list of packages.
	 *
	 * @return array List of packages.
	 */
	public function get_packages() {
		return $this->packages ? $this->packages : array();
	}

	/**
	 * Packing process.
	 *
	 * @throws Exception When items is empty.
	 *
	 * @return void
	 */
	public function pack() {
		try {
			// We need items.
			if ( empty( $this->items ) || ! is_array( $this->items ) ) {
				throw new Exception( 'No items to pack!' );
			}

			// Clear packages.
			$this->packages = array();

			// Order the boxes by volume.
			$this->boxes = $this->order_boxes( $this->boxes );

			if ( ! $this->boxes ) {
				$this->cannot_pack = $this->items;
				$this->items       = array();
			}

			// Keep looping until packed.
			$number_of_items = count( $this->items );
			while ( $number_of_items > 0 ) {
				$this->items       = $this->order_items( $this->items );
				$possible_packages = array();
				$best_package      = '';

				// Attempt to pack all items in each box.
				foreach ( $this->boxes as $box ) {
					$possible_packages[] = $box->pack( $this->items );
				}

				// Find the best success rate.
				$best_percent = 0;

				foreach ( $possible_packages as $package ) {
					if ( $package->percent > $best_percent ) {
						$best_percent = $package->percent;
					}
				}

				if ( 0 === $best_percent ) {
					$this->cannot_pack = $this->items;
					$this->items       = array();
				} else {
					// Get smallest box with best_percent.
					$possible_packages = array_reverse( $possible_packages );

					foreach ( $possible_packages as $package ) {
						if ( $package->percent === $best_percent ) {
							$best_package = $package;
							break; // Done packing.
						}
					}

					// Update items array.
					$this->items = $best_package->unpacked;

					// Store package.
					$this->packages[] = $best_package;
				}

				$number_of_items = count( $this->items );
			}

			// Items we cannot pack (by now) get packaged individually.
			if ( $this->cannot_pack ) {
				foreach ( $this->cannot_pack as $item ) {
					$package           = new stdClass();
					$package->id       = '';
					$package->weight   = $item->get_weight();
					$package->length   = $item->get_length();
					$package->width    = $item->get_width();
					$package->height   = $item->get_height();
					$package->value    = $item->get_value();
					$package->unpacked = true;
					$this->packages[]  = $package;
				}
			}
		} catch ( Exception $e ) {

			// Display a packing error for admins.
			if ( current_user_can( 'manage_options' ) ) {
				echo esc_html( 'Packing error: ' . $e->getMessage() . "\n" );
			}
		}
	}

	/**
	 * Order boxes by weight and volume.
	 *
	 * @param array $sort Boxes to be sorted.
	 *
	 * @return array
	 */
	private function order_boxes( $sort ) {
		if ( ! empty( $sort ) ) {
			uasort( $sort, array( $this, 'box_sorting' ) );
		}
		return $sort;
	}

	/**
	 * Order items by weight and volume.
	 *
	 * @param array $sort Item to be sorted.
	 *
	 * @return array
	 */
	private function order_items( $sort ) {
		if ( ! empty( $sort ) ) {
			uasort( $sort, array( $this, 'item_sorting' ) );
		}
		return $sort;
	}

	/**
	 * Order by volume.
	 *
	 * @param array $sort Item to be sorted.
	 *
	 * @return array
	 */
	private function order_by_volume( $sort ) {
		if ( ! empty( $sort ) ) {
			uasort( $sort, array( $this, 'volume_based_sorting' ) );
		}
		return $sort;
	}

	/**
	 * Sorting the item.
	 *
	 * @param mixed $a First item.
	 * @param mixed $b Second item.
	 *
	 * @return int
	 */
	public function item_sorting( $a, $b ) {
		if ( $a->get_volume() === $b->get_volume() ) {
			if ( $a->get_weight() === $b->get_weight() ) {
				return 0;
			}
			return ( $a->get_weight() < $b->get_weight() ) ? 1 : -1;
		}

		return ( $a->get_volume() < $b->get_volume() ) ? 1 : -1;
	}

	/**
	 * Sorting the box.
	 *
	 * @param mixed $a First box.
	 * @param mixed $b Second box.
	 *
	 * @return int
	 */
	public function box_sorting( $a, $b ) {
		if ( $a->get_volume() === $b->get_volume() ) {
			if ( $a->get_max_weight() === $b->get_max_weight() ) {
				return 0;
			}
			return ( $a->get_max_weight() < $b->get_max_weight() ) ? 1 : -1;
		}

		return ( $a->get_volume() < $b->get_volume() ) ? 1 : -1;
	}

	/**
	 * Sorting based on volume.
	 *
	 * @param mixed $a First item.
	 * @param mixed $b Second item.
	 *
	 * @return int
	 */
	public function volume_based_sorting( $a, $b ) {
		if ( $a->get_volume() === $b->get_volume() ) {
			return 0;
		}

		return ( $a->get_volume() < $b->get_volume() ) ? 1 : -1;
	}
}
