<?php
/**
 * Handle enqueuing scripts and styles.
 *
 * @package WooCommerce Delivery Time Pro
 */

namespace WDTCore;

class Assets {
	protected $base_url;
	protected $core_url;

	protected $handle;
	protected $cdnjs;

	protected $settings;

	public function __construct( $plugin_url, $handle, Settings $settings, Url $core_url ) {
		$this->base_url = "{$plugin_url}assets";
		$this->core_url = $core_url;
		$this->cdnjs    = 'https://cdnjs.cloudflare.com/ajax/libs';
		$this->handle   = $handle;
		$this->settings = $settings;
	}

	public function enqueue() {
		$this->enqueue_styles();
		$this->enqueue_scripts();
		$this->localize_scripts();
	}

	protected function enqueue_styles() {
		wp_enqueue_style( 'jquery-ui', "{$this->cdnjs}/jqueryui/1.12.1/themes/" . $this->settings->get( 'theme' ) . '/jquery-ui.min.css', '', '1.12.1' );
		wp_enqueue_style( 'wdtcore', $this->core_url->base_url() . '/assets/css/style.css' );
	}

	protected function enqueue_scripts() {
		wp_enqueue_script( $this->handle, "{$this->base_url}/js/script.js", $this->script_dependencies(), '1.0', true );
	}

	protected function script_dependencies() {
		wp_register_script( 'wdtcore', $this->core_url->base_url() . '/assets/js/script.js', [ 'jquery' ], '1.0', true );
		$dependencies = [ 'wdtcore', 'jquery-ui-datepicker' ];
		if ( ! is_admin() ) {
			$dependencies[] = 'js-cookie';
		}
		return $dependencies;
	}

	protected function localize_scripts() {
		wp_localize_script( $this->handle, strtoupper( $this->handle ), $this->get_localized_params() );
	}

	protected function get_localized_params() {
		$holidays = array_filter( array_map( 'trim', (array) $this->settings->get( 'holidays' ) ) );
		$this->add_disabled_days( $holidays );

		$weekdays = array_map( 'intval', (array) $this->settings->get( 'restricted_week_days' ) );

		// Get object last date what cannot access and min time available.
		$format_date  = 'F j, Y';
		$min_datetime = date( $format_date );

		$next = $this->settings->get( 'restricted_sequence_days' ) === '' ? 0 : intval( $this->settings->get( 'restricted_sequence_days' ) ) + 1;

		// Add time restricted sequence days.
		$min_datetime = date( $format_date, strtotime( $min_datetime . ' + ' . $next . ' days' ) );

		// Languages.
		$custom_date = array_filter( [
			'dayNamesMin' => $this->settings->get( 'custom_day_names' ) ? explode( ',', $this->settings->get( 'custom_day_names' ) ) : '',
			'nextText'    => $this->settings->get( 'custom_next_text' ),
			'prevText'    => $this->settings->get( 'custom_prev_text' ),
			'closeText'   => $this->settings->get( 'custom_close_text' ),
			'currentText' => $this->settings->get( 'custom_current_text' ),
			'monthNames'  => $this->settings->get( 'custom_month_names' ) ? explode( ',', $this->settings->get( 'custom_month_names' ) ) : '',
		] );

		return [
			'dateFormat'              => $this->settings->get( 'date_format' ),
			'restrictedWeekDays'      => $weekdays,
			'holidays'                => $holidays,
			'minDateTime'             => $min_datetime,
			'maxDate'                 => $this->settings->get( 'max_date' ) ? '+' . $this->settings->get( 'max_date' ) . 'd' : '+9999d',
			'ajaxUrl'                 => admin_url( 'admin-ajax.php' ),
			'showTimepicker'          => intval( $this->settings->get( 'show_timepicker' ) ),
			'nonceCheckShipping'      => wp_create_nonce( 'check-shipping' ),
			'customDate'              => $custom_date,
			'isAdminPage'             => intval( is_admin() ),
			'disabledShippingMethods' => $this->settings->get( 'disabled_shipping_methods' ),
		];
	}

	/**
	 * Add disabled days by rule "Disable shipping in the next" to list of holidays.
	 * Consider the "Count only working day" and "Restricted Week Days" rules.
	 *
	 * @param array $holidays List of holidays.
	 */
	protected function add_disabled_days( &$holidays ) {
		$next = intval( $this->settings->get( 'restricted_sequence_days' ) );
		if ( ! $next ) {
			return;
		}
		$next ++;

		$weekdays      = array_map( 'intval', $this->settings->get( 'restricted_week_days' ) );
		$only_work_day = $this->settings->get( 'count_only_working_day' );

		// Get list of disabled days.
		$disabled_days = [];
		$time          = current_time( 'timestamp' );
		$count         = 0;
		while ( $count < $next ) {
			if ( $only_work_day ) {
				$week_day = intval( date( 'w', $time ) );
				while ( in_array( $week_day, $weekdays, true ) ) {
					$time     += DAY_IN_SECONDS;
					$week_day = date( 'w', $time );
				}
			}

			$disabled_days[] = date( 'm/d/Y', $time );
			$time            += DAY_IN_SECONDS;
			$count ++;
		}

		if ( empty( $holidays ) ) {
			$holidays = $disabled_days;
			return;
		}

		for ( $index = 0, $length = count( $disabled_days ); $only_work_day && $index < $length; $index ++ ) {
			while ( $this->is_holiday( $disabled_days[ $index ] ) ) {
				$disabled_days[ $index ] = date( 'm/d/Y', strtotime( $disabled_days[ $index ] . ' + 1 day' ) );
			}

			$holidays[] = $disabled_days[ $index ];
		}
	}

	/**
	 * Check if a date is a holiday.
	 *
	 * @param string $date Date string.
	 *
	 * @return bool
	 */
	protected function is_holiday( $date ) {
		$holidays = array_filter( array_map( 'trim', $this->settings->get( 'holidays' ) ) );

		for ( $index = 0, $length = count( $holidays ); $index < $length; $index ++ ) {
			// Not a date range.
			if ( false === strpos( $holidays[ $index ], '-' ) ) {
				return $this->is_date_restricted( $date, $holidays[ $index ] );
			}

			// Date range.
			$range = explode( '-', $holidays[ $index ] );
			return $this->is_date_restricted_in_range( $date, trim( $range[0] ), trim( $range[1] ) );
		}

		return false;
	}

	/**
	 * Check if a date is a holiday.
	 *
	 * @param string $date       Date need to check (m/d/Y).
	 * @param string $restricted Restricted date (m/d/Y with/without stars).
	 *
	 * @return bool
	 */
	protected function is_date_restricted( $date, $restricted ) {
		// Make sure 5/1/2016 is the same as 5/01/2016.
		$date = date( 'm/d/Y', strtotime( $date ) );

		// If no stars.
		if ( false === strpos( $restricted, '*' ) ) {
			$restricted = date( 'm/d/Y', strtotime( $restricted ) );
			return $date === $restricted;
		}

		$restricted_parts = explode( '/', $restricted );
		$date_parts       = explode( '/', $date );

		return ( '*' === $restricted_parts[2] || $restricted_parts[2] === $date_parts[2] ) &&
		       ( '*' === $restricted_parts[0] || $restricted_parts[0] === $date_parts[0] ) &&
		       ( '*' === $restricted_parts[1] || $restricted_parts[1] === $date_parts[1] );
	}

	/**
	 * Check if a date is restricted by a date range
	 *
	 * @param string $date Date need to check (m/d/Y).
	 * @param string $from From date (m/d/Y with/without stars).
	 * @param string $to   To date (m/d/Y with/without stars).
	 *
	 * @return bool
	 */
	protected function is_date_restricted_in_range( $date, $from, $to ) {
		$date_parts = array_map( 'trim', explode( '/', $date ) );
		$from_parts = array_map( 'trim', explode( '/', $from ) );
		$to_parts   = array_map( 'trim', explode( '/', $to ) );

		// Do not check with rule */*/*.
		if (
			( '*' === $from_parts[0] && '*' === $from_parts[1] && '*' === $from_parts[2] ) ||
			( '*' === $to_parts[0] && '*' === $to_parts[1] && '*' === $to_parts[2] )
		) {
			return true;
		}

		// Set from date.
		if ( '*' === $from_parts[2] ) {
			$from_parts[2] = $date_parts[2];
		}
		if ( '*' === $from_parts[0] ) {
			$from_parts[0] = $date_parts[0];
		}
		if ( '*' === $from_parts[1] ) {
			$from_parts[1] = 1;
		}
		$from = implode( '/', $from_parts );

		// Set to date.
		if ( '*' === $to_parts[2] ) {
			$to_parts[2] = $date_parts[2];
		}
		if ( '*' === $to_parts[0] ) {
			$to_parts[0] = $date_parts[0];
		}
		if ( '*' === $to_parts[1] ) {
			$to_parts[1] = cal_days_in_month( CAL_GREGORIAN, $to_parts[0], $to_parts[2] );
		}
		$to = implode( '/', $to_parts );

		// Compare.
		$from = date( 'Y-m-d', strtotime( $from ) );
		$to   = date( 'Y-m-d', strtotime( $to ) );
		$date = date( 'Y-m-d', strtotime( $date ) );
		return $date >= $from && $date <= $to;
	}
}
