<?php

namespace WDTCore;

class Settings {
	/**
	 * Settings page ID.
	 *
	 * @var string
	 */
	protected $id;

	/**
	 * Option name.
	 *
	 * @var string
	 */
	protected $option;

	/**
	 * URL to assets folder.
	 *
	 * @var string
	 */
	protected $assets;

	/**
	 * List of settings tabs.
	 *
	 * @var array
	 */
	protected $tabs = [];

	/**
	 * The link to online documentation page.
	 *
	 * @var string
	 */
	protected $docs;

	public function __construct( $id, $option, $docs, $assets ) {
		$this->id     = $id;
		$this->option = $option;
		$this->docs   = $docs;
		$this->assets = $assets;
		$this->tabs   = [
			'general'    => __( 'General', 'woocommerce-delivery-time' ),
			'date'       => __( 'Date', 'woocommerce-delivery-time' ),
			'time'       => __( 'Time', 'woocommerce-delivery-time' ),
			'language'   => __( 'Language', 'woocommerce-delivery-time' ),
			'appearance' => __( 'Appearance', 'woocommerce-delivery-time' ),
		];
	}

	public function init() {
		add_filter( 'mb_settings_pages', [ $this, 'register_settings_page' ] );
		add_filter( 'rwmb_meta_boxes', [ $this, 'register_settings' ] );
	}

	public function register_settings_page( $settings_pages ) {
		$settings_pages[] = [
			'id'          => $this->id,
			'option_name' => $this->option,
			'menu_title'  => __( 'Delivery Time', 'woocommerce-delivery-time' ),
			'icon_url'    => 'dashicons-clock',
			'style'       => 'no-boxes',
			'columns'     => 1,
			'tabs'        => $this->tabs,
			'position'    => '56',
		];
		return $settings_pages;
	}

	public function register_settings( $meta_boxes ) {
		foreach ( $this->tabs as $tab => $label ) {
			$fields       = apply_filters( "wdtcore_settings_$tab", $this->$tab() );
			$meta_boxes[] = [
				'id'             => $tab,
				'title'          => $label,
				'settings_pages' => [ $this->id ],
				'tab'            => $tab,
				'fields'         => $fields,
			];
		}

		return $meta_boxes;
	}

	protected function general() {
		$settings = [];

		// Required?
		$settings[] = [
			'id'   => 'required',
			'type' => 'checkbox',
			'name' => __( 'Required?', 'woocommerce-delivery-time' ),
		];

		// Field label.
		$settings[] = [
			'name' => __( 'Label', 'woocommerce-delivery-time' ),
			'id'   => 'label',
			'type' => 'text',
			'std'  => $this->get_default( 'label' ),
		];

		// Enable date time picker along shipping methods selected.
		$options                                              = [];
		$zones                                                = [];
		$default_zone                                         = \WC_Shipping_Zones::get_zone( 0 );
		$zones[ $default_zone->get_id() ]                     = $default_zone->get_data();
		$zones[ $default_zone->get_id() ]['shipping_methods'] = $default_zone->get_shipping_methods();
		$zones                                                = array_merge( $zones, \WC_Shipping_Zones::get_zones() );
		foreach ( $zones as $id => $zone ) {
			$methods = $zone['shipping_methods'];
			foreach ( $methods as $method ) {
				$options[ $method->get_rate_id() ] = '[' . $zone['zone_name'] . '] ' . esc_html( $method->get_method_title() );
			}
		}
		$settings[] = [
			'name'    => __( 'Disable for shipping methods', 'woocommerce-delivery-time' ),
			'id'      => 'disabled_shipping_methods',
			'type'    => 'checkbox_list',
			'options' => $options,
			'desc'    => '',
		];

		// Number of shipments per day.
		$settings[] = [
			'name' => __( 'Maximum number of shipments per day', 'woocommerce-delivery-time' ),
			'id'   => 'shipments_per_day',
			'type' => 'text',
			'size' => 5,
		];
		// Message displayed when number of shipments exceeds plugin settings.
		$settings[] = [
			'name' => __( 'Notification for exceeded shipments', 'woocommerce-delivery-time' ),
			'desc' => __( 'Message displayed when number of shipments exceeds plugin setting above', 'woocommerce-delivery-time' ),
			'id'   => 'notification_exceeded_shipments',
			'type' => 'textarea',
		];

		return $settings;
	}

	protected function date() {
		$settings = [];

		// Date format.
		$desc       = [];
		$desc[]     = __( '<code>d</code>: Day of month (no leading zero)', 'woocommerce-delivery-time' );
		$desc[]     = __( '<code>dd</code>: Day of month (two digit)', 'woocommerce-delivery-time' );
		$desc[]     = __( '<code>m</code>: Month of year (no leading zero)', 'woocommerce-delivery-time' );
		$desc[]     = __( '<code>mm</code>: Month of year (two digit)', 'woocommerce-delivery-time' );
		$desc[]     = __( '<code>y</code>: Year (two digit)', 'woocommerce-delivery-time' );
		$desc[]     = __( '<code>yy</code>: Year (four digit)', 'woocommerce-delivery-time' );
		$desc[]     = $this->docs;
		$settings[] = [
			'name' => __( 'Date format', 'woocommerce-delivery-time' ),
			'id'   => 'date_format',
			'type' => 'text',
			'std'  => $this->get_default( 'date_format' ),
			'desc' => '<br>' . implode( '<br>', $desc ),
		];

		// Holidays.
		$desc       = [];
		$desc[]     = __( 'Date format: <code>mm/dd/yyyy</code>. Date range format: <code>mm/dd/yyyy - mm/dd/yyyy</code>.', 'woocommerce-delivery-time' );
		$desc[]     = __( 'Stars (<code>*</code>) can be used for any day, month or year.', 'woocommerce-delivery-time' );
		$desc[]     = __( 'Sample dates: <code>5/6/2017</code>, <code>6/*/*</code> (June), <code>*/8/*</code> (8th of all months).', 'woocommerce-delivery-time' );
		$desc[]     = __( 'Sample date ranges: <code>5/6/2013 - 8/*/2013</code>, <code>6/*/* - 8/*/*</code> (from June to August), <code>*/1/* - */5/*</code> (from 1st to 5th all months).', 'woocommerce-delivery-time' );
		$desc[]     = $this->docs;
		$settings[] = [
			'name'              => __( 'Holidays', 'woocommerce-delivery-time' ),
			'desc'              => '<br><span class="description">' . implode( '<br>', $desc ) . '</span>',
			'label_description' => __( 'Enter dates OR date ranges', 'woocommerce-delivery-time' ),
			'id'                => 'holidays',
			'type'              => 'text',
			'clone'             => true,
			'size'              => 45,
		];

		// Restricted week days.
		$week_days  = [
			0 => __( 'Sunday', 'woocommerce-delivery-time' ),
			1 => __( 'Monday', 'woocommerce-delivery-time' ),
			2 => __( 'Tuesday', 'woocommerce-delivery-time' ),
			3 => __( 'Wednesday', 'woocommerce-delivery-time' ),
			4 => __( 'Thursday', 'woocommerce-delivery-time' ),
			5 => __( 'Friday', 'woocommerce-delivery-time' ),
			6 => __( 'Saturday', 'woocommerce-delivery-time' ),
		];
		$settings[] = [
			'name'    => __( 'Do not deliver on', 'woocommerce-delivery-time' ),
			'id'      => 'restricted_week_days',
			'type'    => 'checkbox_list',
			'options' => $week_days,
		];

		// Restricted sequence days.
		$settings[] = [
			'name' => __( 'Do not delivery in the next (days)', 'woocommerce-delivery-time' ),
			'desc' => __( '0 = today, 1 = today and tomorrow, empty = no restriction', 'woocommerce-delivery-time' ),
			'id'   => 'restricted_sequence_days',
			'type' => 'text',
			'size' => 5,
		];

		// Option allow Disable shipping in the next option just applies on working days.
		$settings[] = [
			'name' => __( 'Count only working day', 'woocommerce-delivery-time' ),
			'desc' => __( 'Count only working day in the "Do not delivery in the next XXX days" above.<br><p class="description">Do not count holidays and restricted week days above.</p>', 'woocommerce-delivery-time' ),
			'id'   => 'count_only_working_day',
			'type' => 'checkbox',
		];

		// Allow to ship in sequence days.
		$settings[] = [
			'name' => __( 'Deliver only in the next (days)', 'woocommerce-delivery-time' ),
			'desc' => __( '0 = today, 1 = today and tomorrow, empty = no restriction', 'woocommerce-delivery-time' ),
			'id'   => 'max_date',
			'type' => 'text',
			'size' => 5,
		];

		return $settings;
	}

	protected function time() {
		$settings = [];

		// Show time slots?
		$settings[] = [
			'name' => __( 'Show time', 'woocommerce-delivery-time' ),
			'id'   => 'show_timepicker',
			'type' => 'checkbox',
		];

		return $settings;
	}

	protected function language() {
		$settings = [];

		$settings[] = [
			'type' => 'custom_html',
			'std'  => __( 'If you see the text are not translated to your language, please set them below.', 'woocommerce-delivery-time' ),
		];

		$settings[] = [
			'name' => __( 'Close', 'woocommerce-delivery-time' ),
			'id'   => 'custom_close_text',
			'type' => 'text',
		];
		$settings[] = [
			'name' => __( 'Previous', 'woocommerce-delivery-time' ),
			'id'   => 'custom_prev_text',
			'type' => 'text',
		];
		$settings[] = [
			'name' => __( 'Next', 'woocommerce-delivery-time' ),
			'id'   => 'custom_next_text',
			'type' => 'text',
		];
		$settings[] = [
			'name' => __( 'Today', 'woocommerce-delivery-time' ),
			'id'   => 'custom_current_text',
			'type' => 'text',
		];
		$settings[] = [
			'name' => __( 'Month names', 'woocommerce-delivery-time' ),
			'desc' => __( 'Separated by commas: January, February, etc.', 'woocommerce-delivery-time' ),
			'id'   => 'custom_month_names',
			'type' => 'text',
		];
		$settings[] = [
			'name' => __( 'Day names (short)', 'woocommerce-delivery-time' ),
			'desc' => __( 'Separated by commas: Su, Mo, Tu, We, Th, Fr, Sa', 'woocommerce-delivery-time' ),
			'id'   => 'custom_day_names',
			'type' => 'text',
		];

		return $settings;
	}

	protected function appearance() {
		$settings = [];

		$options = [];
		$themes  = [
			'base',
			'black-tie',
			'blitzer',
			'cupertino',
			'dark-hive',
			'dot-luv',
			'eggplant',
			'excite-bike',
			'flick',
			'hot-sneaks',
			'humanity',
			'le-frog',
			'mint-choc',
			'overcast',
			'pepper-grinder',
			'smoothness',
			'south-street',
			'start',
			'sunny',
			'swanky-purse',
			'trontastic',
			'ui-darkness',
			'ui-lightness',
			'vader',
		];
		foreach ( $themes as $theme ) {
			$options[ $theme ] = "{$this->assets}/images/$theme.png";
		}
		$settings[] = [
			'name'    => __( 'Themes', 'woocommerce-delivery-time' ),
			'id'      => 'theme',
			'type'    => 'image_select',
			'options' => $options,
			'std'     => $this->get_default( 'theme' ),
		];
		return $settings;
	}

	/**
	 * Get plugin setting.
	 *
	 * @param string $name Setting name.
	 *
	 * @return mixed
	 */
	public function get( $name ) {
		$option = get_option( $this->option );
		return isset( $option[ $name ] ) ? $option[ $name ] : $this->get_default( $name );
	}

	protected function get_default( $name ) {
		$defaults = $this->defaults();
		return isset( $defaults[ $name ] ) ? $defaults[ $name ] : '';
	}

	protected function defaults() {
		return [
			'required'                        => 0,
			'disabled_shipping_methods'       => [],
			'label'                           => __( 'Delivery Time', 'woocommerce-delivery-time' ),
			'date_format'                     => 'mm/dd/yy',
			'holidays'                        => [],
			'restricted_week_days'            => [],
			'show_timepicker'                 => 0,
			'notification_exceeded_shipments' => __( 'You cannot request shipping on this day!', 'woocommerce-delivery-time' ),
			'theme'                           => 'base',
		];
	}
}
