<?php
/**
 * Check number of shipments via ajax.
 *
 * @package WooCommerce Delivery Time Pro
 */

namespace WDTCore;

class Shipments {
	protected $settings;

	/**
	 * Data of shipments by date.
	 *
	 * @var array
	 */
	protected $data = [];

	/**
	 * Path to the data file.
	 *
	 * @var string
	 */
	protected $data_file = '';

	public function __construct( Settings $settings ) {
		$this->settings = $settings;
		$upload_dir     = wp_upload_dir();
		$upload_path    = $upload_dir['basedir'];

		$this->data_file = $upload_path . '/woocommerce-delivery-time.json';
	}

	public function init() {
		add_action( 'wp_ajax_wdtcore_check_number_of_shipments', [ $this, 'check_number_of_shipments' ] );
		add_action( 'wp_ajax_nopriv_wdtcore_check_number_of_shipments', [ $this, 'check_number_of_shipments' ] );
	}

	public function check_number_of_shipments() {
		$max = intval( trim( $this->settings->get( 'shipments_per_day' ) ) );

		if ( $max < 1 ) {
			wp_send_json_success();
		}

		$this->get_data();
		$date = filter_input( INPUT_POST, 'date', FILTER_SANITIZE_STRING );
		$date = date( 'Y-m-d', strtotime( $date ) );

		if ( isset( $this->data[ $date ] ) && $max <= $this->data[ $date ] ) {
			wp_send_json_error( [
				'message' => $this->settings->get( 'notification_exceeded_shipments' ),
				'type'    => 'exceed',
			] );
		}

		$number = isset( $this->data[ $date ] ) ? absint( $this->data[ $date ] ) : 0;
		$number ++;
		$this->data[ $date ] = $number;

		$this->delete_past_dates();
		$this->update_report();

		wp_send_json_success();
	}

	/**
	 * Get shipping report
	 */
	protected function get_data() {
		if ( file_exists( $this->data_file ) ) {
			$data       = @file_get_contents( $this->data_file );
			$this->data = json_decode( $data, true );
		}
	}

	/**
	 * Delete log of shipping for past dates.
	 */
	protected function delete_past_dates() {
		$today = current_time( 'timestamp' );
		foreach ( $this->data as $key => $value ) {
			if ( strtotime( $key ) < $today ) {
				unset( $this->data[ $key ] );
			}
		}
	}

	/**
	 * Update shipping days in file JSON.
	 */
	protected function update_report() {
		@file_put_contents( $this->data_file, wp_json_encode( $this->data ) );
	}
}
