<?php
/*
Plugin Name: Everslider
Plugin URI: http://plugins.gravitysign.com/everslider-wp/
Description: Plugin to create responsive, mobile-friendly carousel sliders from your blog posts or custom post types
Version: 1.4
Author: flGravity
Author URI: http://codecanyon.net/user/flGravity
*/


/**
* Define constans
*/

define('ES_PLUGIN_URL', plugin_dir_url(__FILE__));
define('ES_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('ES_PLUGIN_DOCS', ES_PLUGIN_URL . 'docs/');


/**
* Include functions to work with slider options
*/

require(ES_PLUGIN_DIR . 'includes/options.php');


/**
* Include admin options page
*/

require(ES_PLUGIN_DIR . 'includes/admin.php');


/**
* Include custom post type and taxonomy
*/

require(ES_PLUGIN_DIR . 'includes/post.php');


/**
* Load plugin scripts and styles
*/
	
function everslider_enqueue_scripts() {
	// scripts
	wp_enqueue_script('jquery-easing', ES_PLUGIN_URL . 'jquery.easing.1.3.js', array('jquery'), '1.3', true);
	wp_enqueue_script('jquery-mousewheel',ES_PLUGIN_URL . 'jquery.mousewheel.js', array('jquery'), '3.1.3', true);
	wp_enqueue_script('jquery-everslider', ES_PLUGIN_URL . 'jquery.everslider.min.js', array('jquery'), '1.5', true);
	wp_enqueue_script('templates', ES_PLUGIN_URL . 'templates.js', array('jquery','jquery-everslider'), '1.0', true);
	// styles
	wp_enqueue_style('everslider-main', ES_PLUGIN_URL . 'everslider.css');
	wp_enqueue_style('everslider-templates', ES_PLUGIN_URL . 'templates.css', 'everslider');
}

add_action('wp_enqueue_scripts', 'everslider_enqueue_scripts');


/**
* Load plugin translations
*/

function everslider_load_textdomain(){
	$domain = 'everslider'; // text domain name
	load_plugin_textdomain($domain, false, dirname(plugin_basename(__FILE__)) . '/languages/');
}

add_action('plugins_loaded', 'everslider_load_textdomain');


/**
* Function to print custom styles for sliders that have template set to "custom"
*/

function everslider_print_custom_styles() {
	echo '<style type="text/css">' . PHP_EOL;
	// loop sliders in configuration
	foreach( everslider_get_slider_list() as $slider_id ) {
		// get slider options
		$slider_options = everslider_get_slider_options($slider_id);
		if( $slider_options && $slider_options['template'] == 'custom' ) {
			$css_tmpl = $slider_options['custom_template_style'];
			if( !empty($css_tmpl) ) {
				// replace paceholders and print this style
				$css_tmpl = str_replace(array('%slider%', '%slider_id%'), '#' . $slider_id, $css_tmpl);
				$css_tmpl = str_replace('%plugin_url%', plugins_url('', __FILE__), $css_tmpl);
				// replace few spaces with one and remove new lines
				$css_tmpl = preg_replace('/\s+/', ' ', $css_tmpl);
				$css_tmpl = str_replace(array("\n","\r"), ' ', $css_tmpl);
				echo $css_tmpl . PHP_EOL;
			}		
		}
	}
	echo '</style>' . PHP_EOL;
}

add_action('wp_head', 'everslider_print_custom_styles');


/**
* Function to print javascript init code for carousel sliders that
* where added with get_everslider() template tag or shortcode. Use 
* 'everslider_settings' filter to alter json settings that will be 
* passed over to jQuery plugin
*/

function everslider_print_init_scripts() {
	echo '<script type="text/javascript">' . PHP_EOL . 'jQuery(function(){' . PHP_EOL;
	// loop sliders in configuration
	foreach( everslider_get_slider_list() as $slider_id) {
		// get slider options
		$slider_options = everslider_get_slider_options($slider_id);
		if( $slider_options && isset($slider_options['used']) ) {
			// set "used" param for slider back to false
			unset($slider_options['used']);
			everslider_update_slider_options($slider_id, $slider_options);
			// run 'everslider_settings' filter
			$plugin_settings = apply_filters('everslider_settings', $slider_options['plugin_settings'], $slider_id);
			// output javascript init for slider
			echo 'jQuery("#' . $slider_id . '").everslider(' . json_encode($plugin_settings) . ');' . PHP_EOL;
		}
	}
	echo '});' . PHP_EOL . '</script>' . PHP_EOL;
}

add_action('wp_footer', 'everslider_print_init_scripts');


/**
* Function that returns actual markup for carousel slider
*/

function everslider_get_markup($slider_id) {
	global $post;
	
	// include slide templates ($templates array)
	include('templates.php');
		
	// get slider ID
	$slider_id = trim($slider_id);
	if( empty($slider_id) )
		return __('Slider ID has not been set','everslider');
	
	// get options for the slider
	$slider_options = everslider_get_slider_options($slider_id);
	if(false === $slider_options) 
		return sprintf( __('There is no slider with ID <strong>%s</strong>','everslider'), $slider_id );
		
	// get slider template name
	$tmpl_name = isset($slider_options['template']) ? $slider_options['template'] : 'default';
	
	if($tmpl_name === 'custom') {
		$tmpl = $slider_options['custom_slide_template'];
	} else {
		if( isset($templates, $templates[$tmpl_name]) )
			$tmpl = $templates[$tmpl_name];
		else 
			return sprintf( __('Template <em>%s</em> is not defined','everslider'), $tmpl_name);
	}
	
	// tax query
	$tax_query = array( 'relation' => 'AND' );
	
	// process groups (hierarchical taxonomy)
	if( isset($slider_options['groups']) ) {
		$tax_query_groups = array();
		foreach ( $slider_options['groups'] as $group) {
			list($taxonomy, $term) = explode('&', $group);
			if( !array_key_exists($taxonomy, $tax_query_groups) ) {
				$tax_query_groups[$taxonomy] = array(
					'taxonomy' => $taxonomy,
					'field' => 'slug',
					'terms' => array($term)
				);
			} else {
				$tax_query_groups[$taxonomy]['terms'][] = $term;
			}
		}
		$tax_query = array_merge($tax_query, $tax_query_groups);
	}
		
	// process tags (non-hierarchical taxonomy)
	if( isset($slider_options['tags']) ) {
		$tax_query_tags = array();
		foreach ($slider_options['tags'] as $tag) {
			list($taxonomy, $term) = explode('&', $tag);
			if( !array_key_exists($taxonomy, $tax_query_tags) ) {
				$tax_query_tags[$taxonomy] = array(
					'taxonomy' => $taxonomy,
					'field' => 'slug',
					'terms' => array($term)
				);
			} else {
				$tax_query_tags[$taxonomy]['terms'][] = $term;
			}
		}
		$tax_query = array_merge($tax_query, $tax_query_tags);
	}
				
	// run WP_Query
	$query = new WP_Query(array(
		'post_type' => $slider_options['source'],
		'tax_query' => $tax_query,
		'orderby' => $slider_options['orderby'],
		'order' => $slider_options['order'],
		'posts_per_page' => isset($slider_options['limit']) ? $slider_options['limit'] : '-1' 
	));
			
	if( $query->have_posts() ) {
		
		// add classes to Everslider container
		$class = "everslider es-{$tmpl_name}-template es-{$slider_options['source']}-source "; 
		$class .= "es-align-{$slider_options['align']} ";
		$ps = $slider_options['plugin_settings'];
		$class .= $ps['ticker'] && $ps['mode'] != 'normal' ? "es-has-ticker " : "";
		$class .= $ps['pagination'] ? "es-has-pagination " : "";
		$class .= $ps['navigation'] ? "es-has-navigation " : "";
		// apply filter to add some other classes
		$class = apply_filters('everslider_class', $class);
		
		// begin markup
		$markup = '<div id="' . esc_attr($slider_id) . '" class="' . $class . '">' . PHP_EOL;
		$markup .= '<ul class="es-slides">' . PHP_EOL;
		
		while( $query->have_posts() ) {
			$query->the_post();
			$markup .= '<li>' . apply_filters('everslider_slide', $tmpl, $post) . '</li>' . PHP_EOL;
		}
		
		// end markup
		$markup .= '</ul>' . PHP_EOL . '</div>' . PHP_EOL;
		
		// mark carousel slider as used
		$slider_options['used'] = true;
		everslider_update_slider_options($slider_id, $slider_options);
		
		// restore post data
		wp_reset_postdata();

	} else {
		return __('Oops! Slider has no slides','everslider');
	}
		
	// apply filters and return markup	
	return apply_filters('everslider_markup', $markup);
}


/**
* Filter that replaces placeholders in HTML template for slide with actual info
*
* @param string $tmpl HTML template for slide
* @param object $post_raw Raw $post data
*/

function everslider_slide_template_placeholders($tmpl, $post_raw){
	
	// replace %id% placeholder
	$tmpl = str_replace('%id%', get_the_ID(), $tmpl);
	
	// replace %author% placeholder
	$tmpl = str_replace('%author%', get_the_author(), $tmpl);
	
	// replace %content% placeholder
	$tmpl = str_replace('%content%', get_the_content(), $tmpl);
	
	// replace %title% placeholder
	$tmpl = str_replace('%title%', get_the_title(), $tmpl);
	
	// replace %excerpt% placeholder
	$excerpt = wp_trim_words(get_the_excerpt(), 12, '');
	$excerpt = preg_replace('/\W+$/', '', $excerpt);
    $tmpl = str_replace('%excerpt%', $excerpt . ' ..', $tmpl);
	
	// replace %comments_number% placeholder
	$tmpl = str_replace('%comments_number%', get_comments_number(), $tmpl);
	
	// replace %comments_link% placeholder
	$tmpl = str_replace('%comments_link%', get_comments_link(), $tmpl);
	
	// replace %date% placeholder
	$tmpl = str_replace('%date%', get_the_date(), $tmpl);
	
	// replace %date_modified% placeholder
	$tmpl = str_replace('%date_modified%', get_the_modified_date(), $tmpl);
	
	// replace %permalink% placeholder
	$tmpl = str_replace('%permalink%', get_permalink(), $tmpl);
	
	// replace %slug% placeholder
	$tmpl = str_replace('%slug%', $post_raw->post_name, $tmpl);
	
	// replace %image% placeholder
	if( has_post_thumbnail() ) {
		$tmpl = str_replace('%image%', get_the_post_thumbnail($post_raw->ID, 'full'), $tmpl);
	} else {
		$tmpl = str_replace('%image%', __('No image', 'everslider'), $tmpl);
	}
	
	// replace image meta data placeholders, like %image_title%, %image_alt%, 
	// %image_caption%, %image_description% and %image_src%
	$attachment_id = get_post_thumbnail_id($post_raw->ID);
	if( $attachment_id ) {
		$attachment = get_post($attachment_id);
		$attachment_alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
		$attachment_src = wp_get_attachment_image_src($attachment_id, 'full');
		$tmpl = str_replace('%image_title%', esc_html($attachment->post_title), $tmpl);
		$tmpl = str_replace('%image_alt%', esc_html($attachment_alt), $tmpl);
		$tmpl = str_replace('%image_caption%', wp_kses_post($attachment->post_excerpt), $tmpl);
		$tmpl = str_replace('%image_description%', wp_kses_post($attachment->post_content) , $tmpl);
		$tmpl = str_replace('%image_src%', $attachment_src[0], $tmpl);
	} else {
		$tmpl = preg_replace('/%image_[a-z]+%/', '', $tmpl);
	}
		
	// replace post taxonomies placeholders
	$groups_list = ''; $tags_list = '';
	$taxonomies = get_object_taxonomies($post_raw->post_type, 'object');
	foreach( $taxonomies as $taxonomy) {
		$separator = $taxonomy->hierarchical ? ' / ' : ', ';
		$term_list  = get_the_term_list($post_raw->ID, $taxonomy->name, '', $separator, '');
		$tmpl = str_replace('%' . $taxonomy->name  . '%', $term_list, $tmpl); 
		if( !empty($term_list) ) {
			if( $taxonomy->hierarchical ) {
					$groups_list .=  $term_list . $separator;
			} else {
					$tags_list .= $term_list  . $separator;
			}
		}
	}
		
	// replace %tags% and %groups% placeholders with all hierarchical and non-hierarchical terms
	$tmpl = str_replace('%groups%', substr($groups_list, 0, -2), $tmpl);
	$tmpl = str_replace('%tags%', substr($tags_list, 0, -2), $tmpl);
	
	// replace post custom fields placeholders, like %meta_{key}%
	$post_meta = get_post_custom($post_raw->ID);
	foreach($post_meta as $post_meta_key => $post_meta_value) {
		// check if key is not hidden
		if( $post_meta_key[0] !== '_' ) {
			$tmpl = str_replace('%meta_' . $post_meta_key . '%', wp_kses_post($post_meta_value[0]), $tmpl);
		}
	}
	
	// return slide template with replaced placeholders
	return $tmpl;
}

add_filter('everslider_slide', 'everslider_slide_template_placeholders', 5, 2);


/**
* EVERSLIDER FILTERS 
*
*
* everslider_class :        filter to modify/add classes to carousel container
*
* everslider_slide :        filter HTML of carousel slide(item). Filter accepts
*               			2 arguments - HTML markup and WP $post object
* 
* everslider_markup :       filter Everslider HTML before it is printed with 
*                           get_everslider() template tag or [everslider] shortcode                  
*
* everslider_settings :     filter jQuery plugin settings for carousel instance.
*                           Filter accepts 2 arguments - settings and slider ID                       
*/


/**
* Public. Everslider template tag
*
* @param $slide Slider ID
* @return string HTML markup
*
*/

function get_everslider($slider_id) {
	return everslider_get_markup($slider_id);
}


/**
* Public. Everslider shortcode
*
* [everslider id="slider"]
*/

function everslider_shortcode($atts, $content = '') {
	$slider_id = $atts['id'];
	return everslider_get_markup($slider_id);
}

add_shortcode('everslider', 'everslider_shortcode');


?>