<?php

/********************************************************************************
	
	This file creates admin menu for Everslider and handles
	content output for options page
		
********************************************************************************/

/**
* Inclide settings
*/

require( ES_PLUGIN_DIR . 'includes/settings.php' );


/**
* Function to generate slider id from slider name
*
* @param string $name Slider name
* @return string Slider id
*/

function everslider_get_slider_id( $name ){
	$id = strtolower( $name );
	$id = preg_replace('/[^_a-z0-9]/', '_', $id);
	$id = preg_replace('/_+/', '_', $id);
	$id = preg_replace('/^_|_$/', '', $id);
	return $id;
}


/** 
* Function to add create new slider with default configuration
*
* @param string $slider_name Slider name
* @return string Slider id on success. False on failure
*/

function everslider_create_new_slider( $slider_name ){

	// jQuery default plugin settings
	$plugin_settings = array(
		'mode' => 'normal',
		'effect' => 'slide',
		'useCSS' => true,
		'itemWidth' => false,
		'itemHeight' => false,
		'itemMargin' => false,
		'itemKeepRatio' => true,
		'maxWidth' => '100%',
		'maxVisible' => 0,
		'moveSlides' => 1,
		'slideDelay' => 0,
		'slideDuration' => 500,
		'slideEasing' => 'swing',
		'fitDuration' => 200,
		'fitEasing' => 'swing',
		'syncHeight' => false,
		'syncHeightDuration' => 200,
		'syncHeightEasing' => 'swing',
		'navigation' => true,
		'pagination' => true,
		'touchSwipe' => true,
		'swipeThreshold' => 50,
		'swipePage' => false,
		'mouseWheel' => false,
		'keyboard' => false,
		'ticker' => false,
		'tickerAutoStart' => false,
		'tickerTimeout' => 2000,
		'tickerHover' => false
	);

	// generate slider id
	$slider_id = everslider_get_slider_id($slider_name);
	
	// save slider options to database
	$result = everslider_add_slider_options($slider_id, array(
		'id' => $slider_id,
		'name' => $slider_name,
		'source' => 'post',
		'limit' => -1,
		'orderby' => 'post_date',
		'order' => 'ASC',
		'groups' => array(),
		'tags' => array(),
		'align' => 'none',
		'template' => 'default',
		'plugin_settings' => $plugin_settings,
		'custom_slide_template' => '',
		'custom_template_style' => ''
	));
	
	// return slider id or false on failure
	return $result ? $slider_id : false;	
}


/**
* Create admin menu, add thickbox and admin scripts
*/

function everslider_add_menu() {
	global $everslider_menu_hook;
	// add options page
	$everslider_menu_hook = add_menu_page( __('Everslider', 'everslider'), __('Everslider', 'everslider'), 'manage_options', 'everslider', 'everslider_display_options_page');
	// enable thickbox
	add_action('admin_print_scripts-' . $everslider_menu_hook, 'add_thickbox');
	// add custom styles/script to the menu page
	add_action('admin_print_scripts-' . $everslider_menu_hook, 'everslider_add_menu_page_custom_styles');
	add_action('admin_footer-' . $everslider_menu_hook, 'everslider_add_custom_scripts');
	// add contextual help
	add_filter('contextual_help', 'everslider_contextual_help', 10, 3);	
}

add_action('admin_menu', 'everslider_add_menu');


/**
* Function to add custom styles for menu page
*/

function everslider_add_menu_page_custom_styles(){
	?>
	<style type="text/css" media="screen">
		/* Everslider WP - Admin Styles */
		.es-settings-wrap { margin-right: 350px; }
		.es-sidebar-wrap { float: right; margin-right: -350px; width: 330px; }
		.es-sidebar-wrap .postbox h3 { cursor: pointer; }
		.es-sidebar-wrap .description { margin-top: 8px; }
		.es-sidebar-wrap .es-m4 { display: block; margin: 4px 0; }
		.es-sidebar-wrap .es-m4 input[type="checkbox"] { margin-left: 2px; }
		.form-table { float: left; }
		.form-table select { min-width: 130px; }
		.es-post-type { float: right; font-size: 1em; margin: 1.33em 0; }
		.widefat th.sortable { min-width: 100px; }
	</style>
	<?php
}


/**
* Function to add custom scrips to the footer
*/

function everslider_add_custom_scripts(){
	?>
		<script type="text/javascript">
			/* Everslider WP - Admin Scripts */
			(function($){
				
				// show confirm when removing slider
				$('.es-remove-slider').click(function(){
					var name = $(this).closest('tr').find('.es-slider-name').text();
					return confirm('<?php _e('Are you sure you want to remove', 'everslider'); ?> "' + name + '"?');
				});
				
				// move focus to input when trying to creating new slider without a name
				$('#new_slider_form').submit( function(){
					var field = $('input[name="slider_name"]', this);
					if(field.val() == '') {
						field.get(0).focus();
						return false;
					}
				});
				
				// postbox toggle
				$('.postbox').prepend('<div class="handlediv" title="Click to toggle"><br></div>').
				find('> h3').click(function(){
					$(this).parent().toggleClass('closed');
				});
				
				// text selection on click
				$('.es-shortcode').click(function(){
					var text = this;
					if(document.body.createTextRange) { // ie
						var range = document.body.createTextRange();
						range.moveToElementText(text);
						range.select();
					} else if(window.getSelection && document.createRange) { // other
						var selection = window.getSelection();
						var range = document.createRange();
						range.selectNodeContents(text);
						selection.removeAllRanges();
						selection.addRange(range);
					}				
				});
				
				// source change on "Edit" page triggers page reload
				$('.es-source-input').change(function(){
					document.createElement('form').submit.call(this.form);
				});
				
				
				// enable/disable slides post type
				$('#es_post_type').change(function(){
					this.form.submit();
				});
				
				// modify "_wp_http_referer" on "Edit" page when slider has been renamed
				$('form[action="options.php"]').submit(function(){
					var slider_name = $('.es-slider-name', this).val();
					var slider_id = slider_name.toLowerCase().
									replace(/[^_0-9a-z]/g,'_').replace(/_+/g,'_').replace(/^_|_$/g,'');
					var referer_field = $('[name="_wp_http_referer"]', this);
					var referer = referer_field.val();
					if(slider_id !== $('[name="slider_id"]').val()) {						
						referer_field.val(referer.replace(/slider_id=/, 'slider_id=' + slider_id + '&orig_slider_id='));
					} else {
						referer_field.val(referer.replace(/orig_slider_id=\w+&/,''));
					}
					return true;
				});
			})(jQuery);
		</script>
	<?php
}


/**
* Function to add contextual help to Everslider menu page
*/

function everslider_contextual_help( $contextual_help, $screen_id, $screen ){
	global $everslider_menu_hook;
	
	if( $screen_id == $everslider_menu_hook ) {
		$contextual_help = '<h4>Plugin Usage</h4>' .
		'<p>To insert carousel slider copy shortcode that you can find next to each slider name in <em>"All Sliders"</em> list and paste it where you want slider to appear</p>' . 
		'<p><code>[everslider id="slider_one"]</code></p>' .
		'<p>If you work with theme PHP files, then to insert Everslider carousel you should use <em>get_everslider()</em> template tag</p>' . 
		'<p><code>&lt;?php echo get_everslider("slider_one"); ?&gt;</code></p>' .
		'<p>For more information about plugin please check its <a href="' . ES_PLUGIN_DOCS . '" target="_blank">documentation</a>' .
		'<br>If you have any ideas how to improve this plugin or want to leave feedback you can always email me <a href="mailto:bestesee@gmail.com">here</a>. Thanks!</p>';
	} 
	
	return $contextual_help;
}


/**
* This is main function for displaying Everslider options page 

* To display different content on just one page function check "action" 
* query argument and executes functions hooked to 'everslider_options_page_{action}' 
* action hook. Also it passes array to the hooked functions with info about slider
*/

function everslider_display_options_page(){
	// every request should contain "action" so we check this on first place
	$action = isset($_GET['action']) ? trim($_GET['action']) : 'default';
	
	// next we check nonce for all actions except "default" which is quite safe
	$nonce = isset($_GET['_wpnonce']) ? trim($_GET['_wpnonce']) : '';
	if( $action !== 'default' && false === wp_verify_nonce($nonce, __FILE__) ) {
		$message = __('You do not have permissions to perform this action', 'everslider');
		wp_die($message, '', 'back_link=true');
	}
	
	// query arguments may also contain slider id or name, so we get them also
	$slider_id = isset($_GET['slider_id']) ? trim($_GET['slider_id']) : '';
	$slider_name = isset($_GET['slider_name']) ? trim($_GET['slider_name']) : '';
	
	// execute hooked functions for action
	do_action('everslider_options_page_' . $action, array('slider_id' => $slider_id, 'slider_name' => $slider_name));
}


/**
* Display options page for "default" action
*/

function everslider_default_page_content( $args ) {
	$message = isset($args['message']) ? $args['message']: '';
	$order = isset($_GET['order']) ? trim($_GET['order']) : 'desc';
?>
	<div class="wrap">
		<div id="everslider-icon" class="icon32">
			<img src="<?php echo ES_PLUGIN_URL . 'includes/images/icon.png'; ?>" alt="Icon">
		</div>
		<h2><?php echo __('All Sliders', 'everslider'); ?></h2>
		<br>
		<?php if( $message !== '' ) : ?>
		<!--// status message //-->
		<div id="message" class="updated">
			<p><?php echo $message; ?></p>
		</div>
		<?php endif; ?>
		<!--// all sliders //-->
		<table class="widefat" cellspacing="0">
			<thead>
				<tr>
					<th class="sortable <?php echo $order; ?>">
					<a href="<?php echo admin_url('admin.php?page=everslider&order=' . 
						($order === 'desc' ? 'asc' : 'desc')); ?>">
						<span><?php _e('Name', 'everslider'); ?></span>
						<span class="sorting-indicator"></span>
					</a></th>
					<th><?php _e('Shortcode', 'everslider'); ?></th>
					<th><?php _e('Actions', 'everslider'); ?></th>
					<th><?php _e('Preview', 'everslider'); ?></th>
				</tr>
			</thead>
			<tfoot>
				<tr>
					<th class="sortable <?php echo $order; ?>">
					<a href="<?php echo admin_url('admin.php?page=everslider&order=' . 
						($order === 'desc' ? 'asc' : 'desc')); ?>">
						<span><?php _e('Name', 'everslider'); ?></span>
						<span class="sorting-indicator"></span>
					</a></th>
					<th><?php _e('Shortcode', 'everslider'); ?></th>
					<th><?php _e('Actions', 'everslider'); ?></th>
					<th><?php _e('Preview', 'everslider'); ?></th>
				</tr>
			</tfoot>
			<tbody>
				<?php
					// sort all sliders
					$slider_list = everslider_get_slider_list();
					if( $order === 'desc' ) {
						rsort($slider_list);
					} else {
						sort($slider_list);
					}
					// loop sliders in database
					$row_count = 0;
					foreach( $slider_list as $slider_id ) {
						$row_count++;
						// link to open slider edit
						$edit_link = add_query_arg( array(
							'page' => 'everslider', 'action' => 'edit', 'slider_id' => $slider_id
						), admin_url('admin.php') );
						$edit_link = wp_nonce_url( $edit_link, __FILE__ );
						// link to remove slider					
						$remove_link = add_query_arg( array(
							'page' => 'everslider', 'action' => 'remove', 'slider_id' => $slider_id
						), admin_url('admin.php') );
						$remove_link = wp_nonce_url( $remove_link, __FILE__ );
						// link to duplicate slider
						$duplicate_link = add_query_arg( array(
							'page' => 'everslider', 'action' => 'duplicate', 'slider_id' => $slider_id
						), admin_url('admin.php') );
						$duplicate_link = wp_nonce_url( $duplicate_link, __FILE__ );
						// lint to open template dialog (Thickbox)
						$template_link = add_query_arg( array(
							'action' => 'everslider_template',
							'slider_id' => $slider_id,
							'TB_iframe' => 1,
							'width' => 600,
							'height' => 400
						), admin_url('admin-ajax.php') );
						// link to open slider preview (Thickbox)
						$preview_link = add_query_arg( array(
							'action' => 'everslider_preview', 
							'slider_id' => $slider_id,
							'TB_iframe' => 1,
							'width' => 700,
							'height' => 466
						), admin_url('admin-ajax.php') );
						// get options for current slider
						$slider_options = everslider_get_slider_options($slider_id);
						?>
							<tr class="<?php echo $row_count % 2 ? 'alternate' : ''; ?>">
								<td>
									<a href="<?php echo $edit_link; ?>" title="<?php _e('Edit Slider','everslider'); ?>">
										<strong class="es-slider-name"><?php 
											echo esc_html($slider_options['name']); 
										?></strong>
									</a>
								</td>
								<td><em class="es-shortcode">[everslider id="<?php echo $slider_options['id']; ?>"]</em></td>
								<td>
									<a class="button-primary thickbox" href="<?php echo $template_link; ?>" title="<?php 
										_e('Select Template','everslider'); ?>"><?php _e('Template','everslider'); ?></a>
									<a class="button-primary" href="<?php echo $edit_link; ?>" title="<?php 
										_e('Edit Slider','everslider'); ?>"><?php _e('Edit','everslider'); ?></a>
									<a class="button-secondary" href="<?php echo $duplicate_link; ?>" title="<?php 
										_e('Duplicate Slider','everslider'); ?>"><?php _e('Duplicate','everslider'); ?></a>
									<a class="es-remove-slider button-secondary" href="<?php echo $remove_link; ?>" title="<?php
										_e('Remove Slider','everslider'); ?>"><?php _e('Remove','everslider'); ?></a>
								</td>
								<td>
									<a class="thickbox" href="<?php echo $preview_link; ?>" title="<?php 
										echo esc_attr( sprintf(__('Preview for "%s"','everslider'), $slider_options['name']) ); 
									?>">
										<img src="<?php echo ES_PLUGIN_URL . 'includes/images/preview.png'; ?>" alt="Preview">
									</a>
								</td>
							</tr>
						<?php
					}
				?>
			</tbody>
		</table>
		<div class="es-post-type">
			<form method="get" action="<?php echo admin_url('admin.php'); ?>">
				<input type="hidden" name="page" value="everslider">
				<input type="hidden" name="action" value="post_type">
				<input type="checkbox" id="es_post_type" name="enable" value="1" <?php
					checked(get_option('everslider_post_type'), 1); ?>>
				<label for="es_post_type" title="<?php
					_e('A special post type to create carousel items','everslider'); ?>"><?php 
					_e('Enable <strong>Slides</strong>','everslider'); ?></label>
				<?php wp_nonce_field( __FILE__, '_wpnonce', false ); ?>
			</form>
		</div>
		<h4><?php _e('Create new slider', 'everslider'); ?></h4>
		<p><?php _e('To create new carousel slider enter name in field below and click <em>Add New</em>', 'everslider'); ?></p>
		<form id="new_slider_form" method="get" action="<?php echo admin_url('admin.php'); ?>">
			<input type="hidden" name="page" value="everslider">
			<input type="hidden" name="action" value="new">
			<input type="text" id="slider_name" autocomplete="off" placeholder="<?php 
				_e('Slider Name', 'everslider'); ?>" name="slider_name" value="">
			<input type="submit" class="button-primary" name="submit" value="<?php _e('Add New', 'everslider'); ?>">
			<?php wp_nonce_field( __FILE__, '_wpnonce', false ); ?>
		</form>
	</div>
<?php
}

add_action('everslider_options_page_default', 'everslider_default_page_content');


/**
* Display options page for "new" action
*/

function everslider_new_page_content( $args ) {
	extract( $args );
	
	// if form was submitted with empty slider name
	if( $slider_name == '' ) {
		$message = __('Please specify slider name and then press <em>Add New</em>','everslider');
		everslider_default_page_content( array('message' => $message));
		return;
	}
	
	// try to create slider
	$slider_id = everslider_create_new_slider($slider_name);
	
	// slider name is taken
	if ( !$slider_id ) {
		$message = __('Slider with this name already exists','everslider');
		everslider_default_page_content( array('message' => $message));
		return;
	}
	
	// ok, we got new slider
	$message = sprintf(__('Slider <strong>%s</strong> created','everslider'), esc_html($slider_name));
	everslider_default_page_content( array('message' => $message));
}

add_action('everslider_options_page_new', 'everslider_new_page_content');


/**
* Display options page for "edit" action
*/

function everslider_edit_page_content( $args ) {
	extract( $args );
	
	// get slider options
	$slider_options = everslider_get_slider_options( $slider_id );
	
	// if we try to edit unexisting slider, go back to default page - sliders list
	if ( !$slider_options ) {
		$message = __('Slider you want to edit does not exist', 'everslider');
		everslider_default_page_content( array('message' => $message ));
		return;
	}
	
	// show message if slider has been updated (URL has &settings-updated=true)
	if( isset($_GET['settings-updated']) ) {
		// slider list link
		$list_link = admin_url('admin.php?page=everslider');
		// link to open slider preview (Thickbox)
		$preview_link = add_query_arg( array(
			'action' => 'everslider_preview', 
			'slider_id' => $slider_id,
			'TB_iframe' => 1,
			'width' => 700,
			'height' => 466
		), admin_url('admin-ajax.php') );
		$message = sprintf( __('Your changes were saved. Click to <a class="thickbox" href="%s" title="Preview for %s">Preview</a> slider or return to <a href="%s">All Sliders</a>', 'everslider'), $preview_link, esc_attr('"'. $slider_options['name'].'"'), $list_link );
	}
	
	// output "Edit" page content
	?>
	<div class="wrap">
		<div id="everslider-icon" class="icon32">
			<img src="<?php echo ES_PLUGIN_URL . 'includes/images/icon.png'; ?>" alt="Icon">
		</div>
			<h2><?php echo sprintf( __('Editing <strong>%s</strong>', 'everslider'), esc_html($slider_options['name']) ); ?></h2>
			<br>
			<?php if ( isset($message) ) : ?>
			<div id="message" class="updated below-h2">
				<p><?php echo $message; ?></p>
			</div>
			<?php endif; ?>
			<div class="es-settings-wrap">
				<form method="post" action="options.php">
	  				<?php
	  					everslider_print_hidden_fields($slider_id);
	  					settings_fields('everslider_edit');
						do_settings_sections('everslider');
					?>
					
					<div class="es-sidebar-wrap metabox-holder">
						<?php echo do_action('everslider_edit_page_sidebar', array('slider_options' => $slider_options)); ?>
					</div>
					<br class="clear">
			
					<?php submit_button(); ?>
				</form>
			</div>
	</div>
	<?php
}

add_action('everslider_options_page_edit', 'everslider_edit_page_content');


/**
* Display options page for "post_type" action
*/

function everslider_post_type_page_content(){
	// check if slides post type was enabled or disabled
	if( get_option('everslider_post_type') ) {
		$message = sprintf(__('<a href="%s">Slides</a> post type was enabled!','everslider'), admin_url('edit.php?post_type=es-slide'));
	} else {
		$message = __('Slides post type was disabled','everslider');
	}
	// display default page content
	everslider_default_page_content(array('message' => $message));
}

add_action('everslider_options_page_post_type', 'everslider_post_type_page_content');


/**
* Display options page for "remove" action
*/

function everslider_remove_page_content( $args ) {
	extract($args);
	if( $slider_id ) {
		if( everslider_delete_slider_options($slider_id) ) {
			$message = __('Slider removed','everslider');
		} else {
			$message = __('Slider you want to remove does not exist','everslider');
		}
		everslider_default_page_content( array('message' => $message) );
	} else {
		wp_die( __('Missing slider_id or slider does not exist','everslider'), '', 'back_link=true' );
	}
}

add_action('everslider_options_page_remove', 'everslider_remove_page_content');


/**
* Display options page for "duplicate" action
*/

function everslider_duplicate_page_content( $args ) {
	extract($args);
	
	$slider_options = everslider_get_slider_options($slider_id);
	// if source slider does not exist
	if ( !$slider_options ) {
		wp_die( __('Missing slider_id or slider does not exist','everslider'), '', 'back_link=true' );
	}
	
	// if slider has been duplicated strip number from name
	if( preg_match('/\s+\(\d+\)$/', $slider_options['name']) ) {
		$slider_options['name'] = preg_replace('/\s+\(\d+\)$/', '', $slider_options['name']);
	} 
	
	// icrease $slider_copy_number to find good name for duplicated slider
	$slider_copy_number = 1;
	$slider_copy_name =  $slider_options['name'] . ' (' . $slider_copy_number . ')';
	while( in_array(everslider_get_slider_id($slider_copy_name), everslider_get_slider_list()) ) {
		$slider_copy_name =  $slider_options['name'] . ' (' . ++$slider_copy_number . ')';
	}
	
	//add duplicated slider
	$slider_options['name'] = $slider_copy_name;
	$slider_options['id'] = everslider_get_slider_id($slider_copy_name);
	everslider_add_slider_options( $slider_options['id'], $slider_options );
	
	// go to default page - sliders list
	$message = sprintf( __('Slider was duplicated as <strong>%s</strong>','everslider'), $slider_copy_name );
	everslider_default_page_content( array('message' => $message) );
}

add_action('everslider_options_page_duplicate', 'everslider_duplicate_page_content');


/**
* Function that generates simple preview for slider (Thickbox)
*/

function everslider_generate_slider_preview(){
	// check user permissions
	if( !current_user_can('manage_options') ) {
		wp_die( __('You do not have permissions to access this page','everslider') );
	}
	// get slider id
	$slider_id = isset($_GET['slider_id']) ? trim( $_GET['slider_id'] ) : '';
	?>
	<!DOCTYPE html>
	<html lang="en">
	<head>
		<meta http-equiv="content-type" content="<?php 
			get_option('html_type') . '; charset=' . get_option('blog_charset'); 
		?>">
		<title>Preview</title>
		<style type="text/css"> 
			/* Basic reset */
			html, body, div, span, applet, object, iframe,
			h1, h2, h3, h4, h5, h6, p, blockquote, pre,
			a, abbr, acronym, address, big, cite, code,
			del, dfn, em, font, img, ins, kbd, q, s, samp,
			small, strike, strong, sub, sup, tt, var,
			b, u, i, center,
			dl, dt, dd, ol, ul, li,
			fieldset, form, label, legend,
			table, caption, tbody, tfoot, thead, tr, th, td {
				margin: 0;
				padding: 0;
				border: 0;
				outline: 0;
				font-size: 100%;
				vertical-align: baseline;
				background: transparent;
			}
			body { font-size: 12px;  line-height: 1.4em; font-family: sans-serif; }
			a { color: #21759b; text-decoration: none; }
			a:hover { color: #d54e21; }
			p { margin-bottom: 0.7em; }
			.wrap { width: 660px; margin: 30px auto; }
			.everslider { float: none!important; margin: 0 auto!important; }
			#tb-close { position: absolute; bottom: 0; right: 0; font-weight: bold; line-height: 1em; padding: 10px; }
		</style>
		<script type="text/javascript" src="http://ajax.googleapis.com/ajax/libs/jquery/1.10.2/jquery.min.js"></script>
		<link rel="stylesheet" type="text/css" href="<?php echo ES_PLUGIN_URL . 'everslider.css'; ?>" media="screen">
		<link rel="stylesheet" type="text/css" href="<?php echo ES_PLUGIN_URL . 'templates.css'; ?>" media="screen">
		<script type="text/javascript" src="<?php echo ES_PLUGIN_URL . 'jquery.easing.1.3.js'; ?>"></script>
		<script type="text/javascript" src="<?php echo ES_PLUGIN_URL . 'jquery.mousewheel.js'; ?>"></script>
		<script type="text/javascript" src="<?php echo ES_PLUGIN_URL . 'jquery.everslider.min.js'; ?>"></script>
		<script type="text/javascript" src="<?php echo ES_PLUGIN_URL . 'templates.js'; ?>"></script>
		<script type="text/javascript">
			jQuery(function($){
				$('a').click(function(){ return false; });
				$('#tb-close').click(function(){ top.tb_remove(); });
			});
		</script>
		<?php everslider_print_custom_styles(); ?>
	</head>
	<body>
		<div class="wrap">
			<?php echo get_everslider($slider_id); ?>
		</div>
		<?php everslider_print_init_scripts(); ?>
		<a id="tb-close" href='#' title="<?php _e('Click to close','everslider'); ?>"><?php _e('Close','everslider'); ?></a>
	</body>
	</html>
	<?php
	die();
}

add_action('wp_ajax_everslider_preview', 'everslider_generate_slider_preview');


/**
* Function to generate slide template dialog (Thickbox)
*/

function everslider_generate_template_select() {
	// check user permissions
	if( !current_user_can('manage_options') ) {
		wp_die( __('You do not have permissions to access this page','everslider') );
	}
	// get slider id
	$slider_id = isset($_GET['slider_id']) ? trim( $_GET['slider_id'] ) : '';
	$slider_options = everslider_get_slider_options( $slider_id ); 
	// if there is no slider with this id
	if( !$slider_options ) 
		wp_die( __('No such slider','everslider') );
	?>
	<!DOCTYPE html>
	<html lang="en">
	<head>
		<meta http-equiv="content-type" content="<?php 
			get_option('html_type') . '; charset=' . get_option('blog_charset'); 
		?>">
		<title>Template</title>
		<style type="text/css">
			/* Basic reset */
			html, body, div, span, applet, object, iframe,
			h1, h2, h3, h4, h5, h6, p, blockquote, pre,
			a, abbr, acronym, address, big, cite, code,
			del, dfn, em, font, img, ins, kbd, q, s, samp,
			small, strike, strong, sub, sup, tt, var,
			b, u, i, center,
			dl, dt, dd, ol, ul, li,
			fieldset, form, label, legend,
			table, caption, tbody, tfoot, thead, tr, th, td {
				margin: 0;
				padding: 0;
				border: 0;
				outline: 0;
				font-size: 100%;
				vertical-align: baseline;
				background: transparent;
			}
			body { font-size: 12px;  line-height: 1.4em; font-family: sans-serif; }
			h4 { font-weight: bold; font-size: 100%; margin-top: 0.4em; }
			p { margin-bottom: 0.7em; }
			a { color: #21759b; text-decoration: none; }
			a:hover { color: #d54e21; }
			.wrap { width: 500px; margin: 15px auto; }
			.clear { clear: both; }
			label { float: left; margin: 20px; margin-bottom: 0; cursor: pointer; }
			label input { display: none; }
			label h4 { color: #666; }
			label img { width: 200px; height: 130px; border: 5px solid #eee; background: #dadada; -moz-transition: border 0.1s; -webkit-transition: border 0.1s; -o-transition: border 0.1s;  transition: border 0.1s; }
			.active-template img { border-color: #ddd; }
			.active-template h4 { color: #000; padding-left: 20px; background: url(<?php echo ES_PLUGIN_URL . 'includes/images/check.png'; ?>) no-repeat 0 0; }
			#tb-close { position: absolute; bottom: 0; right: 0; font-weight: bold; line-height: 1em; padding: 10px; }
		</style>
		<script type="text/javascript" src="http://ajax.googleapis.com/ajax/libs/jquery/1.10.2/jquery.min.js"></script>
		<script type="text/javascript">
			jQuery(function($){
				var form = $('form');
				$('input:checked').parent().addClass('active-template');
				$('input').click(function(){
					$('.active-template').removeClass('active-template');
					$(this).parent().addClass('active-template');
					$.post(form.attr('action'), form.serialize());
				});
				$('#tb-close').click(function(){ top.tb_remove(); });
			});
		</script>
	</head>
	<body>
		<div class="wrap">
			<form method="post" action="<?php echo admin_url('admin-ajax.php'); ?>">
				<input type="hidden" name="action" value="everslider_template_change">
				<input type="hidden" name="slider_id" value="<?php echo $slider_id; ?>">
				<label title="<?php echo esc_attr(__('Click to select "Default Template"','everslider')); ?>">
					<input type="radio" id="default-template" name="template" value="default" <?php 
						checked($slider_options['template'], 'default'); ?>>
					<img src="<?php echo ES_PLUGIN_URL . 'includes/images/templates/default.png'; ?>" alt="">
					<h4><?php _e('Default Template','everslider'); ?></h4>
				</label>
				<label title="<?php echo esc_attr(__('Click to select "Basic Template"','everslider')); ?>">
					<input type="radio" id="basic-template" name="template" value="basic" <?php
						checked($slider_options['template'], 'basic'); ?>>
					<img src="<?php echo ES_PLUGIN_URL . 'includes/images/templates/basic.png'; ?>" alt="">
					<h4><?php _e('Basic Template','everslider'); ?></h4>
				</label>
				<label title="<?php echo esc_attr(__('Click to select "Photo Template"','everslider')); ?>">
					<input type="radio" id="photo-template" name="template" value="photo" <?php
						checked($slider_options['template'], 'photo'); ?>>
					<img src="<?php echo ES_PLUGIN_URL . 'includes/images/templates/photo.png'; ?>" alt="">
					<h4><?php _e('Photo Template','everslider'); ?></h4>
				</label>
				<label title="<?php echo esc_attr(__('Click to select "Custom Template"','everslider')); ?>">
					<input type="radio" id="custom-template" name="template" value="custom" <?php
						checked($slider_options['template'], 'custom'); ?>>
					<img src="<?php echo ES_PLUGIN_URL . 'includes/images/templates/custom.png'; ?>" alt="">
					<h4><?php _e('Custom Template','everslider'); ?></h4>
				</label>
				<div class="clear"></div>
			</form>
		</div>
		<a id="tb-close" href='#' title="<?php _e('Click to close','everslider'); ?>"><?php _e('Done','everslider'); ?></a>
	</body>
	</html>
	<?php
	die();
}

add_action('wp_ajax_everslider_template', 'everslider_generate_template_select');


/**
* Function hooked to admin-ajax.php?action=template_change to switch template for slider
*/

function everslider_change_slider_template(){
	// check user permissions
	if( !current_user_can('manage_options') ) {
		wp_die( __('You do not have permissions to access this page','everslider') );
	}
	
	// get slider id and slider options
	$slider_id = isset($_REQUEST['slider_id']) ? trim($_REQUEST['slider_id']) : '';
	$slider_options = everslider_get_slider_options($slider_id);
	if( !$slider_options ) {
		wp_die( __('Missing slider_id or slider does not exist','everslider') );
	}
	
	// get new template 
	$template_name = isset($_REQUEST['template']) ? trim($_REQUEST['template']) : '';
	if( $template_name == '' ) {
		wp_die( __('Template name is missing in request','everslider') );
	}
	
	$slider_options['template'] = $template_name;
	everslider_update_slider_options($slider_id, $slider_options);
	die();
}

add_action('wp_ajax_everslider_template_change', 'everslider_change_slider_template');


/**
* Function that renames slider if its name was changed on "Edit" page. Is hooked to admin_init and 
* must run prior to registering settings in settings.php
*/

function everslider_edit_page_slider_rename(){
	// check if query has 'orig_slider_id' param
	if( !isset($_GET['orig_slider_id']) ) 
		return;
		
	// get slider options
	$slider_id = isset($_GET['orig_slider_id']) ? trim($_GET['orig_slider_id']) : '';
	$slider_options = everslider_get_slider_options($slider_id);
	if( !$slider_options) return;
	
	// save options for new slider
	$slider_options['id'] = everslider_get_slider_id($slider_options['name']);
	$result = everslider_add_slider_options($slider_options['id'], $slider_options);
	if( $result ) {
		everslider_delete_slider_options($slider_id);
	} 
}

add_action('admin_init', 'everslider_edit_page_slider_rename', 9);


/**
* Function that enables or disables slides post type before all other "init" hooks will run
*/

function everslider_enable_slides_post_type(){
	if( isset($_GET['action']) && trim($_GET['action']) === 'post_type' ) {
		$enable = isset($_GET['enable']) ? 1 : 0;
		update_option('everslider_post_type', $enable);
	}
}

add_action('init', 'everslider_enable_slides_post_type', 9);

?>