<?php

/********************************************************************************
	
	Register settings for Everslider carousel sliders
	that will be displayed on "Edit" page
			
********************************************************************************/


/**
* Function to register settings for slider by slider id. Function is hooked to admin_init and
* checkes $_REQUEST to figure what slider settings to update
*/

function everslider_register_slider_settings(){

	// check permissions
	if( !current_user_can('manage_options') ) return;
	
	// take 'action' and 'slider_id' either from URL or from form submit
	$action = isset($_REQUEST['action']) ? $_REQUEST['action'] : '';
	$slider_id = isset($_REQUEST['slider_id']) ? $_REQUEST['slider_id'] : '';

	// return if either action or slider id are empty
	if( $action == '' || $slider_id == '' ) return; 
		
	// return if action is not 'edit' or 'update' (form submit) 
	if( $action != 'edit' && $action != 'update' ) return;

	// get options for slider
	$slider_options = everslider_get_slider_options( $slider_id );
	
	// do nothing if slider does not exist
	if( !$slider_options ) return; 
		
	// "General" setting section and related fields
	add_settings_section('everslider_general_settings', __('Slider Options','everslider'), 'everslider_general_section_description' , 'everslider');
	add_settings_field('name', __('Name','everslider'), 'everslider_print_name_field', 'everslider', 'everslider_general_settings', array('slider_options' => $slider_options) );
	add_settings_field('source', __('Source','everslider'), 'everslider_print_source_field', 'everslider', 'everslider_general_settings', array('slider_options' => $slider_options) );
	add_settings_field('groups', __('Groups','everslider'), 'everslider_print_groups_field', 'everslider', 'everslider_general_settings', array('slider_options' => $slider_options) );
	add_settings_field('tags', __('Tags','everslider'), 'everslider_print_tags_field', 'everslider', 'everslider_general_settings', array('slider_options' => $slider_options) );
	add_settings_field('orderby', __('Order By','everslider'), 'everslider_print_orderby_field', 'everslider', 'everslider_general_settings', array('slider_options' => $slider_options) );
	add_settings_field('order', __('Order','everslider'), 'everslider_print_order_field', 'everslider', 'everslider_general_settings', array('slider_options' => $slider_options) );
	add_settings_field('limit', __('Limit','everslider'), 'everslider_print_limit_field', 'everslider', 'everslider_general_settings', array('slider_options' => $slider_options) );
	add_settings_field('align', __('Alignment', 'everslider'), 'everslider_print_align_field', 'everslider', 'everslider_general_settings', array('slider_options' => $slider_options) );

	// display two field below for "custom" template only
	if( isset( $slider_options['template'] ) && $slider_options['template'] === 'custom') {
		add_settings_field('cst', __('Custom Slide Template','everslider'), 'everslider_print_custom_slide_template_field', 'everslider', 'everslider_general_settings', array('slider_options' => $slider_options) );
		add_settings_field('cts', __('Custom Template Style','everslider'), 'everslider_print_custom_template_style_field', 'everslider', 'everslider_general_settings', array('slider_options' => $slider_options) );
	}
	
	// register setting
	register_setting('everslider_edit', ES_SLIDER_PREFIX . $slider_id, 'everslider_sanitize_slider_settings');
}

add_action( 'admin_init', 'everslider_register_slider_settings' );


/**
* Function to display descriptive text for "general" settings section
*/

function everslider_general_section_description(){
	//echo '<p></p>';
}

/**
* Function to sanitize slider options before they will be saved to database
*/

function everslider_sanitize_slider_settings( $slider_options ) {
	// general options
	$slider_options['limit'] = $slider_options['limit'] ? (int) $slider_options['limit'] : -1;
	// plugin settings
	$ps = $slider_options['plugin_settings'];
	$slider_options['plugin_settings'] = array(
		'mode' => $ps['mode'],
		'effect' => $ps['effect'],
		'useCSS' => isset($ps['useCSS']) ? true : false,
		'itemWidth' => empty($ps['itemWidth']) ? false : $ps['itemWidth'],
		'itemHeight' => empty($ps['itemHeight']) ? false : $ps['itemHeight'],
		'itemMargin' => empty($ps['itemMargin']) ? false : $ps['itemMargin'],
		'itemKeepRatio' => isset($ps['itemKeepRatio']) ? true : false,
		'maxWidth' => $ps['maxWidth'],
		'maxVisible' => (int) $ps['maxVisible'],
		'moveSlides' => $ps['moveSlides'] == 'auto' ? 'auto' : max(1, (int) $ps['moveSlides']),
		'slideDelay' => (int) $ps['slideDelay'],
		'slideDuration' => (int) $ps['slideDuration'],
		'slideEasing' => $ps['slideEasing'],
		'fadeDelay' => (int) $ps['slideDelay'],
		'fadeDuration' => (int) $ps['slideDuration'],
		'fadeEasing' => $ps['slideEasing'],
		'fitDuration' => (int) $ps['fitDuration'],
		'fitEasing' => $ps['fitEasing'],
		'syncHeight' => isset($ps['syncHeight']) ? true : false,
		'syncHeightDuration' => (int) $ps['syncHeightDuration'],
		'syncHeightEasing' => $ps['syncHeightEasing'],
		'navigation' => isset($ps['navigation']) ? true : false,
		'pagination' => isset($ps['pagination']) ? true : false,
		'touchSwipe' => isset($ps['touchSwipe']) ? true : false,
		'swipeThreshold' => (int) $ps['swipeThreshold'],
		'swipePage' => isset($ps['swipePage']) ? true : false,
		'mouseWheel' => isset($ps['mouseWheel']) ? true : false,
		'keyboard' => isset($ps['keyboard']) ? true : false,
		'ticker' => isset($ps['ticker']) ? true : false,
		'tickerAutoStart' => isset($ps['tickerAutoStart']) ? true : false,
		'tickerTimeout' => (int) $ps['tickerTimeout'],
		'tickerHover' => isset($ps['tickerHover']) ? true : false
	);
	// return sanitized settings 
	return $slider_options;
}


/**
* Function to add hidden input elements to the form. These inputs are to keep slider 
* options array consistent when it's saved to the database
*/

function everslider_print_hidden_fields( $slider_id ) {
	$slider_options = everslider_get_slider_options( $slider_id );
	?>
	<input type="hidden" name="slider_id" value="<?php echo $slider_id; ?>">
	<input type="hidden" id="<?php echo $slider_id; ?>_id" name="<?php 
		echo ES_SLIDER_PREFIX . $slider_id . '[id]'; ?>" value="<?php echo $slider_options['id']; ?>">
	<input type="hidden" id="<?php echo $slider_id; ?>_template" name="<?php 
		echo ES_SLIDER_PREFIX . $slider_id . '[template]'; ?>" value="<?php echo $slider_options['template']; ?>">	
	<?php if( $slider_options['template'] !== 'custom' ) : ?>
	<input type="hidden" id="<?php echo $slider_id; ?>_custom_slide_template" name="<?php 
		echo ES_SLIDER_PREFIX . $slider_id . '[custom_slide_template]'; ?>" value="<?php 
		echo esc_attr($slider_options['custom_slide_template']); ?>">
	<input type="hidden" id="<?php echo $slider_id; ?>_custom_template_style" name="<?php 
		echo ES_SLIDER_PREFIX . $slider_id . '[custom_template_style]'; ?>" value="<?php 
		echo esc_attr($slider_options['custom_template_style']); ?>">
	<?php endif; ?>
	<?php
}


/**
* Functions to print settings "name" field
*/

function everslider_print_name_field( $args ){
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[name]';
	?>
		<input type="text" id="<?php echo $slider_id; ?>_name" class="es-slider-name" name="<?php 
			echo $option_name; ?>" value="<?php echo esc_attr($slider_options['name']);	?>">
	<?php
}


/**
* Function to print settings "source" field
*/

function everslider_print_source_field( $args ){
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[source]';
	// get all public post types
	$skip_post_types = array('attachment'); // skip post types
	$post_types = get_post_types(array('public' => true), 'objects');
	foreach( $post_types as $post_type) {
		if( !in_array($post_type->name, $skip_post_types) ) {
	?>
		<label>
			<input type="radio" class="es-source-input" id="<?php echo $slider_id . '_' . $post_type->name; 
				?>_source" name="<?php echo $option_name; ?>" value="<?php 
				echo $post_type->name; ?>" <?php checked( $slider_options['source'], $post_type->name ); ?>>
				<span><?php echo $post_type->label; ?>&nbsp;&nbsp;</span>
		</label>
	<?php
		}
	}
	?>
		<p class="description"><?php _e('Select post type for carousel slides','everslider'); ?></p>
	<?php
}


/**
* Function to print settings "groups" field
*/

function everslider_print_groups_field( $args ){
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[groups][]';
	// get terms for hierarchical taxonomies for post type (source)
	$post_type = $slider_options['source'];
	$taxonomies = get_object_taxonomies($post_type);
	?>
		<select multiple="multiple" size="5" id="<?php echo $slider_id; ?>_groups" name="<?php echo $option_name; ?>">
	<?php
	foreach( $taxonomies as $taxonomy ) {
			if( is_taxonomy_hierarchical($taxonomy) ) {
				$taxonomy_object = get_taxonomy($taxonomy);
				echo '<optgroup label="' . $taxonomy_object->label . '">';
				$terms = get_terms($taxonomy);
				foreach ($terms as $term) {
					$opt_value =  $taxonomy . '&'. $term->slug;
					$opt_selected = in_array($opt_value, (array) $slider_options['groups']) ? ' selected' : '';
					?>
					<option value="<?php echo $opt_value; ?>"<?php 
						echo $opt_selected; ?>><?php echo $term->name; ?>
					</option>	
				<?php
				} // terms
				echo '</optgroup>';
			}
	} // taxonomies
	?>
	</select>
	<p class="description"><?php 
		_e('Include slides only from selected groups (multiple groups can be selected)','everslider'); 
	?></p>
	<?php
}


/**
* Function to print settings "tags" field
*/

function everslider_print_tags_field( $args ){
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[tags][]';
	// get terms for non-hierarchical taxonomies for post type (source)
	$post_type = $slider_options['source'];
	$taxonomies = get_object_taxonomies($post_type);
	?>
		<select multiple="multiple" size="5" id="<?php echo $slider_id; ?>_tags" name="<?php echo $option_name; ?>">
	<?php
	foreach( $taxonomies as $taxonomy ) {
			if( !is_taxonomy_hierarchical($taxonomy) ) {
				$taxonomy_object = get_taxonomy($taxonomy);
				$terms = get_terms($taxonomy);
				if( empty($terms) ) // don't show empty taxonomies
					continue;
				echo '<optgroup label="' . $taxonomy_object->label . '">';
				foreach ($terms as $term) {
					$opt_value =  $taxonomy . '&'. $term->slug;
					$opt_selected = in_array($opt_value, (array) $slider_options['tags']) ? ' selected' : '';
					?>
					<option value="<?php echo $opt_value; ?>"<?php 
						echo $opt_selected; ?>><?php echo $term->name; ?>
					</option>	
				<?php
				} // terms
				echo '</optgroup>';
			}
	} // taxonomies
	?>
	</select>
	<p class="description"><?php 
		_e('Include slides only with selected tags (multiple tags can be selected)','everslider'); 
	?></p>
	<?php
}


/**
* Function to print settings "orderby" field
*/

function everslider_print_orderby_field( $args ){
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[orderby]';
	?>
		<select id="<?php echo $slider_id; ?>_orderby" name="<?php echo $option_name; ?>">
			<option value="none" <?php selected( $slider_options['orderby'], 'none'); ?>> - </option>
			<option value="title" <?php selected( $slider_options['orderby'], 'title'); ?>>Title</option>
			<option value="author" <?php selected( $slider_options['orderby'], 'author'); ?>>Author</option>
			<option value="date" <?php selected( $slider_options['orderby'], 'date'); ?>>Date</option>
			<option value="comment_count" <?php selected( $slider_options['orderby'], 'comment_count'); ?>>Popularity</option>
			<option value="rand" <?php selected( $slider_options['orderby'], 'rand'); ?>>Random</option>
		</select>
	<?php
}


/**
* Function to print settings "order" field
*/

function everslider_print_order_field( $args ){
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[order]';
	?>
		<label><input type="radio" id="<?php echo $slider_id; ?>_order_asc" name="<?php echo $option_name; ?>" value="ASC" <?php 
			checked($slider_options['order'], 'ASC'); ?>> <span>ASC&nbsp;</span>
		</label>
		<label>
			<input type="radio" id="<?php echo $slider_id; ?>_order_desc" name="<?php echo $option_name; ?>" value="DESC" <?php 
			checked($slider_options['order'], 'DESC'); ?>> <span>DESC&nbsp;</span>
		</label>
	<?php
}


/**
* Function to print settings "limit" field
*/

function everslider_print_limit_field( $args ){
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[limit]';
	?>
		<input type="number" id="<?php echo $slider_id; ?>_limit" name="<?php 
			echo $option_name; ?>" step="1" min="1" value="<?php echo $slider_options['limit']; ?>">
		<p class="description"><?php _e('Add this number of slides to the carousel or -1 for unlimited','everslider'); ?></p>
	<?php
}


/**
* Function to print settings "align" field
*/

function everslider_print_align_field( $args ){
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[align]';
	?>
	<label>
		<input type="radio" id="<?php echo $slider_id; ?>_align_none" name="<?php 
		echo $option_name; ?>" value="none" <?php checked($slider_options['align'],'none'); ?>> <span>None&nbsp;</span>
	</label>
	<label>
		<input type="radio" id="<?php echo $slider_id; ?>_align_left" name="<?php 
		echo $option_name; ?>" value="left" <?php checked($slider_options['align'],'left'); ?>> <span>Left&nbsp;</span>
	</label>
	<label>
		<input type="radio" id="<?php echo $slider_id; ?>_align_center" name="<?php 
		echo $option_name; ?>" value="center" <?php checked($slider_options['align'],'center'); ?>> <span>Center&nbsp;</span>
	</label>
	<label>
		<input type="radio" id="<?php echo $slider_id; ?>_align_right" name="<?php 
		echo $option_name; ?>" value="right" <?php checked($slider_options['align'],'right'); ?>> <span>Right&nbsp;</span>
	</label>
	<p class="description"><?php _e('Set carousel alignment on page','everslider'); ?></p>
	<?php
}


/**
* Function to print settings "custom_slide_template" field
*/

function everslider_print_custom_slide_template_field( $args ){
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[custom_slide_template]';
	?>
	<textarea cols="70" rows="8" id="<?php echo $slider_id; ?>_cts" name="<?php echo $option_name; ?>"><?php 
		echo esc_textarea( $slider_options['custom_slide_template'] ); 
	?></textarea>
	<p class="description"><?php _e('HTML markup for carousel slide. You can insert following placeholders into the markup to get actual info - %id%, %author%, %content%, %title%, %excerpt%, %comments_number%, %comments_link%, %date%, %date_modified%, %permalink%, %slug%, %image%, %image_src%, %image_title%, %image_alt%, %image_caption%, %image_description%, %groups%, %tags%, %{taxonomy}% and %meta_{key}%', 'everslider'); ?></p>
	<?php
}


/**
* Function to print settings "custom_template_style" field
*/

function everslider_print_custom_template_style_field( $args ){
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[custom_template_style]';
	?>
	<textarea cols="70" rows="8" id="<?php echo $slider_id; ?>_cts" name="<?php echo $option_name; ?>"><?php 
		echo esc_textarea($slider_options['custom_template_style']); 
	?></textarea>
	<p class="description"><?php _e('Custom styles for carousel slide. Every CSS rule that you will enter should start with %slider% placeholder to reference specific slider on page. Also to get URL to plugin directory you may use %plugin_url% placeholder','everslider'); ?></p>
	<?php
}


/**
* Sidebar. Function to print "Mode" postbox fields
*/

function everslider_print_slider_mode_sidebar_fields( $args ) {
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$slider_plugin_settings = $slider_options['plugin_settings'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[plugin_settings]';
	?>
	<div class="postbox">
		<h3><span><?php _e('Mode','everslider'); ?></span></h3>
		<div class="inside">
			<label><input type="radio" id="<?php echo $slider_id;?>_mode_normal" name="<?php 
				echo $option_name . '[mode]';
			?>" value="normal" <?php checked($slider_plugin_settings['mode'], 'normal'); ?>>
			<span>Normal&nbsp;</span>
			</label>
			<label><input type="radio" id="<?php echo $slider_id;?>_mode_circular" name="<?php 
				echo $option_name . '[mode]';
			?>" value="circular" <?php checked($slider_plugin_settings['mode'], 'circular'); ?>>
			<span>Circular&nbsp;</span>
			</label>
			<label><input type="radio" id="<?php echo $slider_id;?>_mode_carousel" name="<?php 
				echo $option_name . '[mode]';
			?>" value="carousel" <?php checked($slider_plugin_settings['mode'], 'carousel'); ?>>
			<span>Carousel&nbsp;</span>
			</label>
			<p class="description"><?php _e('Select mode for Everslider carousel','everslider');?></p>
		</div>
	</div>
	<?php
}

add_action('everslider_edit_page_sidebar', 'everslider_print_slider_mode_sidebar_fields');


/**
* Sidebar. Function to print "Slide Size" postbox fields
*/

function everslider_print_item_size_sidebar_fields( $args ) {
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$slider_plugin_settings = $slider_options['plugin_settings'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[plugin_settings]';
	?>
	<div class="postbox closed">
		<h3><span><?php _e('Slide Size','everslider'); ?></span></h3>
		<div class="inside">
			<label>
				<input type="text" id="<?php echo $slider_id;?>_item_width" name="<?php
					echo $option_name . '[itemWidth]';
				?>" value="<?php echo esc_attr($slider_plugin_settings['itemWidth']); ?>">
				<span><?php _e('px, width', 'everslider'); ?></span>
			</label>
			<label>
				<input type="text" id="<?php echo $slider_id;?>_item_height" name="<?php
					echo $option_name . '[itemHeight]';
				?>" value="<?php echo esc_attr($slider_plugin_settings['itemHeight']); ?>">
				<span><?php _e('px, height', 'everslider'); ?></span>
			</label>
			<label>
				<input type="text" id="<?php echo $slider_id;?>_item_margin" name="<?php
					echo $option_name . '[itemMargin]';
				?>" value="<?php echo esc_attr($slider_plugin_settings['itemMargin']); ?>">
				<span><?php _e('px, margin', 'everslider'); ?></span>
			</label>
			<label class="es-m4">
				<input type="checkbox" id="<?php echo $slider_id; ?>_item_keep_ratio" name="<?php 
					echo $option_name . '[itemKeepRatio]'; ?>" value="1" <?php
					checked($slider_plugin_settings['itemKeepRatio'], true); ?>>
				<span><?php _e('Preserve sides ratio','everslider'); ?></span>
			</label>
			<p class="description"><?php _e('Here you can set size and margin for items in carousel or leave empty for default values','everslider');?></p>
		</div>
	</div>
	<?php
}

add_action('everslider_edit_page_sidebar', 'everslider_print_item_size_sidebar_fields');


/**
* Sidebar. Function to print "Animations" postbox fields
*/

function everslider_print_animations_sidebar_fields( $args ) {
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$slider_plugin_settings = $slider_options['plugin_settings'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[plugin_settings]';
	?>
	<div class="postbox closed">
		<h3><span><?php _e('Animations','everslider'); ?></span></h3>
		<div class="inside">
			<label>
				<input type="checkbox" id="<?php echo $slider_id; ?>_use_css" name="<?php
					echo $option_name . '[useCSS]';
				?>" value="1" <?php checked($slider_plugin_settings['useCSS'], true); ?>>
				<span><?php _e('Use CSS3 transitions','everslider'); ?></span>
			</label>
			<h4><?php _e('Animation Effect','everslider'); ?></h4>
			<label><input type="radio" id="<?php echo $slider_id; ?>_effect_slide" name="<?php 
				echo $option_name . '[effect]'; ?>" value="slide" <?php 
				checked($slider_plugin_settings['effect'], 'slide'); ?>>
				<span>Slide&nbsp;</span></label>
			<label><input type="radio" id="<?php echo $slider_id; ?>_effect_fade" name="<?php 
				echo $option_name . '[effect]'; ?>" value="fade" <?php 
				checked($slider_plugin_settings['effect'], 'fade'); ?>>
				<span>Fade&nbsp;</span></label>
			<p class="description"><?php _e('Select animation effect (for "Fade" effect consider setting delay in "Main Transition" to any positive value)','everslider'); ?></p>
			<h4><?php _e('Main Transition','everslider'); ?></h4>
			<label>
				<input type="number" min="0" id="<?php echo $slider_id; ?>_slide_delay" name="<?php 
					echo $option_name . '[slideDelay]'; ?>" value="<?php 
					echo esc_attr($slider_plugin_settings['slideDelay']); ?>">
				<span><?php _e('ms, delay','everslider'); ?></span>
			</label>
			<label>
				<input type="number" min="10" id="<?php echo $slider_id; ?>_slide_duration" name="<?php
					echo $option_name . '[slideDuration]'; ?>" value="<?php 
					echo esc_attr($slider_plugin_settings['slideDuration']); ?>">
				<span><?php _e('ms, duration','everslider'); ?></span>
			</label><br>
			<label>
				<select id="<?php echo $slider_id; ?>_slide_easing" name="<?php echo $option_name . '[slideEasing]'; ?>">
					<?php 
						$easing_list = array(
							'swing', 'linear', 
							'easeInQuad', 'easeOutQuad', 'easeInOutQuad',
							'easeInCubic', 'easeOutCubic', 'easeInOutCubic',
							'easeInQuart', 'easeOutQuart', 'easeInOutQuart', 
							'easeInQuint', 'easeOutQuint', 'easeInOutQuint', 
							'easeInSine', 'easeOutSine', 'easeInOutSine', 
							'easeInExpo', 'easeOutExpo', 'easeInOutExpo', 
							'easeInCirc', 'easeOutCirc', 'easeInOutCirc', 
							'easeInBack', 'easeOutBack', 'easeInOutBack'
						);
						foreach( $easing_list as $easing ) {
							echo '<option value="' . $easing . '" ' . 
							selected($slider_plugin_settings['slideEasing'], $easing, false) . '>' . 
							$easing . '</option>';
						}
					?>
				</select>
				<span><?php echo _e('easing','everslider'); ?></span>
			</label>
			<h4><?php _e('Resize Transition','everslider'); ?></h4>
			<label>
				<input type="number" min="10" id="<?php echo $slider_id; ?>_fit_duration" name="<?php
					echo $option_name . '[fitDuration]'; ?>" value="<?php 
					echo esc_attr($slider_plugin_settings['fitDuration']); ?>">
				<span><?php _e('ms, duration','everslider'); ?></span>
			</label>
			<label>
				<select id="<?php echo $slider_id; ?>_fit_easing" name="<?php echo $option_name . '[fitEasing]'; ?>">
					<?php 
						$easing_list = array(
							'swing', 'linear', 
							'easeInQuad', 'easeOutQuad', 'easeInOutQuad',
							'easeInCubic', 'easeOutCubic', 'easeInOutCubic',
							'easeInQuart', 'easeOutQuart', 'easeInOutQuart', 
							'easeInQuint', 'easeOutQuint', 'easeInOutQuint', 
							'easeInSine', 'easeOutSine', 'easeInOutSine', 
							'easeInExpo', 'easeOutExpo', 'easeInOutExpo', 
							'easeInCirc', 'easeOutCirc', 'easeInOutCirc', 
							'easeInBack', 'easeOutBack', 'easeInOutBack'
						);
						foreach( $easing_list as $easing ) {
							echo '<option value="' . $easing . '" ' . 
							selected($slider_plugin_settings['fitEasing'], $easing, false) . '>' . 
							$easing . '</option>';
						}
					?>
				</select>
				<span><?php echo _e('easing','everslider'); ?></span>
			</label>
			<h4><?php _e('SyncHeight Transition','everslider'); ?></h4>
			<label class="es-m4">
				<input type="checkbox" id="<?php echo $slider_id; ?>_sync_height" name="<?php 
					echo $option_name . '[syncHeight]'; ?>" <?php
					checked($slider_plugin_settings['syncHeight'], true); ?>>
				<span><?php _e('Allow carousel height sync','everslider'); ?></span>
			</label>
			<label>
				<input type="number" min="10" id="<?php echo $slider_id; ?>_sync_hight_duration" name="<?php
					echo $option_name . '[syncHeightDuration]'; ?>" value="<?php 
					echo esc_attr($slider_plugin_settings['syncHeightDuration']); ?>">
				<span><?php _e('ms, duration','everslider'); ?></span>
			</label><br>
			<label>
				<select id="<?php echo $slider_id; ?>_sync_height_easing" name="<?php 
					echo $option_name . '[syncHeightEasing]'; ?>">
					<?php 
						$easing_list = array(
							'swing', 'linear', 
							'easeInQuad', 'easeOutQuad', 'easeInOutQuad',
							'easeInCubic', 'easeOutCubic', 'easeInOutCubic',
							'easeInQuart', 'easeOutQuart', 'easeInOutQuart', 
							'easeInQuint', 'easeOutQuint', 'easeInOutQuint', 
							'easeInSine', 'easeOutSine', 'easeInOutSine', 
							'easeInExpo', 'easeOutExpo', 'easeInOutExpo', 
							'easeInCirc', 'easeOutCirc', 'easeInOutCirc', 
							'easeInBack', 'easeOutBack', 'easeInOutBack'
						);
						foreach( $easing_list as $easing ) {
							echo '<option value="' . $easing . '" ' . 
							selected($slider_plugin_settings['syncHeightEasing'], $easing, false) . '>' . 
							$easing . '</option>';
						}
					?>
				</select>
				<span><?php echo _e('easing','everslider'); ?></span>
			</label>
			<p class="description"><?php _e('Automatically adjust carousel height accordingly to the heights of visible slides (note that "Slide Size -> Preserve sides ratio" checkbox must be unchecked!)','everslider'); ?></p>
		</div>
	</div>
	<?php
}

add_action('everslider_edit_page_sidebar', 'everslider_print_animations_sidebar_fields');


/**
* Sidebar. Function to print "Controls" postbox fields
*/

function everslider_print_controls_sidebar_fields( $args ) {
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$slider_plugin_settings = $slider_options['plugin_settings'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[plugin_settings]';
	?>
	<div class="postbox closed">
		<h3><span><?php _e('Controls','everslider'); ?></span></h3>
		<div class="inside">
			<label>
				<input type="checkbox" id="<?php echo $slider_id; ?>_navigation" name="<?php 
				echo $option_name . '[navigation]'; ?>" value="1" <?php 
				checked($slider_plugin_settings['navigation'], true); ?>>
				<span><?php _e('Show navigation controls','everslider'); ?></span>
			</label><br>
			<label>
				<input type="checkbox" id="<?php echo $slider_id; ?>_pagination" name="<?php 
				echo $option_name . '[pagination]'; ?>" value="1" <?php 
				checked($slider_plugin_settings['pagination'], true); ?>>
				<span><?php _e('Show pagination controls','everslider'); ?></span>
			</label><br>
			<label>
				<input type="checkbox" id="<?php echo $slider_id; ?>_mousewheel" name="<?php 
				echo $option_name . '[mouseWheel]'; ?>" value="1" <?php 
				checked($slider_plugin_settings['mouseWheel'], true); ?>>
				<span><?php _e('Enable mousewheel','everslider'); ?></span>
			</label><br>
			<label>
				<input type="checkbox" id="<?php echo $slider_id; ?>_keyboard" name="<?php 
				echo $option_name . '[keyboard]'; ?>" value="1" <?php 
				checked($slider_plugin_settings['keyboard'], true); ?>>
				<span><?php _e('Enable keyboard','everslider'); ?></span>
			</label><br>
			<label>
				<input type="checkbox" id="<?php echo $slider_id; ?>_touch_swipe" name="<?php 
				echo $option_name . '[touchSwipe]'; ?>" value="1" <?php 
				checked($slider_plugin_settings['touchSwipe'], true); ?>>
				<span><?php _e('Enable touchswipe','everslider'); ?></span>
			</label><br>
			<label>
				<input type="checkbox" id="<?php echo $slider_id; ?>_swipe_page" name="<?php 
				echo $option_name . '[swipePage]'; ?>" value="1" <?php 
				checked($slider_plugin_settings['swipePage'], true); ?>>
				<span><?php _e('Allow vertical swipe to scroll page','everslider'); ?></span>
			</label><br>
			<label class="es-m4">
				<input type="text" id="<?php echo $slider_id; ?>_swipe_threshold" name="<?php echo $option_name . '[swipeThreshold]';?>" value="<?php echo $slider_plugin_settings['swipeThreshold']; ?>">
				<span><?php _e('px, swipe threshold','everslider'); ?></span>
			</label>
		</div>
	</div>
	<?php
}

add_action('everslider_edit_page_sidebar', 'everslider_print_controls_sidebar_fields');


/**
* Sidebar. Function to print "Autoplay" postbox fields
*/

function everslider_print_autoplay_sidebar_fields( $args ) {
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$slider_plugin_settings = $slider_options['plugin_settings'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[plugin_settings]';
	?>
	<div class="postbox closed">
		<h3><span><?php _e('Autoplay','everslider'); ?></span></h3>
		<div class="inside">
			<label>
				<input type="checkbox" id="<?php echo $slider_id; ?>_ticker" name="<?php 
				echo $option_name . '[ticker]'; ?>" value="1" <?php 
				checked($slider_plugin_settings['ticker'], true); ?>>
				<span><?php _e('Enable autoplay','everslider'); ?></span>
			</label><br>
			<label>
				<input type="checkbox" id="<?php echo $slider_id; ?>_ticker_auto_start" name="<?php 
				echo $option_name . '[tickerAutoStart]'; ?>" value="1" <?php 
				checked($slider_plugin_settings['tickerAutoStart'], true); ?>>
				<span><?php _e('Start autoplay automatically','everslider'); ?></span>
			</label><br>
			<label>
				<input type="checkbox" id="<?php echo $slider_id; ?>_ticker_hover" name="<?php 
				echo $option_name . '[tickerHover]'; ?>" value="1" <?php 
				checked($slider_plugin_settings['tickerHover'], true); ?>>
				<span><?php _e('Pause autoplay on mousehover','everslider'); ?></span>
			</label>
			<label class="es-m4">
				<input type="number" min="10" id="<?php echo $slider_id; ?>_ticker_timeout" name="<?php 
				echo $option_name . '[tickerTimeout]'; ?>" value="<?php 
				echo esc_attr($slider_plugin_settings['tickerTimeout']); ?>">
				<span><?php _e('ms, autoplay timeout','everslider'); ?></span>
			</label>
			<p class="description"><?php 
				_e('Autoplay will only work if carousel is in "Circular" or "Carousel" mode','everslider'); 
			?></p>
		</div>
	</div>
	<?php
}

add_action('everslider_edit_page_sidebar', 'everslider_print_autoplay_sidebar_fields');


/**
* Sidebar. Function to print "Misc" postbox field
*/

function everslider_print_misc_sidebar_fields( $args ) {
	$slider_options = $args['slider_options'];
	$slider_id = $slider_options['id'];
	$slider_plugin_settings = $slider_options['plugin_settings'];
	$option_name =  ES_SLIDER_PREFIX . $slider_options['id'] . '[plugin_settings]';
	?>
	<div class="postbox closed">
		<h3><span><?php _e('Misc','everslider'); ?></span></h3>
		<div class="inside">
			<input type="number" min="1" id="<?php echo $slider_id; ?>_move_slides" name="<?php 
				echo $option_name . '[moveSlides]'; ?>" value="<?php 
				echo esc_attr($slider_plugin_settings['moveSlides']); ?>">
			<label>
				<input type="checkbox" id="<?php echo $slider_id; ?>_auto_move_slides" name="<?php 
				echo $option_name . '[moveSlides]'; ?>" value="auto" <?php 
				checked($slider_plugin_settings['moveSlides'], 'auto'); ?>>
				<span><?php _e('Scroll visible','everslider'); ?></span>
			</label>
			<p class="description"><?php _e('Specify how many items carousel should scroll at once, or select <em>"Scroll visible"</em> checkbox to make this number equal to visible items','everslider'); ?></p>
			<label>
				<input type="text" id="<?php echo $slider_id; ?>_max_width" name="<?php 
				echo $option_name . '[maxWidth]'; ?>" value="<?php 
				echo esc_attr($slider_plugin_settings['maxWidth']); ?>">
				<span>px or %</span>
			</label>
			<p class="description"><?php _e('Set carousel width to fixed pixel value or use % to make it responsive','everslider'); ?></p>
			<label>
				<input type="number" min="0" id="<?php echo $slider_id; ?>_max_visible" name="<?php 
					echo $option_name . '[maxVisible]'; ?>" value="<?php 
					echo esc_attr($slider_plugin_settings['maxVisible']); ?>">
			</label>
			<p class="description"><?php _e('Limit carousel width to make this number of items visible, or set 0 for no limit','everslider'); ?></p>
		</div>
	</div>
	<?php
}

add_action('everslider_edit_page_sidebar', 'everslider_print_misc_sidebar_fields');

?>