<?php
/**
 * Helper functions used in the WP Post Modules plugin
 */

// Return parameters for BFI image resize
if ( ! function_exists( 'wppm_image_resize' ) ) :
	function wppm_image_resize( $src, $imgwidth, $imgheight, $imgcrop, $imgquality, $imgcolor, $imggrayscale ) {
		$params = array();

		// Validate boolean params
		$crop = ( '' == $imgcrop || 'false' == $imgcrop ) ? false : true;
		$grayscale = ( '' == $imggrayscale || 'false' == $imggrayscale ) ? false : true;

		// Params array
		if ( $crop ) {
			$params['crop'] = true;
		}

		if ( $grayscale ) {
			$params['grayscale'] = true;
		}

		if ( '' != $imgquality ) {
			if ( (int)$imgquality < 1 ) {
				$quality = 1;
			} elseif ( (int)$imgquality > 100 ) {
				$quality = 100;
			} else {
				$quality = $imgquality;
			}
			$params['quality'] = (int)$quality;
		}

		if ( '' != $imgcolor ) {
			$color = preg_match( '/#([a-f]|[A-F]|[0-9]){3}(([a-f]|[A-F]|[0-9]){3})?\b/', $imgcolor ) ? $imgcolor : '';
			$params['color'] = $color;
		}

		// Validate width and height
		if ( isset( $imgwidth ) && (int)$imgwidth > 4 && '' != $imgwidth ) {
			$params['width'] = $imgwidth;
		}

		if ( isset( $imgheight ) && (int)$imgheight > 4 && '' != $imgheight ) {
			$params['height'] = $imgheight;
		}

		if ( function_exists( 'bfi_thumb' ) && ! empty( $params ) ) {
			return bfi_thumb( $src, $params );
		} else {
			return $src;
		}
	}
endif;

// Remove p and br tags from nested short codes
if ( ! function_exists( 'wppm_clean' ) ) :
	function wppm_clean( $content, $p_tag = false, $br_tag = false ) {
		$content = preg_replace( '#^<\/p>|^<br \/>|<p>$#', '', $content );

		if ( $br_tag )
			$content = preg_replace( '#<br \/>#', '', $content );

		if ( $p_tag )
			$content = preg_replace( '#<p>|</p>#', '', $content );

		$array = array (
			'<p>[' => '[',
			']</p>' => ']',
			']<br />' => ']',
			']<br>' => ']',
			'<p></p>' => ''
		);

		$content = strtr( $content, $array );

		return apply_filters( 'the_content', do_shortcode( shortcode_unautop( trim( $content ) ) ) );
	}
endif;


/**
 * Function to shorten any text by word
 */
if ( ! function_exists( 'wppm_short' ) ) :
	function wppm_short( $phrase, $max_words = 10, $allowed_tags = 'p,br,a,em,i,strong,b' ) {

		$more = apply_filters( 'wppm_more_text', '&hellip;' );

		/**
		 * For Chinese and Japanese languages
		 * Trimming of multi byte text is different
		 */
		if ( in_array( get_locale(), apply_filters( 'wppm_multi_byte_lang_list', array( 'zh_CN', 'zh_HK', 'zh_TW', 'ja' ) ) ) ) {
			return wp_trim_words( $phrase, $max_words, $more );
		}

		else {
			if ( '' == $max_words ) {
				$max_words = 100;
			}
			$phrase_array = explode( ' ', $phrase );
			if ( count( $phrase_array ) > $max_words && $max_words > 0 ) {
				$phrase = implode( ' ', array_slice( $phrase_array, 0, $max_words ) );
			}
			$phrase = do_shortcode( $phrase );
			return strip_tags( $phrase, $allowed_tags ) . ' ' . $more;
		}
	}
endif;

/**
 * Parse attributes of a nested shortcode
 * https://wordpress.org/plugins/bootstrap-3-shortcodes/
 */

function wppm_parse_nested_atts($str, $att = null) {
	$result = array();
	$return = array();
	$reg = get_shortcode_regex();
	preg_match_all('~'.$reg.'~',$str, $matches);
	foreach( $matches[2] as $key => $name ) {
		$parsed = shortcode_parse_atts($matches[3][$key]);
		$parsed = is_array($parsed) ? $parsed : array();
		$result[$name] = $parsed;
		$return[] = $result;
	}
	return $return;
}

// Post meta for post modules
if ( ! function_exists( 'wppm_meta' ) ) :
	function wppm_meta( $args = array() ) {
		global $post;
		$custom_link = get_post_meta( $post->ID, 'wppm_custom_link', true );
			$defaults = array(
				'template'	=> 'grid',
				'date_format' => get_option( 'date_format' ),
				'enable_schema' => false,
				'hide_cats' => false,
				'hide_reviews' => false,
				'show_cats' => false,
				'show_reviews' => false,
				'hide_date' => false,
				'hide_author' => false,
				'show_avatar' => false,
				'hide_views' => false,
				'hide_comments' => false,
				'readmore' => false,
				'ext_link' => false,
				'readmore_text' => esc_attr__( 'Read more', 'wppm' ),
				'readmore_link_src' 	=> 'permalink',
				'readmore_target_blank' => false,
				'readmore_cust_link' 	=> '',
				'sharing'	=> false,
				'share_style'	=> 'popup',
				'share_btns' => '',

				// Schema props
				'datecreated_prop'		=> 'datePublished',
				'datemodified_prop'		=> 'dateModified',
				'publisher_type'		=> 'Organization',
				'publisher_prop'		=> 'publisher',
				'publisher_name'		=> esc_attr( get_bloginfo( 'name' ) ),
				'publisher_logo'		=> '',
				'authorbox_type'		=> 'Person',
				'authorbox_prop'		=> 'author',
				'authorname_prop'		=> 'name',
				'authoravatar_prop'		=> 'image',
				'category_prop'			=> 'about',
				'commentcount_prop'		=> 'commentCount',
				'commenturl_prop'		=> 'discussionUrl',
			);

			$args = wp_parse_args( $args, $defaults );

			extract( $args );

			$protocol = is_ssl() ? 'https' : 'http';
			$schema = $protocol . '://schema.org/';
			// Date format
			$date = get_the_time( get_option( 'date_format' ) );

			if ( ! empty( $date_format ) ) {
				if ( $date_format == 'human' ) {
					$date = sprintf( _x( '%s ago', 'human time difference. E.g. 10 days ago', 'wppm' ), human_time_diff( get_the_time( 'U' ), current_time( 'timestamp' ) ) );
				}
				else {
					$date = get_the_time( esc_attr( $date_format ) );
				}
			}

			// Category and review stars
			$review_meta = '';

			// Create category list
			$cat_list = '';
			$cat_out = '';
			$hasmore = false;
			$i = 0;
			$cats = get_the_category();
			$cat_limit = apply_filters( 'wppm_cat_list_limit', 3 );
			$cat_count = intval( count( $cats ) - $cat_limit );
			if ( isset( $cats ) ) {
				foreach( $cats as $cat ) {
					if ( $i == $cat_limit ) {
						$hasmore = true;
						$cat_list .= '<li class="submenu-parent"><a class="wppm-cat-toggle" href="#">' . sprintf( esc_attr_x( '+ %d more', 'more count for category list', 'wppm' ), number_format_i18n( $cat_count ) ) . '</a><ul class="cat-sub submenu">';
					}
					$cat_list .= '<li><a class="cat-' . $cat->slug . '" href="' . get_category_link( $cat->cat_ID ) . '">' . $cat->cat_name . '</a></li>';
					$i++;
				}
				if ( $cat_list ) {
					$cat_out = '<ul class="post-cats">' . $cat_list;
					$cat_out .= $hasmore ? '</ul></li></ul>' : '</ul>';
				}
			}

			$cat_meta = ( 'true' != $hide_cats ) ? $cat_out : '';
			if ( function_exists( 'wp_review_show_total' ) && 'true' !== $hide_reviews ) {
				$review_meta = wp_review_show_total( $echo = false );
			}

			// Author and date meta
			$meta_data = '';

			$author = get_the_author();
			if ( $show_avatar ) {
				$meta_data .= sprintf( '<div%s%s class="author-avatar-32%s"><a%s href="%s" title="%s">%s%s</a></div>',
					$enable_schema ? ' itemscope itemtype="' . $schema . $authorbox_type . '"' : '',
					$enable_schema ? ' itemprop="' . $authorbox_prop . '"' : '',
					$hide_author && $hide_date ? ' avatar-only' : '',
					$enable_schema ? ' itemprop="' . $authorname_prop . '"' : '',
					esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
					sprintf( esc_html__( 'More posts by %s', 'wppm' ), esc_attr( $author ) ),
					$enable_schema ? '<span itemprop="' . $authoravatar_prop . '">' . get_avatar( get_the_author_meta( 'user_email' ), 32 ) . '</span>' : get_avatar( get_the_author_meta( 'user_email' ), 32 ),
					$enable_schema ? '<span class="schema-only" itemprop="' . $authorname_prop . '">' . esc_attr( $author ) . '</span>' : ''

				);
			}

			$meta_data .= sprintf( '<ul class="entry-meta%s">',
				$show_avatar ? ' avatar-enabled' : ''
			);

			// Publisher Schema
			if ( $enable_schema ) {
				$meta_data .= '<li class="publisher-schema" itemscope itemtype="' . $schema . $publisher_type . '" itemprop="' . $publisher_prop . '"><meta itemprop="name" content="' . $publisher_name . '"/><div itemprop="logo" itemscope itemtype="' . $schema . 'ImageObject"><img itemprop="url" src="' . esc_url( $publisher_logo ) . '" alt="' . esc_attr( get_bloginfo( 'name' ) ) . '"/></div></li>';
			}

			$modified_date_format = 'human' == $date_format ? get_option( 'date_format' ) : $date_format;
			$meta_data .= sprintf( '<li class="post-time%1$s"><time%2$s class="entry-date updated" datetime="%3$s">%4$s</time>%5$s</li>',
				$hide_date ? ' schema-only' : '',
				$enable_schema ? ' itemprop="' . $datecreated_prop . '"' : '',
				esc_attr( get_the_date( 'c' ) ),
				esc_html( $date ),
				$enable_schema ? '<meta itemprop="' . $datemodified_prop . '" content="' . esc_attr( the_modified_date( $modified_date_format, '', '', false ) ) . '"/>' : ''
			);

			$meta_data .= sprintf( '<li%1$s%2$s class="post-author%3$s"><span class="screen-reader-text">%4$s</span><a href="%5$s">%6$s</a></li>',
				$enable_schema ? ' itemscope itemtype="' . $schema . $authorbox_type . '"' : '',
				$enable_schema ? ' itemprop="' . $authorbox_prop . '"' : '',
				$hide_author ? ' schema-only' : '',
				esc_html_x( 'Author', 'Used before post author name.', 'wppm' ),
				esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
				$enable_schema ? '<span class="vcard author ' . esc_attr( $author ) . '" itemprop="' . $authorname_prop . '"><span class="fn">' . esc_attr( $author ) . '</span></span>' : '<span class="vcard author ' . esc_attr( $author ) . '"><span class="fn">' . esc_attr( $author ) . '</span></span>'
			);

			$meta_data .= '</ul>';

			// Comment link
			$num_comments = get_comments_number();
			$comment_meta = '';
			if ( comments_open() && ( $num_comments >= 1 ) && ! $hide_comments ) {
				$comment_meta = sprintf( '<a href="%1$s" class="post-comment" title="%2$s">%3$s%4$s</a>',
					esc_url( get_comments_link() ),
					sprintf( esc_html__( 'Comment on %s', 'wppm' ), esc_attr( get_the_title() ) ),
					$enable_schema ? '<meta itemprop="' . $commenturl_prop . '" content="' . esc_url( get_comments_link() ) . '" />' : '',
					$enable_schema ? '<span itemprop="' . $commentcount_prop . '">' . $num_comments . '</span>' : $num_comments
				);
			}

			/**
			 * Post views
			 * Requires Plugin https://wordpress.org/plugins/post-views-counter/
			 */
			$views_meta = '';
			if ( function_exists( 'pvc_get_post_views' ) && ! $hide_views ) {
				$views_meta = sprintf( '<span class="post-views">%s</span>',
					pvc_get_post_views()
				);
			}

			/**
			 * Social share buttons
			 * Uses wppm_share_btns() function
			 */
			$share_btns_output =  $sharing ? wppm_share_btns( $share_btns, $share_style ) : '';

			// Generate rows of content
			$row_1 = '';
			$row_2 = '';
			$row_3 = '';
			$row_4 = '';
			if ( $review_meta != '' || $cat_meta != '' ) {
				$row_1 .= '<aside class="meta-row cat-row">';
				if ( $cat_meta != '' ) {
					$row_1 .= sprintf( '<div%s class="meta-col%s">%s</div>',
						$enable_schema ? ' itemprop="' . $category_prop . '"' : '',
						$review_meta != '' ? ' col-60' : '',
						$cat_meta
					);
				}

				if ( $review_meta != '' ) {
					$row_1 .= sprintf( '<div class="meta-col%s">%s</div>',
						$cat_meta != '' ? ' col-40 text-right' : '',
						$review_meta
					);
				}
				$row_1 .= '</aside>';
			}

			$row_4 .= sprintf( '<aside class="meta-row row-3%s">',
				( $hide_date && $hide_author && $hide_views && $hide_comments && 'true' !== $sharing ) ? ' schema-only' : ''
			);

			if ( '' == $views_meta && '' == $comment_meta && 'true' !== $sharing ) {
				$row_4 .= sprintf( '<div class="meta-col">%s</div>', $meta_data );
			}

			elseif ( '' == $meta_data ) {
				$row_4 .= sprintf( '<div class="meta-col">%s%s%s</div>', $views_meta, $comment_meta, $share_btns_output );
			}

			elseif ( 'inline' == $share_style ) {
				$row_4 .= sprintf( '<div class="meta-col col-60">%s</div><div class="meta-col col-40 text-right">%s%s</div>%s', $meta_data, $views_meta, $comment_meta, $share_btns_output );
			}

			else {
				$row_4 .= sprintf( '<div class="meta-col col-60">%s</div><div class="meta-col col-40 text-right">%s%s%s</div>', $meta_data, $views_meta, $comment_meta, $share_btns_output );
			}
			$row_4 .= '</aside>';

			if ( $readmore ) {
				if ( $meta_data ) {
					$row_2 = sprintf( '<aside class="meta-row row-2%s"><div class="meta-col">%s</div></aside>',
						( $hide_date && $hide_author && $hide_views && $hide_comments && 'true' !== $sharing ) ? ' schema-only' : '',
						$meta_data
					);
				}

				if ( $readmore || $views_meta || $comment_meta || $sharing ) {
					$row_3 = sprintf( '<aside class="meta-row row-3"><div class="meta-col col-50"><a class="readmore-link" href="%s"%s>%s</a></div><div class="meta-col col-50 text-right">%s%s%s</div></aside>',
						'ext' == $readmore_link_src && '' !== $custom_link ? esc_url( $custom_link ) : ( 'custom' == $readmore_link_src && '' !== $readmore_cust_link ? esc_url( $readmore_cust_link ) : esc_url( get_permalink() ) ),
						$readmore_target_blank ? ' target="_blank"' : '',
						esc_attr( $readmore_text ),
						$views_meta,
						$comment_meta,
						$share_btns_output
					);
				}
			}

			else {
				$row_3 = $row_4;
			}

		$meta_arr = array();
		$meta_arr['row_1'] = $row_1;
		$meta_arr['row_2'] = $row_2;
		$meta_arr['row_3'] = $row_3;
		$meta_arr['row_4'] = $row_4;
		return $meta_arr;
	}
endif;

/**
 * Get color pallete from image
 * http://stackoverflow.com/questions/10290259/detect-main-colors-in-an-image-with-php#answer-41044459
 */
if ( ! function_exists( 'wppm_get_color_pallet' ) ) :
	function wppm_get_color_pallet( $imageURL, $palletSize = array( 16, 8 ) ) {

		if ( ! $imageURL ) {
			return false;
		}

		// Create pallet from jpeg image
		$img = imagecreatefromjpeg( $imageURL );

		// Scale down image
		$imgSizes = getimagesize( $imageURL );

		$resizedImg = imagecreatetruecolor( $palletSize[0], $palletSize[1] );

		imagecopyresized( $resizedImg, $img, 0, 0, 0, 0, $palletSize[0], $palletSize[1], $imgSizes[0], $imgSizes[1] );

		imagedestroy( $img );

		// Get collors in array
		$colors = array();

		for( $i = 0; $i < $palletSize[1]; $i++ ) {
			for( $j = 0; $j < $palletSize[0]; $j++ ) {
				$colors[] = dechex( imagecolorat( $resizedImg, $j, $i ) );
			}
		}

		imagedestroy( $resizedImg );

		// Remove duplicates
		$colors= array_unique( $colors );

		return $colors;

	}
endif;

/**
 * Get dominant color from a given pallete
 */
if ( ! function_exists( 'wppm_get_dominant_color' ) ) :
	function wppm_get_dominant_color( $pallet ) {

		$lsum = 0;
		$larr = array();

		if ( isset( $pallet ) && is_array( $pallet ) ) {
			foreach( $pallet as $key => $val ) {
				// Split hex value of color in RGB
				$r = hexdec( substr( $val, 0, 2 ) );
				$g = hexdec( substr( $val, 2, 2 ) );
				$b = hexdec( substr( $val, 4, 2 ) );

				// Calculate luma (brightness in an image)
				$luma = 0.2126 * $r + 0.7152 * $g + 0.0722 * $b;
				$larr[] = array( 'luma' => $luma, 'rgb' => $r . ',' . $g . ',' . $b, 'rgbsum' => intval($r + $g + $b) );
				$lsum = $lsum + $luma;
			}

			// Average luma from all available colors in pallet
			$lavg = $lsum / count( $pallet );
			$rgbsum = 0;

			/**
			 * Find dominant color
			 *
			 * Compares each luma value to luma average
			 * and returns the closest match
			 */
			$closest = null;
			foreach ( $larr as $item ) {
				if ( $closest === null || abs( (int)$lavg - (int)$closest ) > abs( (int)$item['luma'] - (int)$lavg ) ) {
					$closest = $item['rgb'];
					$dluma = $item['luma'];
				}
			}

			// Check if the pallet is too light and needs dark text
			$dark_text = $dluma > 178 ? 'true' : '';

			return array( 'rgb' => $closest, 'dark_text' => $dark_text );
		}

		else {
			return array( 'rgb' => '77, 77, 77', 'dark_text' => '' );
		}
	}
endif;

/**
 * Social Sharing feature on single posts
 */
if ( ! function_exists( 'wppm_share_btns' ) ) :
	function wppm_share_btns( $share_btns, $share_style = 'popup' ) {
		global $post;
		setup_postdata( $post );
		$share_btns = ( $share_btns ) ? explode( ',', $share_btns ) : '';

		// Set variables
		$out = '';
		$list = '';
		$share_image = '';
		$protocol = is_ssl() ? 'https' : 'http';

		if ( has_post_thumbnail( $post->ID ) ) {
			$share_image = wp_get_attachment_url( get_post_thumbnail_id( $post->ID ) );
		}

		$share_content = strip_tags( get_the_excerpt() );

		if ( 'inline' == $share_style ) {
			$out .= '<ul class="wppm-sharing inline">';
		} else {
		$out .= sprintf( '<div class="wppm-sharing-container"><a class="share-trigger" title="%1$s"><span class="screen-reader-text">%1$s</span></a><ul class="wppm-sharing">', esc_html__( 'Share this post', 'wppm' ) );
		}

		if ( ! empty( $share_btns ) && is_array( $share_btns ) ) {
			foreach ( $share_btns as $button ) {

				switch( $button ) {

					case 'twitter':
						$list .= sprintf( '<li class="wppm-twitter"><a href="%s://twitter.com/intent/tweet?text=%s" target="_blank" title="%s">%s</a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on X (twitter)', 'wppm' ), esc_attr__( 'X (twitter)', 'wppm-el' ) );
					break;

					case 'facebook':
						$list .= sprintf( '<li class="wppm-facebook"><a href="%s://www.facebook.com/sharer/sharer.php?u=%s" target="_blank" title="%s">%s</a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on facebook', 'wppm' ), esc_attr__( 'Facebook', 'wppm' ) );
					break;

					case 'whatsapp':
						if ( wp_is_mobile() ) {
							$list .= sprintf( '<li class="wppm-whatsapp"><a href="whatsapp://send?text=%s" title="%s" data-action="share/whatsapp/share">%s</a></li>', urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on Whatsapp', 'wppm' ), esc_attr__( 'Whatsapp', 'wppm' ) );
						}
					break;

					case 'linkedin':
						$list .= sprintf( '<li class="wppm-linkedin"><a href="%s://www.linkedin.com/shareArticle?mini=true&amp;url=%s" target="_blank" title="%s">%s</a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on LinkedIn', 'wppm' ), esc_attr__( 'LinkedIn', 'wppm' ) );
					break;

					case 'pinterest':
						$list .= sprintf( '<li class="wppm-pinterest"><a href="%s://pinterest.com/pin/create/button/?url=%s&amp;media=%s" target="_blank" title="%s">%s</a></li>',
							esc_attr( $protocol ),
							urlencode( esc_url( get_permalink() ) ),
							esc_url( $share_image ),
							esc_attr__( 'Pin it', 'wppm' ),
							esc_attr__( 'Pinterest', 'wppm' )
						);
					break;

					case 'vkontakte':
						$list .= sprintf( '<li class="wppm-vkontakte"><a href="%s://vkontakte.ru/share.php?url=%s" target="_blank" title="%s">%s</a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share via VK', 'wppm' ), esc_attr__( 'VKOntakte', 'wppm' ) );
					break;

					case 'line':
						$list .= sprintf( '<li class="wppm-line"><a href="%s://social-plugins.line.me/lineit/share?url=%s" target="_blank" title="%s">%s</a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on LINE', 'wppm' ), esc_attr__( 'LINE', 'wppm' ) );
					break;

					case 'reddit':
						$list .= sprintf( '<li class="wppm-reddit"><a href="//www.reddit.com/submit?url=%s" title="%s">%s</a></li>', urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on Reddit', 'wppm' ), esc_attr__( 'Reddit', 'wppm' ) );
					break;

					case 'email':
						$list .= sprintf( '<li class="wppm-email no-popup"><a href="mailto:someone@example.com?Subject=%s" title="%s">%s</a></li>', urlencode( esc_attr( get_the_title() ) ), esc_attr__( 'Email this', 'wppm' ), esc_attr__( 'Email', 'wppm' ) );

					break;
				} // switch

			} // foreach
		} // if

		// Support extra meta items via action hook
		ob_start();
		do_action( 'wppm_sharing_buttons_li' );
		$out .= ob_get_contents();
		ob_end_clean();

		if ( 'inline' == $share_style ) {
			$out .= $list . '</ul>';
		} else {
			$out .= $list . '</ul></div>';
		}
		return $out;
	}
endif;

/**
 * Add Social meta tags for Facebook
 */
function wppm_social_meta_tags() {

	if ( is_single() && apply_filters( 'wppm_enable_social_tags', 'true' ) ) {
		global $post;
		setup_postdata( $post );
		$image = '';
		if ( has_post_thumbnail( $post->ID ) ) {
			$image = wp_get_attachment_url( get_post_thumbnail_id( $post->ID ) );
		}
		$title = get_the_title();
		$permalink = get_permalink();
		$excerpt = get_the_excerpt();
		$site_name = get_bloginfo( 'name' );
		?>

        <!-- Begin Social Meta tags by WP Post Modules -->

            <!-- Twitter Card data -->
            <meta name="twitter:card" content="summary_large_image">
            <meta name="twitter:title" content="<?php echo esc_attr( $title ); ?>">
            <meta name="twitter:description" content="<?php echo strip_tags( $excerpt ); ?>">
            <meta name="twitter:image:src" content="<?php echo esc_url( $image ); ?>">

            <!-- Open Graph data -->
            <meta property="og:title" content="<?php echo esc_attr( $title ); ?>" />
            <meta property="og:type" content="article" />
            <meta property="og:url" content="<?php echo esc_url( $permalink ); ?>" />
            <meta property="og:image" content="<?php echo esc_url( $image ); ?>" />
            <meta property="og:image:url" content="<?php echo esc_url( $image ); ?>" />
            <meta property="og:description" content="<?php echo strip_tags( $excerpt ); ?>" />
            <meta property="og:site_name" content="<?php echo esc_attr( $site_name ); ?>" />
        <!-- End Social Meta tags by WP Post Modules -->
		<?php wp_reset_postdata();
	}
}
add_action( 'wp_head', 'wppm_social_meta_tags', 99 );

function wppm_animate_in_list() {
	return apply_filters( 'wppm_animate_in_list', array(

				__( 'Default', 'wppm' ) =>	array(
						__( 'Slide', 'wppm' ) => '',
					),

				__( 'Fading Entrances', 'wppm' )	=>	array(
					"fadeIn",
					"fadeInDown",
					"fadeInDownBig",
					"fadeInLeft",
					"fadeInLeftBig",
					"fadeInRight",
					"fadeInRightBig",
					"fadeInUp",
					"fadeInUpBig"
				),

				__( 'Sliding Entrances', 'wppm' )	=>	array(
					"slideInUp",
					"slideInDown",
					"slideInLeft",
					"slideInRight"
				),

				__( 'Zoom Entrances', 'wppm' )	=>	array(
					"zoomIn",
					"zoomInDown",
					"zoomInLeft",
					"zoomInRight",
					"zoomInUp"
				)
			)
		);
}

function wppm_animate_out_list() {
	return apply_filters( 'wppm_animate_out_list', array(

				__( 'Default', 'wppm' ) =>	array(
						__( 'Slide', 'wppm' ) => '',
					),

				__( 'Fading Exits', 'wppm' )	=>	array(
					"fadeOut",
					"fadeOutDown",
					"fadeOutDownBig",
					"fadeOutLeft",
					"fadeOutLeftBig",
					"fadeOutRight",
					"fadeOutRightBig",
					"fadeOutUp",
					"fadeOutUpBig"
				),

				__( 'Sliding Exits', 'wppm' )	=>	array(
					"slideOutUp",
					"slideOutDown",
					"slideOutLeft",
					"slideOutRight"
				),

				__( 'Zoom Exits', 'wppm' )	=>	array(
					"zoomOut",
					"zoomOutDown",
					"zoomOutLeft",
					"zoomOutRight",
					"zoomOutUp"
				)
		) );
}

if ( ! function_exists( 'wppm_enqueue_fonts' ) ) :
	function wppm_enqueue_fonts( $fontsData ) {
		// Get extra subsets for settings (latin/cyrillic/etc)
		$settings = get_option( 'wpb_js_google_fonts_subsets' );
		if ( is_array( $settings ) && ! empty( $settings ) ) {
			$subsets = '&subset=' . implode( ',', $settings );
		} else {
			$subsets = '';
		}

		// Enqueue font
		if ( isset( $fontsData ) ) {
			wp_enqueue_style( 'wppm_google_fonts_' . wppm_create_safe_class( $fontsData ), '//fonts.googleapis.com/css?family=' . $fontsData . $subsets );
		}
	}
endif;

if ( ! function_exists( 'wppm_create_font_style' ) ) :
	function wppm_create_font_style( $fontsData ) {
        $inline_style = '';
		if ( isset( $fontsData ) ) {
			$fontFamily = explode( ':', $fontsData );
			$styles[] = isset( $fontFamily[0] ) ? 'font-family:\'' . $fontFamily[0] . '\'': '';

			if ( isset( $styles ) && is_array( $styles ) ) {
				foreach( $styles as $attribute ){
					$inline_style .= $attribute.'; ';
				}
			}
		}
        return $inline_style;
    }
endif;

if ( ! function_exists( 'wppm_create_safe_class' ) ) :
	function wppm_create_safe_class( $class ) {
		return preg_replace( '/\W+/', '', strtolower( str_replace( ' ', '_', strip_tags( $class ) ) ) );
	}
endif;

/**
 * Get size information for all currently-registered image sizes.
 *
 * @global $_wp_additional_image_sizes
 * @uses   get_intermediate_image_sizes()
 * @return array $sizes Data for all currently-registered image sizes.
 */
if ( ! function_exists( 'wppm_get_image_sizes' ) ) :
function wppm_get_image_sizes() {
	global $_wp_additional_image_sizes;

	$sizes = array( esc_html__( 'Original', 'wppm' ) => 'full' );

	foreach ( get_intermediate_image_sizes() as $_size ) {
		if ( in_array( $_size, array('thumbnail', 'medium', 'medium_large', 'large') ) ) {
			/*$sizes[ $_size ]['width']  = get_option( "{$_size}_size_w" );
			$sizes[ $_size ]['height'] = get_option( "{$_size}_size_h" );
			$sizes[ $_size ]['crop']   = (bool) get_option( "{$_size}_crop" );*/
			$sizes[ $_size . '-' . get_option( "{$_size}_size_w" ) . 'x' . get_option( "{$_size}_size_h" ) ] = $_size;
		} elseif ( isset( $_wp_additional_image_sizes[ $_size ] ) ) {
			$sizes[ $_size . '-' . $_wp_additional_image_sizes[ $_size ]['width'] . 'x' . $_wp_additional_image_sizes[ $_size ]['height'] ] = $_size;
		}
	}

	return $sizes;
}
endif;

if ( ! function_exists( 'wppm_calc_image_sizes' ) ) :
function wppm_calc_image_sizes() {
	global $_wp_additional_image_sizes;

	foreach ( get_intermediate_image_sizes() as $_size ) {
		if ( in_array( $_size, array('thumbnail', 'medium', 'medium_large', 'large') ) ) {
			$sizes[ $_size ]['width']  = get_option( "{$_size}_size_w" );
			$sizes[ $_size ]['height'] = get_option( "{$_size}_size_h" );
		} elseif ( isset( $_wp_additional_image_sizes[ $_size ] ) ) {
			$sizes[ $_size ]['width']  = $_wp_additional_image_sizes[ $_size ]['width'];
			$sizes[ $_size ]['height'] = $_wp_additional_image_sizes[ $_size ]['width'];
		}
	}

	return $sizes;
}
endif;

/**
 * Get size information for a specific image size.
 *
 * @uses   wppm_get_image_sizes()
 * @param  string $size The image size for which to retrieve data.
 * @return bool|array $size Size data about an image size or false if the size doesn't exist.
 */
if ( ! function_exists( 'wppm_get_image_size' ) ) :
function wppm_get_image_size( $size ) {
	$sizes = wppm_calc_image_sizes();

	if ( isset( $sizes[ $size ] ) ) {
		return $sizes[ $size ];
	}

	return false;
}
endif;

/**
 * Get the width of a specific image size.
 *
 * @uses   wppm_get_image_size()
 * @param  string $size The image size for which to retrieve data.
 * @return bool|string $size Width of an image size or false if the size doesn't exist.
 */
if ( ! function_exists( 'wppm_get_image_width' ) ) :
function wppm_get_image_width( $size ) {
	if ( ! $size = wppm_get_image_size( $size ) ) {
		return false;
	}

	if ( isset( $size['width'] ) ) {
		return $size['width'];
	}

	return false;
}
endif;

/**
 * Get the height of a specific image size.
 *
 * @uses   wppm_get_image_size()
 * @param  string $size The image size for which to retrieve data.
 * @return bool|string $size Height of an image size or false if the size doesn't exist.
 */
if ( ! function_exists( 'wppm_get_image_height' ) ) :
	function wppm_get_image_height( $size ) {
		if ( ! $size = wppm_get_image_size( $size ) ) {
			return false;
		}

		if ( isset( $size['height'] ) ) {
			return $size['height'];
		}

		return false;
	}
endif;

if ( ! function_exists( 'wppm_google_font_list' ) ) :
function wppm_google_font_list() {
	return apply_filters( 'wppm_google_font_list',
				array(
					__( '-- Select --', 'wppm' ) => '',
					'ABeeZee' => 'ABeeZee: regular,italic',
					'ADLaM Display' => 'ADLaM Display: regular',
					'AR One Sans' => 'AR One Sans: regular,500,600,700',
					'Abel' => 'Abel: regular',
					'Abhaya Libre' => 'Abhaya Libre: regular,500,600,700,800',
					'Aboreto' => 'Aboreto: regular',
					'Abril Fatface' => 'Abril Fatface: regular',
					'Abyssinica SIL' => 'Abyssinica SIL: regular',
					'Aclonica' => 'Aclonica: regular',
					'Acme' => 'Acme: regular',
					'Actor' => 'Actor: regular',
					'Adamina' => 'Adamina: regular',
					'Advent Pro' => 'Advent Pro: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Afacad' => 'Afacad: regular,500,600,700,italic,500italic,600italic,700italic',
					'Agbalumo' => 'Agbalumo: regular',
					'Agdasima' => 'Agdasima: regular,700',
					'Aguafina Script' => 'Aguafina Script: regular',
					'Akatab' => 'Akatab: regular,500,600,700,800,900',
					'Akaya Kanadaka' => 'Akaya Kanadaka: regular',
					'Akaya Telivigala' => 'Akaya Telivigala: regular',
					'Akronim' => 'Akronim: regular',
					'Akshar' => 'Akshar: 300,regular,500,600,700',
					'Aladin' => 'Aladin: regular',
					'Alata' => 'Alata: regular',
					'Alatsi' => 'Alatsi: regular',
					'Albert Sans' => 'Albert Sans: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Aldrich' => 'Aldrich: regular',
					'Alef' => 'Alef: regular,700',
					'Alegreya' => 'Alegreya: regular,500,600,700,800,900,italic,500italic,600italic,700italic,800italic,900italic',
					'Alegreya SC' => 'Alegreya SC: regular,italic,500,500italic,700,700italic,800,800italic,900,900italic',
					'Alegreya Sans' => 'Alegreya Sans: 100,100italic,300,300italic,regular,italic,500,500italic,700,700italic,800,800italic,900,900italic',
					'Alegreya Sans SC' => 'Alegreya Sans SC: 100,100italic,300,300italic,regular,italic,500,500italic,700,700italic,800,800italic,900,900italic',
					'Aleo' => 'Aleo: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Alex Brush' => 'Alex Brush: regular',
					'Alexandria' => 'Alexandria: 100,200,300,regular,500,600,700,800,900',
					'Alfa Slab One' => 'Alfa Slab One: regular',
					'Alice' => 'Alice: regular',
					'Alike' => 'Alike: regular',
					'Alike Angular' => 'Alike Angular: regular',
					'Alkalami' => 'Alkalami: regular',
					'Alkatra' => 'Alkatra: regular,500,600,700',
					'Allan' => 'Allan: regular,700',
					'Allerta' => 'Allerta: regular',
					'Allerta Stencil' => 'Allerta Stencil: regular',
					'Allison' => 'Allison: regular',
					'Allura' => 'Allura: regular',
					'Almarai' => 'Almarai: 300,regular,700,800',
					'Almendra' => 'Almendra: regular,italic,700,700italic',
					'Almendra Display' => 'Almendra Display: regular',
					'Almendra SC' => 'Almendra SC: regular',
					'Alumni Sans' => 'Alumni Sans: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Alumni Sans Collegiate One' => 'Alumni Sans Collegiate One: regular,italic',
					'Alumni Sans Inline One' => 'Alumni Sans Inline One: regular,italic',
					'Alumni Sans Pinstripe' => 'Alumni Sans Pinstripe: regular,italic',
					'Amarante' => 'Amarante: regular',
					'Amaranth' => 'Amaranth: regular,italic,700,700italic',
					'Amatic SC' => 'Amatic SC: regular,700',
					'Amethysta' => 'Amethysta: regular',
					'Amiko' => 'Amiko: regular,600,700',
					'Amiri' => 'Amiri: regular,italic,700,700italic',
					'Amiri Quran' => 'Amiri Quran: regular',
					'Amita' => 'Amita: regular,700',
					'Anaheim' => 'Anaheim: regular',
					'Andada Pro' => 'Andada Pro: regular,500,600,700,800,italic,500italic,600italic,700italic,800italic',
					'Andika' => 'Andika: regular,italic,700,700italic',
					'Anek Bangla' => 'Anek Bangla: 100,200,300,regular,500,600,700,800',
					'Anek Devanagari' => 'Anek Devanagari: 100,200,300,regular,500,600,700,800',
					'Anek Gujarati' => 'Anek Gujarati: 100,200,300,regular,500,600,700,800',
					'Anek Gurmukhi' => 'Anek Gurmukhi: 100,200,300,regular,500,600,700,800',
					'Anek Kannada' => 'Anek Kannada: 100,200,300,regular,500,600,700,800',
					'Anek Latin' => 'Anek Latin: 100,200,300,regular,500,600,700,800',
					'Anek Malayalam' => 'Anek Malayalam: 100,200,300,regular,500,600,700,800',
					'Anek Odia' => 'Anek Odia: 100,200,300,regular,500,600,700,800',
					'Anek Tamil' => 'Anek Tamil: 100,200,300,regular,500,600,700,800',
					'Anek Telugu' => 'Anek Telugu: 100,200,300,regular,500,600,700,800',
					'Angkor' => 'Angkor: regular',
					'Annie Use Your Telescope' => 'Annie Use Your Telescope: regular',
					'Anonymous Pro' => 'Anonymous Pro: regular,italic,700,700italic',
					'Antic' => 'Antic: regular',
					'Antic Didone' => 'Antic Didone: regular',
					'Antic Slab' => 'Antic Slab: regular',
					'Anton' => 'Anton: regular',
					'Antonio' => 'Antonio: 100,200,300,regular,500,600,700',
					'Anuphan' => 'Anuphan: 100,200,300,regular,500,600,700',
					'Anybody' => 'Anybody: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Aoboshi One' => 'Aoboshi One: regular',
					'Arapey' => 'Arapey: regular,italic',
					'Arbutus' => 'Arbutus: regular',
					'Arbutus Slab' => 'Arbutus Slab: regular',
					'Architects Daughter' => 'Architects Daughter: regular',
					'Archivo' => 'Archivo: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Archivo Black' => 'Archivo Black: regular',
					'Archivo Narrow' => 'Archivo Narrow: regular,500,600,700,italic,500italic,600italic,700italic',
					'Are You Serious' => 'Are You Serious: regular',
					'Aref Ruqaa' => 'Aref Ruqaa: regular,700',
					'Aref Ruqaa Ink' => 'Aref Ruqaa Ink: regular,700',
					'Arima' => 'Arima: 100,200,300,regular,500,600,700',
					'Arimo' => 'Arimo: regular,500,600,700,italic,500italic,600italic,700italic',
					'Arizonia' => 'Arizonia: regular',
					'Armata' => 'Armata: regular',
					'Arsenal' => 'Arsenal: regular,italic,700,700italic',
					'Artifika' => 'Artifika: regular',
					'Arvo' => 'Arvo: regular,italic,700,700italic',
					'Arya' => 'Arya: regular,700',
					'Asap' => 'Asap: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Asap Condensed' => 'Asap Condensed: 200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Asar' => 'Asar: regular',
					'Asset' => 'Asset: regular',
					'Assistant' => 'Assistant: 200,300,regular,500,600,700,800',
					'Astloch' => 'Astloch: regular,700',
					'Asul' => 'Asul: regular,700',
					'Athiti' => 'Athiti: 200,300,regular,500,600,700',
					'Atkinson Hyperlegible' => 'Atkinson Hyperlegible: regular,italic,700,700italic',
					'Atma' => 'Atma: 300,regular,500,600,700',
					'Atomic Age' => 'Atomic Age: regular',
					'Aubrey' => 'Aubrey: regular',
					'Audiowide' => 'Audiowide: regular',
					'Autour One' => 'Autour One: regular',
					'Average' => 'Average: regular',
					'Average Sans' => 'Average Sans: regular',
					'Averia Gruesa Libre' => 'Averia Gruesa Libre: regular',
					'Averia Libre' => 'Averia Libre: 300,300italic,regular,italic,700,700italic',
					'Averia Sans Libre' => 'Averia Sans Libre: 300,300italic,regular,italic,700,700italic',
					'Averia Serif Libre' => 'Averia Serif Libre: 300,300italic,regular,italic,700,700italic',
					'Azeret Mono' => 'Azeret Mono: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'B612' => 'B612: regular,italic,700,700italic',
					'B612 Mono' => 'B612 Mono: regular,italic,700,700italic',
					'BIZ UDGothic' => 'BIZ UDGothic: regular,700',
					'BIZ UDMincho' => 'BIZ UDMincho: regular,700',
					'BIZ UDPGothic' => 'BIZ UDPGothic: regular,700',
					'BIZ UDPMincho' => 'BIZ UDPMincho: regular,700',
					'Babylonica' => 'Babylonica: regular',
					'Bacasime Antique' => 'Bacasime Antique: regular',
					'Bad Script' => 'Bad Script: regular',
					'Bagel Fat One' => 'Bagel Fat One: regular',
					'Bahiana' => 'Bahiana: regular',
					'Bahianita' => 'Bahianita: regular',
					'Bai Jamjuree' => 'Bai Jamjuree: 200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Bakbak One' => 'Bakbak One: regular',
					'Ballet' => 'Ballet: regular',
					'Baloo 2' => 'Baloo 2: regular,500,600,700,800',
					'Baloo Bhai 2' => 'Baloo Bhai 2: regular,500,600,700,800',
					'Baloo Bhaijaan 2' => 'Baloo Bhaijaan 2: regular,500,600,700,800',
					'Baloo Bhaina 2' => 'Baloo Bhaina 2: regular,500,600,700,800',
					'Baloo Chettan 2' => 'Baloo Chettan 2: regular,500,600,700,800',
					'Baloo Da 2' => 'Baloo Da 2: regular,500,600,700,800',
					'Baloo Paaji 2' => 'Baloo Paaji 2: regular,500,600,700,800',
					'Baloo Tamma 2' => 'Baloo Tamma 2: regular,500,600,700,800',
					'Baloo Tammudu 2' => 'Baloo Tammudu 2: regular,500,600,700,800',
					'Baloo Thambi 2' => 'Baloo Thambi 2: regular,500,600,700,800',
					'Balsamiq Sans' => 'Balsamiq Sans: regular,italic,700,700italic',
					'Balthazar' => 'Balthazar: regular',
					'Bangers' => 'Bangers: regular',
					'Barlow' => 'Barlow: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Barlow Condensed' => 'Barlow Condensed: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Barlow Semi Condensed' => 'Barlow Semi Condensed: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Barriecito' => 'Barriecito: regular',
					'Barrio' => 'Barrio: regular',
					'Basic' => 'Basic: regular',
					'Baskervville' => 'Baskervville: regular,italic',
					'Battambang' => 'Battambang: 100,300,regular,700,900',
					'Baumans' => 'Baumans: regular',
					'Bayon' => 'Bayon: regular',
					'Be Vietnam Pro' => 'Be Vietnam Pro: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Beau Rivage' => 'Beau Rivage: regular',
					'Bebas Neue' => 'Bebas Neue: regular',
					'Belanosima' => 'Belanosima: regular,600,700',
					'Belgrano' => 'Belgrano: regular',
					'Bellefair' => 'Bellefair: regular',
					'Belleza' => 'Belleza: regular',
					'Bellota' => 'Bellota: 300,300italic,regular,italic,700,700italic',
					'Bellota Text' => 'Bellota Text: 300,300italic,regular,italic,700,700italic',
					'BenchNine' => 'BenchNine: 300,regular,700',
					'Benne' => 'Benne: regular',
					'Bentham' => 'Bentham: regular',
					'Berkshire Swash' => 'Berkshire Swash: regular',
					'Besley' => 'Besley: regular,500,600,700,800,900,italic,500italic,600italic,700italic,800italic,900italic',
					'Beth Ellen' => 'Beth Ellen: regular',
					'Bevan' => 'Bevan: regular,italic',
					'BhuTuka Expanded One' => 'BhuTuka Expanded One: regular',
					'Big Shoulders Display' => 'Big Shoulders Display: 100,200,300,regular,500,600,700,800,900',
					'Big Shoulders Inline Display' => 'Big Shoulders Inline Display: 100,200,300,regular,500,600,700,800,900',
					'Big Shoulders Inline Text' => 'Big Shoulders Inline Text: 100,200,300,regular,500,600,700,800,900',
					'Big Shoulders Stencil Display' => 'Big Shoulders Stencil Display: 100,200,300,regular,500,600,700,800,900',
					'Big Shoulders Stencil Text' => 'Big Shoulders Stencil Text: 100,200,300,regular,500,600,700,800,900',
					'Big Shoulders Text' => 'Big Shoulders Text: 100,200,300,regular,500,600,700,800,900',
					'Bigelow Rules' => 'Bigelow Rules: regular',
					'Bigshot One' => 'Bigshot One: regular',
					'Bilbo' => 'Bilbo: regular',
					'Bilbo Swash Caps' => 'Bilbo Swash Caps: regular',
					'BioRhyme' => 'BioRhyme: 200,300,regular,500,600,700,800',
					'BioRhyme Expanded' => 'BioRhyme Expanded: 200,300,regular,700,800',
					'Birthstone' => 'Birthstone: regular',
					'Birthstone Bounce' => 'Birthstone Bounce: regular,500',
					'Biryani' => 'Biryani: 200,300,regular,600,700,800,900',
					'Bitter' => 'Bitter: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Black And White Picture' => 'Black And White Picture: regular',
					'Black Han Sans' => 'Black Han Sans: regular',
					'Black Ops One' => 'Black Ops One: regular',
					'Blaka' => 'Blaka: regular',
					'Blaka Hollow' => 'Blaka Hollow: regular',
					'Blaka Ink' => 'Blaka Ink: regular',
					'Blinker' => 'Blinker: 100,200,300,regular,600,700,800,900',
					'Bodoni Moda' => 'Bodoni Moda: regular,500,600,700,800,900,italic,500italic,600italic,700italic,800italic,900italic',
					'Bokor' => 'Bokor: regular',
					'Bona Nova' => 'Bona Nova: regular,italic,700',
					'Bonbon' => 'Bonbon: regular',
					'Bonheur Royale' => 'Bonheur Royale: regular',
					'Boogaloo' => 'Boogaloo: regular',
					'Borel' => 'Borel: regular',
					'Bowlby One' => 'Bowlby One: regular',
					'Bowlby One SC' => 'Bowlby One SC: regular',
					'Braah One' => 'Braah One: regular',
					'Brawler' => 'Brawler: regular,700',
					'Bree Serif' => 'Bree Serif: regular',
					'Bricolage Grotesque' => 'Bricolage Grotesque: 200,300,regular,500,600,700,800',
					'Bruno Ace' => 'Bruno Ace: regular',
					'Bruno Ace SC' => 'Bruno Ace SC: regular',
					'Brygada 1918' => 'Brygada 1918: regular,500,600,700,italic,500italic,600italic,700italic',
					'Bubblegum Sans' => 'Bubblegum Sans: regular',
					'Bubbler One' => 'Bubbler One: regular',
					'Buda' => 'Buda: 300',
					'Buenard' => 'Buenard: regular,700',
					'Bungee' => 'Bungee: regular',
					'Bungee Hairline' => 'Bungee Hairline: regular',
					'Bungee Inline' => 'Bungee Inline: regular',
					'Bungee Outline' => 'Bungee Outline: regular',
					'Bungee Shade' => 'Bungee Shade: regular',
					'Bungee Spice' => 'Bungee Spice: regular',
					'Butcherman' => 'Butcherman: regular',
					'Butterfly Kids' => 'Butterfly Kids: regular',
					'Cabin' => 'Cabin: regular,500,600,700,italic,500italic,600italic,700italic',
					'Cabin Condensed' => 'Cabin Condensed: regular,500,600,700',
					'Cabin Sketch' => 'Cabin Sketch: regular,700',
					'Caesar Dressing' => 'Caesar Dressing: regular',
					'Cagliostro' => 'Cagliostro: regular',
					'Cairo' => 'Cairo: 200,300,regular,500,600,700,800,900',
					'Cairo Play' => 'Cairo Play: 200,300,regular,500,600,700,800,900',
					'Caladea' => 'Caladea: regular,italic,700,700italic',
					'Calistoga' => 'Calistoga: regular',
					'Calligraffitti' => 'Calligraffitti: regular',
					'Cambay' => 'Cambay: regular,italic,700,700italic',
					'Cambo' => 'Cambo: regular',
					'Candal' => 'Candal: regular',
					'Cantarell' => 'Cantarell: regular,italic,700,700italic',
					'Cantata One' => 'Cantata One: regular',
					'Cantora One' => 'Cantora One: regular',
					'Caprasimo' => 'Caprasimo: regular',
					'Capriola' => 'Capriola: regular',
					'Caramel' => 'Caramel: regular',
					'Carattere' => 'Carattere: regular',
					'Cardo' => 'Cardo: regular,italic,700',
					'Carlito' => 'Carlito: regular,italic,700,700italic',
					'Carme' => 'Carme: regular',
					'Carrois Gothic' => 'Carrois Gothic: regular',
					'Carrois Gothic SC' => 'Carrois Gothic SC: regular',
					'Carter One' => 'Carter One: regular',
					'Castoro' => 'Castoro: regular,italic',
					'Castoro Titling' => 'Castoro Titling: regular',
					'Catamaran' => 'Catamaran: 100,200,300,regular,500,600,700,800,900',
					'Caudex' => 'Caudex: regular,italic,700,700italic',
					'Caveat' => 'Caveat: regular,500,600,700',
					'Caveat Brush' => 'Caveat Brush: regular',
					'Cedarville Cursive' => 'Cedarville Cursive: regular',
					'Ceviche One' => 'Ceviche One: regular',
					'Chakra Petch' => 'Chakra Petch: 300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Changa' => 'Changa: 200,300,regular,500,600,700,800',
					'Changa One' => 'Changa One: regular,italic',
					'Chango' => 'Chango: regular',
					'Charis SIL' => 'Charis SIL: regular,italic,700,700italic',
					'Charm' => 'Charm: regular,700',
					'Charmonman' => 'Charmonman: regular,700',
					'Chathura' => 'Chathura: 100,300,regular,700,800',
					'Chau Philomene One' => 'Chau Philomene One: regular,italic',
					'Chela One' => 'Chela One: regular',
					'Chelsea Market' => 'Chelsea Market: regular',
					'Chenla' => 'Chenla: regular',
					'Cherish' => 'Cherish: regular',
					'Cherry Bomb One' => 'Cherry Bomb One: regular',
					'Cherry Cream Soda' => 'Cherry Cream Soda: regular',
					'Cherry Swash' => 'Cherry Swash: regular,700',
					'Chewy' => 'Chewy: regular',
					'Chicle' => 'Chicle: regular',
					'Chilanka' => 'Chilanka: regular',
					'Chivo' => 'Chivo: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Chivo Mono' => 'Chivo Mono: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Chokokutai' => 'Chokokutai: regular',
					'Chonburi' => 'Chonburi: regular',
					'Cinzel' => 'Cinzel: regular,500,600,700,800,900',
					'Cinzel Decorative' => 'Cinzel Decorative: regular,700,900',
					'Clicker Script' => 'Clicker Script: regular',
					'Climate Crisis' => 'Climate Crisis: regular',
					'Coda' => 'Coda: regular,800',
					'Codystar' => 'Codystar: 300,regular',
					'Coiny' => 'Coiny: regular',
					'Combo' => 'Combo: regular',
					'Comfortaa' => 'Comfortaa: 300,regular,500,600,700',
					'Comforter' => 'Comforter: regular',
					'Comforter Brush' => 'Comforter Brush: regular',
					'Comic Neue' => 'Comic Neue: 300,300italic,regular,italic,700,700italic',
					'Coming Soon' => 'Coming Soon: regular',
					'Comme' => 'Comme: 100,200,300,regular,500,600,700,800,900',
					'Commissioner' => 'Commissioner: 100,200,300,regular,500,600,700,800,900',
					'Concert One' => 'Concert One: regular',
					'Condiment' => 'Condiment: regular',
					'Content' => 'Content: regular,700',
					'Contrail One' => 'Contrail One: regular',
					'Convergence' => 'Convergence: regular',
					'Cookie' => 'Cookie: regular',
					'Copse' => 'Copse: regular',
					'Corben' => 'Corben: regular,700',
					'Corinthia' => 'Corinthia: regular,700',
					'Cormorant' => 'Cormorant: 300,regular,500,600,700,300italic,italic,500italic,600italic,700italic',
					'Cormorant Garamond' => 'Cormorant Garamond: 300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Cormorant Infant' => 'Cormorant Infant: 300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Cormorant SC' => 'Cormorant SC: 300,regular,500,600,700',
					'Cormorant Unicase' => 'Cormorant Unicase: 300,regular,500,600,700',
					'Cormorant Upright' => 'Cormorant Upright: 300,regular,500,600,700',
					'Courgette' => 'Courgette: regular',
					'Courier Prime' => 'Courier Prime: regular,italic,700,700italic',
					'Cousine' => 'Cousine: regular,italic,700,700italic',
					'Coustard' => 'Coustard: regular,900',
					'Covered By Your Grace' => 'Covered By Your Grace: regular',
					'Crafty Girls' => 'Crafty Girls: regular',
					'Creepster' => 'Creepster: regular',
					'Crete Round' => 'Crete Round: regular,italic',
					'Crimson Pro' => 'Crimson Pro: 200,300,regular,500,600,700,800,900,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Crimson Text' => 'Crimson Text: regular,italic,600,600italic,700,700italic',
					'Croissant One' => 'Croissant One: regular',
					'Crushed' => 'Crushed: regular',
					'Cuprum' => 'Cuprum: regular,500,600,700,italic,500italic,600italic,700italic',
					'Cute Font' => 'Cute Font: regular',
					'Cutive' => 'Cutive: regular',
					'Cutive Mono' => 'Cutive Mono: regular',
					'DM Mono' => 'DM Mono: 300,300italic,regular,italic,500,500italic',
					'DM Sans' => 'DM Sans: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'DM Serif Display' => 'DM Serif Display: regular,italic',
					'DM Serif Text' => 'DM Serif Text: regular,italic',
					'Dai Banna SIL' => 'Dai Banna SIL: 300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Damion' => 'Damion: regular',
					'Dancing Script' => 'Dancing Script: regular,500,600,700',
					'Dangrek' => 'Dangrek: regular',
					'Darker Grotesque' => 'Darker Grotesque: 300,regular,500,600,700,800,900',
					'Darumadrop One' => 'Darumadrop One: regular',
					'David Libre' => 'David Libre: regular,500,700',
					'Dawning of a New Day' => 'Dawning of a New Day: regular',
					'Days One' => 'Days One: regular',
					'Dekko' => 'Dekko: regular',
					'Dela Gothic One' => 'Dela Gothic One: regular',
					'Delicious Handrawn' => 'Delicious Handrawn: regular',
					'Delius' => 'Delius: regular',
					'Delius Swash Caps' => 'Delius Swash Caps: regular',
					'Delius Unicase' => 'Delius Unicase: regular,700',
					'Della Respira' => 'Della Respira: regular',
					'Denk One' => 'Denk One: regular',
					'Devonshire' => 'Devonshire: regular',
					'Dhurjati' => 'Dhurjati: regular',
					'Didact Gothic' => 'Didact Gothic: regular',
					'Diphylleia' => 'Diphylleia: regular',
					'Diplomata' => 'Diplomata: regular',
					'Diplomata SC' => 'Diplomata SC: regular',
					'Do Hyeon' => 'Do Hyeon: regular',
					'Dokdo' => 'Dokdo: regular',
					'Domine' => 'Domine: regular,500,600,700',
					'Donegal One' => 'Donegal One: regular',
					'Dongle' => 'Dongle: 300,regular,700',
					'Doppio One' => 'Doppio One: regular',
					'Dorsa' => 'Dorsa: regular',
					'Dosis' => 'Dosis: 200,300,regular,500,600,700,800',
					'DotGothic16' => 'DotGothic16: regular',
					'Dr Sugiyama' => 'Dr Sugiyama: regular',
					'Duru Sans' => 'Duru Sans: regular',
					'DynaPuff' => 'DynaPuff: regular,500,600,700',
					'Dynalight' => 'Dynalight: regular',
					'EB Garamond' => 'EB Garamond: regular,500,600,700,800,italic,500italic,600italic,700italic,800italic',
					'Eagle Lake' => 'Eagle Lake: regular',
					'East Sea Dokdo' => 'East Sea Dokdo: regular',
					'Eater' => 'Eater: regular',
					'Economica' => 'Economica: regular,italic,700,700italic',
					'Eczar' => 'Eczar: regular,500,600,700,800',
					'Edu NSW ACT Foundation' => 'Edu NSW ACT Foundation: regular,500,600,700',
					'Edu QLD Beginner' => 'Edu QLD Beginner: regular,500,600,700',
					'Edu SA Beginner' => 'Edu SA Beginner: regular,500,600,700',
					'Edu TAS Beginner' => 'Edu TAS Beginner: regular,500,600,700',
					'Edu VIC WA NT Beginner' => 'Edu VIC WA NT Beginner: regular,500,600,700',
					'El Messiri' => 'El Messiri: regular,500,600,700',
					'Electrolize' => 'Electrolize: regular',
					'Elsie' => 'Elsie: regular,900',
					'Elsie Swash Caps' => 'Elsie Swash Caps: regular,900',
					'Emblema One' => 'Emblema One: regular',
					'Emilys Candy' => 'Emilys Candy: regular',
					'Encode Sans' => 'Encode Sans: 100,200,300,regular,500,600,700,800,900',
					'Encode Sans Condensed' => 'Encode Sans Condensed: 100,200,300,regular,500,600,700,800,900',
					'Encode Sans Expanded' => 'Encode Sans Expanded: 100,200,300,regular,500,600,700,800,900',
					'Encode Sans SC' => 'Encode Sans SC: 100,200,300,regular,500,600,700,800,900',
					'Encode Sans Semi Condensed' => 'Encode Sans Semi Condensed: 100,200,300,regular,500,600,700,800,900',
					'Encode Sans Semi Expanded' => 'Encode Sans Semi Expanded: 100,200,300,regular,500,600,700,800,900',
					'Engagement' => 'Engagement: regular',
					'Englebert' => 'Englebert: regular',
					'Enriqueta' => 'Enriqueta: regular,500,600,700',
					'Ephesis' => 'Ephesis: regular',
					'Epilogue' => 'Epilogue: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Erica One' => 'Erica One: regular',
					'Esteban' => 'Esteban: regular',
					'Estonia' => 'Estonia: regular',
					'Euphoria Script' => 'Euphoria Script: regular',
					'Ewert' => 'Ewert: regular',
					'Exo' => 'Exo: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Exo 2' => 'Exo 2: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Expletus Sans' => 'Expletus Sans: regular,500,600,700,italic,500italic,600italic,700italic',
					'Explora' => 'Explora: regular',
					'Fahkwang' => 'Fahkwang: 200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Familjen Grotesk' => 'Familjen Grotesk: regular,500,600,700,italic,500italic,600italic,700italic',
					'Fanwood Text' => 'Fanwood Text: regular,italic',
					'Farro' => 'Farro: 300,regular,500,700',
					'Farsan' => 'Farsan: regular',
					'Fascinate' => 'Fascinate: regular',
					'Fascinate Inline' => 'Fascinate Inline: regular',
					'Faster One' => 'Faster One: regular',
					'Fasthand' => 'Fasthand: regular',
					'Fauna One' => 'Fauna One: regular',
					'Faustina' => 'Faustina: 300,regular,500,600,700,800,300italic,italic,500italic,600italic,700italic,800italic',
					'Federant' => 'Federant: regular',
					'Federo' => 'Federo: regular',
					'Felipa' => 'Felipa: regular',
					'Fenix' => 'Fenix: regular',
					'Festive' => 'Festive: regular',
					'Figtree' => 'Figtree: 300,regular,500,600,700,800,900,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Finger Paint' => 'Finger Paint: regular',
					'Finlandica' => 'Finlandica: regular,500,600,700,italic,500italic,600italic,700italic',
					'Fira Code' => 'Fira Code: 300,regular,500,600,700',
					'Fira Mono' => 'Fira Mono: regular,500,700',
					'Fira Sans' => 'Fira Sans: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Fira Sans Condensed' => 'Fira Sans Condensed: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Fira Sans Extra Condensed' => 'Fira Sans Extra Condensed: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Fjalla One' => 'Fjalla One: regular',
					'Fjord One' => 'Fjord One: regular',
					'Flamenco' => 'Flamenco: 300,regular',
					'Flavors' => 'Flavors: regular',
					'Fleur De Leah' => 'Fleur De Leah: regular',
					'Flow Block' => 'Flow Block: regular',
					'Flow Circular' => 'Flow Circular: regular',
					'Flow Rounded' => 'Flow Rounded: regular',
					'Foldit' => 'Foldit: 100,200,300,regular,500,600,700,800,900',
					'Fondamento' => 'Fondamento: regular,italic',
					'Fontdiner Swanky' => 'Fontdiner Swanky: regular',
					'Forum' => 'Forum: regular',
					'Fragment Mono' => 'Fragment Mono: regular,italic',
					'Francois One' => 'Francois One: regular',
					'Frank Ruhl Libre' => 'Frank Ruhl Libre: 300,regular,500,600,700,800,900',
					'Fraunces' => 'Fraunces: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Freckle Face' => 'Freckle Face: regular',
					'Fredericka the Great' => 'Fredericka the Great: regular',
					'Fredoka' => 'Fredoka: 300,regular,500,600,700',
					'Freehand' => 'Freehand: regular',
					'Fresca' => 'Fresca: regular',
					'Frijole' => 'Frijole: regular',
					'Fruktur' => 'Fruktur: regular,italic',
					'Fugaz One' => 'Fugaz One: regular',
					'Fuggles' => 'Fuggles: regular',
					'Fuzzy Bubbles' => 'Fuzzy Bubbles: regular,700',
					'GFS Didot' => 'GFS Didot: regular',
					'GFS Neohellenic' => 'GFS Neohellenic: regular,italic,700,700italic',
					'Gabarito' => 'Gabarito: regular,500,600,700,800,900',
					'Gabriela' => 'Gabriela: regular',
					'Gaegu' => 'Gaegu: 300,regular,700',
					'Gafata' => 'Gafata: regular',
					'Gajraj One' => 'Gajraj One: regular',
					'Galada' => 'Galada: regular',
					'Galdeano' => 'Galdeano: regular',
					'Galindo' => 'Galindo: regular',
					'Gamja Flower' => 'Gamja Flower: regular',
					'Gantari' => 'Gantari: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Gasoek One' => 'Gasoek One: regular',
					'Gayathri' => 'Gayathri: 100,regular,700',
					'Gelasio' => 'Gelasio: regular,italic,500,500italic,600,600italic,700,700italic',
					'Gemunu Libre' => 'Gemunu Libre: 200,300,regular,500,600,700,800',
					'Genos' => 'Genos: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Gentium Book Plus' => 'Gentium Book Plus: regular,italic,700,700italic',
					'Gentium Plus' => 'Gentium Plus: regular,italic,700,700italic',
					'Geo' => 'Geo: regular,italic',
					'Geologica' => 'Geologica: 100,200,300,regular,500,600,700,800,900',
					'Georama' => 'Georama: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Geostar' => 'Geostar: regular',
					'Geostar Fill' => 'Geostar Fill: regular',
					'Germania One' => 'Germania One: regular',
					'Gideon Roman' => 'Gideon Roman: regular',
					'Gidugu' => 'Gidugu: regular',
					'Gilda Display' => 'Gilda Display: regular',
					'Girassol' => 'Girassol: regular',
					'Give You Glory' => 'Give You Glory: regular',
					'Glass Antiqua' => 'Glass Antiqua: regular',
					'Glegoo' => 'Glegoo: regular,700',
					'Gloock' => 'Gloock: regular',
					'Gloria Hallelujah' => 'Gloria Hallelujah: regular',
					'Glory' => 'Glory: 100,200,300,regular,500,600,700,800,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic',
					'Gluten' => 'Gluten: 100,200,300,regular,500,600,700,800,900',
					'Goblin One' => 'Goblin One: regular',
					'Gochi Hand' => 'Gochi Hand: regular',
					'Goldman' => 'Goldman: regular,700',
					'Golos Text' => 'Golos Text: regular,500,600,700,800,900',
					'Gorditas' => 'Gorditas: regular,700',
					'Gothic A1' => 'Gothic A1: 100,200,300,regular,500,600,700,800,900',
					'Gotu' => 'Gotu: regular',
					'Goudy Bookletter 1911' => 'Goudy Bookletter 1911: regular',
					'Gowun Batang' => 'Gowun Batang: regular,700',
					'Gowun Dodum' => 'Gowun Dodum: regular',
					'Graduate' => 'Graduate: regular',
					'Grand Hotel' => 'Grand Hotel: regular',
					'Grandiflora One' => 'Grandiflora One: regular',
					'Grandstander' => 'Grandstander: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Grape Nuts' => 'Grape Nuts: regular',
					'Gravitas One' => 'Gravitas One: regular',
					'Great Vibes' => 'Great Vibes: regular',
					'Grechen Fuemen' => 'Grechen Fuemen: regular',
					'Grenze' => 'Grenze: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Grenze Gotisch' => 'Grenze Gotisch: 100,200,300,regular,500,600,700,800,900',
					'Grey Qo' => 'Grey Qo: regular',
					'Griffy' => 'Griffy: regular',
					'Gruppo' => 'Gruppo: regular',
					'Gudea' => 'Gudea: regular,italic,700',
					'Gugi' => 'Gugi: regular',
					'Gulzar' => 'Gulzar: regular',
					'Gupter' => 'Gupter: regular,500,700',
					'Gurajada' => 'Gurajada: regular',
					'Gwendolyn' => 'Gwendolyn: regular,700',
					'Habibi' => 'Habibi: regular',
					'Hachi Maru Pop' => 'Hachi Maru Pop: regular',
					'Hahmlet' => 'Hahmlet: 100,200,300,regular,500,600,700,800,900',
					'Halant' => 'Halant: 300,regular,500,600,700',
					'Hammersmith One' => 'Hammersmith One: regular',
					'Hanalei' => 'Hanalei: regular',
					'Hanalei Fill' => 'Hanalei Fill: regular',
					'Handjet' => 'Handjet: 100,200,300,regular,500,600,700,800,900',
					'Handlee' => 'Handlee: regular',
					'Hanken Grotesk' => 'Hanken Grotesk: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Hanuman' => 'Hanuman: 100,300,regular,700,900',
					'Happy Monkey' => 'Happy Monkey: regular',
					'Harmattan' => 'Harmattan: regular,500,600,700',
					'Headland One' => 'Headland One: regular',
					'Hedvig Letters Sans' => 'Hedvig Letters Sans: regular',
					'Hedvig Letters Serif' => 'Hedvig Letters Serif: regular',
					'Heebo' => 'Heebo: 100,200,300,regular,500,600,700,800,900',
					'Henny Penny' => 'Henny Penny: regular',
					'Hepta Slab' => 'Hepta Slab: 100,200,300,regular,500,600,700,800,900',
					'Herr Von Muellerhoff' => 'Herr Von Muellerhoff: regular',
					'Hi Melody' => 'Hi Melody: regular',
					'Hina Mincho' => 'Hina Mincho: regular',
					'Hind' => 'Hind: 300,regular,500,600,700',
					'Hind Guntur' => 'Hind Guntur: 300,regular,500,600,700',
					'Hind Madurai' => 'Hind Madurai: 300,regular,500,600,700',
					'Hind Siliguri' => 'Hind Siliguri: 300,regular,500,600,700',
					'Hind Vadodara' => 'Hind Vadodara: 300,regular,500,600,700',
					'Holtwood One SC' => 'Holtwood One SC: regular',
					'Homemade Apple' => 'Homemade Apple: regular',
					'Homenaje' => 'Homenaje: regular',
					'Honk' => 'Honk: regular',
					'Hubballi' => 'Hubballi: regular',
					'Hurricane' => 'Hurricane: regular',
					'IBM Plex Mono' => 'IBM Plex Mono: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'IBM Plex Sans' => 'IBM Plex Sans: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'IBM Plex Sans Arabic' => 'IBM Plex Sans Arabic: 100,200,300,regular,500,600,700',
					'IBM Plex Sans Condensed' => 'IBM Plex Sans Condensed: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'IBM Plex Sans Devanagari' => 'IBM Plex Sans Devanagari: 100,200,300,regular,500,600,700',
					'IBM Plex Sans Hebrew' => 'IBM Plex Sans Hebrew: 100,200,300,regular,500,600,700',
					'IBM Plex Sans JP' => 'IBM Plex Sans JP: 100,200,300,regular,500,600,700',
					'IBM Plex Sans KR' => 'IBM Plex Sans KR: 100,200,300,regular,500,600,700',
					'IBM Plex Sans Thai' => 'IBM Plex Sans Thai: 100,200,300,regular,500,600,700',
					'IBM Plex Sans Thai Looped' => 'IBM Plex Sans Thai Looped: 100,200,300,regular,500,600,700',
					'IBM Plex Serif' => 'IBM Plex Serif: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'IM Fell DW Pica' => 'IM Fell DW Pica: regular,italic',
					'IM Fell DW Pica SC' => 'IM Fell DW Pica SC: regular',
					'IM Fell Double Pica' => 'IM Fell Double Pica: regular,italic',
					'IM Fell Double Pica SC' => 'IM Fell Double Pica SC: regular',
					'IM Fell English' => 'IM Fell English: regular,italic',
					'IM Fell English SC' => 'IM Fell English SC: regular',
					'IM Fell French Canon' => 'IM Fell French Canon: regular,italic',
					'IM Fell French Canon SC' => 'IM Fell French Canon SC: regular',
					'IM Fell Great Primer' => 'IM Fell Great Primer: regular,italic',
					'IM Fell Great Primer SC' => 'IM Fell Great Primer SC: regular',
					'Ibarra Real Nova' => 'Ibarra Real Nova: regular,500,600,700,italic,500italic,600italic,700italic',
					'Iceberg' => 'Iceberg: regular',
					'Iceland' => 'Iceland: regular',
					'Imbue' => 'Imbue: 100,200,300,regular,500,600,700,800,900',
					'Imperial Script' => 'Imperial Script: regular',
					'Imprima' => 'Imprima: regular',
					'Inclusive Sans' => 'Inclusive Sans: regular,italic',
					'Inconsolata' => 'Inconsolata: 200,300,regular,500,600,700,800,900',
					'Inder' => 'Inder: regular',
					'Indie Flower' => 'Indie Flower: regular',
					'Ingrid Darling' => 'Ingrid Darling: regular',
					'Inika' => 'Inika: regular,700',
					'Inknut Antiqua' => 'Inknut Antiqua: 300,regular,500,600,700,800,900',
					'Inria Sans' => 'Inria Sans: 300,300italic,regular,italic,700,700italic',
					'Inria Serif' => 'Inria Serif: 300,300italic,regular,italic,700,700italic',
					'Inspiration' => 'Inspiration: regular',
					'Instrument Sans' => 'Instrument Sans: regular,500,600,700,italic,500italic,600italic,700italic',
					'Instrument Serif' => 'Instrument Serif: regular,italic',
					'Inter' => 'Inter: 100,200,300,regular,500,600,700,800,900',
					'Inter Tight' => 'Inter Tight: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Irish Grover' => 'Irish Grover: regular',
					'Island Moments' => 'Island Moments: regular',
					'Istok Web' => 'Istok Web: regular,italic,700,700italic',
					'Italiana' => 'Italiana: regular',
					'Italianno' => 'Italianno: regular',
					'Itim' => 'Itim: regular',
					'Jacques Francois' => 'Jacques Francois: regular',
					'Jacques Francois Shadow' => 'Jacques Francois Shadow: regular',
					'Jaldi' => 'Jaldi: regular,700',
					'JetBrains Mono' => 'JetBrains Mono: 100,200,300,regular,500,600,700,800,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic',
					'Jim Nightshade' => 'Jim Nightshade: regular',
					'Joan' => 'Joan: regular',
					'Jockey One' => 'Jockey One: regular',
					'Jolly Lodger' => 'Jolly Lodger: regular',
					'Jomhuria' => 'Jomhuria: regular',
					'Jomolhari' => 'Jomolhari: regular',
					'Josefin Sans' => 'Josefin Sans: 100,200,300,regular,500,600,700,100italic,200italic,300italic,italic,500italic,600italic,700italic',
					'Josefin Slab' => 'Josefin Slab: 100,200,300,regular,500,600,700,100italic,200italic,300italic,italic,500italic,600italic,700italic',
					'Jost' => 'Jost: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Joti One' => 'Joti One: regular',
					'Jua' => 'Jua: regular',
					'Judson' => 'Judson: regular,italic,700',
					'Julee' => 'Julee: regular',
					'Julius Sans One' => 'Julius Sans One: regular',
					'Junge' => 'Junge: regular',
					'Jura' => 'Jura: 300,regular,500,600,700',
					'Just Another Hand' => 'Just Another Hand: regular',
					'Just Me Again Down Here' => 'Just Me Again Down Here: regular',
					'K2D' => 'K2D: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic',
					'Kablammo' => 'Kablammo: regular',
					'Kadwa' => 'Kadwa: regular,700',
					'Kaisei Decol' => 'Kaisei Decol: regular,500,700',
					'Kaisei HarunoUmi' => 'Kaisei HarunoUmi: regular,500,700',
					'Kaisei Opti' => 'Kaisei Opti: regular,500,700',
					'Kaisei Tokumin' => 'Kaisei Tokumin: regular,500,700,800',
					'Kalam' => 'Kalam: 300,regular,700',
					'Kalnia' => 'Kalnia: 100,200,300,regular,500,600,700',
					'Kameron' => 'Kameron: regular,500,600,700',
					'Kanit' => 'Kanit: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Kantumruy Pro' => 'Kantumruy Pro: 100,200,300,regular,500,600,700,100italic,200italic,300italic,italic,500italic,600italic,700italic',
					'Karantina' => 'Karantina: 300,regular,700',
					'Karla' => 'Karla: 200,300,regular,500,600,700,800,200italic,300italic,italic,500italic,600italic,700italic,800italic',
					'Karma' => 'Karma: 300,regular,500,600,700',
					'Katibeh' => 'Katibeh: regular',
					'Kaushan Script' => 'Kaushan Script: regular',
					'Kavivanar' => 'Kavivanar: regular',
					'Kavoon' => 'Kavoon: regular',
					'Kay Pho Du' => 'Kay Pho Du: regular,500,600,700',
					'Kdam Thmor Pro' => 'Kdam Thmor Pro: regular',
					'Keania One' => 'Keania One: regular',
					'Kelly Slab' => 'Kelly Slab: regular',
					'Kenia' => 'Kenia: regular',
					'Khand' => 'Khand: 300,regular,500,600,700',
					'Khmer' => 'Khmer: regular',
					'Khula' => 'Khula: 300,regular,600,700,800',
					'Kings' => 'Kings: regular',
					'Kirang Haerang' => 'Kirang Haerang: regular',
					'Kite One' => 'Kite One: regular',
					'Kiwi Maru' => 'Kiwi Maru: 300,regular,500',
					'Klee One' => 'Klee One: regular,600',
					'Knewave' => 'Knewave: regular',
					'KoHo' => 'KoHo: 200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Kodchasan' => 'Kodchasan: 200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Koh Santepheap' => 'Koh Santepheap: 100,300,regular,700,900',
					'Kolker Brush' => 'Kolker Brush: regular',
					'Konkhmer Sleokchher' => 'Konkhmer Sleokchher: regular',
					'Kosugi' => 'Kosugi: regular',
					'Kosugi Maru' => 'Kosugi Maru: regular',
					'Kotta One' => 'Kotta One: regular',
					'Koulen' => 'Koulen: regular',
					'Kranky' => 'Kranky: regular',
					'Kreon' => 'Kreon: 300,regular,500,600,700',
					'Kristi' => 'Kristi: regular',
					'Krona One' => 'Krona One: regular',
					'Krub' => 'Krub: 200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Kufam' => 'Kufam: regular,500,600,700,800,900,italic,500italic,600italic,700italic,800italic,900italic',
					'Kulim Park' => 'Kulim Park: 200,200italic,300,300italic,regular,italic,600,600italic,700,700italic',
					'Kumar One' => 'Kumar One: regular',
					'Kumar One Outline' => 'Kumar One Outline: regular',
					'Kumbh Sans' => 'Kumbh Sans: 100,200,300,regular,500,600,700,800,900',
					'Kurale' => 'Kurale: regular',
					'La Belle Aurore' => 'La Belle Aurore: regular',
					'Labrada' => 'Labrada: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Lacquer' => 'Lacquer: regular',
					'Laila' => 'Laila: 300,regular,500,600,700',
					'Lakki Reddy' => 'Lakki Reddy: regular',
					'Lalezar' => 'Lalezar: regular',
					'Lancelot' => 'Lancelot: regular',
					'Langar' => 'Langar: regular',
					'Lateef' => 'Lateef: 200,300,regular,500,600,700,800',
					'Lato' => 'Lato: 100,100italic,300,300italic,regular,italic,700,700italic,900,900italic',
					'Lavishly Yours' => 'Lavishly Yours: regular',
					'League Gothic' => 'League Gothic: regular',
					'League Script' => 'League Script: regular',
					'League Spartan' => 'League Spartan: 100,200,300,regular,500,600,700,800,900',
					'Leckerli One' => 'Leckerli One: regular',
					'Ledger' => 'Ledger: regular',
					'Lekton' => 'Lekton: regular,italic,700',
					'Lemon' => 'Lemon: regular',
					'Lemonada' => 'Lemonada: 300,regular,500,600,700',
					'Lexend' => 'Lexend: 100,200,300,regular,500,600,700,800,900',
					'Lexend Deca' => 'Lexend Deca: 100,200,300,regular,500,600,700,800,900',
					'Lexend Exa' => 'Lexend Exa: 100,200,300,regular,500,600,700,800,900',
					'Lexend Giga' => 'Lexend Giga: 100,200,300,regular,500,600,700,800,900',
					'Lexend Mega' => 'Lexend Mega: 100,200,300,regular,500,600,700,800,900',
					'Lexend Peta' => 'Lexend Peta: 100,200,300,regular,500,600,700,800,900',
					'Lexend Tera' => 'Lexend Tera: 100,200,300,regular,500,600,700,800,900',
					'Lexend Zetta' => 'Lexend Zetta: 100,200,300,regular,500,600,700,800,900',
					'Libre Barcode 128' => 'Libre Barcode 128: regular',
					'Libre Barcode 128 Text' => 'Libre Barcode 128 Text: regular',
					'Libre Barcode 39' => 'Libre Barcode 39: regular',
					'Libre Barcode 39 Extended' => 'Libre Barcode 39 Extended: regular',
					'Libre Barcode 39 Extended Text' => 'Libre Barcode 39 Extended Text: regular',
					'Libre Barcode 39 Text' => 'Libre Barcode 39 Text: regular',
					'Libre Barcode EAN13 Text' => 'Libre Barcode EAN13 Text: regular',
					'Libre Baskerville' => 'Libre Baskerville: regular,italic,700',
					'Libre Bodoni' => 'Libre Bodoni: regular,500,600,700,italic,500italic,600italic,700italic',
					'Libre Caslon Display' => 'Libre Caslon Display: regular',
					'Libre Caslon Text' => 'Libre Caslon Text: regular,italic,700',
					'Libre Franklin' => 'Libre Franklin: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Licorice' => 'Licorice: regular',
					'Life Savers' => 'Life Savers: regular,700,800',
					'Lilita One' => 'Lilita One: regular',
					'Lily Script One' => 'Lily Script One: regular',
					'Limelight' => 'Limelight: regular',
					'Linden Hill' => 'Linden Hill: regular,italic',
					'Linefont' => 'Linefont: 100,200,300,regular,500,600,700,800,900',
					'Lisu Bosa' => 'Lisu Bosa: 200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Literata' => 'Literata: 200,300,regular,500,600,700,800,900,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Liu Jian Mao Cao' => 'Liu Jian Mao Cao: regular',
					'Livvic' => 'Livvic: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,900,900italic',
					'Lobster' => 'Lobster: regular',
					'Lobster Two' => 'Lobster Two: regular,italic,700,700italic',
					'Londrina Outline' => 'Londrina Outline: regular',
					'Londrina Shadow' => 'Londrina Shadow: regular',
					'Londrina Sketch' => 'Londrina Sketch: regular',
					'Londrina Solid' => 'Londrina Solid: 100,300,regular,900',
					'Long Cang' => 'Long Cang: regular',
					'Lora' => 'Lora: regular,500,600,700,italic,500italic,600italic,700italic',
					'Love Light' => 'Love Light: regular',
					'Love Ya Like A Sister' => 'Love Ya Like A Sister: regular',
					'Loved by the King' => 'Loved by the King: regular',
					'Lovers Quarrel' => 'Lovers Quarrel: regular',
					'Luckiest Guy' => 'Luckiest Guy: regular',
					'Lugrasimo' => 'Lugrasimo: regular',
					'Lumanosimo' => 'Lumanosimo: regular',
					'Lunasima' => 'Lunasima: regular,700',
					'Lusitana' => 'Lusitana: regular,700',
					'Lustria' => 'Lustria: regular',
					'Luxurious Roman' => 'Luxurious Roman: regular',
					'Luxurious Script' => 'Luxurious Script: regular',
					'M PLUS 1' => 'M PLUS 1: 100,200,300,regular,500,600,700,800,900',
					'M PLUS 1 Code' => 'M PLUS 1 Code: 100,200,300,regular,500,600,700',
					'M PLUS 1p' => 'M PLUS 1p: 100,300,regular,500,700,800,900',
					'M PLUS 2' => 'M PLUS 2: 100,200,300,regular,500,600,700,800,900',
					'M PLUS Code Latin' => 'M PLUS Code Latin: 100,200,300,regular,500,600,700',
					'M PLUS Rounded 1c' => 'M PLUS Rounded 1c: 100,300,regular,500,700,800,900',
					'Ma Shan Zheng' => 'Ma Shan Zheng: regular',
					'Macondo' => 'Macondo: regular',
					'Macondo Swash Caps' => 'Macondo Swash Caps: regular',
					'Mada' => 'Mada: 200,300,regular,500,600,700,800,900',
					'Magra' => 'Magra: regular,700',
					'Maiden Orange' => 'Maiden Orange: regular',
					'Maitree' => 'Maitree: 200,300,regular,500,600,700',
					'Major Mono Display' => 'Major Mono Display: regular',
					'Mako' => 'Mako: regular',
					'Mali' => 'Mali: 200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Mallanna' => 'Mallanna: regular',
					'Mandali' => 'Mandali: regular',
					'Manjari' => 'Manjari: 100,regular,700',
					'Manrope' => 'Manrope: 200,300,regular,500,600,700,800',
					'Mansalva' => 'Mansalva: regular',
					'Manuale' => 'Manuale: 300,regular,500,600,700,800,300italic,italic,500italic,600italic,700italic,800italic',
					'Marcellus' => 'Marcellus: regular',
					'Marcellus SC' => 'Marcellus SC: regular',
					'Marck Script' => 'Marck Script: regular',
					'Margarine' => 'Margarine: regular',
					'Marhey' => 'Marhey: 300,regular,500,600,700',
					'Markazi Text' => 'Markazi Text: regular,500,600,700',
					'Marko One' => 'Marko One: regular',
					'Marmelad' => 'Marmelad: regular',
					'Martel' => 'Martel: 200,300,regular,600,700,800,900',
					'Martel Sans' => 'Martel Sans: 200,300,regular,600,700,800,900',
					'Martian Mono' => 'Martian Mono: 100,200,300,regular,500,600,700,800',
					'Marvel' => 'Marvel: regular,italic,700,700italic',
					'Mate' => 'Mate: regular,italic',
					'Mate SC' => 'Mate SC: regular',
					'Material Icons' => 'Material Icons: regular',
					'Material Icons Outlined' => 'Material Icons Outlined: regular',
					'Material Icons Round' => 'Material Icons Round: regular',
					'Material Icons Sharp' => 'Material Icons Sharp: regular',
					'Material Icons Two Tone' => 'Material Icons Two Tone: regular',
					'Material Symbols Outlined' => 'Material Symbols Outlined: 100,200,300,regular,500,600,700',
					'Material Symbols Rounded' => 'Material Symbols Rounded: 100,200,300,regular,500,600,700',
					'Material Symbols Sharp' => 'Material Symbols Sharp: 100,200,300,regular,500,600,700',
					'Maven Pro' => 'Maven Pro: regular,500,600,700,800,900',
					'McLaren' => 'McLaren: regular',
					'Mea Culpa' => 'Mea Culpa: regular',
					'Meddon' => 'Meddon: regular',
					'MedievalSharp' => 'MedievalSharp: regular',
					'Medula One' => 'Medula One: regular',
					'Meera Inimai' => 'Meera Inimai: regular',
					'Megrim' => 'Megrim: regular',
					'Meie Script' => 'Meie Script: regular',
					'Meow Script' => 'Meow Script: regular',
					'Merienda' => 'Merienda: 300,regular,500,600,700,800,900',
					'Merriweather' => 'Merriweather: 300,300italic,regular,italic,700,700italic,900,900italic',
					'Merriweather Sans' => 'Merriweather Sans: 300,regular,500,600,700,800,300italic,italic,500italic,600italic,700italic,800italic',
					'Metal' => 'Metal: regular',
					'Metal Mania' => 'Metal Mania: regular',
					'Metamorphous' => 'Metamorphous: regular',
					'Metrophobic' => 'Metrophobic: regular',
					'Michroma' => 'Michroma: regular',
					'Milonga' => 'Milonga: regular',
					'Miltonian' => 'Miltonian: regular',
					'Miltonian Tattoo' => 'Miltonian Tattoo: regular',
					'Mina' => 'Mina: regular,700',
					'Mingzat' => 'Mingzat: regular',
					'Miniver' => 'Miniver: regular',
					'Miriam Libre' => 'Miriam Libre: regular,700',
					'Mirza' => 'Mirza: regular,500,600,700',
					'Miss Fajardose' => 'Miss Fajardose: regular',
					'Mitr' => 'Mitr: 200,300,regular,500,600,700',
					'Mochiy Pop One' => 'Mochiy Pop One: regular',
					'Mochiy Pop P One' => 'Mochiy Pop P One: regular',
					'Modak' => 'Modak: regular',
					'Modern Antiqua' => 'Modern Antiqua: regular',
					'Mogra' => 'Mogra: regular',
					'Mohave' => 'Mohave: 300,regular,500,600,700,300italic,italic,500italic,600italic,700italic',
					'Moirai One' => 'Moirai One: regular',
					'Molengo' => 'Molengo: regular',
					'Molle' => 'Molle: italic',
					'Monda' => 'Monda: regular,700',
					'Monofett' => 'Monofett: regular',
					'Monomaniac One' => 'Monomaniac One: regular',
					'Monoton' => 'Monoton: regular',
					'Monsieur La Doulaise' => 'Monsieur La Doulaise: regular',
					'Montaga' => 'Montaga: regular',
					'Montagu Slab' => 'Montagu Slab: 100,200,300,regular,500,600,700',
					'MonteCarlo' => 'MonteCarlo: regular',
					'Montez' => 'Montez: regular',
					'Montserrat' => 'Montserrat: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Montserrat Alternates' => 'Montserrat Alternates: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Montserrat Subrayada' => 'Montserrat Subrayada: regular,700',
					'Moo Lah Lah' => 'Moo Lah Lah: regular',
					'Mooli' => 'Mooli: regular',
					'Moon Dance' => 'Moon Dance: regular',
					'Moul' => 'Moul: regular',
					'Moulpali' => 'Moulpali: regular',
					'Mountains of Christmas' => 'Mountains of Christmas: regular,700',
					'Mouse Memoirs' => 'Mouse Memoirs: regular',
					'Mr Bedfort' => 'Mr Bedfort: regular',
					'Mr Dafoe' => 'Mr Dafoe: regular',
					'Mr De Haviland' => 'Mr De Haviland: regular',
					'Mrs Saint Delafield' => 'Mrs Saint Delafield: regular',
					'Mrs Sheppards' => 'Mrs Sheppards: regular',
					'Ms Madi' => 'Ms Madi: regular',
					'Mukta' => 'Mukta: 200,300,regular,500,600,700,800',
					'Mukta Mahee' => 'Mukta Mahee: 200,300,regular,500,600,700,800',
					'Mukta Malar' => 'Mukta Malar: 200,300,regular,500,600,700,800',
					'Mukta Vaani' => 'Mukta Vaani: 200,300,regular,500,600,700,800',
					'Mulish' => 'Mulish: 200,300,regular,500,600,700,800,900,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Murecho' => 'Murecho: 100,200,300,regular,500,600,700,800,900',
					'MuseoModerno' => 'MuseoModerno: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'My Soul' => 'My Soul: regular',
					'Mynerve' => 'Mynerve: regular',
					'Mystery Quest' => 'Mystery Quest: regular',
					'NTR' => 'NTR: regular',
					'Nabla' => 'Nabla: regular',
					'Nanum Brush Script' => 'Nanum Brush Script: regular',
					'Nanum Gothic' => 'Nanum Gothic: regular,700,800',
					'Nanum Gothic Coding' => 'Nanum Gothic Coding: regular,700',
					'Nanum Myeongjo' => 'Nanum Myeongjo: regular,700,800',
					'Nanum Pen Script' => 'Nanum Pen Script: regular',
					'Narnoor' => 'Narnoor: regular,500,600,700,800',
					'Neonderthaw' => 'Neonderthaw: regular',
					'Nerko One' => 'Nerko One: regular',
					'Neucha' => 'Neucha: regular',
					'Neuton' => 'Neuton: 200,300,regular,italic,700,800',
					'New Rocker' => 'New Rocker: regular',
					'New Tegomin' => 'New Tegomin: regular',
					'News Cycle' => 'News Cycle: regular,700',
					'Newsreader' => 'Newsreader: 200,300,regular,500,600,700,800,200italic,300italic,italic,500italic,600italic,700italic,800italic',
					'Niconne' => 'Niconne: regular',
					'Niramit' => 'Niramit: 200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Nixie One' => 'Nixie One: regular',
					'Nobile' => 'Nobile: regular,italic,500,500italic,700,700italic',
					'Nokora' => 'Nokora: 100,300,regular,700,900',
					'Norican' => 'Norican: regular',
					'Nosifer' => 'Nosifer: regular',
					'Notable' => 'Notable: regular',
					'Nothing You Could Do' => 'Nothing You Could Do: regular',
					'Noticia Text' => 'Noticia Text: regular,italic,700,700italic',
					'Noto Color Emoji' => 'Noto Color Emoji: regular',
					'Noto Emoji' => 'Noto Emoji: 300,regular,500,600,700',
					'Noto Kufi Arabic' => 'Noto Kufi Arabic: 100,200,300,regular,500,600,700,800,900',
					'Noto Music' => 'Noto Music: regular',
					'Noto Naskh Arabic' => 'Noto Naskh Arabic: regular,500,600,700',
					'Noto Nastaliq Urdu' => 'Noto Nastaliq Urdu: regular,500,600,700',
					'Noto Rashi Hebrew' => 'Noto Rashi Hebrew: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans' => 'Noto Sans: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Noto Sans Adlam' => 'Noto Sans Adlam: regular,500,600,700',
					'Noto Sans Adlam Unjoined' => 'Noto Sans Adlam Unjoined: regular,500,600,700',
					'Noto Sans Anatolian Hieroglyphs' => 'Noto Sans Anatolian Hieroglyphs: regular',
					'Noto Sans Arabic' => 'Noto Sans Arabic: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Armenian' => 'Noto Sans Armenian: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Avestan' => 'Noto Sans Avestan: regular',
					'Noto Sans Balinese' => 'Noto Sans Balinese: regular,500,600,700',
					'Noto Sans Bamum' => 'Noto Sans Bamum: regular,500,600,700',
					'Noto Sans Bassa Vah' => 'Noto Sans Bassa Vah: regular,500,600,700',
					'Noto Sans Batak' => 'Noto Sans Batak: regular',
					'Noto Sans Bengali' => 'Noto Sans Bengali: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Bhaiksuki' => 'Noto Sans Bhaiksuki: regular',
					'Noto Sans Brahmi' => 'Noto Sans Brahmi: regular',
					'Noto Sans Buginese' => 'Noto Sans Buginese: regular',
					'Noto Sans Buhid' => 'Noto Sans Buhid: regular',
					'Noto Sans Canadian Aboriginal' => 'Noto Sans Canadian Aboriginal: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Carian' => 'Noto Sans Carian: regular',
					'Noto Sans Caucasian Albanian' => 'Noto Sans Caucasian Albanian: regular',
					'Noto Sans Chakma' => 'Noto Sans Chakma: regular',
					'Noto Sans Cham' => 'Noto Sans Cham: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Cherokee' => 'Noto Sans Cherokee: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Chorasmian' => 'Noto Sans Chorasmian: regular',
					'Noto Sans Coptic' => 'Noto Sans Coptic: regular',
					'Noto Sans Cuneiform' => 'Noto Sans Cuneiform: regular',
					'Noto Sans Cypriot' => 'Noto Sans Cypriot: regular',
					'Noto Sans Cypro Minoan' => 'Noto Sans Cypro Minoan: regular',
					'Noto Sans Deseret' => 'Noto Sans Deseret: regular',
					'Noto Sans Devanagari' => 'Noto Sans Devanagari: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Display' => 'Noto Sans Display: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Noto Sans Duployan' => 'Noto Sans Duployan: regular,700',
					'Noto Sans Egyptian Hieroglyphs' => 'Noto Sans Egyptian Hieroglyphs: regular',
					'Noto Sans Elbasan' => 'Noto Sans Elbasan: regular',
					'Noto Sans Elymaic' => 'Noto Sans Elymaic: regular',
					'Noto Sans Ethiopic' => 'Noto Sans Ethiopic: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Georgian' => 'Noto Sans Georgian: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Glagolitic' => 'Noto Sans Glagolitic: regular',
					'Noto Sans Gothic' => 'Noto Sans Gothic: regular',
					'Noto Sans Grantha' => 'Noto Sans Grantha: regular',
					'Noto Sans Gujarati' => 'Noto Sans Gujarati: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Gunjala Gondi' => 'Noto Sans Gunjala Gondi: regular,500,600,700',
					'Noto Sans Gurmukhi' => 'Noto Sans Gurmukhi: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans HK' => 'Noto Sans HK: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Hanifi Rohingya' => 'Noto Sans Hanifi Rohingya: regular,500,600,700',
					'Noto Sans Hanunoo' => 'Noto Sans Hanunoo: regular',
					'Noto Sans Hatran' => 'Noto Sans Hatran: regular',
					'Noto Sans Hebrew' => 'Noto Sans Hebrew: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Imperial Aramaic' => 'Noto Sans Imperial Aramaic: regular',
					'Noto Sans Indic Siyaq Numbers' => 'Noto Sans Indic Siyaq Numbers: regular',
					'Noto Sans Inscriptional Pahlavi' => 'Noto Sans Inscriptional Pahlavi: regular',
					'Noto Sans Inscriptional Parthian' => 'Noto Sans Inscriptional Parthian: regular',
					'Noto Sans JP' => 'Noto Sans JP: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Javanese' => 'Noto Sans Javanese: regular,500,600,700',
					'Noto Sans KR' => 'Noto Sans KR: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Kaithi' => 'Noto Sans Kaithi: regular',
					'Noto Sans Kannada' => 'Noto Sans Kannada: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Kawi' => 'Noto Sans Kawi: regular,500,600,700',
					'Noto Sans Kayah Li' => 'Noto Sans Kayah Li: regular,500,600,700',
					'Noto Sans Kharoshthi' => 'Noto Sans Kharoshthi: regular',
					'Noto Sans Khmer' => 'Noto Sans Khmer: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Khojki' => 'Noto Sans Khojki: regular',
					'Noto Sans Khudawadi' => 'Noto Sans Khudawadi: regular',
					'Noto Sans Lao' => 'Noto Sans Lao: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Lao Looped' => 'Noto Sans Lao Looped: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Lepcha' => 'Noto Sans Lepcha: regular',
					'Noto Sans Limbu' => 'Noto Sans Limbu: regular',
					'Noto Sans Linear A' => 'Noto Sans Linear A: regular',
					'Noto Sans Linear B' => 'Noto Sans Linear B: regular',
					'Noto Sans Lisu' => 'Noto Sans Lisu: regular,500,600,700',
					'Noto Sans Lycian' => 'Noto Sans Lycian: regular',
					'Noto Sans Lydian' => 'Noto Sans Lydian: regular',
					'Noto Sans Mahajani' => 'Noto Sans Mahajani: regular',
					'Noto Sans Malayalam' => 'Noto Sans Malayalam: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Mandaic' => 'Noto Sans Mandaic: regular',
					'Noto Sans Manichaean' => 'Noto Sans Manichaean: regular',
					'Noto Sans Marchen' => 'Noto Sans Marchen: regular',
					'Noto Sans Masaram Gondi' => 'Noto Sans Masaram Gondi: regular',
					'Noto Sans Math' => 'Noto Sans Math: regular',
					'Noto Sans Mayan Numerals' => 'Noto Sans Mayan Numerals: regular',
					'Noto Sans Medefaidrin' => 'Noto Sans Medefaidrin: regular,500,600,700',
					'Noto Sans Meetei Mayek' => 'Noto Sans Meetei Mayek: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Mende Kikakui' => 'Noto Sans Mende Kikakui: regular',
					'Noto Sans Meroitic' => 'Noto Sans Meroitic: regular',
					'Noto Sans Miao' => 'Noto Sans Miao: regular',
					'Noto Sans Modi' => 'Noto Sans Modi: regular',
					'Noto Sans Mongolian' => 'Noto Sans Mongolian: regular',
					'Noto Sans Mono' => 'Noto Sans Mono: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Mro' => 'Noto Sans Mro: regular',
					'Noto Sans Multani' => 'Noto Sans Multani: regular',
					'Noto Sans Myanmar' => 'Noto Sans Myanmar: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans NKo' => 'Noto Sans NKo: regular',
					'Noto Sans NKo Unjoined' => 'Noto Sans NKo Unjoined: regular,500,600,700',
					'Noto Sans Nabataean' => 'Noto Sans Nabataean: regular',
					'Noto Sans Nag Mundari' => 'Noto Sans Nag Mundari: regular,500,600,700',
					'Noto Sans Nandinagari' => 'Noto Sans Nandinagari: regular',
					'Noto Sans New Tai Lue' => 'Noto Sans New Tai Lue: regular,500,600,700',
					'Noto Sans Newa' => 'Noto Sans Newa: regular',
					'Noto Sans Nushu' => 'Noto Sans Nushu: regular',
					'Noto Sans Ogham' => 'Noto Sans Ogham: regular',
					'Noto Sans Ol Chiki' => 'Noto Sans Ol Chiki: regular,500,600,700',
					'Noto Sans Old Hungarian' => 'Noto Sans Old Hungarian: regular',
					'Noto Sans Old Italic' => 'Noto Sans Old Italic: regular',
					'Noto Sans Old North Arabian' => 'Noto Sans Old North Arabian: regular',
					'Noto Sans Old Permic' => 'Noto Sans Old Permic: regular',
					'Noto Sans Old Persian' => 'Noto Sans Old Persian: regular',
					'Noto Sans Old Sogdian' => 'Noto Sans Old Sogdian: regular',
					'Noto Sans Old South Arabian' => 'Noto Sans Old South Arabian: regular',
					'Noto Sans Old Turkic' => 'Noto Sans Old Turkic: regular',
					'Noto Sans Oriya' => 'Noto Sans Oriya: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Osage' => 'Noto Sans Osage: regular',
					'Noto Sans Osmanya' => 'Noto Sans Osmanya: regular',
					'Noto Sans Pahawh Hmong' => 'Noto Sans Pahawh Hmong: regular',
					'Noto Sans Palmyrene' => 'Noto Sans Palmyrene: regular',
					'Noto Sans Pau Cin Hau' => 'Noto Sans Pau Cin Hau: regular',
					'Noto Sans Phags Pa' => 'Noto Sans Phags Pa: regular',
					'Noto Sans Phoenician' => 'Noto Sans Phoenician: regular',
					'Noto Sans Psalter Pahlavi' => 'Noto Sans Psalter Pahlavi: regular',
					'Noto Sans Rejang' => 'Noto Sans Rejang: regular',
					'Noto Sans Runic' => 'Noto Sans Runic: regular',
					'Noto Sans SC' => 'Noto Sans SC: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Samaritan' => 'Noto Sans Samaritan: regular',
					'Noto Sans Saurashtra' => 'Noto Sans Saurashtra: regular',
					'Noto Sans Sharada' => 'Noto Sans Sharada: regular',
					'Noto Sans Shavian' => 'Noto Sans Shavian: regular',
					'Noto Sans Siddham' => 'Noto Sans Siddham: regular',
					'Noto Sans SignWriting' => 'Noto Sans SignWriting: regular',
					'Noto Sans Sinhala' => 'Noto Sans Sinhala: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Sogdian' => 'Noto Sans Sogdian: regular',
					'Noto Sans Sora Sompeng' => 'Noto Sans Sora Sompeng: regular,500,600,700',
					'Noto Sans Soyombo' => 'Noto Sans Soyombo: regular',
					'Noto Sans Sundanese' => 'Noto Sans Sundanese: regular,500,600,700',
					'Noto Sans Syloti Nagri' => 'Noto Sans Syloti Nagri: regular',
					'Noto Sans Symbols' => 'Noto Sans Symbols: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Symbols 2' => 'Noto Sans Symbols 2: regular',
					'Noto Sans Syriac' => 'Noto Sans Syriac: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Syriac Eastern' => 'Noto Sans Syriac Eastern: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans TC' => 'Noto Sans TC: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Tagalog' => 'Noto Sans Tagalog: regular',
					'Noto Sans Tagbanwa' => 'Noto Sans Tagbanwa: regular',
					'Noto Sans Tai Le' => 'Noto Sans Tai Le: regular',
					'Noto Sans Tai Tham' => 'Noto Sans Tai Tham: regular,500,600,700',
					'Noto Sans Tai Viet' => 'Noto Sans Tai Viet: regular',
					'Noto Sans Takri' => 'Noto Sans Takri: regular',
					'Noto Sans Tamil' => 'Noto Sans Tamil: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Tamil Supplement' => 'Noto Sans Tamil Supplement: regular',
					'Noto Sans Tangsa' => 'Noto Sans Tangsa: regular,500,600,700',
					'Noto Sans Telugu' => 'Noto Sans Telugu: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Thaana' => 'Noto Sans Thaana: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Thai' => 'Noto Sans Thai: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Thai Looped' => 'Noto Sans Thai Looped: 100,200,300,regular,500,600,700,800,900',
					'Noto Sans Tifinagh' => 'Noto Sans Tifinagh: regular',
					'Noto Sans Tirhuta' => 'Noto Sans Tirhuta: regular',
					'Noto Sans Ugaritic' => 'Noto Sans Ugaritic: regular',
					'Noto Sans Vai' => 'Noto Sans Vai: regular',
					'Noto Sans Vithkuqi' => 'Noto Sans Vithkuqi: regular,500,600,700',
					'Noto Sans Wancho' => 'Noto Sans Wancho: regular',
					'Noto Sans Warang Citi' => 'Noto Sans Warang Citi: regular',
					'Noto Sans Yi' => 'Noto Sans Yi: regular',
					'Noto Sans Zanabazar Square' => 'Noto Sans Zanabazar Square: regular',
					'Noto Serif' => 'Noto Serif: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Noto Serif Ahom' => 'Noto Serif Ahom: regular',
					'Noto Serif Armenian' => 'Noto Serif Armenian: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif Balinese' => 'Noto Serif Balinese: regular',
					'Noto Serif Bengali' => 'Noto Serif Bengali: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif Devanagari' => 'Noto Serif Devanagari: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif Display' => 'Noto Serif Display: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Noto Serif Dogra' => 'Noto Serif Dogra: regular',
					'Noto Serif Ethiopic' => 'Noto Serif Ethiopic: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif Georgian' => 'Noto Serif Georgian: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif Grantha' => 'Noto Serif Grantha: regular',
					'Noto Serif Gujarati' => 'Noto Serif Gujarati: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif Gurmukhi' => 'Noto Serif Gurmukhi: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif HK' => 'Noto Serif HK: 200,300,regular,500,600,700,800,900',
					'Noto Serif Hebrew' => 'Noto Serif Hebrew: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif JP' => 'Noto Serif JP: 200,300,regular,500,600,700,900',
					'Noto Serif KR' => 'Noto Serif KR: 200,300,regular,500,600,700,900',
					'Noto Serif Kannada' => 'Noto Serif Kannada: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif Khitan Small Script' => 'Noto Serif Khitan Small Script: regular',
					'Noto Serif Khmer' => 'Noto Serif Khmer: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif Khojki' => 'Noto Serif Khojki: regular,500,600,700',
					'Noto Serif Lao' => 'Noto Serif Lao: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif Makasar' => 'Noto Serif Makasar: regular',
					'Noto Serif Malayalam' => 'Noto Serif Malayalam: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif Myanmar' => 'Noto Serif Myanmar: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif NP Hmong' => 'Noto Serif NP Hmong: regular,500,600,700',
					'Noto Serif Old Uyghur' => 'Noto Serif Old Uyghur: regular',
					'Noto Serif Oriya' => 'Noto Serif Oriya: regular,500,600,700',
					'Noto Serif Ottoman Siyaq' => 'Noto Serif Ottoman Siyaq: regular',
					'Noto Serif SC' => 'Noto Serif SC: 200,300,regular,500,600,700,900',
					'Noto Serif Sinhala' => 'Noto Serif Sinhala: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif TC' => 'Noto Serif TC: 200,300,regular,500,600,700,900',
					'Noto Serif Tamil' => 'Noto Serif Tamil: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Noto Serif Tangut' => 'Noto Serif Tangut: regular',
					'Noto Serif Telugu' => 'Noto Serif Telugu: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif Thai' => 'Noto Serif Thai: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif Tibetan' => 'Noto Serif Tibetan: 100,200,300,regular,500,600,700,800,900',
					'Noto Serif Toto' => 'Noto Serif Toto: regular,500,600,700',
					'Noto Serif Vithkuqi' => 'Noto Serif Vithkuqi: regular,500,600,700',
					'Noto Serif Yezidi' => 'Noto Serif Yezidi: regular,500,600,700',
					'Noto Traditional Nushu' => 'Noto Traditional Nushu: 300,regular,500,600,700',
					'Nova Cut' => 'Nova Cut: regular',
					'Nova Flat' => 'Nova Flat: regular',
					'Nova Mono' => 'Nova Mono: regular',
					'Nova Oval' => 'Nova Oval: regular',
					'Nova Round' => 'Nova Round: regular',
					'Nova Script' => 'Nova Script: regular',
					'Nova Slim' => 'Nova Slim: regular',
					'Nova Square' => 'Nova Square: regular',
					'Numans' => 'Numans: regular',
					'Nunito' => 'Nunito: 200,300,regular,500,600,700,800,900,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Nunito Sans' => 'Nunito Sans: 200,300,regular,500,600,700,800,900,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Nuosu SIL' => 'Nuosu SIL: regular',
					'Odibee Sans' => 'Odibee Sans: regular',
					'Odor Mean Chey' => 'Odor Mean Chey: regular',
					'Offside' => 'Offside: regular',
					'Oi' => 'Oi: regular',
					'Old Standard TT' => 'Old Standard TT: regular,italic,700',
					'Oldenburg' => 'Oldenburg: regular',
					'Ole' => 'Ole: regular',
					'Oleo Script' => 'Oleo Script: regular,700',
					'Oleo Script Swash Caps' => 'Oleo Script Swash Caps: regular,700',
					'Onest' => 'Onest: 100,200,300,regular,500,600,700,800,900',
					'Oooh Baby' => 'Oooh Baby: regular',
					'Open Sans' => 'Open Sans: 300,regular,500,600,700,800,300italic,italic,500italic,600italic,700italic,800italic',
					'Oranienbaum' => 'Oranienbaum: regular',
					'Orbit' => 'Orbit: regular',
					'Orbitron' => 'Orbitron: regular,500,600,700,800,900',
					'Oregano' => 'Oregano: regular,italic',
					'Orelega One' => 'Orelega One: regular',
					'Orienta' => 'Orienta: regular',
					'Original Surfer' => 'Original Surfer: regular',
					'Oswald' => 'Oswald: 200,300,regular,500,600,700',
					'Outfit' => 'Outfit: 100,200,300,regular,500,600,700,800,900',
					'Over the Rainbow' => 'Over the Rainbow: regular',
					'Overlock' => 'Overlock: regular,italic,700,700italic,900,900italic',
					'Overlock SC' => 'Overlock SC: regular',
					'Overpass' => 'Overpass: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Overpass Mono' => 'Overpass Mono: 300,regular,500,600,700',
					'Ovo' => 'Ovo: regular',
					'Oxanium' => 'Oxanium: 200,300,regular,500,600,700,800',
					'Oxygen' => 'Oxygen: 300,regular,700',
					'Oxygen Mono' => 'Oxygen Mono: regular',
					'PT Mono' => 'PT Mono: regular',
					'PT Sans' => 'PT Sans: regular,italic,700,700italic',
					'PT Sans Caption' => 'PT Sans Caption: regular,700',
					'PT Sans Narrow' => 'PT Sans Narrow: regular,700',
					'PT Serif' => 'PT Serif: regular,italic,700,700italic',
					'PT Serif Caption' => 'PT Serif Caption: regular,italic',
					'Pacifico' => 'Pacifico: regular',
					'Padauk' => 'Padauk: regular,700',
					'Padyakke Expanded One' => 'Padyakke Expanded One: regular',
					'Palanquin' => 'Palanquin: 100,200,300,regular,500,600,700',
					'Palanquin Dark' => 'Palanquin Dark: regular,500,600,700',
					'Palette Mosaic' => 'Palette Mosaic: regular',
					'Pangolin' => 'Pangolin: regular',
					'Paprika' => 'Paprika: regular',
					'Parisienne' => 'Parisienne: regular',
					'Passero One' => 'Passero One: regular',
					'Passion One' => 'Passion One: regular,700,900',
					'Passions Conflict' => 'Passions Conflict: regular',
					'Pathway Extreme' => 'Pathway Extreme: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Pathway Gothic One' => 'Pathway Gothic One: regular',
					'Patrick Hand' => 'Patrick Hand: regular',
					'Patrick Hand SC' => 'Patrick Hand SC: regular',
					'Pattaya' => 'Pattaya: regular',
					'Patua One' => 'Patua One: regular',
					'Pavanam' => 'Pavanam: regular',
					'Paytone One' => 'Paytone One: regular',
					'Peddana' => 'Peddana: regular',
					'Peralta' => 'Peralta: regular',
					'Permanent Marker' => 'Permanent Marker: regular',
					'Petemoss' => 'Petemoss: regular',
					'Petit Formal Script' => 'Petit Formal Script: regular',
					'Petrona' => 'Petrona: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Philosopher' => 'Philosopher: regular,italic,700,700italic',
					'Phudu' => 'Phudu: 300,regular,500,600,700,800,900',
					'Piazzolla' => 'Piazzolla: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Piedra' => 'Piedra: regular',
					'Pinyon Script' => 'Pinyon Script: regular',
					'Pirata One' => 'Pirata One: regular',
					'Pixelify Sans' => 'Pixelify Sans: regular,500,600,700',
					'Plaster' => 'Plaster: regular',
					'Play' => 'Play: regular,700',
					'Playball' => 'Playball: regular',
					'Playfair' => 'Playfair: 300,regular,500,600,700,800,900,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Playfair Display' => 'Playfair Display: regular,500,600,700,800,900,italic,500italic,600italic,700italic,800italic,900italic',
					'Playfair Display SC' => 'Playfair Display SC: regular,italic,700,700italic,900,900italic',
					'Playpen Sans' => 'Playpen Sans: 100,200,300,regular,500,600,700,800',
					'Plus Jakarta Sans' => 'Plus Jakarta Sans: 200,300,regular,500,600,700,800,200italic,300italic,italic,500italic,600italic,700italic,800italic',
					'Podkova' => 'Podkova: regular,500,600,700,800',
					'Poiret One' => 'Poiret One: regular',
					'Poller One' => 'Poller One: regular',
					'Poltawski Nowy' => 'Poltawski Nowy: regular,500,600,700,italic,500italic,600italic,700italic',
					'Poly' => 'Poly: regular,italic',
					'Pompiere' => 'Pompiere: regular',
					'Pontano Sans' => 'Pontano Sans: 300,regular,500,600,700',
					'Poor Story' => 'Poor Story: regular',
					'Poppins' => 'Poppins: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Port Lligat Sans' => 'Port Lligat Sans: regular',
					'Port Lligat Slab' => 'Port Lligat Slab: regular',
					'Potta One' => 'Potta One: regular',
					'Pragati Narrow' => 'Pragati Narrow: regular,700',
					'Praise' => 'Praise: regular',
					'Prata' => 'Prata: regular',
					'Preahvihear' => 'Preahvihear: regular',
					'Press Start 2P' => 'Press Start 2P: regular',
					'Pridi' => 'Pridi: 200,300,regular,500,600,700',
					'Princess Sofia' => 'Princess Sofia: regular',
					'Prociono' => 'Prociono: regular',
					'Prompt' => 'Prompt: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Prosto One' => 'Prosto One: regular',
					'Proza Libre' => 'Proza Libre: regular,italic,500,500italic,600,600italic,700,700italic,800,800italic',
					'Public Sans' => 'Public Sans: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Puppies Play' => 'Puppies Play: regular',
					'Puritan' => 'Puritan: regular,italic,700,700italic',
					'Purple Purse' => 'Purple Purse: regular',
					'Qahiri' => 'Qahiri: regular',
					'Quando' => 'Quando: regular',
					'Quantico' => 'Quantico: regular,italic,700,700italic',
					'Quattrocento' => 'Quattrocento: regular,700',
					'Quattrocento Sans' => 'Quattrocento Sans: regular,italic,700,700italic',
					'Questrial' => 'Questrial: regular',
					'Quicksand' => 'Quicksand: 300,regular,500,600,700',
					'Quintessential' => 'Quintessential: regular',
					'Qwigley' => 'Qwigley: regular',
					'Qwitcher Grypen' => 'Qwitcher Grypen: regular,700',
					'REM' => 'REM: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Racing Sans One' => 'Racing Sans One: regular',
					'Radio Canada' => 'Radio Canada: 300,regular,500,600,700,300italic,italic,500italic,600italic,700italic',
					'Radley' => 'Radley: regular,italic',
					'Rajdhani' => 'Rajdhani: 300,regular,500,600,700',
					'Rakkas' => 'Rakkas: regular',
					'Raleway' => 'Raleway: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Raleway Dots' => 'Raleway Dots: regular',
					'Ramabhadra' => 'Ramabhadra: regular',
					'Ramaraja' => 'Ramaraja: regular',
					'Rambla' => 'Rambla: regular,italic,700,700italic',
					'Rammetto One' => 'Rammetto One: regular',
					'Rampart One' => 'Rampart One: regular',
					'Ranchers' => 'Ranchers: regular',
					'Rancho' => 'Rancho: regular',
					'Ranga' => 'Ranga: regular,700',
					'Rasa' => 'Rasa: 300,regular,500,600,700,300italic,italic,500italic,600italic,700italic',
					'Rationale' => 'Rationale: regular',
					'Ravi Prakash' => 'Ravi Prakash: regular',
					'Readex Pro' => 'Readex Pro: 200,300,regular,500,600,700',
					'Recursive' => 'Recursive: 300,regular,500,600,700,800,900',
					'Red Hat Display' => 'Red Hat Display: 300,regular,500,600,700,800,900,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Red Hat Mono' => 'Red Hat Mono: 300,regular,500,600,700,300italic,italic,500italic,600italic,700italic',
					'Red Hat Text' => 'Red Hat Text: 300,regular,500,600,700,300italic,italic,500italic,600italic,700italic',
					'Red Rose' => 'Red Rose: 300,regular,500,600,700',
					'Redacted' => 'Redacted: regular',
					'Redacted Script' => 'Redacted Script: 300,regular,700',
					'Redressed' => 'Redressed: regular',
					'Reem Kufi' => 'Reem Kufi: regular,500,600,700',
					'Reem Kufi Fun' => 'Reem Kufi Fun: regular,500,600,700',
					'Reem Kufi Ink' => 'Reem Kufi Ink: regular',
					'Reenie Beanie' => 'Reenie Beanie: regular',
					'Reggae One' => 'Reggae One: regular',
					'Rethink Sans' => 'Rethink Sans: regular,500,600,700,800,italic,500italic,600italic,700italic,800italic',
					'Revalia' => 'Revalia: regular',
					'Rhodium Libre' => 'Rhodium Libre: regular',
					'Ribeye' => 'Ribeye: regular',
					'Ribeye Marrow' => 'Ribeye Marrow: regular',
					'Righteous' => 'Righteous: regular',
					'Risque' => 'Risque: regular',
					'Road Rage' => 'Road Rage: regular',
					'Roboto' => 'Roboto: 100,100italic,300,300italic,regular,italic,500,500italic,700,700italic,900,900italic',
					'Roboto Condensed' => 'Roboto Condensed: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Roboto Flex' => 'Roboto Flex: regular',
					'Roboto Mono' => 'Roboto Mono: 100,200,300,regular,500,600,700,100italic,200italic,300italic,italic,500italic,600italic,700italic',
					'Roboto Serif' => 'Roboto Serif: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Roboto Slab' => 'Roboto Slab: 100,200,300,regular,500,600,700,800,900',
					'Rochester' => 'Rochester: regular',
					'Rock 3D' => 'Rock 3D: regular',
					'Rock Salt' => 'Rock Salt: regular',
					'RocknRoll One' => 'RocknRoll One: regular',
					'Rokkitt' => 'Rokkitt: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Romanesco' => 'Romanesco: regular',
					'Ropa Sans' => 'Ropa Sans: regular,italic',
					'Rosario' => 'Rosario: 300,regular,500,600,700,300italic,italic,500italic,600italic,700italic',
					'Rosarivo' => 'Rosarivo: regular,italic',
					'Rouge Script' => 'Rouge Script: regular',
					'Rowdies' => 'Rowdies: 300,regular,700',
					'Rozha One' => 'Rozha One: regular',
					'Rubik' => 'Rubik: 300,regular,500,600,700,800,900,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Rubik 80s Fade' => 'Rubik 80s Fade: regular',
					'Rubik Beastly' => 'Rubik Beastly: regular',
					'Rubik Broken Fax' => 'Rubik Broken Fax: regular',
					'Rubik Bubbles' => 'Rubik Bubbles: regular',
					'Rubik Burned' => 'Rubik Burned: regular',
					'Rubik Dirt' => 'Rubik Dirt: regular',
					'Rubik Distressed' => 'Rubik Distressed: regular',
					'Rubik Doodle Shadow' => 'Rubik Doodle Shadow: regular',
					'Rubik Doodle Triangles' => 'Rubik Doodle Triangles: regular',
					'Rubik Gemstones' => 'Rubik Gemstones: regular',
					'Rubik Glitch' => 'Rubik Glitch: regular',
					'Rubik Glitch Pop' => 'Rubik Glitch Pop: regular',
					'Rubik Iso' => 'Rubik Iso: regular',
					'Rubik Lines' => 'Rubik Lines: regular',
					'Rubik Maps' => 'Rubik Maps: regular',
					'Rubik Marker Hatch' => 'Rubik Marker Hatch: regular',
					'Rubik Maze' => 'Rubik Maze: regular',
					'Rubik Microbe' => 'Rubik Microbe: regular',
					'Rubik Mono One' => 'Rubik Mono One: regular',
					'Rubik Moonrocks' => 'Rubik Moonrocks: regular',
					'Rubik Pixels' => 'Rubik Pixels: regular',
					'Rubik Puddles' => 'Rubik Puddles: regular',
					'Rubik Scribble' => 'Rubik Scribble: regular',
					'Rubik Spray Paint' => 'Rubik Spray Paint: regular',
					'Rubik Storm' => 'Rubik Storm: regular',
					'Rubik Vinyl' => 'Rubik Vinyl: regular',
					'Rubik Wet Paint' => 'Rubik Wet Paint: regular',
					'Ruda' => 'Ruda: regular,500,600,700,800,900',
					'Rufina' => 'Rufina: regular,700',
					'Ruge Boogie' => 'Ruge Boogie: regular',
					'Ruluko' => 'Ruluko: regular',
					'Rum Raisin' => 'Rum Raisin: regular',
					'Ruslan Display' => 'Ruslan Display: regular',
					'Russo One' => 'Russo One: regular',
					'Ruthie' => 'Ruthie: regular',
					'Ruwudu' => 'Ruwudu: regular,500,600,700',
					'Rye' => 'Rye: regular',
					'STIX Two Text' => 'STIX Two Text: regular,500,600,700,italic,500italic,600italic,700italic',
					'Sacramento' => 'Sacramento: regular',
					'Sahitya' => 'Sahitya: regular,700',
					'Sail' => 'Sail: regular',
					'Saira' => 'Saira: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Saira Condensed' => 'Saira Condensed: 100,200,300,regular,500,600,700,800,900',
					'Saira Extra Condensed' => 'Saira Extra Condensed: 100,200,300,regular,500,600,700,800,900',
					'Saira Semi Condensed' => 'Saira Semi Condensed: 100,200,300,regular,500,600,700,800,900',
					'Saira Stencil One' => 'Saira Stencil One: regular',
					'Salsa' => 'Salsa: regular',
					'Sanchez' => 'Sanchez: regular,italic',
					'Sancreek' => 'Sancreek: regular',
					'Sansita' => 'Sansita: regular,italic,700,700italic,800,800italic,900,900italic',
					'Sansita Swashed' => 'Sansita Swashed: 300,regular,500,600,700,800,900',
					'Sarabun' => 'Sarabun: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic',
					'Sarala' => 'Sarala: regular,700',
					'Sarina' => 'Sarina: regular',
					'Sarpanch' => 'Sarpanch: regular,500,600,700,800,900',
					'Sassy Frass' => 'Sassy Frass: regular',
					'Satisfy' => 'Satisfy: regular',
					'Sawarabi Gothic' => 'Sawarabi Gothic: regular',
					'Sawarabi Mincho' => 'Sawarabi Mincho: regular',
					'Scada' => 'Scada: regular,italic,700,700italic',
					'Scheherazade New' => 'Scheherazade New: regular,500,600,700',
					'Schibsted Grotesk' => 'Schibsted Grotesk: regular,500,600,700,800,900,italic,500italic,600italic,700italic,800italic,900italic',
					'Schoolbell' => 'Schoolbell: regular',
					'Scope One' => 'Scope One: regular',
					'Seaweed Script' => 'Seaweed Script: regular',
					'Secular One' => 'Secular One: regular',
					'Sedgwick Ave' => 'Sedgwick Ave: regular',
					'Sedgwick Ave Display' => 'Sedgwick Ave Display: regular',
					'Sen' => 'Sen: regular,500,600,700,800',
					'Send Flowers' => 'Send Flowers: regular',
					'Sevillana' => 'Sevillana: regular',
					'Seymour One' => 'Seymour One: regular',
					'Shadows Into Light' => 'Shadows Into Light: regular',
					'Shadows Into Light Two' => 'Shadows Into Light Two: regular',
					'Shalimar' => 'Shalimar: regular',
					'Shantell Sans' => 'Shantell Sans: 300,regular,500,600,700,800,300italic,italic,500italic,600italic,700italic,800italic',
					'Shanti' => 'Shanti: regular',
					'Share' => 'Share: regular,italic,700,700italic',
					'Share Tech' => 'Share Tech: regular',
					'Share Tech Mono' => 'Share Tech Mono: regular',
					'Shippori Antique' => 'Shippori Antique: regular',
					'Shippori Antique B1' => 'Shippori Antique B1: regular',
					'Shippori Mincho' => 'Shippori Mincho: regular,500,600,700,800',
					'Shippori Mincho B1' => 'Shippori Mincho B1: regular,500,600,700,800',
					'Shizuru' => 'Shizuru: regular',
					'Shojumaru' => 'Shojumaru: regular',
					'Short Stack' => 'Short Stack: regular',
					'Shrikhand' => 'Shrikhand: regular',
					'Siemreap' => 'Siemreap: regular',
					'Sigmar' => 'Sigmar: regular',
					'Sigmar One' => 'Sigmar One: regular',
					'Signika' => 'Signika: 300,regular,500,600,700',
					'Signika Negative' => 'Signika Negative: 300,regular,500,600,700',
					'Silkscreen' => 'Silkscreen: regular,700',
					'Simonetta' => 'Simonetta: regular,italic,900,900italic',
					'Single Day' => 'Single Day: regular',
					'Sintony' => 'Sintony: regular,700',
					'Sirin Stencil' => 'Sirin Stencil: regular',
					'Six Caps' => 'Six Caps: regular',
					'Sixtyfour' => 'Sixtyfour: regular',
					'Skranji' => 'Skranji: regular,700',
					'Slabo 13px' => 'Slabo 13px: regular',
					'Slabo 27px' => 'Slabo 27px: regular',
					'Slackey' => 'Slackey: regular',
					'Slackside One' => 'Slackside One: regular',
					'Smokum' => 'Smokum: regular',
					'Smooch' => 'Smooch: regular',
					'Smooch Sans' => 'Smooch Sans: 100,200,300,regular,500,600,700,800,900',
					'Smythe' => 'Smythe: regular',
					'Sniglet' => 'Sniglet: regular,800',
					'Snippet' => 'Snippet: regular',
					'Snowburst One' => 'Snowburst One: regular',
					'Sofadi One' => 'Sofadi One: regular',
					'Sofia' => 'Sofia: regular',
					'Sofia Sans' => 'Sofia Sans: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Sofia Sans Condensed' => 'Sofia Sans Condensed: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Sofia Sans Extra Condensed' => 'Sofia Sans Extra Condensed: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Sofia Sans Semi Condensed' => 'Sofia Sans Semi Condensed: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Solitreo' => 'Solitreo: regular',
					'Solway' => 'Solway: 300,regular,500,700,800',
					'Sometype Mono' => 'Sometype Mono: regular,500,600,700,italic,500italic,600italic,700italic',
					'Song Myung' => 'Song Myung: regular',
					'Sono' => 'Sono: 200,300,regular,500,600,700,800',
					'Sonsie One' => 'Sonsie One: regular',
					'Sora' => 'Sora: 100,200,300,regular,500,600,700,800',
					'Sorts Mill Goudy' => 'Sorts Mill Goudy: regular,italic',
					'Source Code Pro' => 'Source Code Pro: 200,300,regular,500,600,700,800,900,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Source Sans 3' => 'Source Sans 3: 200,300,regular,500,600,700,800,900,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Source Serif 4' => 'Source Serif 4: 200,300,regular,500,600,700,800,900,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Space Grotesk' => 'Space Grotesk: 300,regular,500,600,700',
					'Space Mono' => 'Space Mono: regular,italic,700,700italic',
					'Special Elite' => 'Special Elite: regular',
					'Spectral' => 'Spectral: 200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic',
					'Spectral SC' => 'Spectral SC: 200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic',
					'Spicy Rice' => 'Spicy Rice: regular',
					'Spinnaker' => 'Spinnaker: regular',
					'Spirax' => 'Spirax: regular',
					'Splash' => 'Splash: regular',
					'Spline Sans' => 'Spline Sans: 300,regular,500,600,700',
					'Spline Sans Mono' => 'Spline Sans Mono: 300,regular,500,600,700,300italic,italic,500italic,600italic,700italic',
					'Squada One' => 'Squada One: regular',
					'Square Peg' => 'Square Peg: regular',
					'Sree Krushnadevaraya' => 'Sree Krushnadevaraya: regular',
					'Sriracha' => 'Sriracha: regular',
					'Srisakdi' => 'Srisakdi: regular,700',
					'Staatliches' => 'Staatliches: regular',
					'Stalemate' => 'Stalemate: regular',
					'Stalinist One' => 'Stalinist One: regular',
					'Stardos Stencil' => 'Stardos Stencil: regular,700',
					'Stick' => 'Stick: regular',
					'Stick No Bills' => 'Stick No Bills: 200,300,regular,500,600,700,800',
					'Stint Ultra Condensed' => 'Stint Ultra Condensed: regular',
					'Stint Ultra Expanded' => 'Stint Ultra Expanded: regular',
					'Stoke' => 'Stoke: 300,regular',
					'Strait' => 'Strait: regular',
					'Style Script' => 'Style Script: regular',
					'Stylish' => 'Stylish: regular',
					'Sue Ellen Francisco' => 'Sue Ellen Francisco: regular',
					'Suez One' => 'Suez One: regular',
					'Sulphur Point' => 'Sulphur Point: 300,regular,700',
					'Sumana' => 'Sumana: regular,700',
					'Sunflower' => 'Sunflower: 300,500,700',
					'Sunshiney' => 'Sunshiney: regular',
					'Supermercado One' => 'Supermercado One: regular',
					'Sura' => 'Sura: regular,700',
					'Suranna' => 'Suranna: regular',
					'Suravaram' => 'Suravaram: regular',
					'Suwannaphum' => 'Suwannaphum: 100,300,regular,700,900',
					'Swanky and Moo Moo' => 'Swanky and Moo Moo: regular',
					'Syncopate' => 'Syncopate: regular,700',
					'Syne' => 'Syne: regular,500,600,700,800',
					'Syne Mono' => 'Syne Mono: regular',
					'Syne Tactile' => 'Syne Tactile: regular',
					'Tai Heritage Pro' => 'Tai Heritage Pro: regular,700',
					'Tajawal' => 'Tajawal: 200,300,regular,500,700,800,900',
					'Tangerine' => 'Tangerine: regular,700',
					'Tapestry' => 'Tapestry: regular',
					'Taprom' => 'Taprom: regular',
					'Tauri' => 'Tauri: regular',
					'Taviraj' => 'Taviraj: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Teko' => 'Teko: 300,regular,500,600,700',
					'Tektur' => 'Tektur: regular,500,600,700,800,900',
					'Telex' => 'Telex: regular',
					'Tenali Ramakrishna' => 'Tenali Ramakrishna: regular',
					'Tenor Sans' => 'Tenor Sans: regular',
					'Text Me One' => 'Text Me One: regular',
					'Texturina' => 'Texturina: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Thasadith' => 'Thasadith: regular,italic,700,700italic',
					'The Girl Next Door' => 'The Girl Next Door: regular',
					'The Nautigal' => 'The Nautigal: regular,700',
					'Tienne' => 'Tienne: regular,700,900',
					'Tillana' => 'Tillana: regular,500,600,700,800',
					'Tilt Neon' => 'Tilt Neon: regular',
					'Tilt Prism' => 'Tilt Prism: regular',
					'Tilt Warp' => 'Tilt Warp: regular',
					'Timmana' => 'Timmana: regular',
					'Tinos' => 'Tinos: regular,italic,700,700italic',
					'Tiro Bangla' => 'Tiro Bangla: regular,italic',
					'Tiro Devanagari Hindi' => 'Tiro Devanagari Hindi: regular,italic',
					'Tiro Devanagari Marathi' => 'Tiro Devanagari Marathi: regular,italic',
					'Tiro Devanagari Sanskrit' => 'Tiro Devanagari Sanskrit: regular,italic',
					'Tiro Gurmukhi' => 'Tiro Gurmukhi: regular,italic',
					'Tiro Kannada' => 'Tiro Kannada: regular,italic',
					'Tiro Tamil' => 'Tiro Tamil: regular,italic',
					'Tiro Telugu' => 'Tiro Telugu: regular,italic',
					'Titan One' => 'Titan One: regular',
					'Titillium Web' => 'Titillium Web: 200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,900',
					'Tomorrow' => 'Tomorrow: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Tourney' => 'Tourney: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Trade Winds' => 'Trade Winds: regular',
					'Train One' => 'Train One: regular',
					'Trirong' => 'Trirong: 100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Trispace' => 'Trispace: 100,200,300,regular,500,600,700,800',
					'Trocchi' => 'Trocchi: regular',
					'Trochut' => 'Trochut: regular,italic,700',
					'Truculenta' => 'Truculenta: 100,200,300,regular,500,600,700,800,900',
					'Trykker' => 'Trykker: regular',
					'Tsukimi Rounded' => 'Tsukimi Rounded: 300,regular,500,600,700',
					'Tulpen One' => 'Tulpen One: regular',
					'Turret Road' => 'Turret Road: 200,300,regular,500,700,800',
					'Twinkle Star' => 'Twinkle Star: regular',
					'Ubuntu' => 'Ubuntu: 300,300italic,regular,italic,500,500italic,700,700italic',
					'Ubuntu Condensed' => 'Ubuntu Condensed: regular',
					'Ubuntu Mono' => 'Ubuntu Mono: regular,italic,700,700italic',
					'Uchen' => 'Uchen: regular',
					'Ultra' => 'Ultra: regular',
					'Unbounded' => 'Unbounded: 200,300,regular,500,600,700,800,900',
					'Uncial Antiqua' => 'Uncial Antiqua: regular',
					'Underdog' => 'Underdog: regular',
					'Unica One' => 'Unica One: regular',
					'UnifrakturCook' => 'UnifrakturCook: 700',
					'UnifrakturMaguntia' => 'UnifrakturMaguntia: regular',
					'Unkempt' => 'Unkempt: regular,700',
					'Unlock' => 'Unlock: regular',
					'Unna' => 'Unna: regular,italic,700,700italic',
					'Updock' => 'Updock: regular',
					'Urbanist' => 'Urbanist: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'VT323' => 'VT323: regular',
					'Vampiro One' => 'Vampiro One: regular',
					'Varela' => 'Varela: regular',
					'Varela Round' => 'Varela Round: regular',
					'Varta' => 'Varta: 300,regular,500,600,700',
					'Vast Shadow' => 'Vast Shadow: regular',
					'Vazirmatn' => 'Vazirmatn: 100,200,300,regular,500,600,700,800,900',
					'Vesper Libre' => 'Vesper Libre: regular,500,700,900',
					'Viaoda Libre' => 'Viaoda Libre: regular',
					'Vibes' => 'Vibes: regular',
					'Vibur' => 'Vibur: regular',
					'Victor Mono' => 'Victor Mono: 100,200,300,regular,500,600,700,100italic,200italic,300italic,italic,500italic,600italic,700italic',
					'Vidaloka' => 'Vidaloka: regular',
					'Viga' => 'Viga: regular',
					'Vina Sans' => 'Vina Sans: regular',
					'Voces' => 'Voces: regular',
					'Volkhov' => 'Volkhov: regular,italic,700,700italic',
					'Vollkorn' => 'Vollkorn: regular,500,600,700,800,900,italic,500italic,600italic,700italic,800italic,900italic',
					'Vollkorn SC' => 'Vollkorn SC: regular,600,700,900',
					'Voltaire' => 'Voltaire: regular',
					'Vujahday Script' => 'Vujahday Script: regular',
					'Waiting for the Sunrise' => 'Waiting for the Sunrise: regular',
					'Wallpoet' => 'Wallpoet: regular',
					'Walter Turncoat' => 'Walter Turncoat: regular',
					'Warnes' => 'Warnes: regular',
					'Water Brush' => 'Water Brush: regular',
					'Waterfall' => 'Waterfall: regular',
					'Wavefont' => 'Wavefont: 100,200,300,regular,500,600,700,800,900',
					'Wellfleet' => 'Wellfleet: regular',
					'Wendy One' => 'Wendy One: regular',
					'Whisper' => 'Whisper: regular',
					'WindSong' => 'WindSong: regular,500',
					'Wire One' => 'Wire One: regular',
					'Wix Madefor Display' => 'Wix Madefor Display: regular,500,600,700,800',
					'Wix Madefor Text' => 'Wix Madefor Text: regular,italic,500,500italic,600,600italic,700,700italic,800,800italic',
					'Work Sans' => 'Work Sans: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Workbench' => 'Workbench: regular',
					'Xanh Mono' => 'Xanh Mono: regular,italic',
					'Yaldevi' => 'Yaldevi: 200,300,regular,500,600,700',
					'Yanone Kaffeesatz' => 'Yanone Kaffeesatz: 200,300,regular,500,600,700',
					'Yantramanav' => 'Yantramanav: 100,300,regular,500,700,900',
					'Yatra One' => 'Yatra One: regular',
					'Yellowtail' => 'Yellowtail: regular',
					'Yeon Sung' => 'Yeon Sung: regular',
					'Yeseva One' => 'Yeseva One: regular',
					'Yesteryear' => 'Yesteryear: regular',
					'Yomogi' => 'Yomogi: regular',
					'Young Serif' => 'Young Serif: regular',
					'Yrsa' => 'Yrsa: 300,regular,500,600,700,300italic,italic,500italic,600italic,700italic',
					'Ysabeau' => 'Ysabeau: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Ysabeau Infant' => 'Ysabeau Infant: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Ysabeau Office' => 'Ysabeau Office: 100,200,300,regular,500,600,700,800,900,100italic,200italic,300italic,italic,500italic,600italic,700italic,800italic,900italic',
					'Ysabeau SC' => 'Ysabeau SC: 100,200,300,regular,500,600,700,800,900',
					'Yuji Boku' => 'Yuji Boku: regular',
					'Yuji Hentaigana Akari' => 'Yuji Hentaigana Akari: regular',
					'Yuji Hentaigana Akebono' => 'Yuji Hentaigana Akebono: regular',
					'Yuji Mai' => 'Yuji Mai: regular',
					'Yuji Syuku' => 'Yuji Syuku: regular',
					'Yusei Magic' => 'Yusei Magic: regular',
					'ZCOOL KuaiLe' => 'ZCOOL KuaiLe: regular',
					'ZCOOL QingKe HuangYou' => 'ZCOOL QingKe HuangYou: regular',
					'ZCOOL XiaoWei' => 'ZCOOL XiaoWei: regular',
					'Zen Antique' => 'Zen Antique: regular',
					'Zen Antique Soft' => 'Zen Antique Soft: regular',
					'Zen Dots' => 'Zen Dots: regular',
					'Zen Kaku Gothic Antique' => 'Zen Kaku Gothic Antique: 300,regular,500,700,900',
					'Zen Kaku Gothic New' => 'Zen Kaku Gothic New: 300,regular,500,700,900',
					'Zen Kurenaido' => 'Zen Kurenaido: regular',
					'Zen Loop' => 'Zen Loop: regular,italic',
					'Zen Maru Gothic' => 'Zen Maru Gothic: 300,regular,500,700,900',
					'Zen Old Mincho' => 'Zen Old Mincho: regular,500,600,700,900',
					'Zen Tokyo Zoo' => 'Zen Tokyo Zoo: regular',
					'Zeyada' => 'Zeyada: regular',
					'Zhi Mang Xing' => 'Zhi Mang Xing: regular',
					'Zilla Slab' => 'Zilla Slab: 300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Zilla Slab Highlight' => 'Zilla Slab Highlight: regular,700'
				)
			);
}
endif;

if ( ! function_exists( 'wppm_custom_meta' ) ) :
	function wppm_custom_meta() {
		$meta = array();
		$meta['author'] = sprintf( '<a href="%1$s"><span class="vcard author %2$s" itemprop="name"><span class="fn">%2$s</span></span></a>',
			esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
			get_the_author()
		);
		$meta['date'] = sprintf( '<span class="posted-on"><time itemprop="datePublished" class="entry-date updated" datetime="%s">%s</time></span>', get_the_date( DATE_W3C ), get_the_date() );
		$meta['date_modified'] = sprintf( '<span class="updated-on"><meta itemprop="dateModified" content="%s">%s %s</span>', get_the_modified_date( DATE_W3C ), get_the_modified_date(), get_the_modified_date('H:i:s') );

		// Comment link
		$num_comments = get_comments_number();
		$meta['comments'] = '';
		if ( comments_open() && ( $num_comments >= 1 ) ) {
			$meta['comments'] = sprintf( '<a href="%1$s" class="post-comment" title="%2$s">%3$s</a>',
				esc_url( get_comments_link() ),
				sprintf( esc_html__( 'Comment on %s', 'newsplus' ), esc_attr( get_the_title() ) ),
				$num_comments
			);
		}

		$meta['categories'] = get_the_category_list( _x( ', ', 'category items separator', 'wppm' ) );
		$meta['permalink'] = get_permalink();

		return $meta;
	}
endif;

if ( ! function_exists( 'wppm_generate_excerpt' ) ) :
	function wppm_generate_excerpt( $psource = 'excerpt', $allowed_tags = '', $content_filter = '', $cust_field_key = '', $excerpt_length = '10', $meta_box = '' ) {
		$excerpt_text = '';
		$post_id = get_the_id();

		if ( 'content' == $psource ) {
			$c = apply_filters( 'the_content', get_the_content() );
			$excerpt_text = wppm_short( $c, $excerpt_length, $allowed_tags );
		}

		elseif ( 'meta_box' == $psource ) {
			$meta_box_arr = get_post_meta( $post_id, $meta_box, true );
			if ( isset( $meta_box_arr ) && is_array( $meta_box_arr ) && isset( $meta_box_arr[ $cust_field_key ] ) && '' !== $meta_box_arr[ $cust_field_key ] ) {
				$excerpt_text = do_shortcode( $meta_box_arr[ $cust_field_key ] );
			}
			else {
				$excerpt_text = wppm_short( get_the_excerpt(), $excerpt_length, '' );
			}
		}

		elseif ( 'custom_field' == $psource ) {
			$excerpt_text = do_shortcode( get_post_meta( $post_id, $cust_field_key, true ) );
			if ( '' == $excerpt_text ) {
				$excerpt_text = wppm_short( get_the_excerpt(), $excerpt_length, '' );
			}
		}

		else {
			$excerpt_text = wppm_short( get_the_excerpt(), $excerpt_length, '' );
		}

		if ( $content_filter && ( 'meta_box' == $psource || 'custom_field' == $psource ) ) {
			$excerpt_text = apply_filters( 'the_content', $excerpt_text );
		}

		return $excerpt_text;
	}
endif;

if ( ! function_exists( 'wppm_generate_title' ) ) :
	function wppm_generate_title( $hsource = 'title', $h_cust_field_key = '', $h_length = '', $h_meta_box = '', $more = null ) {
		$post_id = get_the_id();
		$title_text = get_the_title( $post_id );

		if ( 'meta_box' == $hsource ) {
			$meta_box_arr = get_post_meta( $post_id, $h_meta_box, true );
			if ( isset( $meta_box_arr ) && is_array( $meta_box_arr ) && isset( $meta_box_arr[ $h_cust_field_key ] ) && '' !==  $meta_box_arr[ $h_cust_field_key ] ) {
				$title_text = $meta_box_arr[ $h_cust_field_key ];
			}
		} elseif ( 'custom_field' == $hsource ) {
			$cust_field_title = get_post_meta( $post_id, $h_cust_field_key, true );
			if ( isset( $cust_field_title ) && '' !== $cust_field_title ) {
				$title_text = $cust_field_title;
			}
		}

		if ( '' == $h_length ) {
			return $title_text;
		} else {
			return wp_trim_words( $title_text, $h_length, $more );
		}
	}
endif;