<?php
/**
 * Affiliate Email After Registration for Elementor
 *
 * @package    AffiliateWP
 * @subpackage Integrations
 * @copyright  Copyright (c) 2023, Sandhills Development, LLC
 * @since      2.19.0
 */

class Affiliate_Email_After_Registration extends \ElementorPro\Modules\Forms\Classes\Action_Base {

	/**
	 * Get the unique name of the action.
	 *
	 * @since 2.19.0
	 * @return string The action's unique name.
	 */
	public function get_name() : string {
		return 'affiliate_email_action';
	}

	/**
	 * Get the display label of the action.
	 *
	 * @since 2.19.0
	 * @return string The action's display label.
	 */
	public function get_label() : string {
		return __( 'Email Affiliate', 'affiliate-wp' );
	}

	/**
	 * Execute the action when the form is submitted.
	 *
	 * Handles the logic to send an email to the registered affiliate after form submission.
	 * Retrieves user data from the form, constructs the email content, and sends it to
	 * the registered affiliate's email address.
	 *
	 * @since 2.19.0
	 * @param $record Form record data.
	 * @param $ajax_handler AJAX handler for the form.
	 */
	public function run( $record, $ajax_handler ) : void {

		$affiliate_registration = $record->get_form_settings( 'affiliate_registration' );
		if ( empty( $affiliate_registration ) || 'yes' !== $affiliate_registration ) {
			return;
		}

		// Get submitted data.
		$raw_fields = $record->get('fields');
		$fields = array();

		foreach ( $raw_fields as $id => $field ) {
			$fields[$id] = $field['value'];
		}

		$fields_map = $record->get_form_settings( 'affiliatewp_fields_map' );
		$map = array();

		foreach ( $fields_map as $map_item ) {
			if ( ! empty( $map_item['remote_id'] ) ) {
				$map[ $map_item['remote_id'] ] = $map_item['local_id'];
			}
		}

		// Get email.
		$email_to = ! empty( $fields[ $map['user_email'] ] ) ? $fields[ $map['user_email'] ] : '';

		// Return if no email.
		if ( empty( $email_to ) ) {
			return;
		}

		// Subject.
		$subject = sprintf( /* translators: %s - Site name. */
			esc_html__( '%s - Your affiliate username and password info', 'affiliate-wp' ),
			wp_specialchars_decode( get_option( 'blogname' ), ENT_QUOTES )
		);

		// Affiliate login URL.
		$affiliate_login_url = get_permalink( affiliate_wp()->settings->get( 'affiliates_page' ) );

		// Affiliate username.
		$current_user = wp_get_current_user();
		$affiliate_user_login = $current_user->user_login;

		// Password if provided, otherwise autogenerated.
		$password = ! empty( $fields[ $map['user_pass'] ] ) ? $fields[ $map['user_pass'] ] : AffiliateWP_Action_After_Submit::get_password();

		// Build message
		$message = sprintf( /* translators: %s - Affiliate username. */
			esc_html__( 'Username: %s', 'affiliate-wp' ),
			$affiliate_user_login
		);
		$message .= "\r\n";
		$message .= sprintf( /* translators: %s - Affiliate password. */
			esc_html__( 'Password: %s', 'affiliate-wp' ),
			$password
		);
		$message .= "\r\n\r\n";
		$message .= sprintf( /* translators: %s - Affiliate login URL */
			esc_html__( 'Log into your affiliate area at: %s', 'affiliate-wp' ),
			$affiliate_login_url
		);
		$message .= "\r\n\r\n";

		// Send the email.
		wp_mail( $email_to, $subject, $message );
	}

	/**
	 * Register the settings section for the widget.
	 *
	 * @since 2.19.0
	 * @param $widget The Elementor widget instance.
	 */
	public function register_settings_section( $widget ) : void {}

	/**
	 * Handle actions on export.
	 *
	 * @since 2.19.0
	 * @param array $element The Elementor element being exported.
	 */
	public function on_export( $element ) : void {}
}
