<?php
if (class_exists('WP_Analytify_WooCommerce')) {
	return;
}

class WP_Analytify_WooCommerce extends WP_Analytify_Pro_Base
{

	/**
	 * @var WP_Analytify_WooCommerce $instance The one true WP_Analytify_WooCommerce
	 * @since 1.0.0
	 */
	private static $instance;

	public $catpage_json = array();
	public $catpage_json_ATC_link = array();
	public $google_measurement_url = 'https://ssl.google-analytics.com/collect';
	public $google_debug_measurement_url = 'https://www.google-analytics.com/debug/collect';
	public $order_fully_refunded = null;

	/**
	 * Product Data Container
	 * Used to send hits, will collect data from 'woocommerce_before_shop_loop_item'
	 *
	 * @var array
	 */
	private $product_list = array();

	/**
	 * Get active instance
	 *
	 * @access      public
	 * @since       1.0.0
	 * @return      object self::$instance The one true WP_Analytify_WooCommerce
	 */
	public static function instance()
	{

		if (!self::$instance) {
			self::$instance = new WP_Analytify_WooCommerce();
			self::$instance->setup_constants();
			self::$instance->includes();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Setup plugin constants
	 *
	 * @access      private
	 * @since       1.0.0
	 * @return      void
	 */
	private function setup_constants()
	{

		// Setting Global Values.
		$this->define('WC_ANALYTIFY_PLUGIN_FILE', __FILE__);
		$this->define('WC_ANALYTIFY_PLUGIN_BASENAME', plugin_basename(__FILE__));
		$this->define('WC_ANALYTIFY_PLUGIN_DIR', dirname(__FILE__));
		$this->define('WC_ANALYTIFY_PRODUCT_NAME', 'Enhanced Ecommerce Google Analytics for WooCommerce Addon');
		$this->define('WC_ANALYTIFY_PRODUCT_ID', 1351);
		$this->define('WC_ANALYTIFY_STORE_URL', 'https://analytify.io');
	}

	/**
	 * Define constant if not already set
	 *
	 * @param  string      $name
	 * @param  string|bool $value
	 */
	private function define($name, $value)
	{

		if (!defined($name)) {
			define($name, $value);
		}
	}

	/**
	 * Include necessary files
	 *
	 * @access      private
	 * @since       1.0.0
	 * @return      void
	 */
	private function includes()
	{

		// Track WooCommerce Subscription tracking.
		if (class_exists('WC_Subscriptions')) {
			include WC_ANALYTIFY_PLUGIN_DIR . '/classes/woocommerce-subscription.php';
		}

		// Track WooCommerce Emails.
		include WC_ANALYTIFY_PLUGIN_DIR . '/classes/email-tracking.php';

		// Rest endpoints.
		include WC_ANALYTIFY_PLUGIN_DIR . '/classes/analytify-woo-rest-api.php';
	}

	/**
	 * Run action and filter hooks
	 *
	 * @access      private
	 * @since       1.0.0
	 * @return      void
	 */
	private function hooks()
	{

		// Add menu for WooCommerce Analytics Dashboard.
		add_action('analytify_add_submenu', array($this, 'add_menu'), 10);
		add_action('admin_init', array($this, 'check_plugin_updates'), 0);
		add_action('admin_init', array($this, 'wp_analytify_woo_register_option'));

		// Add WooCommerce navigation bar.
		add_action('admin_menu', array($this, 'woo_navigation_bar'), 10);

		// License form.
		add_action('woocommerce_license_key', array($this, 'display_license_form'));
		add_action('wp_ajax_wpanalytifywoo_activate_license', array($this, 'ajax_woo_activate_license'));

		add_action('admin_notices', array($this, 'show_admin_notices'));
		add_action('ga_ecommerce_js', array($this, 'add_ga_ecommerce_js'), 1);

		// Feeds product data from hits.
		add_action('woocommerce_before_shop_loop_item', array($this, 'feed_product_data'));
		add_action('wp_footer', array($this, 'print_product_data'));

		add_action('woocommerce_after_add_to_cart_button', array($this, 'add_to_cart'));
		add_action('woocommerce_after_cart', array($this, 'remove_from_cart'));
		add_action('woocommerce_before_single_product', array($this, 'view_product_detail'));
		add_action('woocommerce_before_shop_loop_item', array($this, 'bind_product_metadata'));

		// Coupons tracking.
		add_action('woocommerce_applied_coupon', array($this, 'applied_coupon'));
		if (!empty($_GET['remove_coupon'])) {
			add_action('woocommerce_init', array($this, 'removed_coupon'));
		}

		// Order tracking process.
		add_action('woocommerce_after_cart_totals', array($this, 'product_review_cart_step1'));
		add_action('woocommerce_after_checkout_billing_form', array($this, 'product_checkout_step2'));
		add_action('woocommerce_after_checkout_shipping_form', array($this, 'product_checkout_step3'));
		add_action('woocommerce_after_checkout_form', array($this, 'product_checkout_step4'));
		add_action('woocommerce_checkout_order_created', array($this, 'order_created'));
		add_action('woocommerce_cancelled_order', array($this, 'cancelled_order'));

		add_action('woocommerce_order_status_on-hold', array($this, 'purchase_on_hold'));
		add_action('woocommerce_payment_complete', array($this, 'analytify_track_complete_payment'));
		add_action('woocommerce_order_status_completed', array($this, 'analytify_track_complete_payment'));
		add_action('woocommerce_order_status_processing', array($this, 'analytify_track_complete_payment'));
		add_action('woocommerce_thankyou', array($this, 'analytify_track_purchase'), 2);

		add_action('woocommerce_order_refunded', array($this, 'order_refunded'), 10, 2);
		add_action('woocommerce_ordered_again', array($this, 'reordered'));
		add_action('woocommerce_checkout_update_order_meta', array($this, 'store_ga_identity'));

		add_action('analytify_dashboad_dropdown_option', array($this, 'dashboad_dropdown_option'));
		add_action('add_meta_boxes', array($this, 'source_meta_box'));
		
		/**
		 * Filter for Source Medium Column (Heading) at WooCommerce Orders Page
		 */
		add_filter('manage_edit-shop_order_columns', array($this, 'source_medium_column')); // Legacy Mode.
		add_filter('manage_woocommerce_page_wc-orders_columns', array($this, 'source_medium_column')); //HPOS Mode.
		
		/**
		 * Action for getting Source Medium Column (Values) at WooCommerce Orders Page.
		 */
		add_action('manage_shop_order_posts_custom_column', array($this, 'source_medium_value_order_posts')); // Legacy Mode.
		add_action('manage_woocommerce_page_wc-orders_custom_column', array($this, 'source_medium_value_wc_orders'), 10, 2); //HPOS Mode.
		
		/**
		 * Add: Fetch source button visible at WooCommerce Orders Page.
		 */
		add_action('manage_posts_extra_tablenav', array($this, 'show_source_fetch_button')); // Legacy Mode.
		add_action('woocommerce_order_list_table_extra_tablenav', array($this, 'show_source_fetch_button')); //HPOS Mode.
		
		add_filter('analytify_install_ga_text', array($this, 'install_ga_text'));

		add_filter('analytify_add_ecommerce_report', array($this, 'analytify_add_woo_report'), 10, 4);
		add_filter('analytify_email_stats_option', array($this, 'analytify_email_stats_option'));
		add_filter('analytify_admin_bar_menu', array($this, 'add_admin_bar_menu'), 15, 1);
	}

	/**
	 * New order created.
	 * Send event hit, later used for getting order source.
	 *
	 * @param object $order
	 * @return void
	 */
	public function order_created($order)
	{

		$order_id = $order->get_id();

		if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {

			$session_id = $this->analytify_filter_session_id($_COOKIE);

			$event[] = array(
				'name' => 'order_created',
				'params' => array(
					'currency' => get_woocommerce_currency(),
					'wpa_category' => 'order_id',
					'wpa_label' => $order_id,
					'session_id' => $session_id
				),
			);

			analytify_mp_ga4($event);
		} else {
			$attr = array(
				't' => 'event',
				'ec' => 'analytify_orders', // Category
				'ea' => 'order_created', // Action
				'el' => $order_id, // Label
				'ev' => $order_id, // Value
			);

			$this->hit($attr);
		}
	}

	/**
	 * Provides the product list name.
	 *
	 * @return string
	 */
	private function get_product_list()
	{

		$list_name = '';

		if (is_product_category()) {
			$list_name = 'Category';
		} elseif (is_product_tag()) {
			$list_name = 'Tag';
		} elseif (is_search()) {
			$list_name = 'Search Results';
		} elseif (is_cart()) {
			$list_name = 'Cart';
		} elseif (is_product()) {
			$list_name = 'Product';
		} else {
			$list_name = 'Archive';
		}

		return $list_name;
	}

	/**
	 * Plugin row meta links
	 *
	 * @param array  $input already defined meta links
	 * @param string $file plugin file path and name being processed
	 * @return array $input
	 */
	public function plugin_row_meta($input, $file)
	{

		// exit if its other plugins
		if ('wp-analytify-woocommerce/wp-analytify-woocommerce.php' !== $file) {
			return $input;
		}

		// remove the 'view details' org link
		unset($input[2]);

		// add the plugin site link
		$input[] = '<a href="https://analytify.io/add-ons/woocommerce/">Visit plugin site</a>';

		return $input;
	}

	/**
	 * Add woocommerce navigation bar in dashboard.
	 */
	public function woo_navigation_bar()
	{

		if (function_exists('wc_admin_connect_page') && isset($_GET['page']) && 'analytify-woocommerce' === $_GET['page']) {

			// wp-element required for woo navigation bar.
			wp_enqueue_script('wp-element');

			// Add woo navigation bar.
			wc_admin_connect_page(
				array(
					'id' => 'wp-analytify-woocommerce-navigation',
					'screen_id' => 'analytify_page_analytify-woocommerce',
					'title' => __('Analytics', 'wp-analytify-woocommerce'),
				),
			);
		}
	}

	/**
	 * Add Metabox on order page.
	 *
	 * @since 2.0.6
	 */
	public function source_meta_box()
	{
		add_meta_box('analytify_woo_source', __('Source/Medium', 'wp-analytify-woocommerce'), array($this, 'get_source_medium'), 'shop_order', 'side', 'core');
	}

	/**
	 * Get Order Source/Medium
	 *
	 * @since 2.0.6
	 * @return string
	 */
	public function get_source_medium()
	{

		global $post;

		$source_medium_deprecated = get_post_meta($post->ID, 'analytify_woo_single_source', true);

		if ($source_medium_deprecated) {
			echo $source_medium_deprecated;
			return;
		}

		$source = get_post_meta($post->ID, 'analytify_woo_order_source', true);

		if (!$source) {
			$post_date = get_the_date('Y-m-d', $post->ID);
			$start_date = date('Y-m-d', strtotime($post_date . '-5 days'));
			$end_date = date('Y-m-d', strtotime($post_date . '+5 days'));

			if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {
				$stats = $GLOBALS['WP_ANALYTIFY']->get_reports(
					'analytify_woo_order_source',
					array(),
					array(
						'start' => $start_date,
						'end' => $end_date
					),
					array(
						'sourceMedium',
						'transactionId',
					),
					array(),
					array(
						'logic' => 'AND',
						'filters' => array(
							array(
								'type' => 'dimension',
								'name' => 'transactionId',
								'match_type' => 1,
								'value' => $post->ID
							)
						)
					),
					0,
					false
				);

				if (!empty($stats['rows'][0]['sourceMedium'])) {
					update_post_meta($post->ID, 'analytify_woo_order_source', $stats['rows'][0]['sourceMedium']);
				}
			} else {
				$stats = $GLOBALS['WP_ANALYTIFY']->pa_get_analytics_dashboard('ga:totalEvents', $start_date, $end_date, 'ga:sourceMedium,ga:eventCategory,ga:eventLabel', false, 'ga:eventCategory==analytify_orders;ga:eventAction==order_created;ga:eventLabel==' . $post->ID);
				if (!empty($stats['rows'][0][0])) {
					update_post_meta($post->ID, 'analytify_woo_order_source', $stats['rows'][0][0]);
				}
			}
		}

		echo $source;
	}

	/**
	 * Add Source/Medium column.
	 *
	 * @since 2.0.6
	 */
	public function source_medium_column($columns)
	{

		$new_columns = (is_array($columns)) ? $columns : array();
		$new_columns['analytify_source'] = __('Source/Medium', 'wp-analytify-woocommerce');

		return $new_columns;
	}

	/**
	 * Add Source/Medium value for order.
	 *
	 * For WooCommerce HPOS settings only
	 */
	public function source_medium_value_wc_orders($column, $order)
	{
		if ($column === 'analytify_source') {

			// Check for the new meta key first
			$source = $order->get_meta('analytify_woo_order_source', true);

			if ($source) {
				echo $source;
				return;
			}

			// If the new meta key is not found, try the deprecated one
			$source_medium_deprecated = $order->get_meta('analytify_woo_single_source', true);

			if ($source_medium_deprecated) {
				echo $source_medium_deprecated;
				return;
			}

			// Fetch source/medium on request.
			if (!$source && isset($_GET['analytify_woo_fetch_sale_source'])) {

				$date_created = $order->get_date_created()->format('Y-m-d');
				$start_date = date('Y-m-d', strtotime($date_created . ' - 5 days'));
				$end_date = date('Y-m-d', strtotime($date_created . ' + 5 days'));

				// Get sources from ga4.
				if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {
					$stats = $GLOBALS['WP_ANALYTIFY']->get_reports(
						'analytify_woo_order_source',
						array(),
						array(
							'start' => $start_date,
							'end' => $end_date
						),
						array(
							'sourceMedium',
							'transactionId',
						),
						array(),
						array(
							'logic' => 'AND',
							'filters' => array(
								array(
									'type' => 'dimension',
									'name' => 'transactionId',
									'match_type' => 1,
									'value' => $order->get_id()
								)
							)
						),
						0,
						false
					);

					if (!empty($stats['rows'][0]['sourceMedium'])) {
						$source = $stats['rows'][0]['sourceMedium'];
						$order->update_meta_data('analytify_woo_order_source', $source);
						$order->save();
					}
				} else {
					$stats = $GLOBALS['WP_ANALYTIFY']->pa_get_analytics_dashboard('ga:totalEvents', $start_date, $end_date, 'ga:sourceMedium,ga:eventCategory,ga:eventLabel', false, 'ga:eventCategory==analytify_orders;ga:eventAction==order_created;ga:eventLabel==' . $order->get_id());

					if (isset($stats['rows'][0][0])) {
						$source = $stats['rows'][0][0];
						$order->update_meta_data('analytify_woo_order_source', $source);
						$order->save();
					}
				}
			}
			echo $source;
		}
	}
	/**
	 * Add Source/Medium value.
	 *
	 * For WooCommerce Legacy Orders Column.
	 *
	 * @since 2.0.6
	 */
	public function source_medium_value_order_posts($column)
	{

		global $post;

		if ($column == 'analytify_source') {

			$source_medium_deprecated = get_post_meta($post->ID, 'analytify_woo_single_source', true);

			if ($source_medium_deprecated) {
				echo $source_medium_deprecated;
				return;
			}

			$source = get_post_meta($post->ID, 'analytify_woo_order_source', true);

			// Fetch source/medium on request.
			if (!$source && isset($_GET['analytify_woo_fetch_sale_source'])) {
				$post_date = get_the_date('Y-m-d', $post->ID);
				$start_date = date('Y-m-d', strtotime($post_date . ' - 5 days'));
				$end_date = date('Y-m-d', strtotime($post_date . ' + 5 days'));

				// Get sources from ga4.
				if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {
					$stats = $GLOBALS['WP_ANALYTIFY']->get_reports(
						'analytify_woo_order_source',
						array(),
						array(
							'start' => $start_date,
							'end' => $end_date
						),
						array(
							'sourceMedium',
							'transactionId',
						),
						array(),
						array(
							'logic' => 'AND',
							'filters' => array(
								array(
									'type' => 'dimension',
									'name' => 'transactionId',
									'match_type' => 1,
									'value' => $post->ID
								)
							)
						),
						0,
						false
					);

					if (!empty($stats['rows'][0]['sourceMedium'])) {
						update_post_meta($post->ID, 'analytify_woo_order_source', $stats['rows'][0]['sourceMedium']);
					}
				} else {
					$stats = $GLOBALS['WP_ANALYTIFY']->pa_get_analytics_dashboard('ga:totalEvents', $start_date, $end_date, 'ga:sourceMedium,ga:eventCategory,ga:eventLabel', false, 'ga:eventCategory==analytify_orders;ga:eventAction==order_created;ga:eventLabel==' . $post->ID);

					if (isset($stats['rows'][0][0])) {
						$source = $stats['rows'][0][0];
						update_post_meta($post->ID, 'analytify_woo_order_source', $source);
					}
				}
			}

			echo $source;
		}
	}

	/**
	 * Show Fetch Source Button on Orders List Page.
	 *
	 * @since 2.0.7
	 */
	public function show_source_fetch_button($which)
	{

		global $post_type;

		$screen = get_current_screen();

		if ($post_type === 'shop_order' && 'top' === $which) {
			echo '<input style="background: #fff url(' . ANALYTIFY_PLUGIN_URL . 'assets/img/analytify-logo-small.svg' . ') no-repeat left 10px center; padding-left: 35px;" type="submit" class="analytify_woo_fetch_sale_source button-secondary" value="Fetch Traffic Sources" name="analytify_woo_fetch_sale_source" style="margin-top:3px">';
		} elseif ('woocommerce_page_wc-orders' === $screen->id) {
			echo '<input style="background: #fff url(' . ANALYTIFY_PLUGIN_URL . 'assets/img/analytify-logo-small.svg' . ') no-repeat left 10px center; padding-left: 35px;" type="submit" class="analytify_woo_fetch_sale_source button-secondary" value="Fetch Traffic Sources" name="analytify_woo_fetch_sale_source" style="margin-top:3px">';
		}
	}

	/**
	 * Add dropdown option for woo dashboard.
	 *
	 * @since 1.0.0
	 */
	public function dashboad_dropdown_option()
	{

		echo '<li><a href="' . admin_url('admin.php?page=analytify-woocommerce') . '">WooCommerce</a></li>';
	}

	/**
	 * Track when an order is refunded
	 *
	 * @since 1.0.0
	 * @param int $order_id Order ID
	 */
	public function order_refunded($order_id, $refund_id)
	{

		// Don't track if the refund is already tracked
		if ('yes' === get_post_meta($refund_id, 'wpa_woo_refund_already', true)) {
			return;
		}

		$order = wc_get_order($order_id);
		$refund = wc_get_order($refund_id);

		if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {

			$session_id = $this->analytify_filter_session_id($_COOKIE);

			$event[] = array(
				'name' => 'refund',
				'params' => array(
					'currency' => get_woocommerce_currency(),
					'wpa_category' => 'transaction_id',
					'wpa_label' => $order->get_order_number(),
					'session_id' => $session_id
				),
			);

			analytify_mp_ga4($event);
		} else {
			$attr = array(
				't' => 'event',
				'ec' => 'Tracking by Analytify Woo', // Category
				'ea' => 'Refund',
				'ti' => $order->get_order_number(), // Transaction ID
				'tr' => $refund->get_refund_amount(),
				'pa' => 'refund',
			);
			$this->hit($attr);
		}

		update_post_meta($refund_id, 'wpa_woo_refund_already', 'yes');
	}

	/**
	 * Get tracked data for a single item for GA4.
	 *
	 * @param int $product_id   Product ID.
	 * @param int $variation_id Variation ID.
	 * @param int $quantity     Quantity.
	 * @param int $price        Price.
	 * @return array
	 */
	private function get_tracked_item_ga4($product_id, $variation_id, $quantity, $price = null)
	{

		$variation = '';
		$product = null;

		if (!empty($variation_id)) {
			$product = wc_get_product($variation_id);
			if (method_exists($product, 'get_name')) {
				$variation = $product->get_name();
			} else {
				$variation = $product->post->post_title;
			}

			if (empty($product_id)) {
				$product_id = $variation_id;
			}
		} else {
			$product = wc_get_product($product_id);
		}

		if (is_object($product) && !empty($product)) {
			if (is_null($price)) {
				$price = number_format((float) $product->get_price(), 2);
			}
		} else {
			if (!empty($product_id)) {
				if (is_null($price)) {
					$price = get_post_meta($product_id, '_regular_price', true);
				}
			}
		}

		$categories = get_the_terms($product_id, 'product_cat');
		$category_names = is_array($categories) ? wp_list_pluck($categories, 'name') : array();
		$first_category = reset($category_names);

		return array(
			'item_id' => $product_id,
			'item_name' => get_the_title($product_id),
			'item_category' => $first_category,
			'item_variant' => $variation,
			'price' => $price,
			'quantity' => (int) $quantity,
		);
	}

	/**
	 * Implementing purchase event using js
	 * instead of mpsecret on thank you page.
	 *
	 * @version 5.0.1
	 * @since 5.0.1
	 *
	 */
	public function analytify_track_purchase($order_id)
	{
		/**
		 * Filters whether the completed purchase event should be tracked or not.
		 *
		 * @since 2.1
		 * @param bool $do_not_track true to not track the event, false otherwise
		 * @param int $order_id the order ID
		 */
		if (true === apply_filters('analytify_do_not_track_completed_purchase', false, $order_id)) {
			return;
		}

		// don't track order when its already tracked - note that this uses a different meta key from completed purchase on purpose :)
		if ('yes' === get_post_meta($order_id, '_analytify_woo_tracked_completed_payment', true)) {
			return;
		}

		$order = wc_get_order($order_id);

		// Return, if order is not type of WC Order.
		if (!$order || !$order instanceof WC_Order) {
			return;
		}

		// only track orders with a 'paid' order status
		if (!$order->is_paid()) {
			return;
		}

		if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {

			$discount = '';
			$coupons = method_exists($order, 'get_coupon_codes') ? $order->get_coupon_codes() : $order->get_used_coupons();
			if (sizeof($coupons) > 0) {
				foreach ($coupons as $code) {
					if (!$code) {
						continue;
					} else {
						$discount = $code;
						break;
					}
				}
			}

			$order_items = $order->get_items();
			$items = array();

			foreach ($order_items as $key => $item) {
				$variation_id = $item->get_variation_id() ? $item->get_variation_id() : $item->get_product_id();
				$product_id = $variation_id > 0 ? wp_get_post_parent_id($variation_id) : 0;

				$items[] = $this->get_tracked_item_ga4($product_id, $variation_id, $item->get_quantity(), $order->get_item_total($item));
			}

			$shipping = method_exists($order, 'get_shipping_total') ? $order->get_shipping_total() : $order->get_total_shipping();

			$items = json_encode($items);

			echo "<script>gtag('event', 'purchase', {
				transaction_id: '{$order->get_order_number()}',
				value: {$order->get_total()},
				tax: {$order->get_total_tax()},
				shipping: {$shipping},
				currency: '{$order->get_currency()}',
				coupon: '{$discount}',
				items: {$items}
			});</script>";
			update_post_meta($order_id, '_analytify_woo_tracked_completed_payment', 'yes');
		} else {
			$items = $this->get_items($order);

			$attr = array_merge(
				array(
					't' => 'event',
					'ec' => 'Tracking by Analytify Woo', // Category
					'el' => $order->get_order_number(), // Event Label
					'ev' => round($order->get_total() * 100),
					'cid' => $this->get_cid($order_id),
					'ta' => null, // Affiliation,
					'uid' => $order->get_user_id(),
					'ea' => 'Purchase',
					'pa' => 'purchase', // Product action
					'ti' => $order->get_order_number(), // Transaction ID. Required
					'tr' => $order->get_total(), // Revenue
					'tt' => $order->get_total_tax(), // Tax
					'ts' => $order->get_total_shipping(), // Shipping
					'tcc' => implode(',', $order->get_coupon_codes()), // Coupon code
					'uip' => $this->is_wc_version_lt_3_0() ? $order->customer_ip_address : $order->get_customer_ip_address(),
					'ua' => $this->is_wc_version_lt_3_0() ? $order->customer_user_agent : $order->get_customer_user_agent(),
				),
				$items
			);

			$this->hit($attr);

			// Payment checkout step 5.
			$checkout_step_completed_attr = array_merge(
				array(
					't' => 'event',
					'ec' => 'Tracking by Analytify Woo', // Category
					'el' => $order->get_order_number(), // Event Label
					'ev' => round($order->get_total() * 100),
					'cid' => $this->get_cid($order_id),
					'ta' => null, // Affiliation,
					'uid' => $order->get_user_id(),
					'ea' => 'Checkout',
					'pa' => 'checkout', // Product action
					'ti' => $order->get_order_number(), // Transaction ID. Required
					'tr' => $order->get_total(), // Revenue
					'tt' => $order->get_total_tax(), // Tax
					'ts' => $order->get_total_shipping(), // Shipping
					'tcc' => implode(',', $order->get_coupon_codes()), // Coupon code
					'uip' => $this->is_wc_version_lt_3_0() ? $order->customer_ip_address : $order->get_customer_ip_address(),
					'ua' => $this->is_wc_version_lt_3_0() ? $order->customer_user_agent : $order->get_customer_user_agent(),
					'cos' => 5,
				),
				$items
			);

			$this->hit($checkout_step_completed_attr);
			update_post_meta($order_id, '_analytify_woo_tracked_completed_payment', 'yes');
		}
	}

	/**
	 * This function is responsible for filtering the raw ids in cookies
	 * It will return the extracted ids that could be used in mp secret hit.
	 *
	 * @param mixed $value the raw id or cookies array in case of session_id.
	 *
	 * @return string $id
	 *
	 * @since 5.0.3
	 */
	public function analytify_filter_session_id($value)
	{

		$id = '';

		foreach ($value as $cookieName => $cookieContent) {

			if (strpos($cookieName, '_ga_') === 0) {

				$parts = explode(".", $cookieContent);
				$id = (int) $parts[2];
				break;
			}
		}

		return $id;
	}

	/**
	 * Use to track payment that are previously on hold.
	 *
	 * @param int $order_id
	 * @return void
	 */
	public function analytify_track_complete_payment($order_id)
	{


		/**
		 * Filters whether the completed purchase event should be tracked or not.
		 *
		 * @since 2.1
		 * @param bool $do_not_track true to not track the event, false otherwise
		 * @param int $order_id the order ID
		 */
		if (true === apply_filters('analytify_do_not_track_completed_purchase', false, $order_id)) {
			return;
		}

		// don't track order when its already tracked - note that this uses a different meta key from completed purchase on purpose :)
		if ('yes' === get_post_meta($order_id, '_analytify_woo_tracked_completed_payment', true)) {
			return;
		}

		$order = wc_get_order($order_id);

		// Return, if order is not type of WC Order.
		if (!$order || !$order instanceof WC_Order) {
			return;
		}

		// only track orders with a 'paid' order status
		if (!$order->is_paid()) {
			return;
		}

		if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {

			$discount = '';
			$coupons = method_exists($order, 'get_coupon_codes') ? $order->get_coupon_codes() : $order->get_used_coupons();
			if (sizeof($coupons) > 0) {
				foreach ($coupons as $code) {
					if (!$code) {
						continue;
					} else {
						$discount = $code;
						break;
					}
				}
			}

			$order_items = $order->get_items();
			$items = array();

			foreach ($order_items as $key => $item) {
				$variation_id = $item->get_variation_id() ? $item->get_variation_id() : $item->get_product_id();
				$product_id = $variation_id > 0 ? wp_get_post_parent_id($variation_id) : 0;

				$items[] = $this->get_tracked_item_ga4($product_id, $variation_id, $item->get_quantity(), $order->get_item_total($item));
			}

			$session_id = $this->analytify_filter_session_id($_COOKIE);

			$event[] = array(
				'name' => 'purchase',
				'params' => array(
					'transaction_id' => $order->get_order_number(),
					'value' => $order->get_total(),
					'tax' => $order->get_total_tax(),
					'shipping' => method_exists($order, 'get_shipping_total') ? $order->get_shipping_total() : $order->get_total_shipping(),
					'coupon' => $discount,
					'currency' => $order->get_currency(),
					'items' => $items,
					'session_id' => $session_id,
				),
			);

			analytify_mp_ga4($event);

			update_post_meta($order_id, '_analytify_woo_tracked_completed_payment', 'yes');
		} else {

			$items = $this->get_items($order);

			$attr = array_merge(
				array(
					't' => 'event',
					'ec' => 'Tracking by Analytify Woo', // Category
					'el' => $order->get_order_number(), // Event Label
					'ev' => round($order->get_total() * 100),
					'cid' => $this->get_cid($order_id),
					'ta' => null, // Affiliation,
					'uid' => $order->get_user_id(),
					'ea' => 'Purchase',
					'pa' => 'purchase', // Product action
					'ti' => $order->get_order_number(), // Transaction ID. Required
					'tr' => $order->get_total(), // Revenue
					'tt' => $order->get_total_tax(), // Tax
					'ts' => $order->get_total_shipping(), // Shipping
					'tcc' => implode(',', $order->get_coupon_codes()), // Coupon code
					'uip' => $this->is_wc_version_lt_3_0() ? $order->customer_ip_address : $order->get_customer_ip_address(),
					'ua' => $this->is_wc_version_lt_3_0() ? $order->customer_user_agent : $order->get_customer_user_agent(),
				),
				$items
			);

			$this->hit($attr);

			// Payment checkout step 5.
			$checkout_step_completed_attr = array_merge(
				array(
					't' => 'event',
					'ec' => 'Tracking by Analytify Woo', // Category
					'el' => $order->get_order_number(), // Event Label
					'ev' => round($order->get_total() * 100),
					'cid' => $this->get_cid($order_id),
					'ta' => null, // Affiliation,
					'uid' => $order->get_user_id(),
					'ea' => 'Checkout',
					'pa' => 'checkout', // Product action
					'ti' => $order->get_order_number(), // Transaction ID. Required
					'tr' => $order->get_total(), // Revenue
					'tt' => $order->get_total_tax(), // Tax
					'ts' => $order->get_total_shipping(), // Shipping
					'tcc' => implode(',', $order->get_coupon_codes()), // Coupon code
					'uip' => $this->is_wc_version_lt_3_0() ? $order->customer_ip_address : $order->get_customer_ip_address(),
					'ua' => $this->is_wc_version_lt_3_0() ? $order->customer_user_agent : $order->get_customer_user_agent(),
					'cos' => 5,
				),
				$items
			);

			$this->hit($checkout_step_completed_attr);
			update_post_meta($order_id, '_analytify_woo_tracked_completed_payment', 'yes');
		}
	}

	/**
	 * Track purchase on hold.
	 *
	 * @param int $order_id
	 * @return void
	 */
	public function purchase_on_hold($order_id)
	{

		$order = wc_get_order($order_id);

		if ('paypal' === $order->get_payment_method()) {
			$this->analytify_track_complete_payment($order_id);
		}
	}

	/**
	 * Track purchase event.
	 *
	 * @param int $order_id
	 * @return void
	 */
	public function track_purchase($order_id)
	{

		// Don't track order when its already tracked
		if ('yes' === get_post_meta($order_id, '_analytify_woo_tracked', true)) {
			return;
		}

		$order = wc_get_order($order_id);
		$items = $this->get_items($order);
		$attr = array_merge(
			array(
				't' => 'event',
				'ec' => 'Tracking by Analytify Woo', // Category
				'ea' => 'Purchase',
				'el' => $order->get_order_number(), // Event Label
				'ev' => round($order->get_total() * 100),
				'cid' => $this->get_cid($order_id),
				'ti' => $order_id, // Transaction ID
				'ta' => null, // Affiliation
				'uid' => $order->get_user_id(),
				'pa' => 'purchase', // Product action
				'ti' => $order->get_order_number(), // Transaction ID. Required
				'tr' => $order->get_total(), // Revenue
				'tt' => $order->get_total_tax(), // Tax
				'ts' => $order->get_total_shipping(), // Shipping
				'tcc' => implode(',', $order->get_coupon_codes()), // Coupon code
				'uip' => $this->is_wc_version_lt_3_0() ? $order->customer_ip_address : $order->get_customer_ip_address(),
				'ua' => $this->is_wc_version_lt_3_0() ? $order->customer_user_agent : $order->get_customer_user_agent(),
			),
			$items
		);

		$this->hit($attr);
		update_post_meta($order_id, '_analytify_woo_tracked', 'yes');
	}

	/**
	 * Get order items.
	 *
	 * @param object $order
	 * @return array
	 */
	public function get_items($order)
	{

		$c = 0;
		// Add the purchased products.
		foreach ($order->get_items() as $item) {
			$c++;
			$product = wc_get_product(!empty($item['variation_id']) ? $item['variation_id'] : $item['product_id']);

			$variant = $this->get_product_variation_attributes($product);
			$product_identifier = $this->get_product_identifier($product);
			$category_hierarchy = $this->get_category_hierarchy($product);

			$params["pr{$c}id"] = $product_identifier; // Product ID
			$params["pr{$c}nm"] = $item['name']; // Product name
			$params["pr{$c}ca"] = $category_hierarchy; // Product category
			$params["pr{$c}br"] = ''; // Product brand
			$params["pr{$c}pr"] = $order->get_item_total($item); // Product price
			$params["pr{$c}qt"] = $item['qty']; // Product Quantity
			$params["pr{$c}va"] = $variant; // Product variant
		}

		return $params;
	}

	/**
	 * Helping function to get the variation.
	 *
	 * @since 2.0.8
	 */
	public function get_product_variation_attributes($product)
	{

		if (!$product instanceof WC_Product) {
			$product = wc_get_product($product);
		}

		if (!$product) {
			return '';
		}

		$variant = '';

		if ('variation' === $product->get_type()) {
			$variant = implode(',', array_values($product->get_variation_attributes()));
		} elseif ('variable' === $product->get_type()) {
			if ($this->is_wc_version_lt_3_0()) { // if woocommerce is old than 3
				$attributes = $product->get_variation_default_attributes();
			} else {
				$attributes = $product->get_default_attributes();
			}

			$variant = implode(', ', array_values($attributes));
		}

		return $variant;
	}

	/**
	 * Get product identifier.
	 *
	 * @param int/object $product
	 * @return string
	 */
	public function get_product_identifier($product)
	{

		if (!$product instanceof WC_Product) {
			$product = wc_get_product($product);
		}

		if (!$product) {
			return '';
		}

		if ($product->get_sku()) {
			$identifier = $product->get_sku();
		} else {
			$identifier = '#' . $product->get_ID();
		}

		return $identifier;
	}

	/**
	 * Get category hierarchy.
	 *
	 * @param int/object $product
	 * @return string
	 */
	public function get_category_hierarchy($product)
	{

		if (!$product instanceof WC_Product) {
			$product = wc_get_product($product);
		}

		if (!$product) {
			return '';
		}

		$categories = wc_get_product_terms(
			$product->get_ID(),
			'product_cat',
			array(
				'orderby' => 'parent',
				'order' => 'DESC',
			)
		);

		if (!is_array($categories) || empty($categories)) {
			return '';
		}

		$child_term = $categories[0];

		return trim($this->get_category_parents($child_term->term_id), '/');
	}

	/**
	 * Get category parents.
	 *
	 * @return string
	 */
	function get_category_parents($term_id, $separator = '/', $visited = array())
	{

		$chain = '';
		$parent = get_term($term_id, 'product_cat');

		if (is_wp_error($parent)) {
			return $parent;
		}

		$name = $parent->name;

		if ($parent->parent && ($parent->parent !== $parent->term_id) && !in_array($parent->parent, $visited) && count($visited) < 4) {
			$visited[] = $parent->parent;
			$chain .= $this->get_category_parents($parent->parent, $separator, $visited);
		}

		$chain .= $name . $separator;

		return $chain;
	}

	/**
	 * Store clint ID into order meta.
	 *
	 * @since 2.1
	 */
	function store_ga_identity($order_id, $cid = null)
	{

		if ($cid === null) {
			$cid = $this->get_cid($order_id);
		}

		// store clint ID in order meta if it is not empty.
		if (!empty($cid) && is_string($cid)) {
			$cid = trim($cid);
			update_post_meta($order_id, 'analytify_woo_cid', $cid);
		}

		return !is_string($cid) || '' === $cid ? null : $cid;
	}

	/**
	 * Get product cid.
	 *
	 * @param string $payment_id
	 * @return string
	 */
	public function get_cid($payment_id = '')
	{

		$saved_cid = get_post_meta($payment_id, 'analytify_woo_cid', true);

		if (!empty($_COOKIE['_ga'])) {

			list($version, $domainDepth, $cid1, $cid2) = preg_split('[\.]', $_COOKIE['_ga'], 4);
			$contents = array(
				'version' => $version,
				'domainDepth' => $domainDepth,
				'cid' => $cid1 . '.' . $cid2,
			);
			$cid = $contents['cid'];

			return $cid;
		} elseif (!empty($payment_id) && !empty($saved_cid)) {
			return $saved_cid;
		} else {
			return $this->generate_uuid();
		}
	}

	/**
	 * Generate UUID v4 function - needed to generate a CID when one isn't available
	 *
	 * @link http://www.stumiller.me/implementing-google-analytics-measurement-protocol-in-php-and-wordpress/
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function generate_uuid()
	{

		return sprintf(
			'%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
			// 32 bits for "time_low"
			mt_rand(0, 0xffff),
			mt_rand(0, 0xffff),
			// 16 bits for "time_mid"
			mt_rand(0, 0xffff),
			// 16 bits for "time_hi_and_version",
			// four most significant bits holds version number 4
			mt_rand(0, 0x0fff) | 0x4000,
			// 16 bits, 8 bits for "clk_seq_hi_res",
			// 8 bits for "clk_seq_low",
			// two most significant bits holds zero and one for variant DCE1.1
			mt_rand(0, 0x3fff) | 0x8000,
			// 48 bits for "node"
			mt_rand(0, 0xffff),
			mt_rand(0, 0xffff),
			mt_rand(0, 0xffff)
		);
	}

	// sends hits via the PHP API
	function hit($attr, $debug = false)
	{

		$default_args = array(
			't' => 'event', // Required - Hit type
			'ec' => '', // Event category
			'ea' => '', // Event Action
			'el' => '', // Event Label
			'ev' => null, // Event Value
		);

		$body = array_merge($default_args, $attr);

		$ip = $this->get_client_ip();

		$user_language = isset($_SERVER['HTTP_ACCEPT_LANGUAGE']) ? explode(',', $_SERVER['HTTP_ACCEPT_LANGUAGE']) : array();
		$user_language = reset($user_language);

		$default_body = array(
			'v' => '1', // Required - Version
			'tid' => WP_ANALYTIFY_FUNCTIONS::get_UA_code(), // Required - UA code
			'cid' => $this->get_cid(), // Required - Unique (anonymous) visitor ID
			't' => 'pageview', // Required - Hit type
			'ni' => true, // Non interaction

			'dh' => str_replace(array('http://', 'https://'), '', site_url()),
			'dp' => $_SERVER['REQUEST_URI'],
			'dt' => get_the_title(),

			// Hits that usually also go with JS
			'ul' => $user_language, // Optional - User language

			'uip' => $ip, // Optional - User IP, to make sure its not the servers'
			'ua' => $_SERVER['HTTP_USER_AGENT'], // Optional - User Agent
		);

		$body = wp_parse_args($body, $default_body);

		// Requests without ID are ignored by GA
		if (false == $body['cid']) {
			return false;
		}

		if ($debug) {

			$response = wp_remote_post(
				$this->google_debug_measurement_url,
				array(
					'method' => 'POST',
					'timeout' => '5',
					'blocking' => true,
					'body' => array_merge($body, array('z' => time())),
				)
			);

			print '<pre>';
			print_r($response);
			print '</pre>';
		} else {

			$response = wp_remote_post(
				$this->google_measurement_url,
				array(
					'method' => 'POST',
					'timeout' => '5',
					'blocking' => false,
					'body' => array_merge($body, array('z' => time())),
				)
			);
		}
	}

	/**
	 * Internationalization
	 *
	 * @access      public
	 * @since       1.0.0
	 * @return      void
	 */
	public function load_textdomain()
	{

		$plugin_dir = dirname(WC_ANALYTIFY_PLUGIN_BASENAME);
		load_plugin_textdomain('wp-analytify', false, $plugin_dir . '/languages/');
	}

	/**
	 * Add menu for WooCommerce dashboard page
	 *
	 * @access public
	 * @return void
	 */
	public static function add_menu()
	{

		add_submenu_page('analytify-dashboard', esc_html__('Track everything in WooCommerce with Analytify.', 'wp-analytify-woocommerce'), esc_html__('WooCommerce', 'wp-analytify-woocommerce'), 'edit_posts', 'analytify-woocommerce', array(__CLASS__, 'analytify_page_file_path'), 15);
	}

	/**
	 * WooCommerce Addon Update plugins
	 *
	 * @access public
	 * @return void
	 */
	function check_plugin_updates()
	{

		// retrieve our license key from the DB
		$wpa_license_key = trim(get_option('analytify_woo_license_key'));

		// setup the updater
		if (class_exists('ANALYTIFY_SL_Plugin_Updater')) {
			$edd_updater = new ANALYTIFY_SL_Plugin_Updater(
				WC_ANALYTIFY_STORE_URL,
				ANALYTIFY_WOOCOMMERCE_UPGRADE_PATH,
				array(
					'version' => ANALTYIFY_WOOCOMMERCE_VERSION, // current version number
					'license' => $wpa_license_key, // license key (used get_option above to retrieve from DB)
					'item_id' => WC_ANALYTIFY_PRODUCT_ID, // ID of this plugin
					'author' => 'Muhammad Adnan', // author of this plugin
				)
			);
		}
	}

	/*
	 * Output Analytify for WooCommerce license activation key fields/inputs
	 * @access public
	 *
	 */
	function display_license_form()
	{

		$license_woo = get_option('analytify_woo_license_key');
		$status_woo = get_option('analytify_woo_license_status'); ?>

		<tr valign="top">

			<th scope="row" valign="top">
				<?php esc_html_e('Analytify for WooCommerce (License Key):', 'wp-analytify-woocommerce'); ?>
			</th>

			<?php if ('valid' === $status_woo) { ?>

				<td class="woo-license-row">
					<?php echo $this->get_formatted_masked_woo_license($license_woo); ?>
				</td>

			<?php } else { ?>

				<td class="woo-license-row">
					<input id="analytify_woo_license_key" name="analytify_woo_license_key" type="text" class="regular-text"
						value="<?php esc_attr_e($license_woo); ?>" />
					<input type="submit" class="button-secondary" id="analytify_woo_license_activate"
						name="analytify_license_activate"
						value="<?php esc_html_e('Activate License', 'wp-analytify-woocommerce'); ?>" />
					<br />
					<p id="woo-license-status">
						<?php
						if ($status_woo) {
							echo $status_woo;
						}
						?>
					</p>
				</td>

			<?php } ?>

		</tr>

		<?php
	}

	function wp_analytify_woo_register_option()
	{

		// creates our settings in the options table
		register_setting('analytify-settings', 'analytify_woo_license_key', array($this, 'analytify_woo_sanitize_license'));
	}

	function analytify_woo_sanitize_license($new)
	{

		$old = get_option('analytify_woo_license_key');

		if ($old && $old != $new) {
			delete_option('analytify_woo_license_status'); // new license has been entered, so must reactivate
		}

		return $new;
	}

	/************************************
	 * Activate a license key
	 *************************************/
	public function ajax_woo_activate_license()
	{

		$this->check_ajax_referer('activate-license');

		$key_rules = array(
			'action' => 'key',
			'woo_license_key' => 'string',
			'context' => 'key',
			'nonce' => 'key',
		);

		$this->set_post_data($key_rules);

		$api_params = array(
			'edd_action' => 'activate_license',
			'license' => $this->state_data['woo_license_key'],
			'item_id' => WC_ANALYTIFY_PRODUCT_ID, // the name of our product in EDD
			'url' => home_url(),
		);

		$response = wp_remote_get(
			esc_url_raw(add_query_arg($api_params, WC_ANALYTIFY_STORE_URL)),
			array(
				'timeout' => 30,
				'sslverify' => false,
			)
		);

		// make sure the response came back okay
		if (is_wp_error($response)) {
			return false;
		}

		$license_response = json_decode(wp_remote_retrieve_body($response));

		if ('valid' === $license_response->license) {
			$this->set_woo_license_key($this->state_data['woo_license_key']);
			$license_response->masked_license = $this->get_formatted_masked_license($this->state_data['woo_license_key']);
		} else {
			if ('invalid' === $license_response->license) {
				set_site_transient('wpanalytify_woo_license_response', $license_response, $this->transient_timeout);
				$license_response->error = $this->get_woo_license_status_message($license_response, $this->state_data['context']);
			}
		}

		// $license_response->license will be either "valid" or "invalid"
		update_option('analytify_woo_license_status', $license_response->license);
		update_option('analytify_woo_license_key', $this->state_data['woo_license_key']);

		$result = $this->end_ajax(json_encode($license_response));

		return $result;
	}

	/**
	 * Include file for WooCommerce dashboard page
	 *
	 * @access public
	 * @return void
	 */
	public static function analytify_page_file_path()
	{

		$screen = get_current_screen();

		if (strpos($screen->base, 'analytify-woocommerce') !== false) {
			include WC_ANALYTIFY_PLUGIN_DIR . '/includes/woocommerce-dashboard.php';
		}
	}

	// Initialize e-commerce JS
	public static function add_ga_ecommerce_js()
	{

		if ('gtag' === ANALYTIFY_TRACKING_MODE) {
			// set the currency
			echo "gtag('set', {currency: '" . get_woocommerce_currency() . "'});";
		} else {
			echo "ga('require', 'ec');";
			echo "var ga_currency = '" . get_woocommerce_currency() . "';";
			echo 'ga("set", "&cu", ga_currency);';
		}
	}

	/**
	 * Add Inline JS depending on WooCommerce version
	 *
	 * @access public
	 * @return void
	 */
	function add_inline_js_wpa_woo($code)
	{

		global $woocommerce;

		if (version_compare($woocommerce->version, '2.1', '>=')) {
			wc_enqueue_js($code);
		} else {
			$woocommerce->add_inline_js($code);
		}
	}

	/**
	 * Send coupon code applied event.
	 *
	 * @var $coupon_code Coupon code.
	 */
	function applied_coupon($coupon_code)
	{

		if ('gtag' === ANALYTIFY_TRACKING_MODE) {
			// $code = "gtag('event', 'Coupon Applied', {
			// 'event_category': 'Coupons',
			// 'event_label': '".esc_html( $coupon_code )."',
			// });";
			// Sending this event through mp in ga4
			$event[] = array(
				'name' => 'coupon_applied',
				'params' => array(
					'event_category' => 'Coupons',
					'event_label' => esc_html($coupon_code),
				),
			);
			analytify_mp_ga4($event);
			return;
		} else {
			$coupon_code_property = array(
				'ID' => esc_html($coupon_code),
				'Name' => esc_html($coupon_code),
			);

			$this->add_inline_js_wpa_woo('coupon_code=' . json_encode($coupon_code_property) . ';');

			$code = '
			// Enhanced E-commerce Add Coupon Event
			ga("ec:addPromo", {
				"id": coupon_code.ID,
				"name": coupon_code.Name,
			});

			ga("ec:setAction", "promo_click");
			ga("send", "event", "Coupons", "Coupon Applied", coupon_code.Name);
			';
		}

		$this->add_inline_js_wpa_woo($code);
	}

	/**
	 * Send coupon code remove event.
	 */
	function removed_coupon()
	{

		if (false == $this->is_track_user()) {
			return;
		}

		if (false == $this->is_ga_enable()) {
			return;
		}

		if ('gtag' === ANALYTIFY_TRACKING_MODE) {
			$code = "gtag( 'event', 'Coupon Removed', {
				'event_category': 'Coupons',
				'event_label': '" . esc_html($coupon_code) . "',
			});";
		} else {
			$coupon_code_property = array(
				'ID' => esc_html($_GET['remove_coupon']),
				'Name' => esc_html($_GET['remove_coupon']),
			);

			$this->add_inline_js_wpa_woo('coupon_code=' . json_encode($coupon_code_property) . ';');

			$code = '
			// Enhanced E-commerce Remove Coupon Event
			ga("ec:addPromo", {
				"id": coupon_code.ID,
				"name": coupon_code.Name,
			});

			ga("ec:setAction", "promo_click");
			ga("send", "event", "Coupons", "Coupon Removed", coupon_code.Name);
			';
		}

		$this->add_inline_js_wpa_woo($code);
	}

	/**
	 * Track when user cancelled the order.
	 *
	 * @since 2.0.12
	 */
	function cancelled_order($order_id)
	{

		if (false == $this->is_track_user()) {
			return;
		}

		if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {

			$event[] = array(
				'name' => 'order_cancelled',
				'params' => array(
					'currency' => get_woocommerce_currency(),
					'wpa_category' => 'order_id',
					'wpa_label' => $order_id,
				),
			);

			analytify_mp_ga4($event);
		} else {

			$attr = array(
				't' => 'event',
				'ec' => 'Tracking by Analytify Woo',
				'ea' => 'Order Cancelled',
				'el' => $order_id,
			);

			$this->hit($attr);
		}
	}

	/**
	 * Feeds product data in an array for hits
	 *
	 * @return void
	 */
	public function feed_product_data()
	{

		if ('gtag' !== ANALYTIFY_TRACKING_MODE) {
			return;
		}

		global $product, $woocommerce_loop;

		// Exclude feeds.
		if (is_feed()) {
			return;
		}

		$product_id = version_compare(WC_VERSION, '3.0', '<') ? $product->id : $product->get_id();

		// Generate category list.
		$categories = (array) get_the_terms($product_id, 'product_cat');
		$category_names = wp_list_pluck($categories, 'name');

		// Product list name.
		$list_name = $this->get_product_list();

		$item = array();

		if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {

			$item['item_id'] = $product_id;
			$item['item_name'] = get_the_title($product_id);
			$item['price'] = $product->get_price();
			$item['item_brand'] = '';
			$item['item_category'] = reset($category_names);
		} else {
			$item['id'] = $product_id;
			$item['name'] = get_the_title($product_id);
			$item['brand'] = '';
			$item['category'] = implode(',', $category_names);
			$item['coupon'] = '';
			$item['list_name'] = $list_name;
			$item['list_position'] = isset($woocommerce_loop['loop']) ? $woocommerce_loop['loop'] : 1;
			$item['price'] = $product->get_price();
			$item['quantity'] = 1;
			$item['variant'] = '';
		}

		// Append the item to the list.
		$this->product_list[] = $item;
	}

	/**
	 * Prints the product list as JSON
	 *
	 * @return void
	 */
	public function print_product_data()
	{

		// Check of all requirements are filled for tracking.
		$is_tracking_available = WPANALYTIFY_Utils::is_tracking_available(true);

		if (!$is_tracking_available || empty($this->product_list)) {
			return;
		}

		$select_event = (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) ? 'select_item' : 'select_content';

		// build product list
		$this->add_inline_js_wpa_woo('const analytify_wc_product_list = {');
		foreach ($this->product_list as $product) {
			$ID = isset($product['item_id']) ? $product['item_id'] : $product['id'];
			$this->add_inline_js_wpa_woo(" 'item_" . $ID . "' : " . json_encode($product) . ',');
		}
		$this->add_inline_js_wpa_woo('};');

		foreach ($this->product_list as $product) {

			$ID = isset($product['item_id']) ? $product['item_id'] : $product['id'];

			$this->add_inline_js_wpa_woo(
				"
			jQuery( 'body' ).on( 'click', '.products .post-" . $ID . " a.add_to_wishlist', function() {
				gtag( 'event', 'add_to_wishlist', { items: analytify_wc_product_list.item_" . $ID . " } );
			});
			jQuery( '.products .post-" . $ID . " a:not(.add_to_cart_button.product_type_simple, .add_to_wishlist)' ).on( 'click', function() {
				gtag( 'event', '" . $select_event . "', { content_type : 'product', items: analytify_wc_product_list.item_" . $ID . " } );
			});
			jQuery( '.products .post-" . $ID . " a.add_to_cart_button.product_type_simple' ).on( 'click', function() {
				   gtag( 'event', 'add_to_cart', { items: [{ 'item_id' : analytify_wc_product_list.item_" . $ID . '.item_id }] } );
			});
			'
			);
		}

		// product impression
		$this->add_inline_js_wpa_woo(
			"
			gtag('event', 'view_item_list', {
				items: Object.values(analytify_wc_product_list)
			});
		"
		);
	}

	/**
	 * Enhanced E-commerce tracking for single product add to cart
	 *
	 * @access public
	 * @return void
	 */
	function add_to_cart()
	{

		if (!$this->is_track_user() || !$this->is_ga_enable()) {
			return;
		}

		if ('gtag' === ANALYTIFY_TRACKING_MODE) {
			global $product;

			if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {

				$item = $this->get_tracked_item_ga4($product->get_id(), null, 0, $product->get_price());
				$code = "
				jQuery( '.single_add_to_cart_button' ).on( 'click', function() {
					gtag('event', 'add_to_cart', {
						'items': [{
							'item_id': '" . $item['item_id'] . "',
							'item_name': '" . $item['item_name'] . "',
							'item_category': '" . $item['item_category'] . "',
							'price': '" . $item['price'] . "',
							'quantity': jQuery( 'form.cart input.qty' ).val() ? jQuery( 'input.qty' ).val() : '1'
						}]
					});
				});
				";
			} else {

				// Will hold the categories as comma separated string.
				$categories = '';

				// Get all product categories.
				$all_categories = get_the_terms($product->get_id(), 'product_cat');

				if ($all_categories) {
					foreach ($all_categories as $term) {
						$categories .= $term->name . ',';
					}
				}

				$categories = rtrim($categories, ',');

				$code = "
				jQuery( '.single_add_to_cart_button' ).on( 'click', function() {
					gtag('event', 'add_to_cart', {
						'items': [
							{
								'id': '" . esc_js($product->get_id()) . "',
								'name': '" . esc_js($product->get_title()) . "',
								'list_name': 'Single Product Page',
								'category': '" . esc_js($categories) . "',
								'brand': '',
								'variant': '',
								'list_position': 1,
								'quantity': jQuery( 'form.cart input.qty' ).val() ? jQuery( 'input.qty' ).val() : '1',
								'price': '" . esc_js($product->get_price()) . "'
							}
						]
					});
				});
				";
			}
		} elseif ('gtag' !== ANALYTIFY_TRACKING_MODE && is_product()) {
			$code = '
			jQuery("button[class*=add_to_cart_button]").click(function(e) {

				// Enhanced E-commerce Add to cart clicks
				ga("ec:addProduct", {
					"id" : wpa_woo.p_ID,
					"name": wpa_woo.p_name,
					"category" :wpa_woo.p_category,
					"price": wpa_woo.p_price,
					"quantity" :jQuery(this).parent().find("input[name=quantity]").val()
				});

				ga("ec:setAction", "add");
				ga("send", "event", "Enhanced-Ecommerce","click", "add_to_cart_click");
				//e.preventDefault();
			});
			';
		}

		$this->add_inline_js_wpa_woo($code);
	}

	/**
	 * Enhanced E-commerce tracking for single product remove from cart
	 *
	 * @access public
	 * @return void
	 */
	function remove_from_cart()
	{

		if (!$this->is_track_user() || !$this->is_ga_enable()) {
			return;
		}

		global $woocommerce;

		$cart_products = array();
		$categories = '';
		$code = '';

		if ('gtag' === ANALYTIFY_TRACKING_MODE) {

			$k = 1;

			foreach ($woocommerce->cart->cart_contents as $key => $value) {

				if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {

					$cart_products[$value['product_id']] = $this->get_tracked_item_ga4($value['product_id'], null, $woocommerce->cart->cart_contents[$key]['quantity'], null);
				} else {
					$product = wc_get_product($value['product_id']);

					$cart_products[$product->get_id()]['id'] = esc_js($product->get_id());
					$cart_products[$product->get_id()]['name'] = esc_js($product->get_title());
					$cart_products[$product->get_id()]['list_name'] = 'Cart';
					$cart_products[$product->get_id()]['brand'] = '';
					$cart_products[$product->get_id()]['category'] = esc_js($categories);
					$cart_products[$product->get_id()]['variant'] = '';
					$cart_products[$product->get_id()]['list_position'] = $k;
					$cart_products[$product->get_id()]['quantity'] = $woocommerce->cart->cart_contents[$key]['quantity'];
					$cart_products[$product->get_id()]['price'] = esc_js($product->get_price());
				}

				$k++;
			}

			if (!empty($cart_products)) {
				$this->add_inline_js_wpa_woo('const analytify_cart_items =' . json_encode($cart_products) . ';');
				$this->add_inline_js_wpa_woo(
					"$(document).on('click', 'a.remove[data-product_id]', function(e){
					let analytify_remove_cart_item_id = $(this).attr('data-product_id');
					gtag('event', 'remove_from_cart', {
						'items': [ analytify_cart_items[analytify_remove_cart_item_id] ]
					});
				});"
				);
			}
		} else {

			foreach ($woocommerce->cart->cart_contents as $key => $value) {
				$p = wc_get_product($value['product_id']);
				$remove_url = html_entity_decode(wc_get_cart_remove_url($key));
				$category = get_the_terms($value['product_id'], 'product_cat');

				if ($category) {
					foreach ($category as $term) {
						$categories .= $term->name . ',';
					}
				}

				$categories = rtrim($categories, ',');
				$cart_products[$remove_url] = array(
					'p_ID' => esc_html($p->get_ID()),
					'p_sku' => esc_html($p->get_sku() ? $p->get_sku() : $p->get_ID()),
					'p_name' => esc_html($p->get_title()),
					'p_price' => esc_html($p->get_price()),
					'p_category' => esc_html($categories),
					'p_quantity' => $woocommerce->cart->cart_contents[$key]['quantity'],
				);
			}

			$this->add_inline_js_wpa_woo('cart_products=' . json_encode($cart_products) . ';');

			$code .= "
			$('a[href*=\"?remove_item\"]').click(function(e){

				item = jQuery(this).attr('href');

				ga('ec:addProduct', {
					'id':cart_products[item].p_ID,
					'name': cart_products[item].p_name,
					'category': cart_products[item].p_category,
					'price': cart_products[item].p_price,
					'quantity': cart_products[item].p_quantity
				});

				ga('ec:setAction', 'remove');
				ga('send', 'event', 'Enhanced-Ecommerce', 'click', 'removeFromCartClick');

			});
			";

			$this->add_inline_js_wpa_woo($code);
		}
	}

	/*
	 * Track product detail events
	 *
	 * @access public
	 * @return void
	 */
	function view_product_detail()
	{

		if (!$this->is_track_user() || !$this->is_ga_enable()) {
			return;
		}

		global $product;

		$_product_id = $product->get_id();
		$categories = '';
		$code = '';

		if (is_product()) {

			$category = get_the_terms($_product_id, 'product_cat');
			if ($category) {
				foreach ($category as $term) {
					$categories .= $term->name . ',';
				}
			}

			$categories = rtrim($categories, ',');

			if ('gtag' === ANALYTIFY_TRACKING_MODE) {

				if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {

					$item = $this->get_tracked_item_ga4($_product_id, null, 0, $product->get_price());
					unset($item['quantity']);

					$code = 'gtag("event", "view_item", {
						"items": [ ' . json_encode($item) . ' ]
					});';
				} else {

					$code = 'gtag("event", "view_item", {
						"items": [{
							"id": "' . esc_html($_product_id) . '",
							"name": "' . esc_html($product->get_title()) . '",
							"category": "' . $categories . '",
							"price": "' . $product->get_price() . '",
							"list_name": "",
							"brand": "",
							"variant": "",
						}]
					});';
				}
			} else {

				// product detail view json
				$p_detail_view_json = array(
					'p_ID' => esc_html($_product_id),
					'p_sku' => $product->get_sku() ? $product->get_sku() : $_product_id,
					'p_name' => $product->get_title(),
					'p_category' => $categories,
					'p_price' => $product->get_price(),
				);

				if (empty($p_detail_view_json)) {
					$p_detail_view_json = array();
				}

				// product detail view json
				$this->add_inline_js_wpa_woo('wpa_woo=' . json_encode($p_detail_view_json) . ';');

				$code = '
				ga("ec:addProduct", {
					"id": wpa_woo.p_ID,
					"name": wpa_woo.p_name,
					"category": wpa_woo.p_category,
				});
				ga("ec:setAction", "detail");
				ga("send", "event", "Tracking by Analytify", "detail","product detail view");
				';
			}

			$this->add_inline_js_wpa_woo($code);
		}
	}

	/*
	 * Bind product data
	 *
	 * @access public
	 * @return void
	 */
	function bind_product_metadata()
	{

		global $product, $woocommerce_loop;
		global $catpage_json, $catpage_json_ATC_link;

		$_product_id = $product->get_id();
		$category = get_the_terms($_product_id, 'product_cat');
		$categories = '';

		if ($category) {
			foreach ($category as $term) {
				$categories .= $term->name . ',';
			}
		}

		// Remove last comma(,) if there are multiple categories.
		$categories = rtrim($categories, ',');

		if (is_product_category() || is_search() || is_shop()) {
			// cat page link array
			$catpage_json_ATC_link[$product->add_to_cart_url()] = array('link' => get_permalink($_product_id));

			$catpage_json[get_permalink($_product_id)] = array(
				'p_ID' => esc_html($_product_id),
				'p_sku' => esc_html($product->get_sku() ? $product->get_sku() : $_product_id),
				'p_name' => esc_html($product->get_title()),
				'p_price' => esc_html($product->get_price()),
				'p_category' => esc_html($categories),
				'p_position' => isset($woocommerce_loop['loop']) ? $woocommerce_loop['loop'] : '',
			);
		}
	}

	/**
	 * Enhanced E-commerce tracking for product impressions
	 *
	 * @access public
	 * @return void
	 */
	function product_impressions()
	{

		if (!$this->is_track_user() || !$this->is_ga_enable()) {
			return;
		}

		global $catpage_json, $catpage_json_ATC_link;
		$code = '';

		$this->add_inline_js_wpa_woo('cat_page_link=' . json_encode($catpage_json_ATC_link) . ';');
		$this->add_inline_js_wpa_woo('cat_page=' . json_encode($catpage_json) . ';');

		$this->js_calls();

		// Track impressions for only Shop Page.
		if (is_shop() and !is_search()) {

			$code .= '
			if( cat_page !== null && cat_page.length !== 0 ){
				product_impressions( cat_page, "Shop Page", "shopPage" );
			}

			// Track Clicks on Products
			jQuery("a:not(.product_type_variable, .product_type_grouped)").on( "click", function() {
				url = jQuery(this).attr("href");
				product_clicks( url, cat_page, "Shop Page", "shopPage" );
			});';
		}

		if (is_product_category()) {

			$code .= '
			if( cat_page !== null && cat_page.length !== 0 ){
				product_impressions( cat_page, "Category Page", "categoryPage" );
			}

			// Track Clicks on Products
			jQuery("a:not(.product_type_variable, .product_type_grouped)").on( "click", function() {
				url = jQuery(this).attr("href");
				product_clicks( url, cat_page, "Category Page", "categoryPage" );
			});';
		}

		if (is_search()) {

			$code .= '
			if( cat_page !== null ) {

				if( cat_page.length !== 0 ) {
					product_impressions( cat_page, "Search Page", "searchPage" );
				}

				// Track Clicks on Products
				jQuery("a:not(.product_type_variable, .product_type_grouped)").on( "click", function() {
					url = jQuery(this).attr("href");
					product_clicks( url, cat_page, "Search Page", "searchPage" );
				});
			}
			';
		}

		if (is_product_category() || is_shop() || is_search()) {

			$code .= '
			jQuery("a[href*=add-to-cart]").on( "click", function() {

				url = $(this).attr("href");
				qty = $(this).parent().find("input[name=quantity]").val();

				if( qty == "" || qty === undefined) {
					qty = "1";
				}

				products_addToCart( url, cat_page_link, cat_page, qty );
			});
			';
		}

		if (is_home() || is_product() || is_product_category() || is_search() || is_shop()) {
			$this->add_inline_js_wpa_woo($code);
		}
	}

	/**
	 * Register product impression, click and add to cart scripts.
	 *
	 * @return void
	 */
	function js_calls()
	{

		$non_interaction = (apply_filters('analytify_non_interaction_impression', false)) ? '{"nonInteraction": 1}' : '';

		$code = '
		function product_impressions ( products_list, list_name, label ) {

			t_json_length = Object.keys(products_list).length;

			for(var p_index in products_list) {
				ga("ec:addImpression", {
					"id": products_list[p_index].p_ID,
					"name": products_list[p_index].p_name,
					"category": products_list[p_index].p_category,
					"list": list_name,
					"price": products_list[p_index].p_price,
					"position": products_list[p_index].p_position,
				});
			}
			ga("send", "event", "Tracking by Analytify", "onLoad", "productImpression_" + label, ' . $non_interaction . ' );
		}

		// Track product clicks
		function product_clicks( url, products_list, list_name, label ) {

			if( products_list.hasOwnProperty( url ) ) {

				t_call_fired = true;

				ga( "ec:addProduct", {
					"id": products_list[url].p_ID,
					"name": products_list[url].p_name,
					"category": products_list[url].p_category,
					"price": products_list[url].p_price,
					"position": products_list[url].p_position,
				});

				ga( "ec:setAction", "click", { "list" : list_name } );
				ga( "send", "event", "Tracking by Analytify", "click", "productClick_" + label );

			} else {
				t_call_fired = false;
			}

			return t_call_fired;
		}

		// Track product (Add to Cart) clicks

		function products_addToCart( url, url_addToCart, products_list, qty ) {

			index = url_addToCart[url]["link"];

			if ( products_list.hasOwnProperty( index ) ) {

				t_call_fired = true;

				ga("ec:addProduct", {
					"id": products_list[index].p_ID,
					"name": products_list[index].p_name,
					"category": products_list[index].p_category,
					"price": products_list[index].p_price,
					"quantity" : qty
				});

				ga("ec:setAction", "add");
				ga("send", "event", "Tracking by Analytify", "click", "addToCartClick" );

			} else {
				t_call_fired = false;
			}

			return t_call_fired;

		}
		';

		$this->add_inline_js_wpa_woo($code);
	}

	/*
	 * Tracking Product Checkouts - Step 1
	 * Review Cart Page
	 */
	function product_review_cart_step1()
	{

		if (!$this->is_track_user() || !$this->is_ga_enable()) {
			return;
		}

		if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {

			// Get cart contents.
			$cart_contents = WC()->cart->get_cart();

			// Check to make sure cart has items.
			if (empty($cart_contents)) {
				return;
			}

			$items = array();

			foreach ($cart_contents as $item) {
				$product_id = $item['product_id'];
				$variation_id = isset($item['variation_id']) ? $item['variation_id'] : null;

				$items[] = $this->get_tracked_item_ga4($product_id, $variation_id, $item['quantity']);
			}

			$session_id = $this->analytify_filter_session_id($_COOKIE);

			$event[] = array(
				'name' => 'begin_checkout',
				'params' => array(
					'items' => $items,
					'session_id' => $session_id
				),
			);

			analytify_mp_ga4($event);

			return;
		}

		global $woocommerce;

		$cart_products = array();
		$variations = array();
		$categories = '';

		// Product list name.
		$list_name = $this->get_product_list();

		foreach ($woocommerce->cart->cart_contents as $key => $value) {
			$p = wc_get_product($value['product_id']);

			$category = get_the_terms($value['product_id'], 'product_cat');

			if ($category) {
				foreach ($category as $term) {
					$categories .= $term->name . ',';
				}
			}

			if (isset($value['variation'])) {
				foreach ($value['variation'] as $variation => $variation_value) {
					array_push($variations, $variation_value);
				}
			}

			$categories = rtrim($categories, ',');

			$cart_checkout_products[get_permalink($p->get_id())] = array(
				'p_ID' => esc_html($p->get_id()),
				'p_sku' => esc_html($p->get_sku() ? $p->get_sku() : $p->get_id()),
				'p_name' => esc_html($p->get_title()),
				'p_price' => esc_html($p->get_price()),
				'p_category' => esc_html($categories),
				'p_quantity' => $value['quantity'],
				'p_variant' => !empty($variations) ? implode(',', $variations) : '',
				'list_name' => $list_name,
			);
		}

		$this->add_inline_js_wpa_woo('cart_checkout_products=' . json_encode($cart_checkout_products) . ';');

		$code = '';

		if ('gtag' === ANALYTIFY_TRACKING_MODE) {

			$code .= '
			var cartItems = [];
			for(var item in cart_checkout_products) {
				cartItems.push({
					"id": cart_checkout_products[item].p_ID,
					"name": cart_checkout_products[item].p_name,
					"category": cart_checkout_products[item].p_category,
					"price": cart_checkout_products[item].p_price,
					"quantity": cart_checkout_products[item].p_quantity,
					"variant": cart_checkout_products[item].p_variant,
					"brand": "",
				});
			}
			gtag("event", "checkout_progress", {
				currency: "' . get_woocommerce_currency() . '",
				checkout_option: "Review Cart",
				checkout_step: 1,
				value: ' . $woocommerce->cart->get_cart_contents_total() . ',
				items: cartItems
			});
			gtag("event", "set_checkout_option", {
				checkout_step: 1,
				checkout_option: "Review Cart",
				value: 1
			});
			';
		} else {

			$code = '
			for(var item in cart_checkout_products) {
				ga("ec:addProduct", {
					"id": cart_checkout_products[item].p_ID,
					"name": cart_checkout_products[item].p_name,
					"category": cart_checkout_products[item].p_category,
					"price": cart_checkout_products[item].p_price,
					"quantity": cart_checkout_products[item].p_quantity
				});
			}';

			$code .= 'ga("ec:setAction","checkout", { "step": 1 });';
			$code .= 'ga( "send", "event", "Tracking by Analytify","onLoad","Product Review Cart" );';
		}

		$this->add_inline_js_wpa_woo($code);
	}

	/*
	 * Tracking Product Checkouts - Step 2
	 * Billing form
	 */
	function product_checkout_step2()
	{

		// revoke if page refreshed.
		if ($this->is_page_reload()) {
			return;
		}

		if (!$this->is_track_user() || !$this->is_ga_enable()) {
			return;
		}

		if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {

			// Get cart contents.
			$cart_contents = WC()->cart->get_cart();

			// Check to make sure cart has items.
			if (empty($cart_contents)) {
				return;
			}

			$items = array();

			foreach ($cart_contents as $item) {
				$product_id = $item['product_id'];
				$variation_id = isset($item['variation_id']) ? $item['variation_id'] : null;

				$items[] = $this->get_tracked_item_ga4($product_id, $variation_id, $item['quantity']);
			}

			$session_id = $this->analytify_filter_session_id($_COOKIE);

			$event[] = array(
				'name' => 'add_payment_info',
				'params' => array(
					'items' => $items,
					'session_id' => $session_id
				),
			);

			analytify_mp_ga4($event);

			return;
		}

		global $woocommerce;

		$cart_products = array();
		$variations = array();
		$categories = '';

		// Product list name.
		$list_name = $this->get_product_list();

		foreach ($woocommerce->cart->cart_contents as $key => $value) {
			$p = wc_get_product($value['product_id']);

			$category = get_the_terms($value['product_id'], 'product_cat');

			if ($category) {
				foreach ($category as $term) {
					$categories .= $term->name . ',';
				}
			}

			if (isset($value['variation'])) {
				foreach ($value['variation'] as $variation => $variation_value) {
					array_push($variations, $variation_value);
				}
			}

			$categories = rtrim($categories, ',');

			$cart_checkout_products[get_permalink($p->get_id())] = array(
				'p_ID' => esc_html($p->get_id()),
				'p_sku' => esc_html($p->get_sku() ? $p->get_sku() : $p->get_id()),
				'p_name' => esc_html($p->get_title()),
				'p_price' => esc_html($p->get_price()),
				'p_category' => esc_html($categories),
				'p_quantity' => $value['quantity'],
				'p_variant' => !empty($variations) ? implode(',', $variations) : '',
				'list_name' => $list_name,
			);
		}

		$this->add_inline_js_wpa_woo('cart_checkout_products=' . json_encode($cart_checkout_products) . ';');

		$code = '';

		if ('gtag' === ANALYTIFY_TRACKING_MODE) {

			$code .= '
			var cartItems = [];
			for(var item in cart_checkout_products) {
				cartItems.push({
					"id": cart_checkout_products[item].p_ID,
					"name": cart_checkout_products[item].p_name,
					"category": cart_checkout_products[item].p_category,
					"price": cart_checkout_products[item].p_price,
					"quantity": cart_checkout_products[item].p_quantity,
					"variant": cart_checkout_products[item].p_variant,
					"brand": "",
				});
			}
			gtag("event", "checkout_progress", {
				currency: "' . get_woocommerce_currency() . '",
				checkout_option: "Billing",
				checkout_step: 2,
				value: ' . $woocommerce->cart->get_cart_contents_total() . ',
				items: cartItems
			});
			gtag("event", "set_checkout_option", {
				checkout_step: 2,
				checkout_option: "Billing",
				value: 2
			});
			';
		} else {

			$code .= '
			for(var item in cart_checkout_products) {
				ga("ec:addProduct", {
					"id": cart_checkout_products[item].p_ID,
					"name": cart_checkout_products[item].p_name,
					"category": cart_checkout_products[item].p_category,
					"price": cart_checkout_products[item].p_price,
					"quantity": cart_checkout_products[item].p_quantity
				});
			}';

			$code .= 'ga("ec:setAction","checkout", { "step": 2 });';
			$code .= 'ga( "send", "event", "Tracking by Analytify", "onLoad", "Product Checkout Billing Form" );';
		}

		$this->add_inline_js_wpa_woo($code);
	}

	/*
	 * Tracking Product Checkouts - Step 3
	 * Shipping
	 */
	function product_checkout_step3()
	{

		// Revoke if page refreshed.
		if ($this->is_page_reload() || !$this->is_track_user() || !$this->is_ga_enable()) {
			return;
		}

		if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {

			// Get cart contents.
			$cart_contents = WC()->cart->get_cart();

			// Check to make sure cart has items.
			if (empty($cart_contents)) {
				return;
			}

			$items = array();

			foreach ($cart_contents as $item) {
				$product_id = $item['product_id'];
				$variation_id = isset($item['variation_id']) ? $item['variation_id'] : null;

				$items[] = $this->get_tracked_item_ga4($product_id, $variation_id, $item['quantity']);
			}

			$session_id = $this->analytify_filter_session_id($_COOKIE);

			$event[] = array(
				'name' => 'add_shipping_info',
				'params' => array(
					'items' => $items,
					'session_id' => $session_id
				),
			);

			analytify_mp_ga4($event);

			return;
		}

		global $woocommerce;

		$cart_products = array();
		$variations = array();
		$categories = '';

		// Product list name.
		$list_name = $this->get_product_list();

		foreach ($woocommerce->cart->cart_contents as $key => $value) {
			$p = wc_get_product($value['product_id']);

			$category = get_the_terms($value['product_id'], 'product_cat');

			if ($category) {
				foreach ($category as $term) {
					$categories .= $term->name . ',';
				}
			}

			if (isset($value['variation'])) {
				foreach ($value['variation'] as $variation => $variation_value) {
					array_push($variations, $variation_value);
				}
			}

			$categories = rtrim($categories, ',');

			$cart_checkout_products[get_permalink($p->get_id())] = array(
				'p_ID' => esc_html($p->get_id()),
				'p_sku' => esc_html($p->get_sku() ? $p->get_sku() : $p->get_id()),
				'p_name' => esc_html($p->get_title()),
				'p_price' => esc_html($p->get_price()),
				'p_category' => esc_html($categories),
				'p_quantity' => $value['quantity'],
				'p_variant' => !empty($variations) ? implode(',', $variations) : '',
				'list_name' => $list_name,
			);
		}

		$this->add_inline_js_wpa_woo('cart_checkout_products=' . json_encode($cart_checkout_products) . ';');

		$code = '';

		if ('gtag' === ANALYTIFY_TRACKING_MODE) {

			$code .= '
			var cartItems = [];
			for(var item in cart_checkout_products) {
				cartItems.push({
					"id": cart_checkout_products[item].p_ID,
					"name": cart_checkout_products[item].p_name,
					"category": cart_checkout_products[item].p_category,
					"price": cart_checkout_products[item].p_price,
					"quantity": cart_checkout_products[item].p_quantity,
					"variant": cart_checkout_products[item].p_variant,
					"brand": "",
				});
			}
			gtag("event", "checkout_progress", {
				currency: "' . get_woocommerce_currency() . '",
				checkout_option: "Shipping",
				checkout_step: 3,
				value: ' . $woocommerce->cart->get_cart_contents_total() . ',
				items: cartItems
			});
			gtag("event", "set_checkout_option", {
				checkout_step: 3,
				checkout_option: "Shipping",
				value: 3
			});
			';
		} else {

			$code .= '
			for(var item in cart_checkout_products) {
				ga("ec:addProduct", {
					"id": cart_checkout_products[item].p_ID,
					"name": cart_checkout_products[item].p_name,
					"category": cart_checkout_products[item].p_category,
					"price": cart_checkout_products[item].p_price,
					"quantity": cart_checkout_products[item].p_quantity
				});
			}';

			$code .= 'ga("ec:setAction","checkout", { "step": 3 });';
			$code .= 'ga( "send", "event", "Tracking by Analytify", "onLoad", "Product Checkout Shipping Form" );';
		}

		$this->add_inline_js_wpa_woo($code);
	}

	/*
	 * Tracking Product Checkouts - Step 4
	 * Checkout
	 */
	function product_checkout_step4()
	{

		// Revoke if page refreshed.
		if ($this->is_page_reload() || !$this->is_track_user() || !$this->is_ga_enable()) {
			return;
		}

		if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {
			return;
		}

		global $woocommerce;

		$cart_products = array();
		$variations = array();
		$categories = '';

		// Product list name.
		$list_name = $this->get_product_list();

		foreach ($woocommerce->cart->cart_contents as $key => $value) {
			$p = wc_get_product($value['product_id']);

			$category = get_the_terms($value['product_id'], 'product_cat');

			if ($category) {
				foreach ($category as $term) {
					$categories .= $term->name . ',';
				}
			}

			if (isset($value['variation'])) {
				foreach ($value['variation'] as $variation => $variation_value) {
					array_push($variations, $variation_value);
				}
			}

			$categories = rtrim($categories, ',');

			$cart_checkout_products[get_permalink($p->get_id())] = array(
				'p_ID' => esc_html($p->get_id()),
				'p_sku' => esc_html($p->get_sku() ? $p->get_sku() : $p->get_id()),
				'p_name' => esc_html($p->get_title()),
				'p_price' => esc_html($p->get_price()),
				'p_category' => esc_html($categories),
				'p_quantity' => $value['quantity'],
				'p_variant' => !empty($variations) ? implode(',', $variations) : '',
				'list_name' => $list_name,
			);
		}

		$this->add_inline_js_wpa_woo('cart_checkout_products=' . json_encode($cart_checkout_products) . ';');

		$code = '';

		if ('gtag' === ANALYTIFY_TRACKING_MODE) {

			$code = '
			var cartItems = [];
			for(var item in cart_checkout_products) {
				cartItems.push({
					"id": cart_checkout_products[item].p_ID,
					"name": cart_checkout_products[item].p_name,
					"category": cart_checkout_products[item].p_category,
					"price": cart_checkout_products[item].p_price,
					"quantity": cart_checkout_products[item].p_quantity,
					"variant": cart_checkout_products[item].p_variant,
					"brand": "",
				});
			}
			gtag("event", "checkout_progress", {
				currency: "' . get_woocommerce_currency() . '",
				checkout_option: "Checkout",
				checkout_step: 4,
				value: ' . $woocommerce->cart->get_cart_contents_total() . ',
				items: cartItems
			});
			gtag("event", "set_checkout_option", {
				checkout_step: 4,
				checkout_option: "Checkout",
				value: 4
			});
			';
		} else {

			$code = '
			for(var item in cart_checkout_products) {
				ga("ec:addProduct", {
					"id": cart_checkout_products[item].p_ID,
					"name": cart_checkout_products[item].p_name,
					"category": cart_checkout_products[item].p_category,
					"price": cart_checkout_products[item].p_price,
					"quantity": cart_checkout_products[item].p_quantity
				});
			}';

			$code = $code . 'ga("ec:setAction","checkout", { "step": 4 });';
			$code .= 'ga( "send", "event", "Tracking by Analytify", "onLoad", "Product Checkout Form" );';
		}

		$this->add_inline_js_wpa_woo($code);
	}

	/**
	 * Returns true if on a page which uses WooCommerce templates (cart and checkout are standard pages with short-codes and which are also included)
	 *
	 * @access public
	 * @return bool
	 */
	function is_realy_woocommerce_page()
	{

		if (!is_woocommerce() && !is_shop() && !is_product_category() && !is_product_tag() && !is_product() && !is_cart() && !is_checkout() && !is_account_page()) {
			echo '<script>';
			echo "ga('send', 'pageview');";
			echo '</script>';
		}
	}

	/**
	 * Display notices for woocommerce users
	 *
	 * @since 1.0.2
	 */
	function show_admin_notices()
	{

		if (!current_user_can('manage_options')) {
			return;
		}
	}

	/**
	 * Ignore tracking if user excluded.
	 */
	function is_track_user()
	{

		global $current_user;
		$roles = $current_user->roles;
		$is_track = $GLOBALS['WP_ANALYTIFY']->settings->get_option('exclude_users_tracking', 'wp-analytify-profile', array());

		if (is_user_logged_in() && isset($roles[0]) && in_array($roles[0], $is_track)) {
			return false;
		} else {
			return true;
		}
	}

	/**
	 * Ignore if analytics code is not added.
	 */
	function is_ga_enable()
	{

		$is_code = $GLOBALS['WP_ANALYTIFY']->settings->get_option('install_ga_code', 'wp-analytify-profile');

		if ('on' !== $is_code) {
			$code = 'console.warn("`Activate Install Google Analytics tracking code` from Analytify Profile Tab Settings")';
			$this->add_inline_js_wpa_woo($code);
			return false;
		} else {
			return true;
		}
	}

	/**
	 * Change Install GA text.
	 *
	 * @since 2.0.8
	 */
	function install_ga_text()
	{

		return __('Insert Google Analytics (GA) JavaScript code between the HEADER tags in your website. Check this option if you have installed Analytify E-Commerce add-ons for E-commerce Tracking.', 'wp-analytify-woocommerce');
	}

	/**
	 * Check if WooCommerce version is lower than 3
	 *
	 * @since 2.0.8
	 */
	function is_wc_version_lt_3_0()
	{

		return self::get_wc_version() && version_compare(self::get_wc_version(), '3.0', '<');
	}

	/**
	 * Get WooCommerce version
	 *
	 *  @since 2.0.8
	 */
	public static function get_wc_version()
	{

		return defined('WC_VERSION') && WC_VERSION ? WC_VERSION : null;
	}

	/**
	 * Delete Woo Stats Cache.
	 *
	 * @since 2.0.8
	 */
	private static function delete_woo_transients($dashboard_profile_ID, $start_date, $end_date)
	{

		delete_transient(md5('show-woo-general-stats' . $dashboard_profile_ID . $start_date . $end_date . apply_filters('analytify_enhanced_general_filter', false)));
		delete_transient(md5('show-woo-geographical-stats' . $dashboard_profile_ID . $start_date . $end_date . apply_filters('analytify_enhanced_geographic_filter', 'ga:country!=(not set);ga:itemQuantity>0')));
		delete_transient(md5('show-woo-roi-stats' . $dashboard_profile_ID . $start_date . $end_date . apply_filters('analytify_enhanced_roi_filter', 'ga:transactionRevenue>0')));
		delete_transient(md5('show-woo-product-performance-stats' . $dashboard_profile_ID . $start_date . $end_date));
		delete_transient(md5('show-woo-product-list-stats' . $dashboard_profile_ID . $start_date . $end_date));
		delete_transient(md5('show-woo-category-analysis-stats' . $dashboard_profile_ID . $start_date . $end_date));
		delete_transient(md5('show-woo-coupons-analysis-stats' . $dashboard_profile_ID . $start_date . $end_date . apply_filters('analytify_enhanced_coupon_filter', 'ga:orderCouponCode!=(not set)')));
		delete_transient(md5('show-woo-coupons-analysis-stats' . $dashboard_profile_ID . $start_date . $end_date . apply_filters('analytify_enhanced_coupon_filter', 'ga:orderCouponCode!=(not set)')));
	}

	/**
	 * Check if page reload.
	 *
	 * @since 2.0.8
	 */
	function is_page_reload()
	{

		// Can't be a refresh without having a referrer
		if (!isset($_SERVER['HTTP_REFERER'])) {
			return false;
		}

		// IF the referrer is same to the current page request, then it's a refresh
		return (parse_url($_SERVER['HTTP_REFERER'], PHP_URL_PATH) === parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH));
	}

	/**
	 * Add WooCommerce reports in email notification.
	 *
	 * @since 2.0.10
	 */
	function analytify_add_woo_report($message, $Analytify_Email, $start_date, $end_date)
	{

		if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {
			$stats = $WP_Analytify->get_reports(
				'analytify-email-woo-general-stats',
				array(
					'transactions',
					'purchaseRevenue',
					// TODO: (missing_ga4) 'revenuePerTransaction',
					'ecommercePurchases',
					// TODO: (missing_ga4) 'transactionsPerSession',
					'addToCarts',
					// TODO: (missing_ga4) 'productRemovesFromCart',
					'checkouts',
					'itemListClicks',
					'itemViews',
				),
				array(
					'start' => $start_date,
					'end' => $end_date,
				)
			);

			include WC_ANALYTIFY_PLUGIN_DIR . '/views/email/woo-general-stats.php';
			$message .= wpa_email_woo_general($Analytify_Email, $stats);
		} else {
			$stats = $Analytify_Email->pa_get_analytics_dashboard('ga:transactions,ga:transactionRevenue,ga:revenuePerTransaction,ga:uniquePurchases,ga:transactionsPerSession,ga:productAddsToCart,ga:productRemovesFromCart,ga:productCheckouts,ga:productListClicks,ga:productDetailViews', $start_date, $end_date, false, false, false, false, 'analytify-email-woo-general-stats');

			include WC_ANALYTIFY_PLUGIN_DIR . '/views/email/woo-general-stats-deprecated.php';
			$message .= pa_email_include_woo_general($Analytify_Email, $stats);
		}

		return $message;
	}

	/**
	 * Add WooCommerce stats option.
	 *
	 * @since 2.0.10
	 */
	function analytify_email_stats_option($option)
	{

		$option['show-woo-stats'] = __('Ecommerce Stats', 'wp-analytify-woocommerce');

		return $option;
	}

	/**
	 * Add option in admin bar.
	 *
	 * @param [array] $menus.
	 *
	 * @since 2.0.10
	 */
	function add_admin_bar_menu($menus)
	{

		$menus['analytify-woocommerce'] = __('WooCommerce', 'wp-analytify-woocommerce');

		return $menus;
	}

	/**
	 * Get IP by using the WooCommercce Geolocation class.
	 *
	 * @since 2.0.11
	 */
	private function get_client_ip()
	{

		return WC_Geolocation::get_ip_address();
	}

	/**
	 * Track Reorder. Trigger when user click reorder button.
	 *
	 * @since 2.1
	 */
	function reordered($order_id)
	{

		if (false == $this->is_track_user()) {
			return;
		}

		if (method_exists('WPANALYTIFY_Utils', 'get_ga_mode') && 'ga4' === WPANALYTIFY_Utils::get_ga_mode()) {

			$session_id = $this->analytify_filter_session_id($_COOKIE);
			$event[] = array(
				'name' => 'order_reorder',
				'params' => array(
					'currency' => get_woocommerce_currency(),
					'wpa_category' => 'order_id',
					'wpa_label' => $order_id,
					'session_id' => $session_id
				),
			);

			analytify_mp_ga4($event);
		} else {

			$attr = array(
				't' => 'event',
				'ec' => 'Tracking by Analytify Woo',
				'ea' => 'Reoder',
				'el' => $order_id,
				'ni' => true,
			);

			$this->hit($attr);
		}
	}

	/**
	 * Check if tracking is enabled.
	 *
	 * @param int $user_id
	 * @return boolean
	 */
	function is_tracking_is_enabled_for_user($user_id = null)
	{

		if (!$user_id) {
			return false;
		}

		$user_id = get_current_user_id();

		if (0 == $user_id) {
			return false;
		}

		$user_meta = get_userdata($user_id);

		if (!is_object($user_meta)) {
			return false;
		}

		$user_roles = $user_meta->roles;
		$is_track = $GLOBALS['WP_ANALYTIFY']->settings->get_option('exclude_users_tracking', 'wp-analytify-profile', array());

		if (isset($user_roles[0]) && in_array($user_roles[0], $is_track)) {
			return false;
		}

		return true;
	}
}
