<?php
/**
 * Handle Analytify REST end points
 */
class Analytify_Woo_Rest_API {

	/**
	 * The single instance of the class.
	 *
	 * @var object
	 */
	private static $instance;

	private $wp_analytify;
	private $ga_mode;
	private $start_date;
	private $end_date;
	private $compare_start_date;
	private $compare_end_date;
	private $date_difference;
	private $report_url;
	private $report_date_range;

	/**
	 * Returns the single instance of the class.
	 *
	 * @return object Class instance
	 */
	public static function get_instance() {

		if ( empty( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Class constructor.
	 */
	private function __construct() {

		$this->wp_analytify = $GLOBALS['WP_ANALYTIFY'];
		$this->ga_mode      = method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) ? WPANALYTIFY_Utils::get_ga_mode() : 'ga3';

		add_action( 'rest_api_init', array( $this, 'rest_api_init' ) );
	}

	/**
	 * Register end point.
	 */
	public function rest_api_init() {

		register_rest_route(
			'wp-analytify/v1',
			'/get_pro_woo_report/(?P<profile_id>\d+)/(?P<request_type>[a-zA-Z0-9-]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE, // Get Request
					'callback'            => array( $this, 'handle_request' ),
					'permission_callback' => array( $this, 'permission_check' ),
				),
			)
		);
	}

	/**
	 * Checks access permission.
	 * Checks if the user is logged-in and checks of the user role has access.
	 *
	 * @return boolean
	 */
	public function permission_check() {

		$is_access_level = $this->wp_analytify->settings->get_option( 'show_analytics_roles_dashboard', 'wp-analytify-dashboard', array( 'administrator' ) );

		return (bool) $this->wp_analytify->pa_check_roles( $is_access_level );
	}

	/**
	 * Handle the request.
	 *
	 * @param WP_REST_Request $request

	 * @since 2.1.23
	 *
	 * @return array|WP_Error
	 */
	public function handle_request( WP_REST_Request $request ) {

		$is_access_level = $this->wp_analytify->settings->get_option( 'show_analytics_roles_dashboard', 'wp-analytify-dashboard', array( 'Administrator' ) );

		// Generate error if unauthorized user send the request.
		if ( ! $this->wp_analytify->pa_check_roles( $is_access_level ) ) {
			return new WP_Error( 'analytify_forbidden', __( 'You are not allowed to access Analytify Dashboard.', 'wp-analytify' ), array( 'status' => 403 ) );
		}

		$request_type = $request->get_param( 'request_type' );

		$this->start_date         = $request->get_param( 'sd' );
		$this->end_date           = $request->get_param( 'ed' );
		$this->compare_start_date = $request->get_param( 'csd' );
		$this->compare_end_date   = $request->get_param( 'ced' );
		$this->date_difference    = $request->get_param( 'dd' );
		$this->report_url         = $request->get_param( 'ru' );
		$this->report_date_range  = $request->get_param( 'rdr' );

		switch ( $request_type ) {
			case 'general-stats':
				return $this->general_stats();

			case 'categories-performance':
				return $this->categories_performance();

			case 'country-performance':
				return $this->country_performance();

			case 'coupons':
				return $this->coupons();

			case 'list-performance':
				return $this->list_performance();

			case 'measuring-roi':
				return $this->measuring_roi();

			case 'products-performance':
				return $this->products_performance();
		}

		// If no request type match, Return error.
		return new WP_Error( 'analytify_invalid_endpoint', __( 'Invalid endpoint.', 'wp-analytify' ), array( 'status' => 404 ) );
	}

	private function categories_performance() {

		if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {
			$stats = $this->wp_analytify->get_reports(
				'show-woo-categories-performance-stats',
				array(
					'itemRevenue',
					// 'ecommercePurchases', commented out due to incompatibility.
					'itemPurchaseQuantity',
					'cartToViewRate',
					// TODO: (missing_ga4) buyToDetailRate
				),
				array(
					'start' => $this->start_date,
					'end'   => $this->end_date,
				),
				array(
					'itemCategory',
				),
				array(
					'type' => 'metric',
					'name' => 'itemPurchaseQuantity',
				),
				array(
					'logic'   => 'AND',
					'filters' => array(
						array(
							'type'           => 'dimension',
							'name'           => 'itemCategory',
							'match_type'     => 4,
							'value'          => '(not set)',
							'not_expression' => true,
						),
					),
				),
				10
			);

			include WC_ANALYTIFY_PLUGIN_DIR . '/views/default/categories-performance.php';
			return wpa_woocommerce_categories_performance( $this->wp_analytify, $stats );
		} else {
			$stats = $this->wp_analytify->pa_get_analytics_dashboard( 'ga:itemRevenue,ga:uniquePurchases,ga:itemQuantity,ga:cartToDetailRate,ga:buyToDetailRate', $this->start_date, $this->end_date, 'ga:productCategoryHierarchy', '-ga:itemQuantity', 'ga:productCategoryHierarchy!=(not set)', 10, 'show-woo-categories-performance-stats' );

			if ( isset( $stats->totalsForAllResults ) ) {
				include WC_ANALYTIFY_PLUGIN_DIR . '/views/default/categories-performance-deprecated.php';
				return pa_include_wocommerce_categories_performance( $this->wp_analytify, $stats );
			}
		}
	}

	private function country_performance() {

		if ( 'ga4' === $this->ga_mode ) {
			$stats = $this->wp_analytify->get_reports(
				'show-woo-geographical-stats',
				array(
					'addToCarts',
					'itemViews',
				),
				array(
					'start' => $this->start_date,
					'end'   => $this->end_date,
				),
				array(
					'country',
				),
				array(
					'type' => 'metric',
					'name' => 'itemViews',
				),
				array(
					'logic'   => 'AND',
					'filters' => array(
						array(
							'type'           => 'dimension',
							'name'           => 'country',
							'match_type'     => 4,
							'value'          => '(not set)',
							'not_expression' => true,
						),
					),
				),
				10
			);

			include WC_ANALYTIFY_PLUGIN_DIR . '/views/default/country-performance.php';
			return wpa_country_performance( $this->wp_analytify, $stats, $this->report_url, $this->report_date_range );
		} else {
			$analytify_enhanced_geographic_filter = apply_filters( 'analytify_enhanced_geographic_filter', 'ga:country!=(not set);ga:itemQuantity>0' );
			$stats                                = $this->wp_analytify->pa_get_analytics_dashboard( 'ga:itemQuantity,ga:transactionRevenue', $this->start_date, $this->end_date, 'ga:country', '-ga:itemQuantity', $analytify_enhanced_geographic_filter, 10, 'show-woo-geographical-stats' );

			if ( isset( $stats->totalsForAllResults ) ) {
				include WC_ANALYTIFY_PLUGIN_DIR . '/views/default/country-performance-deprecated.php';
				return pa_include_country_performance( $this->wp_analytify, $stats, $this->report_url, $this->report_date_range );
			}
		}
	}

	private function coupons() {

	}

	private function list_performance() {

	}

	private function measuring_roi() {

	}

	private function products_performance() {

	}

	private function general_stats() {
		$analytify_enhanced_general_filter = apply_filters( 'analytify_enhanced_general_filter', false );

		if ( 'ga4' === $this->ga_mode ) {

			$general_stats = $this->wp_analytify->get_reports(
				'show-woo-general-stats',
				array(
					'transactions',
					'totalRevenue',
					// 'transactionsPerPurchaser', this was being used for average order value which is wrong
					'averagePurchaseRevenue',
					'ecommercePurchases',
					// 'ga:transactionsPerSession, TODO: missing_ga4
					'addToCarts',
					// 'ga:productRemovesFromCart, TODO: missing_ga4
					'checkouts',
					'itemListClicks',
					'itemViews',
				),
				array(
					'start' => $this->start_date,
					'end'   => $this->end_date,
				),
				array(),
				array(),
				array(),
				10
			);

			$remove_from_cart = $this->wp_analytify->get_reports(
				'remove-from-cart-stats',
				array(
					'eventCount',
				),
				array(
					'start' => $this->start_date,
					'end'   => $this->end_date,
				),
				array(
					'eventName',
				),
				array(),
				array(
					'logic'   => 'AND',
					'filters' => array(
						array(
							'type'       => 'dimension',
							'name'       => 'eventName',
							'match_type' => 1,
							'value'      => 'remove_from_cart',
						),
					),
				),
				0
			);

			if ( isset( $remove_from_cart['aggregations']['eventCount'] ) ) {
				$general_stats['remove_from_cart'] = $remove_from_cart['aggregations']['eventCount'];
			}

			if ( isset( $general_stats['aggregations'] ) ) {
				include WC_ANALYTIFY_PLUGIN_DIR . '/views/default/shopping-analysis.php';
				return pa_ga_include_woocommerce( $this->wp_analytify, $general_stats );
			}
		} else {

			$stats = $this->wp_analytify->pa_get_analytics_dashboard( 'ga:transactions,ga:transactionRevenue,ga:revenuePerTransaction,ga:uniquePurchases,ga:transactionsPerSession,ga:productAddsToCart,ga:productRemovesFromCart,ga:productCheckouts,ga:productListClicks,ga:productDetailViews', $this->start_date, $this->end_date, false, false, $analytify_enhanced_general_filter, false, 'show-woo-general-stats' );

			if ( isset( $stats->totalsForAllResults ) ) {
				include WC_ANALYTIFY_PLUGIN_DIR . '/views/default/shopping-analysis-deprecated.php';
				return pa_include_woocommerce( $this->wp_analytify, $stats );
			}
		}
	}

}

/**
 * Init the instance.
 */
Analytify_Woo_Rest_API::get_instance();
