/**
 * Import react and react-dom for the externals:
 *
 * Ref:
 *
 * - https://webpack.js.org/configuration/externals/
 * - https://www.cssigniter.com/how-to-use-external-react-components-in-your-gutenberg-blocks/
 */
import React from 'react';
import ReactDom from 'react-dom';

/**
 * Import react-select.
 *
 * Ref: https://github.com/JedWatson/react-select
 */
import Select from 'react-select';

//Import the CSS files used in this block
import './editor.css';
import './style.css';

//Get the registerBlockType() function used for the registration of the block
const {registerBlockType} = wp.blocks;

//Get the base Component used to defined new components
const {Component} = wp.element;

//Import block components
const {InspectorControls} = wp.editor;

//Import Inspector components
const {
  PanelBody,
  SelectControl
} = wp.components;

//Deconstruct just the __ function from wp.i18n global object
const {__} = wp.i18n;

class BlockEdit extends Component {

  constructor(props) {

    super(...arguments);
    this.props = props;
    this.state = {
      eventList: [],
      selectedOption: null,
      placeholder: '',
    };

  }

  /**
   * This method is invoked immediately after a component is mounted (inserted
   * into the tree). Initializations that requires DOM nodes should go here. If
   * you need to load data from a remote endpoint, this is a good place to
   * instantiate the network requests.
   *
   * https://reactjs.org/docs/react-component.html#componentdidmount
   */
  componentDidMount() {

    //Init the "Event" React Select ------------------------------------------------------------------------------------
    let {attributes: {eventId}} = this.props;

    let params = 'security=' + dale_nonce;
    params += '&action=dale_get_event_list';

    fetch(dale_ajax_url, {
      method: "POST",
      credentials: 'include',//Required to avoid the 400 Bad Request error with Edge
      headers: {'Content-type': 'application/x-www-form-urlencoded'},
      body: params,
    })
    .then((response) => {

      if(response.ok) {
        return response.json();
      }else{
        throw new Error('Network response was not ok.');
      }

    })
    .then((data) => {

      //Update the React Select options
      let options = data.map((item) => ({
        value: item.event_id,
        label: item.name,
      }));
      this.setState({
        eventList: options,
      });

      //Find the selected item
      const selectedItem = options.find(function(item) {
        if(item.value === eventId){
          return item;
        }
      });

      //Set the React Select Placeholder or set the React Select selected option
      if(typeof selectedItem === 'undefined'){
        this.setState({
          placeholder: __( 'Select...', 'dale' )
      });
      }else{
        this.setState({
          selectedOption: {
            value: eventId,
            label: selectedItem.label
          }
        });
      }

    })
    .catch(error => {

      console.log('There has been a problem with your fetch operation: ', error.message);

    });

  }

  render() {

    const {attributes: {eventId}, setAttributes} = this.props;

    return <Select
            classNamePrefix="dale-react-select"
            value={this.state.selectedOption}
            onChange={( selectedOption ) => {
              setAttributes({eventId: selectedOption.value});
              this.setState({selectedOption});
            }}
            options={this.state.eventList}
            placeholder={this.state.placeholder}
            noOptionsMessage={() => __('No options', 'dale')}
        />;

  }

}

/**
 * Register the Gutenberg block
 */
registerBlockType('dale/event', {
  // Block name. Block names must be string that contains a namespace prefix. Example: my-plugin/my-custom-block.
  title: 'Event', // Block title.
  icon: 'microphone', //Block icon.
  category: 'widgets', // Block category — Group blocks together based on common traits E.g. common, formatting, layout widgets, embed.
  keywords: [
    __('live', 'dale'),
    __('event', 'dale'),
    __('commentary', 'dale'),
  ],
  attributes: {
    eventId: {
      type: 'string',
    }
  },

  /**
   * The edit function describes the structure of your block in the context of the editor.
   * This represents what the editor will render when the block is used.
   *
   * The "edit" property must be a valid function.
   *
   * @link https://wordpress.org/gutenberg/handbook/block-api/block-edit-save/
   */
  edit: BlockEdit,

  /**
   * The save function defines the way in which the different attributes should be combined
   * into the final markup, which is then serialized by Gutenberg into post_content.
   *
   * The "save" property must be specified and must be a valid function.
   *
   * @link https://wordpress.org/gutenberg/handbook/block-api/block-edit-save/
   */
  save: function() {

    /**
     * This is a dynamic block and the rendering is performed with PHP:
     *
     * https://wordpress.org/gutenberg/handbook/blocks/creating-dynamic-blocks/
     */
    return null;

  },

});
