<?php

/*
 * this class should be used to stores properties and methods shared by the
 * admin and public side of wordpress
 */

class dale_Shared
{

    //regex
    public $regex_number_ten_digits = '/^\s*\d{1,10}\s*$/';
    public $regex_capability = '/^\s*[A-Za-z0-9_]+\s*$/';
    public $hex_rgb_regex = '/^#(?:[0-9a-fA-F]{3}){1,2}$/';
    public $font_family_regex = '/^([A-Za-z0-9-\'", ]*)$/';
    public $url_regex = '/^(http|https):\/\/[-A-Za-z0-9+&@#\/%?=~_|$!:,.;]+$/';

    protected static $instance = null;

    private $data = array();

    private function __construct()
    {

        //Set plugin textdomain
        load_plugin_textdomain('dale', false, 'daext-live-events/lang/');

        $this->data['slug'] = 'dale';
        $this->data['ver']  = '1.34';
        $this->data['dir']  = substr(plugin_dir_path(__FILE__), 0, -7);
        $this->data['url']  = substr(plugin_dir_url(__FILE__), 0, -7);

    }

    public static function get_instance()
    {

        if (null == self::$instance) {
            self::$instance = new self;
        }

        return self::$instance;

    }

    //retrieve data
    public function get($index)
    {
        return $this->data[$index];
    }

    /**
     *
     * Get the timeline name from the timeline_id.
     *
     * @param Int $timeline_id
     *
     * @return String
     */
    public function get_timeline_name($timeline_id)
    {

        global $wpdb;
        $table_name   = $wpdb->prefix . $this->get('slug') . "_timeline";
        $safe_sql     = $wpdb->prepare("SELECT * FROM $table_name WHERE timeline_id = %d ", $timeline_id);
        $timeline_obj = $wpdb->get_row($safe_sql);

        return $timeline_obj->name;

    }

    public function get_timeline_time_mode($timeline_id)
    {

        global $wpdb;
        $table_name   = $wpdb->prefix . $this->get('slug') . "_timeline";
        $safe_sql     = $wpdb->prepare("SELECT * FROM $table_name WHERE timeline_id = %d ", $timeline_id);
        $timeline_obj = $wpdb->get_row($safe_sql);

        return $timeline_obj->time_mode;

    }

    /**
     * Get the stat name based on the stat id.
     *
     * @param $stat_id
     *
     * @return string
     */
    public function get_stat_name($stat_id)
    {

        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_stat";
        $safe_sql   = $wpdb->prepare("SELECT * FROM $table_name WHERE stat_id = %d ", $stat_id);
        $stat_obj   = $wpdb->get_row($safe_sql);

        return $stat_obj->name;

    }

    /**
     * Get the stat_id based on the stat_group_id
     *
     * @param $stat_group_id
     *
     * @return int
     */
    public function get_stat_section($stat_group_id){

        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_stat_group";
        $safe_sql   = $wpdb->prepare("SELECT stat_id FROM $table_name WHERE stat_group_id = %d ", $stat_group_id);
        $stat_obj   = $wpdb->get_row($safe_sql);

        return $stat_obj->stat_id;

    }

    /**
     * Get the stat group name based on the stat group id.
     *
     * @param $stat_group_id
     *
     * @return string
     */
    public function get_stat_group_name($stat_group_id)
    {

        global $wpdb;
        $table_name     = $wpdb->prefix . $this->get('slug') . "_stat_group";
        $safe_sql       = $wpdb->prepare("SELECT * FROM $table_name WHERE stat_group_id = %d ", $stat_group_id);
        $stat_group_obj = $wpdb->get_row($safe_sql);

        return $stat_group_obj->name;

    }

    /**
     * Get the stat group object based on the stat group id.
     *
     * @param $stat_group_id
     *
     * @return string
     */
    public function get_stat_group_obj($stat_group_id)
    {

        global $wpdb;
        $table_name     = $wpdb->prefix . $this->get('slug') . "_stat_group";
        $safe_sql       = $wpdb->prepare("SELECT * FROM $table_name WHERE stat_group_id = %d ", $stat_group_id);
        $stat_group_obj = $wpdb->get_row($safe_sql);

        return $stat_group_obj;

    }

    /**
     * Returns the stat_group_id of the specified stat_item_id.
     *
     * @param $stat_group_id
     *
     * @return int
     */
    public function get_stat_group_from_stat_item($stat_item_id)
    {

        global $wpdb;
        $table_name     = $wpdb->prefix . $this->get('slug') . "_stat_item";
        $safe_sql       = $wpdb->prepare("SELECT * FROM $table_name WHERE stat_item_id = %d ", $stat_item_id);
        $stat_item_obj = $wpdb->get_row($safe_sql);

        return intval($stat_item_obj->stat_group_id, 10);

    }

    /**
     * Returns an object with all the information associated with the stat item.
     *
     * @param $stat_item_id
     *
     * @return Object
     */
    public function get_stat_item_obj($stat_item_id){

        global $wpdb;
        $table_name     = $wpdb->prefix . $this->get('slug') . "_stat_item";
        $safe_sql       = $wpdb->prepare("SELECT * FROM $table_name WHERE stat_item_id = %d ", $stat_item_id);
        $stat_item_obj = $wpdb->get_row($safe_sql);

        return $stat_item_obj;

    }

    /**
     * Returns an object with the data of the comment specified with the $comment_id
     *
     * @param $comment_id
     *
     * @return object
     */
    public function get_comment_data($comment_id)
    {

        global $wpdb;
        $table_name  = $wpdb->prefix . $this->get('slug') . "_comment";
        $safe_sql    = $wpdb->prepare("SELECT * FROM $table_name WHERE comment_id = %d ", $comment_id);
        $comment_obj = $wpdb->get_row($safe_sql);

        return $comment_obj;

    }

    /**
     * Given the comment_id True is returned if the comments are closed and False is returned if the comments are
     * open.
     *
     * @param $comment_id
     *
     * @return bool True if the comments are closed and False if the comment are open.
     */
    public function comments_are_closed($comment_id)
    {

        global $wpdb;
        $table_name  = $wpdb->prefix . $this->get('slug') . "_comment";
        $safe_sql    = $wpdb->prepare("SELECT * FROM $table_name WHERE comment_id = %d ", $comment_id);
        $comment_obj = $wpdb->get_row($safe_sql);

        return intval($comment_obj->closed, 10);

    }

    public function get_number_of_events()
    {

        global $wpdb;
        $table_name  = $wpdb->prefix . $this->get('slug') . "_event";
        $total_items = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");

        return intval($total_items, 10);

    }

    public function get_number_of_timelines()
    {

        global $wpdb;
        $table_name  = $wpdb->prefix . $this->get('slug') . "_timeline";
        $total_items = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");

        return intval($total_items, 10);

    }

    public function get_number_of_comment_sections()
    {

        global $wpdb;
        $table_name  = $wpdb->prefix . $this->get('slug') . "_comment";
        $total_items = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");

        return intval($total_items, 10);

    }

    public function get_number_of_timeline_items($timeline_id)
    {

        global $wpdb;
        $table_name  = $wpdb->prefix . $this->get('slug') . "_timeline_item";
        $safe_sql = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE timeline_id = %d", $timeline_id);
        $total_items = $wpdb->get_var($safe_sql);

        return intval($total_items, 10);

    }

    public function get_number_of_stats()
    {

        global $wpdb;
        $table_name  = $wpdb->prefix . $this->get('slug') . "_stat";
        $total_items = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");

        return intval($total_items, 10);

    }

    public function get_number_of_stats_groups()
    {

        global $wpdb;
        $table_name  = $wpdb->prefix . $this->get('slug') . "_stat_group";
        $total_items = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");

        return intval($total_items, 10);

    }

    /**
     * Given an event_id True is returned if the event exists, otherwise False is returned.
     *
     * @param $event_id
     *
     * @return bool
     */
    public function event_exists($event_id)
    {
        global $wpdb;
        $table_name       = $wpdb->prefix . $this->get('slug') . "_event";
        $safe_sql         = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE event_id = %d ",
            intval($event_id, 10));
        $number_of_events = intval($wpdb->get_var($safe_sql), 10);
        if ($number_of_events === 1) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Given a timeline_id True is returned if the timeline exists, otherwise False is returned.
     *
     * @param $timeline_id
     *
     * @return bool
     */
    public function timeline_exists($timeline_id)
    {
        global $wpdb;
        $table_name          = $wpdb->prefix . $this->get('slug') . "_timeline";
        $safe_sql            = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE timeline_id = %d ",
            intval($timeline_id, 10));
        $number_of_timelines = intval($wpdb->get_var($safe_sql), 10);
        if ($number_of_timelines === 1) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Given a timeline_item_id True is returned if the timeline item exists, otherwise False is returned.
     *
     * @param $timeline_item_id
     *
     * @return bool
     */
    public function timeline_item_exists($timeline_item_id)
    {
        global $wpdb;
        $table_name               = $wpdb->prefix . $this->get('slug') . "_timeline_item";
        $safe_sql                 = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE timeline_item_id = %d ",
            intval($timeline_item_id, 10));
        $number_of_timeline_items = intval($wpdb->get_var($safe_sql), 10);
        if ($number_of_timeline_items === 1) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Given a timeline_icon_id True is returned if the timeline icon exists (if it's the default icon or if it's found
     * in the database), otherwise False is returned.
     *
     * @param $timeline_icon_id
     *
     * @return bool
     */
    public function timeline_icon_exists($timeline_icon_id)
    {

        if ($timeline_icon_id === 0) {
            //Default Timeline Icon
            return true;
        } else {
            //Search in the Database
            global $wpdb;
            $table_name               = $wpdb->prefix . $this->get('slug') . "_timeline_icon";
            $safe_sql                 = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE timeline_icon_id = %d ",
                intval($timeline_icon_id, 10));
            $number_of_timeline_icons = intval($wpdb->get_var($safe_sql), 10);
            if ($number_of_timeline_icons === 1) {
                return true;
            } else {
                return false;
            }
        }


    }

    /**
     * Given a stat_id True is returned if the stat exists, otherwise False is returned.
     *
     * @param $stat_id
     *
     * @return bool
     */
    public function stat_exists($stat_id)
    {
        global $wpdb;
        $table_name      = $wpdb->prefix . $this->get('slug') . "_stat";
        $safe_sql        = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE stat_id = %d ", intval($stat_id, 10));
        $number_of_stats = intval($wpdb->get_var($safe_sql), 10);
        if ($number_of_stats === 1) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Given a stat_group_id True is returned if the stat group exists, otherwise False is returned.
     *
     * @param $stat_group_id
     *
     * @return bool
     */
    public function stat_group_exists($stat_group_id)
    {
        global $wpdb;
        $table_name           = $wpdb->prefix . $this->get('slug') . "_stat_group";
        $safe_sql             = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE stat_group_id = %d ",
            intval($stat_group_id, 10));
        $number_of_stat_group = intval($wpdb->get_var($safe_sql), 10);
        if ($number_of_stat_group === 1) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Given a stat_item_id True is returned if the stat item exists, otherwise False is returned.
     *
     * @param $stat_group_id
     *
     * @return bool
     */
    public function stat_item_exists($stat_item_id)
    {
        global $wpdb;
        $table_name          = $wpdb->prefix . $this->get('slug') . "_stat_item";
        $safe_sql            = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE stat_item_id = %d ",
            intval($stat_item_id, 10));
        $number_of_stat_item = intval($wpdb->get_var($safe_sql), 10);
        if ($number_of_stat_item === 1) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Given a comment_id True is returned if the comment exists, otherwise False is returned.
     *
     * @param $comment_id
     *
     * @return bool
     */
    public function comment_exists($comment_id)
    {
        global $wpdb;
        $table_name         = $wpdb->prefix . $this->get('slug') . "_comment";
        $safe_sql           = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE comment_id = %d ",
            intval($comment_id, 10));
        $number_of_comments = intval($wpdb->get_var($safe_sql), 10);
        if ($number_of_comments === 1) {
            return true;
        } else {
            return false;
        }
    }

    public function display_stat_type($type_code)
    {

        switch ($type_code) {
            case 0:
                return esc_attr__('Single Data', 'dale');
                break;
            case 1:
                return esc_attr__('Single Visual', 'dale');
                break;
            case 2:
                return esc_attr__('Comparison Data', 'dale');
                break;
            case 3:
                return esc_attr__('Comparison Visual', 'dale');
                break;
        }

    }

    /**
     * Get the stat name based on the stat id.
     *
     * @param $stat_id
     *
     * @return string
     */
    public function get_event_name($event_id)
    {

        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_event";
        $safe_sql   = $wpdb->prepare("SELECT * FROM $table_name WHERE event_id = %d ", $event_id);
        $event_obj  = $wpdb->get_row($safe_sql);

        return $event_obj->name;

    }

    /**
     * If the timeline_icon is used in a timeline_item True is returned, otherwise False is returned.
     *
     * @param $timeline_icon_id
     *
     * @return bool
     */
    public function timeline_icon_is_used($timeline_icon_id)
    {

        global $wpdb;
        $table_name          = $wpdb->prefix . $this->get('slug') . "_timeline_item";
        $safe_sql            = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE icon = %d ", $timeline_icon_id);
        $timeline_item_count = $wpdb->get_var($safe_sql);

        if (intval($timeline_item_count, 10) > 0) {
            return true;
        } else {
            return false;
        }

    }

    /*
     * Applies wp_kses() with the allowed tags and parameters defined in the plugin options.
     *
     * @param String The HTML that should be filtered
     * @return String The HTML filter by wp_kses() (if "Apply kses" is enabled) or the unfilter HTML (if "Apply kses" is
     * not enabled).
     */
    public function apply_custom_kses($string)
    {

        if (get_option($this->get('slug') . '_advanced_apply_kses') == 1) {

            /*
             * The final $allowed_html should have this format, for more info see
             * https://codex.wordpress.org/Function_Reference/wp_kses
             *
             * $allowed_html = array(
             *     'a' => array(
             *         'href' => array(),
             *         'title' => array()
             *     ),
             *     'br' => array(),
             *     'em' => array(),
             *     'strong' => array(),
             * );
             */

            $allowed_html           = array();
            $kses_allowed_html_tags = get_option($this->get('slug') . '_advanced_kses_allowed_html_tags');
            $list_of_tags_a         = explode(',', $kses_allowed_html_tags);

            foreach ($list_of_tags_a as $tag_with_attributes) {

                //store the allowed attributes of this tag in an array with a regex
                preg_match_all('/\[([a-z-]*)\]/', $tag_with_attributes, $matches);
                $allowed_attributes_a = $matches[1];

                //remove the allowed attributes from the string and store the allowed tag in a variable with a regex
                $allowed_tag = trim(preg_replace('(\[[a-z-]*\])', '', $tag_with_attributes));

                //create an array with the proper form required by the wp_kses() function
                $attributes = array();
                foreach ($allowed_attributes_a as $allowed_attribute) {
                    $attributes[$allowed_attribute] = array();
                }

                //add the tag only if it's validated with a regex
                if (preg_match('/^[a-z]*$/', $allowed_tag)) {
                    $allowed_html[$allowed_tag] = $attributes;
                }

            }

            /*
             * Set the allowed protocols.
             *
             * The final $allowed_protocols should have this format, for more info see
             * https://codex.wordpress.org/Function_Reference/wp_kses
             *
             * $allowed_protocols = array('http', 'https', 'ftp');
             */

            //get the value in the option
            $allowed_protocols = get_option($this->get('slug') . '_advanced_kses_allowed_protocols');

            //remove all the spaces
            $allowed_protocols = str_replace(' ', '', $allowed_protocols);

            //convert to an array
            $allowed_protocols = explode(',', $allowed_protocols);

            //Use the custom $allowed_html and $allowed_protocols to apply wp_kses()
            return wp_kses($string, $allowed_html, $allowed_protocols);

        } else {

            return $string;

        }

    }

    /**
     * Given the timeline_id the event_id of the event associated with the timeline is returned.
     *
     * @param $timeline_id
     *
     * @return mixed
     */
    public function get_event_of_timeline($timeline_id)
    {

        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_timeline";
        $safe_sql   = $wpdb->prepare("SELECT event_id FROM $table_name WHERE timeline_id = %d ", $timeline_id);
        $event_obj  = $wpdb->get_row($safe_sql);

        return $event_obj->event_id;

    }

    /**
     * Given the stat_id the event_id of the event associated with the stat is returned.
     *
     * @param $stat_id
     *
     * @return mixed
     */
    public function get_event_of_stat($stat_id)
    {

        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_stat";
        $safe_sql   = $wpdb->prepare("SELECT event_id FROM $table_name WHERE stat_id = %d ", $stat_id);
        $event_obj  = $wpdb->get_row($safe_sql);

        return $event_obj->event_id;

    }

    /**
     * Given the comment_id the event_id of the event associated with the comment is returned.
     *
     * @param $comment_id
     *
     * @return mixed
     */
    public function get_event_of_comment($comment_id)
    {

        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_comment";
        $safe_sql   = $wpdb->prepare("SELECT event_id FROM $table_name WHERE comment_id = %d ", $comment_id);
        $event_obj  = $wpdb->get_row($safe_sql);

        return $event_obj->event_id;

    }

    public function is_comment_spam($user_id)
    {

        //get all the comment_item posted by the user in the last 24 hours
        global $wpdb;
        $table_name             = $wpdb->prefix . $this->get('slug') . "_comment_item";
        $safe_sql               = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE user_id = %d AND date_gmt BETWEEN DATE_SUB(NOW(), INTERVAL 1 DAY) AND NOW()",
            $user_id);
        $number_of_comment_item = $wpdb->get_var($safe_sql);

        if ($number_of_comment_item >= intval(get_option($this->get('slug') . '_advanced_comment_limit'))) {
            return true;
        } else {
            return false;
        }

    }

    /**
     * The tweet id is searched in $string and is detected in the following cases:
     *
     * - When $string is a number
     * - When $string is a Tweet URL
     * - When $string is a Tweet Embed Code
     *
     * Note that the only tweet id that are below 50 characters are considered valid.
     *
     * @param $string
     *
     * @return bool
     */
    public function find_tweet_id($string){

        /**
         * If it's numeric return it.
         *
         * Example of a tweet id: 938080604268220416
         */
        if(preg_match('/^\s*\d{1,50}\s*$/', $string, $match) === 1){

           if(strlen($match[0]) <= 50){
               return $match[0];
           }else{
               return false;
           }

        }

        /**
         * Use the same regex to find the tweet_id inside the Tweet URL or inside a Tweet Embed Code:
         *
         * Example of Tweet URL: https://twitter.com/Bitbucket/status/938080604268220416
         * Example of Tweet Embed Code: <blockquote class="twitter-tweet" data-lang="en"><p lang="en" dir="ltr">Introducing Bitbucket Deployments: giving you confidence to release early and often. <a href="https://t.co/RRi0wFgnek">https://t.co/RRi0wFgnek</a> <a href="https://t.co/qZOTct6eiw">pic.twitter.com/qZOTct6eiw</a></p>&mdash; Atlassian Bitbucket (@Bitbucket) <a href="https://twitter.com/Bitbucket/status/938080604268220416?ref_src=twsrc%5Etfw">December 5, 2017</a></blockquote> <script async src="https://platform.twitter.com/widgets.js" charset="utf-8"></script>
         */
        if(preg_match('/https?:\/\/twitter.com\/[a-zA-Z_]{1,20}\/status\/([0-9]{1,50})/', $string, $match) === 1){

            if(strlen($match[1]) <= 50){
                return $match[1];
            }else{
                return false;
            }

        };

        //if it's not numeric and not a tweet embed return false
        return false;

    }

    /**
     * Returns true is the event is live or false if the event is not live
     *
     * @param $event_id
     *
     * @return bool
     */
    public function is_event_live($event_id){

        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_event";
        $safe_sql   = $wpdb->prepare("SELECT live FROM $table_name WHERE event_id = %d", $event_id);
        $event_obj  = $wpdb->get_row($safe_sql);

        if(intval($event_obj->live, 10) === 1){
            return true;
        }else{
            return false;
        }

    }

	/**
	 * Generate the timezone offsets from UTC-12:00 to UTC+14:00
	 *
	 * @return Array An array with the timezone offsets
	 */
	public function generate_timezone_offsets() {

		return [
			'-12:00',
			'-11:30',
			'-11:00',
			'-10:30',
			'-10:00',
			'-09:30',
			'-09:00',
			'-08:30',
			'-08:00',
			'-07:30',
			'-07:00',
			'-06:30',
			'-06:00',
			'-05:30',
			'-05:00',
			'-04:30',
			'-04:00',
			'-03:30',
			'-03:00',
			'-02:30',
			'-02:00',
			'-01:30',
			'-01:00',
			'-00:30',
			'+00:00',
			'+00:30',
			'+01:00',
			'+01:30',
			'+02:00',
			'+02:30',
			'+03:00',
			'+03:30',
			'+04:00',
			'+04:30',
			'+05:00',
			'+05:30',
			'+05:45',
			'+06:00',
			'+06:30',
			'+07:00',
			'+07:30',
			'+08:00',
			'+08:30',
			'+08:45',
			'+09:00',
			'+09:30',
			'+10:00',
			'+10:30',
			'+11:00',
			'+11:30',
			'+12:00',
			'+12:45',
			'+13:00',
			'+13:45',
			'+14:00'
		];

	}

}