/**
 * DeBlocker
 * Most effective way to detect ad blockers. Ask the visitors to disable their ad blockers.
 * Exclusively on https://1.envato.market/deblocker
 *
 * @encoding        UTF-8
 * @version         3.3.9
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

/**
 * @param window.mdpDeBlockerAdmin.translationsCount
 */

jQuery( function ( $ ) {

    "use strict";

    $( document ).ready( function () {

        let { mdpDeblockerAdmin } = window;

        const transitionDuration = 100;
        let showedTranslations = parseInt( mdpDeblockerAdmin.translationsCount ?? '35' );

        /**
         * Hide or close next <tr> after switch
         * @param $element
         * @param num
         */
        function switchSingle( $element, num ) {

            for ( let i = 0; i < num; i++ ) {

                $element.is( ':checked' ) ?
                    $element.closest( 'tr' ).nextAll( 'tr' ).eq( i ).show( transitionDuration ) :
                    $element.closest( 'tr' ).nextAll( 'tr' ).eq( i ).hide( transitionDuration );

            }

        }

        /**
         * Init single switch
         * @param $element
         * @param num
         */
        function initSingleSwitch( $element, num = 1 ) {

            switchSingle( $element, num );

            $element.on( 'change', () => {

                switchSingle( $element, num );

            } );

        }

        /**
         * Init single select
         * @param $element
         * @param condition
         * @param num
         */
        function initSingleSelect( $element, condition, num = 1 ) {

            selectSingle( $element, num, condition );

            $element.on( 'change', () => {

                selectSingle( $element, num, condition );

            } );

        }

        /**
         * Hide or close next <tr> after select
         * @param $element
         * @param num
         * @param conditionValue
         */
        function selectSingle( $element, num, conditionValue ) {

            for ( let i = 0; i < num; i++ ) {

                if ( typeof conditionValue === 'object' ) {

                    let showElement = true
                    conditionValue.forEach( conditionValue => {

                        showElement = $element.val() !== conditionValue && showElement;

                    } );

                    showElement ?
                        $element.closest( 'tr' ).nextAll( 'tr' ).eq( i ).show( 300 ) :
                        $element.closest( 'tr' ).nextAll( 'tr' ).eq( i ).hide( 300 );

                } else {

                    $element.val() !== conditionValue ?
                        $element.closest( 'tr' ).nextAll( 'tr' ).eq( i ).show( 300 ) :
                        $element.closest( 'tr' ).nextAll( 'tr' ).eq( i ).hide( 300 );

                }

            }

        }

        /**
         * Init translation switcher
         */
        function initTranslationSwitcher( translationSwitcher ) {

            initTranslationFields( translationSwitcher );

            translationSwitcher.on( 'change', () => {

                translationSwitcher.is( ':checked' ) ?
                    translationSwitcher.closest( 'tr' ).nextAll().show( transitionDuration ):
                    translationSwitcher.closest( 'tr' ).nextAll().hide( transitionDuration );

                manageEmptyLocales( 'hide' );

            } );

        }

        /**
         * Init translation fields (show or hide)
         * @param translationSwitcher
         */
        function initTranslationFields( translationSwitcher ) {

            translationSwitcher.is( ':checked' ) ?
                translationSwitcher.closest( 'tr' ).nextAll().show( transitionDuration ):
                translationSwitcher.closest( 'tr' ).nextAll().hide( transitionDuration );

        }

        /**
         * Hide empty locales
         */
        function manageEmptyLocales( action ) {

            const localeFields = $( '.translations-locale' );
            const hideBefore = 1;
            const hideAfter = $( `#mdp_deblocker_translations_settings_translate_button_caption_1` ).length > 0 ? 3 : 2;

            localeFields.each( function() {

                const locale = $( this ).children( 'input' ).val();

                // Hide all groups with empty locales
                if ( locale === '' ) {

                    // Hide field-group with empty locale
                    if ( action === 'hide' ) {

                        $( this ).closest( 'tr' ).prevAll().slice( 0, hideBefore ).hide( transitionDuration );
                        $( this ).closest( 'tr' ).nextAll().slice( 0, hideAfter ).hide( transitionDuration );
                        $( this ).closest( 'tr' ).hide();

                        showedTranslations--;

                    }

                    // Show field-group with empty locale
                    if ( action === 'show' ) {

                        $( this ).closest( 'tr' ).prevAll().slice( 0, hideBefore ).show( transitionDuration );
                        $( this ).closest( 'tr' ).nextAll().slice( 0, hideAfter ).show( transitionDuration );
                        $( this ).closest( 'tr' ).show();

                        showedTranslations++;

                        return false;

                    }

                }

            } );

        }

        /**
         * Init translation buttons
         */
        function initTranslationButtons() {

            const $addButton = $( '#mdp-add-translation' );
            $addButton.on( 'click', ( e ) => {

                e.preventDefault();
                manageEmptyLocales( 'show' );
                manageButtons();

            } );

            const $removeButton = $( '#mdp-remove-translation' );
            $removeButton.detach();
            $addButton.parent().append( $removeButton );

            $removeButton.on( 'click', ( e ) => {

                e.preventDefault();
                notEmptyLocaleNotice();
                manageEmptyLocales( 'hide' );
                manageButtons();

            } );

        }

        /**
         * Show or hide buttons if possible to show or hide something
         */
        function manageButtons() {

            const $addButton = $( '#mdp-add-translation' );
            const $removeButton = $( '#mdp-remove-translation' );

            // Hide button if nothing to show
            calculateLocales( 'show' ) >= parseInt( mdpDeblockerAdmin.translationsCount ?? '35' ) ?
                $addButton.hide( 0 ) : $addButton.show( 0 );

            // Hide button if nothing to hide
            calculateLocales( 'hide' ) >= parseInt( mdpDeblockerAdmin.translationsCount ?? '35' ) ?
                $removeButton.hide( 0 ) : $removeButton.show( 0 );

        }

        /**
         * Calculate showing or hided locales
         * @param action
         * @returns {number}
         */
        function calculateLocales( action ) {

            const localeFields = $('.translations-locale');
            let showingLocales = 0;
            let hidedLocales = 0;

            localeFields.each( function () {

                $(this).closest('tr').css( 'display' ) !== 'none' ?
                    showingLocales++ :
                    hidedLocales++;

            } );

            if ( action === 'show' ) {

                return showingLocales;

            } else if ( action === 'hide' ) {

                return hidedLocales;

            } else {

                return 0;

            }

        }

        /**
         * Check locale fields and show notices for no=empty locales
         */
        function notEmptyLocaleNotice() {

            const localeFields = $( '.translations-locale' );
            let hideEmpty = false;
            let $targetLocale = false;

            localeFields.each( function() {

                const locale = $( this ).children( 'input' ).val();
                const isVisible = $( this ).parent().parent().css( 'display' ) !== 'none';

                if ( locale === '' && isVisible ) {

                    hideEmpty = true;

                }

                // Find last filled locale
                if ( locale !== '' && isVisible ) {

                    $targetLocale = $( this );

                }


            } );

            if ( !hideEmpty ) {

                addNotice( 'warning', 5000 );

                $targetLocale.addClass( 'mdc-warning-field' );

                // Make filed focused and looks like focused
                $targetLocale.addClass( 'mdc-text-field--focused' );
                $targetLocale.children( 'input' ).focus();

                // Bind remove action to button
                $( '#deblocker-translation-notice button' ).on( 'click', () => {

                    $targetLocale.children( 'input' ).val( '' );
                    $( '#submit' ).click();

                } );

            }

        }

        /**
         *
         * @param design
         * @param timeout
         */
        function addNotice( design= 'info', timeout = 5000 ) {

            // Show only one notice
            if ( $( '#deblocker-translation-notice' ).length > 0 ) { return; }

            $( '#wpbody-content form .wrap' ).append( `
<div id="deblocker-translation-notice" class="mdc-snackbar mdc-${design} mdc-snackbar--open" data-timeout="${timeout}" data-mdc-index="1">
    <div class="mdc-snackbar__surface">
        <div class="mdc-snackbar__label" role="status" aria-live="polite">Clear the Locale field before deleting the translation</div>
        <div class="mdc-snackbar__actions">
            <button class="mdc-button mdc-snackbar__action" type="button" title="Clear locale">Clear locale</button>
        </div>
    </div>   
</div>
            ` );

            setTimeout( () => {

                $( '#deblocker-translation-notice' ).remove();
                $( '.mdc-warning-field' ).removeClass( 'mdc-warning-field' );

            }, timeout + 10 );

        }

        /**
         * Init meta-boxes user interface
         */
        function initUI() {

            /** Translations tab */
            initTranslationButtons();
            initTranslationSwitcher( $( '#mdp_deblocker_translations_settings_translations' ) );
            manageEmptyLocales( 'hide' );
            manageButtons();

            /** Show/Hide fields on switcher check. */
            initSingleSwitch( $( '#mdp_deblocker_general_settings_javascript' ) );
            initSingleSwitch( $( '#mdp_deblocker_behaviour_settings_cross' ) );
            initSingleSwitch( $( '#mdp_deblocker_behaviour_settings_is_loop' ) );
            initSingleSwitch( $( '#mdp_deblocker_behaviour_settings_is_redirect' ) );
            initSingleSwitch( $( '#mdp_deblocker_behaviour_settings_button' ) );
            initSingleSwitch( $( '#mdp_deblocker_style_settings_button_style' ), 4 );

            /** Show/Hide fields related to select */
            initSingleSelect( $( '#mdp_deblocker_general_settings_algorithm' ), [ 'inline', 'proxy' ], 1 );

        }

        initUI();

    } );

} );
