<?php
/**
 * DeBlocker
 * Most effective way to detect ad blockers. Ask the visitors to disable their ad blockers.
 * Exclusively on https://1.envato.market/deblocker
 *
 * @encoding        UTF-8
 * @version         3.3.9
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 * @license         Envato License https://1.envato.market/KYbje
 **/

namespace Merkulove\Deblocker;

use Merkulove\Deblocker\Unity\Settings;

/** Get Plugin Settings. */
$options = Settings::get_instance()->options;

/** Prepare variables. */
$style              = $options['style'];
$timeout            = $options['timeout'];
$cross              = $options['cross'];
$button             = $options['button'];
$button_caption     = esc_html__( $options['button_caption'], 'deblocker' );
$button_style       = $options[ 'button_style' ] ?? '';
$title              = esc_html__( $options['title'], 'deblocker' );
$bg_color           = $options['bg_color'];
$modal_color        = $options['modal_color'];
$text_color         = $options['text_color'];
$blur               = $options['blur'];
$redirect           = $options['is_redirect'] === 'on' ? $options['redirect'] : '';
$prefix             = get_option( 'mdp_deblocker_random_folder_fake_prefix', 'mdpDeblocker' );
$mdpDeblocker       = substr( str_shuffle( 'abcdefghijklmnopqrstuvwxyz' ), 0, mt_rand( 12, 18 ) );
$folder             = get_option( 'mdp_deblocker_random_folder_fake_folder', 'deblocker' );
$loop               = $options[ 'is_loop' ] === 'on' ? $options['loop_timeout'] : '0';
$translations       = $options[ 'translations' ];
$custom_css         = $options[ 'custom_css' ];
$guide              = $options[ 'guide' ];
$adsScriptHandleID  = get_option( 'mdp_deblocker_random_folder_fake_prefix', 'mdp-deblocker-ads' );

/** Prepare content */
if ( $options['content'] !== strip_tags( $options['content'] ) ){

	$content = wp_kses_post( $options['content'] ); // HTML

} else {

	$content = esc_html__( $options['content'], 'deblocker' ); // Plain text

}

/** Prepare multilingual content */

$translations_counter = intval( apply_filters( 'mdp_deblocker_advanced_translations_count', Config::$translations_count ) );
$lang_json = '{';
for ( $i = 1; $i <= $translations_counter; $i++ ) {

	if ( ! empty( $options[ 'translate_locale_' . $i ] ) ) {

		// Add a comma after the previous language
		if ( $i > 1 ) { $lang_json .= ','; }

		$caption = ! empty( $options[ 'translate_button_caption_' . $i ] ) ? $options[ 'translate_button_caption_' . $i ] : $button_caption;

		$lang_json .= '"' . str_replace( '_', '-', strtolower( $options[ 'translate_locale_' . $i ] ) ) . '":';
		$lang_json .= '{';
		$lang_json .= '"title":"' . $options[ 'translate_title_' . $i ] ?? $title;
		$lang_json .= '",';
		$lang_json .= '"content":"' . str_replace( '"', '\\\"', $options[ 'translate_content_' . $i ] ?? $content );
		$lang_json .= '",';
		$lang_json .= '"caption":"' . $caption;
		$lang_json .= '"';
		$lang_json .= '}';

	}
}
$lang_json .= '}';

// language=JavaScript
return "
let {$mdpDeblocker}load = false;
let {$mdpDeblocker}received = false;
const {$mdpDeblocker}on = function() {
    
    if ( {$mdpDeblocker}load ) { return; } // Run only once
    
    let {$mdpDeblocker} = {
        style: '{$style}',
        loop: parseInt( '{$loop}' ),      
		timeout: '{$timeout}',    
		cross: '{$cross}',  
		button: '{$button}',  
		caption: '{$button_caption}',  
		buttonStyle: '{$button_style}',  
		title: `{$title}`,
		content: `{$content}`,
		bg_color: '{$bg_color}',
		modal_color: '{$modal_color}',
		text_color: '{$text_color}', 
		blur: '{$blur}',
		prefix: '{$prefix}',
		folder: '{$folder}',
		redirect: '{$redirect}',
		translations: '{$translations}',
		guide: '{$guide}'
    };
    
    /** Basic localization */
    wpLocalization();
    
    /** Advacned localization */
    advancedTranslation();
    
    /** Detect ad blockers. */
    adsBlocked( callbackAdsBlocked );
    
    /**
    * Callback function
	* @param blocked
	* @param readyState - keep it here for preventing redirects
	*/
    function callbackAdsBlocked( blocked, readyState = 0 ) {
        
        if ( blocked && readyState === 2 ) { //Ublock
            
            doAction();
            
        } else if ( blocked && readyState === 4 ) { // Other
        
            doAction();
            
        } else if ( ! {$mdpDeblocker}received && readyState === 666 ) {
			
			doAction();
			
        } else if ( ! document.getElementById( '{$adsScriptHandleID}-js' ) ) { 
            
            if ( ! document.getElementById( {$mdpDeblocker} ) ) {
                
                doAction();
                
            }
            
        }
        
    }
    
    /** Do some action if ad blockers detected */
    function doAction() {
        
        {$mdpDeblocker}received = true;
        
        let t = {$mdpDeblocker}.timeout ?? 0;
        t = parseInt( t );
        
        if ( t > 0 ) {
            
            setTimeout( () => { chooseAction() }, t );
            
        } else {
            
            chooseAction();
            
        }
        
    }
    
    /** 
    * Chose action between popup opening or redirect link
	*/
    function chooseAction() {
        
        let redURL = {$mdpDeblocker}.redirect;
            
        if ( redURL.length > 1  ) {
            
            window.location.replace( redURL );
            
        } else {
            
            showModal();
            
        }
        
    }
    
    /** Disable text selection on page. */
    function disableTextSelection( e ) {
        
        if ( typeof e.onselectstart !== 'undefined' ){
            e.onselectstart = function(){ return false; };
        } else if ( typeof e.style.MozUserSelect != 'undefined' ) {
            e.style.MozUserSelect = 'none';
        } else if ( typeof e.style.webkitUserSelect != 'undefined' ) {
            e.style.webkitUserSelect = 'none';
        } else {
            e.onmousedown = function(){ return false; };
        }
        
        e.style.cursor = 'default';
    }
    
    /** Enable text selection on page. */
    function enableSelection( e ) {
        
        if ( typeof e.onselectstart != 'undefined' ){
            e.onselectstart = function(){ return true; };
        } else if ( typeof e.style.MozUserSelect != 'undefined' ) {
            e.style.MozUserSelect = 'text';
        } else if ( typeof e.style.webkitUserSelect != 'undefined' ) {
            e.style.webkitUserSelect = 'text';
        } else {
            e.onmousedown = function(){ return true; };
        }
        
        e.style.cursor = 'auto';
    }
    
    /** Disable context menu on page. */
    function disableContextMenu() {
        document.oncontextmenu = function( e ) { 
            let t = e || window.event;
            let n = t.target || t.srcElement;
            
            if ( n.nodeName != 'A' ) {
                return false;  
            }
        };
        
        document.body.oncontextmenu = function () { return false; };
        
        document.ondragstart = function() { return false; };
    }

    /** Enable context menu on page. */
    function enableContextMenu() {
        document.oncontextmenu = null;
        document.body.oncontextmenu = null;
        document.ondragstart = null;
    }
    
    /** Hotkeys handler */
    function hotkeysHandler( e ) {
        
        const keys = [ 65, 66, 67, 70, 73, 80, 83, 85, 86 ];
        if( ( e.ctrlKey || e.metaKey ) && keys.includes( e.keyCode ) ) {
            e.preventDefault();
        }
        
    }
    
    /** Disable HotKeys on page. */
    function disableHotKeys() {
        
        /** For Windows check ctrl. */
        window.addEventListener( 'keydown', hotkeysHandler );
        
        /** Disable DevTools. */
        document.onkeydown = function( e ) {
            if (
                e.keyCode == 123 || // F12
                ( ( e.ctrlKey || e.metaKey ) && e.shiftKey && e.keyCode == 73 ) // CTRL+SHIFT+I, CMD+OPTION+I
            ) {
                e.preventDefault();
            }
        };

    }
    
    
    
    /** Disable Developer Tool on page. */
    function disableDeveloperTools() {
        
        window.addEventListener( 'keydown', function( e ) {

            if (
                e.keyCode === 123 || // F12
                ( ( e.ctrlKey || e.metaKey ) && e.shiftKey && e.keyCode === 73 ) // Ctrl+Shift+I, ⌘+⌥+I
            ) {
                e.preventDefault();
            }
            
        } );
        
        /** Remove body, if you can open dev tools. */
        let checkStatus;

        let element = new Image();
        Object.defineProperty( element, 'id', {
            get:function() {
                checkStatus = 'on';
                throw new Error( 'Dev tools checker' );
            }
        } );

        requestAnimationFrame( function check() {
            checkStatus = 'off';
            if ( 'on' === checkStatus ) {
                document.body.parentNode.removeChild( document.body );
                document.head.parentNode.removeChild( document.head );
            } else {
                requestAnimationFrame( check );
            }
        } );
                
    }
    
    /** Enable HotKeys on page. */
    function enableHotKeys() {
        
        /** For Windows check ctrl. */
        window.removeEventListener( 'keydown', hotkeysHandler );
        
        /** Enable DevTools. */
        document.onkeydown = function( e ) {
            e = e || window.event; 
            if ( e.keyCode == 123 || e.keyCode == 18 || ( e.ctrlKey && e.shiftKey && e.keyCode == 73 ) ) { return true; }
     
        };
    }
    
    /**
     * Adds Front-end CSS.
     **/
    function addStyles() {

        /** Add styles only once */
        if ( document.querySelector( `#${prefix}-css` ) ) { return; }

        /** Create our stylesheet. */
        let style = document.createElement( 'style' );
        style.id = `${prefix}-css`;

        style.innerHTML = `" . Bricks::basic_css( $prefix ) . "`; // phpcs:ignore
        
        /** Add custom CSS */
        style.innerHTML += getCustomCSS();

        /** Get the random script tag. */
        let ref = document.querySelectorAll('script');
        
        /** Brave can't calculta math function, so set 1 only for Brave browseer */
        let rand = ! navigator.brave ? ref[ Math.floor( Math.random() * ref.length ) ] : ref[ ref.length - 1 ];

        /** Insert our new styles before the first script tag. */
        rand.parentNode.insertBefore( style, rand );

    }
    
    /** Make raplces and get custom CSS */
    function getCustomCSS() {
        
        let customCSS = `${custom_css}`;
        let prefix = `.${prefix}`;
        
        customCSS = customCSS.replaceAll( `{{deblocker}}`, prefix );
        customCSS = customCSS.replaceAll( `{{deBlocker}}`, prefix );
        
        return customCSS;
        
    }

    /**
	 * Manage Body Classes.
	 * @param prefix
	 */
    function manageBodyClasses( prefix ) {

        /** Add only one popup */
        if ( document.body.classList.contains( `${prefix}-style-` + {$mdpDeblocker}.style ) ) { return }
        
        /** Set Style class. */
        document.body.classList.add( `${prefix}-style-` + {$mdpDeblocker}.style );

        /** Blur Content: */
        if ( {$mdpDeblocker}.blur === 'on' ) {
            document.body.classList.add( `${prefix}-blur` );
        }

    }

    /**
    * Append Modal.
	* @param prefix
	*/
    function appendModal( prefix ) {

        let overlay = document.createElement( 'div' );
        overlay.classList.add( `${prefix}-blackout` );
        overlay.style.backgroundColor = {$mdpDeblocker}.bg_color; // Set Overlay Color.
        overlay.classList.add( 'active' );
        document.body.appendChild( overlay );

        /** Create the Modal Wrapper. */
        let modalWrapper = document.createElement( 'div' );
        modalWrapper.classList.add( `${prefix}-wrapper` );
        document.body.appendChild( modalWrapper );

        /** Create Modal. */
        let modal = document.createElement( 'div' );
        modal.classList.add( `${prefix}-modal` );
        modal.classList.add( `${prefix}-the-modal` );
        modal.style.backgroundColor = {$mdpDeblocker}.modal_color; // Set Modal Color.
        modal.classList.add( 'active' );
        modalWrapper.appendChild(modal);

        /** Cross button */
        rednerCloseCross( modal );

        /** Create Title. */
        let title = document.createElement( 'h4' );
        title.innerHTML = {$mdpDeblocker}.title;
        title.style.color = {$mdpDeblocker}.text_color; // Set Text Color.
        modal.appendChild( title );

        /** Create Content. */
        let content = document.createElement( 'div' );
        content.classList.add( `${prefix}-content` );
        content.innerHTML = {$mdpDeblocker}.content;
        content.style.color = {$mdpDeblocker}.text_color; // Set Text Color.
        modal.appendChild( content );
        
        /** Just a button */
        rednerCloseButton( modal );
        
        /** Add a guide */
        if ( typeof renderGuide !== 'undefined' ) {
            renderGuide( modalWrapper, modal );
        }

    }
    
    /**
    * Insert modal HTML. 
	*/
    function insertModal() {
        
        document.body.innerHTML += `" . Bricks::modalMarkup( $options ) . "`;
        
        if ( document.querySelector( `.${prefix}-close` ) ) {
            document.querySelector( `.${prefix}-close` ).addEventListener( 'click', closeClickHandler );
        }
        
        if ( document.querySelector( `.${prefix}-close-button` ) ) {
            document.querySelector( `.${prefix}-close-button` ).addEventListener( 'click', closeClickHandler );
        }
        
        document.querySelectorAll( `.{$prefix}-guide-triger` ).forEach( function( item ) {
			item.addEventListener( 'click', guideTriggerHadler );
		} );
        
    }

    /** Show DeBlocker Modal. */
    function showModal() {
        
        if ( {$mdpDeblocker}load ) { return; }

        let prefix = {$mdpDeblocker}.prefix;
        
        /** Create body overlay. */
        if ( document.querySelector( `${prefix}-modal` ) ) { return; }

        // Add body classes.
        manageBodyClasses( prefix );

        // Add styles.
        try {

            /** Adds Front-end CSS. */
        	addStyles();

        } catch ( e ) {

            const styleTag = document.querySelector( 'style' );
            if ( ! styleTag ) {
                console.warn( 'Imposible get style tags. Please change blocking alghrothm.' );
				return;
            }

            styleTag.innerHTML += `" . Bricks::basic_css( $prefix ) . "`;

		}

        // Add modal.
        try {

            // Append Modal.
            appendModal( prefix );
            {$mdpDeblocker}load = true;
            
        } catch ( e ) {

            console.warn( 'Imposible append modal. Please change blocking alghrothm.' );
            insertModal( prefix );
            {$mdpDeblocker}load = true;

		}

        disableTextSelection( document.body );
        disableContextMenu();
        disableHotKeys();
        disableDeveloperTools();
        
        {$mdpDeblocker}load = true;

    }
    
    /**
    * Render X close button
	* @param modal
	*/
    function rednerCloseCross( modal ) {
        
        if ( {$mdpDeblocker}.cross === 'on' ) {
                
            let close = document.createElement( 'span' );
            close.classList.add( `${prefix}-close` );
            close.innerHTML = '&nbsp;';
            close.setAttribute( 'href', '#' );

            /** Close Event. */
            close.addEventListener( 'click', closeClickHandler );
            
            /** Append Close Button */
            modal.appendChild(close);
            
        }
        
    }
    
    /**
    * Render text close button
	* @param modal
	*/
    function rednerCloseButton( modal ) {
        
        if ( {$mdpDeblocker}.button === 'on' ) {
                
            let close = document.createElement( 'button' );
            close.classList.add( `${prefix}-close-button` );
            
            // Add default WP button classes
            if ( {$mdpDeblocker}.buttonStyle !== 'on' ) {
	            close.classList.add( `wp-block-button__link` );
	            close.classList.add( `elementor-button-link` );
            }
            
            close.innerHTML = {$mdpDeblocker}.caption;

            /** Close Event. */
            close.addEventListener( 'click', closeClickHandler );
            
            /** Append Close Button */
            modal.appendChild( close );
            
        }
        
    }
    
    /**
    * Close click handeler
	* @param e
	*/
    function closeClickHandler( e ) {
        
        e.preventDefault();
        
        let elem = document.querySelector( `.${prefix}-modal` );
        elem.parentNode.removeChild(elem);
        elem = document.querySelector( `.${prefix}-wrapper` );
        elem.parentNode.removeChild(elem);
        elem = document.querySelector( `.${prefix}-blackout` );
        elem.parentNode.removeChild(elem);

        /** Remove Blur. */
        document.body.classList.remove( `${prefix}-blur` );
        enableSelection( document.body );
        enableContextMenu();
        enableHotKeys();
        
        /** Remove body class */
        const bodyClass = `${prefix}-style-` + {$mdpDeblocker}.style;
        if ( document.body.classList.contains( bodyClass ) ) {
            document.body.classList.remove( bodyClass );
        }
        
        /** Rund loop */
        runLoop();
        
    }
    
	" .  Bricks::guide_js( $prefix, $mdpDeblocker ) . "
    
    /**
     * Run popup loop
     **/
    function runLoop() {
        
        if ( {$mdpDeblocker}.loop > 0 ) {
            
            let loopTimeout = setTimeout( () => {
                
                {$mdpDeblocker}load = false;
                {$mdpDeblocker}.timeout = 0;

                /** Detect ad blockers. */
                adsBlocked( callbackAdsBlocked );
                
            }, {$mdpDeblocker}.loop * 1000 );
            
        }

    }

    /**
     * Detect Fair AdBlocker extension.
     **/
    function isFairAdBlocker() {

        let stndzStyle = document.getElementById('stndz-style');

        return null !== stndzStyle;

    }

    /** Detect ad blockers. */
    function adsBlocked( callback ) {

        let adsSrc = 'https://googleads.g.doubleclick.net/pagead/id';

        /** Check Fair AdBlocker. */
        if ( isFairAdBlocker() ) {

            callback( true ); // Blocked!

        } else {

            adsSrc = 'https://pagead2.googlesyndication.com/pagead/js/adsbygoogle.js';

            let head = document.getElementsByTagName('head')[0];

            let script = '';
            try {
                script = document.createElement('script');
                script.setAttribute( 'src', adsSrc );
	            script.setAttribute( 'type', 'text/javascript' ); // 'text/javascript' 'application/json'
	            script.setAttribute( 'charset', 'utf-8' );
			} catch (e) {
                console.warn( 'Blocked creation a new script element!' );
                callback( true ); // Blocked creation a new script element!
            }

            let done = false;
            script.onload = script.onreadstatechange = function() {

                if ( ! done && ( ! this.readyState || this.readyState === 'loaded' || this.readyState === 'complete') ) {

                    done = true;
                    script.onload = script.onreadystatechange = null;

                    if ( 'undefined' === typeof window.adsbygoogle ) {
                        callback( true ); // Blocked!
                    } else {
                        {$mdpDeblocker}received = true;
                        callback( false ); // All fine.
                    }

                    script.parentNode.removeChild( script );

                }

            };

            /** On Error. */
            script.onerror = function() {
                callback( true ); // Blocked request send!
            };

            /** Just wait a bit ;) if no answere - detect blocking */
            if ( ! {$mdpDeblocker}received ) {
                const fishMemory = Math.floor( Math.random() * 4 + 2 ); // 2-5sec
	            setTimeout( () => {
					if ( ! {$mdpDeblocker}received ) {
	                  callback( true, 666 ); // Blocked!
					}
				}, fishMemory * 1000 );
            }

            " . Bricks::async2() . "

           	" . Bricks::async1() . "

			if ( script ) {
                head.insertBefore( script, head.firstChild );
			}
            
            " . Bricks::async3() . "

        }

    }
    
    /** 
    * Advacnec translation 
	*/
    function advancedTranslation() {
        
        if ( {$mdpDeblocker}.translations !== 'on' ) { return; }
        if ( ! document.documentElement.lang ) { return; }
        
        let lang = JSON.parse( '{$lang_json}' );
        let locale = false;
        const localeCode = document.documentElement.lang.toLowerCase();
        const localeLang = localeCode.split( '-' )[ 0 ];
        
        // Find tranaltion by language code or just a language
        if ( lang[ localeCode ] ) {
            
            locale = localeCode;

        } else if ( lang[ localeLang ] ) {

            locale = localeLang;

        }

        // Make translation
        if ( locale ) {

            localizeSingleString( lang, locale, 'title' );
            localizeSingleString( lang, locale, 'content' );

            if ( {$mdpDeblocker}.button === 'on' ) {
            	localizeSingleString( lang, locale, 'caption' );
            }

	    }

    }

    /**
    * Localize throth WP
	*/
    function wpLocalization() {

        if ( window[{$mdpDeblocker}.folder] ) {

            const locale = window[{$mdpDeblocker}.folder];

            for ( const [ key, localizedString ] of Object.entries( locale ) ) {
                
				{$mdpDeblocker}[ key ] = localizedString;  
                
			}
            
        }
        
    }
    
    /**
    * Localize single stting by key
	* @param lang
	* @param locale
	* @param key
	*/
    function localizeSingleString( lang, locale, key ) {
        
        if ( lang[ locale ][ key ].length !== '' ) {
            {$mdpDeblocker}[ key ] = lang[ locale ][ key ];
        }
        
    }
    
}

document.readyState === 'loading' ?
        document.addEventListener( 'DOMContentLoaded', () => { {$mdpDeblocker}on(); } ) :
        {$mdpDeblocker}on();

setTimeout( () => { {$mdpDeblocker}on() }, 1000 );
";
