<?php
/**
 * Creates analytics for fast
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Adds extra endpoints for api
 */
class FastAnalytics
{
    /**
     * I am a singleton
     *
     * @var object
     */
    private static $instance;

    /**
     * Constructor for the api.
     */
    public function __construct()
    {
        add_action('save_post_fast_ticket', array(&$this, 'log_on_ticket_creation'), 99, 3);
        add_action('set_object_terms', array(&$this, 'log_on_taxonomy_change'), 99, 6);
        add_action('fast_action_post_comment', array(&$this, 'log_on_ticket_comment'), 99, 2);
    }

    /**
     * Creates a log event on ticket creation.
     *
     * @param int  $post_id Post id being saved.
     * @param post $post The post object.
     * @param bool $update Whether this is an existing post being updated or not.
     */
    public function log_on_ticket_creation($post_id, $post, $update)
    {
        if ((defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) || wp_is_post_revision($post_id)) {
            return;
        }

        if (true === $update) {
            return;
        }

        $log_id = fast_get_daily_log_id(getdate());
        if (is_wp_error($log_id)) {
            return;
        }

        do_action('fast_filter_analytics_new_ticket', $post, $log_id);
    }

    /**
     * Creates a log event upon taxonomy change.
     *
     * @param int    $object_id  Object ID.
     * @param array  $terms      An array of object terms.
     * @param array  $tt_ids     An array of term taxonomy IDs.
     * @param string $taxonomy   Taxonomy slug.
     * @param bool   $append     Whether to append new terms to the old terms.
     * @param array  $old_tt_ids Old array of term taxonomy IDs.
     */
    public function log_on_taxonomy_change($object_id, $terms, $tt_ids, $taxonomy, $append, $old_tt_ids)
    {
        if ('fast_ticket' !== get_post_type($object_id)) {
            return;
        }

        $log_id = fast_get_daily_log_id(getdate());
        if (is_wp_error($log_id)) {
            return;
        }

        if (empty($terms)) {
            return;
        }

        $field = is_numeric($terms[0]) ? 'id' : 'slug';
        $term = get_term_by($field, $terms[0], $taxonomy);
        if (false === $term) {
            return;
        }

        do_action('fast_filter_analytics_taxonomy_change', $object_id, $taxonomy, $term, $log_id);
    }

    /**
     * Creates a log event upon ticket comment.
     *
     * @param WP_Comment $comment The comment object.
     */
    public function log_on_ticket_comment($comment)
    {
        $log_id = fast_get_daily_log_id(getdate());
        if (is_wp_error($log_id)) {
            return;
        }
        do_action('fast_filter_analytics_ticket_comment', $comment, $log_id);
    }

    /**
     * Instance accessor. If instance doesn't exist, we'll initialize the class.
     *
     * @return object FastAnalytics::$instance
     */
    public static function instance()
    {
        if (!isset(self::$instance)) {
            self::$instance = new FastAnalytics;
        }
        return self::$instance;
    }

    /**
     * Alias for ::instance
     * For semantics. init can be called when the intention is the first initialization
     *
     * @return object FastAnalytics::$instance
     */
    public static function init()
    {
         return self::instance();
    }
}
