<?php
/**
 * Creates extra api endpoints for the plugin.
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Adds extra endpoints for api
 */
class FastAPI extends WP_REST_Controller
{
    /**
     * Constructor for the api.
     */
    public function __construct()
    {
        $this->namespace     = '/fast-plugin/v1';
    }

    /**
     * Registers routes with WordPress.
     */
    public function register_routes()
    {
        register_rest_route($this->namespace, '/is_logged_in', array(
            array(
                'methods'  => 'GET',
                'callback' => array($this, 'user_logged_in_permission_check'),
                'args'     => array()
            ),
        ));

        register_rest_route($this->namespace, '/envato_login', array(
            array(
                'methods'   => 'POST',
                'callback'  => array($this, 'route_envato_login'),
                'args' => array(
                    'code' => array(
                        'required' => true
                    ),
                    'nonce' => array(
                        'required' => true
                    )
                )
            ),
        ));

        register_rest_route($this->namespace, '/register', array(
            array(
                'methods'   => 'POST',
                'callback'  => array($this, 'route_register'),
                'args' => array(
                    'first_name' => array(
                        'required' => true
                    ),
                    'last_name' => array(
                        'required' => true
                    ),
                    'email' => array(
                        'required' => true
                    ),
                    'password' => array(
                        'required' => true
                    ),
                    'captcha' => array(
                        'required' => false
                    ),
                    'nonce' => array(
                        'required' => true
                    ),
                    'language' => array(
                        'required' => false
                    )
                )
            )
        ));

        register_rest_route($this->namespace, '/agent_register', array(
            array(
                'methods'   => 'POST',
                'callback'  => array($this, 'route_agent_register'),
                'permission_callback' => array($this, 'user_logged_in_permission_check'),
                'args' => array(
                    'first_name' => array(
                        'required' => true
                    ),
                    'last_name' => array(
                        'required' => true
                    ),
                    'email' => array(
                        'required' => true
                    ),
                    'password' => array(
                        'required' => true
                    )
                )
            )
        ));


        register_rest_route($this->namespace, '/login', array(
            array(
                'methods'   => 'POST',
                'callback'  => array($this, 'route_login'),
                'args' => array(
                    'email' => array(
                        'required' => true
                    ),
                    'password' => array(
                        'required' => true
                    ),
                    'remember' => array(
                        'required' => false
                    ),
                    'nonce' => array(
                        'required' => true
                    )
                )
            )
        ));

        register_rest_route($this->namespace, '/current_user', array(
            array(
                'methods'             => 'GET',
                'callback'            => array($this, 'route_current_user'),
                'permission_callback' => array($this, 'user_logged_in_permission_check'),
                'args'                => array()
            ),
        ));

        register_rest_route($this->namespace, '/logout', array(
            array(
                'methods'             => 'GET',
                'permission_callback' => array($this, 'user_logged_in_permission_check'),
                'callback'            => array($this, 'route_logout')
            )
        ));

        register_rest_route($this->namespace, '/extra_fields', array(
            array(
                'methods'             => 'GET,POST',
                'callback'            => array($this, 'route_extra_fields'),
                'permission_callback' => array($this, 'user_logged_in_permission_check'),
            )
        ));

        register_rest_route($this->namespace, '/ticket-taxonomy/(?P<id>\d+)', array(
            array(
                'methods'             => 'POST',
                'callback'            => array($this, 'route_ticket_taxonomy'),
                'permission_callback' => array($this, 'user_logged_in_permission_check'),
                'args' => array(
                    'id' => array(
                        'validate_callback' => function ($param, $request, $key) {
                            return is_numeric($param);
                        }
                    ),
                )
            )
        ));

        register_rest_route($this->namespace, '/ticket-mark-as-read/(?P<id>\d+)', array(
            array(
                'methods'             => 'POST',
                'callback'            => array($this, 'route_ticket_as_read'),
                'permission_callback' => array($this, 'user_logged_in_permission_check'),
                'args' => array(
                    'id' => array(
                        'validate_callback' => function ($param, $request, $key) {
                            return is_numeric($param);
                        }
                    ),
                )
            )
        ));

        register_rest_route($this->namespace, '/envato_user_suggestions', array(
            array(
                'methods'             => 'GET',
                'callback'            => array($this, 'route_envato_user_suggestions'),
                'permission_callback' => array($this, 'user_logged_in_and_agent_permission_check'),
                'args' => array(
                    'searchUser' => array(
                        'validate_callback' => function ($param, $request, $key) {
                            return is_string($param);
                        }
                    ),
                )
            )
        ));

        register_rest_route($this->namespace, '/user_suggestions', array(
            array(
                'methods'             => 'GET',
                'callback'            => array($this, 'route_user_suggestions'),
                'permission_callback' => array($this, 'user_logged_in_and_agent_permission_check'),
                'args' => array(
                    'search' => array(
                        'validate_callback' => function ($param, $request, $key) {
                            return is_string($param);
                        }
                    ),
                )
            )
        ));

        if (fast_get_option('analytics_enabled') === 'on') {
            register_rest_route($this->namespace, '/analytics', array(
                array(
                    'methods'             => 'GET',
                    'callback'            => array($this, 'route_fetch_analytics'),
                    'args' => array(
                        'from' => array(
                            'required' => false
                        ),
                        'to' => array(
                            'required' => false
                        ),
                        'span' => array(
                            'required' => true
                        ),
                        'chart' => array(
                            'required' => false
                        ),
                        'series' => array(
                            'required' => false
                        )
                    )
                )
            ));
        }

        if (fast_get_option('customer_search_enabled') === 'on') {
            register_rest_route($this->namespace, '/public-search', array(
                array(
                    'methods' => 'GET',
                    'callback' => array($this, 'route_public_search'),
                    'permission_callback' => array($this, 'user_logged_in_permission_check'),
                    'args' => array(
                        'search' => array(
                            'required' => false
                        ),
                        'ticket_product' => array(
                            'required' => false
                        ),
                        'order' => array(
                            'required' => true
                        ),
                        'orderby' => array(
                            'required' => false
                        ),
                        'page' => array(
                            'required' => false
                        ),
                        'per_page' => array(
                            'required' => false
                        )
                    )
                )
            ));

            register_rest_route($this->namespace, '/public-ticket/(?P<id>\d+)', array(
                array(
                    'methods'             => 'GET',
                    'callback'            => array($this, 'route_public_ticket'),
                    'permission_callback' => array($this, 'user_logged_in_permission_check'),
                    'args' => array(
                        'id' => array(
                            'validate_callback' => function ($param, $request, $key) {
                                return is_numeric($param);
                            }
                        ),
                    )
                )
            ));
        } // End if().
    }

    /**
     * Used for permission_callback on routes
     *
     * @return boolean If user is logged in.
     */
    public function user_logged_in_permission_check()
    {
        return is_user_logged_in();
    }

    /**
     * Checks if a user is logged in now.
     *
     * @param object $request WP_REST_Request object.
     */
    public function user_logged_in_and_agent_permission_check($request)
    {
        $user_id = get_current_user_id();
        return is_user_logged_in() && fast_is_user_agent($user_id);
    }

    /**
     * Returns current user
     *
     * @param object $request WP_REST_Request object.
     */
    public function route_current_user($request)
    {
        return $this->get_user_data();
    }

    /**
     * Logs into envato api using code provided by login page param.
     *
     * @param object $request WP_REST_Request object.
     */
    public function route_envato_login($request)
    {
        if (false === wp_verify_nonce($request->get_param('nonce'), 'ticket-api-action')) {
            return new WP_Error('cant-login', esc_html__('Invalid Nonce', 'fast-plugin'), array(
                'status' => 401
            ));
        }

        // Get envato token.
        $token_response = wp_remote_post('https://api.envato.com/token/', array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-type' => 'application/x-www-form-urlencoded'
            ),
            'body' => array(
                'grant_type'    => 'authorization_code',
                'code'          => $request->get_param('code'),
                'client_id'     => fast_get_option('envato_client_id'),
                'client_secret' => fast_get_option('envato_client_secret')
            )
        ));

        if (is_wp_error($token_response)) {
            return $token_response;
        }

        // Decode the token.
        $token = json_decode(wp_unslash(wp_remote_retrieve_body($token_response)));

        // Check for error responses.
        if (200 !== wp_remote_retrieve_response_code($token_response)) {
            return new WP_Error($token->error, $token->error_description);
        }

        // Use token to get the user details.
        $user_details = $this->envato_get_user($token->access_token);

        if (is_wp_error($user_details)) {
            return $user_details;
        }

        // If user exists log in, otherwise register user.
        $user = get_user_by('email', $user_details['email']);
        if (false === $user) {
            // No user so we need to register one.
            $user = fast_register_user(
                $user_details['account']['firstname'],
                $user_details['account']['firstname'],
                $user_details['account']['surname'],
                $user_details['email'],
                wp_generate_password()
            );

            if (is_wp_error($user)) {
                return $user;
            }
        }

        $purchases = $this->envato_get($token->access_token, 'https://api.envato.com/v3/market/buyer/purchases');

        // Check for errors.
        if (is_wp_error($purchases)) {
            return $purchases;
        }

        // Create user purchases.
        if (isset($purchases['purchases']) && is_array($purchases['purchases'])) {
            // Allow empty content & title for posts.
            add_filter('wp_insert_post_empty_content', '__return_false');

            foreach ($purchases['purchases'] as $purchase) {
                $this->add_envato_purchase($purchase, $user, $user_details);
            }
        }

        return $this->login_user($user);
    }

    /**
     * Adds a purchase to the users purchase list.
     *
     * @param array $envato_purchase Purchase data to add.
     * @param array $wp_user WordPress user.
     * @param array $envato_user  Envato user data.
     *
     * @return array New list of purchases
     */
    private function add_envato_purchase($envato_purchase, $wp_user, $envato_user)
    {
        $purchase = fast_get_user_envato_purchase($envato_purchase['code'], $wp_user->ID);
        if (false === $purchase) {
            // This purchase is new.
            $purchase = array(
                'post_status' => 'private',
                'post_author' => $wp_user->ID,
                'post_type'   => 'fast_purchase'
            );
        } else {
            // Convert the WP_Post to array.
            $purchase = $purchase->to_array();
        }

        // Create purchase meta data.
        $purchase['meta_input'] = array(
            '_buyer'           => $envato_user['username'],
            '_created_at'      => $envato_purchase['sold_at'],
            '_item_id'         => $envato_purchase['item']['id'],
            '_item_name'       => $envato_purchase['item']['name'],
            '_licence'         => $envato_purchase['license'],
            '_purchase_code'   => $envato_purchase['code'],
            '_supported_until' => $envato_purchase['supported_until'],
            '_last_refreshed'  => time()
        );

        // Now look up the product taxonomy for this product so we can link it.
        $products = get_terms(array(
            'hide_empty' => false,
            'taxonomy'   => 'fast_product',
            'meta_key'   => '_fast-plugin-product_envato_item',
            'meta_value' => $envato_purchase['item']['id'],
            'fields'     => 'ids'
        ));

        if (count($products) > 0) {
            $purchase['meta_input']['_ticket_product'] = (int) $products[0];
        } else {
            // Product not supported so return.
            return;
        }

        return wp_insert_post($purchase, true);
    }

    /**
     * Fetches user info from Envato API.
     *
     * @param string $token Bearer token.
     *
     * @return array User information.
     */
    public function envato_get_user($token)
    {
        // Get account details.
        $account = $this->envato_get($token, 'https://api.envato.com/v1/market/private/user/account.json');

        if (is_wp_error($account)) {
            return $account;
        }

        $email = $this->envato_get($token, 'https://api.envato.com/v1/market/private/user/email.json');

        if (is_wp_error($email)) {
            return $email;
        }

        $username = $this->envato_get($token, 'https://api.envato.com/v1/market/private/user/username.json');

        if (is_wp_error($username)) {
            return $username;
        }

        return array_merge($account, $email, $username);
    }

    /**
     * Fetches info from Envato API.
     *
     * @param string $token Bearer token.
     * @param string $url to call.
     *
     * @return object Response from get or WP_Error.
     */
    private function envato_get($token, $url)
    {
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'headers' => array(
                'Authorization' => 'Bearer ' . $token
            )
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $value = json_decode(wp_remote_retrieve_body($response), true);

        // Check for error responses.
        if (200 !== wp_remote_retrieve_response_code($response)) {
            return new WP_Error($value->error, $value->error_description);
        }

        return $value;
    }

    /**
     * Logs a user in to WordPress
     *
     * @param object $request WP_REST_Request object.
     */
    public function route_login($request)
    {   
        $referer = strtok($request->get_header('Referer'), '?');
        if (trailingslashit(get_the_permalink(fast_get_option('support_page'))) !== $referer &&
            trailingslashit(get_the_permalink(fast_get_option('support_page'))) . 'login' !== $referer) {
            return new WP_Error('cant-login', $referer, array(
                'status' => 401
            ));
        }

        if (false === wp_verify_nonce($request->get_param('nonce'), 'ticket-api-action')) {
            return new WP_Error('cant-login', esc_html__('Invalid Nonce', 'fast-plugin'), array(
                'status' => 401
            ));
        }
        $remember = $request->get_param('remember');
        $user = wp_authenticate($request->get_param('email'), $request->get_param('password'));
        return $this->login_user($user, $remember);
    }

    /**
     * Logs a user into WordPress.
     *
     * @param object $user User object.
     *
     * @return object|array User data in array or WP_Error object.
     */
    private function login_user($user, $remember = false)
    {   
        if (is_wp_error($user)) {
            return new WP_Error('cant-login', esc_html__('Invalid username or password', 'fast-plugin'), array(
                'status' => 401
            ));
        }

        if (!in_array('administrator', (array) $user->roles, true) &&
            !in_array('fast_manager', (array) $user->roles, true) &&
            !in_array('fast_agent', (array) $user->roles, true) &&
            !in_array('fast_customer', (array) $user->roles, true)) {
            return new WP_Error('cant-login', esc_html__('User does not possess the necessary role', 'fast-plugin'), array(
                'status' => 401
            ));
        }

        // Login user.
        wp_set_current_user($user->ID, $user->email);
        wp_set_auth_cookie($user->ID, $remember);
        do_action('wp_login', $user->ID, $user);

        $cookie = wp_parse_auth_cookie('', 'logged_in');

        // We have a new nonce now cookies have changed make sure we return it in the headers.
        global $wp_rest_server;
        $wp_rest_server->send_header('X-WP-Nonce', wp_create_nonce('wp_rest'));

        // return user data.
        return $this->get_user_data();
    }

    /**
     * Registers a user in to WordPress
     *
     * @param object $request WP_REST_Request object.
     */
    public function route_register($request)
    {
        if ($request->get_header('referer') !== trailingslashit(get_the_permalink(fast_get_option('support_page'))) . 'register') {
            return new WP_Error('cant-register', esc_html__('Must register in the app', 'fast-plugin'), array(
                'status' => 401
            ));
        }

        if (false === wp_verify_nonce($request->get_param('nonce'), 'ticket-api-action')) {
            return new WP_Error('cant-register', esc_html__('Invalid Nonce', 'fast-plugin'), array(
                'status' => 401
            ));
        }

        $recaptcha_site_key = fast_get_option('recaptcha_site_key');
        $recaptcha_secret_key = fast_get_option('recaptcha_secret_key');

        if (!empty($recaptcha_site_key) && !empty($recaptcha_secret_key)) {
            $response = wp_remote_post('https://www.google.com/recaptcha/api/siteverify', array(
                'method' => 'POST',
                'blocking' => true,
                'timeout' => 60,
                'body' => array(
                    'secret' => $recaptcha_secret_key,
                    'response' => $request->get_param('captcha')
                )
            ));

            if (is_wp_error($response)) {
                return $response;
            }

            $response = json_decode($response['body'], true);
            if (true !== $response['success']) {
                return new WP_Error('invalid-captcha', $response['error-codes'], array(
                    'status' => 401
                ));
            }
        }

        $language = $request->get_param('language');

        $user = fast_register_user(
            $request->get_param('first_name'),
            $request->get_param('first_name'),
            $request->get_param('last_name'),
            $request->get_param('email'),
            $request->get_param('password'),
            null === $language ? '' : $language
        );

        if (is_wp_error($user)) {
            return $user;
        } else {
            // Reset purchases array.
            update_user_meta($user->ID, '_purchases', array());

            if ('on' === fast_get_option('register_agreement_enabled')) {
                $agreement_checked = $request->get_param('agreement_checked');
                update_user_meta($user->ID, '_agreement_checked', $agreement_checked);
            }

            // Send email if option is set.
            if ('on' === fast_get_option('email_new_user_email_registration')) {
                fast_send_registration_email($request->get_param('email'), $request->get_param('password'));
            }

            // Return the new user.
            return $this->login_user($user);
        }
    }

    /**
     * Registers a user in to WordPress
     *
     * @param object $request WP_REST_Request object.
     */
    public function route_agent_register($request)
    {
        $referer = $request->get_header('referer');
        if (trailingslashit(get_the_permalink(fast_get_option('support_page'))) . 'register' !== $referer &&
            trailingslashit(get_the_permalink(fast_get_option('support_page'))) . 'ticket/new' !== $referer) {
            return new WP_Error('cant-register', esc_html__('Must register in the app', 'fast-plugin'), array(
                'status' => 401
            ));
        }

        $language = $request->get_param('language');

        $user = fast_register_user(
            $request->get_param('first_name'),
            $request->get_param('first_name'),
            $request->get_param('last_name'),
            $request->get_param('email'),
            $request->get_param('password'),
            null === $language ? '' : $language
        );

        if (is_wp_error($user)) {
            return $user;
        } else {
            // Reset purchases array.
            update_user_meta($user->ID, '_purchases', array());

            if ('on' === fast_get_option('register_agreement_enabled')) {
                $agreement_checked = $request->get_param('agreement_checked');
                update_user_meta($user->ID, '_agreement_checked', $agreement_checked);
            }

            // Send email if option is set.
            if ('on' === fast_get_option('email_new_user_email_registration')) {
                fast_send_registration_email($request->get_param('email'), $request->get_param('password'));
            }

            // Return the new user.
            return $this->get_user_data($user->ID);
        }
    }


    /**
     * Logs a user out of WordPress
     *
     * @param object $request WP_REST_Request object.
     */
    public function route_logout($request)
    {
        if (!is_user_logged_in()) {
            return new WP_Error('cant-logout', esc_html__('User not logged in', 'fast-plugin'), array(
                'status' => 404
            ));
        }

        // Logout user.
        wp_logout();

        $cookie = wp_parse_auth_cookie('', 'logged_in');

        // We have a new nonce now cookies have changed make sure we return it in the headers.
        global $wp_rest_server;
        $wp_rest_server->send_header('X-WP-Nonce', wp_create_nonce('wp_rest'));

        return array(
            'message' => 'Successfully logged out',
        );
    }

    /**
     * Gets and sets the extra fields option.
     *
     * @param object $request HTTP Request.
     */
    public function route_extra_fields($request)
    {
        switch ($request->get_method()) {
            case 'GET':
                // do nothing but return the fields.
                break;
            case 'POST':
                $body = $request->get_body();
                $fields = json_decode($body);
                update_option('fast_extra_fields', (object) array(
                    'fields' => $fields->extra_fields,
                ));
                update_option('fast_secure_fields', (object) array(
                    'fields' => $fields->secure_fields
                ));
                break;
        }

        $extra_fields = fast_get_extra_fields();
        $secure_fields = fast_get_secure_fields();
        return array(
            'extra_fields' => apply_filters('fast_filter_get_extra_fields', $extra_fields->fields),
            'secure_fields' => apply_filters('fast_filter_get_secure_fields', $secure_fields->fields)
        );
    }

    /**
     * Gets and sets the extra fields option.
     *
     * @param object $request HTTP Request.
     */
    public function route_ticket_taxonomy($request)
    {
        $id = $request->get_param('id');
        $body = $request->get_json_params();
        foreach ($body as $taxonomy => $term) {
            if (taxonomy_exists($taxonomy)) {
                $permission = str_replace('fast_', 'assign_', $taxonomy);
                if (current_user_can($permission, $id)) {
                    wp_set_object_terms($id, $term, $taxonomy, false);
                } else {
                    return new WP_Error('fast_no_permission_taxonomy_change', esc_html__('Sorry, you are not allowed to update this taxonomy.'), array(
                        'status' => 401
                    ));
                }
            }
        }
        $ticket = get_post($id);
        $rest_post = new WP_REST_Posts_Controller('fast_ticket');
        return $rest_post->prepare_item_for_response($ticket, $request);
    }

    /**
     * Gets and sets the extra fields option.
     *
     * @param object $request HTTP Request.
     */
    public function route_ticket_as_read($request)
    {
        $id = $request->get_param('id');
        if (!current_user_can('edit_ticket', $id)) {
            return new WP_Error('fast_no_permission_mark_as_read_change', esc_html__('Sorry, you are not allowed to update this ticket.'), array(
                'status' => 401
            ));
        }

        $agent_replied = $request->get_param('agent_replied');
        update_post_meta($id, '_agent_replied', $agent_replied);
        $ticket = get_post($id);
        $rest_post = new WP_REST_Posts_Controller('fast_ticket');
        return $rest_post->prepare_item_for_response($ticket, $request);
    }

    /**
     * Returns user suggestions for autocomplete in search form.
     *
     * @param object $request WP_REST_Request object.
     */
    public function route_envato_user_suggestions($request)
    {
        $purchases = get_posts(array(
            'posts_per_page' => -1,
            'post_type'      => 'fast_purchase',
            'post_status'    => 'private',
            'meta_query'     => array(
                array(
                    'key'     => '_buyer',
                    'value'   => $request->get_param('searchUser'),
                    'compare' => 'LIKE'
                )
            )
        ));

        $users = array();

        foreach ($purchases as $purchase) {
            $buyer = get_post_meta($purchase->ID, '_buyer', true);
            $users[] = array(
                'username' => $buyer,
                'envato_purchase_id' => $purchase->ID
            );
        }
        return $users;
    }

    /**
     * Returns user suggestions for autocomplete in new ticket form.
     *
     * @param object $request WP_REST_Request object.
     */
    public function route_user_suggestions($request)
    {
        $user_query = new WP_User_Query(
            array(
                'role__in' => array(
                    'fast_customer',
                    'fast_agent',
                    'fast_manager',
                    'administrator'
                ),
                'search' => '*' . $request->get_param('search') . '*',
            )
        );

        $managers = array();
        $agents = array();
        $customers = array();

        $results = $user_query->get_results();
        foreach ($results as $user) {
            $id = $user->get('ID');
            $user_data = array(
                'id' => $id,
                'firstName' => $user->get('first_name'),
                'lastName' => $user->get('last_name'),
                'email' => $user->get('user_email'),
                'username' => $user->get('user_login'),
                'avatar' => get_avatar_url($id)
            );
            if (in_array('administrator', (array) $user->roles)) {
                $managers[] = $user_data;
            } else if (in_array('fast_manager', (array) $user->roles)) {
                $managers[] = $user_data;
            } else if (in_array('fast_agent', (array) $user->roles)) {
                $agents[] = $user_data;
            } else if (in_array('fast_customer', (array) $user->roles)) {
                $customers[] = $user_data;
            } else {
                $customers[] = $user_data;
            }
        }
        return array(
            'count' => count($results),
            'results' => array(
                array(
                    'name' => esc_html__('Managers', 'fast-plugin'),
                    'users' => $managers
                ),
                array(
                    'name' => esc_html__('Agents', 'fast-plugin'),
                    'users' => $agents
                ),
                array(
                    'name' => esc_html__('Customers', 'fast-plugin'),
                    'users' => $customers
                )
            )
        );
    }


    /**
     * Returns Analytics data.
     *
     * @param object $request WP_REST_Request object.
     */
    public function route_fetch_analytics($request)
    {
        $date_format = DateTime::ISO8601;

        $log_query = array(
            'post_type'      => 'fast_log',
            'orderby'        => 'date',
            'order'          => 'ASC',
            'posts_per_page' => -1,
            'date_query'     => array()
        );

        // Create from and to date if we have them.
        $from_date = $request->get_param('from');
        $to_date = $request->get_param('to');
        $span_param = $request->get_param('span');
        $chart_param = $request->get_param('chart');

        // Check if we have a from date.
        if (!empty($from_date)) {
            $from_date = new DateTime($from_date);
            $log_query['date_query'][] = array(
                'column' => 'post_date_gmt',
                'after' => $from_date->format($date_format),
                'inclusive' => true,
            );
        }

        // Check if we have a to date.
        if (!empty($to_date)) {
            $to_date = new DateTime($to_date);
            $log_query['date_query'][] = array(
                'column' => 'post_date_gmt',
                'before' => $to_date->format($date_format),
                'inclusive' => true,
            );
        }

        $logs = get_posts($log_query);

        $chart = apply_filters('fast_filter_chart_' . $chart_param, array());

        $chart['active_series'] = apply_filters($chart['series_param_filter'], $chart['series'], $request);

        if (!empty($logs)) {
            // If we still don't have a from and to date we need to use the returned data.
            if (empty($from_date)) {
                $from_date = new DateTime($logs[0]->post_date_gmt);
            }
            if (empty($to_date)) {
                $to_date = new DateTime(end($logs)->post_date_gmt);
            }

            if (!empty($chart['active_series'])) {
                // Create chart data lookup table.
                $lookup_table = fast_analytics_create_lookup_table($logs, $chart['active_series'], $span_param);

                // Create chart data.
                $chart['data'] = fast_analytics_create_chart_data($chart['active_series'], $from_date, $to_date, $span_param, $lookup_table);
            }
        }

        global $wp_rest_server;
        $wp_rest_server->send_header('X-Fast-From', $from_date->format($date_format));
        $wp_rest_server->send_header('X-Fast-To', $to_date->format($date_format));
        $wp_rest_server->send_header('X-Fast-Span', $span_param);

        return $chart;
    }

    /**
     * Returns public ticket search data
     *
     * @param object $request WP_REST_Request object.
     */
    public function route_public_search($request)
    {
        // WordPress first page is 1.
        // Angular Material pagination first page is 0.
        // So increment by 1.
        $query_args = array(
            'post_type'      => 'fast_ticket',
            'order'          => $request->get_param('order'),
            'orderby'        => $request->get_param('orderby'),
            'paged'          => $request->get_param('page') + 1,
            'posts_per_page' => $request->get_param('page_size'),
            'post_status'    => 'publish'
        );

        $ticket_search = $request->get_param('search');
        if (null !== $ticket_search && '' !== $ticket_search) {
            $query_args['s'] = $ticket_search;
        }

        // If we have a ticket product add to query.
        $ticket_product = $request->get_param('ticket_product');
        if (null !== $ticket_product && 'all' !== $ticket_product) {
            $query_args['tax_query'] = array(
                array(
                    'taxonomy' => 'fast_product',
                    'field'    => 'term_id',
                    'terms'    => $ticket_product
                )
            );
        }

        // Make query.
        $posts_query  = new WP_Query();
        $posts = $posts_query->query($query_args);

        // Add header with total.
        global $wp_rest_server;
        $wp_rest_server->send_header('x-wp-total', $posts_query->found_posts);

        // Prepare return tickets.
        $tickets = array();
        foreach ($posts as $post) {
            $ticket = new stdClass();
            $ticket->id = $post->ID;
            $ticket->title = new stdClass();
            $ticket->title->rendered = $post->post_title;
            $ticket->ticket_product = wp_get_post_terms($post->ID, 'fast_product', array(
                'fields' => 'ids'
            ));
            $ticket->ticket_status = wp_get_post_terms($post->ID, 'fast_status', array(
                'fields' => 'ids'
            ));
            $tickets[] = $ticket;
        }
        return $tickets;
    }

    /**
     * Gets public ticket.
     *
     * @param object $request HTTP Request.
     */
    public function route_public_ticket($request)
    {
        $id = $request->get_param('id');
        $post = get_post($id);

        if ('publish' !== $post->post_status) {
            return new WP_Error('fast_access_denied', esc_html__('Sorry, you can not view this ticket.'), array(
                'status' => 403
            ));
        }

        $ticket = new stdClass();
        $ticket->id = $post->ID;
        $ticket->title = $post->post_title;
        $ticket->author = (int) $post->post_author;
        $ticket->ticket_product = wp_get_post_terms($post->ID, 'fast_product', array(
            'fields' => 'ids'
        ));
        $ticket->ticket_status = wp_get_post_terms($post->ID, 'fast_status', array(
            'fields' => 'ids'
        ));
        $ticket->comments = array();

        // Get comments.
        $comments = get_comments(array(
            'post_id' => $post->ID,
            'meta_query' => array(
                array(
                    'key' => '_agent_only',
                    'compare' => 'NOT EXISTS'
                )
            )
        ));

        foreach ($comments as $comment) {
            $rest_comment = new WP_REST_Comments_Controller();
            $data = $rest_comment->prepare_item_for_response($comment, $request);
            $ticket_comment = $rest_comment->prepare_response_for_collection($data);
            unset($ticket_comment['capabilities']);
            $ticket->comments[] = $ticket_comment;
        }

        return $ticket;
    }

    /**
     * Returns user data that we need for the app
     *
     * @param int $user_id (optional) id of user data to return
     */
    private function get_user_data($user_id = null)
    {
        global $post;
        if (null === $user_id) {
            $current_user = wp_get_current_user();
        } else {
            $current_user = get_user_by('id', $user_id);
        }

        fast_create_default_options();

        return array(
            'id'                => $current_user->ID,
            'first_name'        => $current_user->first_name,
            'last_name'         => $current_user->last_name,
            'description'       => $current_user->description,
            'roles'             => fast_get_user_roles(),
            'avatar'            => get_avatar_url($current_user->user_email),
            'ticket_count'      => (int) get_user_meta($current_user->ID, '_ticket_count', true),
            'purchases'         => get_user_meta($current_user->ID, '_purchases', true),
            'profile_edit_url'  => fast_get_option('profile_edit_url'),
            'language'          => get_user_locale(),
        );
    }
}
