<?php
/**
 * Creates email piping for fast
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

use PhpImap\Mailbox as ImapMailbox;
use PhpImap\IncomingMail;
use PhpImap\IncomingMailAttachment;
use EmailReplyParser\Parser\EmailParser;

/**
 * Adds extra endpoints for api
 */
class FastEmailPipe
{
    /**
     * Constructor for the api.
     */
    public function __construct()
    {
    }

    /**
     * Checks imap for emails
     */
    public function check_emails()
    {
        $report = new stdClass();
        $report->date = time();
        $report->email_count = 0;
        $report->replies_created = 0;
        $report->new_tickets_created = 0;
        $report->errors = array();

        $path = '{';
        $path .= fast_get_option('emailp_url') . ':' . fast_get_option('emailp_port');
        $path .= '/imap' . fast_get_option('emailp_encryption');
        $path .= fast_get_option('emailp_valid_cert');
        $path .= '}';
        $path .= fast_get_option('emailp_mailbox');

        $email_password = fast_get_option('emailp_password');
        $email_password = fast_store_sensitive_e($email_password, 'd');

        if (!function_exists('imap_open')) {
            $report->errors[] = esc_html__('PHP IMAP is not installed, please see Email Piping section of the docs.', 'fast-plugin');
        }

        // Save in case of errors.
        update_option('fast_email_pipe_report', $report);

        try {
            $mailbox = new PhpImap\Mailbox(
                $path,
                fast_get_option('emailp_username'),
                $email_password,
                sys_get_temp_dir(),
                fast_get_option('emailp_encoding')
            );
        } catch (Exception $e) {
            $report->errors[] = $e->getMessage();
        }

        // Save in case of errors.
        update_option('fast_email_pipe_report', $report);

        // Read all messaged into an array.
        try {
            $mail_ids = $mailbox->searchMailbox('ALL UNSEEN');

            // Do we have any mails?
            if ($mail_ids) {
                foreach ($mail_ids as $mail_id) {
                    $mail = $mailbox->getMail($mail_id);

                    $report->email_count += 1;

                    // Get all the info we need from the mail.
                    $reply_ticket  = $this->mail_is_ticket_reply($mail);
                    $reply_content = $this->get_email_reply_content($mail);
                    $reply_user    = $this->get_email_user($mail);

                    // Check if we have  a reply.
                    if (false !== $reply_ticket) {
                        // Only send reply if we have content user and enabled option.
                        if ('on' === fast_get_option('emailp_reply_enabled') && false !== $reply_user) {
                            wp_set_current_user($reply_user->ID);
                            $this->create_reply_comment($mail, $reply_ticket, $reply_content, $reply_user);
                            $report->replies_created += 1;
                        }
                        $mailbox->markMailAsRead($mail_id);
                    } elseif ('on' === fast_get_option('emailp_create_enabled')) {
                        // Try to make a new ticket with email.
                        $this->create_new_ticket($mail, $reply_content, $reply_user);
                        $report->new_tickets_created += 1;
                        $mailbox->markMailAsRead($mail_id);
                    }
                }
            }
        } catch (Exception $e) {
            $report->errors[] = $e->getMessage();
        }

        update_option('fast_email_pipe_report', $report);
    }

    /**
     * Checks mail to see if it is a valid reply to a ticket and makes commment if valid.
     *
     * @param object $mail Email to check.
     *
     * @return boolean|object false if not a reply Ticket post if valid reply
     */
    private function mail_is_ticket_reply($mail)
    {
        $matches_count = preg_match_all('/#(\d+)/', $mail->subject, $matches);
        // Did we find any #numbers?
        if ($matches_count > 0) {
            // Get the id of the ticket and fetch it.
            $ticket_id = $matches[1][0];
            $ticket = get_post($ticket_id);
            // Make sure ticket is a real post and a ticket post type.
            if (null !== $ticket && 'fast_ticket' === $ticket->post_type) {
                return $ticket;
            }
        }

        return false;
    }

    /**
     * Parses mail for reply content.
     *
     * @param object $mail Email to parse.
     *
     * @return string Reply text
     */
    private function get_email_reply_content($mail)
    {
        // Make an email parser, were gonna need it.
        $email_parser = new EmailParser();
        // Use the email parser to get the reply from the rest of the junk in the email.
        $email_content = $email_parser->parse($mail->textPlain);
        return $email_content->getVisibleText();
    }

    /**
     * Gets WordPress user from the email.
     *
     * @param object $mail Email to check.
     *
     * @return object WordPress user.
     */
    private function get_email_user($mail)
    {
        return get_user_by('email', $this->get_sender_address($mail));
    }

    /**
     * Gets
     *
     * @param object $mail Email to check.
     *
     * @return object WordPress user.
     */
    private function get_sender_address($mail)
    {

        if ('on' === fast_get_option('emailp_use_reply_to_header') && !empty($mail->replyTo)) {
            reset($mail->replyTo);
            return key($mail->replyTo);
        }
        return $mail->fromAddress;
    }

    /**
     * Creates a new comment
     *
     * @param object $mail Email to use to create ticket comment.
     * @param object $ticket Ticket to use to create ticket comment.
     * @param string $content Comment content.
     * @param object $user User to use to create ticket comment.
     */
    private function create_reply_comment($mail, $ticket, $content, $user)
    {
        // We have a valid user and some content lets make a reply on the ticket.
        $new_comment_id = wp_insert_comment(array(
            'comment_post_ID'      => $ticket->ID,
            'comment_author'       => $user->get('display_name'),
            'comment_author_email' => $user->get('user_email'),
            'comment_author_url'   => $user->get('user_url'),
            'comment_content'      => $content,
            'user_id'              => $user->ID,
            // Maybe change date to email date? 'comment_date' => $time.
        ));

        update_comment_meta($new_comment_id, '_from_email', true);

        // Add any attachments to the comment.
        $this->create_attachments_from_email($mail, $ticket, $new_comment_id);

        // Get comment object.
        $comment = get_comment($new_comment_id);

        // Call action for comment creation.
        do_action('fast_action_post_comment', $comment);
    }

    /**
     * Creates a new ticket from the email.
     *
     * @param object $mail Email to use to create ticket comment.
     * @param string $content Comment content.
     * @param object $user User to use to create ticket comment.
     */
    private function create_new_ticket($mail, $content, $user)
    {
        if (false === $user) {
            // By default use from name as first name and skip last name.
            $first_name = $mail->fromName;
            $last_name = '';

            // Check if we have a spacee to split the name up.
            if (false !== strpos($first_name, ' ')) {
                // Split the name into 2 parts.
                $names = explode(' ', $mail->fromName, 2);
                $first_name = $names[0];
                $last_name = $names[1];
            }

            $password = wp_generate_password(10, false, false);
            $user = fast_register_user(
                $first_name,
                $first_name,
                $last_name,
                $this->get_sender_address($mail),
                $password
            );

            wp_set_current_user($user->ID);

            if ('on' === fast_get_option('email_new_user')) {
                fast_send_registration_email($this->get_sender_address($mail), $password);
            }
        } // End if().

        // Check the default privacy settings to set the post status.
        $default_privacy = fast_get_option('private_tickets');
        $post_status = $default_privacy === 'always' ? 'private' : 'publish';

        // Create ticket Data.
        $ticket_data = array(
            'post_author' => $user->ID,
            'post_title'  => $mail->subject,
            'post_status' => $post_status,
            'post_type'   => 'fast_ticket',
        );

        // Create ticket.
        $ticket_id = wp_insert_post($ticket_data);

        // Create comment on new ticket.
        if (!is_wp_error($ticket_id)) {
            // Add a meta data to ticket so we know its from an email.
            update_post_meta($ticket_id, '_from_email', true);

            // Assign default product if we have one.
            $ticket_default_product = fast_get_option('emailp_default_product');
            if (!empty($ticket_default_product)) {
                wp_set_post_terms($ticket_id, array(
                    (int) $ticket_default_product
                ), 'fast_product');
            }

            // Assign default category if we have one.
            $ticket_default_category = fast_get_option('emailp_default_category');
            if (!empty($ticket_default_category)) {
                wp_set_post_terms($ticket_id, array(
                    (int) $ticket_default_category
                ), 'fast_category');
            }

            $ticket = get_post($ticket_id);
            // Create reply comment.
            $this->create_reply_comment($mail, $ticket, $content, $user);
        }
    }

    /**
     * Creates attachment for comment if there is one
     *
     * @param object  $mail Email to use to get attachments.
     * @param object  $ticket Ticket post.
     * @param integer $comment_id Comment id.
     */
    private function create_attachments_from_email($mail, $ticket, $comment_id)
    {
        $attachments = $mail->getAttachments();

        if (!empty($attachments)) {
            $attachment_meta = array();

            foreach ($attachments as $attachment) {
                $upload_file = wp_upload_bits($attachment->name, null, file_get_contents($attachment->filePath));

                if (!$upload_file['error']) {
                    $wp_filetype = wp_check_filetype($attachment->name, null);

                    $attachment_post = array(
                        'post_mime_type' => $wp_filetype['type'],
                        'post_parent'    => $ticket->ID,
                        'post_title'     => $attachment->name,
                        'post_content'   => '',
                        'post_status'    => 'inherit'
                    );
                    $attachment_id = wp_insert_attachment($attachment_post, $upload_file['file'], $ticket->ID);

                    if (!is_wp_error($attachment_id)) {
                        require_once(ABSPATH . 'wp-admin/includes/image.php');
                        require_once(ABSPATH . 'wp-admin/includes/media.php');
                        $attachment_data = wp_generate_attachment_metadata($attachment_id, $upload_file['file']);
                        wp_update_attachment_metadata($attachment_id, $attachment_data);

                        $save_attachment = new stdClass();

                        $save_attachment->loading = false;
                        switch ($wp_filetype['type']) {
                            case 'image/jpeg':
                            case 'image/png':
                            case 'image/gif':
                                $save_attachment->media_type = 'image';
                                break;

                            default:
                                $save_attachment->media_type = 'file';
                                break;
                        }
                        $save_attachment->date = $ticket->post_date_gmt;
                        $save_attachment->id = $attachment_id;
                        $save_attachment->source_url = wp_get_attachment_url($attachment_id);
                        $save_attachment->thumbnail = wp_get_attachment_thumb_url($attachment_id);

                        $attachment_ids[] = $save_attachment;
                    }
                } // End if().
            } // End foreach().

            if (!empty($attachment_ids)) {
                update_comment_meta($comment_id, '_attachments', $attachment_ids);
            }
        } // End if().
    }
}
