<?php
/**
 * Initialises the app
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Class to setup the app
 */
class FastTicketApp
{
    /**
     * Stores the singleton instance of the class
     *
     * @var FastTicketApp
     */
    private static $instance;

    /**
     * Creates a new instance of the class
     */
    public function __construct()
    {
        // Start the clean slate so we have WordPress all to ourselves.
        require FAST_PLUGIN_DIR . 'classes/class-clean-slate.php';
        FastCleanSlate::init();

        add_action('wp_enqueue_scripts_clean', array(&$this, 'enqueue_scripts_and_styles'));

        add_action('wp_clean_slate_head', array(&$this, 'add_base_tag'));

        add_action('wp_clean_slate_head', array(&$this, 'add_css_styling'));

        add_action('wp_clean_slate_content', array(&$this, 'add_content'));
    }

    /**
     * Enqueues app styles and scripts
     */
    public function enqueue_scripts_and_styles()
    {
        wp_register_script('fast-script-app_inline', FAST_PLUGIN_URI . 'assets/app/inline.bundle.js', array(), '1.15.4', true);
        $current_user = wp_get_current_user();
        $home_url = function_exists('icl_get_home_url') ? icl_get_home_url() : home_url();
        $logout_url = fast_get_option('logout_url');
        $support_page_id = fast_get_option('support_page');

        $options = array(
            'nonce'                       => wp_create_nonce('ticket-api-action'),
            'app_logo_image'              => fast_get_option('app_logo_image'),
            'private_tickets'             => fast_get_option('private_tickets'),
            'assets_folder_url'           => FAST_PLUGIN_URI . 'assets/',
            'envato_login'                => fast_get_option('envato_login') === 'on',
            'ticket_comments_order'       => fast_get_option('ticket_comments_order'),
            'ticket_default_internal_replies'     => fast_get_option('ticket_default_internal_replies') === 'on',
            'envato_create_ticket_when_supported' => fast_get_option('envato_create_ticket_when_supported') === 'on',
            'envato_reply_to_ticket_when_supported'  => fast_get_option('envato_reply_to_ticket_when_supported') === 'on',
            'envato_support_update_url'   => fast_get_option('envato_support_update_url'),
            'envato_login_url'            => fast_generate_envato_oauth_login_url(),
            'recaptcha_site_key'          => fast_get_option('recaptcha_site_key'),
            'register_user_enabled'       => fast_get_option('register_user_enabled') === 'on',
            'register_page_title'         => fast_get_option('register_page_title'),
            'register_page_content'       => wpautop(fast_get_option('register_page_content')),
            'register_types'              => fast_get_option('register_types'),
            'login_page_title'            => fast_get_option('login_page_title'),
            'login_page_login_types'      => fast_get_option('login_page_login_types'),
            'login_page_content'          => wpautop(fast_get_option('login_page_content')),
            'new_ticket_type'             => is_array(fast_get_option('new_ticket_type')) ? fast_get_option('new_ticket_type') : array(fast_get_option('new_ticket_type')),
            'new_ticket'                  => array(
                'intro_tab' => array(
                    'show' => fast_get_option('intro_tab_show') === 'on',
                    'label' => fast_get_option('intro_tab_title'),
                    'title' => fast_get_option('new_ticket_page_title'),
                    'content' => fast_get_option('new_ticket_page_content'),
                ),
                'user_tab' => array(
                    'label' => fast_get_option('user_tab_title'),
                ),
                'enquiry_tab' => array(
                    'label'       => fast_get_option('enquiry_tab_label'),
                    'title'       => fast_get_option('enquiry_tab_title'),
                    'options' => array(
                        'regular'     => fast_get_option('enquiry_tab_regular_text'),
                        'general'     => fast_get_option('enquiry_tab_general_text'),
                        'envato'      => fast_get_option('enquiry_tab_envato_text'),
                        'woocommerce' => fast_get_option('enquiry_tab_woocommerce_text'),
                    )
                ),
                'ticket_tab' => array(
                    'label' => fast_get_option('ticket_tab_label'),
                    'text' => fast_get_option('ticket_tab_text'),
                ),
                'product_tab' => array(
                    'label' => fast_get_option('product_tab_label'),
                    'text' => fast_get_option('product_tab_text'),
                ),
                'category_tab' => array(
                    'label' => fast_get_option('category_tab_label'),
                    'text' => fast_get_option('category_tab_text'),
                ),
                'extra_fields_tab' => array(
                    'label' => fast_get_option('extra_fields_tab_label'),
                    'text' => fast_get_option('extra_fields_tab_text'),
                ),
                'secure_fields_tab' => array(
                    'label' => fast_get_option('secure_fields_tab_label'),
                    'text' => fast_get_option('secure_fields_tab_text'),
                ),
                'envato_tab' => array(
                    'label' => fast_get_option('envato_tab_label'),
                    'text' => fast_get_option('envato_tab_text'),
                ),
                'wc_product_tab' => array(
                    'label' => fast_get_option('wc_product_tab_label'),
                    'text' => fast_get_option('wc_product_tab_text'),
                ),
                'wc_order_tab' => array(
                    'label' => fast_get_option('wc_order_tab_label'),
                    'text' => fast_get_option('wc_order_tab_text'),
                ),
            ),
            'extra_fields'                => fast_get_option('extra_fields'),
            'attachment_uploads_enabled'  => fast_get_option('attachment_uploads_enabled') === 'on',
            'max_upload_size'             => fast_get_option('max_upload_size'),
            'translate_based_on_user_locale' => fast_get_option('translate_based_on_user_locale') === 'on',
            'site_language'               => get_locale(),
            'available_languages'         => fast_get_app_available_languages(),
            'agent_ticket_creation'       => fast_get_option('agent_ticket_creation') === 'on',
            'agent_user_ticket_creation' => fast_get_option('agent_user_ticket_creation') === 'on',
            'register_user_language'      => fast_get_option('register_user_language') === 'on',
            'register_agreement_enabled'  => fast_get_option('register_agreement_enabled') === 'on',
            'register_agreement_label'    => fast_get_option('register_agreement_label'),
            'register_agreement_text'     => wpautop(fast_get_option('register_agreement_text')),
            'favicon_color'               => fast_get_option('favicon_color'),
            'favicon_bg_color'            => fast_get_option('favicon_bg_color'),
            'admin_url'                   => admin_url(),
            'home_url'                    => $home_url,
            'show_home_icon'              => fast_get_option('show_home_icon') === 'on',
            'wp_lostpassword_url'         => wp_lostpassword_url(),
            'customer_reply_to_closed'    => fast_get_option('customer_reply_to_closed') === 'on',
            'product_singular_text'       => fast_get_option('tax_product_singular_name'),
            'product_plural_text'         => fast_get_option('tax_product_name'),
            'login_url'                   => fast_get_option('login_url'),
            'logout_url'                  => empty($logout_url) ? fast_get_app_url() : fast_get_option('logout_url'),
            'app_url'                     => fast_get_app_url(),
            'support_policy_link_url'     => esc_url(fast_get_option('support_policy_link_url')),
            'support_policy_link_text'    => fast_get_option('support_policy_link_text'),
            'exit_icon_link'              => fast_get_option('exit_icon_link'),
            'show_user_profile'           => fast_get_option('show_user_profile') === 'on',
            'show_keyboard_shortcuts'     => fast_get_option('show_keyboard_shortcuts') === 'on',
            'ticket_show_client_mail'     => fast_get_option('ticket_show_client_mail') === 'on',
            'extra_fields_enabled'       => fast_get_option('extra_fields_enabled') === 'on',
            'secure_fields_enabled'       => fast_get_option('secure_fields_enabled') === 'on',
            'secure_fields_external_url'  => fast_get_option('secure_fields_external_url'),
            'allow_agent_private_comments' => fast_get_option('allow_agent_private_comments') === 'on',
            'allow_customer_notes'        => fast_get_option('allow_customer_notes') === 'on',
            'envato_allow_enter_purchases' => fast_get_option('envato_allow_enter_purchases') === 'on',
            'page_title'                  => get_the_title($support_page_id),
            'preferable_upload_formats'   => fast_get_option('preferable_upload_formats'),
            'sidebar_customer_info'       => fast_get_option('sidebar_customer_info'),
            'analytics_access'            => fast_get_option('analytics_enabled') === 'on' ? fast_get_option('analytics_access') : array(),
            'show_response_times'         => fast_get_option('show_response_times'),
            'woocommerce_order_statuses'  => fast_get_option('woocommerce_order_statuses'),
            'translations_folder_url'     => defined('FAST_SINGLE_TRANSLATION_FILE') ? FAST_PLUGIN_URI . 'languages' : FAST_PLUGIN_URI . 'assets/i18n',
            'customer_search_enabled'     => fast_get_option('customer_search_enabled') === 'on',
            'show_avatars'                => get_option('show_avatars') === '1'
        );

        if (fast_is_woocommerce_active()) {
            $options['wc_order_status'] = wc_get_order_statuses();
        }

        wp_localize_script('fast-script-app_inline', 'wpLocal', array(
            'root'    => rest_url(),
            'nonce'   => wp_create_nonce('wp_rest'),
            'options' => $options
        ));

        $app_color = fast_get_option('app_color');
        // Are we using a ready made color scheme?
        if (is_numeric($app_color)) {
            // Custom Skin.
            $wp_upload_dir = wp_upload_dir();
            $suffix = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';
            $style_url = $wp_upload_dir['baseurl'] . '/fast-plugin/' . $app_color . '/stack' . $suffix . '.css';

            $google_fonts_url = get_post_meta($app_color, '_fast-plugin-stack-google-url', true);
        } else {
            // Ready made.
            $style_url = FAST_PLUGIN_URI . 'assets/app/themes/' . fast_get_option('app_color') . '.bundle.css';
            // Load default font if not custom.
            $google_fonts_url = 'https://fonts.googleapis.com/icon?family=Roboto:300,400,400i,700|Material+Icons';
        }

        wp_enqueue_style('fast-style-google-fonts', $google_fonts_url);
        wp_enqueue_style('fast-style-app_styles', set_url_scheme($style_url), array(), '1.15.4');

        wp_enqueue_script('fast-script-app_inline');
        wp_enqueue_script('fast-script-app_polyfills', FAST_PLUGIN_URI . 'assets/app/polyfills.bundle.js', array(), '1.15.4', true);
        wp_enqueue_script('fast-script-app_scripts', FAST_PLUGIN_URI . 'assets/app/scripts.bundle.js', array(), '1.15.4', true);
        wp_enqueue_script('fast-script-app_main', FAST_PLUGIN_URI . 'assets/app/main.bundle.js', array(), '1.15.4', true);
    }

    /**
     * Adds the base tag to the head (needed by angular)
     */
    public function add_base_tag()
    {
        echo '<base href="' . esc_url(get_the_permalink()) . '">';
        if (has_site_icon()) {
            wp_site_icon();
        }
    }

    /**
     * Adds the base tag to the head (needed by angular)
     */
    public function add_css_styling()
    {
        echo '<style type="text/css">' . fast_custom_css() . '</style>';
        echo '<style type="text/css" id="app-extra-css">' . fast_get_option('extra_css') . '</style>';
    }

    /**
     * Adds the app root to the content of the page
     */
    public function add_content()
    {
        echo '<tick-app><div class="app-spinner"><div class="loader"></div></div></tick-app>';
    }

    /**
     * Instance accessor. If instance doesn't exist, we'll initialize the class.
     *
     * @return object FastTicketApp
     */
    public static function instance()
    {
        if (!isset(self::$instance)) {
            self::$instance = new FastTicketApp;
        }
        return self::$instance;
    }
}
