<?php
/**
 * Actions for the plugin
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/*
 * register_activation_hook
 */

/**
 * Run when the plugin activates.
 */
function fast_install_plugin()
{
    add_option('fast-plugin-activated', 'fast-plugin');
}
register_activation_hook(FAST_PLUGIN_FILE, 'fast_install_plugin');

/*
 * plugins_loaded
 */

/**
 * Load plugin textdomain.
 */
function fast_load_textdomain()
{
    load_plugin_textdomain('fast-plugin', false, 'fast-plugin/languages');
}
add_action('plugins_loaded', 'fast_load_textdomain');


/**
 * Loads woocommerce code if needed.
 */
function fast_load_woocommerce()
{
    // If woocommerce is active and plugin is in woocommerce mode then load the woocommerce code.
    if (fast_is_woocommerce_active()) {
        require FAST_PLUGIN_DIR . 'inc/woocommerce/filters.php';
        require FAST_PLUGIN_DIR . 'inc/woocommerce/actions.php';
    }
}
add_action('plugins_loaded', 'fast_load_woocommerce');

/*
 * init
 */

/**
 * Adds custom user roles.
 */
function fast_add_user_roles()
{
    if (is_admin() &&  'fast-plugin' === get_option('fast-plugin-activated')) {
        delete_option('fast-plugin-activated');

        // Add ticket capabilities to administrator role.
        $caps = apply_filters('fast_filter_ticket_capabilities', array());
        $admin = get_role('administrator');
        foreach ($caps as $cap => $val) {
            $admin->add_cap($cap);
        }

        fast_create_plugin_default_roles();

        $taxonomies = array(
            'fast_status' => array(
                'Open',
                'Closed'
            )
        );
        foreach ($taxonomies as $taxonomy => $terms) {
            foreach ($terms as $term) {
                if (! get_term_by('slug', sanitize_title($term), $taxonomy)) {
                    $result = wp_insert_term($term, $taxonomy);
                }
            }
        }
        // Save Permalinks.
        flush_rewrite_rules();
    }
}
add_action('init', 'fast_add_user_roles', 100);

/**
 * Registers a custom post types.
 */
function fast_register_custom_posts()
{
    register_post_type('fast_ticket', array(
        'labels' => array(
            'name'          => esc_html__('Tickets', 'fast-plugin'),
            'singular_name' => esc_html__('Ticket', 'fast-plugin')
        ),
        'supports'              => array('title', 'author', 'custom-fields', 'comments'),
        'public'                => true,
        'has_archive'           => false,
        'menu_icon'             => 'dashicons-tickets',
        'show_in_rest'          => true,
        'rest_base'             => 'tickets',
        'rest_controller_class' => 'FastRESTTicketController',
        'rewrite'               => false,
        'exclude_from_search'   => true,
        'publicly_queryable'    => true,
        'capabilities' => array(
            'edit_post'              => 'edit_ticket',
            'read_post'              => 'read_ticket',
            'delete_post'            => 'delete_ticket',

            'edit_posts'             => 'edit_tickets',
            'edit_others_posts'      => 'edit_others_tickets',
            'publish_posts'          => 'publish_tickets',
            'read_private_posts'     => 'read_private_tickets',

            'read'                   => 'read',
            'delete_posts'           => 'delete_ticket',
            'delete_private_posts'   => 'delete_ticket',
            'delete_published_posts' => 'delete_ticket',
            'delete_others_posts'    => 'delete_ticket',
            'edit_private_posts'     => 'read_private_tickets',
            'edit_published_posts'   => 'read_private_tickets',
            'create_posts'           => 'edit_tickets',
        )
    ));

    register_post_type('fast_reply', array(
        'labels' => array(
            'name'          => esc_html__('Saved Replies', 'fast-plugin'),
            'singular_name' => esc_html__('Saved Reply', 'fast-plugin')
        ),
        'supports'              => array('title', 'editor', 'author'),
        'public'                => false,
        'has_archive'           => false,
        'menu_icon'             => 'dashicons-book-alt',
        'show_in_rest'          => true,
        'show_ui'               => true,
        'rest_base'             => 'replies',
        'rest_controller_class' => 'FastRESTTicketController',
        'rewrite'               => false,
        'exclude_from_search'   => true,
        'publicly_queryable'    => false,
        'capabilities' => array(
            'edit_post'              => 'edit_reply',
            'read_post'              => 'read_reply',
            'delete_post'            => 'delete_reply',

            'edit_posts'             => 'edit_replies',
            'edit_others_posts'      => 'edit_others_replies',
            'publish_posts'          => 'publish_replies',

            'read'                   => 'read',
            'delete_posts'           => 'delete_reply',
            'delete_private_posts'   => 'delete_reply',
            'delete_published_posts' => 'delete_reply',
            'delete_others_posts'    => 'delete_reply',
            'edit_private_posts'     => 'read_private_replies',
            'edit_published_posts'   => 'read_private_replies',
            'create_posts'           => 'edit_replies',
        )
    ));

    register_post_type('fast_purchase', array(
        'supports'              => array('author'),
        'public'                => false,
        'has_archive'           => false,
        'show_in_rest'          => true,
        'show_ui'               => false,
        'rest_base'             => 'envato_purchases',
        'rest_controller_class' => 'FastRESTTicketController',
        'rewrite'               => false,
        'exclude_from_search'   => true,
        'publicly_queryable'    => false,
        'capabilities' => array(
            'edit_post'              => 'edit_purchase',
            'read_post'              => 'read_purchase',
            'delete_post'            => 'delete_purchase',

            'edit_posts'             => 'edit_purchases',
            'edit_others_posts'      => 'edit_others_purchases',
            'publish_posts'          => 'publish_purchases',

            'read'                   => 'read',
            'delete_posts'           => 'delete_purchases',
            'delete_private_posts'   => 'delete_private_purchases',
            'delete_published_posts' => 'delete_published_purchases',
            'delete_others_posts'    => 'delete_others_purchases',
            'edit_private_posts'     => 'edit_private_purchases',
            'edit_published_posts'   => 'edit_published_purchases',
            'create_posts'           => 'edit_purchases',
        )
    ));

    register_post_type('fast_log', array(
        'supports'              => array('custom-fields'),
        'public'                => false,
        'has_archive'           => false,
        'show_in_rest'          => false,
        'show_ui'               => false,
        'rewrite'               => false,
        'exclude_from_search'   => true,
        'publicly_queryable'    => false,
    ));
}
add_action('init', 'fast_register_custom_posts');

/**
 * Registers a custom post types.
 */
function fast_register_custom_taxonomies()
{
    register_taxonomy('fast_status', array('fast_ticket'), array(
        'labels'                => array(
            'name'              => esc_html_x('Statuses', 'taxonomy general name', 'fast-plugin'),
            'singular_name'     => esc_html_x('Status', 'taxonomy singular name', 'fast-plugin'),
            'search_items'      => esc_html__('Search Statuses', 'fast-plugin'),
            'all_items'         => esc_html__('All Statuses', 'fast-plugin'),
            'parent_item'       => esc_html__('Parent Status', 'fast-plugin'),
            'parent_item_colon' => esc_html__('Parent Status:', 'fast-plugin'),
            'edit_item'         => esc_html__('Edit Status', 'fast-plugin'),
            'update_item'       => esc_html__('Update Status', 'fast-plugin'),
            'add_new_item'      => esc_html__('Add New Status', 'fast-plugin'),
            'new_item_name'     => esc_html__('New Status Name', 'fast-plugin'),
            'menu_name'         => esc_html__('Statuses', 'fast-plugin'),
        ),
        'hierarchical'          => false,
        'show_ui'               => true,
        'show_in_menu'          => true,
        'show_admin_column'     => true,
        'query_var'             => true,
        'rewrite'               => false,
        'show_in_rest'          => true,
        'publicly_queryable'    => false,
        'rest_base'             => 'ticket_status',
        'rest_controller_class' => 'WP_REST_Terms_Controller',
        'update_count_callback' => '_update_generic_term_count',
        'capabilities'          => array(
            'manage_terms' => 'manage_statuses',
            'edit_terms'   => 'manage_statuses',
            'delete_terms' => 'manage_statuses',
            'assign_terms' => 'assign_status'
        )
    ));

    register_taxonomy('fast_product', array('fast_ticket'), array(
        'labels'                => array(
            'name'                       => fast_get_option('tax_product_name'),
            'singular_name'              => fast_get_option('tax_product_singular_name'),
            'search_items'               => fast_get_option('tax_product_search_items'),
            'edit_item'                  => fast_get_option('tax_product_edit_item'),
            'add_new_item'               => fast_get_option('tax_product_add_new_item'),
            'menu_name'                  => fast_get_option('tax_product_menu_name'),
            'separate_items_with_commas' => '',
            'choose_from_most_used'      => '',
        ),
        'hierarchical'          => false,
        'show_ui'               => true,
        'show_in_menu'          => true,
        'show_admin_column'     => true,
        'query_var'             => true,
        'rewrite'               => false,
        'show_in_rest'          => true,
        'publicly_queryable'    => false,
        'rest_base'             => 'ticket_product',
        'rest_controller_class' => 'WP_REST_Terms_Controller',
        'update_count_callback' => '_update_generic_term_count',
        'capabilities'          => array(
            'manage_terms' => 'manage_products',
            'edit_terms'   => 'manage_products',
            'delete_terms' => 'manage_products',
            'assign_terms' => 'assign_product'
        )
    ));

    register_taxonomy('fast_category', array('fast_ticket'), array(
        'labels'                => array(
            'name'              => esc_html_x('Categories', 'taxonomy general name', 'fast-plugin'),
            'singular_name'     => esc_html_x('Category', 'taxonomy singular name', 'fast-plugin'),
            'search_items'      => esc_html__('Search Categories', 'fast-plugin'),
            'all_items'         => esc_html__('All Categories', 'fast-plugin'),
            'parent_item'       => esc_html__('Parent Category', 'fast-plugin'),
            'parent_item_colon' => esc_html__('Parent Category:', 'fast-plugin'),
            'edit_item'         => esc_html__('Edit Category', 'fast-plugin'),
            'update_item'       => esc_html__('Update Category', 'fast-plugin'),
            'add_new_item'      => esc_html__('Add New Category', 'fast-plugin'),
            'new_item_name'     => esc_html__('New Category Name', 'fast-plugin'),
            'menu_name'         => esc_html__('Categories', 'fast-plugin'),
        ),
        'hierarchical'          => false,
        'show_ui'               => true,
        'show_in_menu'          => true,
        'show_admin_column'     => true,
        'query_var'             => true,
        'rewrite'               => false,
        'show_in_rest'          => true,
        'publicly_queryable'    => false,
        'rest_base'             => 'ticket_category',
        'rest_controller_class' => 'WP_REST_Terms_Controller',
        'update_count_callback' => '_update_generic_term_count',
        'capabilities'          => array(
            'manage_terms' => 'manage_categories',
            'edit_terms'   => 'manage_categories',
            'delete_terms' => 'manage_categories',
            'assign_terms' => 'assign_category'
        )
    ));


    register_taxonomy('fast_agent', array('fast_ticket'), array(
        'labels'                => array(
            'name'              => esc_html_x('Agents', 'taxonomy general name', 'fast-plugin'),
            'singular_name'     => esc_html_x('Agent', 'taxonomy singular name', 'fast-plugin'),
            'search_items'      => esc_html__('Search Agents', 'fast-plugin'),
            'all_items'         => esc_html__('All Agents', 'fast-plugin'),
            'parent_item'       => esc_html__('Parent Agent', 'fast-plugin'),
            'parent_item_colon' => esc_html__('Parent Agent:', 'fast-plugin'),
            'edit_item'         => esc_html__('Edit Agent', 'fast-plugin'),
            'update_item'       => esc_html__('Update Agent', 'fast-plugin'),
            'add_new_item'      => esc_html__('Add New Agent', 'fast-plugin'),
            'new_item_name'     => esc_html__('New Agent Name', 'fast-plugin'),
            'menu_name'         => esc_html__('Agents', 'fast-plugin'),
        ),
        'hierarchical'          => false,
        'show_ui'               => true,
        'show_in_menu'          => true,
        'show_admin_column'     => true,
        'query_var'             => true,
        'rewrite'               => false,
        'show_in_rest'          => true,
        'publicly_queryable'    => false,
        'rest_base'             => 'ticket_agent',
        'rest_controller_class' => 'WP_REST_Terms_Controller',
        'update_count_callback' => '_update_generic_term_count',
        'capabilities'          => array(
            'manage_terms' => 'manage_agents',
            'edit_terms'   => 'manage_agents',
            'delete_terms' => 'manage_agents',
            'assign_terms' => 'assign_agent'
        )
    ));

    // Register User ID meta data for agent taxonomy.
    register_meta('fast_agent', '_user_id', array(
        'sanitize_callback' => 'sanitize_text_field',
        'type'              => 'integer',
        'description'       => esc_html__('User id for this agent.', 'fast-plugin'),
        'single'            => true
    ));

    // Register User ID meta data for agent taxonomy.
    register_meta('fast_product', '_agent_ids', array(
        'sanitize_callback' => 'sanitize_text_field',
        'type'              => 'string',
        'description'       => esc_html__('Users that will handle this product.', 'fast-plugin'),
        'single'            => true
    ));

    // Register User ID meta data for agent taxonomy.
    register_meta('fast_category', '_agent_ids', array(
        'sanitize_callback' => 'sanitize_text_field',
        'type'              => 'string',
        'description'       => esc_html__('Users that will handle this category.', 'fast-plugin'),
        'single'            => true
    ));
}
add_action('init', 'fast_register_custom_taxonomies');

/**
 * Allows ticket endpoints for pages & root if front page set to support page
 */
function fast_rewrite_app_endpoints()
{
    $front_page = (int) get_option('page_on_front');
    $support_page_id = (int) fast_get_option('support_page');
    $app_endpoints = array(
        'ticket',
        'ticket-404',
        'login',
        'form-builder',
        'envato',
        'agent-search',
        'agent-results',
        'public-ticket',
        'search',
        'analytics'
    );

    if ('on' === fast_get_option('register_user_enabled')) {
        $app_endpoints[] = 'register';
    }

    if ($front_page === $support_page_id) {
        // Support page set as frontpage.
        foreach ($app_endpoints as $endpoint) {
            add_rewrite_rule(
                $endpoint . '(/(.*))?/?$',
                'index.php?ticket=$matches[2]&page_id=' . $support_page_id,
                'top'
            );
        }
        add_filter('redirect_canonical', 'fast_disable_canonical_redirect_for_front_page');
    } else {
        // Support page set as regular page.
        foreach ($app_endpoints as $endpoint) {
            add_rewrite_endpoint($endpoint, EP_PAGES);
        }
    }
}
add_action('init', 'fast_rewrite_app_endpoints');

/*
 * after_setup_theme
 */

/**
 * Adds the options & metaboxes defaults
 */
function fast_create_default_options()
{
    if (class_exists('FastOptionsDefaults')) {
        // Option page defaults.
        $plugin_options = apply_filters('fast_filter_options', array());
        $options_defaults = FastOptionsDefaults::instance();
        // WPML needs some options to be stored even if they are default.
        $current_options = get_option('fast-plugin-options', array());
        foreach ($plugin_options as $option_page) {
            foreach ($option_page['sections'] as $section) {
                foreach ($section['fields'] as $field) {
                    if (isset($field['id']) && isset($field['default'])) {
                        $options_defaults->add_default('fast-plugin-options', $field['id'], $field['default']);
                    }
                    // WPML needs an actual value stored to work so.
                    // We added the store_default option.
                    if (isset($field['store_default'])) {
                        if (!isset($current_options[$field['id']]) && isset($field['default'])) {
                            $current_options[$field['id']] = $field['default'];
                            // Save options because we need to store this one.
                            update_option('fast-plugin-options', $current_options);
                        }
                    }
                }
            }
        }
        // Register metabox defaults.
        $metabox_options = apply_filters('fast_filter_metabox_options', array());
        foreach ($metabox_options as $section) {
            foreach ($section['fields'] as $field) {
                if (isset($field['id']) && isset($field['default'])) {
                    $options_defaults->add_default('fast-plugin-metabox-options', $field['id'], $field['default']);
                }
            }
        }
    }
}
add_action('after_setup_theme', 'fast_create_default_options');

/*
 * template_redirect
 */

/**
 * Redirects support page to app
 */
function fast_redirect_template_to_app()
{
    global $wp_query;
    $support_page_id = (int) fast_get_option('support_page');
    // WPML check for translated page.
    if (function_exists('icl_object_id')) {
        $support_page_id = icl_object_id($support_page_id, 'post', true, ICL_LANGUAGE_CODE);
    }
    if (get_the_ID() === $support_page_id) {
        require FAST_PLUGIN_DIR . 'classes/class-ticket-app.php';
        FastTicketApp::instance();
    }
}
add_action('template_redirect', 'fast_redirect_template_to_app');

/*
 * set_auth_cookie
 */

/**
 * Make sure that we set the superglobal $_COOKIE when logging in so that the API returns a still valid nonce.
 *
 * @param object $auth_cookie Cookie that has been created.
 */
function fast_set_auth_cookie($auth_cookie)
{
    $_COOKIE[AUTH_COOKIE] = $auth_cookie;
}
add_action('set_auth_cookie', 'fast_set_auth_cookie', 10, 1);

/*
 * set_logged_in_cookie
 */

/**
 * Make sure that we set the superglobal $_COOKIE when logging in so that the API returns a still valid nonce.
 *
 * @param object $logged_in_cookie Cookie that has been created.
 */
function fast_set_logged_in_cookie($logged_in_cookie)
{
    $_COOKIE[LOGGED_IN_COOKIE] = $logged_in_cookie;
}
add_action('set_logged_in_cookie', 'fast_set_logged_in_cookie', 10, 1);

/*
 * save_post_fast_ticket
 */

/**
 * Assigns default status to a new ticket
 *
 * @param int  $post_id Post id being saved.
 * @param post $post The post object.
 * @param bool $update Whether this is an existing post being updated or not.
 */
function fast_on_create_new_ticket($post_id, $post, $update)
{
    if ((defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) || wp_is_post_revision($post_id)) {
        return;
    }

    if (true === $update) {
        return;
    }
    // Make sure ticket is open.
    wp_set_object_terms($post_id, 'open', 'fast_status', false);

    // Increase the user's ticket count.
    $current_user_id = get_current_user_id();
    // If we are creating a ticket for a customer, use the authors id.
    if (!empty($post->post_author)) {
        $current_user_id = $post->post_author;
    }
    $user_ticket_count = (int) get_user_meta($current_user_id, '_ticket_count', true);
    $updated_ticket_count = $user_ticket_count + 1;
    update_user_meta($current_user_id, '_ticket_count', $updated_ticket_count);
}
add_action('save_post_fast_ticket', 'fast_on_create_new_ticket', 10, 3);

/**
 * Decreases ticket count upon ticket deletion.
 *
 * @param int  $post_id Post id being saved.
 * @param post $post The post object.
 * @param bool $update Whether this is an existing post being updated or not.
 */
function fast_on_delete_ticket($post_id, $post, $update)
{
    if ((defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) || wp_is_post_revision($post_id)) {
        return;
    }

    if (false === $update || 'trash' !== get_post_status($post_id)) {
        return;
    }

    // Decrease user ticket count.
    $current_user_id = get_current_user_id();
    $user_ticket_count = (int) get_user_meta($current_user_id, '_ticket_count', true);
    $updated_ticket_count = $user_ticket_count - 1;
    update_user_meta($current_user_id, '_ticket_count', $updated_ticket_count);
}
add_action('save_post_fast_ticket', 'fast_on_delete_ticket', 10, 3);

/*
 * rest_prepare_comment
 */

/**
 * Creates internal actions for comment updates.
 * e.g.
 * fast_action_post_comment
 * fast_action_post_comment_agent_only
 * fast_action_put_comment
 * fast_action_put_comment_agent_only
 *
 * @param WP_REST_Response $response The response object.
 * @param WP_Comment       $comment  The original comment object.
 * @param WP_REST_Request  $request  Request used to generate the response.
 */
function fast_rest_prepare_comment($response, $comment, $request)
{
    // Only do this on tickets.
    if ('fast_ticket' !== get_post_type($comment->comment_post_ID)) {
        return $response;
    }
    // if url params are empty, an id was not included in the request, thus it's a new comment.
    $url_params = $request->get_url_params();
    $method = $request->get_method() === 'GET' ? 'get' : (empty($url_params) ? 'post': 'put');
    $agent_only_reply = get_comment_meta($comment->comment_ID, '_agent_only', false);
    $agent_only_action_name = empty($agent_only_reply) ? '' : '_agent_only';
    do_action('fast_action_' . $method . '_comment' . $agent_only_action_name, $comment);
    return $response;
}
add_action('rest_prepare_comment', 'fast_rest_prepare_comment', 10, 3);

/*
 * fast_action_post_comment
 */

/**
 * Sends an email and/or slack notice to the agent/client upon comment addition
 *
 * @param WP_Comment $comment The original comment object.
 */
function fast_send_email_and_slack_on_ticket_comment($comment)
{
    $ticket                   = get_post($comment->comment_post_ID);
    $sender                   = get_userdata($comment->user_id);
    $is_ticket_author_comment = intval($comment->user_id) === intval($ticket->post_author);
    $receiver                 = $is_ticket_author_comment ? fast_get_ticket_agent($ticket->ID) : get_userdata($ticket->post_author);
    $is_new_ticket            = intval($ticket->comment_count) === 1;

    /* Send Email */

    // Get option for email.
    $email_notify_enabled = $is_new_ticket ? fast_get_option('email_new_ticket') : ($is_ticket_author_comment ? fast_get_option('email_client_reply') : fast_get_option('email_agent_reply'));

    // If we have someone to send to and option is enabled.
    if (false !== $receiver && 'on' === $email_notify_enabled) {
        do_action('wpml_switch_language_for_email', $receiver->user_email);
        // Get subject & content.
        $email_subject = $is_new_ticket ? fast_get_option('ticket_new_email_subject') : fast_get_option('ticket_reply_email_subject');
        $email_content = $is_new_ticket ? fast_get_option('ticket_new_email_content') : fast_get_option('ticket_reply_email_content');
        // Send email.
        fast_send_ticket_email($sender, $receiver, $email_subject, $email_content, $ticket, $comment);
        do_action('wpml_restore_language_from_email');
    }

    /* Send Slack */

    // Get option for slack & hook.
    $slack_notify_enabled = $is_new_ticket ? fast_get_option('slack_new_ticket') : ($is_ticket_author_comment ? fast_get_option('slack_client_reply') : fast_get_option('slack_agent_reply'));
    $slack_webhook = fast_get_option('slack_webhook');

    // If we have a hook and slack is enabled.
    if ('on' === $slack_notify_enabled && !empty($slack_webhook)) {
        // Get content.
        $notice_content = $is_new_ticket ? fast_get_option('ticket_new_slack_content') : fast_get_option('ticket_reply_slack_content');
        $stripped_comment = (object) array(
            'comment_content' => wp_kses($comment->comment_content, array())
        );
        // Send slack notification.
        fast_send_slack_notification($sender, $receiver, $notice_content, $ticket, $stripped_comment);
    }
}
add_action('fast_action_post_comment', 'fast_send_email_and_slack_on_ticket_comment', 10, 1);

/**
 * Marks ticket if needs a reply upon comment addition
 *
 * @param WP_Comment $comment The original comment object.
 */
function fast_mark_ticket_if_needs_reply($comment)
{
    if (get_post_type($comment->comment_post_ID) !== 'fast_ticket') {
        return;
    }

    $ticket                   = get_post($comment->comment_post_ID);
    $is_ticket_author_comment = intval($comment->user_id) === intval($ticket->post_author);
    $agent_replied = $is_ticket_author_comment === true ? 'false' : 'true';
    update_post_meta($comment->comment_post_ID, '_agent_replied', $agent_replied);
}
add_action('fast_action_post_comment', 'fast_mark_ticket_if_needs_reply', 11, 1);

/**
 * Marks ticket as open on comment addition
 *
 * @param WP_Comment $comment The original comment object.
 */
function fast_open_ticket_on_comment($comment)
{
    if (get_post_type($comment->comment_post_ID) !== 'fast_ticket') {
        return;
    }

    $ticket_status = wp_get_post_terms($comment->comment_post_ID, 'fast_status');
    if ($ticket_status && !is_wp_error($ticket_status)) {
        $ticket_status = $ticket_status[0];
        if ('closed' === $ticket_status->slug) {
            wp_set_object_terms($comment->comment_post_ID, 'open', 'fast_status', false);
        }
    }
}
add_action('fast_action_post_comment', 'fast_open_ticket_on_comment', 11, 1);

/**
 * Writes all ticket comments to the excerpt so they may be searched.
 *
 * @param WP_Comment $comment The original comment object.
 */
function fast_update_ticket_excerpt_with_comments($comment)
{
    if (get_post_type($comment->comment_post_ID) !== 'fast_ticket') {
        return;
    }

    $ticket = get_post($comment->comment_post_ID);
    $ticket->post_content = fast_create_ticket_content_from_comments($ticket);
    wp_update_post($ticket);
}
add_action('fast_action_post_comment', 'fast_update_ticket_excerpt_with_comments', 13, 1);

/**
 * Adds current response time to ticket
 *
 * @param WP_Comment $comment The original comment object.
 */
function fast_add_response_time_to_ticket($comment)
{
    if (get_post_type($comment->comment_post_ID) !== 'fast_ticket') {
        return;
    }

    $ticket = get_post($comment->comment_post_ID);

    $response_time = fast_calculate_response_time($ticket, $comment);
    if (null !== $response_time) {
        $response_times = get_post_meta($ticket->ID, '_response_times', true);
        if (empty($response_times)) {
            $response_times = array();
        }
        $response_times[] = $response_time;
        update_post_meta($ticket->ID, '_response_times', $response_times);
    }
}
add_action('fast_action_post_comment', 'fast_add_response_time_to_ticket', 14, 1);

/*
 * rest_insert_comment
 */

/**
 * Make sure ticket content is updated when ticket is edited.
 *
 * @param $comment WP_Comment      $comment  Inserted or updated comment object.
 * @param $request WP_REST_Request $request  Request object.
 * @param $creating bool           $creating True when creating a comment, false when updating.
 */
function fast_update_ticket_excerpt_with_comments_on_edit($comment, $request, $creating)
{
    if (get_post_type($comment->comment_post_ID) !== 'fast_ticket') {
        return;
    }

    $ticket = get_post($comment->comment_post_ID);
    $ticket->post_content = fast_create_ticket_content_from_comments($ticket);
    wp_update_post($ticket);
}
add_action('rest_insert_comment', 'fast_update_ticket_excerpt_with_comments_on_edit', 10, 3);

/*
 * set_object_terms
 */

/**
 * Sets agent on new ticket creation.
 *
 * @param int    $object_id  Object ID.
 * @param array  $terms      An array of object terms.
 * @param array  $tt_ids     An array of term taxonomy IDs.
 * @param string $taxonomy   Taxonomy slug.
 * @param bool   $append     Whether to append new terms to the old terms.
 * @param array  $old_tt_ids Old array of term taxonomy IDs.
 */
function fast_set_agent_after_create_new_ticket($object_id, $terms, $tt_ids, $taxonomy, $append, $old_tt_ids)
{
    // Only run this code if we are changing taxonomy.
    // Of a ticket product or category.
    if ('fast_ticket' !== get_post_type($object_id)) {
        return;
    }

    if ('fast_product' !== $taxonomy &&
        'fast_category' !== $taxonomy) {
        return;
    }

    // Only run if this is a new ticket (0 comments).
    $ticket = get_post($object_id);
    if (0 !== (int) $ticket->comment_count) {
        return;
    }


    $agents = array();
    $tax_terms = wp_get_post_terms($object_id, $taxonomy);
    if (count($tax_terms) > 0) {
        switch ($taxonomy) {
            case 'fast_product':
                $agents = fast_get_taxonomy_metabox_value('product_user_ids', $tax_terms[0]->term_id);
                break;
            case 'fast_category':
                $agents = fast_get_taxonomy_metabox_value('category_user_ids', $tax_terms[0]->term_id);
                break;
        }

        if (!empty($agents)) {
            wp_set_object_terms($object_id, intval($agents[$tax_terms[0]->count % count($agents)]), 'fast_agent', false);
            return;
        }
    }
}
add_action('set_object_terms', 'fast_set_agent_after_create_new_ticket', 10, 6);

/**
 * Sets default agent if no product or category sent with ticket.
 *
 * @param WP_Post         $post     Inserted or updated post object.
 * @param WP_REST_Request $request  Request object.
 * @param bool            $creating True when creating a post, false when updating.
 */
function fast_set_default_agent_if_ticket_has_no_taxonomy($post, $request, $creating)
{
    if (true === $creating) {
        $params = $request->get_params();

        // Do we have an agent set from before?
        $agent_ids = wp_get_object_terms($post->ID, 'fast_agent', array('fields' => 'ids'));
        // If we have a ticket product or category or wc product set we do nothing.
        if (!empty($agent_ids)) {
            return;
        }

        // At this point no product or category has been set so we need to use default agent.
        $default_agents = is_array(fast_get_option('default_agent')) ? fast_get_option('default_agent') : array(fast_get_option('default_agent'));
        if (!empty($default_agents)) {
            $ticket_count = wp_count_posts('fast_ticket');
            $count = intval($ticket_count->publish + $ticket_count->private);
            wp_set_object_terms($post->ID, intval($default_agents[$count % count($default_agents)]), 'fast_agent', false);
            return;
        }

        // if no default agent set, assign a random agent.
        $agents = array();
        $agent_terms = get_terms(array(
            'taxonomy'   => 'fast_agent',
            'hide_empty' => false,
        ));
        foreach ($agent_terms as $agent) {
            $agents[] = $agent->term_id;
        }
        if (!empty($agents)) {
            wp_set_object_terms($post->ID, intval($agents[array_rand($agents)]), 'fast_agent', false);
        }
    }
}
add_action('rest_insert_fast_ticket', 'fast_set_default_agent_if_ticket_has_no_taxonomy', 10, 3);


/**
 * Sends an email to the agent/client upon agent assignement.
 *
 * @param int    $object_id  Object ID.
 * @param array  $terms      An array of object terms.
 * @param array  $tt_ids     An array of term taxonomy IDs.
 * @param string $taxonomy   Taxonomy slug.
 * @param bool   $append     Whether to append new terms to the old terms.
 * @param array  $old_tt_ids Old array of term taxonomy IDs.
 */
function fast_notify_after_agent_change($object_id, $terms, $tt_ids, $taxonomy, $append, $old_tt_ids)
{
    if ('fast_ticket' !== get_post_type($object_id) || 'fast_agent' !== $taxonomy) {
        return;
    }

    // Only do this from manual agent assigns.
    $ticket = get_post($object_id);
    if (0 === (int) $ticket->comment_count) {
        return;
    }

    $current_user_id = get_current_user_id();
    $sender          = get_userdata($current_user_id);
    $agent_id        = fast_get_taxonomy_metabox_value('agent_user_id', $terms[0]);
    $receiver        = get_userdata($agent_id);

    if (false !== $receiver && 'on' === fast_get_option('email_ticket_assignment') && (int) $current_user_id !== (int) $agent_id) {
        $email_subject   = fast_get_option('ticket_assign_email_subject');
        $email_content   = fast_get_option('ticket_assign_email_content');
        fast_send_ticket_email($sender, $receiver, $email_subject, $email_content, $ticket, false);
    }
    $slack_webhook = fast_get_option('slack_webhook');
    if (false !== $receiver && !empty($slack_webhook) && 'on' === fast_get_option('slack_ticket_assignment')) {
        $notice_content  = fast_get_option('ticket_assign_slack_content');
        fast_send_slack_notification($sender, $receiver, $notice_content, $ticket, false);
    }
}
add_action('set_object_terms', 'fast_notify_after_agent_change', 10, 6);

/*
 * rest_api_init
 */

/**
 * Registers api class routes.
 */
function fast_register_extra_api_routes()
{
    include FAST_PLUGIN_DIR . 'classes/class-api.php';
    $api = new FastAPI();
    $api->register_routes();
}
add_action('rest_api_init', 'fast_register_extra_api_routes');

/**
 * Adds post meta fields in the REST GET responses.
 */
function fast_add_post_meta_in_rest()
{
    /**
     * Taxonomy
     */
    register_rest_field('taxonomy', 'singular', array(
        'get_callback' => function ($taxonomy) {
            return $taxonomy['labels']->singular_name;
        },
        'schema' => array(
            'description' => esc_html__('Singular Label', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('taxonomy', 'terms', array(
        'get_callback' => function ($taxonomy) {
            $terms = get_terms(array(
                'taxonomy'   => $taxonomy['slug'],
                'hide_empty' => false,
            ));
            // Run terms through through translator.
            // Not best practice but, the least worst solution.
            foreach ($terms as $term) {
                $term->name = esc_html__($term->name, 'fast-plugin');
                $term->description = esc_html__($term->description, 'fast-plugin');
                // Some products have images & envato.
                if ('fast_product' === $taxonomy['slug']) {
                    $image = fast_get_taxonomy_metabox_value('product_image', $term->term_id);
                    if (!empty($image)) {
                        $term->image = $image;
                    }
                    $envato_product = fast_get_taxonomy_metabox_value('product_envato_item', $term->term_id);
                    if (!empty($envato_product)) {
                        $term->envato_product = $envato_product;
                    }
                }
            }
            return $terms;
        },
        'schema' => array(
            'description' => esc_html__('Terms', 'fast-plugin'),
            'type'        => 'array'
        ),
    ));

    /**
     * Ticket
     */

    register_rest_field('fast_ticket', 'author_email', array(
        'get_callback' => function ($post) {
            return get_the_author_meta('user_email', $post['author']);
        },
        'schema' => array(
            'description' => esc_html__('Author Email Address', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_ticket', 'author_username', array(
        'get_callback' => function ($post) {
            return get_the_author_meta('user_login', $post['author']);
        },
        'schema' => array(
            'description' => esc_html__('Author Username', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_ticket', 'author_name', array(
        'get_callback' => function ($post) {
            $first_name = get_the_author_meta('first_name', $post['author']);
            $last_name = get_the_author_meta('last_name', $post['author']);
            return trim("$first_name $last_name");
        },
        'schema' => array(
            'description' => esc_html__('Author Display Name', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_ticket', 'author_firstname', array(
        'get_callback' => function ($post) {
            return get_the_author_meta('first_name', $post['author']);
        },
        'schema' => array(
            'description' => esc_html__('Author First Name', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_ticket', 'author_avatar', array(
        'get_callback' => function ($post) {
            return get_avatar_url($post['author']);
        },
        'schema' => array(
            'description' => esc_html__('Author Avatar', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_ticket', 'customer_waiting', array(
        'get_callback' => function ($post) {
            return get_post_meta($post['id'], '_customer_waiting', true);
        },
        'schema' => array(
            'description' => esc_html__('Indicates if a user needs to reply to a ticket', 'fast-plugin'),
            'type'        => 'boolean'
        ),
    ));

    register_rest_field('fast_ticket', 'agent_replied', array(
        'get_callback' => function ($post) {
            return get_post_meta($post['id'], '_agent_replied', true);
        },
        'schema' => array(
            'description' => esc_html__('Indicates if a user needs to reply to a ticket', 'fast-plugin'),
            'type'        => 'boolean'
        ),
    ));

    register_rest_field('fast_ticket', 'is_open', array(
        'get_callback' => function ($post) {
            $status = get_the_terms($post['id'], 'fast_status');
            if (is_array($status) && count($status) > 0) {
                return 'open' === $status[0]->slug;
            }
            return false;
        },
        'schema' => array(
            'description' => esc_html__('Indicates if a ticket is open or not', 'fast-plugin'),
            'type'        => 'boolean'
        ),
    ));

    register_rest_field('fast_ticket', 'author_ticket_count', array(
        'get_callback' => function ($post) {
            return (int) get_user_meta($post['author'], '_ticket_count', true);
        },
        'schema' => array(
            'description' => esc_html__('Author\'s total number of created tickets', 'fast-plugin'),
            'type'        => 'number'
        ),
    ));

    register_rest_field('fast_ticket', 'capabilities', array(
        'get_callback' => function ($post) {
            $user_id = get_current_user_id();
            return array(
                'assign_status'  => current_user_can('assign_status', $post['id']),
                'assign_agent'   => current_user_can('assign_agent', $post['id']),
                // Assign product only if you are an agent.
                // But permission is needed for customer to assign product when they create a ticket.
                'assign_product' => fast_is_user_agent($user_id),
                'delete_ticket'  => current_user_can('delete_ticket', $post['id']),
            );
        }
    ));

    register_rest_field('fast_ticket', 'author_notes', array(
        'get_callback' => function ($post) {
            $user_id = get_current_user_id();
            if (fast_is_user_agent($user_id) && 'on' === fast_get_option('allow_customer_notes')) {
                return get_user_meta($post['author'], '_notes', true);
            }
        },
        'update_callback' => function ($value, $post) {
            $user_id = get_current_user_id();
            if (fast_is_user_agent($user_id) && 'on' === fast_get_option('allow_customer_notes')) {
                update_user_meta($post->post_author, '_notes', $value);
            }
        },
        'schema' => array(
            'description' => esc_html__('Notes relevant to the Author of the ticket', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_ticket', 'envato_purchase_id', array(
        'get_callback' => function ($post) {
            return (int) get_post_meta($post['id'], '_envato_purchase_id', true);
        },
        'update_callback' => function ($value, $post) {
            update_post_meta($post->ID, '_envato_purchase_id', $value);
        },
        'schema' => array(
            'description' => esc_html__('Envato purchase id.', 'fast-plugin'),
            'type'        => 'integer'
        ),
    ));

    register_rest_field('fast_ticket', 'purchase', array(
        'get_callback' => 'fast_attach_purchase_data_to_ticket',
        'schema' => array(
            'description' => esc_html__('Envato purchase details.', 'fast-plugin'),
            'type'        => 'object'
        ),
    ));

    register_rest_field('fast_ticket', 'woocommerce_order_id', array(
        'get_callback' => function ($post) {
            return (int) get_post_meta($post['id'], '_woocommerce_order_id', true);
        },
        'update_callback' => function ($value, $post) {
            update_post_meta($post->ID, '_woocommerce_order_id', $value);
        },
        'schema' => array(
            'description' => esc_html__('WooCommerce Order ID.', 'fast-plugin'),
            'type'        => 'integer'
        ),
    ));

    register_rest_field('fast_ticket', 'woocommerce_product_ids', array(
        'get_callback' => function ($post) {
            return get_post_meta($post['id'], '_woocommerce_product_ids', true);
        },
        'update_callback' => function ($value, $post) {
            update_post_meta($post->ID, '_woocommerce_product_ids', $value);
        },
        'schema' => array(
            'description' => esc_html__('WooCommerce Product IDs.', 'fast-plugin'),
            'type'        => 'array',
            'items'       => array(
                'type' => 'integer',
            )
        ),
    ));

    register_rest_field('fast_ticket', 'extra_fields', array(
        'get_callback' => function ($post) {
            return get_post_meta($post['id'], '_extra_fields', true);
        },
        'update_callback' => function ($value, $post) {
            update_post_meta($post->ID, '_extra_fields', apply_filters('fast_filter_save_extra_fields', $value, fast_get_extra_fields()));
        },
        'schema' => array(
            'description' => esc_html__('Extra ticket information.', 'fast-plugin'),
            'type'        => 'object'
        ),
    ));

    register_rest_field('fast_ticket', 'secure_fields', array(
        'get_callback' => function ($post) {
            $secure_fields = get_post_meta($post['id'], '_secure_fields', true);
            return empty($secure_fields) ? array() : $secure_fields;
        },
        'update_callback' => function ($value, $post) {
            // Get secure fields.
            $secure_fields = fast_get_secure_fields();
            // Run values through extra fields filter (for url tidy).
            $value = apply_filters('fast_filter_save_extra_fields', $value, $secure_fields);
            $data_to_store = array();
            $names_of_stored_data = array();
            // Create data for secure storage and secure field names for ticket info.
            foreach ($secure_fields->fields as $field) {
                if (isset($value[$field->id]) && !empty($value[$field->id])) {
                    $data_to_store[$field->id] = $value[$field->id];
                    $names_of_stored_data[] = $field->label;
                }
            }
            // Send data we want to store to filter.
            apply_filters('fast_filter_save_secure_fields', $data_to_store, $post, $secure_fields);
            // Update meta with list of fields we just stored.
            update_post_meta($post->ID, '_secure_fields', $names_of_stored_data);
        },
        'schema' => array(
            'description' => esc_html__('Secure ticket information.', 'fast-plugin'),
            'type'        => 'object'
        ),
    ));

    register_rest_field('fast_ticket', 'from_email', array(
        'get_callback' => function ($post) {
            $from_email = get_post_meta($post['id'], '_from_email', false);
            return !empty($from_email);
        },
        'schema' => array(
            'description' => esc_html__('Is this ticket from email.', 'fast-plugin'),
            'type'        => 'boolean'
        ),
    ));

    register_rest_field('fast_ticket', 'avg_response_time', array(
        'get_callback' => function ($post) {
            $avg_response_time = null;
            $response_times = get_post_meta($post['id'], '_response_times', false);
            if (!empty($response_times)) {
                $response_times = $response_times[0];
                $response_times = array_map('intval', $response_times);
                $total = array_sum($response_times);
                $avg_response_time = $total / count($response_times);
            }
            return $avg_response_time;
        },
        'schema' => array(
            'description' => esc_html__('Average response time', 'fast-plugin'),
            'type'        => 'int'
        ),
    ));

    /**
     * Comment
     */

    register_rest_field('comment', 'attachments', array(
        'get_callback' => function ($comment) {
            $attachment_meta = get_comment_meta($comment['id'], '_attachments', true);
            if (!empty($attachment_meta)) {
                if (is_array($attachment_meta)) {
                    return $attachment_meta;
                }
                // For versions older that 1.3
                // Some names might be badly encoded in a non-standard way. Change encoding notation and convert it to UTF-8.
                $attachments = html_entity_decode(preg_replace('/u+([0-9A-F]{4})/', "&#x\\1;", $attachment_meta), ENT_NOQUOTES, 'UTF-8');
                return json_decode($attachments, true);
            }
            return array();
        },
        'update_callback' => function ($value, $comment) {
             update_comment_meta($comment->comment_ID, '_attachments', $value);
        }
    ));

    register_rest_field('comment', 'from_email', array(
        'get_callback' => function ($comment) {
            $from_email = get_comment_meta($comment['id'], '_from_email', false);
            return !empty($from_email);
        },
        'schema' => array(
            'description' => esc_html__('Is this comment from email.', 'fast-plugin'),
            'type'        => 'boolean'
        ),
    ));

    register_rest_field('comment', 'capabilities', array(
        'get_callback' => function ($comment) {
            return apply_filters('fast_filter_get_comment_capabilities', array(
                'edit_comment'  => current_user_can('edit_comment', $comment['id']),
                'delete_comment' => current_user_can('edit_comment', $comment['id']),
            ), $comment);
        }
    ));

    register_rest_field('comment', 'agent_only', array(
        'get_callback' => function ($comment) {
            $from_email = get_comment_meta($comment['id'], '_agent_only', false);
            return !empty($from_email);
        },
        'update_callback' => function ($value, $comment) {
            if (true === $value) {
                update_comment_meta($comment->comment_ID, '_agent_only', 'true');
            } else {
                delete_comment_meta($comment->comment_ID, '_agent_only');
            }
        },
        'schema' => array(
            'description' => esc_html__('Is this an agent only comment.', 'fast-plugin'),
            'type'        => 'boolean'
        ),
    ));

    /**
     * Purchase
     */

    register_rest_field('fast_purchase', 'purchase_code', array(
        'get_callback' => function ($post) {
            return get_post_meta($post['id'], '_purchase_code', true);
        },
        'update_callback' => function ($value, $post) {
            update_post_meta($post->ID, '_purchase_code', $value);
        },
        'schema' => array(
            'description' => esc_html__('Envato purchase code.', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_purchase', 'item_name', array(
        'get_callback' => function ($post) {
            return get_post_meta($post['id'], '_item_name', true);
        },
        'schema' => array(
            'description' => esc_html__('Envato Item Name', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_purchase', 'item_id', array(
        'get_callback' => function ($post) {
            return get_post_meta($post['id'], '_item_id', true);
        },
        'schema' => array(
            'description' => esc_html__('Envato Item ID', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_purchase', 'created_at', array(
        'get_callback' => function ($post) {
            return get_post_meta($post['id'], '_created_at', true);
        },
        'schema' => array(
            'description' => esc_html__('Envato Created At', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_purchase', 'buyer', array(
        'get_callback' => function ($post) {
            return get_post_meta($post['id'], '_buyer', true);
        },
        'schema' => array(
            'description' => esc_html__('Envato Buyer', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_purchase', 'licence', array(
        'get_callback' => function ($post) {
            return get_post_meta($post['id'], '_licence', true);
        },
        'schema' => array(
            'description' => esc_html__('Envato Licence', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_purchase', 'supported_until', array(
        'get_callback' => function ($post) {
            return get_post_meta($post['id'], '_supported_until', true);
        },
        'schema' => array(
            'description' => esc_html__('Envato Supported Until', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_purchase', 'ticket_product', array(
        'get_callback' => function ($post) {
            return (int) get_post_meta($post['id'], '_ticket_product', true);
        },
        'schema' => array(
            'description' => esc_html__('Fast ticket product id', 'fast-plugin'),
            'type'        => 'integer'
        ),
    ));

    register_rest_field('fast_purchase', 'purchase_status', array(
        'get_callback' => function ($post) {
            return get_post_meta($post['id'], '_purchase_status', true);
        },
        'schema' => array(
            'description' => esc_html__('Envato Purchase Status', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));

    register_rest_field('fast_purchase', 'error_message', array(
        'get_callback' => function ($post) {
            return get_post_meta($post['id'], '_error_message', true);
        },
        'schema' => array(
            'description' => esc_html__('Envato Error Message', 'fast-plugin'),
            'type'        => 'string'
        ),
    ));
}
add_action('rest_api_init', 'fast_add_post_meta_in_rest');

/**
 * Add the necessary filter to tickets post type requests
 */
function fast_rest_api_add_ticket_filter()
{
    add_filter('rest_fast_ticket_query', 'fast_rest_api_filter_add_filter_param', 10, 2);
    add_filter('rest_fast_ticket_query', 'fast_rest_api_filter_add_search_params', 10, 2);

    // Hide users/ endpoints if option set.
    if ('on' === fast_get_option('remove_users_endpoints')) {
        add_filter('rest_endpoints', 'fast_remove_users_rest_endpoints', 10, 1);
    }
}
add_action('rest_api_init', 'fast_rest_api_add_ticket_filter');

/**
 * Adds a filter options as response header.
 *
 * @param boolean $served Indicates if the request has already been served.
 * @param object  $response Response Object.
 * @param object  $request Request Object.
 */
function fast_rest_api_add_filters_in_headers($served, $response, $request)
{
    if ($request->get_method() === 'GET' && $request->get_route() === '/wp/v2/tickets' && null !== $request->get_param('filter')) {
        $user_id        = get_current_user_id();
        $user_type = fast_get_user_type($user_id);
        $filters = apply_filters('fast_filter_get_' . $user_type . '_filters', $user_id);
        $request_filter = $request->get_param('filter');
        $pages = $request->get_param('pages');
        $pages = empty($pages) ? 1 : $pages;

        $returned_filters = array();
        foreach ($filters as $key => $filter) {
            $returned_filter = array();
            $returned_filter['name'] = $filter['name'];
            $returned_filter['id'] = $filter['id'];
            $returned_filter['count'] = isset($filter['count']) ? $filter['count'] : fast_get_open_count($filter['args']);
            $returned_filter['needReplyCount'] = fast_get_need_reply_count($filter['args'], $user_type !== 'customer');
            if ($request_filter === $filter['id']) {
                $returned_filter['active'] = true;
                $returned_filter['pages'] = $pages;
            }
            $returned_filters[] = $returned_filter;
        }
        header('ticket-filters:' . wp_json_encode($returned_filters));
    }
    return $served;
}

/**
 * Adds a filter that modifies response headers.
 */
function fast_rest_api_add_headers()
{
    add_filter('rest_pre_serve_request', 'fast_rest_api_add_filters_in_headers', 10, 3);
}
add_action('rest_api_init', 'fast_rest_api_add_headers');

/*
 * fast_action_demo_cleanup
 */

/**
 * Removes all comments and tickets created after a date (for demo site)
 *
 * @param string $date Date for searching after.
 */
function fast_demo_cleanup_content($date = '')
{
    $date_query = array(
        'after' => $date,
        'inclusive' => true,
    );
    $new_comments = get_comments(array(
        'date_query' => $date_query
    ));
    foreach ($new_comments as $comment) {
        wp_delete_comment($comment->comment_ID, true);
    }
    $new_tickets = get_posts(array(
        'post_type'      => 'fast_ticket',
        'posts_per_page' => -1,
        'post_status'    => 'publish,private',
        'date_query'     => $date_query
    ));

    foreach ($new_tickets as $ticket) {
        wp_delete_post($ticket->ID, true);
    }
}
add_action('fast_action_demo_cleanup', 'fast_demo_cleanup_content', 10, 1);

/*
 * do_feed_rss2
 * do_feed_atom
 */

/**
 * Removes all comments from feeds
 *
 * @param boolean $for_comments If comment feed.
 */
function fast_remove_comment_feeds($for_comments)
{
    if ($for_comments) {
        remove_action('do_feed_rss2', 'do_feed_rss2', 10, 1);
        remove_action('do_feed_atom', 'do_feed_atom', 10, 1);
    }
}
add_action('do_feed_rss2', 'fast_remove_comment_feeds', 9, 1);
add_action('do_feed_atom', 'fast_remove_comment_feeds', 9, 1);

/**
 * Loads the email pipe class because we need to check for emails.
 */
function fast_check_email_pipe()
{
    // Load IMAP reader.
    require FAST_PLUGIN_DIR . 'lib/imap/Mailbox.php';
    require FAST_PLUGIN_DIR . 'lib/imap/IncomingMail.php';
    // Load Email Parser.
    require FAST_PLUGIN_DIR . 'lib/email-reply-parser/Parser/EmailParser.php';
    require FAST_PLUGIN_DIR . 'lib/email-reply-parser/Parser/FragmentDTO.php';
    require FAST_PLUGIN_DIR . 'lib/email-reply-parser/Email.php';
    require FAST_PLUGIN_DIR . 'lib/email-reply-parser/EmailReplyParser.php';
    require FAST_PLUGIN_DIR . 'lib/email-reply-parser/Fragment.php';
    // Load our email pipe class.
    require FAST_PLUGIN_DIR . 'classes/class-email-pipe.php';
    $email_pipe = new FastEmailPipe();
    // Check emails.
    $email_pipe->check_emails();
}
add_action('fast_action_email_pipe', 'fast_check_email_pipe', 10, 0);


/**
 * Queries and closes old tickets
 */
function fast_close_old_tickets()
{
    $current_time = time();
    $report = new stdClass();
    $report->date = $current_time;
    $report->closed_tickets = 0;

    $ticket_maintenance_timespan = fast_get_option('ticket_maintenance_timespan');
    $before = get_date_from_gmt(date('Y-m-d H:i:s', $current_time - ( (int) $ticket_maintenance_timespan * 24 * 3600)), 'F j, Y H:i:s');

    $old_tickets = get_posts(array(
        'post_type' => 'fast_ticket',
        'fields' => 'ids',
        'posts_per_page' => -1,
        'post_status' => 'any',
        'tax_query' => array(
            array(
                'taxonomy' => 'fast_status',
                'field'    => 'slug',
                'terms'    => 'open'
            )
        ),
        'date_query' => array(
            array(
                'column' => 'post_modified_gmt',
                'before' => $before,
                'inclusive' => true,
            )
        )
    ));

    foreach ($old_tickets as $ticket) {
        wp_set_object_terms($ticket, 'closed', 'fast_status', false);
    }

    $report->closed_tickets = count($old_tickets);
    update_option('fast_scheduled_maintenance_report', $report);
}
add_action('fast_action_scheduled_maintenance', 'fast_close_old_tickets');

/**
 * Adds the customer role to all new registered users.
 *
 * @param int $user_id User id of new user.
 */
function fast_add_customer_role_to_all_registrations($user_id)
{
    if ('on' === fast_get_option('add_customer_role_to_all_users')) {
        $user = get_user_by('id', $user_id);
        $user->add_role('fast_customer');
    }
}
add_action('user_register', 'fast_add_customer_role_to_all_registrations', 10, 1);

/**
 * Styles iframe used for embedding support page shortcode.
 */
function fast_add_frontend_shortcodes_css()
{
    wp_register_style('fast-style-frontend_shortcodes_css', FAST_PLUGIN_URI . 'assets/css/frontend.shortcodes.css');
}
add_action('wp_enqueue_scripts', 'fast_add_frontend_shortcodes_css', 10, 0);
