<?php
/**
 * Admin only filters
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Gets all users that are valid ticket agents
 *
 * @param array $values Array of user id => username.
 */
function fast_agent_taxonomy_user_select_option_values($values)
{
    $values = array();

    $valid_users = get_users(array(
        'role__in' => array(
            'fast_agent',
            'fast_manager',
            'administrator'
        )
    ));

    foreach ($valid_users as $user) {
        $values[$user->ID] = $user->display_name;
    }

    return $values;
}
add_filter('fast_filter_taxonomy_user_select_option_values', 'fast_agent_taxonomy_user_select_option_values');

/**
 * Gets all ticket agents.
 *
 * @param array $values Array of agent id => Agent Name.
 */
function fast_ticket_agent_ids_values($values)
{
    $values = array();
    $agent_terms = get_terms(array(
        'taxonomy'   => 'fast_agent',
        'hide_empty' => false,
    ));
    foreach ($agent_terms as $agent) {
        $values[$agent->term_id] = $agent->name;
    }
    return $values;
}
add_filter('fast_filter_ticket_agent_ids_values', 'fast_ticket_agent_ids_values');

/**
 * Gets all ticket products.
 *
 * @param array $values Array of product id => Product Name.
 */
function fast_ticket_product_ids_values($values)
{
    $values = array();
    $agent_terms = get_terms(array(
        'taxonomy'   => 'fast_product',
        'hide_empty' => false,
    ));
    foreach ($agent_terms as $product) {
        $values[$product->term_id] = $product->name;
    }
    return $values;
}
add_filter('fast_filter_ticket_product_ids_values', 'fast_ticket_product_ids_values');

/**
 * Gets all ticket categories.
 *
 * @param array $values Array of category id => Category Name.
 */
function fast_ticket_category_ids_values($values)
{
    $values = array();
    $agent_terms = get_terms(array(
        'taxonomy'   => 'fast_category',
        'hide_empty' => false,
    ));
    foreach ($agent_terms as $category) {
        $values[$category->term_id] = $category->name;
    }
    return $values;
}
add_filter('fast_filter_ticket_category_ids_values', 'fast_ticket_category_ids_values');

/**
 * Adds marketplace item select box to options if envato_products is on
 *
 * @param array $options Plugin meabox options.
 */
function fast_add_envato_product_field_metabox($options)
{
    if ('on' === fast_get_option('envato_products') && isset($options['product_fields'])) {
        array_push($options['product_fields']['fields'], array(
            'name'             => esc_html__('Envato Marketplace Item', 'fast-plugin'),
            'id'               => 'product_envato_item',
            'desc'             => esc_html__('Link this ticket product to an Envato marketplace item.', 'fast-plugin'),
            'type'             => 'select',
            'empty_label'      => esc_html__('No Item Selected', 'fast-plugin'),
            'filter'           => 'fast_filter_product_marketplace_items',
            'select2'          => true,
            'default'          => 0,
            'attributes'       => array(
                'placeholder' => esc_html__('Select a marketplace item', 'fast-plugin'),
                'class'       => 'fast-select2-option',
            )
        ));
    }
    return $options;
}
add_filter('fast_filter_metabox_options', 'fast_add_envato_product_field_metabox', 1, 11);

/**
 * Gets all Envato Marketplace items
 *
 * @param array $items Array of items id => name.
 */
function fast_product_marketplace_items($items)
{
    return get_option('fast-pluginenvato_product_list', array());
}
add_filter('fast_filter_product_marketplace_items', 'fast_product_marketplace_items', 1, 11);

/**
 * Menus that are allowed to be shown in the admin menu when logged in as a customer
 *
 * @param array $node_ids IDs of menu items to allow.
 */
function fast_customer_allowed_admin_menu_nodes($node_ids)
{
    return array(
        'top-secondary',
        'logout',
        'user-info',
        'user-actions',
        'my-account',
        'edit-profile',
        'fast-tickets'
    );
}
add_filter('fast_filter_customer_allowed_admin_menu_nodes', 'fast_customer_allowed_admin_menu_nodes');

/**
 * Adds custom columns in Agents taxonomy admin table.
 *
 * @param array $columns Default columns displayed.
 */
function fast_custom_agent_columns($columns)
{
    $columns['_user_id'] = esc_html__('Wordpress User');
    return $columns;
}
add_filter('manage_edit-fast_agent_columns', 'fast_custom_agent_columns');

/**
 * Adds content for custom columns in Agents taxonomy admin table.
 *
 * @param string $content Default content for displayed column.
 * @param string $column_name Column name.
 * @param int    $term_id ID of the term displayed.
 */
function fast_custom_agent_columns_content( $content, $column_name, $term_id )
{
    if ('_user_id' === $column_name) {
        $agent_id = fast_get_taxonomy_metabox_value('agent_user_id', $term_id);
        $agent = get_userdata($agent_id);
        $content = $agent->user_nicename;
    }
    return $content;
}
add_filter('manage_fast_agent_custom_column', 'fast_custom_agent_columns_content', 10, 3);

/**
 * Adds custom columns in Products taxonomy admin table.
 *
 * @param array $columns Default columns displayed.
 */
function fast_custom_product_columns($columns)
{
    $columns['_agent_ids'] = esc_html__('Product Agents');
    return $columns;
}
add_filter('manage_edit-fast_product_columns', 'fast_custom_product_columns');

/**
 * Adds content for custom columns in Products taxonomy admin table.
 *
 * @param string $content Default content for displayed column.
 * @param string $column_name Column name.
 * @param int    $term_id ID of the term displayed.
 */
function fast_custom_product_columns_content( $content, $column_name, $term_id )
{
    if ('_agent_ids' === $column_name) {
        $agents = array();
        $agent_ids = fast_get_taxonomy_metabox_value('product_user_ids', $term_id);
        foreach ($agent_ids as $key => $agent_id) {
            $agent = get_term_by('id', $agent_id, 'fast_agent');
            if (false !== $agent) {
                $agents[] = $agent->name;
            }
        }
        $content = implode(', ', $agents);
    }
    return $content;
}
add_filter('manage_fast_product_custom_column', 'fast_custom_product_columns_content', 10, 3);

/**
 * Adds custom columns in Categories taxonomy admin table.
 *
 * @param array $columns Default columns displayed.
 */
function fast_custom_category_columns($columns)
{
    $columns['_agent_ids'] = esc_html__('Category Agents');
    return $columns;
}
add_filter('manage_edit-fast_category_columns', 'fast_custom_category_columns');

/**
 * Adds content for custom columns in Categories taxonomy admin table.
 *
 * @param string $content Default content for displayed column.
 * @param string $column_name Column name.
 * @param int    $term_id ID of the term displayed.
 */
function fast_custom_category_columns_content( $content, $column_name, $term_id )
{
    if ('_agent_ids' === $column_name) {
        $agents = array();
        $agent_ids = fast_get_taxonomy_metabox_value('category_user_ids', $term_id);
        foreach ($agent_ids as $key => $agent_id) {
            $agent = get_term_by('id', $agent_id, 'fast_agent');
            if (false !== $agent) {
                $agents[] = $agent->name;
            }
        }
        $content = implode(', ', $agents);
    }
    return $content;
}
add_filter('manage_fast_category_custom_column', 'fast_custom_category_columns_content', 10, 3);

/*
 * admin_footer_text
 */

/**
 * Changes the admin footer text in plugin pages.
 *
 * @param string $footer_text Default wordpress admin footer text.
 *
 * @return string Modified footer text.
 */
function fast_admin_footer_text($footer_text)
{
    $plugin_pages = apply_filters('fast_filter_options', array());
    $page_names = array();
    foreach ($plugin_pages as $page) {
        $page_names[] = $page['slug'];
    }

    if (isset($_GET['page']) && in_array(wp_unslash($_GET['page']), $page_names)) {
        $footer_text = esc_html__('Fast Plugin - Version 1.15.4', 'fast-plugin');
    }
    return $footer_text;
}
add_filter('admin_footer_text', 'fast_admin_footer_text', 1);

/**
 * Filter for email pipe next schedule on option page.
 *
 * @param string $text Original text.
 */
function fast_get_next_email_pipe_run($text)
{
    $timestamp = wp_next_scheduled('fast_action_email_pipe');
    if (false === $timestamp) {
        $text = esc_html__('No Email Check Scheduled', 'fast-plugin');
    } else {
        $text = '<p>' . esc_html__('Current System Time', 'fast-plugin') . ': ' . get_date_from_gmt(date('Y-m-d H:i:s', time()), 'F j, Y H:i:s') . '</p>';
        $text .= '<p>' . esc_html__('Next Email Check At', 'fast-plugin') . ': ' . get_date_from_gmt(date('Y-m-d H:i:s', $timestamp), 'F j, Y H:i:s') . '</p>';
    }
    return $text;
}
add_filter('fast_filter_email_pipe_next_run', 'fast_get_next_email_pipe_run', 10, 1);

/**
 * Filter for email pipe results on option page.
 *
 * @param string $text Original text.
 */
function fast_get_next_email_pipe_results($text)
{
    $report = get_option('fast_email_pipe_report', false);
    if (false !== $report) {
        $text .= '<p><strong>Last Run Report</strong></p>';
        $text .= '<p><strong>Date: </strong>' . get_date_from_gmt(date('Y-m-d H:i:s', $report->date), 'Y-m-d H:i:s') . '</p>';
        $text .= '<p><strong>Parsed Email Count: </strong>' . $report->email_count . '</p>';
        $text .= '<p><strong>Replies Created: </strong>' . $report->replies_created . '</p>';
        $text .= '<p><strong>New Tickets Created: </strong>' . $report->new_tickets_created . '</p>';
        if (count($report->errors) > 0) {
            $text .= '<p><strong>Errors: </strong></p>';
            foreach ($report->errors as $error) {
                $text .= '<p>' . $error . '</p>';
            }
        }
    } else {
        return esc_html__('No Email Check Has Run', 'fast-plugin');
    }
    return $text;
}
add_filter('fast_filter_email_pipe_results', 'fast_get_next_email_pipe_results', 10, 1);

/**
 * Filter for email pipe schedule select box.
 */
function fast_get_cron_schedules()
{
    $schedules = wp_get_schedules();
    $spans = array();
    foreach ($schedules as $value => $span) {
        $spans[$value] = $span['display'];
    }
    return $spans;
}
add_filter('fast_filter_cron_time_spans', 'fast_get_cron_schedules', 10, 0);


/**
 * Filters stored password so we don't store in plaintext.
 *
 * @param string $password Pass to encrypt.
 * @return string Encrypted password.
 **/
function fast_sanitize_password($password)
{
    // Only encrypt if password has changed.
    if (fast_get_option('emailp_password') !== $password && !empty($password)) {
        $password = fast_store_sensitive_e($password, 'e');
    }
    return $password;
}
add_filter('fast_filter_field_sanitize_password', 'fast_sanitize_password', 10, 1);

/**
 * Filter for auto close next schedule on option page.
 *
 * @param string $text Original text.
 */
function fast_get_next_scheduled_maintenance_run($text)
{
    $timestamp = wp_next_scheduled('fast_action_scheduled_maintenance');
    if (false === $timestamp) {
        $text = esc_html__('No Ticket Auto - Close Scheduled', 'fast-plugin');
    } else {
        $ticket_maintenance_timespan = fast_get_option('ticket_maintenance_timespan');
        $before = $timestamp - ( (int) $ticket_maintenance_timespan * 24 * 3600);
        $text = '<p>' . esc_html__('Auto Close will run again on', 'fast-plugin') . ' ' . get_date_from_gmt(date('Y-m-d H:i:s', $timestamp), 'F j, Y H:i:s') . '</p>';
        $text .= '<p>' . esc_html__('All tickets not modified since', 'fast-plugin') . ' ' . get_date_from_gmt(date('Y-m-d H:i:s', $before), 'F j, Y H:i:s') . ' ' . esc_html__('will close.', 'fast-plugin') . '</p>';
    }
    return $text;
}
add_filter('fast_filter_scheduled_maintenance_next_run', 'fast_get_next_scheduled_maintenance_run', 10, 1);

/**
 * Filter for auto close results on option page.
 *
 * @param string $text Original text.
 */
function fast_get_next_scheduled_maintenance_results($text)
{
    $report = get_option('fast_scheduled_maintenance_report', false);
    if (false !== $report) {
        $text .= '<p><strong>Last Run Report</strong></p>';
        $text .= '<p><strong>Date: </strong>' . get_date_from_gmt(date('Y-m-d H:i:s', $report->date), 'Y-m-d H:i:s') . '</p>';
        $text .= '<p><strong>Tickets Closed: </strong>' . $report->closed_tickets . '</p>';
    } else {
         return esc_html__('No Ticket Auto - Close Has Run', 'fast-plugin');
    }
    return $text;
}
add_filter('fast_filter_scheduled_maintenance_results', 'fast_get_next_scheduled_maintenance_results', 10, 1);

/**
 * Add a post display state for the support page in the admin page list.
 *
 * @param array   $post_states An array of post display states.
 * @param WP_Post $post        The current post object.
 */
function fast_display_support_page_label($post_states, $post)
{
    $support_page_id = (int) fast_get_option('support_page');
    if ($support_page_id === $post->ID) {
        $post_states[] = __('Support Page', 'fast-plugin');
    }
    return $post_states;
}
add_filter('display_post_states', 'fast_display_support_page_label', 10, 2);

/*
 * fast_filter_stack_scss
 */

/**
 * Adds skin scss to stack save.
 *
 * @param string $scss Scss code.
 *
 * @return string new scss code.
 */
function fast_theme_stack_scss($scss)
{
    return '@import "skin.scss";';
}
add_filter('fast_filter_stack_scss', 'fast_theme_stack_scss', 10, 1);

/**
 * Grabs the theme stack options from the
 *
 * @return array array of metabox options.
 */
function fast_register_stack_options()
{
    return array(
        array(
            'id' => 'fast-fonts',
            'title' => esc_html__('FAST Fonts', 'fast-plugin'),
            'fields' => array(
                array(
                    'name'    => esc_html__('Font', 'fast-plugin-admin'),
                    'id'      => 'fast-font',
                    'desc'    => esc_html__('The standard font family to use for support system text.', 'fast-plugin-admin'),
                    'type'    => 'font',
                    'default' => '{ "family": "Roboto", "weights": ["300","regular", "italic" ,"700"], "languages": ["latin"] }',
                    'filter'  => 'fast_filter_stack_filter_font_family'
                ),
            )
        ),
        array(
            'id' => 'fast-theme',
            'title' => esc_html__('FAST Theme', 'fast-plugin'),
            'fields' => array(
                array(
                    'name'    => esc_html__('Theme Background Shade', 'fast-plugin-admin'),
                    'id'      => 'fast-theme-shade',
                    'desc'    => esc_html__('The shade of background theme you will be creating (light=white background with black text, dark=black background with white text).', 'fast-plugin-admin'),
                    'type'    => 'select',
                    'default' => 'light',
                    'options' => array(
                        'light' => esc_html__('Light', 'fast-plugin-admin'),
                        'dark' => esc_html__('Dark', 'fast-plugin-admin'),
                    )
                ),
            )
        ),
        array(
            'id' => 'fast-primary-colors',
            'title' => esc_html__('FAST Primary Colors', 'fast-plugin'),
            'fields' => array(
                array(
                    'name'    => esc_html__('Primary Color (main buttons & toolbars)', 'fast-plugin'),
                    'id'      => 'app-primary-color',
                    'desc'    => esc_html__('Color that is most widely used across all screens and components.', 'fast-plugin'),
                    'default' => '#03a9f4',
                    'type'    => 'color',
                    'format'  => 'hex'
                ),
                array(
                    'name'    => esc_html__('Primary Text Color (main buttons & toolbars)', 'fast-plugin'),
                    'id'      => 'app-primary-color-text',
                    'desc'    => esc_html__('Color of text to contrast primary color.', 'fast-plugin'),
                    'default' => 'rgb(255, 255, 255)',
                    'type'    => 'color',
                    'format'  => 'rgba'
                ),
                array(
                    'name'    => esc_html__('Primary Lighter Color', 'fast-plugin'),
                    'id'      => 'app-primary-lighter',
                    'desc'    => esc_html__('Lighter color that is most widely used across all screens and components.', 'fast-plugin'),
                    'default' => '#b3e5fc',
                    'type'    => 'color',
                    'format'  => 'hex'
                ),
                array(
                    'name'    => esc_html__('Primary Lighter Text Color', 'fast-plugin'),
                    'id'      => 'app-primary-lighter-text',
                    'desc'    => esc_html__('Color of text to contrast light primary color.', 'fast-plugin'),
                    'default' => 'rgba(0, 0, 0, 0.87)',
                    'type'    => 'color',
                    'format'  => 'rgba'
                ),
                array(
                    'name'    => esc_html__('Primary Darker Color (Ticket Sidebar)', 'fast-plugin'),
                    'id'      => 'app-primary-darker',
                    'desc'    => esc_html__('Darker color that is most widely used across all screens and components.', 'fast-plugin'),
                    'default' => '#0288d1',
                    'type'    => 'color',
                    'format'  => 'hex'
                ),
                array(
                    'name'    => esc_html__('Primary Darker Text Color (Ticket Sidebar)', 'fast-plugin'),
                    'id'      => 'app-primary-darker-text',
                    'desc'    => esc_html__('Color of text to contrast darker primary color.', 'fast-plugin'),
                    'default' => 'rgb(255, 255, 255)',
                    'type'    => 'color',
                    'format'  => 'rgba'
                ),
            )
        ),
        array(
            'id' => 'fast-accent-colors',
            'title' => esc_html__('FAST Accent Colors', 'fast-plugin'),
            'fields' => array(
                array(
                    'name'    => esc_html__('Accent Color (Reply button, notice icons)', 'fast-plugin'),
                    'id'      => 'app-accent-color',
                    'desc'    => esc_html__('Colors used for the floating action button and interactive elements.', 'fast-plugin'),
                    'default' => '#ffc107',
                    'type'    => 'color',
                    'format'  => 'hex'
                ),
                array(
                    'name'    => esc_html__('Accent Text Color (Reply button, notice icons)', 'fast-plugin'),
                    'id'      => 'app-accent-color-text',
                    'desc'    => esc_html__('Color of text to contrast default accent color.', 'fast-plugin'),
                    'default' => 'rgba(0, 0, 0, 0.87)',
                    'type'    => 'color',
                    'format'  => 'rgba'
                ),
                array(
                    'name'    => esc_html__('Accent Lighter Color', 'fast-plugin'),
                    'id'      => 'app-accent-lighter',
                    'desc'    => esc_html__('Colors used for the floating action button and interactive elements.', 'fast-plugin'),
                    'default' => '#03a9f4',
                    'type'    => 'color',
                    'format'  => 'hex'
                ),
                array(
                    'name'    => esc_html__('Accent Lighter Text Color', 'fast-plugin'),
                    'id'      => 'app-accent-lighter-text',
                    'desc'    => esc_html__('Color of text to contrast lighter accent color.', 'fast-plugin'),
                    'default' => 'rgba(0, 0, 0, 0.87)',
                    'type'    => 'color',
                    'format'  => 'rgba'
                ),
                array(
                    'name'    => esc_html__('Accent Darker Color', 'fast-plugin'),
                    'id'      => 'app-accent-darker',
                    'desc'    => esc_html__('Colors used for the floating action button and interactive elements.', 'fast-plugin'),
                    'default' => '#ffa000',
                    'type'    => 'color',
                    'format'  => 'hex'
                ),
                array(
                    'name'    => esc_html__('Accent Darker Text Color', 'fast-plugin'),
                    'id'      => 'app-accent-darker-text',
                    'desc'    => esc_html__('Color of text to contrast darker accent color.', 'fast-plugin'),
                    'default' => 'rgba(0, 0, 0, 0.87)',
                    'type'    => 'color',
                    'format'  => 'rgba'
                ),
            )
        ),
        array(
            'id' => 'fast-warn-colors',
            'title' => esc_html__('FAST Warn Colors', 'fast-plugin'),
            'fields' => array(
                array(
                    'name'    => esc_html__('Warn Color (Cancel button and form errors) ', 'fast-plugin'),
                    'id'      => 'app-warn-color',
                    'desc'    => esc_html__('Colors used to convey error state.', 'fast-plugin'),
                    'default' => '#f44336',
                    'type'    => 'color',
                    'format'  => 'hex'
                ),
                array(
                    'name'    => esc_html__('Warn Text Color', 'fast-plugin'),
                    'id'      => 'app-warn-color-text',
                    'desc'    => esc_html__('Color of text to contrast default warn color.', 'fast-plugin'),
                    'default' => 'rgb(255, 255, 255)',
                    'type'    => 'color',
                    'format'  => 'rgba'
                ),
                array(
                    'name'    => esc_html__('Warn Lighter Color', 'fast-plugin'),
                    'id'      => 'app-warn-lighter',
                    'desc'    => esc_html__('Colors used to convey error state.', 'fast-plugin'),
                    'default' => '#ffcdd2',
                    'type'    => 'color',
                    'format'  => 'hex'
                ),
                array(
                    'name'    => esc_html__('Warn Lighter Text Color', 'fast-plugin'),
                    'id'      => 'app-warn-lighter-text',
                    'desc'    => esc_html__('Color of text to contrast lighter warn color.', 'fast-plugin'),
                    'default' => 'rgba(0, 0, 0, 0.87)',
                    'type'    => 'color',
                    'format'  => 'rgba'
                ),
                array(
                    'name'    => esc_html__('Warn Darker Color', 'fast-plugin'),
                    'id'      => 'app-warn-darker',
                    'desc'    => esc_html__('Colors used to convey error state.', 'fast-plugin'),
                    'default' => '#d32f2f',
                    'type'    => 'color',
                    'format'  => 'hex'
                ),
                array(
                    'name'    => esc_html__('Warn Darker Text Color', 'fast-plugin'),
                    'id'      => 'app-warn-darker-text',
                    'desc'    => esc_html__('Color of text to contrast darker warn color.', 'fast-plugin'),
                    'default' => 'rgb(255, 255, 255)',
                    'type'    => 'color',
                    'format'  => 'rgba'
                ),
            )
        )
    );
}
add_filter('fast_filter_stack_options', 'fast_register_stack_options', 10, 1);

/**
 * Returns the plugin colors.
 *
 * @param array $colors Array of colors.
 */
function fast_get_app_colors($colors = array())
{
    // Ready made colors.
    $select_options = array(
        'Ready Made Skins' => array(
            'blue-grey_amber'         => esc_html('Blue Grey & Amber', 'fast-plugin'),
            'purple_orange'           => esc_html('Purple & Orange', 'fast-plugin'),
            'cyan_yellow'             => esc_html('Cyan & Yellow', 'fast-plugin'),
            'red_cyan'                => esc_html('Red & Cyan', 'fast-plugin'),
            'brown_light-blue'        => esc_html('Brown & Light Blue', 'fast-plugin'),
            'lime_pink'               => esc_html('Lime & Pink', 'fast-plugin'),
            'deep-orange_teal'        => esc_html('Deep Orange & Teal', 'fast-plugin'),
            'light-blue_purple'       => esc_html('Light Blue & Purple', 'fast-plugin'),
            'light-green_deep-orange' => esc_html('Light Green & Deep Orange', 'fast-plugin'),
            'indigo_pink'             => esc_html('Indigo & Pink', 'fast-plugin'),
            'black_grey'              => esc_html('Black & Grey', 'fast-plugin'),
            'dark_blue_cyan'          => esc_html('(Dark) Blue & Cyan', 'fast-plugin')
        )
    );
    // Check if we have any custom stacks.
    $custom_stacks = get_posts(array(
        'posts_per_page' => -1,
        'post_type'      => 'fast-plugin-stack',
        'orderby'        => 'title',
        'order'          => 'DESC'
    ));
    if (count($custom_stacks) > 0) {
        $custom_stack_options = array();
        foreach ($custom_stacks as $post) {
            $custom_stack_options[$post->ID] = $post->post_title;
        }
        $select_options['Custom Skins'] = $custom_stack_options;
    }

    return $select_options;
}
add_filter('fast_filter_app_colors', 'fast_get_app_colors', 10, 1);


/**
 * Returns the args for reigster_post_type.
 *
 * @param array $args Array arguments for reigster_post_type.
 */
function fast_stack_register_post_type_args($args)
{
    $args['labels'] = array(
        'name'               => esc_html__('Custom Skins', 'fast-plugin'),
        'singular_name'      => esc_html__('Skin', 'fast-plugin'),
        'add_new'            => esc_html__('Create New Skin', 'fast-plugin'),
        'add_new_item'       => esc_html__('Create New Skin', 'fast-plugin'),
        'edit_item'          => esc_html__('Edit Skin', 'fast-plugin'),
        'new_item'           => esc_html__('New Skin', 'fast-plugin'),
        'all_items'          => esc_html__('Custom Skins', 'fast-plugin'),
        'view_item'          => esc_html__('View Skin', 'fast-plugin'),
        'search_items'       => esc_html__('Search Skin', 'fast-plugin'),
        'not_found'          => esc_html__('No Skin found', 'fast-plugin'),
        'not_found_in_trash' => esc_html__('No Skin found in Trash', 'fast-plugin')
    );

    $args['show_in_menu'] = 'fast-plugin-general-options';
    return $args;
}
add_filter('fast_filter_stack_register_post_type', 'fast_stack_register_post_type_args', 10, 1);

/**
 * Filters stack variable to return font family.
 *
 * @param  array $value Stack font values.
 * @return string Font Family Name.
 */
function fast_filter_stack_font_family_value($value)
{
    $font = json_decode($value, true);
    return '"' . $font['family'] . '"';
}
add_filter('fast_filter_stack_filter_font_family', 'fast_filter_stack_font_family_value', 10, 1);

/**
 * Adds material icons to font families of google.
 *
 * @param  array $families Font families to load.
 * @return array Font Family Names.
 */
function fast_add_material_icons_to_fontstack($families)
{
    $families[] = 'Material+Icons';
    return $families;
}
add_filter('fast_filter_stack-google-families', 'fast_add_material_icons_to_fontstack', 10, 1);

/**
 * Returns the links we add to WordPress admin comments page.
 *
 * @param array $links List of links to add.
 */
function fast_comment_list_page_links($links)
{
    $links[] = array(
        'post_type' => 'post',
        'name' => esc_html__('Posts', 'fast-plugin')
    );

    $links[] = array(
        'post_type' => 'page',
        'name' => esc_html__('Pages', 'fast-plugin')
    );

    $links[] = array(
        'post_type' => 'fast_ticket',
        'name' => esc_html__('Ticket Comments', 'fast-plugin')
    );
    return $links;
}
add_filter('fast_filter_comment_page_links', 'fast_comment_list_page_links', 10, 1);


/**
 * Adds to WordPress admin comments page.
 *
 * @param array $status_links Links to add to top.
 */
function fast_comment_page_links( $status_links )
{
    // Get list of filters to add.
    $fast_pages = apply_filters('fast_filter_comment_page_links', array());

    foreach($fast_pages as $page_link) {
        $post_type = $page_link['post_type'];
        $class = $_GET['post_type'] === $post_type ? 'class="current"' : '';
        $count = get_comments(array(
            'post_type' => $post_type,
            'count' => true
        ));
        // Add link.
        $status_links[$post_type] = sprintf(
            '<a href="edit-comments.php?post_type=%s"%s>%s (%s)</a>',
            $post_type,
            $class,
            $page_link['name'],
            $count
        );
    }
    return $status_links;
}
// Added by fast_add_admin_comment_page_links
