<?php
/**
 * Actions for analytics
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Data collection actions.
 */

/*
 * fast_filter_analytics_taxonomy_change
 */

/**
 * Logs a ticket being opened / closed.
 *
 * @param int     $ticket_id ID of ticket post..
 * @param string  $taxonomy Taxonomy that changed.
 * @param WP_Term $term Taxonomy term.
 * @param object  $log_id Log id.
 */
function fast_log_ticket_status_change($ticket_id, $taxonomy, $term, $log_id)
{
    if ('fast_status' === $taxonomy) {
        switch ($term->slug) {
            case 'open':
                $ticket = get_post($ticket_id);
                if (0 === (int) $ticket->comment_count) {
                    // Add this ticket to the list of tickets closed today.
                    fast_add_value_to_log_meta_key_list($ticket_id, '_opened', $log_id, true);
                }
                break;
            case 'closed':
                // Add this ticket to the list of tickets closed today.
                fast_add_value_to_log_meta_key_list($ticket_id, '_closed', $log_id, true);

                // Add this ticket to the list of tickets closed by this agent.
                $ticket_agent = fast_get_ticket_agent($ticket_id);
                fast_add_value_to_log_meta_key_list($ticket_id, '_agent_closed_' . $ticket_agent->ID, $log_id, true);
                break;
        }
    }
}
add_action('fast_filter_analytics_taxonomy_change', 'fast_log_ticket_status_change', 10, 4);

/**
 * Logs a ticket being assigned to a product
 *
 * @param int     $ticket_id ID of ticket post..
 * @param string  $taxonomy Taxonomy that changed.
 * @param WP_Term $term Taxonomy term.
 * @param object  $log_id Log id.
 */
function fast_log_ticket_product_change($ticket_id, $taxonomy, $term, $log_id)
{
    if ('fast_product' === $taxonomy) {
        $ticket = get_post($ticket_id);
        if (0 === (int) $ticket->comment_count) {
            // Add this ticket to the list of tickets closed today.
            fast_add_value_to_log_meta_key_list($ticket_id, '_product_' . $term->term_id, $log_id, true);
        }
    }
}
add_action('fast_filter_analytics_taxonomy_change', 'fast_log_ticket_product_change', 10, 4);

/*
 * fast_filter_analytics_agent_reply
 */

/**
 * Logs a ticket answered.
 *
 * @param object $comment Comment Object.
 * @param object $log_id Log id.
 */
function fast_log_agent_ticket_response($comment, $log_id)
{
    $is_user_agent = fast_is_user_agent($comment->user_id);
    if ($is_user_agent) {
        // Add this ticket to the list of tickets responded by this agent.
        fast_increment_log_meta_key($comment->comment_post_ID, '_agent_responded_' . $comment->user_id, $log_id);
        // Log ticket that was responded to.
        fast_add_value_to_log_meta_key_list($comment->comment_post_ID, '_agent_responded_ids_' . $comment->user_id, $log_id, true);

        // Log ticket agents response time.
        $ticket = get_post($comment->comment_post_ID);
        $response_time = fast_calculate_response_time($ticket, $comment);
        if (null !== $response_time) {
            // Add response time to agent.
            fast_add_value_to_log_meta_key_list($response_time, '_agent_response_time_' . $comment->user_id, $log_id, false);

            // Add ticket id to response time tickets.
            fast_add_value_to_log_meta_key_list($comment->comment_post_ID, '_agent_response_time_ids_' . $comment->user_id, $log_id, true);
        }
    }
}
add_action('fast_filter_analytics_ticket_comment', 'fast_log_agent_ticket_response', 10, 2);

/**
 * Logs the number of total open tickets.
 *
 * @param int $log_id ID of the daily log.
 */
function fast_log_total_open_tickets($log_id)
{
    $total_open_ticket_ids = get_posts(array(
        'post_type' => 'fast_ticket',
        'fields' => 'ids',
        'posts_per_page' => -1,
        'status' => 'publish,private',
        'tax_query' => array(
            array(
                'taxonomy' => 'fast_status',
                'field'    => 'slug',
                'terms'    => 'open'
            )
        )
    ));

    $count = count($total_open_ticket_ids);
    update_post_meta($log_id, '_total_open_tickets', $count);
    update_post_meta($log_id, '_total_open_tickets_ids', $total_open_ticket_ids);
}
add_action('fast_action_on_new_log_creation', 'fast_log_total_open_tickets');
