<?php
/**
 * Filters for analytics
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Chart filters
 */

/**
 * Data for status chart.
 *
 * @param array $chart Chart data.
 */
function fast_chart_status($chart)
{
    $chart = array(
        'title' => esc_html__('Open & Closed Tickets', 'fast-plugin'),
        'x_axis_label' => '',
        'y_axis_label' => esc_html__('Tickets', 'fast-plugin'),
        'series_param_filter' => 'fast_filter_series_param_active_series',
        'series' => array(
            array(
                'id' => 'opened',
                'name' => 'Opened',
                'fields' => array(
                    'value' => array(
                        'meta_key' => '_opened',
                        'lookup_key' => '_opened',
                        'lookup_filter' => 'fast_filter_lookup_get_meta_array',
                        'chart_filter' => 'fast_filter_chart_data_count',
                    ),
                    'ids' => array(
                        'meta_key' => '_opened',
                        'lookup_key' => '_opened_ids',
                        'lookup_filter' => 'fast_filter_lookup_get_meta_array',
                        'chart_filter' => 'fast_filter_chart_data_copy',
                    )
                )
            ),
            array(
                'id' => 'closed',
                'name' => 'Closed',
                'fields' => array(
                    'value' => array(
                        'meta_key' => '_closed',
                        'lookup_key' => '_closed',
                        'lookup_filter' => 'fast_filter_lookup_get_meta_array',
                        'chart_filter' => 'fast_filter_chart_data_count',
                    ),
                    'ids' => array(
                        'meta_key' => '_closed',
                        'lookup_key' => '_closed_ids',
                        'lookup_filter' => 'fast_filter_lookup_get_meta_array',
                        'chart_filter' => 'fast_filter_chart_data_copy',
                    )
                )
            )
        )
    );
    return $chart;
}
add_filter('fast_filter_chart_status', 'fast_chart_status', 10, 2);

/**
 * Data for agent closed chart.
 *
 * @param array $chart Chart data.
 */
function fast_agent_closed_chart($chart)
{
    $chart = array(
        'title' => esc_html__('Agents Closed Tickets', 'fast-plugin'),
        'x_axis_label' => '',
        'y_axis_label' => esc_html__('Tickets', 'fast-plugin'),
        'series_param_filter' => 'fast_filter_series_param_active_series',
        'series' => array()
    );

    $agents = get_terms(array(
        'taxonomy' => 'fast_agent',
        'hide_empty' => false,
    ));

    foreach ($agents as $agent) {
        $user_id = get_term_meta($agent->term_id, '_fast-plugin-agent_user_id');
        if (!empty($user_id)) {
            $chart['series'][] = array(
                'id' => $user_id[0],
                'name' => $agent->name,
                'fields' => array(
                    'value' => array(
                        'meta_key' => '_agent_closed_' . $user_id[0],
                        'lookup_key' => '_agent_closed_' . $user_id[0],
                        'lookup_filter' => 'fast_filter_lookup_get_meta_array',
                        'chart_filter' => 'fast_filter_chart_data_count',
                    ),
                    'ids' => array(
                        'meta_key' => '_agent_closed_' . $user_id[0],
                        'lookup_key' => '_agent_closed_ids_' . $user_id[0],
                        'lookup_filter' => 'fast_filter_lookup_get_meta_array',
                        'chart_filter' => 'fast_filter_chart_data_copy',
                    ),
                )
            );
        }
    }
    return $chart;
}
add_filter('fast_filter_chart_agent_closed', 'fast_agent_closed_chart', 10, 2);

/**
 * Data for agent closed chart.
 *
 * @param array $chart Chart data.
 */
function fast_agent_responded_chart($chart)
{
    $chart = array(
        'title' => esc_html__('Agents Responded Tickets', 'fast-plugin'),
        'x_axis_label' => '',
        'y_axis_label' => esc_html__('Responses', 'fast-plugin'),
        'series_param_filter' => 'fast_filter_series_param_active_series',
        'series' => array()
    );

    $agents = get_terms(array(
        'taxonomy' => 'fast_agent',
        'hide_empty' => false,
    ));

    foreach ($agents as $agent) {
        $user_id = get_term_meta($agent->term_id, '_fast-plugin-agent_user_id');
        if (!empty($user_id)) {
            $chart['series'][] = array(
                'id' => 'agent_closed_' . $user_id[0],
                'name' => $agent->name,
                'fields' => array(
                    'value' => array(
                        'meta_key' => '_agent_responded_' . $user_id[0],
                        'lookup_key' => '_agent_responded_' . $user_id[0],
                        'lookup_filter' => 'fast_filter_lookup_get_meta_value',
                        'chart_filter' => 'fast_filter_chart_data_sum',
                    ),
                    'ids' => array(
                        'meta_key' => '_agent_responded_ids_' . $user_id[0],
                        'lookup_key' => '_agent_responded_ids_' . $user_id[0],
                        'lookup_filter' => 'fast_filter_lookup_get_meta_array',
                        'chart_filter' => 'fast_filter_chart_data_copy',
                    )
                )
            );
        }
    }
    return $chart;
}
add_filter('fast_filter_chart_agent_responded', 'fast_agent_responded_chart', 10, 2);

/**
 * Data for agent closed chart.
 *
 * @param array $chart Chart data.
 */
function fast_agent_response_time_chart($chart)
{
    $chart = array(
        'title' => esc_html__('Agents Response Time', 'fast-plugin'),
        'x_axis_label' => '',
        'y_axis_label' => esc_html__('Time (hours)', 'fast-plugin'),
        'series_param_filter' => 'fast_filter_series_param_active_series',
        'series' => array()
    );

    $agents = get_terms(array(
        'taxonomy' => 'fast_agent',
        'hide_empty' => false,
    ));

    foreach ($agents as $agent) {
        $user_id = get_term_meta($agent->term_id, '_fast-plugin-agent_user_id');
        if (!empty($user_id)) {
            $chart['series'][] = array(
                'id' => 'agent_closed_' . $user_id[0],
                'name' => $agent->name,
                'fields' => array(
                    'value' => array(
                        'meta_key' => '_agent_response_time_' . $user_id[0],
                        'lookup_key' => '_agent_response_time_' . $user_id[0],
                        'lookup_filter' => 'fast_filter_lookup_get_meta_array',
                        'chart_filter' => array(
                            'fast_filter_chart_data_avg',
                            'fast_filter_chart_data_hours',
                        )
                    ),
                    'ids' => array(
                        'meta_key' => '_agent_response_time_ids_' . $user_id[0],
                        'lookup_key' => '_agent_response_time_ids_' . $user_id[0],
                        'lookup_filter' => 'fast_filter_lookup_get_meta_array',
                        'chart_filter' => 'fast_filter_chart_data_copy',
                    )
                )
            );
        }
    }
    return $chart;
}
add_filter('fast_filter_chart_agent_response_time', 'fast_agent_response_time_chart', 10, 2);

/**
 * Data for products opened chart.
 *
 * @param array $chart Chart data.
 */
function fast_products_opened_chart($chart)
{
    $chart = array(
        'title' => esc_html__('New tickets per product', 'fast-plugin'),
        'x_axis_label' => '',
        'y_axis_label' => esc_html__('Tickets', 'fast-plugin'),
        'series_param_filter' => 'fast_filter_series_param_active_series',
        'series' => array()
    );

    $products = get_terms(array(
        'taxonomy' => 'fast_product',
        'hide_empty' => false,
    ));

    foreach ($products as $product) {
        $chart['series'][] = array(
            'id' => 'product_' . $product->term_id,
            'name' => $product->name,
            'fields' => array(
                'value' => array(
                    'meta_key' => '_product_' . $product->term_id,
                    'lookup_key' => '_product_' . $product->term_id,
                    'lookup_filter' => 'fast_filter_lookup_get_meta_array',
                    'chart_filter' => 'fast_filter_chart_data_count',
                ),
                'ids' => array(
                    'meta_key' => '_product_' . $product->term_id,
                    'lookup_key' => '_product_ids_' . $product->term_id,
                    'lookup_filter' => 'fast_filter_lookup_get_meta_array',
                    'chart_filter' => 'fast_filter_chart_data_copy',
                )
            )
        );
    }
    return $chart;
}
add_filter('fast_filter_chart_products_opened', 'fast_products_opened_chart', 10, 2);

/**
 * Data for products opened chart.
 *
 * @param array $chart Chart data.
 */
function fast_total_open_tickets_chart($chart)
{
    $chart = array(
        'title' => esc_html__('Open Tickets', 'fast-plugin'),
        'x_axis_label' => '',
        'y_axis_label' => esc_html__('Tickets', 'fast-plugin'),
        'series_param_filter' => 'fast_filter_series_param_active_series',
        'series' => array(
            array(
                'id' => 'open',
                'name' => esc_html__('Open Tickets', 'fast-plugin'),
                'fields' => array(
                    'value' => array(
                        'meta_key' => '_total_open_tickets',
                        'lookup_key' => '_total_open_tickets',
                        'lookup_filter' => 'fast_filter_lookup_get_meta_value',
                        'chart_filter' => 'fast_filter_chart_data_sum',
                    ),
                    'ids' => array(
                        'meta_key' => '_total_open_tickets_ids',
                        'lookup_key' => '_total_open_tickets_ids',
                        'lookup_filter' => 'fast_filter_lookup_get_meta_array',
                        'chart_filter' => 'fast_filter_chart_data_copy',
                    ),
                )
            )
        )
    );
    return $chart;
}
add_filter('fast_filter_chart_total_open_tickets', 'fast_total_open_tickets_chart', 10, 2);

/**
 * Series param filters
 */

/**
 * Active data for agent closed chart.
 *
 * @param array  $series Chart series data.
 * @param object $request WP_REST_Request object.
 */
function fast_filter_series_param_active_series($series, $request)
{
    $requested_series = $request->get_param('series');
    if (!empty($requested_series)) {
        $requested_series = explode(',', $requested_series);
        $new_series = array();
        foreach ($series as $item) {
            if (in_array($item['id'], $requested_series)) {
                $new_series[] = $item;
            }
        }
        return $new_series;
    }

    return $series;
}
add_filter('fast_filter_series_param_active_series', 'fast_filter_series_param_active_series', 10, 2);

/**
 * Lookup table filters
 */

/**
 * Creates lookup table data by merging arrays together.
 *
 * @param array  $lookup_entry Date to add data to.
 * @param object $field Field data.
 * @param string $log Log object.
 */
function fast_lookup_get_meta_array($lookup_entry, $field, $log)
{
    $lookup_entry = array_merge(array(
        $field['lookup_key'] => array(),
    ), $lookup_entry);

    // Get meta value.
    $value = get_post_meta($log->ID, $field['meta_key']);

    // If no value just use empty array.
    $value = empty($value) ? array() : $value[0];

    // Add value to lookup.
    $lookup_entry[$field['lookup_key']] = array_merge($lookup_entry[$field['lookup_key']], $value);

    return $lookup_entry;
}
add_filter('fast_filter_lookup_get_meta_array', 'fast_lookup_get_meta_array', 10, 4);

/**
 * Creates lookup table data by copying value from meta.
 *
 * @param array  $lookup_entry Date to add data to.
 * @param object $field Field data.
 * @param string $log Log object.
 */
function fast_lookup_get_meta_value($lookup_entry, $field, $log)
{
    $lookup_entry = array_merge(array(
        $field['lookup_key'] => array(),
    ), $lookup_entry);

    // Get meta value.
    $value = get_post_meta($log->ID, $field['meta_key']);

    if (!empty($value)) {
        array_push($lookup_entry[$field['lookup_key']], $value[0]);
    }
    return $lookup_entry;
}
add_filter('fast_filter_lookup_get_meta_value', 'fast_lookup_get_meta_value', 10, 4);

/**
 * Graph data filters.
 */

/**
 * Counts values in lookup table.
 *
 * @param array    $series_data Data to add to.
 * @param string   $value_key Name of value key to save field.
 * @param object   $field Field data.
 * @param string   $span Timespan name.
 * @param array    $lookup_table Data lookup array.
 * @param DateTime $date Date of data to add.
 */
function fast_chart_data_count($series_data, $value_key, $field, $span, $lookup_table, $date)
{
    // Get the timestamp of the data date.
    $timestamp = $date->getTimestamp();

    // Create blank field data.
    $series_data = array_merge(array(
        $value_key => 0
    ), $series_data);

    // Get the key and see if we have any lookup data to calculate.
    $key = apply_filters('fast_filter_series_span_key', $span, $timestamp);
    if (isset($lookup_table[$key][$field['lookup_key']])) {
        // Calculate count.
        $series_data[$value_key] = count($lookup_table[$key][$field['lookup_key']]);
    }
    return $series_data;
}
add_filter('fast_filter_chart_data_count', 'fast_chart_data_count', 10, 6);

/**
 * Sums up values from lookup table.
 *
 * @param array    $series_data Data to add to.
 * @param string   $value_key Name of value key to save field.
 * @param object   $field Field data.
 * @param string   $span Timespan name.
 * @param array    $lookup_table Data lookup array.
 * @param DateTime $date Date of data to add.
 */
function fast_chart_data_sum($series_data, $value_key, $field, $span, $lookup_table, $date)
{
    // Get the timestamp of the data date.
    $timestamp = $date->getTimestamp();

    // Create blank field data.
    $series_data = array_merge(array(
        $value_key => 0
    ), $series_data);

    // Get the key and see if we have any lookup data to calculate.
    $key = apply_filters('fast_filter_series_span_key', $span, $timestamp);
    if (isset($lookup_table[$key][$field['lookup_key']])) {
        // Calculate count.
        $series_data[$value_key] = array_sum($lookup_table[$key][$field['lookup_key']]);
    }
    return $series_data;
}
add_filter('fast_filter_chart_data_sum', 'fast_chart_data_sum', 10, 6);

/**
 * Sums up values from lookup table.
 *
 * @param array    $series_data Data to add to.
 * @param string   $value_key Name of value key to save field.
 * @param object   $field Field data.
 * @param string   $span Timespan name.
 * @param array    $lookup_table Data lookup array.
 * @param DateTime $date Date of data to add.
 */
function fast_chart_data_avg($series_data, $value_key, $field, $span, $lookup_table, $date)
{
    // Get the timestamp of the data date.
    $timestamp = $date->getTimestamp();

    // Create blank field data.
    $series_data = array_merge(array(
        $value_key => 0
    ), $series_data);

    // Get the key and see if we have any lookup data to calculate.
    $key = apply_filters('fast_filter_series_span_key', $span, $timestamp);
    if (isset($lookup_table[$key][$field['lookup_key']]) && count($lookup_table[$key][$field['lookup_key']]) > 0) {
        // Calculate count.
        $sum = array_sum($lookup_table[$key][$field['lookup_key']]);
        $series_data[$value_key] = $sum / count($lookup_table[$key][$field['lookup_key']]);
    }
    return $series_data;
}
add_filter('fast_filter_chart_data_avg', 'fast_chart_data_avg', 10, 6);

/**
 * Sums up values from lookup table.
 *
 * @param array    $series_data Data to add to.
 * @param string   $value_key Name of value key to save field.
 * @param object   $field Field data.
 * @param string   $span Timespan name.
 * @param array    $lookup_table Data lookup array.
 * @param DateTime $date Date of data to add.
 */
function fast_chart_data_hours($series_data, $value_key, $field, $span, $lookup_table, $date)
{
    // Get the timestamp of the data date.
    $timestamp = $date->getTimestamp();

    // Create blank field data.
    $series_data = array_merge(array(
        $value_key => 0
    ), $series_data);

    // Get the key and see if we have any lookup data to calculate.
    $key = apply_filters('fast_filter_series_span_key', $span, $timestamp);
    if (isset($series_data[$value_key])) {
        // Calculate hours.
        $series_data[$value_key] = round($series_data[$value_key] / 60 / 60, 2);
    }
    return $series_data;
}
add_filter('fast_filter_chart_data_hours', 'fast_chart_data_hours', 10, 6);

/**
 * Simply copies array values.
 *
 * @param array    $series_data Data to add to.
 * @param string   $value_key Name of value key to save field.
 * @param object   $field Field data.
 * @param string   $span Timespan name.
 * @param array    $lookup_table Data lookup array.
 * @param DateTime $date Date of data to add.
 */
function fast_chart_data_copy($series_data, $value_key, $field, $span, $lookup_table, $date)
{
    // Get the timestamp of the data date.
    $timestamp = $date->getTimestamp();

    // Create blank field data.
    $series_data = array_merge(array(
        $value_key => array()
    ), $series_data);

    // Get the key and see if we have any lookup data to calculate.
    $key = apply_filters('fast_filter_series_span_key', $span, $timestamp);
    if (isset($lookup_table[$key][$field['lookup_key']])) {
        // Copy value to series.
        $series_data[$value_key] = $lookup_table[$key][$field['lookup_key']];
    }
    return $series_data;
}
add_filter('fast_filter_chart_data_copy', 'fast_chart_data_copy', 10, 6);

/**
 * Utility Filters
 */

/**
 * Gets the array key to use for a timespan.
 *
 * @param string $span Timespan name.
 * @param string $timestamp Date string for key.
 */
function fast_series_span_key($span, $timestamp)
{
    switch ($span) {
        case 'years':
            $key_format = 'Y';
            break;
        case 'months':
            $key_format = 'n Y';
            break;
        case 'days':
        default:
            $key_format = 'j n Y';
            break;
    }
    return date($key_format, $timestamp);
}
add_filter('fast_filter_series_span_key', 'fast_series_span_key', 10, 2);

/**
 * Gets the graph label to use for a timespan.
 *
 * @param string $span Timespan name.
 * @param string $timestamp Date string for key.
 */
function fast_series_span_label($span, $timestamp)
{
    switch_to_locale(get_user_locale());
    switch ($span) {
        case 'years':
            $label_format = 'Y';
            break;
        case 'months':
            $label_format = 'F Y';
            break;
        case 'days':
        default:
            $label_format = 'l jS';
            break;
    }
    return date_i18n($label_format, $timestamp);
}
add_filter('fast_filter_series_span_label', 'fast_series_span_label', 10, 2);

/**
 * Gets the timespan to use to increment dates.
 *
 * @param string $span Timespan name.
 */
function fast_series_span_increment($span)
{
    switch ($span) {
        case 'years':
            return '+1 year';
        case 'months':
            return '+1 month';
        case 'days':
        default:
            return '+1 day';
    }
}
add_filter('fast_filter_series_span_increment', 'fast_series_span_increment', 10, 2);
