<?php
/**
 * Functions for analytics
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Creates a lookup table of log values for a timespan.
 *
 * @param array  $logs List of log posts.
 * @param string $series_list List of series to create lookups for.
 * @param string $span Name of timespan.
 *
 * @return array Lookup data table.
 */
function fast_analytics_create_lookup_table($logs, $series_list, $span)
{
    $lookup = array();
    foreach ($logs as $log) {
        foreach ($series_list as $series) {
            $key = apply_filters('fast_filter_series_span_key', $span, strtotime($log->post_date_gmt));

            // Make sure we have look up array for series.
            if (!isset($lookup[$key])) {
                $lookup[$key] = array();
            }

            foreach ($series['fields'] as $field) {
                $lookup[$key] = apply_filters($field['lookup_filter'], $lookup[$key], $field, $log);
            }
        }
    }
    return $lookup;
}

/**
 * Creates chart data from lookup table.
 *
 * @param array    $series_list List of series to create chart for.
 * @param DateTime $from_date From date.
 * @param DateTime $to_date To date.
 * @param string   $span_param Span increment.
 * @param array    $lookup_table Lookup table data.
 *
 * @return array Chart data table.
 */
function fast_analytics_create_chart_data($series_list, $from_date, $to_date, $span_param, $lookup_table)
{
    $increment = apply_filters('fast_filter_series_span_increment', $span_param);
    $loop_date = clone $from_date;
    while ($loop_date <= $to_date) {
        // Create new row.
        $row_data = array(
            'name' => apply_filters('fast_filter_series_span_label', $span_param, $loop_date->getTimestamp()),
            'series' => array()
        );
        foreach ($series_list as $series) {
            // Create series data.
            $series_data = array(
                'name' => $series['name'],
            );
            // Add fields to series data.
            foreach ($series['fields'] as $value_key => $field) {
                $chart_filters = is_array($field['chart_filter']) ? $field['chart_filter'] : array($field['chart_filter']);
                foreach ($chart_filters as $chart_filter) {
                    $series_data = apply_filters($chart_filter, $series_data, $value_key, $field, $span_param, $lookup_table, $loop_date);
                }
            }
            // Add series data to row.
            $row_data['series'][] = $series_data;
        }
        // Add row to graph.
        $data[] = $row_data;
        // Move on to the next row.
        $loop_date->modify($increment);
    }
    return $data;
}

/**
 * Adds a value to the list stored in meta.
 *
 * @param any     $value Value to add to list.
 * @param string  $meta_key Meta key to store list.
 * @param int     $log_id Log id.
 * @param boolean $unique Only store unique values.
 */
function fast_add_value_to_log_meta_key_list($value, $meta_key, $log_id, $unique)
{
    // Get the list of ticket ids.
    $id_list = get_post_meta($log_id, $meta_key, true);

    // If no list exists create one.
    if (empty($id_list)) {
        $id_list = array();
    }

    // If ticket is not in the list already add it.
    if ($unique && in_array($value, $id_list)) {
        return;
    }

    $id_list[] = $value;
    update_post_meta($log_id, $meta_key, $id_list);
}

/**
 * Adds a ticket to the list stored in meta.
 *
 * @param int    $ticket_id ID of ticket post.
 * @param string $meta_key Meta key to store list.
 * @param int    $log_id Log id.
 */
function fast_increment_log_meta_key($ticket_id, $meta_key, $log_id)
{
    // Get the list of ticket ids.
    $count = get_post_meta($log_id, $meta_key, true);

    // If no count exists create one.
    if (empty($count)) {
        $count = 0;
    }

    // If ticket is not in the list already add it.
    $count++;
    update_post_meta($log_id, $meta_key, $count);
}

/**
 * Returns number of posts found for a query of a specific user.
 *
 * @param object $date Date to retrieve log for.
 *
 * @return int ID for post found / created.
 */
function fast_get_daily_log_id($date)
{
    $logs = get_posts(array(
        'post_type'      => 'fast_log',
        'fields'         => 'ids',
        'posts_per_page' => 1,
        'date_query'     => array(
            array(
                'year'  => $date['year'],
                'month' => $date['mon'],
                'day'   => $date['mday']
            )
        )
    ));

    if (empty($logs)) {
        $id = wp_insert_post(array(
            'post_type' => 'fast_log',
            'post_status' => 'publish'
        ), true);

        do_action('fast_action_on_new_log_creation', $id);

        return $id;
    }

    return $logs[0];
}
