<?php
/**
 * Admin only actions
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/*
* fast_filter_ticket_roles
*/

/**
 * Returns the plugin roles.
 *
 * @param array $roles Array of roles.
 */
function fast_get_ticket_roles($roles = array())
{
    return array_merge($roles, array(
        'fast_customer' => esc_html__('Ticket Customer', 'fast-plugin'),
        'fast_agent'    => esc_html__('Ticket Agent', 'fast-plugin'),
        'fast_manager'  => esc_html__('Ticket Manager', 'fast-plugin')
    ));
}
add_filter('fast_filter_ticket_roles', 'fast_get_ticket_roles', 10, 1);

/*
* fast_filter_get_all_available_roles
*/

/**
 * Returns the plugin roles.
 *
 * @param array $roles Array of roles.
 */
function fast_get_all_available_roles($roles = array())
{
    global $wp_roles;
    // Get FAST roles because we don't want to show them now do we.
    $fast_roles = apply_filters('fast_filter_ticket_roles', array());
    foreach ($wp_roles->roles as $role_id => $role) {
        if (!array_key_exists($role_id, $fast_roles)) {
            $roles[$role_id] = $role['name'];
        }
    }
    return $roles;
}
add_filter('fast_filter_get_all_available_roles', 'fast_get_all_available_roles', 10, 1);


/*
* fast_filter_ticket_capabilities
*/

/**
 * Returns the plugin capabilities.
 *
 * @param array $capabilities Array of capabilities.
 */
function fast_get_ticket_capabilities($capabilities = array())
{
    return array_merge($capabilities, array(
        'read'                  => esc_html__('Read', 'fast-plugin'),
        'edit_ticket'           => esc_html__('Edit Ticket', 'fast-plugin'),
        'read_ticket'           => esc_html__('Read Ticket', 'fast-plugin'),
        'delete_ticket'         => esc_html__('Delete Ticket', 'fast-plugin'),
        'edit_tickets'          => esc_html__('Edit Tickets', 'fast-plugin'),
        'edit_others_tickets'   => esc_html__('Edit Others Tickets', 'fast-plugin'),
        'publish_tickets'       => esc_html__('Publish Tickets', 'fast-plugin'),
        'read_private_tickets'  => esc_html__('Read Private Tickets', 'fast-plugin'),
        'manage_statuses'       => esc_html__('Manage Statuses', 'fast-plugin'),
        'assign_status'         => esc_html__('Change Ticket Status', 'fast-plugin'),
        'manage_agents'         => esc_html__('Manage Agents', 'fast-plugin'),
        'assign_agent'          => esc_html__('Change Ticket Agent', 'fast-plugin'),
        'manage_products'       => esc_html__('Manage Products', 'fast-plugin'),
        'assign_product'        => esc_html__('Change Ticket Product', 'fast-plugin'),
        'manage_categories'     => esc_html__('Manage Category', 'fast-plugin'),
        'assign_category'       => esc_html__('Change Ticket Category', 'fast-plugin'),
        'upload_files'          => esc_html__('Upload Images', 'fast-plugin'),
        'delete_posts'          => esc_html__('Delete Attachments', 'fast-plugin'),
        'edit_comment'          => esc_html__('Edit Ticket Comment', 'fast-plugin'),
        'edit_reply'            => esc_html__('Edit Saved Reply', 'fast-plugin'),
        'edit_others_replies'   => esc_html__('Edit Others Saved Replies', 'fast-plugin'),
        'read_reply'            => esc_html__('Read Saved Reply', 'fast-plugin'),
        'delete_reply'          => esc_html__('Delete Saved Reply', 'fast-plugin'),
        'edit_replies'          => esc_html__('Edit Saved Replies', 'fast-plugin'),
        'publish_replies'       => esc_html__('Publish Replies', 'fast-plugin'),
        'delete_reply'          => esc_html__('Delete Saved Reply', 'fast-plugin'),
        'edit_purchase'         => esc_html__('Edit Purchase', 'fast-plugin'),
        'edit_others_purchases' => esc_html__('Edit Others Purchases', 'fast-plugin'),
        'read_purchase'         => esc_html__('Read Purchase', 'fast-plugin'),
        'edit_purchase'         => esc_html__('Edit Purchase', 'fast-plugin'),
        'delete_purchase'       => esc_html__('Delete Purchase', 'fast-plugin'),
        'edit_purchases'        => esc_html__('Edit Purchases', 'fast-plugin'),
        'publish_purchases'     => esc_html__('Publish Purchases', 'fast-plugin'),
        'delete_purchase'       => esc_html__('Delete Purchase', 'fast-plugin')
    ));
}
add_filter('fast_filter_ticket_capabilities', 'fast_get_ticket_capabilities', 10, 1);

/*
* fast_filter_default_fast_customer_capabilities
*/

/**
 * Returns the default customer capabilities.
 *
 * @param array $capabilities Array of capabilities.
 */
function fast_get_customer_capabilities($capabilities = array())
{
    return array_merge($capabilities, array(
        'read',
        'read_ticket',
        'edit_ticket',
        'publish_tickets',
        'edit_tickets',
        'assign_product',
        'assign_category',
        'upload_files',
        'delete_posts',
        'read_purchase',
        'edit_purchase',
        'edit_purchases',
        'delete_purchase',
        'publish_purchases',
        'edit_comment',
    ));
}
add_filter('fast_filter_default_fast_customer_capabilities', 'fast_get_customer_capabilities', 10, 1);

/*
* fast_filter_default_fast_agent_capabilities
*/

/**
 * Returns the default agent capabilities.
 *
 * @param array $capabilities Array of capabilities.
 */
function fast_get_agent_capabilities($capabilities = array())
{
    return array_merge($capabilities, array(
        'read',
        'edit_ticket',
        'read_ticket',
        'edit_tickets',
        'edit_others_tickets',
        'publish_tickets',
        'read_private_tickets',
        'assign_status',
        'assign_agent',
        'assign_product',
        'assign_category',
        'upload_files',
        'delete_posts',
        'edit_reply',
        'edit_others_replies',
        'read_reply',
        'delete_reply',
        'edit_replies',
        'publish_replies',
        'delete_reply',
        'read_purchase',
        'edit_purchase',
        'edit_purchases',
        'delete_purchase',
        'publish_purchases',
        'edit_comment',
    ));
}
add_filter('fast_filter_default_fast_agent_capabilities', 'fast_get_agent_capabilities', 10, 1);

/*
* fast_filter_default_fast_manager_capabilities
*/

/**
 * Returns the default manager capabilities.
 *
 * @param array $capabilities Array of capabilities.
 */
function fast_get_manager_capabilities($capabilities = array())
{
    return array_merge($capabilities, array(
        'read',
        'edit_ticket',
        'read_ticket',
        'delete_ticket',
        'edit_tickets',
        'edit_others_tickets',
        'publish_tickets',
        'read_private_tickets',
        'manage_statuses',
        'assign_status',
        'manage_agents',
        'assign_agent',
        'manage_products',
        'assign_product',
        'manage_categories',
        'assign_category',
        'upload_files',
        'delete_posts',
        'edit_reply',
        'edit_others_replies',
        'read_reply',
        'delete_reply',
        'edit_replies',
        'publish_replies',
        'delete_reply',
        'read_purchase',
        'edit_purchase',
        'edit_purchases',
        'delete_purchase',
        'publish_purchases',
        'edit_comment',
    ));
}
add_filter('fast_filter_default_fast_manager_capabilities', 'fast_get_manager_capabilities', 10, 1);

/*
* fast_filter_field_sanitize_html
*/

/**
 * Makes sure that only allowed html entities are stored in options
 *
 * @param array $html Options framework option value.
 */
function fast_field_sanitize_html($html)
{
    return wp_kses($html, fast_get_allowed_html_tags());
}
add_filter('fast_filter_field_sanitize_html', 'fast_field_sanitize_html');

/*
* fast_filter_mail_replace_shortcodes
*/

/**
 * Replaces shortcodes in email templates with their respective values
 *
 * @param string $content mail markup.
 * @param array  $shortcodes mail shortcodes.
 */
function fast_mail_replace_shortcodes($content, $shortcodes)
{
    foreach ($shortcodes as $sc => $value) {
        $content = str_replace($sc, $value, $content);
    }
    return $content;
}
add_filter('fast_filter_mail_replace_shortcodes', 'fast_mail_replace_shortcodes', 10, 2);

/*
* fast_filter_build_placeholders
*/

/**
 * Replaces shortcodes in email templates with their respective values
 *
 * @param string $ticket   Ticket data.
 * @param object $comment  Comment object.
 * @param object $sender   Sender user.
 * @param object $receiver Receiver user.
 */
function fast_content_build_placeholders($ticket, $comment, $sender, $receiver)
{
    $ticket_author_user = get_userdata($ticket->post_author);

    $placeholders = array(
        '[ticket_id]'        => '#' . $ticket->ID,
        '[sender_name]'      => $sender->display_name,
        '[ticket_title]'     => $ticket->post_title,
        '[ticket_author]'    => $ticket_author_user->display_name,
        '[comment_content]'  => false === $comment ? '' : $comment->comment_content,
        '[slack_link]'       => sprintf(
            '<%1$s|' . esc_html__('Ticket Link', 'fast-plugin') . '>',
            esc_url(get_permalink($ticket->ID))
        )
    );

    // Add product name if we have one.
    $product = get_the_terms($ticket->ID, 'fast_product');
    if (!empty($product)) {
        $placeholders['[product]'] = $product[0]->name;
    } else {
        $category = get_the_terms($ticket->ID, 'fast_category');
        if (!empty($category)) {
            $placeholders['[product]'] = $category[0]->name;
        }
    }

    // Add Order number if we have one.
    $order_id = get_post_meta($ticket->ID, '_woocommerce_order_id', true);
    if (fast_is_woocommerce_active() && !empty($order_id)) {
        $placeholders['[order_number]'] = $order_id;
    }

    return $placeholders;
}
add_filter('fast_filter_build_placeholders', 'fast_content_build_placeholders', 10, 4);

/**
 * Replaces shortcodes in email templates with their respective values
 *
 * @param string $content Content to change.
 * @param array  $ticket Ticket data.
 * @param object $comment Comment object.
 * @param object $sender   Sender user.
 * @param object $receiver Receiver user.
 */
function fast_content_replace_placeholders($content, $ticket, $comment, $sender, $receiver)
{
    $placeholders = apply_filters('fast_filter_build_placeholders', $ticket, $comment, $sender, $receiver);
    foreach ($placeholders as $placeholder => $value) {
        $content = str_replace($placeholder, $value, $content);
    }
    return $content;
}
add_filter('fast_filter_replace_placeholders', 'fast_content_replace_placeholders', 10, 5);

/**
 * Replaces default sender email of emails sent by the plugin with the email set in the plugin options.
 *
 * @param string $original_email Default email "from address".
 */
function fast_wp_mail_from($original_email)
{
    $email = fast_get_option('ticket_email_from_address');
    return !empty($email) && is_email($email) ? $email : $original_email;
}

/**
 * Replaces default sender name of emails sent by the plugin with the name set in the plugin options.
 *
 * @param string $original_name Default email "from name".
 */
function fast_wp_mail_from_name($original_name)
{
    $name = fast_get_option('ticket_email_from_name');
    return !empty($name) ? $name : $original_name;
}


/*
* rest_fast_ticket_query
*/

/**
 * Add the filter parameter
 *
 * @param  array           $args    The query arguments.
 * @param  WP_REST_Request $request Full details about the request.
 * @return array $args.
 **/
function fast_rest_api_filter_add_filter_param($args, $request)
{
    $user_id = get_current_user_id();
    $user_type = fast_get_user_type($user_id);
    $filters = apply_filters('fast_filter_get_' . $user_type . '_filters', $user_id);

    $request_filter = $request->get_param('filter');
    foreach ($filters as $key => $filter) {
        if ($request_filter === $filter['id']) {
            $active_filter = $filter;
        }
    }
    if (!empty($active_filter)) {
        $args = array_merge($args, $active_filter['args']);
        $pages = $request->get_param('pages');
        $args['posts_per_page'] *= empty($pages) ? 1 : $pages;
    }
    return $args;
}
/* Added in actions.php. */

/**
 * Add the filter parameter
 *
 * @param  array           $args    The query arguments.
 * @param  WP_REST_Request $request Full details about the request.
 * @return array $args.
 **/
function fast_rest_api_filter_add_search_params($args, $request)
{
    // Envato Lookup.
    $user_id = get_current_user_id();
    $is_agent = fast_is_user_agent($user_id);
    $envato_purchase_id = $request->get_param('envato_purchase_id');
    if (!empty($envato_purchase_id)) {
        $args['meta_query'] = array(
            array(
                'key'     => '_envato_purchase_id',
                'value'   => $envato_purchase_id,
            )
        );
    }

    // WordPress user lookup.
    $customer_search = $request->get_param('customer_search');
    if (!empty($customer_search)) {
        // Lookup authors using email / user or name.
        $author_query = new WP_User_Query(array(
            'search' => '*' . $customer_search . '*',
        ));
        $matches = $author_query->get_results();

        // Create an authors array to search with.
        $authors = array();
        if (!empty($matches)) {
            foreach ($matches as $author) {
                $authors[] = $author->ID;
            }
        } else {
            /*
             * We didn't find any authors so send a author id that doesn't exist.
             * It isn't pretty but WordPress returns all authors if author__in is
             * sent an empty array so what's a guy supposed to do?
             **/
            $authors[] = PHP_INT_MAX;
        }
        $args['author__in'] = $authors;
    }

    return $args;
}
/* Added in actions.php. */



/*
* fast_filter_get_manager_filters
*/

/**
 * Returns the manager filters
 *
 * @param string $user_id id of the current user.
 * @return array $filters.
 **/
function fast_get_manager_filters($user_id)
{
    $agent_id = fast_get_user_agent_id($user_id);
    return apply_filters('fast_filter_modify_manager_filters', array_merge(
        array(
            array(
                'id' => 'my-tickets',
                'name' => esc_html__('My Tickets', 'fast-plugin'),
                'args' => array(
                    'posts_per_page' => 25,
                    'post_status'    => 'publish,private',
                    'tax_query'      => array(
                        array(
                            'taxonomy' => 'fast_agent',
                            'field'    => 'term_id',
                            'terms'    => $agent_id,
                        ),
                        array(
                            'taxonomy' => 'fast_status',
                            'field'    => 'slug',
                            'terms'    => 'open'
                        ),
                    ),
                    'orderby' => array(
                        'meta_value' => 'ASC',
                        'modified'   => 'ASC'
                    ),
                    'meta_key' => '_agent_replied'
                )
            ),
            array(
                'id' => 'all-tickets',
                'name' => esc_html__('All Tickets', 'fast-plugin'),
                'args' => array(
                    'posts_per_page' => 25,
                    'post_status'    => 'publish,private',
                    'tax_query'      => array(
                        array(
                            'taxonomy' => 'fast_status',
                            'field'    => 'slug',
                            'terms'    => 'open'
                        ),
                    ),
                    'orderby' => array(
                        'meta_value' => 'ASC',
                        'modified'   => 'ASC'
                    ),
                    'meta_key' => '_agent_replied'
                )
            )
        ),
        fast_create_filters_from_taxonomy_terms(array('fast_agent', 'fast_product', 'fast_category', 'fast_status'), array(
            'orderby' => 'modified',
            'order'   => 'DESC'
        ))
    ), $user_id);
}
add_filter('fast_filter_get_manager_filters', 'fast_get_manager_filters', 10, 2);

/*
* fast_filter_get_agent_filters
*/

/**
 * Returns the agent filters
 *
 * @param string $user_id id of the current user.
 * @return array $filters.
 **/
function fast_get_agent_filters($user_id)
{
    $agent_id = fast_get_user_agent_id($user_id);
    return apply_filters('fast_filter_modify_agent_filters', array_merge(
        array(
            array(
                'id' => 'my-tickets',
                'name' => esc_html__('My Tickets', 'fast-plugin'),
                'args' => array(
                    'posts_per_page' => 25,
                    'post_status'    => 'publish,private',
                    'tax_query'      => array(
                        array(
                            'taxonomy' => 'fast_agent',
                            'field'    => 'term_id',
                            'terms'    => $agent_id,
                        ),
                        array(
                            'taxonomy' => 'fast_status',
                            'field'    => 'slug',
                            'terms'    => 'open'
                        ),
                    ),
                    'orderby' => array(
                        'meta_value' => 'ASC',
                        'modified'   => 'ASC'
                    ),
                    'meta_key' => '_agent_replied'
                )
            )
        ),
        fast_create_filters_from_taxonomy_terms(array('fast_agent', 'fast_product', 'fast_category', 'fast_status'), array(
            'orderby' => 'modified',
            'order'   => 'DESC'
        ))
    ), $user_id);
}
add_filter('fast_filter_get_agent_filters', 'fast_get_agent_filters', 10, 2);

/*
* fast_filter_get_customer_filters
*/

/**
 * Returns the customer filters
 *
 * @param string $user_id id of the current user.
 * @return array $filters.
 **/
function fast_get_customer_filters($user_id)
{
    return apply_filters('fast_filter_modify_customer_filters', array_merge(
        array(
            array(
                'id' => 'my-tickets',
                'name' => esc_html__('My Tickets', 'fast-plugin'),
                'args' => array(
                    'author' => $user_id,
                    'posts_per_page'=> 25,
                    'post_status' => 'publish,private',
                    'tax_query'      => array(
                        array(
                            'taxonomy' => 'fast_status',
                            'field'    => 'slug',
                            'terms'    => 'open'
                        ),
                    ),
                    'orderby' => array(
                        'meta_value' => 'DESC',
                        'modified'   => 'DESC'
                    ),
                    'meta_key' => '_agent_replied'
                )
            )
        ),
        fast_create_filters_from_taxonomy_terms(array('fast_agent', 'fast_product', 'fast_category', 'fast_status'), array(
            'orderby' => 'modified',
            'order'   => 'DESC'
        ))
    ), $user_id);
}
add_filter('fast_filter_get_customer_filters', 'fast_get_customer_filters', 10, 2);

/*
* fast_filter_modify_customer_filters
*/

/**
 * Adds ordering to all agent filters - order by agent replied and date
 *
 * @param array $filters All agent filters.
 * @param int   $user_id current user id.
 * @return array Modified agent filters.
 */
function fast_add_ordering_to_customer_filters($filters, $user_id)
{
    foreach ($filters as &$filter) {
        $filter['args']['author'] = $user_id;
    }
    return $filters;
}
add_filter('fast_filter_modify_customer_filters', 'fast_add_ordering_to_customer_filters', 10, 2);

/*
* post_type_link
*/

/**
 * Changes ticket url to point to url in the app.
 *
 * @param object $permalink Post permalink object.
 * @param object $post Post object.
 */
function fast_change_ticket_view_link($permalink, $post)
{
    if ('fast_ticket' === $post->post_type) {
        $support_page_id = fast_get_option('support_page');
        if (!empty($support_page_id)) {
            $permalink = trailingslashit(get_the_permalink($support_page_id)) . 'ticket/' . $post->ID;
        }
    }
    return $permalink;
}
add_filter('post_type_link', 'fast_change_ticket_view_link', 10, 2);

/*
* the_title
*/

/**
 * Removes the Private: prefix from titles of tickets
 *
 * @param object $private_title_format Post title format.
 * @param object $id Post object.
 */
function fast_remove_private_from_ticket_title($private_title_format, $id)
{
    return get_post_type($id) === 'fast_ticket' ? '%s' : $private_title_format;
}
add_filter('private_title_format', 'fast_remove_private_from_ticket_title', 10, 2);

/*
* fast_filter_options
*/

/**
 * Plugin options
 *
 * @param array $options Plugin option pages.
 */
function fast_plugin_options($options)
{
    return array(
        array(
            'slug' => 'fast-plugin-general-options',
            'page' => array(
                'title' => esc_html__('Fast Support Ticket Plugin Options', 'fast-plugin'),
            ),
            'menu' => array(
                'title' => esc_html__('FAST Support', 'fast-plugin')
            ),
            'sections'   => array(
                'support-pages' => array(
                    'title'   => esc_html__('Support Page', 'fast-plugin'),
                    'header'  => esc_html__('This setting will define the page that is used to load the plugin support app.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'             => esc_html__('Support Page', 'fast-plugin'),
                            'desc'             => esc_html__('Set the page where your support system will be shown.', 'fast-plugin'),
                            'id'               => 'support_page',
                            'type'             => 'select',
                            'database'         => 'custom_post_type',
                            'custom_post_type' => 'page',
                            'default'          => '',
                            'empty_label'      => esc_html__('Not Set', 'fast-plugin'),
                        )
                    )
                ),
                'ticket-options' => array(
                    'title'   => esc_html__('Ticket Options', 'fast-plugin'),
                    'header'  => esc_html__('Configure the types of tickets that customers are able to create.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'       => esc_html__('Available Ticket Types', 'fast-plugin'),
                            'desc'       => esc_html__('Select the kinds of ticket that your customers will be able to create.', 'fast-plugin'),
                            'id'         => 'new_ticket_type',
                            'type'    => 'select',
                            'options' => array(
                                'regular'     => esc_html__('Product Tickets', 'fast-plugin'),
                                'general'     => esc_html__('General Tickets', 'fast-plugin'),
                                'envato'      => esc_html__('Envato Product Tickets', 'fast-plugin'),
                                'woocommerce' => esc_html__('WooCommerce Order Tickets', 'fast-plugin'),
                            ),
                            'default' => array('general'),
                            'select2'          => true,
                            'attributes'       => array(
                                'class' => 'fast-select2-option',
                                'multiple' => '',
                            )
                        ),
                        array(
                            'name'    => esc_html__('Default Ticket Privacy', 'fast-plugin'),
                            'desc'    => esc_html__('Allows users to create private tickets.', 'fast-plugin'),
                            'id'      => 'private_tickets',
                            'type'    => 'select',
                            'default' => 'on',
                            'options' => array(
                                'on'     => esc_html__('User can choose Public or Private', 'fast-plugin'),
                                'off'    => esc_html__('Always use Public Tickets', 'fast-plugin'),
                                'always' => esc_html__('Always use Private Tickets', 'fast-plugin'),
                            ),
                        ),
                        array(
                            'name'        => esc_html__('Default Ticket Agents', 'fast-plugin'),
                            'desc'        => esc_html__('Select the agents that will be assigned tickets by default.', 'fast-plugin'),
                            'id'          => 'default_agent',
                            'type'        => 'select',
                            'filter'      => 'fast_filter_ticket_agent_ids_values',
                            'default'     => array(),
                            'attributes'       => array(
                                'class' => 'fast-select2-option',
                                'multiple' => '',
                            )
                        ),
                        array(
                            'name'    => esc_html__('Comments Order', 'fast-plugin'),
                            'desc'    => esc_html__('Comments order in ticket view. Descending order renders the newest comments at the top.', 'fast-plugin'),
                            'id'      => 'ticket_comments_order',
                            'type'    => 'select',
                            'options' => array(
                                'asc'     => esc_html__('Ascending', 'fast-plugin'),
                                'desc'    => esc_html__('Descending', 'fast-plugin'),
                            ),
                            'default' => 'desc',
                        ),
                        array(
                            'name'    => esc_html__('Agents Can Create Tickets', 'fast-plugin'),
                            'desc'    => esc_html__('Allow agents to create new tickets.', 'fast-plugin'),
                            'id'      => 'agent_ticket_creation',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'    => esc_html__('Agents Can Create Tickets For Other Users', 'fast-plugin'),
                            'desc'    => esc_html__('Allow agents to create new tickets on behalf of customers / agents / managers.', 'fast-plugin'),
                            'id'      => 'agent_user_ticket_creation',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'    => esc_html__('Customers can reply to closed tickets', 'fast-plugin'),
                            'desc'    => esc_html__('Customers can reply to closed tickets and they will be reopened.', 'fast-plugin'),
                            'id'      => 'customer_reply_to_closed',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'    => esc_html__('Customers can search public tickets', 'fast-plugin'),
                            'desc'    => esc_html__('Customers will be able to search all public tickets.', 'fast-plugin'),
                            'id'      => 'customer_search_enabled',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'    => esc_html__('Allow agent private comments', 'fast-plugin'),
                            'desc'    => esc_html__('Agents are able to create private replies only visible by other agents.', 'fast-plugin'),
                            'id'      => 'allow_agent_private_comments',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'    => esc_html__('Allow customer notes', 'fast-plugin'),
                            'desc'    => esc_html__('Agents are able to add notes about a ticket customer and share them with all other agents..', 'fast-plugin'),
                            'id'      => 'allow_customer_notes',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'             => esc_html__('Show response times', 'fast-plugin'),
                            'id'               => 'show_response_times',
                            'desc'             => esc_html__('Show response times to the following roles.', 'fast-plugin'),
                            'type'             => 'select',
                            'empty_label'      => esc_html__('No Role Set', 'fast-plugin'),
                            'filter'           => 'fast_filter_ticket_roles',
                            'select2'          => true,
                            'default'          => array('fast_agent', 'fast_manager'),
                            'attributes'       => array(
                                'placeholder' => esc_html__('Select a user role', 'fast-plugin'),
                                'class'       => 'fast-select2-option',
                                'multiple'    => '',
                            )
                        )
                    )
                ),
                'ticket-sidebar-options' => array(
                    'title'   => esc_html__('Ticket Sidebar Options', 'fast-plugin'),
                    'header'  => esc_html__('Configure the sidebar ticket list.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'       => esc_html__('Show Customer information in Ticket List', 'fast-plugin'),
                            'desc'       => esc_html__('Adds a line that shows the customer username or customer name in the ticket list.', 'fast-plugin'),
                            'id'         => 'sidebar_customer_info',
                            'type'    => 'select',
                            'options' => array(
                                'none'       => esc_html__('Show nothing', 'fast-plugin'),
                                'authorName'  => esc_html__('Customer Full Name', 'fast-plugin'),
                                'authorFirstname' => esc_html__('Customer First Name', 'fast-plugin'),
                                'authorEmail' => esc_html__('Customer Email Address', 'fast-plugin'),
                                'authorUsername'  => esc_html__('WordPress Username', 'fast-plugin'),
                                'envato'     => esc_html__('Envato Username', 'fast-plugin'),
                            ),
                            'default' => 'authorName',
                        ),
                        array(
                            'name'    => esc_html__('Show client\'s Email in ticket details sidebar', 'fast-plugin'),
                            'desc'    => esc_html__('Show or hide the client\'s Email address when presenting a ticket\'s details.', 'fast-plugin'),
                            'id'      => 'ticket_show_client_mail',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                    )
                ),
                'color-options' => array(
                    'title'   => esc_html__('Color Options', 'fast-plugin'),
                    'header'  => esc_html__('Configure the colors used in the ticket system.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('App Skin', 'fast-plugin'),
                            'desc'    => esc_html__('Set the color scheme for your app.', 'fast-plugin'),
                            'id'      => 'app_color',
                            'type'    => 'select',
                            'default' => 'cyan_yellow',
                            'filter'  => 'fast_filter_app_colors',
                        ),
                        array(
                            'name'    => esc_html__('Favicon Alert Text Color', 'fast-plugin'),
                            'id'      => 'favicon_color',
                            'desc'    => esc_html__('Sets the color of the favicon alert text.', 'fast-plugin'),
                            'default' => '#FFFFFF',
                            'type'    => 'color',
                            'format'  => 'hex'
                        ),
                        array(
                            'name'    => esc_html__('Favicon Alert Background Color', 'fast-plugin'),
                            'id'      => 'favicon_bg_color',
                            'desc'    => esc_html__('Sets the background color of the favicon alert.', 'fast-plugin'),
                            'default' => '#FF0000',
                            'type'    => 'color',
                            'format'  => 'hex'
                        ),
                    )
                ),
                'reply-options' => array(
                    'title'   => esc_html__('Reply Options', 'fast-plugin'),
                    'header'  => esc_html__('Configure options for replying on tickets.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Default Agent Reply Type', 'fast-plugin'),
                            'desc'    => esc_html__('Sets the if agents comments are private (only other agents can see) or regular (customer sees) by default', 'fast-plugin'),
                            'id'      => 'ticket_default_internal_replies',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('Private', 'fast-plugin'),
                                'off'    => esc_html__('Regular', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'    => esc_html__('Minimum Reply Interval', 'fast-plugin'),
                            'id'      => 'ticket_minimum_reply_interval',
                            'desc'    => esc_html__('Sets the minimum time (in seconds) that should pass before a client can reply again.', 'fast-plugin'),
                            'type'    => 'slider',
                            'default'   => 15,
                            'attr'      => array(
                                'max'       => 120,
                                'min'       => 1,
                                'step'      => 1
                            )
                        ),
                    )
                ),
                'app-options' => array(
                    'title'   => esc_html__('App Options', 'fast-plugin'),
                    'header'  => esc_html__('General options for the app.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'     => esc_html__('Logo Image', 'fast-plugin'),
                            'desc'     => esc_html__('Sets the logo image that is shown on the toolbar.', 'fast-plugin'),
                            'id'       => 'app_logo_image',
                            'type'     => 'upload',
                            'store'    => 'url',
                            'default'  => ''
                        ),
                        array(
                            'name'    => esc_html__('Show user profile in menu', 'fast-plugin'),
                            'desc'    => esc_html__('Show or hide the profile option from the user menu.', 'fast-plugin'),
                            'id'      => 'show_user_profile',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'    => esc_html__('Show available keyboard shortcuts in menu', 'fast-plugin'),
                            'desc'    => esc_html__('Show or hide the keyboard shortcuts option from the user menu.', 'fast-plugin'),
                            'id'      => 'show_keyboard_shortcuts',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'    => esc_html__('Exit link url', 'fast-plugin'),
                            'desc'    => esc_html__('Shows an exit icon that takes user out of the app.  Leave blank to not show icon.', 'fast-plugin'),
                            'id'      => 'exit_icon_link',
                            'type'    => 'text',
                            'default' => '',
                        ),
                        array(
                            'name'    => esc_html__('Login url', 'fast-plugin'),
                            'desc'    => esc_html__('Url to the page that the user will be redirected to if he tries to access the app without logging in. The default app login screen will be used if option is left blank.', 'fast-plugin'),
                            'id'      => 'login_url',
                            'type'    => 'text',
                            'default' => '',
                        ),
                        array(
                            'name'    => esc_html__('Logout url', 'fast-plugin'),
                            'desc'    => esc_html__('Url to take users to when user logs out, if left blank will return to login page by default.', 'fast-plugin'),
                            'id'      => 'logout_url',
                            'type'    => 'text',
                            'default' => '',
                        ),
                        array(
                            'name'    => esc_html__('User profile url', 'fast-plugin'),
                            'desc'    => esc_html__('Url to the user\'s profile editor', 'fast-plugin'),
                            'id'      => 'profile_edit_url',
                            'type'    => 'text',
                            'default' => get_edit_user_link(),
                        ),
                        array(
                            'name'    => esc_html__('Support policy link url', 'fast-plugin'),
                            'desc'    => esc_html__('Url to the support policy page. The url will appear next to the "Reply to ticket" button', 'fast-plugin'),
                            'id'      => 'support_policy_link_url',
                            'type'    => 'text',
                            'default' => '',
                        ),
                        array(
                            'name'    => esc_html__('Support policy link text', 'fast-plugin'),
                            'desc'    => esc_html__('Text of the link to the support policy page. The url will appear next to the "Reply to ticket" button', 'fast-plugin'),
                            'id'      => 'support_policy_link_text',
                            'type'    => 'text',
                            'default' => '',
                        ),
                        array(
                            'name'    => esc_html__('Enable Analytics', 'fast-plugin'),
                            'desc'    => esc_html__('Allow Fast to log all events taking place in the app, and display an analytics page.', 'fast-plugin'),
                            'id'      => 'analytics_enabled',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'             => esc_html__('Analytics Access', 'fast-plugin'),
                            'id'               => 'analytics_access',
                            'desc'             => esc_html__('Choose the user roles that will be able to access analytics.', 'fast-plugin'),
                            'type'             => 'select',
                            'empty_label'      => esc_html__('No Role Set', 'fast-plugin'),
                            'filter'           => 'fast_filter_ticket_roles',
                            'select2'          => true,
                            'default'          => array('fast_manager'),
                            'attributes'       => array(
                                'placeholder' => esc_html__('Select a user role', 'fast-plugin'),
                                'class'       => 'fast-select2-option',
                                'multiple'    => '',
                            )
                        )
                    )
                ),
            )
        ),
        array(
            'slug' => 'fast-plugin-registration-page',
            'page' => array(
                'title' => esc_html__('Registration Page Content', 'fast-plugin'),
            ),
            'menu' => array(
                'title' => esc_html__('Registration Page', 'fast-plugin')
            ),
            'sections'   => array(
                'register-page' => array(
                    'title'   => esc_html__('User Registration', 'fast-plugin'),
                    'header'  => esc_html__('Set your support system to allow user registration page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Enable User Registration', 'fast-plugin'),
                            'desc'    => esc_html__('Allows users to be registered using the app.', 'fast-plugin'),
                            'id'      => 'register_user_enabled',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'    => esc_html__('Show Register Types', 'fast-plugin'),
                            'desc'    => esc_html__('Which register options to show on the register page.', 'fast-plugin'),
                            'id'      => 'register_types',
                            'select2' => true,
                            'type'    => 'select',
                            'options' => array(
                                'wordpress' => esc_html__('WordPress Register', 'fast-plugin'),
                                'envato'    => esc_html__('Envato Register', 'fast-plugin'),
                            ),
                            'default' => array(
                                'wordpress'
                            ),
                            'attributes' => array(
                                'class'    => 'fast-select2-option',
                                'multiple' => '',
                            )
                        ),
                        array(
                            'name'    => esc_html__('New User can choose language', 'fast-plugin'),
                            'desc'    => esc_html__('A select box will be available to allow the new user to run the app in their own language.', 'fast-plugin'),
                            'id'      => 'register_user_language',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                    )
                ),
                'register-page-content' => array(
                    'title'   => esc_html__('Registration Page', 'fast-plugin'),
                    'header'  => esc_html__('Set the content of your registration page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'          => esc_html__('Register User Page Title', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the title that is shown on the register user page.', 'fast-plugin'),
                            'id'            => 'register_page_title',
                            'type'          => 'text',
                            'default'       => esc_html__('Register for support', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Register User Page Content', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the content that is shown on the register user page.', 'fast-plugin'),
                            'id'            => 'register_page_content',
                            'type'          => 'editor',
                            'sanitize'      => 'html',
                            'store_default' => true,
                            'default'       => '<p>Are you having trouble?  Have no fear we are here to help you!</p><p>Just fill in your details to gain access to our support system and create a support ticket.</p><p>Our support team will then be in touch as soon as possible to help you.</p>'
                        ),
                        array(
                            'name'     => esc_html__('Register User Page Background Image', 'fast-plugin'),
                            'desc'     => esc_html__('Sets the background image that is shown on the register user page.', 'fast-plugin'),
                            'id'       => 'register_page_image',
                            'type'     => 'upload',
                            'store'    => 'url',
                            'default'  => ''
                        ),
                    )
                ),
                'register-page-agreement' => array(
                    'title'   => esc_html__('Registration Agreement', 'fast-plugin'),
                    'header'  => esc_html__('Setup user agreement before registration.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Add agreement checkbox', 'fast-plugin'),
                            'desc'    => esc_html__('A check box will need to be checked by the new user in order for the registration proccess to complete.', 'fast-plugin'),
                            'id'      => 'register_agreement_enabled',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'       => esc_html__('Agreement chekbox label', 'fast-plugin'),
                            'desc'       => esc_html__('The text label that will appear next to the agreement chebkbox.', 'fast-plugin'),
                            'id'         => 'register_agreement_label',
                            'type'       => 'text',
                            'store_default' => true,
                            'default'    => esc_html__('I\'ve read and accept the Privacy Policy', 'fast-plugin'),
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Agreement text', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the content that is shown under the agreement chebkbox.', 'fast-plugin'),
                            'id'            => 'register_agreement_text',
                            'type'          => 'editor',
                            'sanitize'      => 'html',
                            'store_default' => true,
                            'default'       => '<p>By signing up, you agree to our <a href="#" target="_blank">Terms of Service</a>. Learn how we collect and use your data in our <a href="#" target="_blank">Privacy Policy</a>.</p>'
                        ),
                    )
                ),
                'register-recaptcha' => array(
                    'title'   => esc_html__('Google reCAPTCHA', 'fast-plugin'),
                    'header'  => esc_html__('Add a reCAPTCHA check to the user registration page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'        => esc_html__('reCAPTCHA Site key', 'fast-plugin'),
                            'desc'        => esc_html__('Add your site key from https://www.google.com/recaptcha/', 'fast-plugin'),
                            'id'          => 'recaptcha_site_key',
                            'type'        => 'text',
                            'default'     => ''
                        ),
                        array(
                            'name'        => esc_html__('reCAPTCHA Secret key', 'fast-plugin'),
                            'desc'        => esc_html__('Add your secret key from https://www.google.com/recaptcha/', 'fast-plugin'),
                            'id'          => 'recaptcha_secret_key',
                            'type'        => 'text',
                            'default'     => ''
                        ),
                    )
                ),
            )
        ),
        array(
            'slug' => 'fast-plugin-login-page',
            'page' => array(
                'title' => esc_html__('Login Page Content', 'fast-plugin'),
            ),
            'menu' => array(
                'title' => esc_html__('Login Page', 'fast-plugin')
            ),
            'sections'   => array(
                'register-page' => array(
                    'title'   => esc_html__('Login Page', 'fast-plugin'),
                    'header'  => esc_html__('Set the content that is shown on the login page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Show Login Types', 'fast-plugin'),
                            'desc'    => esc_html__('Which login options to show on the login page.', 'fast-plugin'),
                            'id'      => 'login_page_login_types',
                            'select2' => true,
                            'type'    => 'select',
                            'options' => array(
                                'wordpress' => esc_html__('WordPress Login', 'fast-plugin'),
                                'envato'  => esc_html__('Envato Login', 'fast-plugin'),
                            ),
                            'default' => array(
                                'wordpress'
                            ),
                            'attributes' => array(
                                'class'    => 'fast-select2-option',
                                'multiple' => '',
                            )
                        ),
                        array(
                            'name'          => esc_html__('Login Page Title', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the title that is shown on the login page.', 'fast-plugin'),
                            'id'            => 'login_page_title',
                            'type'          => 'text',
                            'default'       => esc_html__('Support Login', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'  => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Login Page Content', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the content that is shown on the login page.', 'fast-plugin'),
                            'id'            => 'login_page_content',
                            'store_default' => true,
                            'type'          => 'editor',
                            'sanitize'      => 'html',
                            'default'       => '<p>Welcome to our support ticket system</p><p>Please login using the form</p>'
                        ),
                        array(
                            'name'     => esc_html__('Login Page Background Image', 'fast-plugin'),
                            'desc'     => esc_html__('Sets the background image that is shown on the login page.', 'fast-plugin'),
                            'id'       => 'login_page_image',
                            'type'     => 'upload',
                            'store'    => 'url',
                            'default'  => ''
                        ),
                    )
                ),
            )
        ),
        array(
            'slug' => 'fast-plugin-new-ticket-page',
            'page' => array(
                'title' => esc_html__('New Ticket Page', 'fast-plugin'),
            ),
            'menu' => array(
                'title' => esc_html__('New Ticket Page', 'fast-plugin')
            ),
            'sections'   => array(
                'new-ticket-general' => array(
                    'title'   => esc_html__('New Ticket Page', 'fast-plugin'),
                    'header'  => esc_html__('General options for the new ticket page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'     => esc_html__('New Ticket Page Background Image', 'fast-plugin'),
                            'desc'     => esc_html__('Sets the background image that is shown on the new ticket page.', 'fast-plugin'),
                            'id'       => 'new_ticket_page_image',
                            'type'     => 'upload',
                            'store'    => 'url',
                            'default'  => ''
                        ),
                    )
                ),
                'new-ticket-page' => array(
                    'title'   => esc_html__('Introduction Tab', 'fast-plugin'),
                    'header'  => esc_html__('Set the content that is shown on the first tab of the new ticket page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Show the introduction tab', 'fast-plugin'),
                            'desc'    => esc_html__('Turns on introduction tab on new ticket page.', 'fast-plugin'),
                            'id'      => 'intro_tab_show',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'          => esc_html__('Intro Tab Label', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the label that is shown on the first tab.', 'fast-plugin'),
                            'id'            => 'intro_tab_title',
                            'type'          => 'text',
                            'default'       => esc_html__('Introduction', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Intro Tab Title', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the title that is shown at the top of the first tab.', 'fast-plugin'),
                            'id'            => 'new_ticket_page_title',
                            'type'          => 'text',
                            'default'       => esc_html__('Create a new ticket', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Intro Tab Content', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the content that is shown on the first tab of the new ticket page.', 'fast-plugin'),
                            'id'            => 'new_ticket_page_content',
                            'type'          => 'editor',
                            'sanitize'      => 'html',
                            'default'       => '<p>Please fill in the form and describe as clearly as possible the problem you are having.</p><p>To help our support team please include a link to your website or a screenshot.</p>',
                            'store_default' => true,
                        ),
                    )
                ),
                'customer-step' => array(
                    'title'   => esc_html__('User Tab', 'fast-plugin'),
                    'header'  => esc_html__('Set the content that is shown on the user tab of the new ticket page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'          => esc_html__('User Tab Label', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the label that is shown on the customer tab.', 'fast-plugin'),
                            'id'            => 'user_tab_title',
                            'type'          => 'text',
                            'default'       => esc_html__('User', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                    )
                ),
                'new-ticket-enquiry' => array(
                    'title'   => esc_html__('Enquiry Tab', 'fast-plugin'),
                    'header'  => esc_html__('Set the content that is shown on the 2nd tab of the new ticket page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'          => esc_html__('Enquiry Tab Label', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the label that is shown on the first tab.', 'fast-plugin'),
                            'id'            => 'enquiry_tab_label',
                            'type'          => 'text',
                            'default'       => esc_html__('Enquiry', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Enquiry Tab Title', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the title that is shown above the options of the enquiry tab.', 'fast-plugin'),
                            'id'            => 'enquiry_tab_title',
                            'type'          => 'text',
                            'default'       => esc_html__('What do you need help with?', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('General Ticket Type Description', 'fast-plugin'),
                            'desc'          => esc_html__('If you use more than one ticket type this text will be used to allow users to select what type of ticket to create in the 2nd tab.', 'fast-plugin'),
                            'id'            => 'enquiry_tab_general_text',
                            'type'          => 'text',
                            'default'       => esc_html__('I have a general enquiry.', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Product Ticket Type Description', 'fast-plugin'),
                            'desc'          => esc_html__('If you use more than one ticket type this text will be used to allow users to select what type of ticket to create in the 2nd tab.', 'fast-plugin'),
                            'id'            => 'enquiry_tab_regular_text',
                            'type'          => 'text',
                            'default'       => esc_html__('I have a problem with a product.', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Envato Product Ticket Type Description', 'fast-plugin'),
                            'desc'          => esc_html__('If you use more than one ticket type this text will be used to allow users to select what type of ticket to create in the 2nd tab.', 'fast-plugin'),
                            'id'            => 'enquiry_tab_envato_text',
                            'type'          => 'text',
                            'default'       => esc_html__('I have a problem with a product I purchased on Themeforest.', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Product Ticket Type Description', 'fast-plugin'),
                            'desc'          => esc_html__('If you use more than one ticket type this text will be used to allow users to select what type of ticket to create in the 2nd tab.', 'fast-plugin'),
                            'id'            => 'enquiry_tab_woocommerce_text',
                            'type'          => 'text',
                            'default'       => esc_html__('I have a problem with an order.', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                    )
                ),
                'new-ticket-tab' => array(
                    'title'   => esc_html__('Ticket Tab', 'fast-plugin'),
                    'header'  => esc_html__('Set the content that is shown on the ticket tab of the new ticket page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'          => esc_html__('Ticket Tab Label', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the label that is shown on the ticket tab.', 'fast-plugin'),
                            'id'            => 'ticket_tab_label',
                            'type'          => 'text',
                            'default'       => esc_html__('Ticket', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Ticket Tab Description', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the text that is shown above the ticket form.', 'fast-plugin'),
                            'id'            => 'ticket_tab_text',
                            'type'          => 'text',
                            'default'       => esc_html__('Please fill in the form below and tell us as much information as you can about the issue you are having.', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                    )
                ),
                'new-ticket-product-tab' => array(
                    'title'   => esc_html__('Product Tab', 'fast-plugin'),
                    'header'  => esc_html__('Set the content that is shown on the product tab of the new ticket page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'          => esc_html__('Product Tab Label', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the label that is shown on the ticket tab.', 'fast-plugin'),
                            'id'            => 'product_tab_label',
                            'type'          => 'text',
                            'default'       => esc_html__('Product', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Product Tab Description', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the text that is shown above the ticket form.', 'fast-plugin'),
                            'id'            => 'product_tab_text',
                            'type'          => 'text',
                            'default'       => esc_html__('Please select a product that you need support for from the list below.', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                    )
                ),
                'new-ticket-category-tab' => array(
                    'title'   => esc_html__('Category Tab', 'fast-plugin'),
                    'header'  => esc_html__('Set the content that is shown on the category tab of the new ticket page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'          => esc_html__('Category Tab Label', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the label that is shown on the ticket tab.', 'fast-plugin'),
                            'id'            => 'category_tab_label',
                            'type'          => 'text',
                            'default'       => esc_html__('Category', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Category Tab Description', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the text that is shown above the ticket form.', 'fast-plugin'),
                            'id'            => 'category_tab_text',
                            'type'          => 'text',
                            'default'       => esc_html__('Please select a category that you need support for from the list below.', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                    )
                ),
                'new-ticket-extra-tab' => array(
                    'title'   => esc_html__('Extra Fields Tab', 'fast-plugin'),
                    'header'  => esc_html__('Set the content that is shown on the extra fields tab of the new ticket page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'          => esc_html__('Extra Fields Tab Label', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the label that is shown on the extra fields tab.', 'fast-plugin'),
                            'id'            => 'extra_fields_tab_label',
                            'type'          => 'text',
                            'default'       => esc_html__('Extra Information', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Extra Fields Tab Description', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the text that is shown above the extra fields form.', 'fast-plugin'),
                            'id'            => 'extra_fields_tab_text',
                            'type'          => 'text',
                            'default'       => esc_html__('Please provide us with any extra information you can.', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                    )
                ),
                'new-ticket-secure-tab' => array(
                    'title'   => esc_html__('Secure Fields Tab', 'fast-plugin'),
                    'header'  => esc_html__('Set the content that is shown on the secure fields tab of the new ticket page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'       => esc_html__('Secure Fields Tab Label', 'fast-plugin'),
                            'desc'       => esc_html__('Sets the label that is shown on the secure fields tab.', 'fast-plugin'),
                            'id'         => 'secure_fields_tab_label',
                            'type'       => 'text',
                            'default'    => esc_html__('Secure Information', 'fast-plugin'),
                            'store_default' => true,
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                        array(
                            'name'       => esc_html__('Secure Fields Tab Description', 'fast-plugin'),
                            'desc'       => esc_html__('Sets the text that is shown above the secure fields form.', 'fast-plugin'),
                            'id'         => 'secure_fields_tab_text',
                            'type'       => 'text',
                            'default'    => esc_html__('The information you provide below will be stored securely.', 'fast-plugin'),
                            'store_default' => true,
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                    )
                ),
                'new-ticket-envato-tab' => array(
                    'title'   => esc_html__('Envato Purchase Tab', 'fast-plugin'),
                    'header'  => esc_html__('Set the content that is shown on the envato fields tab of the new ticket page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'          => esc_html__('Envato Purchase Tab Label', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the label that is shown on the envato purchase tab.', 'fast-plugin'),
                            'id'            => 'envato_tab_label',
                            'type'          => 'text',
                            'default'       => esc_html__('Envato Purchase', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('Envato Purchase Fields Tab Description', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the text that is shown above the envato purchase form.', 'fast-plugin'),
                            'id'            => 'envato_tab_text',
                            'type'          => 'text',
                            'default'       => esc_html__('Please select a purchase that you need support for from the list below.', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                    )
                ),
                'new-ticket-wc-order-tab' => array(
                    'title'   => esc_html__('WooCommerce Order Tab', 'fast-plugin'),
                    'header'  => esc_html__('Set the content that is shown on the WooCommerce order tab of the new ticket page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'          => esc_html__('WooCommerce Order Tab Label', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the label that is shown on the WooCommerce order tab.', 'fast-plugin'),
                            'id'            => 'wc_order_tab_label',
                            'type'          => 'text',
                            'default'       => esc_html__('Order', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('WooCommerce Order Tab Description', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the text that is shown above the WooCommerce Order selection.', 'fast-plugin'),
                            'id'            => 'wc_order_tab_text',
                            'type'          => 'text',
                            'default'       => esc_html__('Please select your order you need help with from the list below.', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                    )
                ),
                'new-ticket-wc-product-tab' => array(
                    'title'   => esc_html__('WooCommerce Product Tab', 'fast-plugin'),
                    'header'  => esc_html__('Set the content that is shown on the WooCommerce product tab of the new ticket page.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'          => esc_html__('WooCommerce Product Tab Label', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the label that is shown on the WooCommerce product tab.', 'fast-plugin'),
                            'id'            => 'wc_product_tab_label',
                            'type'          => 'text',
                            'default'       => esc_html__('Product', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                        array(
                            'name'          => esc_html__('WooCommerce Product Tab Description', 'fast-plugin'),
                            'desc'          => esc_html__('Sets the text that is shown above the WooCommerce product selection.', 'fast-plugin'),
                            'id'            => 'wc_product_tab_text',
                            'type'          => 'text',
                            'default'       => esc_html__('Please select the product(s) that you need support for from the list below.', 'fast-plugin'),
                            'store_default' => true,
                            'attributes'    => array(
                                'class'     => 'widefat'
                            )
                        ),
                    )
                ),
            )
        ),
        array(
            'slug' => 'fast-plugin-extra-fields',
            'page' => array(
                'title' => esc_html__('Ticket Extra Fields', 'fast-plugin'),
            ),
            'menu' => array(
                'title' => esc_html__('Extra Fields', 'fast-plugin')
            ),
            'sections'   => array(
                'extra-fields-options' => array(
                    'title'   => esc_html__('Extra Field Options', 'fast-plugin'),
                    'header'  => esc_html__('Sets up how extra fields are shown to customers and agents.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Enable Extra Fields', 'fast-plugin'),
                            'desc'    => esc_html__('Turns on extra fields for tickets.', 'fast-plugin'),
                            'id'      => 'extra_fields_enabled',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        )
                    )
                ),
                'secure-fields-options' => array(
                    'title'   => esc_html__('Secure Field Options', 'fast-plugin'),
                    'header'  => esc_html__('Sets up how secure fields are shown to customers and agents.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Enable Secure Fields', 'fast-plugin'),
                            'desc'    => esc_html__('Turns on secure fields for tickets.', 'fast-plugin'),
                            'id'      => 'secure_fields_enabled',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'       => esc_html__('Secure Field Access URL', 'fast-plugin'),
                            'desc'       => esc_html__('Sets the external URL that agents can access secure information.', 'fast-plugin'),
                            'id'         => 'secure_fields_external_url',
                            'type'       => 'text',
                            'default'    => '',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                    )
                )
            )
        ),
        array(
            'slug' => 'fast-plugin-roles-capabilities',
            'page' => array(
                'title' => esc_html__('Roles & Capabilities', 'fast-plugin'),
            ),
            'menu' => array(
                'title' => esc_html__('User Permissions', 'fast-plugin')
            ),
            'sections'   => array(
                'capabilities' => array(
                    'title'   => esc_html__('Assign Capabilities to Roles', 'fast-plugin'),
                    'header'  => esc_html__('Sets the capabilities for all user roles.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'             => esc_html__('Customer Role', 'fast-plugin'),
                            'desc'             => esc_html__('Sets capabilites that a customer will have.', 'fast-plugin'),
                            'id'               => 'fast_customer_capabilities',
                            'select2'          => true,
                            'type'             => 'select',
                            'filter'           => 'fast_filter_ticket_capabilities',
                            'default'          => apply_filters('fast_filter_default_fast_customer_capabilities', array()),
                            'attributes'       => array(
                                'class' => 'fast-select2-option',
                                'multiple' => '',
                            )
                        ),
                        array(
                            'name'             => esc_html__('Agent Role', 'fast-plugin'),
                            'desc'             => esc_html__('Sets capabilites that an agent will have.', 'fast-plugin'),
                            'id'               => 'fast_agent_capabilities',
                            'type'             => 'select',
                            'select2'          => true,
                            'filter'           => 'fast_filter_ticket_capabilities',
                            'default'          => apply_filters('fast_filter_default_fast_agent_capabilities', array()),
                            'attributes'       => array(
                                'class' => 'fast-select2-option',
                                'multiple' => '',
                            )
                        ),
                        array(
                            'name'             => esc_html__('Manager Role', 'fast-plugin'),
                            'desc'             => esc_html__('Sets capabilites that a manager will have.', 'fast-plugin'),
                            'id'               => 'fast_manager_capabilities',
                            'type'             => 'select',
                            'select2'          => true,
                            'filter'           => 'fast_filter_ticket_capabilities',
                            'default'          => apply_filters('fast_filter_default_fast_manager_capabilities', array()),
                            'attributes'       => array(
                                'class' => 'fast-select2-option',
                                'multiple' => '',
                            )
                        )
                    )
                ),
                'user-rest' => array(
                    'title'   => esc_html__('REST API', 'fast-plugin'),
                    'header'  => esc_html__('Options for users in the REST API.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name' => esc_html__('Remove users endpoints from Rest API', 'fast-plugin'),
                            'desc' => esc_html__('Removes users/ endpoints from rest API.', 'fast-plugin'),
                            'id'   => 'remove_users_endpoints',
                            'type' => 'select',
                            'options' => array(
                                'on'  => esc_html__('On', 'fast-plugin'),
                                'off' => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on'
                        ),
                    )
                )
            )
        ),
        array(
            'slug' => 'fast-plugin-email-options',
            'page' => array(
                'title' => esc_html__('Email Options', 'fast-plugin'),
            ),
            'menu' => array(
                'title' => esc_html__('Email Options', 'fast-plugin')
            ),
            'sections'   => array(
                'ticket-email-general' => array(
                    'title'   => esc_html__('Ticket Email Template Options', 'fast-plugin'),
                    'header'  => '<p>Set the template options for all ticket emails. Email content may contain shortcodes which will be replaced by the respective email information. See <a target="_blank" href="http://help.oxygenna.com/plugins/fast/documentation.html#customizing-emails">the plugin documentation for details</a> about the shortcodes.</p>',
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Email Background Color', 'fast-plugin'),
                            'id'      => 'ticket_email_template_bg_color',
                            'desc'    => esc_html__('Sets the background color of the email.', 'fast-plugin'),
                            'default' => '#FBFAFA',
                            'type'    => 'color',
                            'format'  => 'hex'
                        ),
                        array(
                            'name'     => esc_html__('Logo Header Image', 'fast-plugin'),
                            'desc'     => esc_html__('Sets the logo image that is shown in the header at the top of them email.', 'fast-plugin'),
                            'id'       => 'ticket_email_logo_image',
                            'type'     => 'upload',
                            'store'    => 'url',
                            'default'  => ''
                        ),
                        array(
                            'name'    => esc_html__('Logo Header Background Color', 'fast-plugin'),
                            'id'      => 'ticket_email_logo_image_bg_color',
                            'desc'    => esc_html__('Sets the background color of the email header.', 'fast-plugin'),
                            'default' => '#FFFFFF',
                            'type'    => 'color',
                            'format'  => 'hex'
                        ),
                        array(
                            'name'     => esc_html__('Email Header Text', 'fast-plugin'),
                            'desc'     => esc_html__('Sets the text that is shown at the top of the email below the logo.', 'fast-plugin'),
                            'id'       => 'ticket_email_logo_title',
                            'type'     => 'text',
                            'store_default' => true,
                            'default'  => 'FAST Support Ticket Plugin',
                            'sanitize' => 'html',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                        array(
                            'name'    => esc_html__('Header Text Color', 'fast-plugin'),
                            'id'      => 'ticket_email_logo_title_color',
                            'desc'    => esc_html__('Sets the text color of the title area.', 'fast-plugin'),
                            'default' => '#555555',
                            'type'    => 'color',
                            'format'  => 'hex'
                        ),
                        array(
                            'name'    => esc_html__('Header Background Color', 'fast-plugin'),
                            'id'      => 'ticket_email_logo_title_bg_color',
                            'desc'    => esc_html__('Sets the background color of the text area.', 'fast-plugin'),
                            'default' => '#FFFFFF',
                            'type'    => 'color',
                            'format'  => 'hex'
                        ),
                        array(
                            'name'    => esc_html__('Divider Color', 'fast-plugin'),
                            'id'      => 'ticket_email_divider_color',
                            'desc'    => esc_html__('Sets the color of the divider that separates the title from the email content. The divider will appear only if there is a logo image or header text set.', 'fast-plugin'),
                            'default' => '#FBFAFA',
                            'type'    => 'color',
                            'format'  => 'hex'
                        ),
                        array(
                            'name'    => esc_html__('Border Radius', 'fast-plugin'),
                            'id'      => 'ticket_email_border_radius',
                            'desc'    => esc_html__('Sets the border radius on the content area of the email.', 'fast-plugin'),
                            'type'    => 'slider',
                            'default'   => 4,
                            'attr'      => array(
                                'max'       => 20,
                                'min'       => 0,
                                'step'      => 1
                            )
                        ),
                        array(
                            'name'    => esc_html__('Button Text Color', 'fast-plugin'),
                            'id'      => 'ticket_email_button_color',
                            'desc'    => esc_html__('Sets the text color of the button.', 'fast-plugin'),
                            'default' => '#FFFFFF',
                            'type'    => 'color',
                            'format'  => 'hex'
                        ),
                        array(
                            'name'    => esc_html__('Button Background Color', 'fast-plugin'),
                            'id'      => 'ticket_email_button_bg_color',
                            'desc'    => esc_html__('Sets the background color of the button.', 'fast-plugin'),
                            'default' => '#E86468',
                            'type'    => 'color',
                            'format'  => 'hex'
                        ),
                        array(
                            'name'        => esc_html__('Above ticket button text', 'fast-plugin'),
                            'desc'        => esc_html__('Set the text that shows before the ticket button.', 'fast-plugin'),
                            'id'          => 'ticket_above_ticket_button',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => '',
                            'sanitize'    => 'html',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                        array(
                            'name'        => esc_html__('Email Template Footer Text', 'fast-plugin'),
                            'desc'        => esc_html__('Set the footer text that is shown inthe email template.', 'fast-plugin'),
                            'id'          => 'ticket_email_footer',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => 'Thanks for using our support.',
                            'sanitize'    => 'html',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                        array(
                            'name'    => esc_html__('Footer Text Color', 'fast-plugin'),
                            'id'      => 'ticket_email_footer_color',
                            'desc'    => esc_html__('Sets the text color of the footer.', 'fast-plugin'),
                            'default' => '#E0A8AA',
                            'type'    => 'color',
                            'format'  => 'hex'
                        ),
                    )
                ),
                'ticket-email-header' => array(
                    'title'   => esc_html__('Ticket Email Header Options', 'fast-plugin'),
                    'header'  => '<p>Set the header options for all ticket emails. The options configure the email name and address that emails should be sent from. Default name and email will be used if options are left blank.</p>',
                    'fields' => array(
                        array(
                            'name'     => esc_html__('Email Sender Name', 'fast-plugin'),
                            'desc'     => esc_html__('Sets the "from name" field in all emails sent using the plugin.', 'fast-plugin'),
                            'id'       => 'ticket_email_from_name',
                            'type'     => 'text',
                            'default'  => '',
                            'sanitize' => 'html',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                        array(
                            'name'     => esc_html__('Email Sender Address', 'fast-plugin'),
                            'desc'     => esc_html__('Sets the "from email address" field in all emails sent using the plugin.', 'fast-plugin'),
                            'id'       => 'ticket_email_from_address',
                            'type'     => 'text',
                            'default'  => '',
                            'sanitize' => 'html',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        )
                    )
                ),
                'ticket-new-email' => array(
                    'title'   => esc_html__('New Ticket Email', 'fast-plugin'),
                    'header'  => esc_html__('Set the content of a new ticket email.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Send Email on New ticket', 'fast-plugin'),
                            'desc'    => esc_html__('Send an email to the assigned agent upon ticket creation', 'fast-plugin'),
                            'id'      => 'email_new_ticket',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'        => esc_html__('New Ticket Subject', 'fast-plugin'),
                            'desc'        => esc_html__('Sets the subject of the new email for the agent.', 'fast-plugin'),
                            'id'          => 'ticket_new_email_subject',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => 'New ticket [ticket_id] from [sender_name] for [product]',
                            'sanitize'    => 'html',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                        array(
                            'name'        => esc_html__('New Ticket Content', 'fast-plugin'),
                            'desc'        => esc_html__('Set the content for the new email for the agent.', 'fast-plugin'),
                            'id'          => 'ticket_new_email_content',
                            'store_default' => true,
                            'type'        => 'editor',
                            'default'     => '<p><strong>You have a new support ticket from [sender_name]</strong></p><p>Here is what they said:</p>[comment_content]',
                            'sanitize'    => 'html'
                        ),
                    )
                ),
                'ticket-reply-email' => array(
                    'title'   => esc_html__('Ticket Reply Email', 'fast-plugin'),
                    'header'  => esc_html__('Set the content of a ticket reply email.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Send Email on Client Reply', 'fast-plugin'),
                            'desc'    => esc_html__('Send an Email to agent upon client\'s reply.', 'fast-plugin'),
                            'id'      => 'email_client_reply',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'    => esc_html__('Send Email on Agent Reply', 'fast-plugin'),
                            'desc'    => esc_html__('Send an Email to client upon agent\'s reply.', 'fast-plugin'),
                            'id'      => 'email_agent_reply',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'        => esc_html__('Ticket Reply Subject', 'fast-plugin'),
                            'desc'        => esc_html__('Set the reply email subject.', 'fast-plugin'),
                            'id'          => 'ticket_reply_email_subject',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => 'Comment from [sender_name] on ticket [ticket_id]',
                            'sanitize'    => 'html',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                        array(
                            'name'        => esc_html__('Ticket Reply Content', 'fast-plugin'),
                            'desc'        => esc_html__('Sets the email content for reply email.', 'fast-plugin'),
                            'id'          => 'ticket_reply_email_content',
                            'type'        => 'editor',
                            'store_default' => true,
                            'default'     => '<p><strong>There is a new comment from [sender_name] in the support ticket [ticket_title]</strong></p>[comment_content]',
                            'sanitize'    => 'html'
                        ),
                    )
                ),
                'ticket-assign-email' => array(
                    'title'   => esc_html__('Ticket Assign Email', 'fast-plugin'),
                    'header'  => esc_html__('Set the content of a ticket assign email.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Send Email on Ticket Assignment', 'fast-plugin'),
                            'desc'    => esc_html__('Send an email upon ticket assignment.', 'fast-plugin'),
                            'id'      => 'email_ticket_assignment',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'        => esc_html__('Ticket Assign Subject', 'fast-plugin'),
                            'desc'        => esc_html__('Sets the subject of the mail that is sent when a ticket is assigned to an agent', 'fast-plugin'),
                            'id'          => 'ticket_assign_email_subject',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => '[sender_name] assigned you ticket [ticket_id]',
                            'sanitize'    => 'html',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                        array(
                            'name'        => esc_html__('Ticket Assign Content', 'fast-plugin'),
                            'desc'        => esc_html__('Set the email content for the ticket assigned email for the agent.', 'fast-plugin'),
                            'id'          => 'ticket_assign_email_content',
                            'type'        => 'editor',
                            'store_default' => true,
                            'default'     => '<p>[sender_name] just assigned you the ticket [ticket_title] from [ticket_author]</p>',
                            'sanitize'    => 'html'
                        ),
                    )
                ),
                'user-new-email' => array(
                    'title'   => esc_html__('New User Email', 'fast-plugin'),
                    'header'  => esc_html__('Set the content of a new user email.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Send User Registration email on user registration', 'fast-plugin'),
                            'desc'    => esc_html__('Send an email with new user details when a user registers using the FAST register user page.', 'fast-plugin'),
                            'id'      => 'email_new_user_email_registration',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'    => esc_html__('Send User Registration email on new user email pipe message', 'fast-plugin'),
                            'desc'    => esc_html__('Send an email with new user details when an unknown user sends an email to the email pipe address.', 'fast-plugin'),
                            'id'      => 'email_new_user',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'        => esc_html__('New User Subject', 'fast-plugin'),
                            'desc'        => esc_html__('Sets the subject of the new user email.', 'fast-plugin'),
                            'id'          => 'user_new_email_subject',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => 'Thank you for registering with our support.',
                            'sanitize'    => 'html',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                        array(
                            'name'        => esc_html__('New User Content', 'fast-plugin'),
                            'desc'        => esc_html__('Set the content for the new user email.', 'fast-plugin'),
                            'id'          => 'user_new_email_content',
                            'type'        => 'editor',
                            'store_default' => true,
                            'default'     => '<p><strong>Welcome to our support.</strong></p><p>Here are your new user details:</p>',
                            'sanitize'    => 'html'
                        ),
                    )
                ),
            )
        ),
        array(
            'slug' => 'fast-plugin-envato',
            'page' => array(
                'title' => esc_html__('Envato Integration', 'fast-plugin'),
            ),
            'menu' => array(
                'title' => esc_html__('Envato Integration', 'fast-plugin')
            ),
            'sections'   => array(
                'envato-credentials' => array(
                    'title'   => esc_html__('Envato User Login / Registration', 'fast-plugin'),
                    'header'  => '<p>The options below will enable users to register and login using their Envato login credentials.</p><p>For how to set this up <a target="_blank" href="http://help.oxygenna.com/plugins/fast/documentation.html#envato-intergration">please see the plugin documentation</a></p>',
                    'fields' => array(
                        array(
                            'name'        => esc_html__('Enable Envato Login', 'fast-plugin'),
                            'desc'        => esc_html__('Allow users to log into the ticket system using their Envato credentials.', 'fast-plugin'),
                            'id'          => 'envato_login',
                            'type'        => 'select',
                            'options'     => array(
                                'on'  => esc_html__('On', 'fast-plugin'),
                                'off' => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default'     => 'off',
                        ),
                        array(
                            'name'        => esc_html__('OAuth Client ID', 'fast-plugin'),
                            'desc'        => esc_html__('This ID uniquely identifies your app.', 'fast-plugin'),
                            'id'          => 'envato_client_id',
                            'type'        => 'text',
                            'default'     => '',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                        array(
                            'name'        => esc_html__('Client Secret Key', 'fast-plugin'),
                            'desc'        => esc_html__('To use Envato login you must register an app with the Envato API', 'fast-plugin'),
                            'id'          => 'envato_client_secret',
                            'type'        => 'text',
                            'default'     => '',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                    )
                ),
                'envato-products' => array(
                    'title'   => esc_html__('Link Envato Products', 'fast-plugin'),
                    'header'  => '<p>The options below will allow you to link your <a href="' . admin_url('edit-tags.php?taxonomy=fast_product&post_type=fast_ticket') . '">ticket products</a> to Envato Products.</p><p>You will then be able to verify purchases on ticket creation and Ticket Agents will be allowed to verify purchases at any time.</p>',
                    'fields' => array(
                        array(
                            'name'        => esc_html__('Link Envato Products to ticket products', 'fast-plugin'),
                            'desc'        => esc_html__('Link your ticket products with Envato marketplace items.', 'fast-plugin'),
                            'id'          => 'envato_products',
                            'type'        => 'select',
                            'options'     => array(
                                'on'  => esc_html__('On', 'fast-plugin'),
                                'off' => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default'     => 'off',
                        ),
                        array(
                            'name'        => esc_html__('Envato Username', 'fast-plugin'),
                            'desc'        => esc_html__('Enter your Envato marketplace username used to verify purchases.', 'fast-plugin'),
                            'id'          => 'envato_username',
                            'type'        => 'text',
                            'default'     => '',
                        ),
                        array(
                            'name'        => esc_html__('Envato Token', 'fast-plugin'),
                            'desc'        => esc_html__('Enter your Envato personal token used to verify purchases.', 'fast-plugin'),
                            'id'          => 'envato_token',
                            'type'        => 'text',
                            'default'     => '',
                        ),
                        array(
                            'name'        => esc_html__('Load Envato Products', 'fast-plugin'),
                            'desc'        => esc_html__('Click the button to contact the Envato API and load all your products.', 'fast-plugin'),
                            'id'          => 'envato_products_load',
                            'button_text' => esc_html__('Click to load all your Envato products', 'fast-plugin'),
                            'type'        => 'button',
                            'attributes'  => array(
                                'class' => 'button',
                                'type'  => 'submit'
                            )
                        ),
                        array(
                            'name'    => esc_html__('Allow users to add purchase codes.', 'fast-plugin'),
                            'desc'    => esc_html__('Shows and hides the add purchase code input and button in the Envato step of create ticket.', 'fast-plugin'),
                            'id'      => 'envato_allow_enter_purchases',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'    => esc_html__('Purchase codes can only be used by one user.', 'fast-plugin'),
                            'desc'    => esc_html__('When a purchase code is entered in the add purchase field it will be checked to see if it is already in use in the system.', 'fast-plugin'),
                            'id'      => 'envato_check_duplicate_purchase_codes',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'    => esc_html__('Refresh purchases every', 'fast-plugin'),
                            'desc'    => esc_html__('Sets how often a purchase for a ticket gets refreshed from envato.', 'fast-plugin'),
                            'id'      => 'envato_purchase_code_refresh_interval',
                            'type'    => 'slider',
                            'default' => 24,
                            'postfix' => esc_html__('hours.', 'fast-plugin'),
                            'attr'      => array(
                                'max'       => 100,
                                'min'       => 1,
                                'step'      => 1
                            )
                        ),
                    )
                ),
                'envato-customer-checks' => array(
                    'title'   => esc_html__('Customer Checks', 'fast-plugin'),
                    'header'  => '<p>Sets the checks that the app will make for Envato Customers</p>',
                    'fields' => array(
                        array(
                            'name'        => esc_html__('Customers must have support to create tickets', 'fast-plugin'),
                            'desc'        => esc_html__('Only allow creation of ticket if support period has not expired.', 'fast-plugin'),
                            'id'          => 'envato_create_ticket_when_supported',
                            'type'        => 'select',
                            'options'     => array(
                                'on'  => esc_html__('On', 'fast-plugin'),
                                'off' => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default'     => 'on',
                        ),
                        array(
                            'name'        => esc_html__('Customers must have support to reply to tickets', 'fast-plugin'),
                            'desc'        => esc_html__('Only allow reply to a ticket if support period has not expired.', 'fast-plugin'),
                            'id'          => 'envato_reply_to_ticket_when_supported',
                            'type'        => 'select',
                            'options'     => array(
                                'on'  => esc_html__('On', 'fast-plugin'),
                                'off' => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default'     => 'on',
                        ),
                        array(
                            'name'       => esc_html__('Envato support update url', 'fast-plugin'),
                            'desc'       => esc_html__('Sets the external URL that links to envato support update.', 'fast-plugin'),
                            'id'         => 'envato_support_update_url',
                            'type'       => 'text',
                            'default'    => 'http://market.envato.com/',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        )
                    )
                )
            )
        ),
        array(
            'slug' => 'fast-plugin-slack',
            'page' => array(
                'title' => esc_html__('Slack Integration Options', 'fast-plugin'),
            ),
            'menu' => array(
                'title' => esc_html__('Slack Integration', 'fast-plugin')
            ),
            'sections'   => array(
                'slack-integration' => array(
                    'title'   => esc_html__('Slack Integration', 'fast-plugin'),
                    'header'  => '<p>The options below will enable users to integrate the ticketing system with Slack in order to allow automatic notifications to your Slack channel.</p><p>Full instructions are in the <a target="_blank" href="http://help.oxygenna.com/plugins/fast/documentation.html#slack-intergration">Slack section of the documentation</a></p>',
                    'fields' => array(
                        array(
                            'name'       => esc_html__('Webhook URL', 'fast-plugin'),
                            'desc'       => esc_html__('Enter the webhook URL of your Slack app in order to dispatch messages to Slack.', 'fast-plugin'),
                            'id'         => 'slack_webhook',
                            'type'       => 'text',
                            'default'    => '',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                    )
                ),
                'ticket-new-slack' => array(
                    'title'   => esc_html__('New Ticket Notification', 'fast-plugin'),
                    'header'  => esc_html__('Set the content of a new ticket notification.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Notify on New ticket', 'fast-plugin'),
                            'desc'    => esc_html__('Dispatch a message to Slack upon ticket creation', 'fast-plugin'),
                            'id'      => 'slack_new_ticket',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'        => esc_html__('New Ticket Notification Content', 'fast-plugin'),
                            'desc'        => esc_html__('Set the content for the new ticket notification.', 'fast-plugin'),
                            'id'          => 'ticket_new_slack_content',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => 'There is a new support ticket for `[product]` from `[sender_name]`. Here is what they said: ```[comment_content]``` View the ticket here: [slack_link]',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                    )
                ),
                'ticket-reply-slack' => array(
                    'title'   => esc_html__('Ticket Reply Notification', 'fast-plugin'),
                    'header'  => esc_html__('Set the content of a ticket reply notification.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Notify on Client Reply', 'fast-plugin'),
                            'desc'    => esc_html__('Dispatch a message to Slack upon client\'s reply.', 'fast-plugin'),
                            'id'      => 'slack_client_reply',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'    => esc_html__('Notify on Agent Reply', 'fast-plugin'),
                            'desc'    => esc_html__('Dispatch a message to Slack upon agent\'s reply.', 'fast-plugin'),
                            'id'      => 'slack_agent_reply',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'        => esc_html__('Ticket Reply Notification Content', 'fast-plugin'),
                            'desc'        => esc_html__('Sets the slack notification content for reply email.', 'fast-plugin'),
                            'id'          => 'ticket_reply_slack_content',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => 'There is a new comment for `[product]` from `[sender_name]` in the support ticket `[ticket_title]`. ```[comment_content]```  View the ticket here: [slack_link]',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                    )
                ),
                'ticket-assign-slack' => array(
                    'title'   => esc_html__('Ticket Assign Notification', 'fast-plugin'),
                    'header'  => esc_html__('Set the content of a ticket assign notification.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Notify on Ticket Assignment', 'fast-plugin'),
                            'desc'    => esc_html__('Dispatch a message to Slack upon ticket assignment.', 'fast-plugin'),
                            'id'      => 'slack_ticket_assignment',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'        => esc_html__('Ticket Assign Content', 'fast-plugin'),
                            'desc'        => esc_html__('Set the content for the ticket assigned notification for the agent. [comment_content] and [comment_author] are not applicable here', 'fast-plugin'),
                            'id'          => 'ticket_assign_slack_content',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => '`[sender_name]` just reassigned the ticket `[ticket_title]` from `[ticket_author]`',
                            'attributes' => array(
                                'class'  => 'widefat'
                            )
                        ),
                    )
                )
            )
        ),
        array(
            'slug' => 'fast-plugin-woocommerce',
            'page' => array(
                'title' => esc_html__('WooCommerce Options', 'fast-plugin'),
            ),
            'menu' => array(
                'title' => esc_html__('WooCommerce', 'fast-plugin')
            ),
            'sections'   => array(
                'woocommerce-integration' => array(
                    'title'   => esc_html__('WooCommerce MyAccount Page', 'fast-plugin'),
                    'header'  => esc_html__('Sets the options for the WooCommerce Customer MyAccount page', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Add Support Link To My Account Menu', 'fast-plugin'),
                            'desc'    => esc_html__('Adds a link to the support tickets to customers MyAccount menu.', 'fast-plugin'),
                            'id'      => 'woocommerce_support_tickets_link',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'    => esc_html__('Add Support Button To My Orders Page', 'fast-plugin'),
                            'desc'    => esc_html__('Adds a link to the support tickets to customers MyAccount page.', 'fast-plugin'),
                            'id'      => 'woocommerce_support_tickets_order_button',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                    )
                ),
                'woocommerce-agents' => array(
                    'title'   => esc_html__('Fallback WooCommerce Agent(s)', 'fast-plugin'),
                    'header'  => esc_html__('Set the agent or agents that will handle your WooCommerce support tickets if no agents have been set for the product or product category.  Each agent will be assigned a ticket in turn.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'             => esc_html__('Default WooCommerce Agent(s)', 'fast-plugin'),
                            'id'               => 'woocommerce_agents',
                            'desc'             => esc_html__('Sets the fast agents that will handle all woocommerce support tickets.', 'fast-plugin'),
                            'type'             => 'select',
                            'filter'           => 'fast_filter_ticket_agent_ids_values',
                            'select2'          => true,
                            'default'          => array(),
                            'attributes'       => array(
                                'placeholder' => esc_html__('Select ticket agents who will handle woocommerce support tickets', 'fast-plugin'),
                                'class'       => 'fast-select2-option',
                                'multiple'    => '',
                            ),
                        )
                    )
                ),
                'woocommerce-existing' => array(
                    'title'   => esc_html__('Existing WooCommerce Site', 'fast-plugin'),
                    'header'  => esc_html__('Utilities for adding FAST to an existing WooCommerce Site', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Add FAST Customer role to all existing WooCommerce Customer Users', 'fast-plugin'),
                            'desc'    => esc_html__('This will allow all the customers that registered before you installed FAST to access the FAST support ticket system.', 'fast-plugin'),
                            'id'      => 'woocommerce_add_fast_customer_roles',
                            'type'    => 'button',
                            'button_text' => esc_html__('Add FAST Customer Role to all customers', 'fast-plugin'),
                            'attributes'  => array(
                                'class' => 'button',
                                'value' => 'woocommerce_add_fast_customer_roles'
                            )
                        ),
                    )
                ),
                'woocommerce-orders-supported' => array(
                    'title'   => esc_html__('Supported WooCommerce Order Statuses', 'fast-plugin'),
                    'header'  => esc_html__('Set the WooCommerce order statuses which you want your customers to create tickets for.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'        => esc_html__('Supported Order Statuses', 'fast-plugin'),
                            'desc'        => esc_html__('Sets the order status. WooCommerce offers the following order statuses by default: any, pending, processing, on-hold, completed, cancelled, refunded and failed', 'fast-plugin'),
                            'id'          => 'woocommerce_order_statuses',
                            'type'        => 'select',
                            'default'     => 'any',
                            'options' => array(
                                'any' => esc_html('Any', 'fast-plugin'),
                                'pending' => esc_html('Pending', 'fast-plugin'),
                                'processing' => esc_html('Processing', 'fast-plugin'),
                                'on-hold' => esc_html('On Hold', 'fast-plugin'),
                                'completed' => esc_html('Completed', 'fast-plugin'),
                                'cancelled' => esc_html('Cancelled', 'fast-plugin'),
                                'refunded' => esc_html('Refunded', 'fast-plugin'),
                                'failed' => esc_html('Failed', 'fast-plugin'),
                            ),
                        ),
                    )
                ),
            )
        ),
        array(
            'slug' => 'fast-plugin-email-pipe',
            'page' => array(
                'title' => esc_html__('Email Piping Options', 'fast-plugin'),
            ),
            'menu' => array(
                'title' => esc_html__('Email Piping', 'fast-plugin')
            ),
            'sections'   => array(
                'email-piping' => array(
                    'title'   => esc_html__('Email Piping', 'fast-plugin'),
                    'header'  => esc_html__('Setup FAST to allow use emails to respond / create tickets.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Enable Email Piping', 'fast-plugin'),
                            'desc'    => esc_html__('Turns on email piping.', 'fast-plugin'),
                            'id'      => 'emailp_enabled',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'    => esc_html__('Check Emails Every', 'fast-plugin'),
                            'desc'    => esc_html__('Set the time period to wait to check emails.', 'fast-plugin'),
                            'id'      => 'emailp_timespan',
                            'type'    => 'select',
                            'filter'  => 'fast_filter_cron_time_spans',
                            'default' => 'ten_minutes',
                        ),
                    )
                ),
                'email-piping-replies' => array(
                    'title'   => esc_html__('Reply to tickets', 'fast-plugin'),
                    'header'  => esc_html__('Setup FAST to allow users to reply to ticket emails.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Enable email replies to tickets', 'fast-plugin'),
                            'desc'    => esc_html__('Users can reply to emails to respond to tickets.', 'fast-plugin'),
                            'id'      => 'emailp_reply_enabled',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                    )
                ),
                'email-piping-new' => array(
                    'title'   => esc_html__('Create tickets with email', 'fast-plugin'),
                    'header'  => esc_html__('Setup FAST to allow customers to create tickets with email.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Enable create ticket with email', 'fast-plugin'),
                            'desc'    => esc_html__('Users can create tickets using email.', 'fast-plugin'),
                            'id'      => 'emailp_create_enabled',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'        => esc_html__('Default New Ticket Product', 'fast-plugin'),
                            'desc'        => esc_html__('Select a product that will be assigned when user creates a ticket via email.', 'fast-plugin'),
                            'id'          => 'emailp_default_product',
                            'type'        => 'select',
                            'empty_label' => esc_html__('Not Set', 'fast-plugin'),
                            'filter'      => 'fast_filter_ticket_product_ids_values',
                            'default'     => '',
                        ),
                        array(
                            'name'        => esc_html__('Default New Ticket Category', 'fast-plugin'),
                            'desc'        => esc_html__('Select a category that will be assigned when user creates a ticket via email.', 'fast-plugin'),
                            'id'          => 'emailp_default_category',
                            'type'        => 'select',
                            'empty_label' => esc_html__('Not Set', 'fast-plugin'),
                            'filter'      => 'fast_filter_ticket_category_ids_values',
                            'default'     => '',
                        ),
                    )
                ),
                'email-piping-header' => array(
                    'title'   => esc_html__('Use Reply-To Email Header', 'fast-plugin'),
                    'header'  => esc_html__('Make FAST use the Reply-To header of an incoming email when creating users and tickets through email piping. Fast will prioritize the value in the Reply-To header when figuring out who the email sender is.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Use Reply-To header', 'fast-plugin'),
                            'desc'    => esc_html__('Make fast prioritize the Reply-To header field value when figuring out who the email sender is.', 'fast-plugin'),
                            'id'      => 'emailp_use_reply_to_header',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        )
                    )
                ),
                'email-piping-imap' => array(
                    'title'   => esc_html__('Email IMAP Details', 'fast-plugin'),
                    'header'  => esc_html__('Enter your IMAP server details so FAST can connect to your email', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Email Address Username', 'fast-plugin'),
                            'desc'    => esc_html__('Email address username to login to email account.', 'fast-plugin'),
                            'id'      => 'emailp_username',
                            'type'    => 'text',
                            'default' => '',
                        ),
                        array(
                            'name'    => esc_html__('Email Address Password', 'fast-plugin'),
                            'desc'    => esc_html__('Email address password to login to email account.', 'fast-plugin'),
                            'id'      => 'emailp_password',
                            'type'    => 'text',
                            'default' => '',
                            'sanitize' => 'password',
                            'attributes'  => array(
                                'type' => 'password',
                            )
                        ),
                        array(
                            'name'    => esc_html__('IMAP Server URL', 'fast-plugin'),
                            'desc'    => esc_html__('Email server url address.', 'fast-plugin'),
                            'id'      => 'emailp_url',
                            'type'    => 'text',
                            'default' => '',
                        ),
                        array(
                            'name'    => esc_html__('IMAP Server port', 'fast-plugin'),
                            'desc'    => esc_html__('Port to use when connecting to email server.', 'fast-plugin'),
                            'id'      => 'emailp_port',
                            'type'    => 'text',
                            'default' => '',
                        ),
                        array(
                            'name'    => esc_html__('IMAP Mailbox', 'fast-plugin'),
                            'desc'    => esc_html__('Mailbox to check for emails (usually default INBOX is used).', 'fast-plugin'),
                            'id'      => 'emailp_mailbox',
                            'type'    => 'text',
                            'default' => 'INBOX',
                        ),
                        array(
                            'name'    => esc_html__('Encryption', 'fast-plugin'),
                            'desc'    => esc_html__('Set encryption to use when connecting to server.', 'fast-plugin'),
                            'id'      => 'emailp_encryption',
                            'type'    => 'select',
                            'options' => array(
                                '/ssl' => esc_html__('SSL', 'fast-plugin'),
                                '/tls' => esc_html__('TLS', 'fast-plugin'),
                                ''     => esc_html__('None', 'fast-plugin'),
                            ),
                            'default' => '/ssl',
                        ),
                        array(
                            'name'    => esc_html__('Validate Certificate', 'fast-plugin'),
                            'desc'    => esc_html__('Sets if certificate from TLS/SSL server is needed.  Set novalidate for self-signed certificates.', 'fast-plugin'),
                            'id'      => 'emailp_valid_cert',
                            'type'    => 'select',
                            'options' => array(
                                '/validate-cert'   => esc_html__('Validate', 'fast-plugin'),
                                '/novalidate-cert' => esc_html__('Dont Validate', 'fast-plugin'),
                            ),
                            'default' => '/validate-cert',
                        ),
                        array(
                            'name'    => esc_html__('Charset Encoding', 'fast-plugin'),
                            'desc'    => esc_html__('Set the charset encoding of the server you are trying to connect to. Set it to US-ASCII if you are using a Microsoft / Office 365 server.', 'fast-plugin'),
                            'id'      => 'emailp_encoding',
                            'type'    => 'select',
                            'options' => array(
                                'UTF-8' => esc_html__('UTF-8', 'fast-plugin'),
                                'US-ASCII' => esc_html__('US-ASCII', 'fast-plugin'),
                            ),
                            'default' => 'UTF-8',
                        ),
                    )
                ),
                'email-piping-test' => array(
                    'title'   => esc_html__('Emil Piping Status', 'fast-plugin'),
                    'header'  => esc_html__('Check the status of your email piping.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'        => esc_html__('Last Email Check Results', 'fast-plugin'),
                            'desc'        => esc_html__('Shows the results of the last email run.', 'fast-plugin'),
                            'id'          => 'emailp_run_results',
                            'type'        => 'filter',
                            'filter'      => 'fast_filter_email_pipe_results'
                        ),
                        array(
                            'name'        => esc_html__('Next Email Check', 'fast-plugin'),
                            'desc'        => esc_html__('Shows the time the next email check will run.', 'fast-plugin'),
                            'id'          => 'emailp_run_time',
                            'type'        => 'filter',
                            'filter'      => 'fast_filter_email_pipe_next_run'
                        ),
                        array(
                            'name'        => esc_html__('Check Now', 'fast-plugin'),
                            'desc'        => esc_html__('Runs the email checker using the above settings.', 'fast-plugin'),
                            'id'          => 'emailp_run_check',
                            'type'        => 'button',
                            'button_text' => esc_html__('Run Email Check Now', 'fast-plugin'),
                            'attributes'  => array(
                                'class' => 'button',
                                'value' => 'run'
                            )
                        ),
                    )
                )
            )
        ),
        array(
            'slug' => 'fast-plugin-ticket-scheduled-maintenance',
            'page' => array(
                'title' => esc_html__('Ticket Scheduled Maintenance', 'fast-plugin'),
            ),
            'menu' => array(
                'title' => esc_html__('Scheduled Maintenance', 'fast-plugin')
            ),
            'sections'   => array(
                'ticket-maintenance' => array(
                    'title'   => esc_html__('Auto Close Tickets', 'fast-plugin'),
                    'header'  => esc_html__('Setup FAST to automatically close old inactive tickets.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Enable Auto Close Ticket', 'fast-plugin'),
                            'desc'    => esc_html__('Turns on ticket auto closing.  FAST will check for inactive tickets at midnight every day.', 'fast-plugin'),
                            'id'      => 'ticket_maintenance_enabled',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        ),
                        array(
                            'name'    => esc_html__('Close Tickets After', 'fast-plugin'),
                            'desc'    => esc_html__('Sets the period of time that a ticket will need to be inactive for them to be closed automatically.', 'fast-plugin'),
                            'id'      => 'ticket_maintenance_timespan',
                            'type'    => 'slider',
                            'default' => 14,
                            'postfix' => esc_html__('days without activity.', 'fast-plugin'),
                            'attr'      => array(
                                'max'       => 30,
                                'min'       => 1,
                                'step'      => 1
                            )
                        ),
                    )
                ),
                'ticket-maintenance-test' => array(
                    'title'   => esc_html__('Auto Close Status', 'fast-plugin'),
                    'header'  => esc_html__('Check the status of your ticket maintenance.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'        => esc_html__('Last Auto Close Results', 'fast-plugin'),
                            'desc'        => esc_html__('Shows the results of the last ticket maintenance.', 'fast-plugin'),
                            'id'          => 'ticket_maintenance_run_results',
                            'type'        => 'filter',
                            'filter'      => 'fast_filter_scheduled_maintenance_results'
                        ),
                        array(
                            'name'        => esc_html__('Next Daily Auto Close Check', 'fast-plugin'),
                            'desc'        => esc_html__('Shows the time the next ticket maintenance will run.', 'fast-plugin'),
                            'id'          => 'ticket_maintenance_run_time',
                            'type'        => 'filter',
                            'filter'      => 'fast_filter_scheduled_maintenance_next_run'
                        ),
                    )
                ),
                'ticket-maintenance-run' => array(
                    'title'   => esc_html__('Close old tickets now', 'fast-plugin'),
                    'header'  => esc_html__('This allows you to run a check for old tickets now by clicking the button below.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'        => esc_html__('Run Auto Close Now', 'fast-plugin'),
                            'desc'        => esc_html__('Runs the scheduled maintenance now.', 'fast-plugin'),
                            'id'          => 'ticket_maintenance_run_check',
                            'type'        => 'button',
                            'button_text' => esc_html__('Close Old Tickets Now', 'fast-plugin'),
                            'attributes'  => array(
                                'class' => 'button',
                                'value' => 'run'
                            )
                        ),
                    )
                )
            )
        ),
        array(
            'slug' => 'fast-plugin-advanced',
            'page' => array(
                'title' => esc_html__('Advanced Options', 'fast-plugin'),
            ),
            'menu' => array(
                'title' => esc_html__('Advanced', 'fast-plugin')
            ),
            'sections'   => array(
                'css-options' => array(
                    'title'   => esc_html__('App CSS Options / Registration', 'fast-plugin'),
                    'header'  => '<p>The options below will enable users to add some extra CSS that will be added to the frontend app.</p>',
                    'fields' => array(
                        array(
                            'name'        => esc_html__('Extra CSS', 'fast-plugin'),
                            'desc'        => esc_html__('Adds extra CSS to the frontend app.', 'fast-plugin'),
                            'id'          => 'extra_css',
                            'type'        => 'textarea',
                            'default'     => '',
                            'sanitize'    => 'sanitize_textarea_field',
                            'attributes'  => array(
                                'class' => 'widefat',
                                'rows' => 10
                            )
                        ),
                    )
                ),
                'taxonomy-options' => array(
                    'title'   => esc_html__('Taxonomy Names', 'fast-plugin'),
                    'header'  => '<p>So that you can quickly modify the ticket taxonomy names we have added the option to name them below.</p>',
                    'fields' => array(
                        array(
                            'name'        => esc_html__('Product Name Text', 'fast-plugin'),
                            'desc'        => esc_html__('Sets the taxonomy name text.', 'fast-plugin'),
                            'id'          => 'tax_product_name',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => esc_html_x('Products', 'taxonomy general name', 'fast-plugin'),
                        ),
                        array(
                            'name'        => esc_html__('Product Singular Name Text', 'fast-plugin'),
                            'desc'        => esc_html__('Sets the taxonomy singular name text.', 'fast-plugin'),
                            'id'          => 'tax_product_singular_name',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => esc_html_x('Product', 'taxonomy singular name text', 'fast-plugin'),
                        ),
                        array(
                            'name'        => esc_html__('Product Search Items Text', 'fast-plugin'),
                            'desc'        => esc_html__('Sets the taxonomy search items text.', 'fast-plugin'),
                            'id'          => 'tax_product_search_items',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => esc_html__('Search Products', 'fast-plugin'),
                        ),
                        array(
                            'name'        => esc_html__('Product Edit Item Text', 'fast-plugin'),
                            'desc'        => esc_html__('Sets the taxonomy edit item text.', 'fast-plugin'),
                            'id'          => 'tax_product_edit_item',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => esc_html__('Edit Product', 'fast-plugin'),
                        ),
                        array(
                            'name'        => esc_html__('Product Add New Item Text', 'fast-plugin'),
                            'desc'        => esc_html__('Sets the taxonomy add new item text.', 'fast-plugin'),
                            'id'          => 'tax_product_add_new_item',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => esc_html__('Add New Product', 'fast-plugin'),
                        ),
                        array(
                            'name'        => esc_html__('Product Menu Item Text', 'fast-plugin'),
                            'desc'        => esc_html__('Sets the taxonomy menu item text.', 'fast-plugin'),
                            'id'          => 'tax_product_menu_name',
                            'type'        => 'text',
                            'store_default' => true,
                            'default'     => esc_html__('Products', 'fast-plugin'),
                        ),
                    )
                ),
                'upload-options' => array(
                    'title'   => esc_html__('App Upload Options', 'fast-plugin'),
                    'header'  => '<p>The option below provides browsers with a hint of what media types the attachement popup dialog should filter by default. A complete list of all media types can be found <a href="http://www.iana.org/assignments/media-types/media-types.xhtml" target="_blank">here</a>. Values should be comma separated.</p>',
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Enable Attachment Uploads', 'fast-plugin'),
                            'desc'    => esc_html__('Turns on attachment uploading. Users will be able to upload attachments with their replies and tickets.', 'fast-plugin'),
                            'id'      => 'attachment_uploads_enabled',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'on',
                        ),
                        array(
                            'name'        => esc_html__('Preferable Upload Formats', 'fast-plugin'),
                            'desc'        => esc_html__('Comma Separated list of preferable media types.', 'fast-plugin'),
                            'id'          => 'preferable_upload_formats',
                            'type'        => 'text',
                            'default'     => esc_html__('image/*, video/*, .pdf', 'fast-plugin'),
                        ),
                        array(
                            'name'    => esc_html__('Max Upload Size', 'fast-plugin'),
                            'id'      => 'max_upload_size',
                            'desc'    => esc_html__('Sets the max upload size for ticket attachments.', 'fast-plugin'),
                            'type'    => 'slider',
                            'default' => round(wp_max_upload_size() / 1000000),
                            'postfix' => esc_html__('Mega Bytes.', 'fast-plugin'),
                            'attr'      => array(
                                'max'       => round(wp_max_upload_size() / 1000000),
                                'min'       => 1,
                                'step'      => 1
                            )
                        ),
                    )
                ),
                'translation-options' => array(
                    'title'   => esc_html__('App Translation Options', 'fast-plugin'),
                    'header'  => '<p>If set to "on", the option below will allow the app to be translated depending on the current user\'s locale. By default the plugin uses the language set under "Settings -> General" as the same translation language for all users.</p>',
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Translate using current user\'s locale', 'fast-plugin'),
                            'desc'    => esc_html__('Setting the option to "on" means each user sees the app translated in his own locale. "Off" means that the app will be displayed in the same language for all users.', 'fast-plugin'),
                            'id'      => 'translate_based_on_user_locale',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off'
                        )
                    )
                ),
                'convert-existing-users' => array(
                    'title'   => esc_html__('Convert existing users', 'fast-plugin'),
                    'header'  => esc_html__('Give FAST support access to users that were registered on your site before the plugin was installed.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'             => esc_html__('Select User Role', 'fast-plugin'),
                            'desc'             => esc_html__('Choose a role that you would like to add .', 'fast-plugin'),
                            'id'               => 'convert_role',
                            'type'             => 'select',
                            'filter'           => 'fast_filter_get_all_available_roles',
                            'default'          => '',
                        ),
                        array(
                            'name'    => esc_html__('Add Customer Role.', 'fast-plugin'),
                            'desc'    => esc_html__('Clicking the button will allow all users with the above user role to access FAST as a customer.', 'fast-plugin'),
                            'id'      => 'convert_role_clicked',
                            'type'    => 'button',
                            'button_text' => esc_html__('Give all users with the above role customer access', 'fast-plugin'),
                            'attributes'  => array(
                                'class' => 'button',
                                'value' => 'convert_role_clicked'
                            )
                        ),
                    )
                ),
                'external-user-registration' => array(
                    'title'   => esc_html__('External User Registration', 'fast-plugin'),
                    'header'  => esc_html__('These options are for users that want to use an external plugin to register their users.', 'fast-plugin'),
                    'fields' => array(
                        array(
                            'name'    => esc_html__('Add ticket customer role to all new registered users', 'fast-plugin'),
                            'desc'    => esc_html__('This will add some code that will always add the ticket customer role to any new customers who register using another plugin.', 'fast-plugin'),
                            'id'      => 'add_customer_role_to_all_users',
                            'type'    => 'select',
                            'options' => array(
                                'on'     => esc_html__('On', 'fast-plugin'),
                                'off'    => esc_html__('Off', 'fast-plugin'),
                            ),
                            'default' => 'off',
                        )
                    )
                )
            )
        )
    );
}
add_filter('fast_filter_options', 'fast_plugin_options');

/*
* fast_filter_metabox_options
*/

/**
 * Plugin metabox options
 *
 * @param array $options Plugin meabox options.
 */
function fast_plugin_metabox_options($options)
{
    return array(
        'product_fields' => array(
            'id' => 'ticket_product_fields',
            'taxonomies' => array(
                'fast_product'
            ),
            'fields' => array(
                array(
                    'name'             => esc_html__('Image', 'fast-plugin'),
                    'id'               => 'product_image',
                    'desc'             => esc_html__('Choose the user roles that will be able to use this product as a ticket filter.', 'fast-plugin'),
                    'type'             => 'upload',
                    'store'            => 'url',
                    'default'          => '',
                ),
                array(
                    'name'             => esc_html__('Product Agents', 'fast-plugin'),
                    'id'               => 'product_user_ids',
                    'desc'             => esc_html__('Set a WordPress user(s) that will be assigned tickets for this product.', 'fast-plugin'),
                    'type'             => 'select',
                    'filter'           => 'fast_filter_ticket_agent_ids_values',
                    'select2'          => true,
                    'default'          => array(),
                    'attributes'       => array(
                        'placeholder' => esc_html__('Select ticket agents who will support this product', 'fast-plugin'),
                        'class'       => 'fast-select2-option',
                        'multiple'    => '',
                    ),
                ),
                array(
                    'name'             => esc_html__('Filter Visibility', 'fast-plugin'),
                    'id'               => 'fast_product_role_visibility',
                    'desc'             => esc_html__('Choose the user roles that will be able to use this product as a ticket filter.', 'fast-plugin'),
                    'type'             => 'select',
                    'empty_label'      => esc_html__('No Role Set', 'fast-plugin'),
                    'filter'           => 'fast_filter_ticket_roles',
                    'select2'          => true,
                    'default'          => array('fast_manager', 'fast_agent'),
                    'attributes'       => array(
                        'placeholder' => esc_html__('Select a user role', 'fast-plugin'),
                        'class'       => 'fast-select2-option',
                        'multiple'    => '',
                    )
                )
            )
        ),
        'category_fields' => array(
            'id' => 'ticket_category_fields',
            'taxonomies' => array(
                'fast_category'
            ),
            'fields' => array(
                array(
                    'name'             => esc_html__('Category Agents', 'fast-plugin'),
                    'id'               => 'category_user_ids',
                    'desc'             => esc_html__('Set a WordPress user(s) that will be assigned tickets for this category.', 'fast-plugin'),
                    'type'             => 'select',
                    'filter'           => 'fast_filter_ticket_agent_ids_values',
                    'select2'          => true,
                    'default'          => array(),
                    'attributes'       => array(
                        'placeholder' => esc_html__('Select ticket agents who will support this category', 'fast-plugin'),
                        'class'       => 'fast-select2-option',
                        'multiple'    => '',
                    ),
                ),
                array(
                    'name'             => esc_html__('Filter Visibility', 'fast-plugin'),
                    'id'               => 'fast_category_role_visibility',
                    'desc'             => esc_html__('Choose the user roles that will be able to see this category as a ticket filter.', 'fast-plugin'),
                    'type'             => 'select',
                    'empty_label'      => esc_html__('No Role Set', 'fast-plugin'),
                    'filter'           => 'fast_filter_ticket_roles',
                    'select2'          => true,
                    'default'          => array('fast_manager', 'fast_agent'),
                    'attributes'       => array(
                        'placeholder' => esc_html__('Select a user role', 'fast-plugin'),
                        'class'       => 'fast-select2-option',
                        'multiple'    => '',
                    )
                )
            )
        ),
        'agent_fields' => array(
            'id' => 'ticket_agent_fields',
            'taxonomies' => array(
                'fast_agent'
            ),
            'fields' => array(
                array(
                    'name'             => esc_html__('WordPress User', 'fast-plugin'),
                    'id'               => 'agent_user_id',
                    'desc'             => esc_html__('Choose a WordPress user that corresponds to this agent.', 'fast-plugin'),
                    'type'             => 'select',
                    'empty_label'      => esc_html__('No User Set', 'fast-plugin'),
                    'filter'           => 'fast_filter_taxonomy_user_select_option_values',
                    'select2'          => true,
                    'default'          => 0,
                    'attributes'       => array(
                        'placeholder' => esc_html__('Select a WordPress user', 'fast-plugin'),
                        'class'       => 'fast-select2-option',
                    )
                ),
                array(
                    'name'             => esc_html__('Filter Visibility', 'fast-plugin'),
                    'id'               => 'fast_agent_role_visibility',
                    'desc'             => esc_html__('Choose the user roles that will be able to use this agent as a ticket filter.', 'fast-plugin'),
                    'type'             => 'select',
                    'empty_label'      => esc_html__('No Role Set', 'fast-plugin'),
                    'filter'           => 'fast_filter_ticket_roles',
                    'select2'          => true,
                    'default'          => array('fast_manager', 'fast_agent'),
                    'attributes'       => array(
                        'placeholder' => esc_html__('Select a user role', 'fast-plugin'),
                        'class'       => 'fast-select2-option',
                        'multiple'    => '',
                    )
                )
            )
        ),
        'status_visibility_fields' => array(
            'id' => 'ticket_status_visibility_field',
            'taxonomies' => array(
                'fast_status'
            ),
            'fields' => array(
                array(
                    'name'             => esc_html__('Filter Visibility', 'fast-plugin'),
                    'id'               => 'fast_status_role_visibility',
                    'desc'             => esc_html__('Choose the user roles that will be able to use this status as a ticket filter.', 'fast-plugin'),
                    'type'             => 'select',
                    'empty_label'      => esc_html__('No Role Set', 'fast-plugin'),
                    'filter'           => 'fast_filter_ticket_roles',
                    'select2'          => true,
                    'default'          => array('fast_manager', 'fast_agent', 'fast_customer'),
                    'attributes'       => array(
                        'placeholder' => esc_html__('Select a user role', 'fast-plugin'),
                        'class'       => 'fast-select2-option',
                        'multiple'    => '',
                    )
                )
            )
        ),
        'woocommerce_product_agent_fields' => array(
            'id'       => 'woocommerce_product_agent_field',
            'title'    => esc_html__('FAST Plugin Agents', 'fast-plugin'),
            'priority' => 'default',
            'context'  => 'advanced',
            'pages'    => array('product'),
            'fields'   => array(
                array(
                    'name'             => esc_html__('Product Agent(s)', 'fast-plugin'),
                    'id'               => 'fast_agents',
                    'desc'             => esc_html__('Sets the fast agents that will handle all woocommerce support tickets.', 'fast-plugin'),
                    'type'             => 'select',
                    'filter'           => 'fast_filter_ticket_agent_ids_values',
                    'select2'          => true,
                    'default'          => array(),
                    'attributes'       => array(
                        'placeholder' => esc_html__('Select agents who will handle tickets for this product', 'fast-plugin'),
                        'class'       => 'fast-select2-option',
                        'multiple'    => '',
                    ),
                )
            )
        ),
        'woocommerce_product_category_agent_fields' => array(
            'id'       => 'woocommerce_product_agent_field',
            'taxonomies' => array(
                'product_cat'
            ),
            'fields'   => array(
                array(
                    'name'             => esc_html__('Product Agent(s)', 'fast-plugin'),
                    'id'               => 'fast_agents',
                    'desc'             => esc_html__('Sets the fast agents that will handle all woocommerce support tickets.', 'fast-plugin'),
                    'type'             => 'select',
                    'filter'           => 'fast_filter_ticket_agent_ids_values',
                    'select2'          => true,
                    'default'          => array(),
                    'attributes'       => array(
                        'placeholder' => esc_html__('Select agents who will handle tickets for this category.', 'fast-plugin'),
                        'class'       => 'fast-select2-option',
                        'multiple'    => '',
                    ),
                )
            )
        ),
    );
}
add_filter('fast_filter_metabox_options', 'fast_plugin_metabox_options', 1, 1);

/*
* duplicate_comment_id
*/

/**
 * Allow duplicate ticket comments.
 */
add_filter('duplicate_comment_id', '__return_false');

/*
* rest_post_dispatch
*/

/**
 * Add custom header in api response.
 *
 * @param object  $response Response Object.
 * @param boolean $server Server Instance.
 * @param object  $request Request Object.
 */
function fast_rest_api_custom_header($response, $server, $request)
{
    if ('GET' === $request->get_method() && '/wp/v2/tickets' === $request->get_route()) {
        $response->header('Access-Control-Expose-Headers', 'X-WP-Total, X-WP-TotalPages, ticket-filters');
    }
    if ('GET' === $request->get_method() && '/fast-plugin/v1/analytics' === $request->get_route()) {
        $response->header('Access-Control-Expose-Headers', 'X-Fast-From, X-Fast-To, X-Fast-Span');
    }
    return $response;
}
add_filter('rest_post_dispatch', 'fast_rest_api_custom_header', 10, 3);

/*
* rest_pre_insert_comment
*/

/**
 * Allows html in comment content.
 *
 * @param array  $prepared_comment Comment.
 * @param object $request HTTP Request.
 */
function fast_rest_remove_html_filter_for_ticket_comments($prepared_comment, $request)
{
    if ('fast_ticket' === get_post_type($prepared_comment['comment_post_ID'])) {
        remove_filter('pre_comment_content', 'wp_filter_kses');
    }
    return $prepared_comment;
}
add_filter('rest_preprocess_comment', 'fast_rest_remove_html_filter_for_ticket_comments', 10, 2);

/**
 * Removes Akismet filters to allow ticket comments.
 *
 * @param array  $prepared_comment Comment.
 * @param object $request HTTP Request.
 */
function fast_rest_remove_akismet_check_for_tickets($prepared_comment, $request)
{
    if ('fast_ticket' === get_post_type($prepared_comment['comment_post_ID'])) {
        remove_filter('preprocess_comment', array('Akismet', 'auto_check_comment'), 1);
        remove_filter('rest_pre_insert_comment', array('Akismet', 'rest_auto_check_comment'), 1);
    }
    return $prepared_comment;
}
if (class_exists('Akismet')) {
    add_filter('rest_pre_insert_comment', 'fast_rest_remove_akismet_check_for_tickets', -1, 2);
}

/**
 * Override default discussion options, always allow comments in tickets.
 *
 * @param array $data Post Data.
 */
function fast_always_allow_ticket_comments($data)
{
    if ('fast_ticket' === $data['post_type']) {
        $data['comment_status'] = 'open';
    }
    return $data;
}
add_filter('wp_insert_post_data', 'fast_always_allow_ticket_comments');

/*
* map_meta_cap
*/

/**
 * Maps capabilities for ticket posts
 *
 * @param array  $caps Returns the user's actual capabilities.
 * @param string $cap Capability name.
 * @param int    $user_id The user ID.
 * @param array  $args Adds the context to the cap. Typically the object ID.
 *
 * @return array Capabilities to check.
 */
function fast_ticket_map_meta_cap($caps, $cap, $user_id, $args)
{
    /* If editing, deleting, or reading a ticket, get the post and post type object. */
    if ('edit_ticket' === $cap || 'delete_ticket' === $cap || 'read_ticket' === $cap) {
        if (empty($args)) {
            return array();
        }

        $post = get_post($args[0]);
        $post_type = get_post_type_object($post->post_type);

        /* Set an empty array for the caps. */
        $caps = array();
    }

    /* If editing a ticket, assign the required capability. */
    switch ($cap) {
        case 'edit_ticket':
            if ($user_id == $post->post_author) {
                $caps[] = $post_type->cap->edit_posts;
            } else {
                $caps[] = $post_type->cap->edit_others_posts;
            }
            break;
        case 'delete_ticket':
            if ($user_id == $post->post_author) {
                $caps[] = $post_type->cap->delete_posts;
            } else {
                $caps[] = $post_type->cap->delete_others_posts;
            }
            break;
        case 'read_ticket':
            if ('private' === $post->post_status) {
                if ($user_id == $post->post_author) {
                    $caps[] = $post_type->cap->edit_posts;
                } else {
                    $caps[] = $post_type->cap->read_private_posts;
                }
            } else {
                if ($user_id == $post->post_author) {
                    $caps[] = $post_type->cap->edit_posts;
                } else {
                    $caps[] = $post_type->cap->edit_others_posts;
                }
            }
            break;
    }

    /* Return the capabilities required by the user. */
    return $caps;
}
add_filter('map_meta_cap', 'fast_ticket_map_meta_cap', 10, 4);

/**
 * Allows moderate_comments on rest requests (permission will be checked using edit_comment below).
 *
 * @param array  $caps Returns the user's actual capabilities.
 * @param string $cap Capability name.
 * @param int    $user_id The user ID.
 * @param array  $args Adds the context to the cap. Typically the object ID.
 *
 * @return array Capabilities to check.
 */
function fast_allow_rest_comment_map_meta_cap($caps, $cap, $user_id, $args)
{
    if (defined('REST_REQUEST') && REST_REQUEST) {
        if ('moderate_comments' === $cap) {
            return array();
        }
    }
    return $caps;
}
add_filter('map_meta_cap', 'fast_allow_rest_comment_map_meta_cap', 10, 4);

/**
 * Filter on the current_user_can() function.
 * This function is used to explicitly allow authors to edit their own comments
 *
 * @param array $allcaps All the capabilities of the user.
 * @param array $cap     [0] Required capability.
 * @param array $args    [0] Requested capability.
 *                       [1] User ID.
 *                       [2] Associated object ID.
 */
function fast_edit_comment_cap_filter($allcaps, $cap, $args)
{
    if ('edit_comment' === $args[0]) {
        // Make sure we have the edit_comment cap or just dont allow.
        if (!array_key_exists('edit_comment', $allcaps)) {
            return array();
        }

        // We have the edit_comment cap so check if this is users comment.
        $comment_id = $args[2];
        // Get comment.
        $comment = get_comment($comment_id);
        // Get post.
        $post = get_post($comment->comment_post_ID);
        // If this is a ticket, check if current user is author.
        if ('fast_ticket' === $post->post_type) {
            if (intval($comment->user_id) !== $args[1]) {
                // Comment is not made by this user so don't allow.
                return array();
            }
        }
    }
    return $allcaps;
}
add_filter('user_has_cap', 'fast_edit_comment_cap_filter', 10, 3);

/*
* pre_comment_approved
*/

/**
 * Filters a comment's approval status before it is set. We always approve ticket comments
 *
 * @param bool|string $approved     The approval status. Accepts 1, 0, or 'spam'.
 * @param array       $comment_data Comment data.
 */
function fast_always_approve_ticket_comment($approved, $comment_data)
{
    if ('fast_ticket' === get_post_type($comment_data['comment_post_ID'])) {
        $approved = 1;
    }
    return $approved;
}
add_filter('pre_comment_approved', 'fast_always_approve_ticket_comment', 99, 2);

/*
* get_avatar_url
*/

/**
 * Use custom avatars in case WP User Avatar plugin is installed.
 *
 * @param string $url           avatar url.
 * @param string $id_or_email   user id or email.
 * @param object $args          avatar args.
 */
function fast_custom_avatar_url($url, $id_or_email, $args)
{
    global $wpua_functions;
    if (null !== $wpua_functions) {
        remove_filter('get_avatar_url', 'fast_custom_avatar_url', 10);
        // Check if this user has an custom avatar.
        if (has_wp_user_avatar($id_or_email)) {
            // Return custom avatar set by user.
            $url = get_wp_user_avatar_src($id_or_email, $args['size']);
        } elseif (!$wpua_functions->wpua_has_gravatar($id_or_email)) {
            // Get default avatar set by plugin.
            $default_image_details = $wpua_functions->wpua_default_image($args['size']);
            $url = $default_image_details['src'];
        }
        add_filter('get_avatar_url', 'fast_custom_avatar_url', 10, 3);
    }
    return $url;
}
add_filter('get_avatar_url', 'fast_custom_avatar_url', 10, 3);

/*
* rest_fast_ticket_query
*/

/**
 * Adds the needs_reply=0|1 to the ticket api.
 *
 * @param array  $args Query vars to use.
 * @param object $request API request.
 *
 * @return array New query args.
 */
function fast_add_needs_reply_var_to_ticket_api($args, $request)
{
    $agent_replied = $request->get_param('agent_replied');
    if (null !== $agent_replied) {
        // Create meta_query if not already there.
        if (!isset($args['meta_query'])) {
            $args['meta_query'] = array();
        }
        // Add query for agent replied.
        array_push($args['meta_query'], array(
            'key'     => '_agent_replied',
            'value'   => $agent_replied,
        ));
    }
    return $args;
}
add_filter('rest_fast_ticket_query', 'fast_add_needs_reply_var_to_ticket_api', 10, 2);

/*
* redirect_canonical
*/

/**
 * Disables redirect to / when frontpage set in General -> Reading.
 * used to allow extra endpoints like /ticket /login to work when
 * support page is set as the frontpage.
 *
 * @param boolean $redirect To redirect or not.
 **/
function fast_disable_canonical_redirect_for_front_page($redirect)
{
    if (is_page()) {
        $front_page = get_option('page_on_front');
        if (is_page($front_page)) {
            $redirect = false;
        }
    }
    return $redirect;
}
/* added by fast_rewrite_app_endpoints in actions.php */

/*
 * widget_comments_args
 */

/**
 * Adds custom arg in widget comment query
 *
 * @param  array $comment_args Comment Query Args.
 * @return array Modified Query Args.
 */
function fast_custom_comments_args_for_widgets($comment_args)
{
    $comment_args['post_type_not_in'] = 'fast_ticket';
    return $comment_args;
}
add_filter('widget_comments_args', 'fast_custom_comments_args_for_widgets');

/*
 * comments_clauses
 */

/**
 * Exclude comments of the fast_ticket post type in widgets.
 *
 * @param  array  $clauses Query Clauses.
 * @param  object $wp_comment_query Actual Comment Query.
 * @return array Modified Query Clauses.
 */
function fast_remove_ticket_comments_from_widgets($clauses, $wp_comment_query)
{
    global $wpdb;
    if (isset($wp_comment_query->query_vars['post_type_not_in'])) {
        if (!$clauses['join']) {
            $clauses['join'] = 'JOIN $wpdb->posts ON $wpdb->posts.ID = $wpdb->comments.comment_post_ID';
        }

        if (!$wp_comment_query->query_vars['post_type']) {
            $clauses['where'] .= $wpdb->prepare(" AND {$wpdb->posts}.post_type != %s", 'fast_ticket');
        }
        unset($wp_comment_query->query_vars['post_type_not_in']);
    }
    return $clauses;
}
add_filter('comments_clauses', 'fast_remove_ticket_comments_from_widgets', 10, 2);

/*
 * comment_flood_filter
 */

/**
 * Whether a client's reply should be blocked because of comment flood.
 *
 * @param bool $block Whether a plugin has already blocked the comment.
 * @param int  $time_lastcomment Timestamp for last comment.
 * @param int  $time_newcomment Timestamp for new comment.
 * @return bool Whether comment should be blocked.
 */
function fast_throttle_comment_flood($block, $time_lastcomment, $time_newcomment)
{
    if ($block) {
        return $block;
    }

    if (($time_newcomment - $time_lastcomment) < (int) fast_get_option('ticket_minimum_reply_interval')) {
        return true;
    }
    return false;
}
add_filter('comment_flood_filter', 'fast_throttle_comment_flood', 100, 3);


/*
 * comment_text
 */

/**
 * Makes sure that all ticket comments open in a new tab.
 *
 * @param string $content Comment content.
 * @param object $comment Comment object.
 *
 * @return string Modified comment content.
 */
function comment_links_in_new_tab($content, $comment)
{
    if ('fast_ticket' !== get_post_type($comment->comment_post_ID)) {
        return $content;
    } else {
        return str_replace('<a', '<a target="_blank"', $content);
    }
}
add_filter('comment_text', 'comment_links_in_new_tab', 10, 2);

/*
 * comment_text
 */

/**
 * Adds extra timespans for cron jobs.
 *
 * @param array $schedules Array of schedule times.
 */
function fast_add_extra_cron_schedules($schedules)
{
    $schedules['five_minutes'] = array(
        'interval' => 300,
        'display'  => esc_html__('Every 5 Minutes', 'fast-plugin'),
    );
    $schedules['ten_minutes'] = array(
        'interval' => 600,
        'display'  => esc_html__('Every 10 Minutes', 'fast-plugin'),
    );
    $schedules['fifteen_minutes'] = array(
        'interval' => 900,
        'display'  => esc_html__('Every 15 Minutes', 'fast-plugin'),
    );
    $schedules['half_hour'] = array(
        'interval' => 1800,
        'display'  => esc_html__('Every 30 Minutes', 'fast-plugin'),
    );
    return $schedules;
}
add_filter('cron_schedules', 'fast_add_extra_cron_schedules');

/**
 * Filters extra fields, prefixing Urls in inputs.
 *
 * @param array $values Extra fields saved for a ticket.
 * @param array $extra_fields Extra fields that have input urls prefixed with 'http://'.
 **/
function fast_prefix_extra_fields_input_url($values, $extra_fields)
{
    foreach ($extra_fields->fields as $field) {
        if ('input' === $field->element && 'url' === $field->inputType) {
            if (isset($values[$field->name]) && !empty($values[$field->name])) {
                $values[$field->name] = wp_parse_url($values[$field->name], PHP_URL_SCHEME) === null ? 'http://' . $values[$field->name] : $values[$field->name];
            }
        }
    }
    return $values;
}
add_filter('fast_filter_save_extra_fields', 'fast_prefix_extra_fields_input_url', 10, 2);

/**
 * Filters extra fields, prefixing Urls in inputs.
 *
 * @param array            $prepared_args Query params for comments.
 * @param WP_REST_Response $request Extra fields saved for a ticket.
 * @return array Modified query params for comments
 **/
function fast_add_agent_to_comment_query($prepared_args, $request)
{
    $roles = fast_get_user_roles();
    $is_customer = in_array('fast_customer', $roles);
    $is_agent = in_array('fast_agent', $roles);
    $is_manager = in_array('fast_manager', $roles);

    // Check we should be allowed to see comments.
    if (!$is_manager && !$is_agent && !$is_customer) {
        // You will get no comments.
        $prepared_args['user_id'] = -1;
    }

    if (isset($prepared_args['post__in']) && !empty($prepared_args['post__in'])) {
        if ('fast_ticket' === get_post_type($prepared_args['post__in'][0])) {
            if ($is_customer) {
                // Return only customer comments, hide agent only ones.
                $prepared_args['meta_query'] = array(
                    array(
                        'key' => '_agent_only',
                        'compare' => 'NOT EXISTS'
                    ),
                );
            }
        }
    }

    return $prepared_args;
}
add_filter('rest_comment_query', 'fast_add_agent_to_comment_query', 10, 2);

/*
 * rest_pre_insert_fast_purchase
 */

/**
 * Checks if the customer has already added this purchase to their purchase list.
 *
 * @param object $prepared_post Post object.
 * @param object $request HTTP Request.
 */
function fast_check_if_customer_has_already_registered_this_product($prepared_post, $request)
{
    // Only need to check this on post.
    if ('POST' !== $request->get_method()) {
        return $prepared_post;
    }

    // Only need to check this on new post.
    if (null !== $request->get_param('id')) {
        return $prepared_post;
    }

    if (false !== fast_get_user_envato_purchase($request->get_param('purchase_code'), get_current_user_id())) {
        return new WP_Error('fast_purchase_already_exits', esc_html__('You already have added this purchase', 'fast-plugin'), array(
            'status' => 400
        ));
    }

    return $prepared_post;
}
add_filter('rest_pre_insert_fast_purchase', 'fast_check_if_customer_has_already_registered_this_product', 1, 2);

/**
 * Checks if this purchase code has already been added by another customer (optional).
 *
 * @param object $prepared_post Post object.
 * @param object $request HTTP Request.
 */
function fast_check_for_duplicate_purchase_codes($prepared_post, $request)
{
    // Don't need to do anything if this is already in error.
    if (is_wp_error($prepared_post)) {
        return $prepared_post;
    }

    // Only need to check this on post.
    if ('POST' !== $request->get_method()) {
        return $prepared_post;
    }

    // Only need to check this on new post.
    if (null !== $request->get_param('id')) {
        return $prepared_post;
    }

    // Don't check if option is off.
    if ('on' !== fast_get_option('envato_check_duplicate_purchase_codes')) {
        return $prepared_post;
    }

    // Ok we need to check the purchases.
    $purchases = get_posts(array(
        'posts_per_page' => -1,
        'post_type' => 'fast_purchase',
        'post_status' => 'private',
        'meta_key' => '_purchase_code',
        'meta_value' => $request->get_param('purchase_code')
    ));

    // Return error if this purchase is already in use.
    if (!empty($purchases)) {
        $user_info = get_userdata($purchases[0]->post_author);
        return new WP_Error('fast_purchase_already_in_use', esc_html__('This purchase code is already registered with another user', 'fast-plugin'), array(
            'status' => 403
        ));
    }

    return $prepared_post;
}
add_filter('rest_pre_insert_fast_purchase', 'fast_check_for_duplicate_purchase_codes', 10, 2);

/**
 * If the fast_check_for_duplicate_purchase_codes fast_check_for_duplicate_purchase_codes
 * functions don't return errors we will call Envato and add meta data to this purchase.
 *
 * @param object $prepared_post Post object.
 * @param object $request HTTP Request.
 */
function fast_add_envato_purchase_data($prepared_post, $request)
{
    // Don't need to do anything if this is already in error.
    if (is_wp_error($prepared_post)) {
        return $prepared_post;
    }

    // Get PC from request or post if already exists.
    $purchase_code = isset($prepared_post->ID) ? get_post_meta($prepared_post->ID, '_purchase_code', true) : $request->get_param('purchase_code');

    // Get purchase from Envato.
    $purchase = fast_fetch_purchase_information($purchase_code);

    if (is_wp_error($purchase)) {
        if (isset($prepared_post->ID)) {
            update_post_meta($prepared_post->ID, '_purchase_status', 'error');
            update_post_meta($prepared_post->ID, '_error_message', $purchase->get_error_message());
        }
        return $purchase;
    }

    // Got the purchase data so save it to meta.
    foreach ($purchase as $field => $value) {
        $prepared_post->meta_input['_' . $field] = $value;
    }

    return $prepared_post;
}
add_filter('rest_pre_insert_fast_purchase', 'fast_add_envato_purchase_data', 20, 2);

/**
 * Makes sure that we use the users locale for translations when using API.
 *
 * @param string $locale Locale of plugin.
 * @param string $domain Text Domain of plugin.
 *
 * @return string New plugin locale.
 */
function fast_change_plugin_locale($locale, $domain)
{
    if ('fast-plugin' === $domain && fast_get_option('translate_based_on_user_locale') === 'on') {
        $user_id = get_current_user_id();
        return empty($user_id) ? get_locale() : get_user_locale();
    }
    return $locale;
}
add_filter('plugin_locale', 'fast_change_plugin_locale', 10, 2);

// Only load this filter if languages are using single file.
if (defined('FAST_SINGLE_TRANSLATION_FILE')) {
    /**
     * Replaces mo filename without domain and - so that it matches the angular filename.
     * This allows us to put all translations into one file (e.g. en_GB.po & en_GB.mo).
     *
     * @param string $mofile Path to mo file.
     * @param string $domain Text Domain of plugin.
     *
     * @return string New mo path without domain-
     */
    function fast_change_mo_filename($mofile, $domain)
    {
        if ('fast-plugin' === $domain) {
            return str_replace($domain . '-', '', $mofile);
        }
        return $mofile;
    }
    add_filter('load_textdomain_mofile', 'fast_change_mo_filename', 10, 2);
}

/**
 * Removes users/ endpoints
 *
 * Added in fast_rest_api_add_ticket_filter action.
 *
 * @param array $endpoints List of endpoints.
 *
 * @return array Modified list of endpoints.
 */
function fast_remove_users_rest_endpoints($endpoints)
{
    if (isset($endpoints['/wp/v2/users'])) {
        unset($endpoints['/wp/v2/users']);
    }
    if (isset($endpoints['/wp/v2/users/(?P<id>[\d]+)'])) {
        unset($endpoints['/wp/v2/users/(?P<id>[\d]+)']);
    }
    return $endpoints;
}

/**
 * WPML Filters
 */
if (function_exists('icl_object_id')) {
    /**
     * Removes comment filter if WPML is active.
     *
     * @param array            $dispatch_result A compacted array of comment query clauses.
     * @param WP_Comment_Query $request Current instance of WP_Comment_Query (passed by reference).
     */
    function fast_filter_wpml($dispatch_result, $request)
    {
        global $sitepress;
        remove_filter('comments_clauses', array($sitepress, 'comments_clauses'), 10, 2);
    }
    add_filter('rest_dispatch_request', 'fast_filter_wpml', 10, 2);
}


/**
 * Adds FAST data to WordPress export personal data.
 *
 * @param  array $exporters An array of personal data exporters.
 * @return array $exporters An array of personal data exporters.
 */
function fast_plugin_data_exporters($exporters)
{
    $exporters[] = array(
        'exporter_friendly_name' => esc_html__('Fast Plugin Tickets', 'fast-plugin'),
        'callback'               => 'fast_ticket_data_exporter',
    );
    $exporters[] = array(
        'exporter_friendly_name' => esc_html__('Fast Plugin Comments', 'fast-plugin'),
        'callback'               => 'fast_comment_data_exporter',
    );
    $exporters[] = array(
        'exporter_friendly_name' => esc_html__('Fast Plugin Purchases', 'fast-plugin'),
        'callback'               => 'fast_purchase_data_exporter',
    );

    return $exporters;
}
add_filter('wp_privacy_personal_data_exporters', 'fast_plugin_data_exporters', 10);

/**
 * Registers the personal data eraser for all customer data.
 *
 * @param  array $erasers An array of personal data erasers.
 * @return array $erasers An array of personal data erasers.
 */
function fast_data_erasers($erasers)
{
    $erasers['fast-comments'] = array(
        'eraser_friendly_name' => esc_html__('FAST Comments', 'fast-plugin'),
        'callback'             => 'fast_comment_data_eraser',
    );

    $erasers['fast-tickets'] = array(
        'eraser_friendly_name' => esc_html__('FAST Tickets', 'fast-plugin'),
        'callback'             => 'fast_ticket_data_eraser',
    );

    $erasers['fast-purchases'] = array(
        'eraser_friendly_name' => esc_html__('FAST Purchases', 'fast-plugin'),
        'callback'             => 'fast_purchase_data_eraser',
    );

    return $erasers;
}
add_filter('wp_privacy_personal_data_erasers', 'fast_data_erasers', 100);
