<?php
/**
 * Functions for the plugin
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Creates the default roles when plugin is activated.
 **/
function fast_create_plugin_default_roles()
{
    $roles = apply_filters('fast_filter_ticket_roles', array());
    foreach ($roles as $role => $name) {
        $defaults = apply_filters('fast_filter_default_' . $role . '_capabilities', array());
        $capabilities = array();
        foreach ($defaults as $default) {
            $capabilities[$default] = true;
        }
        add_role($role, $name, $capabilities);
    }
}

/**
 * Checks if a user is a ticket agent.
 *
 * @param int $user_id Id of user to check.
 */
function fast_is_user_agent($user_id)
{
    $agents_with_user_id = get_terms(array(
        'taxonomy'   => 'fast_agent',
        'hide_empty' => false,
        'meta_query' => array(
            array(
                'key'       => '_fast-plugin-agent_user_id',
                'value'     => $user_id,
            )
        )
    ));
    return !empty($agents_with_user_id) === true || is_wp_error($agents_with_user_id) === true;
}

/**
 * Returns the user type. The user must be assigned as an agent in order to be considered a staff member.
 *
 * @param int $user_id Id of user to check.
 */
function fast_get_user_type($user_id)
{
    $is_staff = fast_is_user_agent($user_id);
    $user_roles = fast_get_user_roles();

    if ($is_staff) {
        return in_array('fast_manager', $user_roles, true) ? 'manager': 'agent';
    }
    return 'customer';
}

/**
 * Gets a user object of an agent from a ticket.
 *
 * @param integer $ticket_id Ticket ID to get the agent from.
 *
 * @return object|boolean User object or false if not found.
 */
function fast_get_ticket_agent($ticket_id)
{
    $agent_terms = wp_get_post_terms($ticket_id, 'fast_agent');
    if (is_wp_error($agent_terms) || empty($agent_terms)) {
        return false;
    }
    $agent_term_id = $agent_terms[0]->term_id;
    $agent_id = fast_get_taxonomy_metabox_value('agent_user_id', $agent_term_id);
    return get_userdata($agent_id);
}

/**
 * Creates an email content subject sends it to the template and then sends the email
 *
 * @param string $sender User who is making the change.
 * @param string $receiver User to send the mail to.
 * @param string $subject Email subject.
 * @param string $message Email message with placeholders.
 * @param array  $ticket  Ticket data.
 * @param object $comment Comment object.
 *
 * @return boolean Whether the email contents were sent successfully.
 */
function fast_send_ticket_email($sender, $receiver, $subject, $message, $ticket, $comment)
{
    $subject                            = apply_filters('fast_filter_replace_placeholders', $subject, $ticket, $comment, $sender, $receiver);
    $message                            = apply_filters('fast_filter_replace_placeholders', $message, $ticket, $comment, $sender, $receiver);
    $ticket_url                         = get_permalink($ticket->ID);
    $ticket_email_template_bg_color     = fast_get_option('ticket_email_template_bg_color');
    $ticket_email_logo_image            = fast_get_option('ticket_email_logo_image');
    $ticket_email_logo_image_bg_color   = fast_get_option('ticket_email_logo_image_bg_color');
    $ticket_email_logo_title            = fast_get_option('ticket_email_logo_title');
    $ticket_email_logo_title_color      = fast_get_option('ticket_email_logo_title_color');
    $ticket_email_logo_title_bg_color   = fast_get_option('ticket_email_logo_title_bg_color');
    $ticket_email_divider_color         = fast_get_option('ticket_email_divider_color');
    $ticket_email_button_color          = fast_get_option('ticket_email_button_color');
    $ticket_email_button_bg_color       = fast_get_option('ticket_email_button_bg_color');
    $ticket_email_above_button          = fast_get_option('ticket_above_ticket_button');
    $ticket_email_footer                = fast_get_option('ticket_email_footer');
    $ticket_email_border_radius         = fast_get_option('ticket_email_border_radius');
    $ticket_email_footer_color          = fast_get_option('ticket_email_footer_color');

    $attachments = array();
    $attachment_meta = array();

    if (false !== $comment) {
        $attachment_meta = get_comment_meta($comment->comment_ID, '_attachments', true);
    }

    if (!empty($attachment_meta)) {
        if (!is_array($attachment_meta)) {
            // For versions older that 1.3
            // Some names might be badly encoded in a non-standard way. Change encoding notation and convert it to UTF-8.
            $attachment_meta = html_entity_decode(preg_replace('/u+([0-9A-F]{4})/', "&#x\\1;", $attachment_meta), ENT_NOQUOTES, 'UTF-8');
            $attachment_meta = json_decode($attachment_meta, true);
        }
        $attachments = $attachment_meta;
    }

    ob_start();
    include apply_filters('fast_filter_email_template', FAST_PLUGIN_DIR . 'templates/email/template-ticket-email.php');
    $email = ob_get_contents();
    ob_end_clean();

    add_filter('wp_mail_from', 'fast_wp_mail_from', 99);
    add_filter('wp_mail_from_name', 'fast_wp_mail_from_name', 99);
    $result = wp_mail($receiver->user_email, $subject, $email, array(
        'Content-Type: text/html; charset="UTF-8"',
    ));
    remove_filter('wp_mail_from_name', 'fast_wp_mail_from_name', 99);
    remove_filter('wp_mail_from', 'fast_wp_mail_from');
    return $result;
}

/**
 * Sends new user email when user registers
 *
 * @param string $to Email address to send to.
 * @param string $subject Subject of the email.
 * @param string $message Message to send in the email.
 **/
function fast_send_regular_email($to, $subject, $message)
{
    $ticket_email_template_bg_color   = fast_get_option('ticket_email_template_bg_color');
    $ticket_email_logo_image          = fast_get_option('ticket_email_logo_image');
    $ticket_email_logo_image_bg_color = fast_get_option('ticket_email_logo_image_bg_color');
    $ticket_email_logo_title          = fast_get_option('ticket_email_logo_title');
    $ticket_email_logo_title_color    = fast_get_option('ticket_email_logo_title_color');
    $ticket_email_logo_title_bg_color = fast_get_option('ticket_email_logo_title_bg_color');
    $ticket_email_divider_color       = fast_get_option('ticket_email_divider_color');
    $ticket_email_border_radius       = fast_get_option('ticket_email_border_radius');

    ob_start();
    include apply_filters('fast_filter_email_template', FAST_PLUGIN_DIR . 'templates/email/template-regular-email.php');
    $email_content = ob_get_contents();
    ob_end_clean();

    add_filter('wp_mail_from', 'fast_wp_mail_from', 99);
    add_filter('wp_mail_from_name', 'fast_wp_mail_from_name', 99);
    $result = wp_mail($to, $subject, $email_content, array(
        'Content-Type: text/html; charset="UTF-8"'
    ));
    remove_filter('wp_mail_from_name', 'fast_wp_mail_from_name', 99);
    remove_filter('wp_mail_from', 'fast_wp_mail_from');
    return $result;
}

/**
 * Creates a slack notification content, sends it to the template and then dispatches the notification
 *
 * @param string $sender   User that triggered the action.
 * @param string $receiver User that is affected by the action.
 * @param string $message  Message to be dispatched to Slack.
 * @param array  $ticket   Ticket data.
 * @param object $comment  Comment object.
 *
 * @return array Http Header / WP_Error if the request failed.
 */
function fast_send_slack_notification($sender, $receiver, $message, $ticket, $comment)
{
    $message = apply_filters('fast_filter_replace_placeholders', $message, $ticket, $comment, $sender, $receiver);

    return wp_remote_post(fast_get_option('slack_webhook'), array(
        'method' => 'POST',
        'blocking' => false,
        'headers' => array(
            'Content-Type' => 'application/json; charset=utf-8'
        ),
        'body' => json_encode(array(
            'mrkdwn' => true,
            'text' => $message,
        ))
    ));
}

/**
 * Returns an agent's user id from his user id.
 *
 * @param int $user_id A user id.
 *
 * @return int User agent's id.
 */
function fast_get_user_agent_id($user_id)
{
    $agents_with_user_id = get_terms(array(
        'taxonomy'   => 'fast_agent',
        'hide_empty' => false,
        'fields' => 'ids',
        'meta_query' => array(
            array(
                'key'       => '_fast-plugin-agent_user_id',
                'value'     => $user_id,
            )
        )
    ));

    if (!empty($agents_with_user_id) && count($agents_with_user_id) > 0) {
        return $agents_with_user_id[0];
    }
}

/**
 * Returns allowed html tags used by the plugin options
 *
 * @return array Allowed HTML tags
 */
function fast_get_allowed_html_tags()
{
    return array(
        'a' => array(
            'class' => array(),
            'href'  => array(),
            'rel'   => array(),
            'title' => array(),
        ),
        'blockquote' => array(
            'cite'  => array(),
        ),
        'div' => array(
            'class' => array(),
            'title' => array(),
            'style' => array(),
        ),
        'h1' => array(),
        'h2' => array(),
        'h3' => array(),
        'h4' => array(),
        'h5' => array(),
        'h6' => array(),
        'i' => array(),
        'li' => array(
            'class' => array(),
        ),
        'p' => array(
            'class' => array(),
        ),
        'span' => array(
            'class' => array(),
            'title' => array(),
            'style' => array(),
        ),
        'strong' => array(),
        'ul' => array(
            'class' => array(),
        ),
        'table' => array(
            'class' => array(),
        ),
        'tbody' => array(),
        'thead' => array(),
        'tr' => array(),
        'td' => array(),
        'th' => array(),
    );
}

/**
 * Returns number of posts found for a query of a specific user.
 *
 * @param object $filter_args Query args.
 * @param int    $is_agent is the current user an agent.
 *
 * @return int Found posts for query
 */
function fast_get_need_reply_count($filter_args, $is_agent)
{
    $filter_args['tax_query'] = isset($filter_args['tax_query']) ? $filter_args['tax_query'] : array();
    $tax_query = array_merge(
        $filter_args['tax_query'],
        array(
            array(
                'taxonomy' => 'fast_status',
                'field'    => 'slug',
                'terms'    => 'open'
            )
        )
    );

    $args = array_merge(
        $filter_args,
        array(
            'post_type' => 'fast_ticket',
            'fields' => 'ids',
            'posts_per_page' => 1,
            'meta_query' => array(
                array(
                    'key' => '_agent_replied',
                    'value' => $is_agent ? 'false' : 'true',
                )
            ),
            'tax_query' => $tax_query
        )
    );
    $query = new WP_Query($args);
    $count = $query->found_posts;
    wp_reset_postdata();
    return $count;
}

/**
 * Returns number of posts found for a query.
 *
 * @param object $filter_args Query args.
 *
 * @return int Found posts for query
 */
function fast_get_open_count($filter_args)
{
    // Only show open tickets.
    // Need to merge in case we have other tax_queries.
    $filter_args['tax_query'] = isset($filter_args['tax_query']) ? $filter_args['tax_query'] : array();
    $tax_query = array_merge(
        $filter_args['tax_query'],
        array(
            array(
                'taxonomy' => 'fast_status',
                'field'    => 'slug',
                'terms'    => 'open'
            )
        )
    );
    $args = array_merge(
        $filter_args,
        array(
            'post_type' => 'fast_ticket',
            'fields' => 'ids',
            'posts_per_page' => 1,
            'tax_query' => $tax_query
        )
    );

    $query = new WP_Query($args);
    $count = $query->found_posts;
    wp_reset_postdata();
    return $count;
}

/**
 * Gets a users roles for checking if customer/agent/manager.
 */
function fast_get_user_roles()
{
    $user = wp_get_current_user();
    // Make sure we have roles in a real array.
    // Some plugins (BBPress) mess this up.
    $user_roles = array_values($user->roles);
    if (in_array('administrator', $user_roles)) {
        $user_roles[] = 'fast_agent';
        $user_roles[] = 'fast_manager';
    }
    return $user_roles;
}

/**
 * Returns a sidebar filter for each taxonomy term, along with it's query args'
 *
 * @param object $taxonomies Array of taxonomies.
 * @param array  $extra_args Array of extra args for query.
 *
 * @return array Term filters
 */
function fast_create_filters_from_taxonomy_terms($taxonomies, $extra_args = array())
{
    $filters = array();
    $current_user_roles = fast_get_user_roles();

    foreach ($taxonomies as $taxonomy) {
        $tax_terms = get_terms(array(
            'taxonomy' => $taxonomy,
            'hide_empty' => false
        ));

        foreach ($tax_terms as $term) {
            $visible_roles = fast_get_taxonomy_metabox_value($taxonomy . '_role_visibility', $term->term_id);
            $intersect = array_intersect($visible_roles, $current_user_roles);
            if (!empty($intersect)) {
                $filter = array(
                    'id' => $term->slug,
                    'name' => $term->name,
                    'args' => array(
                        'posts_per_page' => 25,
                        'post_status' => 'publish,private',
                        'tax_query' => array(
                            array(
                                'taxonomy' => $taxonomy,
                                'field'    => 'slug',
                                'terms'    => $term->slug,
                            ),
                        ),
                    ),
                );

                if (!empty($extra_args)) {
                    $filter['args'] = array_merge($filter['args'], $extra_args);
                }

                // Conditionally add the term count if the user is not a client.
                // For clients we call WP_Query later.
                if (!in_array('fast_customer', $current_user_roles)) {
                    $filter['args']['count'] = $term->count;
                }
                $filters[] = $filter;
            }
        }
    } // End foreach().
    return $filters;
}

/**
 * Returns the custom CSS to be added to the header.
 *
 * @return string CSS code.
 */
function fast_custom_css()
{
    $css                   = '';
    $register_page_image   = fast_get_option('register_page_image');
    $login_page_image      = fast_get_option('login_page_image');
    $new_ticket_page_image = fast_get_option('new_ticket_page_image');

    if (!empty($register_page_image)) {
        $css .= 'div.fast-register-image-background { background-size: cover; background-image:'
        . 'url(' . esc_url(set_url_scheme($register_page_image)) . ');}';
    }
    if (!empty($login_page_image)) {
        $css .= 'div.fast-login-image-background { background-size: cover; background-image:'
        . 'url(' . esc_url(set_url_scheme($login_page_image)) . ');}';
    }
    if (!empty($new_ticket_page_image)) {
        $css .= 'div.fast-new-ticket-image-background { background-size: cover; background-image: '
        . 'url(' . esc_url(set_url_scheme($new_ticket_page_image)) . ');}';
    }

    return apply_filters('fast_filter_custom_css', $css);
}

/**
 * Creates oauth login url from options.
 *
 * @return string URL to envato login.
 */
function fast_generate_envato_oauth_login_url()
{
    $url = '';
    if (fast_get_option('envato_login') === 'on') {
        $support_page_id = fast_get_option('support_page');
        $envato_client_id = fast_get_option('envato_client_id');
        if (!empty($support_page_id) && !empty($envato_client_id)) {
            $redirect_url = trailingslashit(get_the_permalink($support_page_id));
            $url = 'https://api.envato.com/authorization?response_type=code&client_id=' . $envato_client_id . '&redirect_uri=' . $redirect_url;
        }
    }
    return $url;
}

/**
 * Creates oauth login url from options.
 *
 * @param string  $username Envato username.
 * @param integer $page_size Size of page to fetch.
 * @param integer $page Page to fetch.
 *
 * @return object Response object from http request to envato
 */
function fast_generate_envato_get_marketplace_items($username, $page_size, $page)
{
    return wp_remote_get('https://api.envato.com/v1/discovery/search/search/item?page=' . $page . '&page_size=' . $page_size . '&username=' . fast_get_option('envato_username'), array(
        'headers' => array(
            'timeout' => 30,
            'Authorization' => 'Bearer ' . fast_get_option('envato_token')
        )
    ));
}

/**
 * Verify envato purchase code
 *
 * @param string $purchase_code The envato purchase code.
 *
 * @return object|array Purchase details or WP_Error if fail
 */
function fast_fetch_purchase_information($purchase_code)
{
    $url = 'https://api.envato.com/v1/market/private/user/verify-purchase:' . $purchase_code . '.json';
    $token = fast_get_option('envato_token');

    $response = wp_remote_get($url, array(
        'timeout' => 30,
        'headers' => array(
            'Authorization' => 'Bearer ' . $token
        )
    ));

    if (is_wp_error($response)) {
        return $response;
    }

    $value = json_decode(wp_remote_retrieve_body($response), true);

    // Check for error responses.
    if (200 !== wp_remote_retrieve_response_code($response)) {
        // create new error with code from envato api (e.g: unknown_endpoint).
        $code = !empty($value['code']) ? $value['code'] : esc_html__('api-down', 'fast-plugin');
        $message = !empty($value['error']) ? $value['error'] : esc_html__('Could not contact API.', 'fast-plugin');
        return new WP_Error($code, $message, array(
            'status' => 401
        ));
    }

    if (isset($value['verify-purchase']) && !empty($value['verify-purchase'])) {
        $purchase = $value['verify-purchase'];
        // Set Purchase Code.
        $purchase['purchase_code'] = $purchase_code;
        // Set refresh time.
        $purchase['last_refreshed'] = time();
        // Set purchase status.
        $support_expired = time() < strtotime($purchase['support_expires']);
        $purchase['purchase_status'] = $support_expired ? 'expired' : 'unexpired';
        // Set ticket product.
        $products = get_terms(array(
            'hide_empty' => false,
            'taxonomy'   => 'fast_product',
            'meta_key'   => '_fast-plugin-product_envato_item',
            'meta_value' => $purchase['item_id'],
            'fields'     => 'ids'
        ));

        if (count($products) > 0) {
            $purchase['ticket_product'] = (int) $products[0];
        } else {
            return new WP_Error('fast-unknown-product', esc_html__('Could not find this product in our supported products', 'fast-plugin'), array(
                'status' => 404
            ));
        }

        return $purchase;
    } else {
        return new WP_Error('cant-verify', esc_html__('Cant Verify Purchase Code', 'fast-plugin'), array(
            'status' => 401
        ));
    }
}

/**
 * Gets a users envato purchase.
 *
 * @param string $purchase_code Envato Purchase code.
 * @param int    $user_id WP User id.
 *
 * @return boolean|WP_Post Purchase post or false if not found.
 */
function fast_get_user_envato_purchase($purchase_code, $user_id)
{
    $purchase_exists = get_posts(array(
        'post_type'      => 'fast_purchase',
        'post_status'    => 'private',
        'posts_per_page' => 1,
        'meta_key'       => '_purchase_code',
        'meta_value'     => $purchase_code,
        'author'         => $user_id
    ));

    return empty($purchase_exists) ? false : $purchase_exists[0];
}

/**
 * Gets the languages available in the app
 */
function fast_get_app_available_languages()
{
    // Get all WordPress Languages.
    require_once(ABSPATH . 'wp-admin/includes/translation-install.php');
    $wp_languages = wp_get_available_translations();

    // Get all languages in the app.
    $language_folder = defined('FAST_SINGLE_TRANSLATION_FILE') ? FAST_PLUGIN_DIR . 'languages' : FAST_PLUGIN_DIR . 'assets/i18n';
    $app_installed_languages = get_available_languages($language_folder);

    // Create a new array to return.
    $app_languages = array();
    if (in_array('en_US', $app_installed_languages)) {
        $app_languages[] = array(
            'code' => 'en_US',
            'name' => esc_html__('English (United States)', 'fast-plugin')
        );
    }
    // Loacate the app languages in the WP array.
    foreach ($wp_languages as $language_code => $language) {
        if (in_array($language_code, $app_installed_languages)) {
            $app_languages[] = array(
                'code' => $language_code,
                'name' => $language['native_name']
            );
        }
    }
    return $app_languages;
}

/**
 * Forces a recount of all fast ticket taxonomy counts.
 * Used for 1.2.0 update to fix bug when private tickets were not being counted.
 */
function fast_recalculate_all_taxonomy_counts()
{
    $taxonomies = array(
        'fast_status',
        'fast_product',
        'fast_agent'
    );

    foreach ($taxonomies as $taxonomy) {
        $term_ids = get_terms(array(
            'taxonomy' => $taxonomy,
            'hide_empty' => false,
            'fields' => 'ids'
        ));
        wp_update_term_count_now($term_ids, $taxonomy);
    }
}

/**
 * Creates post content based on all comments made so that search works.
 *
 * @param object $ticket Ticket post object.
 *
 * @return string Post content string with concatonated comments.
 **/
function fast_create_ticket_content_from_comments($ticket)
{
    $post_content = '';
    $comments = get_comments(array(
        'post_id' => $ticket->ID
    ));
    foreach ($comments as $comment) {
        $post_content .= $comment->comment_content;
    }
    return $post_content;
}

/**
 * Adds envato products to a list
 *
 * @param array $result Product list call.
 * @param array $products List to add results to.
 *
 * @return array List of products.
 */
function fast_add_envato_products($result, $products = array())
{
    if (is_array($result['matches'])) {
        foreach ($result['matches'] as $item) {
            $products[$item['id']] = $item['name'];
        }
    }
    return $products;
}

/**
 * Gets the extra fields option.
 */
function fast_get_extra_fields()
{
    return get_option('fast_extra_fields', (object) array(
        'fields' => array(
            (object) array(
                'label'     => esc_html__('Site URL', 'fast-plugin'),
                'element'   => 'input',
                'inputType' => 'url',
                'id'        => 'site-url',
                'name'      => 'site-url',
                'required'  => false
            )
        )
    ));
}

/**
 * Gets the secure fields option.
 */
function fast_get_secure_fields()
{
    return get_option('fast_secure_fields', (object) array(
        'fields' => array()
    ));
}

/**
 * Gets the url of the frontend app.
 */
function fast_get_app_url()
{
    $url = '';
    $support_page_id = fast_get_option('support_page');
    if (!empty($support_page_id)) {
        $url = set_url_scheme(trailingslashit(get_the_permalink($support_page_id)));
    }
    return $url;
}

/**
 * Returns if woocommerce is active.
 *
 * @return boolean If woocommerce plugins is active.
 */
function fast_is_woocommerce_active()
{
    $ticket_types = is_array(fast_get_option('new_ticket_type')) ? fast_get_option('new_ticket_type') : array(fast_get_option('new_ticket_type'));

    return class_exists('WooCommerce') && in_array('woocommerce', $ticket_types);
}

/**
 * Registers a user in WordPress.
 *
 * @param string $display_name User display name.
 * @param string $first_name First nam name.
 * @param string $last_name Last name.
 * @param string $email User Email.
 * @param string $password User Password.
 * @param string $language User language.
 *
 * @return object|boolean WP_User or false on failure.
 **/
function fast_register_user($display_name, $first_name, $last_name, $email, $password = null, $language = '')
{
    // Check for valid username.
    if (!validate_username($email)) {
        return new WP_Error('invalid-username', esc_html__('Username not valid', 'fast-plugin'), array(
            'status' => 400
        ));
    }
    // Check for valid email.
    if (!is_email($email)) {
        return new WP_Error('invalid-email', esc_html__('Email not valid', 'fast-plugin'), array(
            'status' => 400
        ));
    }
    // Check for conflicts.
    if (username_exists($email)) {
        return new WP_Error('username-exists', esc_html__('Username already exists', 'fast-plugin'), array(
            'status' => 409
        ));
    }
    if (email_exists($email)) {
        return new WP_Error('email-exists', esc_html__('User already registered with this email', 'fast-plugin'), array(
            'status' => 409
        ));
    }

    if (null === $password) {
        $password = wp_generate_password(10, false, false);
    }

    $user_id = wp_insert_user(array(
        'display_name' => $display_name,
        'first_name'   => $first_name,
        'last_name'    => $last_name,
        'user_login'   => $email,
        'user_pass'    => $password,
        'user_email'   => $email,
        'role'         => 'fast_customer',
        'locale'       => $language
    ));

    // Add flag to show this user has no tickets.
    add_user_meta($user_id, '_ticket_count', 0);

    // Get user object.
    $user = get_user_by('ID', $user_id);

    return $user;
}

/**
 * Encrypt and decrypt string.
 *
 * @param string $string string to be encrypted/decrypted.
 * @param string $action what to do with this? e for encrypt, d for decrypt.
 */
function fast_store_sensitive_e($string, $action = 'e')
{
    $secret_key = 'N0FAKHKJVpFpKQqnDjtFS6oKBkFhTuIVeoxLWShhxeC0Izqxgs';
    $secret_iv = 'COrUW4fb3JcHhI9XZ5vdr5UBwyPX5VFTS1Ya6ziNhmmpNawyFt';

    $encrypt_method = 'AES-256-CBC';
    $key = hash('sha256', $secret_key);
    $iv = substr(hash('sha256', $secret_iv), 0, 16);

    if ('e' === $action) {
        // Skip the $iv on versions less than 5.3.3 http://php.net/manual/en/function.openssl-encrypt.php .
        if (version_compare(PHP_VERSION, '5.3.3', '<')) {
            $string = base64_encode(openssl_encrypt($string, $encrypt_method, $key, 0));
        } else {
            $string = base64_encode(openssl_encrypt($string, $encrypt_method, $key, 0, $iv));
        }
    } elseif ('d' === $action) {
        // Skip the $iv on versions less than 5.3.3 http://php.net/manual/en/function.openssl-decrypt.php .
        if (version_compare(PHP_VERSION, '5.3.3', '<')) {
            $string = openssl_decrypt(base64_decode($string), $encrypt_method, $key, 0);
        } else {
            $string = openssl_decrypt(base64_decode($string), $encrypt_method, $key, 0, $iv);
        }
    }

    return $string;
}

/**
 * Assigns fast customer roles to existing users with a certain role.
 *
 * @param string $role_search Role to search users to convert.
 *
 * @return void
 */
function fast_add_customer_role_to_existing_users($role_search)
{
    $user_query = new WP_User_Query(array(
        'role' => $role_search
    ));

    $customers = $user_query->get_results();
    if (!empty($customers)) {
        foreach ($customers as $customer) {
            $user = get_user_by('id', $customer->ID);
            $user->add_role('fast_customer');
        }
    }
}

/**
 * Sends a new user registration email.
 *
 * @param string $username New user email username.
 * @param string $password New user password.
 **/
function fast_send_registration_email($username, $password)
{
    $new_user_subject = fast_get_option('user_new_email_subject');
    $new_user_content = fast_get_option('user_new_email_content');

    $new_user_content .= '<p><strong>' . esc_html__('Login Here', 'fast-plugin') . ': </strong><a href="' . esc_url(fast_get_app_url()) . '">' . esc_url(fast_get_app_url()) . '</a></p>';
    $new_user_content .= '<p><strong>' . esc_html__('Username', 'fast-plugin') . ': </strong>' . $username . '</p>';
    $new_user_content .= '<p><strong>' . esc_html__('Password', 'fast-plugin') . ': </strong>' . $password . '</p>';

    fast_send_regular_email(
        $username,
        $new_user_subject,
        $new_user_content
    );
}

/**
 * Calculates response time of comment
 *
 * @param WP_Post    $ticket Ticket.
 * @param WP_Comment $comment New comment.
 *
 * @return null|int Response time in sec or null if not found.
 */
function fast_calculate_response_time($ticket, $comment)
{
    $is_new_ticket = intval($ticket->comment_count) === 1;
    $is_user_agent = fast_is_user_agent($comment->user_id);

    if ($is_new_ticket || !$is_user_agent) {
        return;
    }

    $comments = get_comments(array(
        'post_id' => $ticket->ID,
        'author__not_in' => array($comment->user_id)
    ));

    if (empty($comments)) {
        return;
    }

    if (fast_is_user_agent($comments[0]->user_id)) {
        return;
    }

    $response_time_in_secs = strtotime($comment->comment_date_gmt) - strtotime($comments[0]->comment_date_gmt);

    return $response_time_in_secs;
}

/**
 * Exports ticket data.
 *
 * @param string $email_address Email address of user.
 * @param int    $page Current page being exported.
 */
function fast_ticket_data_exporter($email_address, $page = 1)
{
    $user = get_user_by('email', $email_address);
    $number = 500; // Limit us to avoid timing out.
    $page = (int) $page;

    $export_items = array();

    $tickets = get_posts(
        array(
            'author'         => $user->ID,
            'post_type'      => 'fast_ticket',
            'posts_per_page' => $number,
            'paged'          => $page,
            'order_by'       => 'ID',
            'order'          => 'ASC',
            'post_status'    => array('publish','private')
        )
    );


    foreach ($tickets as $ticket) {
        $data = array();

        $data[] = array(
            'name'  => esc_html__('Ticket ID', 'fast-plugin'),
            'value' => $ticket->ID
        );

        $data[] = array(
            'name'  => esc_html__('Ticket Title', 'fast-plugin'),
            'value' => $ticket->post_title
        );

        $data[] = array(
            'name'  => esc_html__('Ticket Author', 'fast-plugin'),
            'value' => get_the_author_meta('display_name', $ticket->post_author)
        );

        $data[] = array(
            'name'  => esc_html__('Ticket Author Email', 'fast-plugin'),
            'value' => get_the_author_meta('user_email', $ticket->post_author)
        );

        $data[] = array(
            'name'  => esc_html__('Ticket Date', 'fast-plugin'),
            'value' => $ticket->post_date
        );

        $data[] = array(
            'name'  => esc_html__('Ticket Content', 'fast-plugin'),
            'value' => $ticket->post_content
        );

        $url = get_the_permalink($ticket);
        $data[] = array(
            'name'  => esc_html__('Ticket URL', 'fast-plugin'),
            'value' => sprintf(
                '<a href="%s" target="_blank" rel="noreferrer noopener">%s</a>',
                esc_url($url),
                esc_html($url)
            )
        );

        if (!empty($data)) {
            $item_id = "fast_ticket-{$ticket->ID}";
            $group_id = 'fast_ticket';
            $group_label = esc_html__('Tickets', 'fast-plugin');

            $export_items[] = array(
                'group_id'    => $group_id,
                'group_label' => $group_label,
                'item_id'     => $item_id,
                'data'        => $data,
            );
        }
    } // End foreach().

    // Tell core if we have more tickets to work on still.
    $done = count($tickets) < $number;

    return array(
        'data' => $export_items,
        'done' => $done,
    );
}

/**
 * Exports ticket data.
 *
 * @param string $email_address Email address of user.
 * @param int    $page Current page being exported.
 */
function fast_purchase_data_exporter($email_address, $page = 1)
{
    $user = get_user_by('email', $email_address);
    $number = 500; // Limit us to avoid timing out.
    $page = (int) $page;

    $export_items = array();

    $purchases = get_posts(
        array(
            'author'         => $user->ID,
            'post_type'      => 'fast_purchase',
            'posts_per_page' => $number,
            'paged'          => $page,
            'order_by'       => 'ID',
            'order'          => 'ASC',
            'post_status' => array('publish','private')
        )
    );

    foreach ($purchases as $purchase) {
        $data = array();
        $purchase_info = get_post_custom($purchase->ID);

        if (!empty($purchase_info)) {
            $data[] = array(
                'name'  => esc_html__('Item', 'fast-plugin'),
                'value' => $purchase_info['_item_name'][0]
            );

            $data[] = array(
                'name'  => esc_html__('Licence', 'fast-plugin'),
                'value' => $purchase_info['_licence'][0]
            );

            $data[] = array(
                'name'  => esc_html__('Envato User', 'fast-plugin'),
                'value' => $purchase_info['_buyer'][0]
            );

            $data[] = array(
                'name'  => esc_html__('Envato Purchase Code', 'fast-plugin'),
                'value' => $purchase_info['_purchase_code'][0]
            );

            $data[] = array(
                'name'  => esc_html__('Envato Support Date', 'fast-plugin'),
                'value' => $purchase_info['_supported_until'][0]
            );
        }

        if (!empty($data)) {
            $item_id = "fast_purchase-{$purchase->ID}";
            $group_id = 'fast_purchase';
            $group_label = esc_html__('Purchases', 'fast-plugin');

            $export_items[] = array(
                'group_id'    => $group_id,
                'group_label' => $group_label,
                'item_id'     => $item_id,
                'data'        => $data,
            );
        }
    } // End foreach().

    // Tell core if we have more purchases to work on still.
    $done = count($purchases) < $number;

    return array(
        'data' => $export_items,
        'done' => $done,
    );
}



/**
 * Exports extra comment data.
 *
 * @param string $email_address Email address of user.
 * @param int    $page Current page being exported.
 */
function fast_comment_data_exporter($email_address, $page = 1)
{
    $number = 500; // Limit us to avoid timing out.
    $page = (int) $page;

    $export_items = array();

    $comments = get_comments(
        array(
            'author_email' => $email_address,
            'number'       => $number,
            'paged'        => $page,
            'order_by'     => 'comment_ID',
            'order'        => 'ASC',
        )
    );


    foreach ($comments as $comment) {
        $data = array();
        // Add attachments data.
        $attachments  = get_comment_meta($comment->comment_ID, '_attachments', true);
        if (!empty($attachments)) {
            // Plugins can add as many items in the item data array as they want.
            foreach ($attachments as $attachment) {
                $data[] = array(
                    'name'  => esc_html__('Comment Image Attachment', 'fast-plugin'),
                    'value' => sprintf(
                        '<a href="%s" target="_blank" rel="noreferrer noopener">%s</a>',
                        esc_url($attachment['source_url']),
                        esc_html($attachment['source_url'])
                    )
                );
            }
        }

        if (!empty($data)) {
            $item_id = "comment-{$comment->comment_ID}";
            $group_id = 'comments';
            $group_label = esc_html__('Comments', 'fast-plugin');

            $export_items[] = array(
                'group_id'    => $group_id,
                'group_label' => $group_label,
                'item_id'     => $item_id,
                'data'        => $data,
            );
        }
    }

    // Tell core if we have more comments to work on still.
    $done = count($comments) < $number;

    return array(
        'data' => $export_items,
        'done' => $done,
    );
}

/**
 * Erases ticket comments data.
 *
 * @param string $email_address Email address of user.
 * @param int    $page Current page being exported.
 */
function fast_comment_data_eraser($email_address, $page = 1)
{
    if (empty($email_address)) {
        return array(
            'items_removed'  => false,
            'items_retained' => false,
            'messages'       => array(),
            'done'           => true,
        );
    }

    $user = get_user_by('email', $email_address);
    $number = 500; // Limit us to avoid timing out.
    $page = (int) $page;
    $items_removed  = false;
    $items_retained = false;
    $messages    = array();

    $tickets = get_posts(
        array(
            'author'         => $user->ID,
            'post_type'      => 'fast_ticket',
            'posts_per_page' => $number,
            'paged'          => $page,
            'order_by'       => 'ID',
            'order'          => 'ASC',
            'post_status'    => array('publish','private')
        )
    );

    foreach ($tickets as $ticket) {
        $comments = get_comments(
            array(
                'post_id' => $ticket->ID
            )
        );
        foreach ($comments as $comment) {
            $comment_id = (int) $comment->comment_ID;
            $messages[] = sprintf(
                // translators: %d: comment id
                esc_html__('Removed Comment %d', 'fast-plugin'),
                $comment_id
            );
            $items_removed = true;
            wp_delete_comment($comment_id, true);
        }
    }

    // Tell core if we have more tickets to work on still.
    $done = count($tickets) < $number;

    return array(
        'items_removed'  => $items_removed,
        'items_retained' => $items_retained,
        'messages'       => $messages,
        'done'           => $done,
    );
}

/**
 * Erases ticket data.
 *
 * @param string $email_address Email address of user.
 * @param int    $page Current page being exported.
 */
function fast_ticket_data_eraser($email_address, $page = 1)
{
    if (empty($email_address)) {
        return array(
            'items_removed'  => false,
            'items_retained' => false,
            'messages'       => array(),
            'done'           => true,
        );
    }

    $user = get_user_by('email', $email_address);
    $number = 500; // Limit us to avoid timing out.
    $page = (int) $page;
    $items_removed  = false;
    $items_retained = false;
    $messages    = array();

    $tickets = get_posts(
        array(
            'author'         => $user->ID,
            'post_type'      => 'fast_ticket',
            'posts_per_page' => $number,
            'paged'          => $page,
            'order_by'       => 'ID',
            'order'          => 'ASC',
            'post_status'    => array('publish','private')
        )
    );

    foreach ($tickets as $ticket) {
        $ticket_id = (int) $ticket->ID;
        $messages[] = sprintf(
            // translators: %d: ticket id
            esc_html__('Removed Ticket %d', 'fast-plugin'),
            $ticket_id
        );
        $items_removed = true;
        wp_delete_post($ticket_id, true);
    }

    // Tell core if we have more tickets to work on still.
    $done = count($tickets) < $number;

    return array(
        'items_removed'  => $items_removed,
        'items_retained' => $items_retained,
        'messages'       => $messages,
        'done'           => $done,
    );
}

/**
 * Erases ticket data.
 *
 * @param string $email_address Email address of user.
 * @param int    $page Current page being exported.
 */
function fast_purchase_data_eraser($email_address, $page = 1)
{
    if (empty($email_address)) {
        return array(
            'items_removed'  => false,
            'items_retained' => false,
            'messages'       => array(),
            'done'           => true,
        );
    }

    $user = get_user_by('email', $email_address);
    $number = 500; // Limit us to avoid timing out.
    $page = (int) $page;
    $items_removed  = false;
    $items_retained = false;
    $messages    = array();

    $purchases = get_posts(
        array(
            'author'         => $user->ID,
            'post_type'      => 'fast_purchase',
            'posts_per_page' => $number,
            'paged'          => $page,
            'order_by'       => 'ID',
            'order'          => 'ASC',
            'post_status'    => array('publish','private')
        )
    );

    foreach ($purchases as $purchase) {
        $purchase_id = (int) $purchase->ID;
        $messages[] = sprintf(
            // translators: %d: purchase id
            esc_html__('Removed Purchase %d', 'fast-plugin'),
            $purchase_id
        );
        $items_removed = true;
        wp_delete_post($purchase_id, true);
    }

    // Tell core if we have more purchases to work on still.
    $done = count($purchases) < $number;

    return array(
        'items_removed'  => $items_removed,
        'items_retained' => $items_retained,
        'messages'       => $messages,
        'done'           => $done,
    );
}

/**
 * Fetches and saves purchase data for ticket.
 *
 * @param array $post Ticket post.
 */
function fast_attach_purchase_data_to_ticket($post)
{
    // Only return data if we are an agent.
    $user_id = get_current_user_id();
    if (!fast_is_user_agent($user_id)) {
        return;
    }

    // Only add data if ticket has a purchase id.
    $envato_purchase_id = get_post_meta($post['id'], '_envato_purchase_id', true);
    if (empty($envato_purchase_id)) {
        return;
    }

    // Check if last_refreshed timestamp has expired.
    $ticket_maintenance_interval = (int) fast_get_option('envato_purchase_code_refresh_interval');
    $last_refreshed = (int) get_post_meta($envato_purchase_id, '_last_refreshed', true);
    $current_time = time();

    if (empty($last_refreshed) || $current_time > $last_refreshed + 3600 * $ticket_maintenance_interval) {
        $purchase_code = get_post_meta($envato_purchase_id, '_purchase_code', true);
        $purchase = fast_fetch_purchase_information($purchase_code);

        // Update refresh time to now.
        update_post_meta($envato_purchase_id, '_last_refreshed', time());

        // Check if we have an error.
        if (is_wp_error($purchase)) {
            update_post_meta($envato_purchase_id, '_purchase_status', 'error');
            update_post_meta($envato_purchase_id, '_error_message', $purchase->get_error_message());
        } else {
            // Save purchase info to the meta data of the purchase.
            foreach ($purchase as $field => $value) {
                update_post_meta($envato_purchase_id, '_' . $field, $value);
            }
        }
    }

    $purchase_info = get_post_custom($envato_purchase_id);
    if (!empty($purchase_info)) {
        $return_purchase = array(
            'id'              => $envato_purchase_id,
            'buyer'           => $purchase_info['_buyer'][0],
            'created_at'      => $purchase_info['_created_at'][0],
            'item_id'         => $purchase_info['_item_id'][0],
            'item_name'       => $purchase_info['_item_name'][0],
            'licence'         => $purchase_info['_licence'][0],
            'purchase_code'   => $purchase_info['_purchase_code'][0],
            'supported_until' => $purchase_info['_supported_until'][0],
            'purchase_status' => $purchase_info['_purchase_status'][0],
            'error_message'   => '',
            'last_refreshed'  => date(DateTime::ATOM, intval($purchase_info['_last_refreshed'][0])),
            'ticket_product'  => $purchase_info['_ticket_product'][0]
        );
        if(isset($purchase_info['_error_message'][0])) {
            $return_purchase['_error_message'] = $purchase_info['_error_message'][0];
        }
        return $return_purchase;
    }
}
