<?php
/**
 * Theme option functions
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Gets an option set in the theme settings
 *
 * @param string $name Option name tag.
 * @param string $options_group name of options group to fetch from.
 * @return option value
 */
function fast_get_option($name, $options_group = null)
{
    // Get the name of the themes options.
    $options_group = null === $options_group ? apply_filters('fast_filter_theme-options-name', 'fast-plugin-options') : $options_group;

    // Get the default registry.
    $options_defaults = FastOptionsDefaults::instance();
    // Grab the options & merge them with the defaults that have been registered
    // This way no defaults need to be saved
    // See https://make.wordpress.org/themes/2014/07/09/using-sane-defaults-in-themes/ article.
    $options = wp_parse_args(get_option($options_group, array()), $options_defaults->get_defaults($options_group));

    // If the option is set then return it.
    if (isset($options[$name])) {
        return $options[$name];
    }
}

/**
 * Sets a theme option
 *
 * @param string $name          Name of option.
 * @param string $value         Value to set option to.
 * @param string $options_group Option group used by theme.
 */
function fast_set_option($name, $value, $options_group = null)
{
    // Get the name of the themes options.
    $options_group = null === $options_group ? apply_filters('fast_filter_theme-options-name', 'fast-plugin-options') : $options_group;

    // Get the default registry.
    $options_defaults = FastOptionsDefaults::instance();
    $defaults = $options_defaults->get_defaults($options_group);

    // Don't set a value if it is the default.
    if (isset($defaults[$name]) && $value === $defaults[$name]) {
        return;
    }

    // Set the option and save it.
    $options = get_option($options_group, array());
    $options[$name] = $value;
    update_option($options_group, $options);
}

/**
 * Gets a metabox value returns default if not set
 *
 * @param  string $field_id Id of field set in metabox options.
 * @param  string $post_id Id of post to get data from.
 * @return string Value of metabox option, default if not set, null no default or value set
 */
function fast_get_metabox_value($field_id, $post_id = null)
{
    $id = null === $post_id ? get_the_id() : $post_id;
    $value = get_post_meta($id, apply_filters('fast_filter_metabox_option_prefix', '_fast-plugin-') . $field_id, false);

    if (empty($value)) {
        $defaults = FastOptionsDefaults::instance();
        $metabox_defaults = $defaults->get_defaults('fast-plugin-metabox-options');
        if (isset($metabox_defaults[$field_id])) {
            $value = $metabox_defaults[$field_id];
        }
    } else {
        $value = $value[0];
    }

    return $value;
}

/**
 * Sets a metabox value on a post.
 *
 * @param string $field_id    Field ID to set.
 * @param string $field_value Field Value to set.
 * @param string $post_id     Post ID to set meta.
 */
function fast_set_metabox_value($field_id, $field_value, $post_id = null)
{
    $id = null === $post_id ? get_the_id() : $post_id;
    $meta_key = apply_filters('fast_filter_metabox_option_prefix', '_fast-plugin-') . $field_id;

    $defaults = FastOptionsDefaults::instance();
    $metabox_defaults = $defaults->get_defaults('fast-plugin-metabox-options');
    if ($field_value !== $metabox_defaults[$field_id]) {
        update_post_meta($post_id, $meta_key, $field_value);
    } else {
        delete_post_meta($post_id, $meta_key);
    }
}

/**
 * Gets a metabox value returns default if not set
 *
 * @param  string $field_id Id of field set in metabox options.
 * @param  string $term_id Id of post to get data from.
 *
 * @return string Value of metabox option, default if not set, null no default or value set
 */
function fast_get_taxonomy_metabox_value($field_id, $term_id)
{
    $value = get_term_meta($term_id, apply_filters('fast_filter_metabox_option_prefix', '_fast-plugin-') . $field_id, false);

    if (false === $value || empty($value)) {
        $defaults = FastOptionsDefaults::instance();
        $metabox_defaults = $defaults->get_defaults('fast-plugin-metabox-options');
        if (isset($metabox_defaults[$field_id])) {
            $value = $metabox_defaults[$field_id];
        }
    } else {
        $value = $value[0];
    }

    return $value;
}

/**
 * Sets a metabox value on a post.
 *
 * @param string $field_id    Field ID to set.
 * @param string $field_value Field Value to set.
 * @param string $term_id     Term ID to set meta.
 */
function fast_set_taxonomy_metabox_value($field_id, $field_value, $term_id)
{
    $meta_key = apply_filters('fast_filter_metabox_option_prefix', '_fast-plugin-') . $field_id;

    $defaults = FastOptionsDefaults::instance();
    $metabox_defaults = $defaults->get_defaults('fast-plugin-metabox-options');
    if ($field_value !== $metabox_defaults[$field_id]) {
        update_term_meta($term_id, $meta_key, $field_value);
    } else {
        delete_term_meta($term_id, $meta_key);
    }
}


/**
 * Registers the options jquery ui CSS and theme file (created with gulp)
 *
 * @return void
 */
function fast_register_jquery_ui_css()
{
    $suffix = SCRIPT_DEBUG ? '' : '.min';
    wp_register_style('fast-style--jquery-ui-options', FAST_OPTIONS_URI . 'assets/css/jquery-ui' . $suffix . '.css');
}

/**
 * Creates an option object using the field data provided.
 *
 * @param  array  $field Field options.
 * @param  string $value Value of the option.
 * @return object Option object | null on error
 */
function fast_create_option($field, $value = '')
{
    $option = null;
    if (isset($field['type'])) {
        $class_file = FAST_OPTIONS_DIR . 'options' . DIRECTORY_SEPARATOR . 'fields' . DIRECTORY_SEPARATOR . $field['type'] . DIRECTORY_SEPARATOR . $field['type'] . '.php';
        if (file_exists($class_file)) {
            require_once $class_file;
            $option_class = 'Fast' . ucwords($field['type']);
            if (class_exists($option_class)) {
                $option = new $option_class($field, $value);
            }
        }
    }

    return $option;
}


/**
 * Creates an array of options from the database (for select and checkbox options)
 *
 * @param  array $field Field options.
 * @return array Options from the database.
 */
function fast_create_database_options($field)
{
    $options = array();

    if (isset($field['default_name'])) {
        $options[$field['default']] = $field['default_name'];
    }

    switch ($field['database']) {
        case 'taxonomy':
            if (isset($field['taxonomy'])) {
                $terms = get_terms(array(
                    'taxonomy' => $field['taxonomy'],
                    'hide_empty' => false,
                ));
                if (count($terms) > 0) {
                    foreach ($terms as $term) {
                        $options[$term->term_id] = $term->name;
                    }
                }
            }
            break;
        case 'custom_post_type':
            if (isset($field['custom_post_type'])) {
                $custom_post_data = get_posts(array(
                    'posts_per_page' => -1,
                    'post_type'      => $field['custom_post_type'],
                    'orderby'        => 'title',
                    'order'          => 'DESC'
                ));
                if (count($custom_post_data) > 0) {
                    foreach ($custom_post_data as $post) {
                        $options[$post->ID] = $post->post_title;
                    }
                }
            }
            break;
    }
    return $options;
}


/**
 * Calls google to get a list of the updated fonts.
 *
 * @return array|WP_Error returns array of options or WP_Error.
 */
function fast_request_google_fonts()
{
    $google_fonts = array();
    // Try DNS lookup check to make sure we can connect to google.
    $google_font_domain = 'googleapis.com';
    $ip = gethostbyname($google_font_domain);

    // If lookup went ok we can go ahead and fetch the fonts.
    if ($ip !== $google_font_domain) {
        $google_api_url = 'https://www.googleapis.com/webfonts/v1/webfonts?key=AIzaSyDVQGrQVBkgCBi9JgPiPpBeKN69jIRk8ZA';
        $google_fonts = wp_remote_get($google_api_url, array(
            'sslverify' => false
        ));
        if (!is_wp_error($google_fonts)) {
            $body = wp_remote_retrieve_body($google_fonts);
            $google_fonts = json_decode($body, true);
        }
    } else {
        // Domain lookup is not working check server.
        $google_fonts = new WP_Error('theme-fonts-no-outgoing', 'Could not resolve domain name googleapis.com - Please check your servers outgoing connections, DNS lookup & firewall settings', 'googleapis.com');
    }

    return $google_fonts;
}
