<?php
/**
 * Creates a single metabox with options inside
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Adds metabox to theme
 */
class FastMetabox {
    /**
     * Stores the options to display inside a metabox.
     *
     * @var array
     */
    private $options;

    /**
     * Stores the key used to store all meta data
     *
     * @var string
     */
    private $meta_key;

    /**
     * Constructior, called by themeadmin.php when metaboxes are created
     *
     * @param array $options array of all theme options to use in construction this theme.
     */
    public function __construct($options)
    {
        $this->options = $options;

        add_action('admin_init', array(&$this, 'admin_init'));

        add_action('admin_enqueue_scripts', array(&$this, 'admin_enqueue_scripts'));

        add_action('save_post', array(&$this, 'save_metabox'));
    }

    /**
     * Adds metabox actions to display the metabox
     */
    public function admin_init()
    {
        if (isset($this->options['pages'])) {
            foreach ($this->options['pages'] as $page) {
                add_action('add_meta_boxes_' . $page, array(&$this, 'add_meta_boxes'));
            }
        }
    }

    /**
     * Tells WordPress that we want to load a metabox
     */
    public function add_meta_boxes()
    {
        foreach ($this->options['pages'] as $page) {
            add_meta_box($this->options['id'], $this->options['title'], array(&$this, 'render_metabox'), $page, $this->options['context'], $this->options['priority']);
        }
    }

    /**
     * Enqueue common styles
     *
     * @return void
     */
    public function admin_enqueue_scripts()
    {
        $screen = get_current_screen();

        if (isset($screen) &&
            'post' == $screen->base &&
            isset($this->options['pages']) &&
            in_array($screen->post_type, $this->options['pages']) || isset($screen) &&
            'edit-tags' == $screen->base) {
            // Register jquery ui CSS.
            fast_register_jquery_ui_css();

            // Load option screen CSS and JS.
            $suffix = SCRIPT_DEBUG ? '' : '.min';
            wp_enqueue_style('fast-style-options-page', FAST_OPTIONS_URI . 'assets/css/options' . $suffix . '.css', array('fast-style--jquery-ui-options'));
            wp_enqueue_script('fast-script-options-page', FAST_OPTIONS_URI . 'assets/js/options' . $suffix . '.js', array('jquery', 'jquery-ui-tooltip'));

            // Load option field css / js.
            foreach ($this->options['fields'] as $field) {
                $metabox_field = fast_create_option($field, '');
                if (null !== $metabox_field) {
                    $metabox_field->enqueue($suffix);
                }
            }

            // Enqueue metabox option javascripts.
            if (isset($this->options['javascripts'])) {
                foreach ($this->options['javascripts'] as $js) {
                    wp_enqueue_script($js['handle'], $js['src'], $js['deps']);
                    if (isset($js['localize'])) {
                        wp_localize_script($js['handle'], $js['localize']['object_handle'], $js['localize']['data']);
                    }
                }
            }
        }
    }

    /**
     * Main function called when metabox is shown
     *
     * @param  object $post Post object.
     * @return void
     */
    public function render_metabox($post)
    {
        $metabox_fields = array();
        foreach ($this->options['fields'] as $field) {
            $value = fast_get_metabox_value($field['id']);
            $metabox_field = fast_create_option($field, $value);
            if (null !== $metabox_field) {
                $metabox_fields[] = $metabox_field;
            }
        }
        include FAST_OPTIONS_DIR . 'metabox' . DIRECTORY_SEPARATOR . 'partials' . DIRECTORY_SEPARATOR . 'metabox.php';
    }

     /**
      * Saves all fields as metadata
      *
      * @param  int $post_id Current saving post id.
      * @return void
      */
    public function save_metabox($post_id)
    {
        // Check if not an autosave.
        if (wp_is_post_autosave($post_id)) {
            return;
        }

        // Check if not a revision.
        if (wp_is_post_revision($post_id)) {
            return;
        }

        // Check nonce.
        $nonce_field = $this->options['id'] . '-theme-metabox-nonce';
        if (isset($_POST[$nonce_field]) && wp_verify_nonce(sanitize_key(wp_unslash($_POST[$nonce_field])), $this->options['id'] . '-metabox')) {
             // First we need to check if the current user is authorised to do this action.
            if (isset($_POST['post_type'])) {
                if ('page' === sanitize_text_field(wp_unslash($_POST['post_type']))) {
                    if (! current_user_can('edit_page', $post_id)) {
                        return;
                    }
                } else {
                    if (! current_user_can('edit_post', $post_id)) {
                        return;
                    }
                }
            }

            // Thirdly we can save the value to the database.
            foreach ($this->options['fields'] as $field) {
                if (isset($_POST[$field['id']])) {
                    $post_value = $_POST[$field['id']];
                    if (is_array($post_value)) {
                        foreach ($post_value as &$sanitize_me) {
                            $sanitize_me = sanitize_text_field($sanitize_me);
                        }
                        $value = $post_value;
                    } else {
                        $value = sanitize_text_field(wp_unslash($post_value));
                    }

                    $meta_key = apply_filters('fast_filter_metabox_option_prefix', '_fast-plugin-') . $field['id'];
                    if ($field['default'] !== $value) {
                        update_post_meta($post_id, $meta_key, $value);
                    } else {
                        delete_post_meta($post_id, $meta_key);
                    }
                }
            }
        }
    }
}
