<?php
/**
 * Creates a single metabox with options inside
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Adds metabox to theme
 */
class FastTaxonomyMetabox {
    /**
     * Stores the options to display inside a metabox.
     *
     * @var array
     */
    private $options;

    /**
     * Stores the key used to store all meta data
     *
     * @var string
     */
    private $meta_key;

    /**
     * Constructior, called by themeadmin.php when metaboxes are created
     *
     * @param array $options array of all theme options to use in construction this theme.
     */
    public function __construct($options)
    {
        $this->options = $options;

        add_action('admin_init', array(&$this, 'admin_init'));

        add_action('admin_enqueue_scripts', array(&$this, 'admin_enqueue_scripts'));

        add_action('edit_term', array($this, 'save_taxonomy_metabox'), 10, 2);
        add_action('created_term', array($this, 'save_taxonomy_metabox'), 10, 2);
    }

    /**
     * Adds metabox actions to display the metabox
     */
    public function admin_init()
    {
        if (isset($this->options['taxonomies'])) {
            foreach (get_taxonomies() as $tax_name) {
                if (in_array($tax_name, $this->options['taxonomies'])) {
                    add_action($tax_name . '_add_form_fields', array($this, 'render_add_form_taxonomy_metabox'));
                    add_action($tax_name . '_edit_form_fields', array($this, 'render_edit_form_taxonomy_metabox'), 10, 2);
                }
            }
        }
    }


    /**
     * Enqueue common styles
     *
     * @return void
     */
    public function admin_enqueue_scripts()
    {
        $screen = get_current_screen();
        if (isset($screen) &&
            ('term' == $screen->base || 'edit-tags' == $screen->base) &&
            isset($this->options['taxonomies']) &&
            in_array($screen->taxonomy, $this->options['taxonomies'])) {
            // Register jquery ui CSS.
            fast_register_jquery_ui_css();

            // Load option screen CSS and JS.
            $suffix = SCRIPT_DEBUG ? '' : '.min';
            wp_enqueue_style('fast-style-options-page', FAST_OPTIONS_URI . 'assets/css/options' . $suffix . '.css', array('fast-style--jquery-ui-options'));
            wp_enqueue_script('fast-script-options-page', FAST_OPTIONS_URI . 'assets/js/options' . $suffix . '.js', array('jquery', 'jquery-ui-tooltip'));

            // Load option field css / js.
            foreach ($this->options['fields'] as $field) {
                $metabox_field = fast_create_option($field, '');
                if (null !== $metabox_field) {
                    $metabox_field->enqueue($suffix);
                }
            }

            // Enqueue metabox option javascripts.
            if (isset($this->options['javascripts'])) {
                foreach ($this->options['javascripts'] as $js) {
                    wp_enqueue_script($js['handle'], $js['src'], $js['deps']);
                    if (isset($js['localize'])) {
                        wp_localize_script($js['handle'], $js['localize']['object_handle'], $js['localize']['data']);
                    }
                }
            }
        }
    }

    /**
     * Main function called when metabox is shown
     *
     * @param  object $taxonomy Taxonomy object.
     * @return void
     */
    public function render_add_form_taxonomy_metabox($taxonomy)
    {
        $metabox_fields = array();
        foreach ($this->options['fields'] as $field) {
            $value = fast_get_taxonomy_metabox_value($field['id'], null);
            $metabox_field = fast_create_option($field, $value);
            if (null !== $metabox_field) {
                $metabox_fields[] = $metabox_field;
            }
        }
        include FAST_OPTIONS_DIR . 'metabox' . DIRECTORY_SEPARATOR . 'partials' . DIRECTORY_SEPARATOR . 'new-taxonomy-metabox.php';
    }


    /**
     * Main function called when metabox is shown
     *
     * @param  object $term Term object.
     * @param  object $taxonomy Taxonomy object.
     * @return void
     */
    public function render_edit_form_taxonomy_metabox($term, $taxonomy)
    {
        $metabox_fields = array();
        foreach ($this->options['fields'] as $field) {
            $value = fast_get_taxonomy_metabox_value($field['id'], $term->term_id);
            $metabox_field = fast_create_option($field, $value);
            if (null !== $metabox_field) {
                $metabox_fields[] = $metabox_field;
            }
        }
        include FAST_OPTIONS_DIR . 'metabox' . DIRECTORY_SEPARATOR . 'partials' . DIRECTORY_SEPARATOR . 'edit-taxonomy-metabox.php';
    }

     /**
      * Saves all fields as metadata
      *
      * @param  int $term_id Current saving term id.
      * @return void
      */
    public function save_taxonomy_metabox($term_id)
    {
        // Check nonce.
        $nonce_field = $this->options['id'] . '-theme-metabox-nonce';
        if (!isset($_POST[$nonce_field]) || !wp_verify_nonce(sanitize_key(wp_unslash($_POST[$nonce_field])), $this->options['id'] . '-metabox')) {
            return;
        }

        foreach ($this->options['fields'] as $field) {
            // Check the default to see if this is an array or single value.
            $is_array_value = is_array($field['default']);
            // If we are using an array and value is not set.
            // The empty value needs to be an empty array.
            // If value is not an array we use empty string.
            $default_null_value = $is_array_value ? array() : '';

            // Get the value of the field, use empty string if not set.
            $value = isset($_POST[$field['id']]) ? $_POST[$field['id']] : $default_null_value;

            // Sanitize the value.
            if ($is_array_value) {
                foreach ($value as &$sanitize_me) {
                    $sanitize_me = sanitize_text_field($sanitize_me);
                }
            } else {
                $value = sanitize_text_field(wp_unslash($value));
            }

            // Check if we have the default value.
            $is_default = false;
            if ($is_array_value) {
                // Need to compare the contents of the array.
                // If no difference then this is the default.
                $is_default = count(array_diff($value, $field['default'])) === 0 &&
                    count(array_diff($field['default'], $value)) === 0;
            } else {
                $is_default = $field['default'] === $value;
            }

            $meta_key = apply_filters('fast_filter_metabox_option_prefix', '_fast-plugin-') . $field['id'];
            if ($is_default) {
                // If we have the default no need to store it so remove the meta.
                delete_term_meta($term_id, $meta_key);
            } else {
                update_term_meta($term_id, $meta_key, $value);
            }
        }
    }
}
