<?php
/**
 * Abstract class for option
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Class to be extended by all options
 **/
abstract class FastOption {
    /**
     * Array of html attributes to add to this option.
     *
     * @var array
     */
    protected $attributes = array();

    /**
     * Stores the field data array
     *
     * @var array
     **/
    protected $field;

    /**
     * Stores the fields value
     *
     * @var mixed
     **/
    protected $value;

    /**
     * Main options construct
     *
     * @return void
     **/
    /**
     * Option constructor
     *
     * @param array $field Field options for this Option.
     * @param mixed $value Value of this option.
     */
    public function __construct($field, $value)
    {
        $this->field = $field;
        $this->value = $value;

        // Check if this field has extra attributes.
        if (isset($field['attributes'])) {
            $this->attributes = $field['attributes'];
        }
    }

    /**
     * Renders the option - to be overriden by sub classes
     *
     * @return void
     **/
    public function render()
    {
        // Override this method.
    }

    /**
     * Sets a html tag attribute.
     *
     * @param string       $name Name of attribute.
     * @param string|array $value value or set of values (e.g. class).
     */
    public function set_attribute($name, $value)
    {
        if (isset($this->attributes[$name])) {
            if (is_array($this->attributes[$name])) {
                $this->attributes[$name] = array_merge($this->attributes[$name], $value);
            }
        } else {
            $this->attributes[$name] = $value;
        }
    }

    /**
     * Outputs all html attrubutes.
     *
     * @return void
     */
    public function esc_attributes()
    {
        foreach ($this->attributes as $name => $value) {
            if (is_array($value)) {
                $value = implode(' ', $value);
            }
            echo esc_attr($name) . '="' . esc_attr($value) . '"';
        }
    }

    /**
     * Gets option
     *
     * @param  string $name Option Name.
     * @return array Option
     */
    public function option($name)
    {
        if (isset($this->field[$name])) {
            return $this->field[$name];
        }
    }

    /**
     * Gets the value of the options
     *
     * @return mixed Option value or default if set null if no value or default
     */
    public function value()
    {
        if (null !== $this->value) {
            return $this->value;
        } else {
            return isset($this->field['default']) ? $this->field['default'] : null;
        }
    }

    /**
     * Checks for any extra scripts defined by the option - can be overridden by sub classes
     * best to call this parent first
     *
     * @param  string $suffix CSS / JS suffix for filename (.min).
     * @return void
     */
    public function enqueue($suffix)
    {
        if (isset($this->field['javascripts'])) {
            foreach ($this->field['javascripts'] as $js) {
                wp_enqueue_script($js['handle'], $js['src'], $js['deps']);
                if (isset($js['localize'])) {
                    wp_localize_script($js['handle'], $js['localize']['object_handle'], $js['localize']['data']);
                }
            }
        }
    }
}
