<?php
/**
 * Creates an admin menu option page
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Handles creation of an admin option page.
 */
class FastMenuPage {
    /**
     * Stores the sections and options to add to the page.
     *
     * @var array
     */
    public $options;

    /**
     * Option array name
     *
     * @var string
     */
    private $option_name;

    /**
     * Registered option group
     *
     * @var string
     */
    private $option_group;

    /**
     * Stores the hook suffix of this menu page.
     *
     * @var string
     */
    public $hook;

    /**
     * Creates a menu page
     *
     * @param array  $options      Options array to add to the options page.
     * @param string $option_name  Options name used to store options.
     * @param string $option_group Registered option group.
     */
    public function __construct($options, $option_name, $option_group)
    {
        $this->option_name = $option_name;
        $this->option_group = $option_group;
        $this->options = $options;

        add_action('admin_init', array(&$this, 'create_menu_page'));

        add_action('admin_menu', array(&$this, 'register_side_menu'), 1);

        add_action('admin_enqueue_scripts', array(&$this, 'admin_enqueue_scripts'));
    }


    /**
     * Registers the option page with WordPress
     */
    public function create_menu_page()
    {
        // Register page sections with WordPress.
        foreach ($this->options['sections'] as $section_id => $section) {
            add_settings_section(
                $section_id,
                '',
                array(&$this, 'render_section_header'),
                $this->options['slug']
            );

            foreach ($section['fields'] as $field) {
                add_settings_field(
                    $field['id'],
                    $field['name'],
                    array(&$this, 'render_option'),
                    $this->options['slug'],
                    $section_id,
                    $field
                );
            }
        }
    }

    /**
     * Registers the option page as a side menu in WordPress admin panel
     */
    public function register_side_menu()
    {
        do_action('fast_action_register_options_menu_' . $this->options['slug'], $this);
    }

    /**
     * Enqueues options CSS and JS and also any extra CSS and JS needed by the options page.
     *
     * @param  string $hook Hook suffix of the current admin page.
     * @return void
     */
    public function admin_enqueue_scripts($hook)
    {
        if ($this->hook === $hook) {
            $suffix = SCRIPT_DEBUG ? '' : '.min';

            // Register jquery ui CSS.
            fast_register_jquery_ui_css();

            // Load option screen CSS and JS.
            wp_enqueue_style('fast-style-options-page', FAST_OPTIONS_URI . 'assets/css/options' . $suffix . '.css', array('fast-style--jquery-ui-options'));
            wp_enqueue_script('fast-script-options-page', FAST_OPTIONS_URI . 'assets/js/options' . $suffix . '.js', array('jquery', 'jquery-ui-tooltip'));

            // Enqueue option scripts / styles.
            foreach ($this->options['sections'] as $section_id => $section) {
                foreach ($section['fields'] as $field) {
                    $option = fast_create_option($field, '');
                    if (null !== $option) {
                        $option->enqueue($suffix);
                    }
                }
            }
        }
    }

    /**
     * Outputs the option page HTML
     *
     * @return void
     */
    public function render_option_page()
    {
        if (isset($_GET['page'])) {
            $page = sanitize_text_field(wp_unslash($_GET['page']));
            $show_footer = isset($this->options['hide_footer']) && true === $this->options['hide_footer'] ? false : true;
            do_action($page . '-before-page');
            include FAST_OPTIONS_DIR . 'options' . DIRECTORY_SEPARATOR . 'partials' . DIRECTORY_SEPARATOR . 'options-page.php';
            do_action($page . '-after-page');
        }
    }

    /**
     * Outputs each sections header description and title.
     *
     * @param  array $section_data Data of each option.
     * @return void
     */
    public function render_section_header($section_data)
    {
        foreach ($this->options['sections'] as $section_id => $section) {
            if ($section_id == $section_data['id']) {
                if (isset($section['title'])) {
                    echo '<h3>' . esc_html($section['title']) . '</h3>';
                }
                if (isset($section['header'])) {
                    echo wp_kses($section['header'], array(
                        'a' => array(
                            'href'   => array(),
                            'title'  => array(),
                            'target' => array()
                        )
                    ));
                }
                break;
            }
        }
    }

    /**
     * Creates a single option in the option page.
     *
     * @param  array $field Option field info.
     * @return void
     */
    public function render_option($field)
    {
        if (isset($field['id'])) {
            $id = $field['id'];
            $data = get_option($this->option_name);
            $value = isset($data[$id]) ? $data[$id] : null;

            $option = fast_create_option($field, $value);
            if (null !== $option) {
                $option->render();
            }

            if (isset($field['desc'])) {
                echo '</td><td>';
                echo '<a class="oxy-tooltip-description oxy-pull-right" title="' . esc_html($field['desc']) . '"><span class="ui-widget-content">?</span></a>';
            }
        } else {
            echo esc_html__('No id set for this option', 'fast-plugin');
        }
    }


    /**
     * Saves this option pages data.
     *
     * @param  array $data Original data.
     * @return array Updated Data Values
     */
    public function save($data)
    {
        if (isset($_POST['_wpnonce']) && wp_verify_nonce(sanitize_key($_POST['_wpnonce']), $this->option_group . '-options')) {
            // Are we resetting all options to defaults?
            if (isset($_POST['reset_options'])) {
                foreach ($this->options['sections'] as $section_id => $section) {
                    foreach ($section['fields'] as $field) {
                        unset($data[$field['id']]);
                    }
                }
            } else {
                foreach ($this->options['sections'] as $section_id => $section) {
                    foreach ($section['fields'] as $field) {
                        $id = $field['id'];

                        if (isset($_POST[$id])) {
                            $sanitize_method = isset($field['sanitize']) ? $field['sanitize'] : 'sanitize_text_field';
                            $form_data = wp_unslash($_POST[$id]);
                            switch ($sanitize_method) {
                                case 'sanitize_text_field':
                                    if (is_array($form_data)) {
                                        $data[$id] = array();
                                        foreach ($form_data as $index => $sanitize_me) {
                                            $data[$id][$index] = sanitize_text_field($sanitize_me);
                                        }
                                    } else {
                                        $data[$id] = sanitize_text_field(wp_unslash($form_data));
                                    }
                                    break;
                                default:
                                    $data[$id] = apply_filters('fast_filter_field_sanitize_' . $field['sanitize'], $form_data);
                            }
                        } else {
                            // No data has been set.
                            // So add blank value.
                            $data[$id] = '';
                            // Check for an array type and set blank to be empty array.
                            if (isset($field['default'])) {
                                $data[$id] = is_array($field['default']) ? array() : '';
                            }
                        }
                    }
                }

                // Check for any default values (they do not need to be saved).
                foreach ($this->options['sections'] as $section_id => $section) {
                    foreach ($section['fields'] as $field) {
                        if (isset($field['default']) && $data[$field['id']] == $field['default']) {
                            unset($data[$field['id']]);
                        }
                    }
                }
            } // End if().
        } // End if().
        do_action('fast_action_option_page_saved_' . $this->options['slug'], $data);
        return $data;
    }
}
