<?php
/**
 * Handles Google Font updates
 *
 * @package Fast Plugin
 *
 * @copyright 2019 Oxygenna.com
 * @license http://themeforest.net/licenses/standard
 * @version 1.15.4
 * @author Oxygenna
 */

/**
 * Handles google font updates
 */
class FastGoogleFonts {
    /**
     * Stores the singleton instance of the class
     *
     * @var FastGoogleFonts
     */
    private static $instance;

    /**
     * Stores the option name of the google fonts list.
     *
     * @var string
     */
    private $google_fonts_option_name = 'fast-plugin_google_fonts';

    /**
     * Constructor for google fonts
     */
    public function __construct()
    {
        // Register google fonts list ajax call.
        add_action('wp_ajax_google_fonts_list', array(&$this, 'google_fetch_fonts_list_ajax'));
    }

    /**
     * Gets the font list from the options db
     *
     * @return array list of fonts available
     */
    public function get_fonts_from_options()
    {
        return get_option($this->google_fonts_option_name, array());
    }

    /**
     * Grabs the fonts from the fonts.json file in the theme in case google fails.
     *
     * @return WP_Error|string JSON of fonts or error.
     */
    public function fallback_google_fonts()
    {
        $url = FAST_OPTIONS_URI . 'typography/fonts.json';
        return wp_remote_get($url, array(
            'sslverify' => false
        ));
    }

    /**
     * Gets the list of google fonts from the google API
     *
     * @return WP_Error|string JSON of fonts or error.
     */
    public function get_google_fonts()
    {
        $google_fonts = null;

        // Do DNS lookup check to make sure we can connect to google.
        $google_font_domain = 'googleapis.com';
        $ip = gethostbyname($google_font_domain);

        // If lookup went ok we can go ahead and fetch the fonts.
        if ($ip !== $google_font_domain) {
            $google_fonts = wp_remote_get('https://www.googleapis.com/webfonts/v1/webfonts?key=AIzaSyDVQGrQVBkgCBi9JgPiPpBeKN69jIRk8ZA', array(
                'sslverify' => false
            ));
        } else {
            $google_fonts = new WP_Error('theme-update-fonts-error', esc_html__('Could not fetch fonts', 'fast-plugin'));
        }

        return $google_fonts;
    }

    /**
     * Ajax function for getting fonts.
     *
     * @return void
     */
    public function google_fetch_fonts_list_ajax()
    {
        header('Content-Type: application/json');
        // 3 minutes should be PLENTY.
        @set_time_limit(180);

        $response = array(
            'message' => esc_html__('Could not verify nonce', 'fast-plugin'),
            'data'    => array()
        );

        if (isset($_POST['nonce']) && wp_verify_nonce(sanitize_key(wp_unslash($_POST['nonce'])), 'fast-plugin-add-fontstack')) {
            $response['message'] = esc_html__('Fonts Fetched OK', 'fast-plugin');
            $response['data'] = $this->update_google_fonts();
        }

        echo wp_json_encode($response);

        die();
    }

    /**
     * Calls google to update fonts
     *
     * @return array Fonts
     */
    public function update_google_fonts()
    {
        $fonts = array();

        // Try to get font list from google.
        $google_fonts = $this->get_google_fonts();

        if (is_wp_error($google_fonts)) {
            // Couldnt get fonts from google url.
            // So get the font list from the theme.
            $google_fonts = $this->fallback_google_fonts();
        }

        if (!is_wp_error($google_fonts)) {
            $data = wp_remote_retrieve_body($google_fonts);
            $fonts = json_decode($data, true);
            update_option($this->google_fonts_option_name, $fonts);
        }

        return $fonts;
    }

    /**
     * Instance accessor. If instance doesn't exist, we'll initialize the class.
     *
     * @return object PagePress
     */
    public static function instance()
    {
        if (!isset(self::$instance)) {
            self::$instance = new FastGoogleFonts();
        }
        return self::$instance;
    }
}
