<?php
/**
 * Main Stacks Class
 *
 * @package Fast Plugin
 * @subpackage Stacks
 *
 * @copyright **COPYRIGHT**
 * @license **LICENSE**
 * @version **VERSION**
 * @author **AUTHOR**
 */

use Leafo\ScssPhp\Compiler;
use MatthiasMullie\Minify;

/**
 * Main stack class
 */
class FastStacks {
    /**
     * Singleton instance
     *
     * @var object
     */
    private static $instance;

    /**
     * Post type prefix.
     *
     * @var string
     */
    private $post_type = 'fast-plugin-stack';

    /**
     * Option name to store the stack settings.
     *
     * @var string
     */
    private $option_name = 'fast-plugin_stack_options';

    /**
     * Option group to store the stack settings.
     *
     * @var string
     */
    private $option_group = 'fast-plugin_stack_options_group';

    /**
     * Meta key for the google url.
     *
     * @var string
     */
    private $google_url_meta_key = '_fast-plugin-stack-google-url';

    /**
     * Creates an instance of this class.
     *
     * @return object Instance of this class.
     */
    public static function instance()
    {
        if (! self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Class constructor
     */
    public function __construct()
    {
        add_action('init', array(&$this, 'register_post_type'));

        add_action('after_setup_theme', array(&$this, 'options_create'));

        add_action('fast_action_register_options_menu_fast-plugin-stack-settings', array(&$this, 'options_create_menu'));

        add_action('after_setup_theme', array(&$this, 'create_stack_metaboxes'));

        add_action('current_screen', array(&$this, 'add_edit_page_css'));

        add_action('save_post', array(&$this, 'save_stack_post'), 100);

        add_action('add_meta_boxes', array(&$this, 'register_extra_metaboxes'));
    }

    /**
     * Registers the stack post type
     *
     * @return void
     */
    public function register_post_type()
    {
        // Create stacks custom post type.
        $labels = array(
            'name'               => esc_html__('Stacks', 'fast-plugin'),
            'singular_name'      => esc_html__('Stack', 'fast-plugin'),
            'add_new'            => esc_html__('Add New', 'fast-plugin'),
            'add_new_item'       => esc_html__('Add New Stack', 'fast-plugin'),
            'edit_item'          => esc_html__('Edit Stack', 'fast-plugin'),
            'new_item'           => esc_html__('New Stack', 'fast-plugin'),
            'all_items'          => esc_html__('All Stacks', 'fast-plugin'),
            'view_item'          => esc_html__('View Stack', 'fast-plugin'),
            'search_items'       => esc_html__('Search Stack', 'fast-plugin'),
            'not_found'          => esc_html__('No Stack found', 'fast-plugin'),
            'not_found_in_trash' => esc_html__('No Stack found in Trash', 'fast-plugin'),
            'menu_name'          => esc_html__('Stacks', 'fast-plugin')
        );

        $args = array(
            'labels'             => $labels,
            'public'             => false,
            'publicly_queryable' => false,
            'show_ui'            => true,
            'show_in_menu'       => true,
            'query_var'          => false,
            'capability_type'    => 'post',
            'menu_position'      => null,
            'menu_icon'          => 'dashicons-art',
            'supports'           => array('title')
        );
        register_post_type($this->post_type,
            apply_filters('fast_filter_stack_register_post_type', $args)
        );
    }

    /**
     * Creates the stack settings page
     *
     * @return void
     */
    public function options_create()
    {
        // Add stack options to appearance menu.
        if (class_exists('FastOptions')) {
            // Create new option set with post type as menu.
            $this->options = new FastOptions($this->option_group, $this->option_name);

            // Load the stack options and add them to the options page.
            $stack_options = include FAST_STACKS_DIR . 'options' . DIRECTORY_SEPARATOR . 'settings-options.php';
            $this->options->add_menu_page($stack_options);
        }
    }

    /**
     * Registers all stack metabox options.
     *
     * @return void
     */
    public function create_stack_metaboxes()
    {
        if (class_exists('FastMetaboxes')) {
            $metaboxes = new FastMetaboxes();
            $metabox_options = $this->get_stack_metabox_options();
            foreach ($metabox_options as $metabox) {
                $metabox['priority'] = 'default';
                $metabox['context']  = 'advanced';
                $metabox['pages']    = array($this->post_type);

                $metaboxes->add_metabox($metabox);
            }
        }
    }

    /**
     * Adds submenu for stack options to the stacks custom post type menu
     *
     * @param  object $menu_page MenuPage object.
     * @return void
     */
    public function options_create_menu($menu_page)
    {
        $menu_page->hook = add_submenu_page('edit.php?post_type=' . $this->post_type, $menu_page->options['page']['title'], $menu_page->options['menu']['title'], 'manage_options', $menu_page->options['slug'], array(&$menu_page, 'render_option_page'));
    }

    /**
     * Adds submenu for stack options to the stacks custom post type menu
     *
     * @return void
     */
    public function options_register_menu()
    {
        $hook = add_submenu_page('edit.php?post_type=' . $this->post_type, esc_html__('Stack Options', 'fast-plugin'), esc_html__(' Options', 'fast-plugin'), 'manage_options', $this->options_page['slug'], array(&$this, 'options_render_page'));
    }

    /**
     * Gets the stack options from the theme using a filter.
     *
     * @return array All the stack options.
     */
    public function get_stack_metabox_options()
    {
        return apply_filters('fast_filter_stack_options', array());
    }

    /**
     * Gets the stack fields from the stack metaboxes.
     *
     * @return array All the stack fields.
     */
    public function get_stack_fields()
    {
        $variables = array();
        // Grab all stack metaboxes.
        $stack_metaboxes = $this->get_stack_metabox_options();
        foreach ($stack_metaboxes as $metabox) {
            foreach ($metabox['fields'] as $field) {
                $variables[] = $field;
            }
        }
        return $variables;
    }

    /**
     * Adds the edit stack page CSS
     *
     * @return void
     */
    public function add_edit_page_css()
    {
        $screen = get_current_screen();
        if ($this->post_type === $screen->post_type && 'post' === $screen->base) {
            $suffix = SCRIPT_DEBUG ? '' : '.min';
            wp_enqueue_style('oxy-edit-stack', FAST_STACKS_URI . 'assets/css/edit-stack' . $suffix . '.css');
        }
        if ($this->post_type === $screen->post_type && 'post' !== $screen->base) {
            $suffix = SCRIPT_DEBUG ? '' : '.min';
            wp_enqueue_style('oxy-stack-admin', FAST_STACKS_URI . 'assets/css/stack-list' . $suffix . '.css');
        }
    }

    /**
     * Saves a stack post
     *
     * @param int $post_id Post ID.
     * @return void
     */
    public function save_stack_post($post_id)
    {
        if (isset($_POST['import_stack_nonce']) && wp_verify_nonce(sanitize_key(wp_unslash($_POST['import_stack_nonce'])), 'fast-plugin_import_stack') && isset($_POST['import_data'])) {
            $this->import_stack(sanitize_text_field(wp_unslash($_POST['import_data'])), $post_id);
        }

        if (defined('DOING_AJAX') || defined('DOING_AUTOSAVE') || 'auto-draft' === get_post_status($post_id)) {
            return;
        }

        if (get_post_type($post_id) !== $this->post_type) {
            return;
        }

        $this->save_post_css($post_id);
    }

    /**
     * Saves a stack post
     *
     * @param int $post_id Post ID.
     * @return void
     */
    public function save_post_css($post_id)
    {
        // Get the default registry.
        $stack_options_defaults = FastOptionsDefaults::instance();
        // Grab the options & merge them with the defaults that have been registered
        // This way no defaults need to be saved
        // See https://make.wordpress.org/themes/2014/07/09/using-sane-defaults-in-themes/ article.
        $stack_options = wp_parse_args(get_option($this->option_name, array()), $stack_options_defaults->get_defaults($this->option_name));

        // Get all the stack variables.
        $variables = $this->get_stack_fields();

        // Save the regular CSS.
        $css = $this->generate_stack_css($post_id, $variables);
        $this->save_css_to_file($post_id, $css, 'stack.css');

        // Save the minified CSS.
        $minified_css = $this->minify_css($css);
        $this->save_css_to_file($post_id, $minified_css, 'stack.min.css');

        // Create google font link.
        $google_url = $this->generate_google_font_url($variables, $post_id);
        update_post_meta($post_id, $this->google_url_meta_key, $google_url);
    }

    /**
     *
     * Takes a template CSS file and replaces variables from the stack options.
     *
     * @param int   $post_id    Id of post we are saving.
     * @param array $variables  Metabox variables to replace in the CSS file.
     *
     * @return string            File contents with variables replaced.
     */
    public function generate_stack_css($post_id, $variables)
    {
        // Get the compiler.
        require FAST_PLUGIN_DIR . 'lib/scssphp/scss.inc.php';
        $scss_compiler = new Compiler();

        // Create the stack scss variables at the top of the scss.
        $stack_variables = array();
        foreach ($variables as $variable) {
            $value = fast_get_metabox_value($variable['id'], $post_id);
            if (isset($variable['filter'])) {
                $value = apply_filters($variable['filter'], $value);
            }
            $stack_variables[$variable['id']] = $value;
        }
        // Set the variables in the compiler.
        $scss_compiler->setVariables($stack_variables);
        // Set up the import paths.
        $scss_compiler->setImportPaths(array(
            FAST_PLUGIN_DIR . 'assets/scss/',
        ));
        // Set encoding to latin for quicker compile time.
        $scss_compiler->setEncoding('iso8859-1');
        // Compile the sass!
        return $scss_compiler->compile(apply_filters('fast_filter_stack_scss', ''));
    }

    /**
     * Minifys CSS
     *
     * @param  string $css Css to minify.
     * @return string Minified CSS
     */
    public function minify_css($css)
    {
        require FAST_PLUGIN_DIR . 'lib/minify/path-converter/src/ConverterInterface.php';
        require FAST_PLUGIN_DIR . 'lib/minify/path-converter/src/Converter.php';
        require FAST_PLUGIN_DIR . 'lib/minify/minify/src/Minify.php';
        require FAST_PLUGIN_DIR . 'lib/minify/minify/src/CSS.php';
        $minifier = new Minify\CSS();
        $minifier->add($css);
        return $minifier->minify();
    }

    /**
     * Saves some CSS to a file.
     *
     * @param  int    $post_id Post ID.
     * @param  string $css CSS to save to the file.
     * @param  string $filename Filename to save the CSS to.
     * @return void
     */
    public function save_css_to_file($post_id, $css, $filename)
    {
        $filesystem_method = $this->get_filesystem_for_upload_dir();
        if ('direct' === $filesystem_method) {
            // Okay, let's see about getting credentials.
            $url = wp_nonce_url('post.php?post=' . $post_id, 'edit');
            $creds = request_filesystem_credentials($url);
            if (false === $creds) {
                // If we get here, then we don't have credentials yet.
                // But have just produced a form for the user to fill in.
                // So stop processing for now.
                wp_die('Please input your credentials above to save the CSS file.');
            }


            // Now we have some credentials, try to get the wp_filesystem running.
            if (!WP_Filesystem($creds)) {
                // Our credentials were no good, ask the user for them again.
                request_filesystem_credentials($url);
                wp_die('Please input your credentials above to save the CSS file.');
            }

            // Get the upload directory and make a test.txt file.
            $stack_base_folder = $this->get_base_upload_path();
            $stack_css_folder = $this->get_stack_css_path($post_id, '');
            $stack_css_file = $this->get_stack_css_path($post_id, $filename);

            global $wp_filesystem;

            // Check if base folder exists first.
            if (!$wp_filesystem->is_dir($stack_base_folder)) {
                /* Directory didn't exist, so let's create it */
                $wp_filesystem->mkdir($stack_base_folder);
                // And lets add a blank index.html to be safe.
                $index_filename = $stack_base_folder . 'index.html';
                if (!$wp_filesystem->put_contents($index_filename, '', FS_CHMOD_FILE)) {
                    wp_die(esc_html__('Error Saving Index File', 'fast-plugin'));
                }
            }

            // Now create stack folder for all stack CSS.
            if (!$wp_filesystem->is_dir($stack_css_folder)) {
                /* Directory didn't exist, so let's create it */
                $wp_filesystem->mkdir($stack_css_folder);
                // And lets add a blank index.html to be safe.
                $index_filename = $stack_css_folder . 'index.html';
                if (!$wp_filesystem->put_contents($index_filename, '', FS_CHMOD_FILE)) {
                    wp_die(esc_html__('Error Saving Index File', 'fast-plugin'));
                }
            }

            // Everything should be set so lets save the CSS file.
            if (!$wp_filesystem->put_contents($stack_css_file, $css, FS_CHMOD_FILE)) {
                wp_die(esc_html__('Error Saving CSS File', 'fast-plugin'));
            }
        }
    }

    /**
     * Creates the google font stylesheet url.
     *
     * @param array  $variables Array of variables for the stack.
     * @param string $post_id Post id.
     *
     * @return string URL to google api.
     */
    public function generate_google_font_url($variables, $post_id)
    {
        // Colate together all the fonts into families, weights and languages.
        $families = array();
        $languages = array();
        foreach ($variables as $variable) {
            $value = fast_get_metabox_value($variable['id'], $post_id);
            if ('font' === $variable['type'] && !empty($value)) {
                $font = json_decode($value, true);

                if (isset($font['family'])) {
                    $family = $font['family'];
                    $weights = isset($families[$family]) ? $families[$family] : array();

                    if (isset($font['weights'])) {
                        $weights = array_merge($weights, $font['weights']);
                    }

                    if (isset($font['languages'])) {
                        $languages = array_merge($languages, $font['languages']);
                    }

                    $families[$family] = $weights;
                }
            }
        }

        // Put all families into Family:Weights for url.
        $families_and_weights = array();
        foreach ($families as $family => $weights) {
            $url_part = str_replace(' ', '+', $family);
            if (!empty($weights)) {
                $url_part .= ':' . implode(',', $weights);
            }
            $families_and_weights[] = $url_part;
        }

        // Create url - separate families using pipe and subsets with commas.
        $url = add_query_arg(array(
            'family' => implode('|', apply_filters('fast_filter_stack-google-families', $families_and_weights)),
            'subsets' => implode(',', apply_filters('fast_filter_stack-google-languages', $languages))
        ), 'https://fonts.googleapis.com/css');

        return $url;
    }

    /**
     * Gets a base upload folder
     *
     * @return string path of the base folder to save stack css files.
     */
    public function get_base_upload_path()
    {
        $wp_upload_dir = wp_upload_dir();
        return trailingslashit($wp_upload_dir['basedir'] . DIRECTORY_SEPARATOR . 'fast-plugin');
    }

    /**
     * Gets a stack css path
     *
     * @param int    $post_id Post id.
     * @param string $filename Filename of the CSS file.
     *
     * @return string path of the stack CSS file
     */
    public function get_stack_css_path($post_id, $filename)
    {
        $wp_upload_dir = wp_upload_dir();
        return trailingslashit($this->get_base_upload_path() . $post_id) . $filename;
    }

    /**
     * Gets a stack css url
     *
     * @param int    $post_id Post id.
     * @param string $filename Filename of the CSS file.
     *
     * @return string URL of the stack CSS file
     */
    public function get_stack_css_url($post_id, $filename)
    {
        return apply_filters('fast_filter_css_url_from_id', '', $post_id, $filename);
    }

    /**
     * Returns the filesystem available for uploading to the uploads folder.
     *
     * @return string ftp,direct filesystem method
     */
    public function get_filesystem_for_upload_dir()
    {
        // Get filesystem available for the uploads folder.
        $upload_dir = wp_upload_dir();
        return get_filesystem_method(array(), $upload_dir['path']);
    }

    /**
     * Imports a stack from import text field
     *
     * @param  string $import_data Serialized data.
     * @param  int    $post_id  Stack post id.
     *
     * @return void
     */
    public function import_stack($import_data, $post_id)
    {
        if (is_serialized($import_data)) {
            $import_data = unserialize($import_data);

            if (is_array($import_data)) {
                $variables = $this->get_stack_fields();
                foreach ($variables as $variable) {
                    if (isset($import_data[$variable['id']])) {
                        fast_set_metabox_value($variable['id'], $import_data[$variable['id']], $post_id);
                    }
                }
            }
        }
    }

    /**
     * Metaboxes
     **/

    /**
     * Adds import export metaboxes
     *
     * @return void
     */
    public function register_extra_metaboxes()
    {
        // Register stack status metabox.
        add_meta_box('fast-script-status', esc_html__('Stack Status', 'fast-plugin'), array(&$this, 'status_metabox'), $this->post_type, 'advanced', 'high');

        // Register import / export metabox.
        add_meta_box('fast-script-import-export', esc_html__('Import / Export', 'fast-plugin'), array(&$this, 'import_metabox'), $this->post_type, 'advanced', 'low');
    }

    /**
     * Creates the status metabox
     *
     * @param  object $post Current stack post.
     * @return void
     */
    public function status_metabox($post)
    {
        // Get filesystem available for the uploads folder.
        $filesystem_method = $this->get_filesystem_for_upload_dir();

        switch ($filesystem_method) {
            case 'direct':
                // Get css file details.
                $css_files = array();
                $stack_css_css_files = apply_filters('fast_filter_stack_css_files', array());
                foreach ($stack_css_css_files as $file_path) {
                    $filename = basename($file_path);
                    $path = $this->get_stack_css_path($post->ID, $filename);
                    $css_files[] = array(
                        'path' => $path,
                        'url'  => $this->get_stack_css_url($post->ID, $filename),
                        'exists' => file_exists($path)
                    );
                }
                include FAST_STACKS_DIR . 'partials' . DIRECTORY_SEPARATOR . 'stack-status-direct.php';
                break;
            default:
                $upload_dir = wp_upload_dir();

                include FAST_STACKS_DIR . 'partials' . DIRECTORY_SEPARATOR . 'stack-status-no-direct.php';
                break;
        }
    }

    /**
     * Creates an import / export metabox
     *
     * @param  object $post Current Post.
     * @return void
     */
    public function import_metabox($post)
    {
        $stack = array();
        $variables = $this->get_stack_fields();
        foreach ($variables as $variable) {
            $value = fast_get_metabox_value($variable['id']);
            $stack[$variable['id']] = $value;
        }

        $export = serialize($stack);

        include FAST_STACKS_DIR . 'partials' . DIRECTORY_SEPARATOR . 'import-metabox.php';
    }
}
