<?php
/**
 * Mobile Smart - Domain Switcher class
 * - manages domain switching
 */

define('MOBILE_SMART_ENABLE_DOMAIN_SWITCHING', 'enable_domain_switching'); 
define('MOBILE_SMART_ENABLE_DOMAIN_SWITCHING_ONLY', 'enable_domain_switching_only');
define('MOBILE_SMART_ONLY_SWITCH_DOMAIN_ON_VALID_DEVICE', 'only_switch_domain_on_valid_device');
define('MOBILE_SMART_OPTION_MOBILE_DOMAIN', 'mobile_domain');

define('MOBILE_SMART_ENABLE_DOMAIN_SWITCHING_TO_DESKTOP', 'enable_domain_switching_to_desktop');
define('MOBILE_SMART_OPTION_DESKTOP_DOMAIN', 'desktop_domain');
 
class MobileSmartDomainSwitcher extends MobileSmartPluginBase {
  
  
  /**
   * constructor - sets up actions & filters
   * 
   * @access public
   * @return void
   */
  public function __construct() {
    parent::__construct();
    
    add_action('init', array($this, 'action_init'));
    add_action('wp_head', array($this, 'action_head'));
    
    add_filter('post_link', array(&$this, 'modify_url'));
    add_filter('walker_nav_menu_start_el', array(&$this, 'modify_url'));
    add_filter('the_category', array($this, 'modify_url'));
    add_filter('the_terms', array($this, 'modify_url'));
    add_filter('the_content', array($this, 'modify_url'));
    add_filter('term_link', array($this, 'modify_url'));
    
    add_filter('redirect_canonical', array($this, 'redirectCanonical'), 100, 2);
    
    //add_filter('nav_menu_link_attributes', array($this, 'modify_menu_url'));
  }
  
  /**
    * Run init action
    */
   public function action_init() {
  
    // if is a mobile device or is mobile due to cookie switching
    if ($this->isMobile()) { 
      // run redirect if necessary
      $redirect = $this->canRedirectDomain($this->isMobile());
      
      if ($redirect && !is_admin()) {
        wp_redirect($redirect);
        exit();
      }
    }
    else {
      $redirect = $this->canRedirectDomain($this->isMobile());
      
      if ($redirect && !is_admin()) {
        wp_redirect($redirect);
        exit();
      }
    }
   }
   
   /**
    * can we redirect to a mobile domain
    * @return string
    */
   public function canRedirectDomain($is_mobile)
   {
     $this->logDebug('switcher_canRedirectDomain: $is_mobile', $is_mobile);
     $redirect = false;
     
     if ($this->getOption(MOBILE_SMART_ENABLE_DOMAIN_SWITCHING)) {
	     
       // if not is mobile device (could be cookie switched) and we're on the mobile domain, then redirect to main domain
       if (!$is_mobile && (isset($_SERVER['HTTP_HOST']) && stripos($_SERVER['HTTP_HOST'], $this->getOption(MOBILE_SMART_OPTION_MOBILE_DOMAIN)) !== false)) {
          $redirect = site_url();
          
          $this->logDebug('switcher_canRedirectDomain: currently on mobile domain but not mobile, $redirect: ', $redirect);
       }
       // else if we're mobile device and not on mobile domain then redirect to mobile domain
       else if ($is_mobile && (stripos($_SERVER['HTTP_HOST'], $this->getOption(MOBILE_SMART_OPTION_MOBILE_DOMAIN)) === false)) {
           $redirect = (is_ssl() ? 'https://' : 'http://') . $this->getOption(MOBILE_SMART_OPTION_MOBILE_DOMAIN);
           
           $this->logDebug('switcher_canRedirectDomain: mobile but not on mobile domain, $redirect ', $redirect);
       }
       else {
         $this->logDebug('switcher_canRedirectDomain: no redirect');
       }
       
       // generate the full redirect link
       $redirect = $this->generateRedirectLink($redirect);
     } else if ($this->getOption(MOBILE_SMART_ENABLE_DOMAIN_SWITCHING_TO_DESKTOP)) {

       // if we're deskop and not on mobile domain then redirect to mobile domain
       if (!$is_mobile && (stripos($_SERVER['HTTP_HOST'], $this->getOption(MOBILE_SMART_OPTION_DESKTOP_DOMAIN)) === false)) {
           $redirect = (is_ssl() ? 'https://' : 'http://') . $this->getOption(MOBILE_SMART_OPTION_DESKTOP_DOMAIN);
           
           $this->logDebug('switcher_canRedirectDomain: mobile but not on mobile domain, $redirect ', $redirect);
       }
       else {
         $this->logDebug('switcher_canRedirectDomain: no redirect');
       }
       
       // generate the full redirect link
       $redirect = $this->generateRedirectLink($redirect);
     }
     
     return $redirect;
   }
   
  /**
   * Generate the redirect link
   */
  public function generateRedirectLink($root) {
    $redirect = $root;
       
     if ($redirect) {
       if (isset($_SERVER['REQUEST_URI'])) {
        $redirect .= $_SERVER['REQUEST_URI'];
       }
       else if (isset($_SERVER['REDIRECT_URL']) && $_SERVER['REDIRECT_URL']) {
         $redirect .= $_SERVER['REDIRECT_URL'];
       }
       if ((stripos($redirect, '?') === false) && isset($_SERVER['QUERY_STRING']) && $_SERVER['QUERY_STRING']) {
         $redirect .= '?'.$_SERVER['QUERY_STRING'];
       }
     }
     return $redirect;
  
  }
    
  /**
   * Add Google Webmasters head
   */ 
  public function action_head() {
    
    if ($this->getOption(MOBILE_SMART_OPTION_MOBILE_DOMAIN)) {
      
      if ($this->isMobile()) {
        $root = site_url();
      }
      else {
        $root = (is_ssl() ? 'https://' : 'http://') . $this->getOption(MOBILE_SMART_OPTION_MOBILE_DOMAIN);
      }
      
      $redirect = $this->generateRedirectLink($root);
      
      // Google-friendly URL
      if ($this->isMobile()) :
        ?>
        <link rel="canonical" href="<?php echo $redirect; ?>" >
        <?php
      else :
       ?>
       <link rel="alternate" media="only screen and (max-width: 640px)" href="<?php echo $redirect; ?>" >
       <?php
      endif;
    }
  }
  
  /**
   * Modify a URL
   */ 
  public function modify_url($original_link) {
    
    $new_link = $original_link;
  
    if (!$this->getOption(MOBILE_SMART_ENABLE_DOMAIN_SWITCHING) && !$this->getOption(MOBILE_SMART_ENABLE_DOMAIN_SWITCHING_ONLY)) {
	    return $original_link;
	  }
    
    // we need to rewrite the permalink to the domain switched url only if on a mobile site
    if ($this->isMobile()) {
      $domain = (is_ssl() ? 'https://' : 'http://') . $this->getOption(MOBILE_SMART_OPTION_MOBILE_DOMAIN);
    
      $new_link = str_replace(home_url(), $domain, $original_link);
    }
    
    return $new_link;
  }
  
  
  /**
   * make sure redirect canonical doesn't mess up our URL to www instead of mobile domain.
   * 
   * @access public
   * @param mixed $redirect_url
   * @param mixed $requested_url
   * @return void
   */
  public function redirectCanonical($redirect_url, $requested_url) {
    return $this->modify_url($redirect_url);
  }
  
  
  /**
   * modify_menu_url function.
   * 
   * @access public
   * @param mixed $atts
   * @return void
   */
  public function modify_menu_url($atts) {
    die();
    if (isset($atts['href'])) {
      $atts['href'] = $this->modify_url($atts);
    }
    
    echo "<pre>";                  
    print_r($atts);
    echo "</pre>";
    die();
    
    return $atts;
  }
}