<?php
/**
 * Mobile Smart Pro - Mobile Smart Pages class
 * - mobile specific content
 */
 
define('MOBILE_SMART_POST_CONTENT_META', 'mobileSmart_mobilepages');
define('MOBILE_SMART_POST_EXCERPT_META', 'mobileSmart_mobileexcerpt');

define('MOBILE_SMART_OPTION_ENABLE_MOBILE_PAGES', 'enable_mobile_pages');
 
class MobileSmartPages extends MobileSmartPluginBase {
  
  
  /**
   * constructor - sets up actions & filters
   * 
   * @access public
   * @return void
   */
  public function __construct() {
    parent::__construct();
    
    add_action('add_meta_boxes', array(&$this, 'addMetaBoxes'));
    add_action( 'save_post', array(&$this, 'savePost'));
    
    add_filter('the_content', array(&$this, 'the_content'), 1, 1);
    add_filter('get_the_excerpt', array(&$this, 'get_the_excerpt'), 1, 1);
    
    add_action('mobile_smart_is_mobile', array($this, 'checkCompatibility'));
  }
  
  
  /**
   * check plugin compatibility.
   * 
   * @access public
   * @param mixed $is_mobile
   * @return void
   */
  public function checkCompatibility($is_mobile) {
    if ($is_mobile) {
      add_filter('siteorigin_panels_filter_content_enabled', array($this, 'disableSiteOriginPageBuilderContent'));
    }
  }
  
  /**
   * SiteOrigin Page Builder clashes with mobile content function - put logic in place to manage compatibility
   * 
   * @access public
   * @param mixed $disable
   * @return void
   */
  public function disableSiteOriginPageBuilderContent($disable) {
    global $post;
    
    // get mobile content
    $mobile_content = $this->getContent($post->ID);
    
    // if mobile content exists, then disable page builder for this page
    if ($mobile_content) {
      $disable = false;
    }
    
    return $disable;
  }
  
  /**
   * Add mobile pages post type
   */
  function addMetaBoxes() {
    
    if ($this->getOption('enable_mobile_pages')) {
      // add the meta box to each post type
      $post_types=get_post_types('','names'); 
    
      foreach ($post_types as $post_type) {
        if (post_type_supports($post_type, 'editor')) {
          add_meta_box( 
            'mobileSmart_mobilePage'
            ,__( 'Mobile Content', MOBILESMART_DOMAIN )
            ,array( &$this, 'displayContentMetaBox' )
            ,$post_type
            ,'normal'
            ,'high'
          );
        }
        if (post_type_supports($post_type, 'excerpt')) {
          add_meta_box( 
            'mobileSmart_mobilePageExcerpt'
            ,__( 'Mobile Excerpt', MOBILESMART_DOMAIN )
            ,array( &$this, 'displayExcerptMetaBox' )
            ,$post_type
            ,'normal'
            ,'core'
          );
        }
      }
    }
  }
  
  /**
   * Display the content meta box
   * @param type $post 
   */
  function displayContentMetaBox($post)
  {
   $mobilePage_content = $this->getContent($post->ID);
   
   // compatibility with WordPress 3.3 - wp_editor only works with version 3.3 or after
   if (function_exists('wp_editor')) {
     wp_editor($mobilePage_content, 'mobileSmartmobilePage');
   }
   else // if not, just show an empty textbox for the moment
   {
     ?>
     <label class="screen-reader-text">Mobile Content</label>
     <textarea name="mobileSmartmobilePage" cols="80" rows="7" style="width:100%;"><?php echo $mobilePage_content; ?></textarea>
     <?php
   }
   
  ?>
   <p><?php echo __('Leave this empty to display the main content on a mobile device, or add your mobile-specific content here', MOBILESMART_DOMAIN); ?></p>
  <?php
   
  }
  
  /**
   * Display the excerpt meta box
   * @param type $post 
   */
  function displayExcerptMetaBox($post)
  {
   $mobilePage_excerpt = $this->getExcerpt($post->ID);
     ?>
     <label class="screen-reader-text">Mobile Excerpt</label>
     <textarea name="mobileSmartmobilePageExcerpt" cols="40" rows="3" style="width:98%; height: 4em; margin: 0;"><?php echo $mobilePage_excerpt; ?></textarea>
     <p><?php echo __('Leave this empty to display use the normal excerpt functionality, or add your mobile-specific manual excerpt here. <a href="http://codex.wordpress.org/Excerpt">Learn more about manual excerpts.</a>', MOBILESMART_DOMAIN); ?></p>
  <?php
  }
  
  /**
   * Get the mobile pages content for a post
   * @param type $post_id
   * @return string 
   */
  function getContent($post_id)
  {
    global $page, $more, $preview, $pages, $multipage;
    
    
    // get the post content from the meta instead
    $post_content = get_post_meta($post_id, MOBILE_SMART_POST_CONTENT_META, true);
    
    if (is_admin()) {
      return $post_content;
    }
    
    // for post_password_required
    $post = get_post();
    
    // need a way to pass this in
    $strip_teaser = false;
   
    // the following is adapted from get_the_content() in /wp-includes/post-template.php
    $more_link_text = __( '(more&hellip;)' );
  
    $output = '';
    $has_teaser = false;
  
    // If post password required and it doesn't match the cookie.
    if ( post_password_required( $post ) )
    	return get_the_password_form( $post );
  
  	$content = $post_content;
  	
  	if ( preg_match( '/<!--more(.*?)?-->/', $content, $matches ) ) {
  		$content = explode( $matches[0], $content, 2 );
  		if ( ! empty( $matches[1] ) && ! empty( $more_link_text ) )
  			$more_link_text = strip_tags( wp_kses_no_null( trim( $matches[1] ) ) );
  
  		$has_teaser = true;
  	} else {
  		$content = array( $content );
  	}
  
  	if ( false !== strpos( $post->post_content, '<!--noteaser-->' ) && ( ! $multipage || $page == 1 ) )
  		$strip_teaser = true;
  
  	$teaser = $content[0];
  
  	if ( $more && $strip_teaser && $has_teaser )
  		$teaser = '';
  
  	$output .= $teaser;
  
  	if ( count( $content ) > 1 ) {
  		if ( $more ) {
  			$output .= '<span id="more-' . $post->ID . '"></span>' . $content[1];
  		} else {
  			if ( ! empty( $more_link_text ) )
  				$output .= apply_filters( 'the_content_more_link', ' <a href="' . get_permalink() . "#more-{$post->ID}\" class=\"more-link\">$more_link_text</a>", $more_link_text );
  			$output = force_balance_tags( $output );
  		}
  	}
  
  	if ( $preview ) // preview fix for javascript bug with foreign languages
  		$output =	preg_replace_callback( '/\%u([0-9A-F]{4})/', '_convert_urlencoded_to_entities', $output );
  		
    return $output;
  
  }
  
  /**
   * Save the mobile pages content
   * @param type $post_id
   * @param type $content 
   */
  function storeContent($post_id, $content)
  {
    update_post_meta($post_id, MOBILE_SMART_POST_CONTENT_META, $content);
  }
  
  /**
   * Save the mobile pages excerpt
   * @param type $post_id
   * @param type $excerpt 
   */
  function storeExcerpt($post_id, $excerpt)
  {
    update_post_meta($post_id, MOBILE_SMART_POST_EXCERPT_META, $excerpt);
  }
  
  
  /**
   * mobilePages_getExcerptMeta function.
   * 
   * @access public
   * @param mixed $post_id
   * @return void
   */
  function getExcerpt($post_id) {
    return get_post_meta($post_id, MOBILE_SMART_POST_EXCERPT_META, true);
  }
  
  /**
   * Save mobile page data
   * @global type $post
   * @param type $post_id
   * @return type 
   */
  function savePost($post_id) {
    // verify if this is an auto save routine. 
    // If it is our form has not been submitted, so we dont want to do anything
    if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) 
      return;
    
    if ($this->getOption('enable_mobile_pages'))
    {
     global $post;
     
     // Check permissions
      if ( !current_user_can( 'edit_post', $post_id ) )
          return;
     
      // save post data
      if (isset($_POST['mobileSmartmobilePage']))
      {
        $this->storeContent($post_id, $_POST['mobileSmartmobilePage']);
      }
      if (isset($_POST['mobileSmartmobilePageExcerpt']))
      {
        $this->storeExcerpt($post_id, $_POST['mobileSmartmobilePageExcerpt']);
      }
    }
  }
  
  /**
  * Display the content if it exists
  * @global obj $post
  * @param type $content 
  */
  function the_content($content = false)
  {    
    if (/*is_main_query() && in_the_loop() && */$this->getOption(MOBILE_SMART_OPTION_ENABLE_MOBILE_PAGES))
    {
     if ($this->isMobile())
     {
       global $post;
    
       $mobile_content = $this->getContent($post->ID);
    
       if ($mobile_content)
       {
         $content = $mobile_content;
       }
     }
    }
    
    return $content;
  }
  
  
  /**
  * Gets the excerpt - hooked onto 'get_the_excerpt' filter
  * 
  * @access public
  * @global obj $post
  * @param string $output
  * @return string
  */
  public function get_the_excerpt($output) {
    global $post;
    
    if ($this->getOption(MOBILE_SMART_OPTION_ENABLE_MOBILE_PAGES))
    {
      if ($this->isMobile() && !is_admin()) {       
        $mobile_excerpt = $this->getExcerpt($post->ID);
    
        // if we have a mobile excerpt, then override, no matter whether there was a manual excerpt or not            
        if ($mobile_excerpt) {
          $output = $mobile_excerpt;
        }
      }
    }
    
    return $output;
  }
  
}