<?php
/**
 * Mobile Smart - Domain Switcher class
 * - manages domain switching
 */

define('MOBILE_SMART_ENABLE_POST_SWITCHING', 'enable_post_switching'); 
 
class MobileSmartPostSwitcher extends MobileSmartPluginBase {
	
	const TYPE_MOBILE = 'mobile';
	const TYPE_TABLET = 'tablet';
	const TYPE_DESKTOP = 'desktop';
	
	const POST_META_REDIRECT_URLS = 'mobile_smart_redirect_url';
	
	const CAPABILITY = 'manage_options';
  
  
  /**
   * constructor - sets up actions & filters
   * 
   * @access public
   * @return void
   */
  public function __construct() {
    parent::__construct();
    
    add_action('template_redirect', array($this, 'postRedirect'));
    
    add_action('add_meta_boxes', array($this, 'addMetaBox'));
    add_action('save_post', array($this, 'savePost'), 10, 3);
  }
  
  /**
    * Run init action
    */
   public function postRedirect() {
      // if is mobile
      if ($this->isMobile() && $this->getRedirectPost(self::TYPE_MOBILE)) {
        // run redirect if necessary
        $redirect = $this->getRedirectPost(self::TYPE_MOBILE);
        
        if ($redirect && !is_admin()) {
          wp_redirect($redirect);
          exit();
        }
      }
      // if is tablet
      else if ($this->isTablet() && $this->getRedirectPost(self::TYPE_TABLET)) {
        $redirect = $this->getRedirectPost(self::TYPE_TABLET);
        
        if ($redirect && !is_admin()) {
          wp_redirect($redirect);
          exit();
        }
      }
      // if is desktop
      else if ($this->isDesktop() && $this->getRedirectPost(self::TYPE_DESKTOP)) {
        $redirect = $this->getRedirectPost(self::TYPE_DESKTOP);
        
        if ($redirect && !is_admin()) {
          wp_redirect($redirect);
          exit();
        }
      }
   }
   
   /**
    * can we redirect to a mobile domain
    * @return string
    */
  public function getRedirectPost($type) {
		global $post;
		
		$redirect = false;
		
		$this->logDebug('canRedirectPost', $type);
		
		// if is a single post / page / etc, then check if the redirect has been set
		if (is_singular() && isset($post->ID)) {
		 	$post_redirect = $this->getPostMeta($post->ID);
		 
		 	if ($post_redirect[$type]) {
		   	$redirect = $post_redirect[$type];
		 	}
		}
		
		return $redirect;
  }
  
  
  /**
   * add meta box to set redirect url for post.
   * 
   * @access public
   * @return void
   */
  public function addMetaBox() {
	  add_meta_box('mobilesmart_post_redirect', 'Mobile Post Redirect', array($this, 'displayPostRedirectMetaBox'), null);
  }
  
  
  /**
   * display the post redirect meta box.
   * 
   * @access public
   * @return void
   */
  public function displayPostRedirectMetaBox($post) {
	  if (!current_user_can(self::CAPABILITY)) return;
	  
	  $post_redirect = $this->getPostMeta($post->ID);
	  
	  $post_type_object = get_post_type_object(get_post_type());
	  
	  ?>
	  <p>You can specifically redirect this <?php echo $post_type_object->labels->singular_name; ?> for mobile and/or tablet. This will override any domain switching.</p>
	  <p>
		  <label for="<?php echo self::POST_META_REDIRECT_URLS.'-'.self::TYPE_MOBILE; ?>">Mobile redirect: </label>
		  <input type="text" name="<?php echo self::POST_META_REDIRECT_URLS.'-'.self::TYPE_MOBILE; ?>" value="<?php echo $post_redirect[self::TYPE_MOBILE]; ?>"/>
	  </p>
	  <p>
		  <label for="<?php echo self::POST_META_REDIRECT_URLS.'-'.self::TYPE_TABLET; ?>">Tablet redirect: </label>
		  <input type="text" name="<?php echo self::POST_META_REDIRECT_URLS.'-'.self::TYPE_TABLET; ?>" value="<?php echo $post_redirect[self::TYPE_TABLET]; ?>"/>
	  </p>
	  <?php
  }
  
  
  /**
   * getPostMeta function.
   * 
   * @access private
   * @return void
   */
  private function getPostMeta($post_id) {
	  $post_redirect = get_post_meta($post_id, self::POST_META_REDIRECT_URLS, true);
	  
	  if (!$post_redirect) {
		  $post_redirect = array(
			  self::TYPE_MOBILE => '',
			  self::TYPE_TABLET => '',
			  self::TYPE_DESKTOP => ''
		  );
	  }
	  
	  return $post_redirect;
  }
  
  
  /**
   * savePostMeta function.
   * 
   * @access private
   * @return void
   */
  private function savePostMeta($post_id, $post_redirect) {
	  update_post_meta($post_id, self::POST_META_REDIRECT_URLS, $post_redirect);
  }
  
  
  /**
   * save the post redirect url.
   * 
   * @access public
   * @return void
   */
  public function savePost($post_id, $post, $update) {
	  if (!current_user_can(self::CAPABILITY)) return;
	  
	  if (isset($_POST[self::POST_META_REDIRECT_URLS.'-'.self::TYPE_MOBILE]) ||
	  		isset($_POST[self::POST_META_REDIRECT_URLS.'-'.self::TYPE_TABLET])) {
		  		
		  // get redirect array
		  $post_redirect = $this->getPostMeta($post_id);
	  
			// update values if set
		  if (isset($_POST[self::POST_META_REDIRECT_URLS.'-'.self::TYPE_MOBILE])) {
			  $post_redirect[self::TYPE_MOBILE] = $_POST[self::POST_META_REDIRECT_URLS.'-'.self::TYPE_MOBILE];
		  }
		  if (isset($_POST[self::POST_META_REDIRECT_URLS.'-'.self::TYPE_TABLET])) {
			  $post_redirect[self::TYPE_TABLET] = $_POST[self::POST_META_REDIRECT_URLS.'-'.self::TYPE_TABLET];
		  }
		  
		  // save updated array
		  $this->savePostMeta($post_id, $post_redirect);
		}
  }
}