<?php
  /**
 * Mobile Smart - Widget Control class
 * - manages visibility of widgets
 */
 
class MobileSmartWidgetControl extends MobileSmartPluginBase {
  
  const SHOW_ALL = 'showAll';
  const SHOW_MOBILE = 'showMobile';
  const SHOW_MOBILE_AND_TABLET = 'showMobileAndTablet';
  const SHOW_MOBILE_AND_DESKTOP = 'showMobileAndDesktop';
  const SHOW_TABLET = 'showTablet';
  const SHOW_TABLET_AND_DESKTOP = 'showTabletAndDesktop';
  const SHOW_DESKTOP = 'showDesktop';
  const HIDE_ALL = 0;
  
  /**
   * constructor - sets up actions & filters - note we're using a singleton pattern for this class due to external callbacks
   * 
   * @access public
   * @return void
   */
  public function __construct() {
    parent::__construct();
    
    // Sidebars
    if (is_admin()) {
      add_action('sidebar_admin_setup', array($this, 'addControl'));
      add_filter('widget_update_callback', 'mobileSidebars_ajax_update_callback', 10, 4); 				// widget changes submitted by ajax method
    }
    else  {
      add_filter( 'sidebars_widgets', array($this, 'filterSidebarWidgets'), 10, 1);
    }
  }
  
  
  /**
   * Singleton pattern.
   * 
   * @access public
   * @static
   * @return void
   */
  public static function Instance() {
    static $instance = null;
    
    if ($instance === null) {
      $instance = new MobileSmartWidgetControl();
    }
    
    return $instance;
  }
     
  
  /**
   * addControl
   * - add our control to the widget
   * 
   * @access public
   * @return void
   */
  public function addControl() {
    global $wp_registered_widgets;
    global $wp_registered_widget_controls;
    
    foreach ( $wp_registered_widgets as $id => $widget )
    {	// controll-less widgets need an empty function so the callback function is called.
    	if (!$wp_registered_widget_controls[$id])
    		wp_register_widget_control($id,$widget['name'], 'mobileSidebars_emptyControl');
    		
      // replace the current callback with our own callback, and save the existing callback to be called by our own method
      $wp_registered_widget_controls[$id]['callback_mobilesmart_redirect'] = $wp_registered_widget_controls[$id]['callback'];
    	$wp_registered_widget_controls[$id]['callback'] = 'mobileSidebars_mobileControl';
    	array_push($wp_registered_widget_controls[$id]['params'],$id);	
    }
   
  }
     
   /**
    * adds the checkbox for each widget
    */
   public function mobileControl($params) {
    global $wp_registered_widget_controls;
    
    $options = $this->getAdminOptions();
  
    // get widget id
    $id = array_pop($params);
  
    // go to the original control function for the widget
    $callback = $wp_registered_widget_controls[$id]['callback_mobilesmart_redirect'];
    if ( is_callable($callback) )
        call_user_func_array($callback, $params);       
  
    // value should be true by default
    $value = isset( $options['mobileSidebar-'.$id] ) ? $options['mobileSidebar-'.$id] : true;
  
    // dealing with multiple widgets - get the number. if -1 this is the 'template' for the admin interface
    $number = $params[0]['number'];
    if ($number == -1) {
        $number = "%i%"; 
        $value = true;
    }
    $id_disp=$id;
    if ( isset($number) ) 
        $id_disp = $wp_registered_widget_controls[$id]['id_base'].'-'.$number;
  
        $show_mobile = $value === self::SHOW_MOBILE || $value === self::SHOW_MOBILE_AND_TABLET || $value === self::SHOW_MOBILE_AND_DESKTOP || $value === self::SHOW_ALL || $value === true;
        $show_tablet = $value === self::SHOW_TABLET || $value === self::SHOW_MOBILE_AND_TABLET || $value === self::SHOW_TABLET_AND_DESKTOP || $value === self::SHOW_ALL || $value === true;
        $show_desktop = $value === self::SHOW_DESKTOP || $value === self::SHOW_MOBILE_AND_DESKTOP || $value === self::SHOW_TABLET_AND_DESKTOP || $value === self::SHOW_ALL || $value === true;
     ?>
      <fieldset style="border: 1px solid #999; padding: 5px; margin: 10px 0;">
        <legend>Mobile Smart Pro</legend>
      <p>
        <input type="checkbox" value="show-mobile" name="<?php echo $id_disp; ?>-mobileSmartShowMobile" id="<?php echo $id_disp; ?>-mobileSmartShowMobile" <?php echo $show_mobile ? 'checked="checked"' : ''; ?>/>
        <label for="<?php echo $id_disp; ?>-mobileSmartShowMobile">Show widget on mobile device?</label>
      </p>
      <p>
        <input type="checkbox" value="show-tablet" name="<?php echo $id_disp; ?>-mobileSmartShowTablet" id="<?php echo $id_disp; ?>-mobileSmartShowTablet" <?php echo $show_tablet ? 'checked="checked"' : ''; ?>/>
        <label for="<?php echo $id_disp; ?>-mobileSmartShowTablet">Show widget on tablet?</label>
      </p>
      <p>
        <input type="checkbox" value="show-tablet" name="<?php echo $id_disp; ?>-mobileSmartShowDesktop" id="<?php echo $id_disp; ?>-mobileSmartShowDesktop" <?php echo $show_desktop ? 'checked="checked"' : ''; ?>/>
        <label for="<?php echo $id_disp; ?>-mobileSmartShowDesktop">Show widget on desktop?</label>
      </p>
      </fieldset>
     <?php
   }
     
     
  /**
   * emptyControl function.
   * 
   * @access public
   * @return void
   */
  function emptyControl() {
    // deliberately empty
  }
   
  /**
   * Callback to save widget control content
   */
  function ajaxUpdateCallback($instance, $new_instance, $old_instance, $this_widget) {
    $options = $this->getAdminOptions();
    
  	$widget_id = $this_widget->id;
  	
  	if (   isset($_POST[$widget_id.'-mobileSmartShowMobile'])
  	    && isset($_POST[$widget_id.'-mobileSmartShowTablet'])
  	    && isset($_POST[$widget_id.'-mobileSmartShowDesktop'])) {
    	$value = self::SHOW_ALL;
  	}
  	else if (   isset($_POST[$widget_id.'-mobileSmartShowMobile'])
  	         && !isset($_POST[$widget_id.'-mobileSmartShowTablet'])
  	         && !isset($_POST[$widget_id.'-mobileSmartShowDesktop'])) {
      $value = self::SHOW_MOBILE;
  	}
  	else if (   isset($_POST[$widget_id.'-mobileSmartShowMobile'])
  	         && !isset($_POST[$widget_id.'-mobileSmartShowTablet'])
  	         && isset($_POST[$widget_id.'-mobileSmartShowDesktop'])) {
      $value = self::SHOW_MOBILE_AND_DESKTOP;
  	}
  	else if (   isset($_POST[$widget_id.'-mobileSmartShowMobile'])
  	         && isset($_POST[$widget_id.'-mobileSmartShowTablet'])
  	         && !isset($_POST[$widget_id.'-mobileSmartShowDesktop'])) {
      $value = self::SHOW_MOBILE_AND_TABLET;
  	}
    else if (   !isset($_POST[$widget_id.'-mobileSmartShowMobile'])
             && isset($_POST[$widget_id.'-mobileSmartShowTablet'])
             && !isset($_POST[$widget_id.'-mobileSmartShowDesktop'])) {
      $value = self::SHOW_TABLET;
  	}
  	else if (   !isset($_POST[$widget_id.'-mobileSmartShowMobile'])
  	         && isset($_POST[$widget_id.'-mobileSmartShowTablet'])
  	         && isset($_POST[$widget_id.'-mobileSmartShowDesktop'])) {
      $value = self::SHOW_TABLET_AND_DESKTOP;
  	}
  	else if (   !isset($_POST[$widget_id.'-mobileSmartShowMobile'])
             && !isset($_POST[$widget_id.'-mobileSmartShowTablet'])
             && isset($_POST[$widget_id.'-mobileSmartShowDesktop'])) {
      $value = self::SHOW_DESKTOP;
  	}
  	else {
  	  $value = self::HIDE_ALL;
    }
    
    $options['mobileSidebar-'.$widget_id] = $value;    
  	
    // update the options
    $this->updateOptions($options);
    
  	return $instance;
  }
  
  /**
   * only display the sidebar if needed
   */
  function filterSidebarWidgets($sidebars_widgets) {
      
    $options = $this->getAdminOptions();
    
  	// loop through every widget in every sidebar (barring 'wp_inactive_widgets') checking WL for each one
  	foreach($sidebars_widgets as $widget_area => $widget_list)
  	{
  	  if ($widget_area == 'wp_inactive_widgets' || empty($widget_list)) continue;
  
  		foreach ($widget_list as $pos => $widget_id)
  		{
  		  // only hide if explicitly set
  		  $display_widget = true;
  		  
  		  // if user has set options
  		  if (isset($options['mobileSidebar-'.$widget_id])) {
    		  // set by default
    		  $display_widget = false;
    		  
    		  $value = $options['mobileSidebar-'.$widget_id];
    		  
    		  $show_mobile = ($this->isMobile() && !$this->isTablet()) && ($value === self::SHOW_MOBILE || $value === self::SHOW_MOBILE_AND_TABLET || $value === self::SHOW_MOBILE_AND_DESKTOP || $value === self::SHOW_ALL || $value === true);
          $show_tablet = $this->isTablet() && ($value === self::SHOW_TABLET || $value === self::SHOW_MOBILE_AND_TABLET || $value === self::SHOW_TABLET_AND_DESKTOP || $value === self::SHOW_ALL || $value === true);
          $show_desktop = !$this->isMobile() && !$this->isTablet() && ($value === self::SHOW_DESKTOP || $value === self::SHOW_MOBILE_AND_DESKTOP || $value === self::SHOW_TABLET_AND_DESKTOP || $value === self::SHOW_ALL || $value === true);
          
/*
          echo "<pre>"; 
          echo "\n\nmobileSidebar-".$widget_id."\nDevices:\n";
          var_dump($this->isMobile());
          var_dump($this->isTablet());
          var_dump(!$this->isMobile());
          echo "Show: \n";
          var_dump($show_mobile);
          var_dump($show_tablet);
          var_dump($show_desktop);
          echo "Stored value: ".$value."\n";
          echo "*" . (($show_mobile || $show_tablet || $show_desktop) ? 'Widget shown' : 'Widget hidden');
          echo "</pre>";
          die();
*/
    		  
    		  // if we are on tablet and not mobile
    		  if ($show_mobile || $show_tablet || $show_desktop) {
      		  $display_widget = true;
    		  }
    		}

        // display widget
  			if (!$display_widget) {
  				unset($sidebars_widgets[$widget_area][$pos]);
        }
  		}
  	}
  	return $sidebars_widgets;
  }
} // end class

/**
 * Callbacks - these need to be outside the class as there's not a simple way of putting the callback inside a class unfortunately
 */

function mobileSidebars_mobileControl() {
   $mobile_smart_sidebars = MobileSmartWidgetControl::Instance();
   
   $params = func_get_args();
   
   $mobile_smart_sidebars->mobileControl($params);
}

function mobileSidebars_emptyControl() {
   $mobile_smart_sidebars = MobileSmartWidgetControl::Instance();
   
   $mobile_smart_sidebars->emptyControl();
}

function mobileSidebars_ajax_update_callback($instance, $new_instance, $old_instance, $this_widget) {
  $mobile_smart_sidebars = MobileSmartWidgetControl::Instance();
  
  return $mobile_smart_sidebars->ajaxUpdateCallback($instance, $new_instance, $old_instance, $this_widget);
}