<?php
/**
 * @author WP Cloud Plugins
 * @copyright Copyright (c) 2023, WP Cloud Plugins
 *
 * @since       2.8.4
 * @see https://www.wpcloudplugins.com
 */

namespace TheLion\ShareoneDrive;

class Thumbnail
{
    /**
     * @var Entry
     */
    private $_entry;

    /**
     * @var int
     */
    private $_width;

    /**
     * @var int
     */
    private $_height;

    /**
     * @var bool
     */
    private $_crop;

    private $thumbnail_url;

    /**
     * @var string
     */
    private $_location_thumbnails = SHAREONEDRIVE_CACHEDIR.'thumbnails/';

    /**
     * @var string
     */
    private $_location_thumbnails_url = SHAREONEDRIVE_CACHEURL.'thumbnails/';

    /**
     * @var string
     */
    private $_thumbnail_name;

    public function __construct(Entry $entry, $thumbnail_url)
    {
        $this->_entry = $entry;
        $this->thumbnail_url = $thumbnail_url;

        $re_size = '/width=(?<width>\d*)&height=(?<height>\d*)/';
        preg_match($re_size, $thumbnail_url, $matches);
        $this->_width = $matches['width'] ?? null;
        $this->_height = $matches['height'] ?? null;

        $re_crop = '/cropmode=(?<width>\w*)/';
        preg_match($re_crop, $thumbnail_url, $matches);
        $this->_crop = $matches['cropmode'] ?? null;

        $this->_thumbnail_name = $this->_get_entry()->get_id().'_w'.$this->_width.'_h'.$this->_height.'_c'.(($this->_crop) ? '1' : '0').'.jpeg';
    }

    public function get_url()
    {
        if (in_array(App::get_current_account()->get_type(), ['personal'])) {
            return $this->thumbnail_url;
        }

        if ('onedrive' === Settings::get('thumbnail_location') && User::can_download()) {
            return $this->thumbnail_url;
        }

        if (false === $this->_get_entry()->has_own_thumbnail()) {
            return $this->thumbnail_url;
        }

        if ($this->does_thumbnail_exist()) {
            return $this->get_location_thumbnail_url();
        }

        return $this->_build_thumbnail_url();
    }

    public function does_thumbnail_exist()
    {
        if (!file_exists($this->get_location_thumbnail())) {
            return false;
        }

        if (filemtime($this->get_location_thumbnail()) !== $this->_get_entry()->get_last_edited()) {
            return false;
        }

        if (filesize($this->get_location_thumbnail()) < 1) {
            return false;
        }

        return $this->get_location_thumbnail();
    }

    public function get_thumbnail_name()
    {
        return $this->_thumbnail_name;
    }

    public function build_thumbnail()
    {
        Helpers::set_time_limit(60); // Downloading the thumbnail can take a while

        $this->_create_thumbnail_dir();

        // Get file
        $request = new \SODOneDrive_Http_Request($this->thumbnail_url, 'GET');

        App::instance()->get_sdk_client()->getIo()->setOptions(
            [
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HEADER => true,
            ]
        );

        $request->enableExpectedRaw();

        try {
            $thumbnail_data = App::instance()->get_sdk_client()->execute($request);

            $thumbnail_location = $this->get_location_thumbnail();
            file_put_contents($thumbnail_location, $thumbnail_data);
            touch($thumbnail_location, $this->_get_entry()->get_last_edited());

            curl_close(App::instance()->get_sdk_client()->getIo()->getHandler());
        } catch (\Exception $ex) {
            return false;
        }

        App::instance()->get_sdk_client()->getIo()->clearOptions();

        return true;
    }

    public function get_location_thumbnail()
    {
        return $this->_location_thumbnails.$this->get_thumbnail_name();
    }

    public function get_location_thumbnail_url()
    {
        return $this->_location_thumbnails_url.$this->get_thumbnail_name();
    }

    private function _build_thumbnail_url()
    {
        return SHAREONEDRIVE_ADMIN_URL.'?action=shareonedrive-thumbnail&src='.$this->_thumbnail_name.'&account_id='.App::get_current_account()->get_id().'&drive_id='.App::get_current_drive_id();
    }

    private function _create_thumbnail_dir()
    {
        if (!file_exists($this->_location_thumbnails)) {
            @mkdir($this->_location_thumbnails, 0755);
        } else {
            return true;
        }

        if (!is_writable($this->_location_thumbnails)) {
            @chmod($this->_location_thumbnails, 0755);
        } else {
            return true;
        }

        return is_writable($this->_location_thumbnails);
    }

    /**
     * @return Entry
     */
    private function _get_entry()
    {
        return $this->_entry;
    }
}
