<?php

// {{{ license

/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4 foldmethod=marker: */
//
// +----------------------------------------------------------------------+
// | This library is free software; you can redistribute it and/or modify |
// | it under the terms of the GNU Lesser General Public License as       |
// | published by the Free Software Foundation; either version 2.1 of the |
// | License, or (at your option) any later version.                      |
// |                                                                      |
// | This library is distributed in the hope that it will be useful, but  |
// | WITHOUT ANY WARRANTY; without even the implied warranty of           |
// | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    |
// | Lesser General Public License for more details.                      |
// |                                                                      |
// | You should have received a copy of the GNU Lesser General Public     |
// | License along with this library; if not, write to the Free Software  |
// | Foundation, Inc., 51 Franklin St, Boston, MA 02110, United States    |
// +----------------------------------------------------------------------+
//
// }}}

/**
 * Encode/decode Internationalized Domain Names.
 *
 * The class allows to convert internationalized domain names
 * (see RFC 3490 for details) as they can be used with various registries worldwide
 * to be translated between their original (localized) form and their encoded form
 * as it will be used in the DNS (Domain Name System).
 *
 * The class provides two public methods, encode() and decode(), which do exactly
 * what you would expect them to do. You are allowed to use complete domain names,
 * simple strings and complete email addresses as well. That means, that you might
 * use any of the following notations:
 *
 * - www.nörgler.com
 * - xn--nrgler-wxa
 * - xn--brse-5qa.xn--knrz-1ra.info
 *
 * Unicode input might be given as either UTF-8 string, UCS-4 string or UCS-4 array.
 * Unicode output is available in the same formats.
 * You can select your preferred format via {@link set_paramter()}.
 *
 * ACE input and output is always expected to be ASCII.
 *
 * @author  Matthias Sommerfeld <mso@phlylabs.de>
 * @copyright 2004-2014 phlyLabs Berlin, http://phlylabs.de
 * @version 0.9.0 2014-12-12
 */
class idna_convert {
	
	protected static $_mb_string_overload = null;
	/**
	 * Holds all relevant mapping tables
	 * See RFC3454 for details
	 *
	 * @private array
	 * @since 0.5.2
	 */
	protected static $NP = [
		'map_nothing'        => [
			0xAD,
			0x34F,
			0x1806,
			0x180B,
			0x180C,
			0x180D,
			0x200B,
			0x200C,
			0x200D,
			0x2060,
			0xFE00,
			0xFE01,
			0xFE02,
			0xFE03,
			0xFE04,
			0xFE05,
			0xFE06,
			0xFE07,
			0xFE08,
			0xFE09,
			0xFE0A,
			0xFE0B,
			0xFE0C,
			0xFE0D,
			0xFE0E,
			0xFE0F,
			0xFEFF
		],
		'general_prohibited' => [
			0,
			1,
			2,
			3,
			4,
			5,
			6,
			7,
			8,
			9,
			10,
			11,
			12,
			13,
			14,
			15,
			16,
			17,
			18,
			19,
			20,
			21,
			22,
			23,
			24,
			25,
			26,
			27,
			28,
			29,
			30,
			31,
			32,
			33,
			34,
			35,
			36,
			37,
			38,
			39,
			40,
			41,
			42,
			43,
			44,
			47,
			59,
			60,
			61,
			62,
			63,
			64,
			91,
			92,
			93,
			94,
			95,
			96,
			123,
			124,
			125,
			126,
			127,
			0x3002
		],
		'prohibit'           => [
			0xA0,
			0x340,
			0x341,
			0x6DD,
			0x70F,
			0x1680,
			0x180E,
			0x2000,
			0x2001,
			0x2002,
			0x2003,
			0x2004,
			0x2005,
			0x2006,
			0x2007,
			0x2008,
			0x2009,
			0x200A,
			0x200B,
			0x200C,
			0x200D,
			0x200E,
			0x200F,
			0x2028,
			0x2029,
			0x202A,
			0x202B,
			0x202C,
			0x202D,
			0x202E,
			0x202F,
			0x205F,
			0x206A,
			0x206B,
			0x206C,
			0x206D,
			0x206E,
			0x206F,
			0x3000,
			0x33C2,
			0xFEFF,
			0xFFF9,
			0xFFFA,
			0xFFFB,
			0xFFFC,
			0xFFFD,
			0xFFFE,
			0xFFFF,
			0x1FFFE,
			0x1FFFF,
			0x2FFFE,
			0x2FFFF,
			0x3FFFE,
			0x3FFFF,
			0x4FFFE,
			0x4FFFF,
			0x5FFFE,
			0x5FFFF,
			0x6FFFE,
			0x6FFFF,
			0x7FFFE,
			0x7FFFF,
			0x8FFFE,
			0x8FFFF,
			0x9FFFE,
			0x9FFFF,
			0xAFFFE,
			0xAFFFF,
			0xBFFFE,
			0xBFFFF,
			0xCFFFE,
			0xCFFFF,
			0xDFFFE,
			0xDFFFF,
			0xE0001,
			0xEFFFE,
			0xEFFFF,
			0xFFFFE,
			0xFFFFF,
			0x10FFFE,
			0x10FFFF
		],
		'prohibit_ranges'    => [
			[ 0x80, 0x9F ],
			[ 0x2060, 0x206F ],
			[ 0x1D173, 0x1D17A ],
			[ 0xE000, 0xF8FF ],
			[ 0xF0000, 0xFFFFD ],
			[ 0x100000, 0x10FFFD ],
			[ 0xFDD0, 0xFDEF ],
			[ 0xD800, 0xDFFF ],
			[ 0x2FF0, 0x2FFB ],
			[ 0xE0020, 0xE007F ]
		],
		'replacemaps_2003'   => [
			0x41    => [ 0x61 ],
			0x42    => [ 0x62 ],
			0x43    => [ 0x63 ],
			0x44    => [ 0x64 ],
			0x45    => [ 0x65 ],
			0x46    => [ 0x66 ],
			0x47    => [ 0x67 ],
			0x48    => [ 0x68 ],
			0x49    => [ 0x69 ],
			0x4A    => [ 0x6A ],
			0x4B    => [ 0x6B ],
			0x4C    => [ 0x6C ],
			0x4D    => [ 0x6D ],
			0x4E    => [ 0x6E ],
			0x4F    => [ 0x6F ],
			0x50    => [ 0x70 ],
			0x51    => [ 0x71 ],
			0x52    => [ 0x72 ],
			0x53    => [ 0x73 ],
			0x54    => [ 0x74 ],
			0x55    => [ 0x75 ],
			0x56    => [ 0x76 ],
			0x57    => [ 0x77 ],
			0x58    => [ 0x78 ],
			0x59    => [ 0x79 ],
			0x5A    => [ 0x7A ],
			0xB5    => [ 0x3BC ],
			0xC0    => [ 0xE0 ],
			0xC1    => [ 0xE1 ],
			0xC2    => [ 0xE2 ],
			0xC3    => [ 0xE3 ],
			0xC4    => [ 0xE4 ],
			0xC5    => [ 0xE5 ],
			0xC6    => [ 0xE6 ],
			0xC7    => [ 0xE7 ],
			0xC8    => [ 0xE8 ],
			0xC9    => [ 0xE9 ],
			0xCA    => [ 0xEA ],
			0xCB    => [ 0xEB ],
			0xCC    => [ 0xEC ],
			0xCD    => [ 0xED ],
			0xCE    => [ 0xEE ],
			0xCF    => [ 0xEF ],
			0xD0    => [ 0xF0 ],
			0xD1    => [ 0xF1 ],
			0xD2    => [ 0xF2 ],
			0xD3    => [ 0xF3 ],
			0xD4    => [ 0xF4 ],
			0xD5    => [ 0xF5 ],
			0xD6    => [ 0xF6 ],
			0xD8    => [ 0xF8 ],
			0xD9    => [ 0xF9 ],
			0xDA    => [ 0xFA ],
			0xDB    => [ 0xFB ],
			0xDC    => [ 0xFC ],
			0xDD    => [ 0xFD ],
			0xDE    => [ 0xFE ],
			0xDF    => [ 0x73, 0x73 ],
			0x100   => [ 0x101 ],
			0x102   => [ 0x103 ],
			0x104   => [ 0x105 ],
			0x106   => [ 0x107 ],
			0x108   => [ 0x109 ],
			0x10A   => [ 0x10B ],
			0x10C   => [ 0x10D ],
			0x10E   => [ 0x10F ],
			0x110   => [ 0x111 ],
			0x112   => [ 0x113 ],
			0x114   => [ 0x115 ],
			0x116   => [ 0x117 ],
			0x118   => [ 0x119 ],
			0x11A   => [ 0x11B ],
			0x11C   => [ 0x11D ],
			0x11E   => [ 0x11F ],
			0x120   => [ 0x121 ],
			0x122   => [ 0x123 ],
			0x124   => [ 0x125 ],
			0x126   => [ 0x127 ],
			0x128   => [ 0x129 ],
			0x12A   => [ 0x12B ],
			0x12C   => [ 0x12D ],
			0x12E   => [ 0x12F ],
			0x130   => [ 0x69, 0x307 ],
			0x132   => [ 0x133 ],
			0x134   => [ 0x135 ],
			0x136   => [ 0x137 ],
			0x139   => [ 0x13A ],
			0x13B   => [ 0x13C ],
			0x13D   => [ 0x13E ],
			0x13F   => [ 0x140 ],
			0x141   => [ 0x142 ],
			0x143   => [ 0x144 ],
			0x145   => [ 0x146 ],
			0x147   => [ 0x148 ],
			0x149   => [ 0x2BC, 0x6E ],
			0x14A   => [ 0x14B ],
			0x14C   => [ 0x14D ],
			0x14E   => [ 0x14F ],
			0x150   => [ 0x151 ],
			0x152   => [ 0x153 ],
			0x154   => [ 0x155 ],
			0x156   => [ 0x157 ],
			0x158   => [ 0x159 ],
			0x15A   => [ 0x15B ],
			0x15C   => [ 0x15D ],
			0x15E   => [ 0x15F ],
			0x160   => [ 0x161 ],
			0x162   => [ 0x163 ],
			0x164   => [ 0x165 ],
			0x166   => [ 0x167 ],
			0x168   => [ 0x169 ],
			0x16A   => [ 0x16B ],
			0x16C   => [ 0x16D ],
			0x16E   => [ 0x16F ],
			0x170   => [ 0x171 ],
			0x172   => [ 0x173 ],
			0x174   => [ 0x175 ],
			0x176   => [ 0x177 ],
			0x178   => [ 0xFF ],
			0x179   => [ 0x17A ],
			0x17B   => [ 0x17C ],
			0x17D   => [ 0x17E ],
			0x17F   => [ 0x73 ],
			0x181   => [ 0x253 ],
			0x182   => [ 0x183 ],
			0x184   => [ 0x185 ],
			0x186   => [ 0x254 ],
			0x187   => [ 0x188 ],
			0x189   => [ 0x256 ],
			0x18A   => [ 0x257 ],
			0x18B   => [ 0x18C ],
			0x18E   => [ 0x1DD ],
			0x18F   => [ 0x259 ],
			0x190   => [ 0x25B ],
			0x191   => [ 0x192 ],
			0x193   => [ 0x260 ],
			0x194   => [ 0x263 ],
			0x196   => [ 0x269 ],
			0x197   => [ 0x268 ],
			0x198   => [ 0x199 ],
			0x19C   => [ 0x26F ],
			0x19D   => [ 0x272 ],
			0x19F   => [ 0x275 ],
			0x1A0   => [ 0x1A1 ],
			0x1A2   => [ 0x1A3 ],
			0x1A4   => [ 0x1A5 ],
			0x1A6   => [ 0x280 ],
			0x1A7   => [ 0x1A8 ],
			0x1A9   => [ 0x283 ],
			0x1AC   => [ 0x1AD ],
			0x1AE   => [ 0x288 ],
			0x1AF   => [ 0x1B0 ],
			0x1B1   => [ 0x28A ],
			0x1B2   => [ 0x28B ],
			0x1B3   => [ 0x1B4 ],
			0x1B5   => [ 0x1B6 ],
			0x1B7   => [ 0x292 ],
			0x1B8   => [ 0x1B9 ],
			0x1BC   => [ 0x1BD ],
			0x1C4   => [ 0x1C6 ],
			0x1C5   => [ 0x1C6 ],
			0x1C7   => [ 0x1C9 ],
			0x1C8   => [ 0x1C9 ],
			0x1CA   => [ 0x1CC ],
			0x1CB   => [ 0x1CC ],
			0x1CD   => [ 0x1CE ],
			0x1CF   => [ 0x1D0 ],
			0x1D1   => [ 0x1D2 ],
			0x1D3   => [ 0x1D4 ],
			0x1D5   => [ 0x1D6 ],
			0x1D7   => [ 0x1D8 ],
			0x1D9   => [ 0x1DA ],
			0x1DB   => [ 0x1DC ],
			0x1DE   => [ 0x1DF ],
			0x1E0   => [ 0x1E1 ],
			0x1E2   => [ 0x1E3 ],
			0x1E4   => [ 0x1E5 ],
			0x1E6   => [ 0x1E7 ],
			0x1E8   => [ 0x1E9 ],
			0x1EA   => [ 0x1EB ],
			0x1EC   => [ 0x1ED ],
			0x1EE   => [ 0x1EF ],
			0x1F0   => [ 0x6A, 0x30C ],
			0x1F1   => [ 0x1F3 ],
			0x1F2   => [ 0x1F3 ],
			0x1F4   => [ 0x1F5 ],
			0x1F6   => [ 0x195 ],
			0x1F7   => [ 0x1BF ],
			0x1F8   => [ 0x1F9 ],
			0x1FA   => [ 0x1FB ],
			0x1FC   => [ 0x1FD ],
			0x1FE   => [ 0x1FF ],
			0x200   => [ 0x201 ],
			0x202   => [ 0x203 ],
			0x204   => [ 0x205 ],
			0x206   => [ 0x207 ],
			0x208   => [ 0x209 ],
			0x20A   => [ 0x20B ],
			0x20C   => [ 0x20D ],
			0x20E   => [ 0x20F ],
			0x210   => [ 0x211 ],
			0x212   => [ 0x213 ],
			0x214   => [ 0x215 ],
			0x216   => [ 0x217 ],
			0x218   => [ 0x219 ],
			0x21A   => [ 0x21B ],
			0x21C   => [ 0x21D ],
			0x21E   => [ 0x21F ],
			0x220   => [ 0x19E ],
			0x222   => [ 0x223 ],
			0x224   => [ 0x225 ],
			0x226   => [ 0x227 ],
			0x228   => [ 0x229 ],
			0x22A   => [ 0x22B ],
			0x22C   => [ 0x22D ],
			0x22E   => [ 0x22F ],
			0x230   => [ 0x231 ],
			0x232   => [ 0x233 ],
			0x345   => [ 0x3B9 ],
			0x37A   => [ 0x20, 0x3B9 ],
			0x386   => [ 0x3AC ],
			0x388   => [ 0x3AD ],
			0x389   => [ 0x3AE ],
			0x38A   => [ 0x3AF ],
			0x38C   => [ 0x3CC ],
			0x38E   => [ 0x3CD ],
			0x38F   => [ 0x3CE ],
			0x390   => [ 0x3B9, 0x308, 0x301 ],
			0x391   => [ 0x3B1 ],
			0x392   => [ 0x3B2 ],
			0x393   => [ 0x3B3 ],
			0x394   => [ 0x3B4 ],
			0x395   => [ 0x3B5 ],
			0x396   => [ 0x3B6 ],
			0x397   => [ 0x3B7 ],
			0x398   => [ 0x3B8 ],
			0x399   => [ 0x3B9 ],
			0x39A   => [ 0x3BA ],
			0x39B   => [ 0x3BB ],
			0x39C   => [ 0x3BC ],
			0x39D   => [ 0x3BD ],
			0x39E   => [ 0x3BE ],
			0x39F   => [ 0x3BF ],
			0x3A0   => [ 0x3C0 ],
			0x3A1   => [ 0x3C1 ],
			0x3A3   => [ 0x3C3 ],
			0x3A4   => [ 0x3C4 ],
			0x3A5   => [ 0x3C5 ],
			0x3A6   => [ 0x3C6 ],
			0x3A7   => [ 0x3C7 ],
			0x3A8   => [ 0x3C8 ],
			0x3A9   => [ 0x3C9 ],
			0x3AA   => [ 0x3CA ],
			0x3AB   => [ 0x3CB ],
			0x3B0   => [ 0x3C5, 0x308, 0x301 ],
			0x3C2   => [ 0x3C3 ],
			0x3D0   => [ 0x3B2 ],
			0x3D1   => [ 0x3B8 ],
			0x3D2   => [ 0x3C5 ],
			0x3D3   => [ 0x3CD ],
			0x3D4   => [ 0x3CB ],
			0x3D5   => [ 0x3C6 ],
			0x3D6   => [ 0x3C0 ],
			0x3D8   => [ 0x3D9 ],
			0x3DA   => [ 0x3DB ],
			0x3DC   => [ 0x3DD ],
			0x3DE   => [ 0x3DF ],
			0x3E0   => [ 0x3E1 ],
			0x3E2   => [ 0x3E3 ],
			0x3E4   => [ 0x3E5 ],
			0x3E6   => [ 0x3E7 ],
			0x3E8   => [ 0x3E9 ],
			0x3EA   => [ 0x3EB ],
			0x3EC   => [ 0x3ED ],
			0x3EE   => [ 0x3EF ],
			0x3F0   => [ 0x3BA ],
			0x3F1   => [ 0x3C1 ],
			0x3F2   => [ 0x3C3 ],
			0x3F4   => [ 0x3B8 ],
			0x3F5   => [ 0x3B5 ],
			0x400   => [ 0x450 ],
			0x401   => [ 0x451 ],
			0x402   => [ 0x452 ],
			0x403   => [ 0x453 ],
			0x404   => [ 0x454 ],
			0x405   => [ 0x455 ],
			0x406   => [ 0x456 ],
			0x407   => [ 0x457 ],
			0x408   => [ 0x458 ],
			0x409   => [ 0x459 ],
			0x40A   => [ 0x45A ],
			0x40B   => [ 0x45B ],
			0x40C   => [ 0x45C ],
			0x40D   => [ 0x45D ],
			0x40E   => [ 0x45E ],
			0x40F   => [ 0x45F ],
			0x410   => [ 0x430 ],
			0x411   => [ 0x431 ],
			0x412   => [ 0x432 ],
			0x413   => [ 0x433 ],
			0x414   => [ 0x434 ],
			0x415   => [ 0x435 ],
			0x416   => [ 0x436 ],
			0x417   => [ 0x437 ],
			0x418   => [ 0x438 ],
			0x419   => [ 0x439 ],
			0x41A   => [ 0x43A ],
			0x41B   => [ 0x43B ],
			0x41C   => [ 0x43C ],
			0x41D   => [ 0x43D ],
			0x41E   => [ 0x43E ],
			0x41F   => [ 0x43F ],
			0x420   => [ 0x440 ],
			0x421   => [ 0x441 ],
			0x422   => [ 0x442 ],
			0x423   => [ 0x443 ],
			0x424   => [ 0x444 ],
			0x425   => [ 0x445 ],
			0x426   => [ 0x446 ],
			0x427   => [ 0x447 ],
			0x428   => [ 0x448 ],
			0x429   => [ 0x449 ],
			0x42A   => [ 0x44A ],
			0x42B   => [ 0x44B ],
			0x42C   => [ 0x44C ],
			0x42D   => [ 0x44D ],
			0x42E   => [ 0x44E ],
			0x42F   => [ 0x44F ],
			0x460   => [ 0x461 ],
			0x462   => [ 0x463 ],
			0x464   => [ 0x465 ],
			0x466   => [ 0x467 ],
			0x468   => [ 0x469 ],
			0x46A   => [ 0x46B ],
			0x46C   => [ 0x46D ],
			0x46E   => [ 0x46F ],
			0x470   => [ 0x471 ],
			0x472   => [ 0x473 ],
			0x474   => [ 0x475 ],
			0x476   => [ 0x477 ],
			0x478   => [ 0x479 ],
			0x47A   => [ 0x47B ],
			0x47C   => [ 0x47D ],
			0x47E   => [ 0x47F ],
			0x480   => [ 0x481 ],
			0x48A   => [ 0x48B ],
			0x48C   => [ 0x48D ],
			0x48E   => [ 0x48F ],
			0x490   => [ 0x491 ],
			0x492   => [ 0x493 ],
			0x494   => [ 0x495 ],
			0x496   => [ 0x497 ],
			0x498   => [ 0x499 ],
			0x49A   => [ 0x49B ],
			0x49C   => [ 0x49D ],
			0x49E   => [ 0x49F ],
			0x4A0   => [ 0x4A1 ],
			0x4A2   => [ 0x4A3 ],
			0x4A4   => [ 0x4A5 ],
			0x4A6   => [ 0x4A7 ],
			0x4A8   => [ 0x4A9 ],
			0x4AA   => [ 0x4AB ],
			0x4AC   => [ 0x4AD ],
			0x4AE   => [ 0x4AF ],
			0x4B0   => [ 0x4B1 ],
			0x4B2   => [ 0x4B3 ],
			0x4B4   => [ 0x4B5 ],
			0x4B6   => [ 0x4B7 ],
			0x4B8   => [ 0x4B9 ],
			0x4BA   => [ 0x4BB ],
			0x4BC   => [ 0x4BD ],
			0x4BE   => [ 0x4BF ],
			0x4C1   => [ 0x4C2 ],
			0x4C3   => [ 0x4C4 ],
			0x4C5   => [ 0x4C6 ],
			0x4C7   => [ 0x4C8 ],
			0x4C9   => [ 0x4CA ],
			0x4CB   => [ 0x4CC ],
			0x4CD   => [ 0x4CE ],
			0x4D0   => [ 0x4D1 ],
			0x4D2   => [ 0x4D3 ],
			0x4D4   => [ 0x4D5 ],
			0x4D6   => [ 0x4D7 ],
			0x4D8   => [ 0x4D9 ],
			0x4DA   => [ 0x4DB ],
			0x4DC   => [ 0x4DD ],
			0x4DE   => [ 0x4DF ],
			0x4E0   => [ 0x4E1 ],
			0x4E2   => [ 0x4E3 ],
			0x4E4   => [ 0x4E5 ],
			0x4E6   => [ 0x4E7 ],
			0x4E8   => [ 0x4E9 ],
			0x4EA   => [ 0x4EB ],
			0x4EC   => [ 0x4ED ],
			0x4EE   => [ 0x4EF ],
			0x4F0   => [ 0x4F1 ],
			0x4F2   => [ 0x4F3 ],
			0x4F4   => [ 0x4F5 ],
			0x4F8   => [ 0x4F9 ],
			0x500   => [ 0x501 ],
			0x502   => [ 0x503 ],
			0x504   => [ 0x505 ],
			0x506   => [ 0x507 ],
			0x508   => [ 0x509 ],
			0x50A   => [ 0x50B ],
			0x50C   => [ 0x50D ],
			0x50E   => [ 0x50F ],
			0x531   => [ 0x561 ],
			0x532   => [ 0x562 ],
			0x533   => [ 0x563 ],
			0x534   => [ 0x564 ],
			0x535   => [ 0x565 ],
			0x536   => [ 0x566 ],
			0x537   => [ 0x567 ],
			0x538   => [ 0x568 ],
			0x539   => [ 0x569 ],
			0x53A   => [ 0x56A ],
			0x53B   => [ 0x56B ],
			0x53C   => [ 0x56C ],
			0x53D   => [ 0x56D ],
			0x53E   => [ 0x56E ],
			0x53F   => [ 0x56F ],
			0x540   => [ 0x570 ],
			0x541   => [ 0x571 ],
			0x542   => [ 0x572 ],
			0x543   => [ 0x573 ],
			0x544   => [ 0x574 ],
			0x545   => [ 0x575 ],
			0x546   => [ 0x576 ],
			0x547   => [ 0x577 ],
			0x548   => [ 0x578 ],
			0x549   => [ 0x579 ],
			0x54A   => [ 0x57A ],
			0x54B   => [ 0x57B ],
			0x54C   => [ 0x57C ],
			0x54D   => [ 0x57D ],
			0x54E   => [ 0x57E ],
			0x54F   => [ 0x57F ],
			0x550   => [ 0x580 ],
			0x551   => [ 0x581 ],
			0x552   => [ 0x582 ],
			0x553   => [ 0x583 ],
			0x554   => [ 0x584 ],
			0x555   => [ 0x585 ],
			0x556   => [ 0x586 ],
			0x587   => [ 0x565, 0x582 ],
			0xE33   => [ 0xE4D, 0xE32 ],
			0x1E00  => [ 0x1E01 ],
			0x1E02  => [ 0x1E03 ],
			0x1E04  => [ 0x1E05 ],
			0x1E06  => [ 0x1E07 ],
			0x1E08  => [ 0x1E09 ],
			0x1E0A  => [ 0x1E0B ],
			0x1E0C  => [ 0x1E0D ],
			0x1E0E  => [ 0x1E0F ],
			0x1E10  => [ 0x1E11 ],
			0x1E12  => [ 0x1E13 ],
			0x1E14  => [ 0x1E15 ],
			0x1E16  => [ 0x1E17 ],
			0x1E18  => [ 0x1E19 ],
			0x1E1A  => [ 0x1E1B ],
			0x1E1C  => [ 0x1E1D ],
			0x1E1E  => [ 0x1E1F ],
			0x1E20  => [ 0x1E21 ],
			0x1E22  => [ 0x1E23 ],
			0x1E24  => [ 0x1E25 ],
			0x1E26  => [ 0x1E27 ],
			0x1E28  => [ 0x1E29 ],
			0x1E2A  => [ 0x1E2B ],
			0x1E2C  => [ 0x1E2D ],
			0x1E2E  => [ 0x1E2F ],
			0x1E30  => [ 0x1E31 ],
			0x1E32  => [ 0x1E33 ],
			0x1E34  => [ 0x1E35 ],
			0x1E36  => [ 0x1E37 ],
			0x1E38  => [ 0x1E39 ],
			0x1E3A  => [ 0x1E3B ],
			0x1E3C  => [ 0x1E3D ],
			0x1E3E  => [ 0x1E3F ],
			0x1E40  => [ 0x1E41 ],
			0x1E42  => [ 0x1E43 ],
			0x1E44  => [ 0x1E45 ],
			0x1E46  => [ 0x1E47 ],
			0x1E48  => [ 0x1E49 ],
			0x1E4A  => [ 0x1E4B ],
			0x1E4C  => [ 0x1E4D ],
			0x1E4E  => [ 0x1E4F ],
			0x1E50  => [ 0x1E51 ],
			0x1E52  => [ 0x1E53 ],
			0x1E54  => [ 0x1E55 ],
			0x1E56  => [ 0x1E57 ],
			0x1E58  => [ 0x1E59 ],
			0x1E5A  => [ 0x1E5B ],
			0x1E5C  => [ 0x1E5D ],
			0x1E5E  => [ 0x1E5F ],
			0x1E60  => [ 0x1E61 ],
			0x1E62  => [ 0x1E63 ],
			0x1E64  => [ 0x1E65 ],
			0x1E66  => [ 0x1E67 ],
			0x1E68  => [ 0x1E69 ],
			0x1E6A  => [ 0x1E6B ],
			0x1E6C  => [ 0x1E6D ],
			0x1E6E  => [ 0x1E6F ],
			0x1E70  => [ 0x1E71 ],
			0x1E72  => [ 0x1E73 ],
			0x1E74  => [ 0x1E75 ],
			0x1E76  => [ 0x1E77 ],
			0x1E78  => [ 0x1E79 ],
			0x1E7A  => [ 0x1E7B ],
			0x1E7C  => [ 0x1E7D ],
			0x1E7E  => [ 0x1E7F ],
			0x1E80  => [ 0x1E81 ],
			0x1E82  => [ 0x1E83 ],
			0x1E84  => [ 0x1E85 ],
			0x1E86  => [ 0x1E87 ],
			0x1E88  => [ 0x1E89 ],
			0x1E8A  => [ 0x1E8B ],
			0x1E8C  => [ 0x1E8D ],
			0x1E8E  => [ 0x1E8F ],
			0x1E90  => [ 0x1E91 ],
			0x1E92  => [ 0x1E93 ],
			0x1E94  => [ 0x1E95 ],
			0x1E96  => [ 0x68, 0x331 ],
			0x1E97  => [ 0x74, 0x308 ],
			0x1E98  => [ 0x77, 0x30A ],
			0x1E99  => [ 0x79, 0x30A ],
			0x1E9A  => [ 0x61, 0x2BE ],
			0x1E9B  => [ 0x1E61 ],
			0x1EA0  => [ 0x1EA1 ],
			0x1EA2  => [ 0x1EA3 ],
			0x1EA4  => [ 0x1EA5 ],
			0x1EA6  => [ 0x1EA7 ],
			0x1EA8  => [ 0x1EA9 ],
			0x1EAA  => [ 0x1EAB ],
			0x1EAC  => [ 0x1EAD ],
			0x1EAE  => [ 0x1EAF ],
			0x1EB0  => [ 0x1EB1 ],
			0x1EB2  => [ 0x1EB3 ],
			0x1EB4  => [ 0x1EB5 ],
			0x1EB6  => [ 0x1EB7 ],
			0x1EB8  => [ 0x1EB9 ],
			0x1EBA  => [ 0x1EBB ],
			0x1EBC  => [ 0x1EBD ],
			0x1EBE  => [ 0x1EBF ],
			0x1EC0  => [ 0x1EC1 ],
			0x1EC2  => [ 0x1EC3 ],
			0x1EC4  => [ 0x1EC5 ],
			0x1EC6  => [ 0x1EC7 ],
			0x1EC8  => [ 0x1EC9 ],
			0x1ECA  => [ 0x1ECB ],
			0x1ECC  => [ 0x1ECD ],
			0x1ECE  => [ 0x1ECF ],
			0x1ED0  => [ 0x1ED1 ],
			0x1ED2  => [ 0x1ED3 ],
			0x1ED4  => [ 0x1ED5 ],
			0x1ED6  => [ 0x1ED7 ],
			0x1ED8  => [ 0x1ED9 ],
			0x1EDA  => [ 0x1EDB ],
			0x1EDC  => [ 0x1EDD ],
			0x1EDE  => [ 0x1EDF ],
			0x1EE0  => [ 0x1EE1 ],
			0x1EE2  => [ 0x1EE3 ],
			0x1EE4  => [ 0x1EE5 ],
			0x1EE6  => [ 0x1EE7 ],
			0x1EE8  => [ 0x1EE9 ],
			0x1EEA  => [ 0x1EEB ],
			0x1EEC  => [ 0x1EED ],
			0x1EEE  => [ 0x1EEF ],
			0x1EF0  => [ 0x1EF1 ],
			0x1EF2  => [ 0x1EF3 ],
			0x1EF4  => [ 0x1EF5 ],
			0x1EF6  => [ 0x1EF7 ],
			0x1EF8  => [ 0x1EF9 ],
			0x1F08  => [ 0x1F00 ],
			0x1F09  => [ 0x1F01 ],
			0x1F0A  => [ 0x1F02 ],
			0x1F0B  => [ 0x1F03 ],
			0x1F0C  => [ 0x1F04 ],
			0x1F0D  => [ 0x1F05 ],
			0x1F0E  => [ 0x1F06 ],
			0x1F0F  => [ 0x1F07 ],
			0x1F18  => [ 0x1F10 ],
			0x1F19  => [ 0x1F11 ],
			0x1F1A  => [ 0x1F12 ],
			0x1F1B  => [ 0x1F13 ],
			0x1F1C  => [ 0x1F14 ],
			0x1F1D  => [ 0x1F15 ],
			0x1F28  => [ 0x1F20 ],
			0x1F29  => [ 0x1F21 ],
			0x1F2A  => [ 0x1F22 ],
			0x1F2B  => [ 0x1F23 ],
			0x1F2C  => [ 0x1F24 ],
			0x1F2D  => [ 0x1F25 ],
			0x1F2E  => [ 0x1F26 ],
			0x1F2F  => [ 0x1F27 ],
			0x1F38  => [ 0x1F30 ],
			0x1F39  => [ 0x1F31 ],
			0x1F3A  => [ 0x1F32 ],
			0x1F3B  => [ 0x1F33 ],
			0x1F3C  => [ 0x1F34 ],
			0x1F3D  => [ 0x1F35 ],
			0x1F3E  => [ 0x1F36 ],
			0x1F3F  => [ 0x1F37 ],
			0x1F48  => [ 0x1F40 ],
			0x1F49  => [ 0x1F41 ],
			0x1F4A  => [ 0x1F42 ],
			0x1F4B  => [ 0x1F43 ],
			0x1F4C  => [ 0x1F44 ],
			0x1F4D  => [ 0x1F45 ],
			0x1F50  => [ 0x3C5, 0x313 ],
			0x1F52  => [ 0x3C5, 0x313, 0x300 ],
			0x1F54  => [ 0x3C5, 0x313, 0x301 ],
			0x1F56  => [ 0x3C5, 0x313, 0x342 ],
			0x1F59  => [ 0x1F51 ],
			0x1F5B  => [ 0x1F53 ],
			0x1F5D  => [ 0x1F55 ],
			0x1F5F  => [ 0x1F57 ],
			0x1F68  => [ 0x1F60 ],
			0x1F69  => [ 0x1F61 ],
			0x1F6A  => [ 0x1F62 ],
			0x1F6B  => [ 0x1F63 ],
			0x1F6C  => [ 0x1F64 ],
			0x1F6D  => [ 0x1F65 ],
			0x1F6E  => [ 0x1F66 ],
			0x1F6F  => [ 0x1F67 ],
			0x1F80  => [ 0x1F00, 0x3B9 ],
			0x1F81  => [ 0x1F01, 0x3B9 ],
			0x1F82  => [ 0x1F02, 0x3B9 ],
			0x1F83  => [ 0x1F03, 0x3B9 ],
			0x1F84  => [ 0x1F04, 0x3B9 ],
			0x1F85  => [ 0x1F05, 0x3B9 ],
			0x1F86  => [ 0x1F06, 0x3B9 ],
			0x1F87  => [ 0x1F07, 0x3B9 ],
			0x1F88  => [ 0x1F00, 0x3B9 ],
			0x1F89  => [ 0x1F01, 0x3B9 ],
			0x1F8A  => [ 0x1F02, 0x3B9 ],
			0x1F8B  => [ 0x1F03, 0x3B9 ],
			0x1F8C  => [ 0x1F04, 0x3B9 ],
			0x1F8D  => [ 0x1F05, 0x3B9 ],
			0x1F8E  => [ 0x1F06, 0x3B9 ],
			0x1F8F  => [ 0x1F07, 0x3B9 ],
			0x1F90  => [ 0x1F20, 0x3B9 ],
			0x1F91  => [ 0x1F21, 0x3B9 ],
			0x1F92  => [ 0x1F22, 0x3B9 ],
			0x1F93  => [ 0x1F23, 0x3B9 ],
			0x1F94  => [ 0x1F24, 0x3B9 ],
			0x1F95  => [ 0x1F25, 0x3B9 ],
			0x1F96  => [ 0x1F26, 0x3B9 ],
			0x1F97  => [ 0x1F27, 0x3B9 ],
			0x1F98  => [ 0x1F20, 0x3B9 ],
			0x1F99  => [ 0x1F21, 0x3B9 ],
			0x1F9A  => [ 0x1F22, 0x3B9 ],
			0x1F9B  => [ 0x1F23, 0x3B9 ],
			0x1F9C  => [ 0x1F24, 0x3B9 ],
			0x1F9D  => [ 0x1F25, 0x3B9 ],
			0x1F9E  => [ 0x1F26, 0x3B9 ],
			0x1F9F  => [ 0x1F27, 0x3B9 ],
			0x1FA0  => [ 0x1F60, 0x3B9 ],
			0x1FA1  => [ 0x1F61, 0x3B9 ],
			0x1FA2  => [ 0x1F62, 0x3B9 ],
			0x1FA3  => [ 0x1F63, 0x3B9 ],
			0x1FA4  => [ 0x1F64, 0x3B9 ],
			0x1FA5  => [ 0x1F65, 0x3B9 ],
			0x1FA6  => [ 0x1F66, 0x3B9 ],
			0x1FA7  => [ 0x1F67, 0x3B9 ],
			0x1FA8  => [ 0x1F60, 0x3B9 ],
			0x1FA9  => [ 0x1F61, 0x3B9 ],
			0x1FAA  => [ 0x1F62, 0x3B9 ],
			0x1FAB  => [ 0x1F63, 0x3B9 ],
			0x1FAC  => [ 0x1F64, 0x3B9 ],
			0x1FAD  => [ 0x1F65, 0x3B9 ],
			0x1FAE  => [ 0x1F66, 0x3B9 ],
			0x1FAF  => [ 0x1F67, 0x3B9 ],
			0x1FB2  => [ 0x1F70, 0x3B9 ],
			0x1FB3  => [ 0x3B1, 0x3B9 ],
			0x1FB4  => [ 0x3AC, 0x3B9 ],
			0x1FB6  => [ 0x3B1, 0x342 ],
			0x1FB7  => [ 0x3B1, 0x342, 0x3B9 ],
			0x1FB8  => [ 0x1FB0 ],
			0x1FB9  => [ 0x1FB1 ],
			0x1FBA  => [ 0x1F70 ],
			0x1FBB  => [ 0x1F71 ],
			0x1FBC  => [ 0x3B1, 0x3B9 ],
			0x1FBE  => [ 0x3B9 ],
			0x1FC2  => [ 0x1F74, 0x3B9 ],
			0x1FC3  => [ 0x3B7, 0x3B9 ],
			0x1FC4  => [ 0x3AE, 0x3B9 ],
			0x1FC6  => [ 0x3B7, 0x342 ],
			0x1FC7  => [ 0x3B7, 0x342, 0x3B9 ],
			0x1FC8  => [ 0x1F72 ],
			0x1FC9  => [ 0x1F73 ],
			0x1FCA  => [ 0x1F74 ],
			0x1FCB  => [ 0x1F75 ],
			0x1FCC  => [ 0x3B7, 0x3B9 ],
			0x1FD2  => [ 0x3B9, 0x308, 0x300 ],
			0x1FD3  => [ 0x3B9, 0x308, 0x301 ],
			0x1FD6  => [ 0x3B9, 0x342 ],
			0x1FD7  => [ 0x3B9, 0x308, 0x342 ],
			0x1FD8  => [ 0x1FD0 ],
			0x1FD9  => [ 0x1FD1 ],
			0x1FDA  => [ 0x1F76 ],
			0x1FDB  => [ 0x1F77 ],
			0x1FE2  => [ 0x3C5, 0x308, 0x300 ],
			0x1FE3  => [ 0x3C5, 0x308, 0x301 ],
			0x1FE4  => [ 0x3C1, 0x313 ],
			0x1FE6  => [ 0x3C5, 0x342 ],
			0x1FE7  => [ 0x3C5, 0x308, 0x342 ],
			0x1FE8  => [ 0x1FE0 ],
			0x1FE9  => [ 0x1FE1 ],
			0x1FEA  => [ 0x1F7A ],
			0x1FEB  => [ 0x1F7B ],
			0x1FEC  => [ 0x1FE5 ],
			0x1FF2  => [ 0x1F7C, 0x3B9 ],
			0x1FF3  => [ 0x3C9, 0x3B9 ],
			0x1FF4  => [ 0x3CE, 0x3B9 ],
			0x1FF6  => [ 0x3C9, 0x342 ],
			0x1FF7  => [ 0x3C9, 0x342, 0x3B9 ],
			0x1FF8  => [ 0x1F78 ],
			0x1FF9  => [ 0x1F79 ],
			0x1FFA  => [ 0x1F7C ],
			0x1FFB  => [ 0x1F7D ],
			0x1FFC  => [ 0x3C9, 0x3B9 ],
			0x20A8  => [ 0x72, 0x73 ],
			0x2102  => [ 0x63 ],
			0x2103  => [ 0xB0, 0x63 ],
			0x2107  => [ 0x25B ],
			0x2109  => [ 0xB0, 0x66 ],
			0x210B  => [ 0x68 ],
			0x210C  => [ 0x68 ],
			0x210D  => [ 0x68 ],
			0x2110  => [ 0x69 ],
			0x2111  => [ 0x69 ],
			0x2112  => [ 0x6C ],
			0x2115  => [ 0x6E ],
			0x2116  => [ 0x6E, 0x6F ],
			0x2119  => [ 0x70 ],
			0x211A  => [ 0x71 ],
			0x211B  => [ 0x72 ],
			0x211C  => [ 0x72 ],
			0x211D  => [ 0x72 ],
			0x2120  => [ 0x73, 0x6D ],
			0x2121  => [ 0x74, 0x65, 0x6C ],
			0x2122  => [ 0x74, 0x6D ],
			0x2124  => [ 0x7A ],
			0x2126  => [ 0x3C9 ],
			0x2128  => [ 0x7A ],
			0x212A  => [ 0x6B ],
			0x212B  => [ 0xE5 ],
			0x212C  => [ 0x62 ],
			0x212D  => [ 0x63 ],
			0x2130  => [ 0x65 ],
			0x2131  => [ 0x66 ],
			0x2133  => [ 0x6D ],
			0x213E  => [ 0x3B3 ],
			0x213F  => [ 0x3C0 ],
			0x2145  => [ 0x64 ],
			0x2160  => [ 0x2170 ],
			0x2161  => [ 0x2171 ],
			0x2162  => [ 0x2172 ],
			0x2163  => [ 0x2173 ],
			0x2164  => [ 0x2174 ],
			0x2165  => [ 0x2175 ],
			0x2166  => [ 0x2176 ],
			0x2167  => [ 0x2177 ],
			0x2168  => [ 0x2178 ],
			0x2169  => [ 0x2179 ],
			0x216A  => [ 0x217A ],
			0x216B  => [ 0x217B ],
			0x216C  => [ 0x217C ],
			0x216D  => [ 0x217D ],
			0x216E  => [ 0x217E ],
			0x216F  => [ 0x217F ],
			0x24B6  => [ 0x24D0 ],
			0x24B7  => [ 0x24D1 ],
			0x24B8  => [ 0x24D2 ],
			0x24B9  => [ 0x24D3 ],
			0x24BA  => [ 0x24D4 ],
			0x24BB  => [ 0x24D5 ],
			0x24BC  => [ 0x24D6 ],
			0x24BD  => [ 0x24D7 ],
			0x24BE  => [ 0x24D8 ],
			0x24BF  => [ 0x24D9 ],
			0x24C0  => [ 0x24DA ],
			0x24C1  => [ 0x24DB ],
			0x24C2  => [ 0x24DC ],
			0x24C3  => [ 0x24DD ],
			0x24C4  => [ 0x24DE ],
			0x24C5  => [ 0x24DF ],
			0x24C6  => [ 0x24E0 ],
			0x24C7  => [ 0x24E1 ],
			0x24C8  => [ 0x24E2 ],
			0x24C9  => [ 0x24E3 ],
			0x24CA  => [ 0x24E4 ],
			0x24CB  => [ 0x24E5 ],
			0x24CC  => [ 0x24E6 ],
			0x24CD  => [ 0x24E7 ],
			0x24CE  => [ 0x24E8 ],
			0x24CF  => [ 0x24E9 ],
			0x3371  => [ 0x68, 0x70, 0x61 ],
			0x3373  => [ 0x61, 0x75 ],
			0x3375  => [ 0x6F, 0x76 ],
			0x3380  => [ 0x70, 0x61 ],
			0x3381  => [ 0x6E, 0x61 ],
			0x3382  => [ 0x3BC, 0x61 ],
			0x3383  => [ 0x6D, 0x61 ],
			0x3384  => [ 0x6B, 0x61 ],
			0x3385  => [ 0x6B, 0x62 ],
			0x3386  => [ 0x6D, 0x62 ],
			0x3387  => [ 0x67, 0x62 ],
			0x338A  => [ 0x70, 0x66 ],
			0x338B  => [ 0x6E, 0x66 ],
			0x338C  => [ 0x3BC, 0x66 ],
			0x3390  => [ 0x68, 0x7A ],
			0x3391  => [ 0x6B, 0x68, 0x7A ],
			0x3392  => [ 0x6D, 0x68, 0x7A ],
			0x3393  => [ 0x67, 0x68, 0x7A ],
			0x3394  => [ 0x74, 0x68, 0x7A ],
			0x33A9  => [ 0x70, 0x61 ],
			0x33AA  => [ 0x6B, 0x70, 0x61 ],
			0x33AB  => [ 0x6D, 0x70, 0x61 ],
			0x33AC  => [ 0x67, 0x70, 0x61 ],
			0x33B4  => [ 0x70, 0x76 ],
			0x33B5  => [ 0x6E, 0x76 ],
			0x33B6  => [ 0x3BC, 0x76 ],
			0x33B7  => [ 0x6D, 0x76 ],
			0x33B8  => [ 0x6B, 0x76 ],
			0x33B9  => [ 0x6D, 0x76 ],
			0x33BA  => [ 0x70, 0x77 ],
			0x33BB  => [ 0x6E, 0x77 ],
			0x33BC  => [ 0x3BC, 0x77 ],
			0x33BD  => [ 0x6D, 0x77 ],
			0x33BE  => [ 0x6B, 0x77 ],
			0x33BF  => [ 0x6D, 0x77 ],
			0x33C0  => [ 0x6B, 0x3C9 ],
			0x33C1  => [ 0x6D, 0x3C9 ], /*
                    0x33C2  => array(0x61, 0x2E, 0x6D, 0x2E), */
			0x33C3  => [ 0x62, 0x71 ],
			0x33C6  => [ 0x63, 0x2215, 0x6B, 0x67 ],
			0x33C7  => [ 0x63, 0x6F, 0x2E ],
			0x33C8  => [ 0x64, 0x62 ],
			0x33C9  => [ 0x67, 0x79 ],
			0x33CB  => [ 0x68, 0x70 ],
			0x33CD  => [ 0x6B, 0x6B ],
			0x33CE  => [ 0x6B, 0x6D ],
			0x33D7  => [ 0x70, 0x68 ],
			0x33D9  => [ 0x70, 0x70, 0x6D ],
			0x33DA  => [ 0x70, 0x72 ],
			0x33DC  => [ 0x73, 0x76 ],
			0x33DD  => [ 0x77, 0x62 ],
			0xFB00  => [ 0x66, 0x66 ],
			0xFB01  => [ 0x66, 0x69 ],
			0xFB02  => [ 0x66, 0x6C ],
			0xFB03  => [ 0x66, 0x66, 0x69 ],
			0xFB04  => [ 0x66, 0x66, 0x6C ],
			0xFB05  => [ 0x73, 0x74 ],
			0xFB06  => [ 0x73, 0x74 ],
			0xFB13  => [ 0x574, 0x576 ],
			0xFB14  => [ 0x574, 0x565 ],
			0xFB15  => [ 0x574, 0x56B ],
			0xFB16  => [ 0x57E, 0x576 ],
			0xFB17  => [ 0x574, 0x56D ],
			0xFF21  => [ 0xFF41 ],
			0xFF22  => [ 0xFF42 ],
			0xFF23  => [ 0xFF43 ],
			0xFF24  => [ 0xFF44 ],
			0xFF25  => [ 0xFF45 ],
			0xFF26  => [ 0xFF46 ],
			0xFF27  => [ 0xFF47 ],
			0xFF28  => [ 0xFF48 ],
			0xFF29  => [ 0xFF49 ],
			0xFF2A  => [ 0xFF4A ],
			0xFF2B  => [ 0xFF4B ],
			0xFF2C  => [ 0xFF4C ],
			0xFF2D  => [ 0xFF4D ],
			0xFF2E  => [ 0xFF4E ],
			0xFF2F  => [ 0xFF4F ],
			0xFF30  => [ 0xFF50 ],
			0xFF31  => [ 0xFF51 ],
			0xFF32  => [ 0xFF52 ],
			0xFF33  => [ 0xFF53 ],
			0xFF34  => [ 0xFF54 ],
			0xFF35  => [ 0xFF55 ],
			0xFF36  => [ 0xFF56 ],
			0xFF37  => [ 0xFF57 ],
			0xFF38  => [ 0xFF58 ],
			0xFF39  => [ 0xFF59 ],
			0xFF3A  => [ 0xFF5A ],
			0x10400 => [ 0x10428 ],
			0x10401 => [ 0x10429 ],
			0x10402 => [ 0x1042A ],
			0x10403 => [ 0x1042B ],
			0x10404 => [ 0x1042C ],
			0x10405 => [ 0x1042D ],
			0x10406 => [ 0x1042E ],
			0x10407 => [ 0x1042F ],
			0x10408 => [ 0x10430 ],
			0x10409 => [ 0x10431 ],
			0x1040A => [ 0x10432 ],
			0x1040B => [ 0x10433 ],
			0x1040C => [ 0x10434 ],
			0x1040D => [ 0x10435 ],
			0x1040E => [ 0x10436 ],
			0x1040F => [ 0x10437 ],
			0x10410 => [ 0x10438 ],
			0x10411 => [ 0x10439 ],
			0x10412 => [ 0x1043A ],
			0x10413 => [ 0x1043B ],
			0x10414 => [ 0x1043C ],
			0x10415 => [ 0x1043D ],
			0x10416 => [ 0x1043E ],
			0x10417 => [ 0x1043F ],
			0x10418 => [ 0x10440 ],
			0x10419 => [ 0x10441 ],
			0x1041A => [ 0x10442 ],
			0x1041B => [ 0x10443 ],
			0x1041C => [ 0x10444 ],
			0x1041D => [ 0x10445 ],
			0x1041E => [ 0x10446 ],
			0x1041F => [ 0x10447 ],
			0x10420 => [ 0x10448 ],
			0x10421 => [ 0x10449 ],
			0x10422 => [ 0x1044A ],
			0x10423 => [ 0x1044B ],
			0x10424 => [ 0x1044C ],
			0x10425 => [ 0x1044D ],
			0x1D400 => [ 0x61 ],
			0x1D401 => [ 0x62 ],
			0x1D402 => [ 0x63 ],
			0x1D403 => [ 0x64 ],
			0x1D404 => [ 0x65 ],
			0x1D405 => [ 0x66 ],
			0x1D406 => [ 0x67 ],
			0x1D407 => [ 0x68 ],
			0x1D408 => [ 0x69 ],
			0x1D409 => [ 0x6A ],
			0x1D40A => [ 0x6B ],
			0x1D40B => [ 0x6C ],
			0x1D40C => [ 0x6D ],
			0x1D40D => [ 0x6E ],
			0x1D40E => [ 0x6F ],
			0x1D40F => [ 0x70 ],
			0x1D410 => [ 0x71 ],
			0x1D411 => [ 0x72 ],
			0x1D412 => [ 0x73 ],
			0x1D413 => [ 0x74 ],
			0x1D414 => [ 0x75 ],
			0x1D415 => [ 0x76 ],
			0x1D416 => [ 0x77 ],
			0x1D417 => [ 0x78 ],
			0x1D418 => [ 0x79 ],
			0x1D419 => [ 0x7A ],
			0x1D434 => [ 0x61 ],
			0x1D435 => [ 0x62 ],
			0x1D436 => [ 0x63 ],
			0x1D437 => [ 0x64 ],
			0x1D438 => [ 0x65 ],
			0x1D439 => [ 0x66 ],
			0x1D43A => [ 0x67 ],
			0x1D43B => [ 0x68 ],
			0x1D43C => [ 0x69 ],
			0x1D43D => [ 0x6A ],
			0x1D43E => [ 0x6B ],
			0x1D43F => [ 0x6C ],
			0x1D440 => [ 0x6D ],
			0x1D441 => [ 0x6E ],
			0x1D442 => [ 0x6F ],
			0x1D443 => [ 0x70 ],
			0x1D444 => [ 0x71 ],
			0x1D445 => [ 0x72 ],
			0x1D446 => [ 0x73 ],
			0x1D447 => [ 0x74 ],
			0x1D448 => [ 0x75 ],
			0x1D449 => [ 0x76 ],
			0x1D44A => [ 0x77 ],
			0x1D44B => [ 0x78 ],
			0x1D44C => [ 0x79 ],
			0x1D44D => [ 0x7A ],
			0x1D468 => [ 0x61 ],
			0x1D469 => [ 0x62 ],
			0x1D46A => [ 0x63 ],
			0x1D46B => [ 0x64 ],
			0x1D46C => [ 0x65 ],
			0x1D46D => [ 0x66 ],
			0x1D46E => [ 0x67 ],
			0x1D46F => [ 0x68 ],
			0x1D470 => [ 0x69 ],
			0x1D471 => [ 0x6A ],
			0x1D472 => [ 0x6B ],
			0x1D473 => [ 0x6C ],
			0x1D474 => [ 0x6D ],
			0x1D475 => [ 0x6E ],
			0x1D476 => [ 0x6F ],
			0x1D477 => [ 0x70 ],
			0x1D478 => [ 0x71 ],
			0x1D479 => [ 0x72 ],
			0x1D47A => [ 0x73 ],
			0x1D47B => [ 0x74 ],
			0x1D47C => [ 0x75 ],
			0x1D47D => [ 0x76 ],
			0x1D47E => [ 0x77 ],
			0x1D47F => [ 0x78 ],
			0x1D480 => [ 0x79 ],
			0x1D481 => [ 0x7A ],
			0x1D49C => [ 0x61 ],
			0x1D49E => [ 0x63 ],
			0x1D49F => [ 0x64 ],
			0x1D4A2 => [ 0x67 ],
			0x1D4A5 => [ 0x6A ],
			0x1D4A6 => [ 0x6B ],
			0x1D4A9 => [ 0x6E ],
			0x1D4AA => [ 0x6F ],
			0x1D4AB => [ 0x70 ],
			0x1D4AC => [ 0x71 ],
			0x1D4AE => [ 0x73 ],
			0x1D4AF => [ 0x74 ],
			0x1D4B0 => [ 0x75 ],
			0x1D4B1 => [ 0x76 ],
			0x1D4B2 => [ 0x77 ],
			0x1D4B3 => [ 0x78 ],
			0x1D4B4 => [ 0x79 ],
			0x1D4B5 => [ 0x7A ],
			0x1D4D0 => [ 0x61 ],
			0x1D4D1 => [ 0x62 ],
			0x1D4D2 => [ 0x63 ],
			0x1D4D3 => [ 0x64 ],
			0x1D4D4 => [ 0x65 ],
			0x1D4D5 => [ 0x66 ],
			0x1D4D6 => [ 0x67 ],
			0x1D4D7 => [ 0x68 ],
			0x1D4D8 => [ 0x69 ],
			0x1D4D9 => [ 0x6A ],
			0x1D4DA => [ 0x6B ],
			0x1D4DB => [ 0x6C ],
			0x1D4DC => [ 0x6D ],
			0x1D4DD => [ 0x6E ],
			0x1D4DE => [ 0x6F ],
			0x1D4DF => [ 0x70 ],
			0x1D4E0 => [ 0x71 ],
			0x1D4E1 => [ 0x72 ],
			0x1D4E2 => [ 0x73 ],
			0x1D4E3 => [ 0x74 ],
			0x1D4E4 => [ 0x75 ],
			0x1D4E5 => [ 0x76 ],
			0x1D4E6 => [ 0x77 ],
			0x1D4E7 => [ 0x78 ],
			0x1D4E8 => [ 0x79 ],
			0x1D4E9 => [ 0x7A ],
			0x1D504 => [ 0x61 ],
			0x1D505 => [ 0x62 ],
			0x1D507 => [ 0x64 ],
			0x1D508 => [ 0x65 ],
			0x1D509 => [ 0x66 ],
			0x1D50A => [ 0x67 ],
			0x1D50D => [ 0x6A ],
			0x1D50E => [ 0x6B ],
			0x1D50F => [ 0x6C ],
			0x1D510 => [ 0x6D ],
			0x1D511 => [ 0x6E ],
			0x1D512 => [ 0x6F ],
			0x1D513 => [ 0x70 ],
			0x1D514 => [ 0x71 ],
			0x1D516 => [ 0x73 ],
			0x1D517 => [ 0x74 ],
			0x1D518 => [ 0x75 ],
			0x1D519 => [ 0x76 ],
			0x1D51A => [ 0x77 ],
			0x1D51B => [ 0x78 ],
			0x1D51C => [ 0x79 ],
			0x1D538 => [ 0x61 ],
			0x1D539 => [ 0x62 ],
			0x1D53B => [ 0x64 ],
			0x1D53C => [ 0x65 ],
			0x1D53D => [ 0x66 ],
			0x1D53E => [ 0x67 ],
			0x1D540 => [ 0x69 ],
			0x1D541 => [ 0x6A ],
			0x1D542 => [ 0x6B ],
			0x1D543 => [ 0x6C ],
			0x1D544 => [ 0x6D ],
			0x1D546 => [ 0x6F ],
			0x1D54A => [ 0x73 ],
			0x1D54B => [ 0x74 ],
			0x1D54C => [ 0x75 ],
			0x1D54D => [ 0x76 ],
			0x1D54E => [ 0x77 ],
			0x1D54F => [ 0x78 ],
			0x1D550 => [ 0x79 ],
			0x1D56C => [ 0x61 ],
			0x1D56D => [ 0x62 ],
			0x1D56E => [ 0x63 ],
			0x1D56F => [ 0x64 ],
			0x1D570 => [ 0x65 ],
			0x1D571 => [ 0x66 ],
			0x1D572 => [ 0x67 ],
			0x1D573 => [ 0x68 ],
			0x1D574 => [ 0x69 ],
			0x1D575 => [ 0x6A ],
			0x1D576 => [ 0x6B ],
			0x1D577 => [ 0x6C ],
			0x1D578 => [ 0x6D ],
			0x1D579 => [ 0x6E ],
			0x1D57A => [ 0x6F ],
			0x1D57B => [ 0x70 ],
			0x1D57C => [ 0x71 ],
			0x1D57D => [ 0x72 ],
			0x1D57E => [ 0x73 ],
			0x1D57F => [ 0x74 ],
			0x1D580 => [ 0x75 ],
			0x1D581 => [ 0x76 ],
			0x1D582 => [ 0x77 ],
			0x1D583 => [ 0x78 ],
			0x1D584 => [ 0x79 ],
			0x1D585 => [ 0x7A ],
			0x1D5A0 => [ 0x61 ],
			0x1D5A1 => [ 0x62 ],
			0x1D5A2 => [ 0x63 ],
			0x1D5A3 => [ 0x64 ],
			0x1D5A4 => [ 0x65 ],
			0x1D5A5 => [ 0x66 ],
			0x1D5A6 => [ 0x67 ],
			0x1D5A7 => [ 0x68 ],
			0x1D5A8 => [ 0x69 ],
			0x1D5A9 => [ 0x6A ],
			0x1D5AA => [ 0x6B ],
			0x1D5AB => [ 0x6C ],
			0x1D5AC => [ 0x6D ],
			0x1D5AD => [ 0x6E ],
			0x1D5AE => [ 0x6F ],
			0x1D5AF => [ 0x70 ],
			0x1D5B0 => [ 0x71 ],
			0x1D5B1 => [ 0x72 ],
			0x1D5B2 => [ 0x73 ],
			0x1D5B3 => [ 0x74 ],
			0x1D5B4 => [ 0x75 ],
			0x1D5B5 => [ 0x76 ],
			0x1D5B6 => [ 0x77 ],
			0x1D5B7 => [ 0x78 ],
			0x1D5B8 => [ 0x79 ],
			0x1D5B9 => [ 0x7A ],
			0x1D5D4 => [ 0x61 ],
			0x1D5D5 => [ 0x62 ],
			0x1D5D6 => [ 0x63 ],
			0x1D5D7 => [ 0x64 ],
			0x1D5D8 => [ 0x65 ],
			0x1D5D9 => [ 0x66 ],
			0x1D5DA => [ 0x67 ],
			0x1D5DB => [ 0x68 ],
			0x1D5DC => [ 0x69 ],
			0x1D5DD => [ 0x6A ],
			0x1D5DE => [ 0x6B ],
			0x1D5DF => [ 0x6C ],
			0x1D5E0 => [ 0x6D ],
			0x1D5E1 => [ 0x6E ],
			0x1D5E2 => [ 0x6F ],
			0x1D5E3 => [ 0x70 ],
			0x1D5E4 => [ 0x71 ],
			0x1D5E5 => [ 0x72 ],
			0x1D5E6 => [ 0x73 ],
			0x1D5E7 => [ 0x74 ],
			0x1D5E8 => [ 0x75 ],
			0x1D5E9 => [ 0x76 ],
			0x1D5EA => [ 0x77 ],
			0x1D5EB => [ 0x78 ],
			0x1D5EC => [ 0x79 ],
			0x1D5ED => [ 0x7A ],
			0x1D608 => [ 0x61 ],
			0x1D609 => [ 0x62 ],
			0x1D60A => [ 0x63 ],
			0x1D60B => [ 0x64 ],
			0x1D60C => [ 0x65 ],
			0x1D60D => [ 0x66 ],
			0x1D60E => [ 0x67 ],
			0x1D60F => [ 0x68 ],
			0x1D610 => [ 0x69 ],
			0x1D611 => [ 0x6A ],
			0x1D612 => [ 0x6B ],
			0x1D613 => [ 0x6C ],
			0x1D614 => [ 0x6D ],
			0x1D615 => [ 0x6E ],
			0x1D616 => [ 0x6F ],
			0x1D617 => [ 0x70 ],
			0x1D618 => [ 0x71 ],
			0x1D619 => [ 0x72 ],
			0x1D61A => [ 0x73 ],
			0x1D61B => [ 0x74 ],
			0x1D61C => [ 0x75 ],
			0x1D61D => [ 0x76 ],
			0x1D61E => [ 0x77 ],
			0x1D61F => [ 0x78 ],
			0x1D620 => [ 0x79 ],
			0x1D621 => [ 0x7A ],
			0x1D63C => [ 0x61 ],
			0x1D63D => [ 0x62 ],
			0x1D63E => [ 0x63 ],
			0x1D63F => [ 0x64 ],
			0x1D640 => [ 0x65 ],
			0x1D641 => [ 0x66 ],
			0x1D642 => [ 0x67 ],
			0x1D643 => [ 0x68 ],
			0x1D644 => [ 0x69 ],
			0x1D645 => [ 0x6A ],
			0x1D646 => [ 0x6B ],
			0x1D647 => [ 0x6C ],
			0x1D648 => [ 0x6D ],
			0x1D649 => [ 0x6E ],
			0x1D64A => [ 0x6F ],
			0x1D64B => [ 0x70 ],
			0x1D64C => [ 0x71 ],
			0x1D64D => [ 0x72 ],
			0x1D64E => [ 0x73 ],
			0x1D64F => [ 0x74 ],
			0x1D650 => [ 0x75 ],
			0x1D651 => [ 0x76 ],
			0x1D652 => [ 0x77 ],
			0x1D653 => [ 0x78 ],
			0x1D654 => [ 0x79 ],
			0x1D655 => [ 0x7A ],
			0x1D670 => [ 0x61 ],
			0x1D671 => [ 0x62 ],
			0x1D672 => [ 0x63 ],
			0x1D673 => [ 0x64 ],
			0x1D674 => [ 0x65 ],
			0x1D675 => [ 0x66 ],
			0x1D676 => [ 0x67 ],
			0x1D677 => [ 0x68 ],
			0x1D678 => [ 0x69 ],
			0x1D679 => [ 0x6A ],
			0x1D67A => [ 0x6B ],
			0x1D67B => [ 0x6C ],
			0x1D67C => [ 0x6D ],
			0x1D67D => [ 0x6E ],
			0x1D67E => [ 0x6F ],
			0x1D67F => [ 0x70 ],
			0x1D680 => [ 0x71 ],
			0x1D681 => [ 0x72 ],
			0x1D682 => [ 0x73 ],
			0x1D683 => [ 0x74 ],
			0x1D684 => [ 0x75 ],
			0x1D685 => [ 0x76 ],
			0x1D686 => [ 0x77 ],
			0x1D687 => [ 0x78 ],
			0x1D688 => [ 0x79 ],
			0x1D689 => [ 0x7A ],
			0x1D6A8 => [ 0x3B1 ],
			0x1D6A9 => [ 0x3B2 ],
			0x1D6AA => [ 0x3B3 ],
			0x1D6AB => [ 0x3B4 ],
			0x1D6AC => [ 0x3B5 ],
			0x1D6AD => [ 0x3B6 ],
			0x1D6AE => [ 0x3B7 ],
			0x1D6AF => [ 0x3B8 ],
			0x1D6B0 => [ 0x3B9 ],
			0x1D6B1 => [ 0x3BA ],
			0x1D6B2 => [ 0x3BB ],
			0x1D6B3 => [ 0x3BC ],
			0x1D6B4 => [ 0x3BD ],
			0x1D6B5 => [ 0x3BE ],
			0x1D6B6 => [ 0x3BF ],
			0x1D6B7 => [ 0x3C0 ],
			0x1D6B8 => [ 0x3C1 ],
			0x1D6B9 => [ 0x3B8 ],
			0x1D6BA => [ 0x3C3 ],
			0x1D6BB => [ 0x3C4 ],
			0x1D6BC => [ 0x3C5 ],
			0x1D6BD => [ 0x3C6 ],
			0x1D6BE => [ 0x3C7 ],
			0x1D6BF => [ 0x3C8 ],
			0x1D6C0 => [ 0x3C9 ],
			0x1D6D3 => [ 0x3C3 ],
			0x1D6E2 => [ 0x3B1 ],
			0x1D6E3 => [ 0x3B2 ],
			0x1D6E4 => [ 0x3B3 ],
			0x1D6E5 => [ 0x3B4 ],
			0x1D6E6 => [ 0x3B5 ],
			0x1D6E7 => [ 0x3B6 ],
			0x1D6E8 => [ 0x3B7 ],
			0x1D6E9 => [ 0x3B8 ],
			0x1D6EA => [ 0x3B9 ],
			0x1D6EB => [ 0x3BA ],
			0x1D6EC => [ 0x3BB ],
			0x1D6ED => [ 0x3BC ],
			0x1D6EE => [ 0x3BD ],
			0x1D6EF => [ 0x3BE ],
			0x1D6F0 => [ 0x3BF ],
			0x1D6F1 => [ 0x3C0 ],
			0x1D6F2 => [ 0x3C1 ],
			0x1D6F3 => [ 0x3B8 ],
			0x1D6F4 => [ 0x3C3 ],
			0x1D6F5 => [ 0x3C4 ],
			0x1D6F6 => [ 0x3C5 ],
			0x1D6F7 => [ 0x3C6 ],
			0x1D6F8 => [ 0x3C7 ],
			0x1D6F9 => [ 0x3C8 ],
			0x1D6FA => [ 0x3C9 ],
			0x1D70D => [ 0x3C3 ],
			0x1D71C => [ 0x3B1 ],
			0x1D71D => [ 0x3B2 ],
			0x1D71E => [ 0x3B3 ],
			0x1D71F => [ 0x3B4 ],
			0x1D720 => [ 0x3B5 ],
			0x1D721 => [ 0x3B6 ],
			0x1D722 => [ 0x3B7 ],
			0x1D723 => [ 0x3B8 ],
			0x1D724 => [ 0x3B9 ],
			0x1D725 => [ 0x3BA ],
			0x1D726 => [ 0x3BB ],
			0x1D727 => [ 0x3BC ],
			0x1D728 => [ 0x3BD ],
			0x1D729 => [ 0x3BE ],
			0x1D72A => [ 0x3BF ],
			0x1D72B => [ 0x3C0 ],
			0x1D72C => [ 0x3C1 ],
			0x1D72D => [ 0x3B8 ],
			0x1D72E => [ 0x3C3 ],
			0x1D72F => [ 0x3C4 ],
			0x1D730 => [ 0x3C5 ],
			0x1D731 => [ 0x3C6 ],
			0x1D732 => [ 0x3C7 ],
			0x1D733 => [ 0x3C8 ],
			0x1D734 => [ 0x3C9 ],
			0x1D747 => [ 0x3C3 ],
			0x1D756 => [ 0x3B1 ],
			0x1D757 => [ 0x3B2 ],
			0x1D758 => [ 0x3B3 ],
			0x1D759 => [ 0x3B4 ],
			0x1D75A => [ 0x3B5 ],
			0x1D75B => [ 0x3B6 ],
			0x1D75C => [ 0x3B7 ],
			0x1D75D => [ 0x3B8 ],
			0x1D75E => [ 0x3B9 ],
			0x1D75F => [ 0x3BA ],
			0x1D760 => [ 0x3BB ],
			0x1D761 => [ 0x3BC ],
			0x1D762 => [ 0x3BD ],
			0x1D763 => [ 0x3BE ],
			0x1D764 => [ 0x3BF ],
			0x1D765 => [ 0x3C0 ],
			0x1D766 => [ 0x3C1 ],
			0x1D767 => [ 0x3B8 ],
			0x1D768 => [ 0x3C3 ],
			0x1D769 => [ 0x3C4 ],
			0x1D76A => [ 0x3C5 ],
			0x1D76B => [ 0x3C6 ],
			0x1D76C => [ 0x3C7 ],
			0x1D76D => [ 0x3C8 ],
			0x1D76E => [ 0x3C9 ],
			0x1D781 => [ 0x3C3 ],
			0x1D790 => [ 0x3B1 ],
			0x1D791 => [ 0x3B2 ],
			0x1D792 => [ 0x3B3 ],
			0x1D793 => [ 0x3B4 ],
			0x1D794 => [ 0x3B5 ],
			0x1D795 => [ 0x3B6 ],
			0x1D796 => [ 0x3B7 ],
			0x1D797 => [ 0x3B8 ],
			0x1D798 => [ 0x3B9 ],
			0x1D799 => [ 0x3BA ],
			0x1D79A => [ 0x3BB ],
			0x1D79B => [ 0x3BC ],
			0x1D79C => [ 0x3BD ],
			0x1D79D => [ 0x3BE ],
			0x1D79E => [ 0x3BF ],
			0x1D79F => [ 0x3C0 ],
			0x1D7A0 => [ 0x3C1 ],
			0x1D7A1 => [ 0x3B8 ],
			0x1D7A2 => [ 0x3C3 ],
			0x1D7A3 => [ 0x3C4 ],
			0x1D7A4 => [ 0x3C5 ],
			0x1D7A5 => [ 0x3C6 ],
			0x1D7A6 => [ 0x3C7 ],
			0x1D7A7 => [ 0x3C8 ],
			0x1D7A8 => [ 0x3C9 ],
			0x1D7BB => [ 0x3C3 ],
			0x3F9   => [ 0x3C3 ],
			0x1D2C  => [ 0x61 ],
			0x1D2D  => [ 0xE6 ],
			0x1D2E  => [ 0x62 ],
			0x1D30  => [ 0x64 ],
			0x1D31  => [ 0x65 ],
			0x1D32  => [ 0x1DD ],
			0x1D33  => [ 0x67 ],
			0x1D34  => [ 0x68 ],
			0x1D35  => [ 0x69 ],
			0x1D36  => [ 0x6A ],
			0x1D37  => [ 0x6B ],
			0x1D38  => [ 0x6C ],
			0x1D39  => [ 0x6D ],
			0x1D3A  => [ 0x6E ],
			0x1D3C  => [ 0x6F ],
			0x1D3D  => [ 0x223 ],
			0x1D3E  => [ 0x70 ],
			0x1D3F  => [ 0x72 ],
			0x1D40  => [ 0x74 ],
			0x1D41  => [ 0x75 ],
			0x1D42  => [ 0x77 ],
			0x213B  => [ 0x66, 0x61, 0x78 ],
			0x3250  => [ 0x70, 0x74, 0x65 ],
			0x32CC  => [ 0x68, 0x67 ],
			0x32CE  => [ 0x65, 0x76 ],
			0x32CF  => [ 0x6C, 0x74, 0x64 ],
			0x337A  => [ 0x69, 0x75 ],
			0x33DE  => [ 0x76, 0x2215, 0x6D ],
			0x33DF  => [ 0x61, 0x2215, 0x6D ]
		],
		'replacemaps'        => [
			0x41    => [ 0x61 ],
			0x42    => [ 0x62 ],
			0x43    => [ 0x63 ],
			0x44    => [ 0x64 ],
			0x45    => [ 0x65 ],
			0x46    => [ 0x66 ],
			0x47    => [ 0x67 ],
			0x48    => [ 0x68 ],
			0x49    => [ 0x69 ],
			0x4A    => [ 0x6A ],
			0x4B    => [ 0x6B ],
			0x4C    => [ 0x6C ],
			0x4D    => [ 0x6D ],
			0x4E    => [ 0x6E ],
			0x4F    => [ 0x6F ],
			0x50    => [ 0x70 ],
			0x51    => [ 0x71 ],
			0x52    => [ 0x72 ],
			0x53    => [ 0x73 ],
			0x54    => [ 0x74 ],
			0x55    => [ 0x75 ],
			0x56    => [ 0x76 ],
			0x57    => [ 0x77 ],
			0x58    => [ 0x78 ],
			0x59    => [ 0x79 ],
			0x5A    => [ 0x7A ],
			0xAA    => [ 0x61 ],
			0xB2    => [ 0x32 ],
			0xB3    => [ 0x33 ],
			0xB5    => [ 0x3BC ],
			0xB9    => [ 0x31 ],
			0xBA    => [ 0x6F ],
			0xBC    => [ 0x31, 0x2044, 0x34 ],
			0xBD    => [ 0x31, 0x2044, 0x32 ],
			0xBE    => [ 0x33, 0x2044, 0x34 ],
			0xC0    => [ 0xE0 ],
			0xC1    => [ 0xE1 ],
			0xC2    => [ 0xE2 ],
			0xC3    => [ 0xE3 ],
			0xC4    => [ 0xE4 ],
			0xC5    => [ 0xE5 ],
			0xC6    => [ 0xE6 ],
			0xC7    => [ 0xE7 ],
			0xC8    => [ 0xE8 ],
			0xC9    => [ 0xE9 ],
			0xCA    => [ 0xEA ],
			0xCB    => [ 0xEB ],
			0xCC    => [ 0xEC ],
			0xCD    => [ 0xED ],
			0xCE    => [ 0xEE ],
			0xCF    => [ 0xEF ],
			0xD0    => [ 0xF0 ],
			0xD1    => [ 0xF1 ],
			0xD2    => [ 0xF2 ],
			0xD3    => [ 0xF3 ],
			0xD4    => [ 0xF4 ],
			0xD5    => [ 0xF5 ],
			0xD6    => [ 0xF6 ],
			0xD8    => [ 0xF8 ],
			0xD9    => [ 0xF9 ],
			0xDA    => [ 0xFA ],
			0xDB    => [ 0xFB ],
			0xDC    => [ 0xFC ],
			0xDD    => [ 0xFD ],
			0xDE    => [ 0xFE ],
			0x100   => [ 0x101 ],
			0x102   => [ 0x103 ],
			0x104   => [ 0x105 ],
			0x106   => [ 0x107 ],
			0x108   => [ 0x109 ],
			0x10A   => [ 0x10B ],
			0x10C   => [ 0x10D ],
			0x10E   => [ 0x10F ],
			0x110   => [ 0x111 ],
			0x112   => [ 0x113 ],
			0x114   => [ 0x115 ],
			0x116   => [ 0x117 ],
			0x118   => [ 0x119 ],
			0x11A   => [ 0x11B ],
			0x11C   => [ 0x11D ],
			0x11E   => [ 0x11F ],
			0x120   => [ 0x121 ],
			0x122   => [ 0x123 ],
			0x124   => [ 0x125 ],
			0x126   => [ 0x127 ],
			0x128   => [ 0x129 ],
			0x12A   => [ 0x12B ],
			0x12C   => [ 0x12D ],
			0x12E   => [ 0x12F ],
			0x130   => [ 0x69, 0x307 ],
			0x132   => [ 0x69, 0x6A ],
			0x133   => [ 0x69, 0x6A ],
			0x134   => [ 0x135 ],
			0x136   => [ 0x137 ],
			0x139   => [ 0x13A ],
			0x13B   => [ 0x13C ],
			0x13D   => [ 0x13E ],
			0x13F   => [ 0x6C, 0xB7 ],
			0x140   => [ 0x6C, 0xB7 ],
			0x141   => [ 0x142 ],
			0x143   => [ 0x144 ],
			0x145   => [ 0x146 ],
			0x147   => [ 0x148 ],
			0x149   => [ 0x2BC, 0x6E ],
			0x14A   => [ 0x14B ],
			0x14C   => [ 0x14D ],
			0x14E   => [ 0x14F ],
			0x150   => [ 0x151 ],
			0x152   => [ 0x153 ],
			0x154   => [ 0x155 ],
			0x156   => [ 0x157 ],
			0x158   => [ 0x159 ],
			0x15A   => [ 0x15B ],
			0x15C   => [ 0x15D ],
			0x15E   => [ 0x15F ],
			0x160   => [ 0x161 ],
			0x162   => [ 0x163 ],
			0x164   => [ 0x165 ],
			0x166   => [ 0x167 ],
			0x168   => [ 0x169 ],
			0x16A   => [ 0x16B ],
			0x16C   => [ 0x16D ],
			0x16E   => [ 0x16F ],
			0x170   => [ 0x171 ],
			0x172   => [ 0x173 ],
			0x174   => [ 0x175 ],
			0x176   => [ 0x177 ],
			0x178   => [ 0xFF ],
			0x179   => [ 0x17A ],
			0x17B   => [ 0x17C ],
			0x17D   => [ 0x17E ],
			0x17F   => [ 0x73 ],
			0x181   => [ 0x253 ],
			0x182   => [ 0x183 ],
			0x184   => [ 0x185 ],
			0x186   => [ 0x254 ],
			0x187   => [ 0x188 ],
			0x189   => [ 0x256 ],
			0x18A   => [ 0x257 ],
			0x18B   => [ 0x18C ],
			0x18E   => [ 0x1DD ],
			0x18F   => [ 0x259 ],
			0x190   => [ 0x25B ],
			0x191   => [ 0x192 ],
			0x193   => [ 0x260 ],
			0x194   => [ 0x263 ],
			0x196   => [ 0x269 ],
			0x197   => [ 0x268 ],
			0x198   => [ 0x199 ],
			0x19C   => [ 0x26F ],
			0x19D   => [ 0x272 ],
			0x19F   => [ 0x275 ],
			0x1A0   => [ 0x1A1 ],
			0x1A2   => [ 0x1A3 ],
			0x1A4   => [ 0x1A5 ],
			0x1A6   => [ 0x280 ],
			0x1A7   => [ 0x1A8 ],
			0x1A9   => [ 0x283 ],
			0x1AC   => [ 0x1AD ],
			0x1AE   => [ 0x288 ],
			0x1AF   => [ 0x1B0 ],
			0x1B1   => [ 0x28A ],
			0x1B2   => [ 0x28B ],
			0x1B3   => [ 0x1B4 ],
			0x1B5   => [ 0x1B6 ],
			0x1B7   => [ 0x292 ],
			0x1B8   => [ 0x1B9 ],
			0x1BC   => [ 0x1BD ],
			0x1C4   => [ 0x64, 0x17E ],
			0x1C5   => [ 0x64, 0x17E ],
			0x1C6   => [ 0x64, 0x17E ],
			0x1C7   => [ 0x6C, 0x6A ],
			0x1C8   => [ 0x6C, 0x6A ],
			0x1C9   => [ 0x6C, 0x6A ],
			0x1CA   => [ 0x6E, 0x6A ],
			0x1CB   => [ 0x6E, 0x6A ],
			0x1CC   => [ 0x6E, 0x6A ],
			0x1CD   => [ 0x1CE ],
			0x1CF   => [ 0x1D0 ],
			0x1D1   => [ 0x1D2 ],
			0x1D3   => [ 0x1D4 ],
			0x1D5   => [ 0x1D6 ],
			0x1D7   => [ 0x1D8 ],
			0x1D9   => [ 0x1DA ],
			0x1DB   => [ 0x1DC ],
			0x1DE   => [ 0x1DF ],
			0x1E0   => [ 0x1E1 ],
			0x1E2   => [ 0x1E3 ],
			0x1E4   => [ 0x1E5 ],
			0x1E6   => [ 0x1E7 ],
			0x1E8   => [ 0x1E9 ],
			0x1EA   => [ 0x1EB ],
			0x1EC   => [ 0x1ED ],
			0x1EE   => [ 0x1EF ],
			0x1F1   => [ 0x64, 0x7A ],
			0x1F2   => [ 0x64, 0x7A ],
			0x1F3   => [ 0x64, 0x7A ],
			0x1F4   => [ 0x1F5 ],
			0x1F6   => [ 0x195 ],
			0x1F7   => [ 0x1BF ],
			0x1F8   => [ 0x1F9 ],
			0x1FA   => [ 0x1FB ],
			0x1FC   => [ 0x1FD ],
			0x1FE   => [ 0x1FF ],
			0x200   => [ 0x201 ],
			0x202   => [ 0x203 ],
			0x204   => [ 0x205 ],
			0x206   => [ 0x207 ],
			0x208   => [ 0x209 ],
			0x20A   => [ 0x20B ],
			0x20C   => [ 0x20D ],
			0x20E   => [ 0x20F ],
			0x210   => [ 0x211 ],
			0x212   => [ 0x213 ],
			0x214   => [ 0x215 ],
			0x216   => [ 0x217 ],
			0x218   => [ 0x219 ],
			0x21A   => [ 0x21B ],
			0x21C   => [ 0x21D ],
			0x21E   => [ 0x21F ],
			0x220   => [ 0x19E ],
			0x222   => [ 0x223 ],
			0x224   => [ 0x225 ],
			0x226   => [ 0x227 ],
			0x228   => [ 0x229 ],
			0x22A   => [ 0x22B ],
			0x22C   => [ 0x22D ],
			0x22E   => [ 0x22F ],
			0x230   => [ 0x231 ],
			0x232   => [ 0x233 ],
			0x23A   => [ 0x2C65 ],
			0x23B   => [ 0x23C ],
			0x23D   => [ 0x19A ],
			0x23E   => [ 0x2C66 ],
			0x241   => [ 0x242 ],
			0x243   => [ 0x180 ],
			0x244   => [ 0x289 ],
			0x245   => [ 0x28C ],
			0x246   => [ 0x247 ],
			0x248   => [ 0x249 ],
			0x24A   => [ 0x24B ],
			0x24C   => [ 0x24D ],
			0x24E   => [ 0x24F ],
			0x2B0   => [ 0x68 ],
			0x2B1   => [ 0x266 ],
			0x2B2   => [ 0x6A ],
			0x2B3   => [ 0x72 ],
			0x2B4   => [ 0x279 ],
			0x2B5   => [ 0x27B ],
			0x2B6   => [ 0x281 ],
			0x2B7   => [ 0x77 ],
			0x2B8   => [ 0x79 ],
			0x2E0   => [ 0x263 ],
			0x2E1   => [ 0x6C ],
			0x2E2   => [ 0x73 ],
			0x2E3   => [ 0x78 ],
			0x2E4   => [ 0x295 ],
			0x340   => [ 0x300 ],
			0x341   => [ 0x301 ],
			0x343   => [ 0x313 ],
			0x344   => [ 0x308, 0x301 ],
			0x345   => [ 0x3B9 ],
			0x370   => [ 0x371 ],
			0x372   => [ 0x373 ],
			0x374   => [ 0x2B9 ],
			0x376   => [ 0x377 ],
			0x37F   => [ 0x3F3 ],
			0x386   => [ 0x3AC ],
			0x387   => [ 0xB7 ],
			0x388   => [ 0x3AD ],
			0x389   => [ 0x3AE ],
			0x38A   => [ 0x3AF ],
			0x38C   => [ 0x3CC ],
			0x38E   => [ 0x3CD ],
			0x38F   => [ 0x3CE ],
			0x391   => [ 0x3B1 ],
			0x392   => [ 0x3B2 ],
			0x393   => [ 0x3B3 ],
			0x394   => [ 0x3B4 ],
			0x395   => [ 0x3B5 ],
			0x396   => [ 0x3B6 ],
			0x397   => [ 0x3B7 ],
			0x398   => [ 0x3B8 ],
			0x399   => [ 0x3B9 ],
			0x39A   => [ 0x3BA ],
			0x39B   => [ 0x3BB ],
			0x39C   => [ 0x3BC ],
			0x39D   => [ 0x3BD ],
			0x39E   => [ 0x3BE ],
			0x39F   => [ 0x3BF ],
			0x3A0   => [ 0x3C0 ],
			0x3A1   => [ 0x3C1 ],
			0x3A3   => [ 0x3C3 ],
			0x3A4   => [ 0x3C4 ],
			0x3A5   => [ 0x3C5 ],
			0x3A6   => [ 0x3C6 ],
			0x3A7   => [ 0x3C7 ],
			0x3A8   => [ 0x3C8 ],
			0x3A9   => [ 0x3C9 ],
			0x3AA   => [ 0x3CA ],
			0x3AB   => [ 0x3CB ],
			0x3CF   => [ 0x3D7 ],
			0x3D0   => [ 0x3B2 ],
			0x3D1   => [ 0x3B8 ],
			0x3D2   => [ 0x3C5 ],
			0x3D3   => [ 0x3CD ],
			0x3D4   => [ 0x3CB ],
			0x3D5   => [ 0x3C6 ],
			0x3D6   => [ 0x3C0 ],
			0x3D8   => [ 0x3D9 ],
			0x3DA   => [ 0x3DB ],
			0x3DC   => [ 0x3DD ],
			0x3DE   => [ 0x3DF ],
			0x3E0   => [ 0x3E1 ],
			0x3E2   => [ 0x3E3 ],
			0x3E4   => [ 0x3E5 ],
			0x3E6   => [ 0x3E7 ],
			0x3E8   => [ 0x3E9 ],
			0x3EA   => [ 0x3EB ],
			0x3EC   => [ 0x3ED ],
			0x3EE   => [ 0x3EF ],
			0x3F0   => [ 0x3BA ],
			0x3F1   => [ 0x3C1 ],
			0x3F2   => [ 0x3C3 ],
			0x3F4   => [ 0x3B8 ],
			0x3F5   => [ 0x3B5 ],
			0x3F7   => [ 0x3F8 ],
			0x3F9   => [ 0x3C3 ],
			0x3FA   => [ 0x3FB ],
			0x3FD   => [ 0x37B ],
			0x3FE   => [ 0x37C ],
			0x3FF   => [ 0x37D ],
			0x400   => [ 0x450 ],
			0x401   => [ 0x451 ],
			0x402   => [ 0x452 ],
			0x403   => [ 0x453 ],
			0x404   => [ 0x454 ],
			0x405   => [ 0x455 ],
			0x406   => [ 0x456 ],
			0x407   => [ 0x457 ],
			0x408   => [ 0x458 ],
			0x409   => [ 0x459 ],
			0x40A   => [ 0x45A ],
			0x40B   => [ 0x45B ],
			0x40C   => [ 0x45C ],
			0x40D   => [ 0x45D ],
			0x40E   => [ 0x45E ],
			0x40F   => [ 0x45F ],
			0x410   => [ 0x430 ],
			0x411   => [ 0x431 ],
			0x412   => [ 0x432 ],
			0x413   => [ 0x433 ],
			0x414   => [ 0x434 ],
			0x415   => [ 0x435 ],
			0x416   => [ 0x436 ],
			0x417   => [ 0x437 ],
			0x418   => [ 0x438 ],
			0x419   => [ 0x439 ],
			0x41A   => [ 0x43A ],
			0x41B   => [ 0x43B ],
			0x41C   => [ 0x43C ],
			0x41D   => [ 0x43D ],
			0x41E   => [ 0x43E ],
			0x41F   => [ 0x43F ],
			0x420   => [ 0x440 ],
			0x421   => [ 0x441 ],
			0x422   => [ 0x442 ],
			0x423   => [ 0x443 ],
			0x424   => [ 0x444 ],
			0x425   => [ 0x445 ],
			0x426   => [ 0x446 ],
			0x427   => [ 0x447 ],
			0x428   => [ 0x448 ],
			0x429   => [ 0x449 ],
			0x42A   => [ 0x44A ],
			0x42B   => [ 0x44B ],
			0x42C   => [ 0x44C ],
			0x42D   => [ 0x44D ],
			0x42E   => [ 0x44E ],
			0x42F   => [ 0x44F ],
			0x460   => [ 0x461 ],
			0x462   => [ 0x463 ],
			0x464   => [ 0x465 ],
			0x466   => [ 0x467 ],
			0x468   => [ 0x469 ],
			0x46A   => [ 0x46B ],
			0x46C   => [ 0x46D ],
			0x46E   => [ 0x46F ],
			0x470   => [ 0x471 ],
			0x472   => [ 0x473 ],
			0x474   => [ 0x475 ],
			0x476   => [ 0x477 ],
			0x478   => [ 0x479 ],
			0x47A   => [ 0x47B ],
			0x47C   => [ 0x47D ],
			0x47E   => [ 0x47F ],
			0x480   => [ 0x481 ],
			0x48A   => [ 0x48B ],
			0x48C   => [ 0x48D ],
			0x48E   => [ 0x48F ],
			0x490   => [ 0x491 ],
			0x492   => [ 0x493 ],
			0x494   => [ 0x495 ],
			0x496   => [ 0x497 ],
			0x498   => [ 0x499 ],
			0x49A   => [ 0x49B ],
			0x49C   => [ 0x49D ],
			0x49E   => [ 0x49F ],
			0x4A0   => [ 0x4A1 ],
			0x4A2   => [ 0x4A3 ],
			0x4A4   => [ 0x4A5 ],
			0x4A6   => [ 0x4A7 ],
			0x4A8   => [ 0x4A9 ],
			0x4AA   => [ 0x4AB ],
			0x4AC   => [ 0x4AD ],
			0x4AE   => [ 0x4AF ],
			0x4B0   => [ 0x4B1 ],
			0x4B2   => [ 0x4B3 ],
			0x4B4   => [ 0x4B5 ],
			0x4B6   => [ 0x4B7 ],
			0x4B8   => [ 0x4B9 ],
			0x4BA   => [ 0x4BB ],
			0x4BC   => [ 0x4BD ],
			0x4BE   => [ 0x4BF ],
			0x4C1   => [ 0x4C2 ],
			0x4C3   => [ 0x4C4 ],
			0x4C5   => [ 0x4C6 ],
			0x4C7   => [ 0x4C8 ],
			0x4C9   => [ 0x4CA ],
			0x4CB   => [ 0x4CC ],
			0x4CD   => [ 0x4CE ],
			0x4D0   => [ 0x4D1 ],
			0x4D2   => [ 0x4D3 ],
			0x4D4   => [ 0x4D5 ],
			0x4D6   => [ 0x4D7 ],
			0x4D8   => [ 0x4D9 ],
			0x4DA   => [ 0x4DB ],
			0x4DC   => [ 0x4DD ],
			0x4DE   => [ 0x4DF ],
			0x4E0   => [ 0x4E1 ],
			0x4E2   => [ 0x4E3 ],
			0x4E4   => [ 0x4E5 ],
			0x4E6   => [ 0x4E7 ],
			0x4E8   => [ 0x4E9 ],
			0x4EA   => [ 0x4EB ],
			0x4EC   => [ 0x4ED ],
			0x4EE   => [ 0x4EF ],
			0x4F0   => [ 0x4F1 ],
			0x4F2   => [ 0x4F3 ],
			0x4F4   => [ 0x4F5 ],
			0x4F6   => [ 0x4F7 ],
			0x4F8   => [ 0x4F9 ],
			0x4FA   => [ 0x4FB ],
			0x4FC   => [ 0x4FD ],
			0x4FE   => [ 0x4FF ],
			0x500   => [ 0x501 ],
			0x502   => [ 0x503 ],
			0x504   => [ 0x505 ],
			0x506   => [ 0x507 ],
			0x508   => [ 0x509 ],
			0x50A   => [ 0x50B ],
			0x50C   => [ 0x50D ],
			0x50E   => [ 0x50F ],
			0x510   => [ 0x511 ],
			0x512   => [ 0x513 ],
			0x514   => [ 0x515 ],
			0x516   => [ 0x517 ],
			0x518   => [ 0x519 ],
			0x51A   => [ 0x51B ],
			0x51C   => [ 0x51D ],
			0x51E   => [ 0x51F ],
			0x520   => [ 0x521 ],
			0x522   => [ 0x523 ],
			0x524   => [ 0x525 ],
			0x526   => [ 0x527 ],
			0x528   => [ 0x529 ],
			0x52A   => [ 0x52B ],
			0x52C   => [ 0x52D ],
			0x52E   => [ 0x52F ],
			0x531   => [ 0x561 ],
			0x532   => [ 0x562 ],
			0x533   => [ 0x563 ],
			0x534   => [ 0x564 ],
			0x535   => [ 0x565 ],
			0x536   => [ 0x566 ],
			0x537   => [ 0x567 ],
			0x538   => [ 0x568 ],
			0x539   => [ 0x569 ],
			0x53A   => [ 0x56A ],
			0x53B   => [ 0x56B ],
			0x53C   => [ 0x56C ],
			0x53D   => [ 0x56D ],
			0x53E   => [ 0x56E ],
			0x53F   => [ 0x56F ],
			0x540   => [ 0x570 ],
			0x541   => [ 0x571 ],
			0x542   => [ 0x572 ],
			0x543   => [ 0x573 ],
			0x544   => [ 0x574 ],
			0x545   => [ 0x575 ],
			0x546   => [ 0x576 ],
			0x547   => [ 0x577 ],
			0x548   => [ 0x578 ],
			0x549   => [ 0x579 ],
			0x54A   => [ 0x57A ],
			0x54B   => [ 0x57B ],
			0x54C   => [ 0x57C ],
			0x54D   => [ 0x57D ],
			0x54E   => [ 0x57E ],
			0x54F   => [ 0x57F ],
			0x550   => [ 0x580 ],
			0x551   => [ 0x581 ],
			0x552   => [ 0x582 ],
			0x553   => [ 0x583 ],
			0x554   => [ 0x584 ],
			0x555   => [ 0x585 ],
			0x556   => [ 0x586 ],
			0x587   => [ 0x565, 0x582 ],
			0x675   => [ 0x627, 0x674 ],
			0x676   => [ 0x648, 0x674 ],
			0x677   => [ 0x6C7, 0x674 ],
			0x678   => [ 0x64A, 0x674 ],
			0x958   => [ 0x915, 0x93C ],
			0x959   => [ 0x916, 0x93C ],
			0x95A   => [ 0x917, 0x93C ],
			0x95B   => [ 0x91C, 0x93C ],
			0x95C   => [ 0x921, 0x93C ],
			0x95D   => [ 0x922, 0x93C ],
			0x95E   => [ 0x92B, 0x93C ],
			0x95F   => [ 0x92F, 0x93C ],
			0x9DC   => [ 0x9A1, 0x9BC ],
			0x9DD   => [ 0x9A2, 0x9BC ],
			0x9DF   => [ 0x9AF, 0x9BC ],
			0xA33   => [ 0xA32, 0xA3C ],
			0xA36   => [ 0xA38, 0xA3C ],
			0xA59   => [ 0xA16, 0xA3C ],
			0xA5A   => [ 0xA17, 0xA3C ],
			0xA5B   => [ 0xA1C, 0xA3C ],
			0xA5E   => [ 0xA2B, 0xA3C ],
			0xB5C   => [ 0xB21, 0xB3C ],
			0xB5D   => [ 0xB22, 0xB3C ],
			0xE33   => [ 0xE4D, 0xE32 ],
			0xEB3   => [ 0xECD, 0xEB2 ],
			0xEDC   => [ 0xEAB, 0xE99 ],
			0xEDD   => [ 0xEAB, 0xEA1 ],
			0xF0C   => [ 0xF0B ],
			0xF43   => [ 0xF42, 0xFB7 ],
			0xF4D   => [ 0xF4C, 0xFB7 ],
			0xF52   => [ 0xF51, 0xFB7 ],
			0xF57   => [ 0xF56, 0xFB7 ],
			0xF5C   => [ 0xF5B, 0xFB7 ],
			0xF69   => [ 0xF40, 0xFB5 ],
			0xF73   => [ 0xF71, 0xF72 ],
			0xF75   => [ 0xF71, 0xF74 ],
			0xF76   => [ 0xFB2, 0xF80 ],
			0xF77   => [ 0xFB2, 0xF71, 0xF80 ],
			0xF78   => [ 0xFB3, 0xF80 ],
			0xF79   => [ 0xFB3, 0xF71, 0xF80 ],
			0xF81   => [ 0xF71, 0xF80 ],
			0xF93   => [ 0xF92, 0xFB7 ],
			0xF9D   => [ 0xF9C, 0xFB7 ],
			0xFA2   => [ 0xFA1, 0xFB7 ],
			0xFA7   => [ 0xFA6, 0xFB7 ],
			0xFAC   => [ 0xFAB, 0xFB7 ],
			0xFB9   => [ 0xF90, 0xFB5 ],
			0x10C7  => [ 0x2D27 ],
			0x10CD  => [ 0x2D2D ],
			0x10FC  => [ 0x10DC ],
			0x1D2C  => [ 0x61 ],
			0x1D2D  => [ 0xE6 ],
			0x1D2E  => [ 0x62 ],
			0x1D30  => [ 0x64 ],
			0x1D31  => [ 0x65 ],
			0x1D32  => [ 0x1DD ],
			0x1D33  => [ 0x67 ],
			0x1D34  => [ 0x68 ],
			0x1D35  => [ 0x69 ],
			0x1D36  => [ 0x6A ],
			0x1D37  => [ 0x6B ],
			0x1D38  => [ 0x6C ],
			0x1D39  => [ 0x6D ],
			0x1D3A  => [ 0x6E ],
			0x1D3C  => [ 0x6F ],
			0x1D3D  => [ 0x223 ],
			0x1D3E  => [ 0x70 ],
			0x1D3F  => [ 0x72 ],
			0x1D40  => [ 0x74 ],
			0x1D41  => [ 0x75 ],
			0x1D42  => [ 0x77 ],
			0x1D43  => [ 0x61 ],
			0x1D44  => [ 0x250 ],
			0x1D45  => [ 0x251 ],
			0x1D46  => [ 0x1D02 ],
			0x1D47  => [ 0x62 ],
			0x1D48  => [ 0x64 ],
			0x1D49  => [ 0x65 ],
			0x1D4A  => [ 0x259 ],
			0x1D4B  => [ 0x25B ],
			0x1D4C  => [ 0x25C ],
			0x1D4D  => [ 0x67 ],
			0x1D4F  => [ 0x6B ],
			0x1D50  => [ 0x6D ],
			0x1D51  => [ 0x14B ],
			0x1D52  => [ 0x6F ],
			0x1D53  => [ 0x254 ],
			0x1D54  => [ 0x1D16 ],
			0x1D55  => [ 0x1D17 ],
			0x1D56  => [ 0x70 ],
			0x1D57  => [ 0x74 ],
			0x1D58  => [ 0x75 ],
			0x1D59  => [ 0x1D1D ],
			0x1D5A  => [ 0x26F ],
			0x1D5B  => [ 0x76 ],
			0x1D5C  => [ 0x1D25 ],
			0x1D5D  => [ 0x3B2 ],
			0x1D5E  => [ 0x3B3 ],
			0x1D5F  => [ 0x3B4 ],
			0x1D60  => [ 0x3C6 ],
			0x1D61  => [ 0x3C7 ],
			0x1D62  => [ 0x69 ],
			0x1D63  => [ 0x72 ],
			0x1D64  => [ 0x75 ],
			0x1D65  => [ 0x76 ],
			0x1D66  => [ 0x3B2 ],
			0x1D67  => [ 0x3B3 ],
			0x1D68  => [ 0x3C1 ],
			0x1D69  => [ 0x3C6 ],
			0x1D6A  => [ 0x3C7 ],
			0x1D78  => [ 0x43D ],
			0x1D9B  => [ 0x252 ],
			0x1D9C  => [ 0x63 ],
			0x1D9D  => [ 0x255 ],
			0x1D9E  => [ 0xF0 ],
			0x1D9F  => [ 0x25C ],
			0x1DA0  => [ 0x66 ],
			0x1DA1  => [ 0x25F ],
			0x1DA2  => [ 0x261 ],
			0x1DA3  => [ 0x265 ],
			0x1DA4  => [ 0x268 ],
			0x1DA5  => [ 0x269 ],
			0x1DA6  => [ 0x26A ],
			0x1DA7  => [ 0x1D7B ],
			0x1DA8  => [ 0x29D ],
			0x1DA9  => [ 0x26D ],
			0x1DAA  => [ 0x1D85 ],
			0x1DAB  => [ 0x29F ],
			0x1DAC  => [ 0x271 ],
			0x1DAD  => [ 0x270 ],
			0x1DAE  => [ 0x272 ],
			0x1DAF  => [ 0x273 ],
			0x1DB0  => [ 0x274 ],
			0x1DB1  => [ 0x275 ],
			0x1DB2  => [ 0x278 ],
			0x1DB3  => [ 0x282 ],
			0x1DB4  => [ 0x283 ],
			0x1DB5  => [ 0x1AB ],
			0x1DB6  => [ 0x289 ],
			0x1DB7  => [ 0x28A ],
			0x1DB8  => [ 0x1D1C ],
			0x1DB9  => [ 0x28B ],
			0x1DBA  => [ 0x28C ],
			0x1DBB  => [ 0x7A ],
			0x1DBC  => [ 0x290 ],
			0x1DBD  => [ 0x291 ],
			0x1DBE  => [ 0x292 ],
			0x1DBF  => [ 0x3B8 ],
			0x1E00  => [ 0x1E01 ],
			0x1E02  => [ 0x1E03 ],
			0x1E04  => [ 0x1E05 ],
			0x1E06  => [ 0x1E07 ],
			0x1E08  => [ 0x1E09 ],
			0x1E0A  => [ 0x1E0B ],
			0x1E0C  => [ 0x1E0D ],
			0x1E0E  => [ 0x1E0F ],
			0x1E10  => [ 0x1E11 ],
			0x1E12  => [ 0x1E13 ],
			0x1E14  => [ 0x1E15 ],
			0x1E16  => [ 0x1E17 ],
			0x1E18  => [ 0x1E19 ],
			0x1E1A  => [ 0x1E1B ],
			0x1E1C  => [ 0x1E1D ],
			0x1E1E  => [ 0x1E1F ],
			0x1E20  => [ 0x1E21 ],
			0x1E22  => [ 0x1E23 ],
			0x1E24  => [ 0x1E25 ],
			0x1E26  => [ 0x1E27 ],
			0x1E28  => [ 0x1E29 ],
			0x1E2A  => [ 0x1E2B ],
			0x1E2C  => [ 0x1E2D ],
			0x1E2E  => [ 0x1E2F ],
			0x1E30  => [ 0x1E31 ],
			0x1E32  => [ 0x1E33 ],
			0x1E34  => [ 0x1E35 ],
			0x1E36  => [ 0x1E37 ],
			0x1E38  => [ 0x1E39 ],
			0x1E3A  => [ 0x1E3B ],
			0x1E3C  => [ 0x1E3D ],
			0x1E3E  => [ 0x1E3F ],
			0x1E40  => [ 0x1E41 ],
			0x1E42  => [ 0x1E43 ],
			0x1E44  => [ 0x1E45 ],
			0x1E46  => [ 0x1E47 ],
			0x1E48  => [ 0x1E49 ],
			0x1E4A  => [ 0x1E4B ],
			0x1E4C  => [ 0x1E4D ],
			0x1E4E  => [ 0x1E4F ],
			0x1E50  => [ 0x1E51 ],
			0x1E52  => [ 0x1E53 ],
			0x1E54  => [ 0x1E55 ],
			0x1E56  => [ 0x1E57 ],
			0x1E58  => [ 0x1E59 ],
			0x1E5A  => [ 0x1E5B ],
			0x1E5C  => [ 0x1E5D ],
			0x1E5E  => [ 0x1E5F ],
			0x1E60  => [ 0x1E61 ],
			0x1E62  => [ 0x1E63 ],
			0x1E64  => [ 0x1E65 ],
			0x1E66  => [ 0x1E67 ],
			0x1E68  => [ 0x1E69 ],
			0x1E6A  => [ 0x1E6B ],
			0x1E6C  => [ 0x1E6D ],
			0x1E6E  => [ 0x1E6F ],
			0x1E70  => [ 0x1E71 ],
			0x1E72  => [ 0x1E73 ],
			0x1E74  => [ 0x1E75 ],
			0x1E76  => [ 0x1E77 ],
			0x1E78  => [ 0x1E79 ],
			0x1E7A  => [ 0x1E7B ],
			0x1E7C  => [ 0x1E7D ],
			0x1E7E  => [ 0x1E7F ],
			0x1E80  => [ 0x1E81 ],
			0x1E82  => [ 0x1E83 ],
			0x1E84  => [ 0x1E85 ],
			0x1E86  => [ 0x1E87 ],
			0x1E88  => [ 0x1E89 ],
			0x1E8A  => [ 0x1E8B ],
			0x1E8C  => [ 0x1E8D ],
			0x1E8E  => [ 0x1E8F ],
			0x1E90  => [ 0x1E91 ],
			0x1E92  => [ 0x1E93 ],
			0x1E94  => [ 0x1E95 ],
			0x1E9A  => [ 0x61, 0x2BE ],
			0x1E9B  => [ 0x1E61 ],
			0x1E9E  => [ 0x73, 0x73 ],
			0x1EA0  => [ 0x1EA1 ],
			0x1EA2  => [ 0x1EA3 ],
			0x1EA4  => [ 0x1EA5 ],
			0x1EA6  => [ 0x1EA7 ],
			0x1EA8  => [ 0x1EA9 ],
			0x1EAA  => [ 0x1EAB ],
			0x1EAC  => [ 0x1EAD ],
			0x1EAE  => [ 0x1EAF ],
			0x1EB0  => [ 0x1EB1 ],
			0x1EB2  => [ 0x1EB3 ],
			0x1EB4  => [ 0x1EB5 ],
			0x1EB6  => [ 0x1EB7 ],
			0x1EB8  => [ 0x1EB9 ],
			0x1EBA  => [ 0x1EBB ],
			0x1EBC  => [ 0x1EBD ],
			0x1EBE  => [ 0x1EBF ],
			0x1EC0  => [ 0x1EC1 ],
			0x1EC2  => [ 0x1EC3 ],
			0x1EC4  => [ 0x1EC5 ],
			0x1EC6  => [ 0x1EC7 ],
			0x1EC8  => [ 0x1EC9 ],
			0x1ECA  => [ 0x1ECB ],
			0x1ECC  => [ 0x1ECD ],
			0x1ECE  => [ 0x1ECF ],
			0x1ED0  => [ 0x1ED1 ],
			0x1ED2  => [ 0x1ED3 ],
			0x1ED4  => [ 0x1ED5 ],
			0x1ED6  => [ 0x1ED7 ],
			0x1ED8  => [ 0x1ED9 ],
			0x1EDA  => [ 0x1EDB ],
			0x1EDC  => [ 0x1EDD ],
			0x1EDE  => [ 0x1EDF ],
			0x1EE0  => [ 0x1EE1 ],
			0x1EE2  => [ 0x1EE3 ],
			0x1EE4  => [ 0x1EE5 ],
			0x1EE6  => [ 0x1EE7 ],
			0x1EE8  => [ 0x1EE9 ],
			0x1EEA  => [ 0x1EEB ],
			0x1EEC  => [ 0x1EED ],
			0x1EEE  => [ 0x1EEF ],
			0x1EF0  => [ 0x1EF1 ],
			0x1EF2  => [ 0x1EF3 ],
			0x1EF4  => [ 0x1EF5 ],
			0x1EF6  => [ 0x1EF7 ],
			0x1EF8  => [ 0x1EF9 ],
			0x1EFA  => [ 0x1EFB ],
			0x1EFC  => [ 0x1EFD ],
			0x1EFE  => [ 0x1EFF ],
			0x1F08  => [ 0x1F00 ],
			0x1F09  => [ 0x1F01 ],
			0x1F0A  => [ 0x1F02 ],
			0x1F0B  => [ 0x1F03 ],
			0x1F0C  => [ 0x1F04 ],
			0x1F0D  => [ 0x1F05 ],
			0x1F0E  => [ 0x1F06 ],
			0x1F0F  => [ 0x1F07 ],
			0x1F18  => [ 0x1F10 ],
			0x1F19  => [ 0x1F11 ],
			0x1F1A  => [ 0x1F12 ],
			0x1F1B  => [ 0x1F13 ],
			0x1F1C  => [ 0x1F14 ],
			0x1F1D  => [ 0x1F15 ],
			0x1F28  => [ 0x1F20 ],
			0x1F29  => [ 0x1F21 ],
			0x1F2A  => [ 0x1F22 ],
			0x1F2B  => [ 0x1F23 ],
			0x1F2C  => [ 0x1F24 ],
			0x1F2D  => [ 0x1F25 ],
			0x1F2E  => [ 0x1F26 ],
			0x1F2F  => [ 0x1F27 ],
			0x1F38  => [ 0x1F30 ],
			0x1F39  => [ 0x1F31 ],
			0x1F3A  => [ 0x1F32 ],
			0x1F3B  => [ 0x1F33 ],
			0x1F3C  => [ 0x1F34 ],
			0x1F3D  => [ 0x1F35 ],
			0x1F3E  => [ 0x1F36 ],
			0x1F3F  => [ 0x1F37 ],
			0x1F48  => [ 0x1F40 ],
			0x1F49  => [ 0x1F41 ],
			0x1F4A  => [ 0x1F42 ],
			0x1F4B  => [ 0x1F43 ],
			0x1F4C  => [ 0x1F44 ],
			0x1F4D  => [ 0x1F45 ],
			0x1F59  => [ 0x1F51 ],
			0x1F5B  => [ 0x1F53 ],
			0x1F5D  => [ 0x1F55 ],
			0x1F5F  => [ 0x1F57 ],
			0x1F68  => [ 0x1F60 ],
			0x1F69  => [ 0x1F61 ],
			0x1F6A  => [ 0x1F62 ],
			0x1F6B  => [ 0x1F63 ],
			0x1F6C  => [ 0x1F64 ],
			0x1F6D  => [ 0x1F65 ],
			0x1F6E  => [ 0x1F66 ],
			0x1F6F  => [ 0x1F67 ],
			0x1F71  => [ 0x3AC ],
			0x1F73  => [ 0x3AD ],
			0x1F75  => [ 0x3AE ],
			0x1F77  => [ 0x3AF ],
			0x1F79  => [ 0x3CC ],
			0x1F7B  => [ 0x3CD ],
			0x1F7D  => [ 0x3CE ],
			0x1F80  => [ 0x1F00, 0x3B9 ],
			0x1F81  => [ 0x1F01, 0x3B9 ],
			0x1F82  => [ 0x1F02, 0x3B9 ],
			0x1F83  => [ 0x1F03, 0x3B9 ],
			0x1F84  => [ 0x1F04, 0x3B9 ],
			0x1F85  => [ 0x1F05, 0x3B9 ],
			0x1F86  => [ 0x1F06, 0x3B9 ],
			0x1F87  => [ 0x1F07, 0x3B9 ],
			0x1F88  => [ 0x1F00, 0x3B9 ],
			0x1F89  => [ 0x1F01, 0x3B9 ],
			0x1F8A  => [ 0x1F02, 0x3B9 ],
			0x1F8B  => [ 0x1F03, 0x3B9 ],
			0x1F8C  => [ 0x1F04, 0x3B9 ],
			0x1F8D  => [ 0x1F05, 0x3B9 ],
			0x1F8E  => [ 0x1F06, 0x3B9 ],
			0x1F8F  => [ 0x1F07, 0x3B9 ],
			0x1F90  => [ 0x1F20, 0x3B9 ],
			0x1F91  => [ 0x1F21, 0x3B9 ],
			0x1F92  => [ 0x1F22, 0x3B9 ],
			0x1F93  => [ 0x1F23, 0x3B9 ],
			0x1F94  => [ 0x1F24, 0x3B9 ],
			0x1F95  => [ 0x1F25, 0x3B9 ],
			0x1F96  => [ 0x1F26, 0x3B9 ],
			0x1F97  => [ 0x1F27, 0x3B9 ],
			0x1F98  => [ 0x1F20, 0x3B9 ],
			0x1F99  => [ 0x1F21, 0x3B9 ],
			0x1F9A  => [ 0x1F22, 0x3B9 ],
			0x1F9B  => [ 0x1F23, 0x3B9 ],
			0x1F9C  => [ 0x1F24, 0x3B9 ],
			0x1F9D  => [ 0x1F25, 0x3B9 ],
			0x1F9E  => [ 0x1F26, 0x3B9 ],
			0x1F9F  => [ 0x1F27, 0x3B9 ],
			0x1FA0  => [ 0x1F60, 0x3B9 ],
			0x1FA1  => [ 0x1F61, 0x3B9 ],
			0x1FA2  => [ 0x1F62, 0x3B9 ],
			0x1FA3  => [ 0x1F63, 0x3B9 ],
			0x1FA4  => [ 0x1F64, 0x3B9 ],
			0x1FA5  => [ 0x1F65, 0x3B9 ],
			0x1FA6  => [ 0x1F66, 0x3B9 ],
			0x1FA7  => [ 0x1F67, 0x3B9 ],
			0x1FA8  => [ 0x1F60, 0x3B9 ],
			0x1FA9  => [ 0x1F61, 0x3B9 ],
			0x1FAA  => [ 0x1F62, 0x3B9 ],
			0x1FAB  => [ 0x1F63, 0x3B9 ],
			0x1FAC  => [ 0x1F64, 0x3B9 ],
			0x1FAD  => [ 0x1F65, 0x3B9 ],
			0x1FAE  => [ 0x1F66, 0x3B9 ],
			0x1FAF  => [ 0x1F67, 0x3B9 ],
			0x1FB2  => [ 0x1F70, 0x3B9 ],
			0x1FB3  => [ 0x3B1, 0x3B9 ],
			0x1FB4  => [ 0x3AC, 0x3B9 ],
			0x1FB7  => [ 0x1FB6, 0x3B9 ],
			0x1FB8  => [ 0x1FB0 ],
			0x1FB9  => [ 0x1FB1 ],
			0x1FBA  => [ 0x1F70 ],
			0x1FBB  => [ 0x3AC ],
			0x1FBC  => [ 0x3B1, 0x3B9 ],
			0x1FBE  => [ 0x3B9 ],
			0x1FC2  => [ 0x1F74, 0x3B9 ],
			0x1FC3  => [ 0x3B7, 0x3B9 ],
			0x1FC4  => [ 0x3AE, 0x3B9 ],
			0x1FC7  => [ 0x1FC6, 0x3B9 ],
			0x1FC8  => [ 0x1F72 ],
			0x1FC9  => [ 0x3AD ],
			0x1FCA  => [ 0x1F74 ],
			0x1FCB  => [ 0x3AE ],
			0x1FCC  => [ 0x3B7, 0x3B9 ],
			0x1FD3  => [ 0x390 ],
			0x1FD8  => [ 0x1FD0 ],
			0x1FD9  => [ 0x1FD1 ],
			0x1FDA  => [ 0x1F76 ],
			0x1FDB  => [ 0x3AF ],
			0x1FE3  => [ 0x3B0 ],
			0x1FE8  => [ 0x1FE0 ],
			0x1FE9  => [ 0x1FE1 ],
			0x1FEA  => [ 0x1F7A ],
			0x1FEB  => [ 0x3CD ],
			0x1FEC  => [ 0x1FE5 ],
			0x1FF2  => [ 0x1F7C, 0x3B9 ],
			0x1FF3  => [ 0x3C9, 0x3B9 ],
			0x1FF4  => [ 0x3CE, 0x3B9 ],
			0x1FF7  => [ 0x1FF6, 0x3B9 ],
			0x1FF8  => [ 0x1F78 ],
			0x1FF9  => [ 0x3CC ],
			0x1FFA  => [ 0x1F7C ],
			0x1FFB  => [ 0x3CE ],
			0x1FFC  => [ 0x3C9, 0x3B9 ],
			0x2011  => [ 0x2010 ],
			0x2033  => [ 0x2032, 0x2032 ],
			0x2034  => [ 0x2032, 0x2032, 0x2032 ],
			0x2036  => [ 0x2035, 0x2035 ],
			0x2037  => [ 0x2035, 0x2035, 0x2035 ],
			0x2057  => [ 0x2032, 0x2032, 0x2032, 0x2032 ],
			0x2070  => [ 0x30 ],
			0x2071  => [ 0x69 ],
			0x2074  => [ 0x34 ],
			0x2075  => [ 0x35 ],
			0x2076  => [ 0x36 ],
			0x2077  => [ 0x37 ],
			0x2078  => [ 0x38 ],
			0x2079  => [ 0x39 ],
			0x207B  => [ 0x2212 ],
			0x207F  => [ 0x6E ],
			0x2080  => [ 0x30 ],
			0x2081  => [ 0x31 ],
			0x2082  => [ 0x32 ],
			0x2083  => [ 0x33 ],
			0x2084  => [ 0x34 ],
			0x2085  => [ 0x35 ],
			0x2086  => [ 0x36 ],
			0x2087  => [ 0x37 ],
			0x2088  => [ 0x38 ],
			0x2089  => [ 0x39 ],
			0x208B  => [ 0x2212 ],
			0x2090  => [ 0x61 ],
			0x2091  => [ 0x65 ],
			0x2092  => [ 0x6F ],
			0x2093  => [ 0x78 ],
			0x2094  => [ 0x259 ],
			0x2095  => [ 0x68 ],
			0x2096  => [ 0x6B ],
			0x2097  => [ 0x6C ],
			0x2098  => [ 0x6D ],
			0x2099  => [ 0x6E ],
			0x209A  => [ 0x70 ],
			0x209B  => [ 0x73 ],
			0x209C  => [ 0x74 ],
			0x20A8  => [ 0x72, 0x73 ],
			0x2102  => [ 0x63 ],
			0x2103  => [ 0xB0, 0x63 ],
			0x2107  => [ 0x25B ],
			0x2109  => [ 0xB0, 0x66 ],
			0x210A  => [ 0x67 ],
			0x210B  => [ 0x68 ],
			0x210C  => [ 0x68 ],
			0x210D  => [ 0x68 ],
			0x210E  => [ 0x68 ],
			0x210F  => [ 0x127 ],
			0x2110  => [ 0x69 ],
			0x2111  => [ 0x69 ],
			0x2112  => [ 0x6C ],
			0x2113  => [ 0x6C ],
			0x2115  => [ 0x6E ],
			0x2116  => [ 0x6E, 0x6F ],
			0x2119  => [ 0x70 ],
			0x211A  => [ 0x71 ],
			0x211B  => [ 0x72 ],
			0x211C  => [ 0x72 ],
			0x211D  => [ 0x72 ],
			0x2120  => [ 0x73, 0x6D ],
			0x2121  => [ 0x74, 0x65, 0x6C ],
			0x2122  => [ 0x74, 0x6D ],
			0x2124  => [ 0x7A ],
			0x2126  => [ 0x3C9 ],
			0x2128  => [ 0x7A ],
			0x212A  => [ 0x6B ],
			0x212B  => [ 0xE5 ],
			0x212C  => [ 0x62 ],
			0x212D  => [ 0x63 ],
			0x212F  => [ 0x65 ],
			0x2130  => [ 0x65 ],
			0x2131  => [ 0x66 ],
			0x2133  => [ 0x6D ],
			0x2134  => [ 0x6F ],
			0x2135  => [ 0x5D0 ],
			0x2136  => [ 0x5D1 ],
			0x2137  => [ 0x5D2 ],
			0x2138  => [ 0x5D3 ],
			0x2139  => [ 0x69 ],
			0x213B  => [ 0x66, 0x61, 0x78 ],
			0x213C  => [ 0x3C0 ],
			0x213D  => [ 0x3B3 ],
			0x213E  => [ 0x3B3 ],
			0x213F  => [ 0x3C0 ],
			0x2140  => [ 0x2211 ],
			0x2145  => [ 0x64 ],
			0x2146  => [ 0x64 ],
			0x2147  => [ 0x65 ],
			0x2148  => [ 0x69 ],
			0x2149  => [ 0x6A ],
			0x2150  => [ 0x31, 0x2044, 0x37 ],
			0x2151  => [ 0x31, 0x2044, 0x39 ],
			0x2152  => [ 0x31, 0x2044, 0x31, 0x30 ],
			0x2153  => [ 0x31, 0x2044, 0x33 ],
			0x2154  => [ 0x32, 0x2044, 0x33 ],
			0x2155  => [ 0x31, 0x2044, 0x35 ],
			0x2156  => [ 0x32, 0x2044, 0x35 ],
			0x2157  => [ 0x33, 0x2044, 0x35 ],
			0x2158  => [ 0x34, 0x2044, 0x35 ],
			0x2159  => [ 0x31, 0x2044, 0x36 ],
			0x215A  => [ 0x35, 0x2044, 0x36 ],
			0x215B  => [ 0x31, 0x2044, 0x38 ],
			0x215C  => [ 0x33, 0x2044, 0x38 ],
			0x215D  => [ 0x35, 0x2044, 0x38 ],
			0x215E  => [ 0x37, 0x2044, 0x38 ],
			0x215F  => [ 0x31, 0x2044 ],
			0x2160  => [ 0x69 ],
			0x2161  => [ 0x69, 0x69 ],
			0x2162  => [ 0x69, 0x69, 0x69 ],
			0x2163  => [ 0x69, 0x76 ],
			0x2164  => [ 0x76 ],
			0x2165  => [ 0x76, 0x69 ],
			0x2166  => [ 0x76, 0x69, 0x69 ],
			0x2167  => [ 0x76, 0x69, 0x69, 0x69 ],
			0x2168  => [ 0x69, 0x78 ],
			0x2169  => [ 0x78 ],
			0x216A  => [ 0x78, 0x69 ],
			0x216B  => [ 0x78, 0x69, 0x69 ],
			0x216C  => [ 0x6C ],
			0x216D  => [ 0x63 ],
			0x216E  => [ 0x64 ],
			0x216F  => [ 0x6D ],
			0x2170  => [ 0x69 ],
			0x2171  => [ 0x69, 0x69 ],
			0x2172  => [ 0x69, 0x69, 0x69 ],
			0x2173  => [ 0x69, 0x76 ],
			0x2174  => [ 0x76 ],
			0x2175  => [ 0x76, 0x69 ],
			0x2176  => [ 0x76, 0x69, 0x69 ],
			0x2177  => [ 0x76, 0x69, 0x69, 0x69 ],
			0x2178  => [ 0x69, 0x78 ],
			0x2179  => [ 0x78 ],
			0x217A  => [ 0x78, 0x69 ],
			0x217B  => [ 0x78, 0x69, 0x69 ],
			0x217C  => [ 0x6C ],
			0x217D  => [ 0x63 ],
			0x217E  => [ 0x64 ],
			0x217F  => [ 0x6D ],
			0x2189  => [ 0x30, 0x2044, 0x33 ],
			0x222C  => [ 0x222B, 0x222B ],
			0x222D  => [ 0x222B, 0x222B, 0x222B ],
			0x222F  => [ 0x222E, 0x222E ],
			0x2230  => [ 0x222E, 0x222E, 0x222E ],
			0x2329  => [ 0x3008 ],
			0x232A  => [ 0x3009 ],
			0x2460  => [ 0x31 ],
			0x2461  => [ 0x32 ],
			0x2462  => [ 0x33 ],
			0x2463  => [ 0x34 ],
			0x2464  => [ 0x35 ],
			0x2465  => [ 0x36 ],
			0x2466  => [ 0x37 ],
			0x2467  => [ 0x38 ],
			0x2468  => [ 0x39 ],
			0x2469  => [ 0x31, 0x30 ],
			0x246A  => [ 0x31, 0x31 ],
			0x246B  => [ 0x31, 0x32 ],
			0x246C  => [ 0x31, 0x33 ],
			0x246D  => [ 0x31, 0x34 ],
			0x246E  => [ 0x31, 0x35 ],
			0x246F  => [ 0x31, 0x36 ],
			0x2470  => [ 0x31, 0x37 ],
			0x2471  => [ 0x31, 0x38 ],
			0x2472  => [ 0x31, 0x39 ],
			0x2473  => [ 0x32, 0x30 ],
			0x24B6  => [ 0x61 ],
			0x24B7  => [ 0x62 ],
			0x24B8  => [ 0x63 ],
			0x24B9  => [ 0x64 ],
			0x24BA  => [ 0x65 ],
			0x24BB  => [ 0x66 ],
			0x24BC  => [ 0x67 ],
			0x24BD  => [ 0x68 ],
			0x24BE  => [ 0x69 ],
			0x24BF  => [ 0x6A ],
			0x24C0  => [ 0x6B ],
			0x24C1  => [ 0x6C ],
			0x24C2  => [ 0x6D ],
			0x24C3  => [ 0x6E ],
			0x24C4  => [ 0x6F ],
			0x24C5  => [ 0x70 ],
			0x24C6  => [ 0x71 ],
			0x24C7  => [ 0x72 ],
			0x24C8  => [ 0x73 ],
			0x24C9  => [ 0x74 ],
			0x24CA  => [ 0x75 ],
			0x24CB  => [ 0x76 ],
			0x24CC  => [ 0x77 ],
			0x24CD  => [ 0x78 ],
			0x24CE  => [ 0x79 ],
			0x24CF  => [ 0x7A ],
			0x24D0  => [ 0x61 ],
			0x24D1  => [ 0x62 ],
			0x24D2  => [ 0x63 ],
			0x24D3  => [ 0x64 ],
			0x24D4  => [ 0x65 ],
			0x24D5  => [ 0x66 ],
			0x24D6  => [ 0x67 ],
			0x24D7  => [ 0x68 ],
			0x24D8  => [ 0x69 ],
			0x24D9  => [ 0x6A ],
			0x24DA  => [ 0x6B ],
			0x24DB  => [ 0x6C ],
			0x24DC  => [ 0x6D ],
			0x24DD  => [ 0x6E ],
			0x24DE  => [ 0x6F ],
			0x24DF  => [ 0x70 ],
			0x24E0  => [ 0x71 ],
			0x24E1  => [ 0x72 ],
			0x24E2  => [ 0x73 ],
			0x24E3  => [ 0x74 ],
			0x24E4  => [ 0x75 ],
			0x24E5  => [ 0x76 ],
			0x24E6  => [ 0x77 ],
			0x24E7  => [ 0x78 ],
			0x24E8  => [ 0x79 ],
			0x24E9  => [ 0x7A ],
			0x24EA  => [ 0x30 ],
			0x2A0C  => [ 0x222B, 0x222B, 0x222B, 0x222B ],
			0x2ADC  => [ 0x2ADD, 0x338 ],
			0x2C00  => [ 0x2C30 ],
			0x2C01  => [ 0x2C31 ],
			0x2C02  => [ 0x2C32 ],
			0x2C03  => [ 0x2C33 ],
			0x2C04  => [ 0x2C34 ],
			0x2C05  => [ 0x2C35 ],
			0x2C06  => [ 0x2C36 ],
			0x2C07  => [ 0x2C37 ],
			0x2C08  => [ 0x2C38 ],
			0x2C09  => [ 0x2C39 ],
			0x2C0A  => [ 0x2C3A ],
			0x2C0B  => [ 0x2C3B ],
			0x2C0C  => [ 0x2C3C ],
			0x2C0D  => [ 0x2C3D ],
			0x2C0E  => [ 0x2C3E ],
			0x2C0F  => [ 0x2C3F ],
			0x2C10  => [ 0x2C40 ],
			0x2C11  => [ 0x2C41 ],
			0x2C12  => [ 0x2C42 ],
			0x2C13  => [ 0x2C43 ],
			0x2C14  => [ 0x2C44 ],
			0x2C15  => [ 0x2C45 ],
			0x2C16  => [ 0x2C46 ],
			0x2C17  => [ 0x2C47 ],
			0x2C18  => [ 0x2C48 ],
			0x2C19  => [ 0x2C49 ],
			0x2C1A  => [ 0x2C4A ],
			0x2C1B  => [ 0x2C4B ],
			0x2C1C  => [ 0x2C4C ],
			0x2C1D  => [ 0x2C4D ],
			0x2C1E  => [ 0x2C4E ],
			0x2C1F  => [ 0x2C4F ],
			0x2C20  => [ 0x2C50 ],
			0x2C21  => [ 0x2C51 ],
			0x2C22  => [ 0x2C52 ],
			0x2C23  => [ 0x2C53 ],
			0x2C24  => [ 0x2C54 ],
			0x2C25  => [ 0x2C55 ],
			0x2C26  => [ 0x2C56 ],
			0x2C27  => [ 0x2C57 ],
			0x2C28  => [ 0x2C58 ],
			0x2C29  => [ 0x2C59 ],
			0x2C2A  => [ 0x2C5A ],
			0x2C2B  => [ 0x2C5B ],
			0x2C2C  => [ 0x2C5C ],
			0x2C2D  => [ 0x2C5D ],
			0x2C2E  => [ 0x2C5E ],
			0x2C60  => [ 0x2C61 ],
			0x2C62  => [ 0x26B ],
			0x2C63  => [ 0x1D7D ],
			0x2C64  => [ 0x27D ],
			0x2C67  => [ 0x2C68 ],
			0x2C69  => [ 0x2C6A ],
			0x2C6B  => [ 0x2C6C ],
			0x2C6D  => [ 0x251 ],
			0x2C6E  => [ 0x271 ],
			0x2C6F  => [ 0x250 ],
			0x2C70  => [ 0x252 ],
			0x2C72  => [ 0x2C73 ],
			0x2C75  => [ 0x2C76 ],
			0x2C7C  => [ 0x6A ],
			0x2C7D  => [ 0x76 ],
			0x2C7E  => [ 0x23F ],
			0x2C7F  => [ 0x240 ],
			0x2C80  => [ 0x2C81 ],
			0x2C82  => [ 0x2C83 ],
			0x2C84  => [ 0x2C85 ],
			0x2C86  => [ 0x2C87 ],
			0x2C88  => [ 0x2C89 ],
			0x2C8A  => [ 0x2C8B ],
			0x2C8C  => [ 0x2C8D ],
			0x2C8E  => [ 0x2C8F ],
			0x2C90  => [ 0x2C91 ],
			0x2C92  => [ 0x2C93 ],
			0x2C94  => [ 0x2C95 ],
			0x2C96  => [ 0x2C97 ],
			0x2C98  => [ 0x2C99 ],
			0x2C9A  => [ 0x2C9B ],
			0x2C9C  => [ 0x2C9D ],
			0x2C9E  => [ 0x2C9F ],
			0x2CA0  => [ 0x2CA1 ],
			0x2CA2  => [ 0x2CA3 ],
			0x2CA4  => [ 0x2CA5 ],
			0x2CA6  => [ 0x2CA7 ],
			0x2CA8  => [ 0x2CA9 ],
			0x2CAA  => [ 0x2CAB ],
			0x2CAC  => [ 0x2CAD ],
			0x2CAE  => [ 0x2CAF ],
			0x2CB0  => [ 0x2CB1 ],
			0x2CB2  => [ 0x2CB3 ],
			0x2CB4  => [ 0x2CB5 ],
			0x2CB6  => [ 0x2CB7 ],
			0x2CB8  => [ 0x2CB9 ],
			0x2CBA  => [ 0x2CBB ],
			0x2CBC  => [ 0x2CBD ],
			0x2CBE  => [ 0x2CBF ],
			0x2CC0  => [ 0x2CC1 ],
			0x2CC2  => [ 0x2CC3 ],
			0x2CC4  => [ 0x2CC5 ],
			0x2CC6  => [ 0x2CC7 ],
			0x2CC8  => [ 0x2CC9 ],
			0x2CCA  => [ 0x2CCB ],
			0x2CCC  => [ 0x2CCD ],
			0x2CCE  => [ 0x2CCF ],
			0x2CD0  => [ 0x2CD1 ],
			0x2CD2  => [ 0x2CD3 ],
			0x2CD4  => [ 0x2CD5 ],
			0x2CD6  => [ 0x2CD7 ],
			0x2CD8  => [ 0x2CD9 ],
			0x2CDA  => [ 0x2CDB ],
			0x2CDC  => [ 0x2CDD ],
			0x2CDE  => [ 0x2CDF ],
			0x2CE0  => [ 0x2CE1 ],
			0x2CE2  => [ 0x2CE3 ],
			0x2CEB  => [ 0x2CEC ],
			0x2CED  => [ 0x2CEE ],
			0x2CF2  => [ 0x2CF3 ],
			0x2D6F  => [ 0x2D61 ],
			0x2E9F  => [ 0x6BCD ],
			0x2EF3  => [ 0x9F9F ],
			0x2F00  => [ 0x4E00 ],
			0x2F01  => [ 0x4E28 ],
			0x2F02  => [ 0x4E36 ],
			0x2F03  => [ 0x4E3F ],
			0x2F04  => [ 0x4E59 ],
			0x2F05  => [ 0x4E85 ],
			0x2F06  => [ 0x4E8C ],
			0x2F07  => [ 0x4EA0 ],
			0x2F08  => [ 0x4EBA ],
			0x2F09  => [ 0x513F ],
			0x2F0A  => [ 0x5165 ],
			0x2F0B  => [ 0x516B ],
			0x2F0C  => [ 0x5182 ],
			0x2F0D  => [ 0x5196 ],
			0x2F0E  => [ 0x51AB ],
			0x2F0F  => [ 0x51E0 ],
			0x2F10  => [ 0x51F5 ],
			0x2F11  => [ 0x5200 ],
			0x2F12  => [ 0x529B ],
			0x2F13  => [ 0x52F9 ],
			0x2F14  => [ 0x5315 ],
			0x2F15  => [ 0x531A ],
			0x2F16  => [ 0x5338 ],
			0x2F17  => [ 0x5341 ],
			0x2F18  => [ 0x535C ],
			0x2F19  => [ 0x5369 ],
			0x2F1A  => [ 0x5382 ],
			0x2F1B  => [ 0x53B6 ],
			0x2F1C  => [ 0x53C8 ],
			0x2F1D  => [ 0x53E3 ],
			0x2F1E  => [ 0x56D7 ],
			0x2F1F  => [ 0x571F ],
			0x2F20  => [ 0x58EB ],
			0x2F21  => [ 0x5902 ],
			0x2F22  => [ 0x590A ],
			0x2F23  => [ 0x5915 ],
			0x2F24  => [ 0x5927 ],
			0x2F25  => [ 0x5973 ],
			0x2F26  => [ 0x5B50 ],
			0x2F27  => [ 0x5B80 ],
			0x2F28  => [ 0x5BF8 ],
			0x2F29  => [ 0x5C0F ],
			0x2F2A  => [ 0x5C22 ],
			0x2F2B  => [ 0x5C38 ],
			0x2F2C  => [ 0x5C6E ],
			0x2F2D  => [ 0x5C71 ],
			0x2F2E  => [ 0x5DDB ],
			0x2F2F  => [ 0x5DE5 ],
			0x2F30  => [ 0x5DF1 ],
			0x2F31  => [ 0x5DFE ],
			0x2F32  => [ 0x5E72 ],
			0x2F33  => [ 0x5E7A ],
			0x2F34  => [ 0x5E7F ],
			0x2F35  => [ 0x5EF4 ],
			0x2F36  => [ 0x5EFE ],
			0x2F37  => [ 0x5F0B ],
			0x2F38  => [ 0x5F13 ],
			0x2F39  => [ 0x5F50 ],
			0x2F3A  => [ 0x5F61 ],
			0x2F3B  => [ 0x5F73 ],
			0x2F3C  => [ 0x5FC3 ],
			0x2F3D  => [ 0x6208 ],
			0x2F3E  => [ 0x6236 ],
			0x2F3F  => [ 0x624B ],
			0x2F40  => [ 0x652F ],
			0x2F41  => [ 0x6534 ],
			0x2F42  => [ 0x6587 ],
			0x2F43  => [ 0x6597 ],
			0x2F44  => [ 0x65A4 ],
			0x2F45  => [ 0x65B9 ],
			0x2F46  => [ 0x65E0 ],
			0x2F47  => [ 0x65E5 ],
			0x2F48  => [ 0x66F0 ],
			0x2F49  => [ 0x6708 ],
			0x2F4A  => [ 0x6728 ],
			0x2F4B  => [ 0x6B20 ],
			0x2F4C  => [ 0x6B62 ],
			0x2F4D  => [ 0x6B79 ],
			0x2F4E  => [ 0x6BB3 ],
			0x2F4F  => [ 0x6BCB ],
			0x2F50  => [ 0x6BD4 ],
			0x2F51  => [ 0x6BDB ],
			0x2F52  => [ 0x6C0F ],
			0x2F53  => [ 0x6C14 ],
			0x2F54  => [ 0x6C34 ],
			0x2F55  => [ 0x706B ],
			0x2F56  => [ 0x722A ],
			0x2F57  => [ 0x7236 ],
			0x2F58  => [ 0x723B ],
			0x2F59  => [ 0x723F ],
			0x2F5A  => [ 0x7247 ],
			0x2F5B  => [ 0x7259 ],
			0x2F5C  => [ 0x725B ],
			0x2F5D  => [ 0x72AC ],
			0x2F5E  => [ 0x7384 ],
			0x2F5F  => [ 0x7389 ],
			0x2F60  => [ 0x74DC ],
			0x2F61  => [ 0x74E6 ],
			0x2F62  => [ 0x7518 ],
			0x2F63  => [ 0x751F ],
			0x2F64  => [ 0x7528 ],
			0x2F65  => [ 0x7530 ],
			0x2F66  => [ 0x758B ],
			0x2F67  => [ 0x7592 ],
			0x2F68  => [ 0x7676 ],
			0x2F69  => [ 0x767D ],
			0x2F6A  => [ 0x76AE ],
			0x2F6B  => [ 0x76BF ],
			0x2F6C  => [ 0x76EE ],
			0x2F6D  => [ 0x77DB ],
			0x2F6E  => [ 0x77E2 ],
			0x2F6F  => [ 0x77F3 ],
			0x2F70  => [ 0x793A ],
			0x2F71  => [ 0x79B8 ],
			0x2F72  => [ 0x79BE ],
			0x2F73  => [ 0x7A74 ],
			0x2F74  => [ 0x7ACB ],
			0x2F75  => [ 0x7AF9 ],
			0x2F76  => [ 0x7C73 ],
			0x2F77  => [ 0x7CF8 ],
			0x2F78  => [ 0x7F36 ],
			0x2F79  => [ 0x7F51 ],
			0x2F7A  => [ 0x7F8A ],
			0x2F7B  => [ 0x7FBD ],
			0x2F7C  => [ 0x8001 ],
			0x2F7D  => [ 0x800C ],
			0x2F7E  => [ 0x8012 ],
			0x2F7F  => [ 0x8033 ],
			0x2F80  => [ 0x807F ],
			0x2F81  => [ 0x8089 ],
			0x2F82  => [ 0x81E3 ],
			0x2F83  => [ 0x81EA ],
			0x2F84  => [ 0x81F3 ],
			0x2F85  => [ 0x81FC ],
			0x2F86  => [ 0x820C ],
			0x2F87  => [ 0x821B ],
			0x2F88  => [ 0x821F ],
			0x2F89  => [ 0x826E ],
			0x2F8A  => [ 0x8272 ],
			0x2F8B  => [ 0x8278 ],
			0x2F8C  => [ 0x864D ],
			0x2F8D  => [ 0x866B ],
			0x2F8E  => [ 0x8840 ],
			0x2F8F  => [ 0x884C ],
			0x2F90  => [ 0x8863 ],
			0x2F91  => [ 0x897E ],
			0x2F92  => [ 0x898B ],
			0x2F93  => [ 0x89D2 ],
			0x2F94  => [ 0x8A00 ],
			0x2F95  => [ 0x8C37 ],
			0x2F96  => [ 0x8C46 ],
			0x2F97  => [ 0x8C55 ],
			0x2F98  => [ 0x8C78 ],
			0x2F99  => [ 0x8C9D ],
			0x2F9A  => [ 0x8D64 ],
			0x2F9B  => [ 0x8D70 ],
			0x2F9C  => [ 0x8DB3 ],
			0x2F9D  => [ 0x8EAB ],
			0x2F9E  => [ 0x8ECA ],
			0x2F9F  => [ 0x8F9B ],
			0x2FA0  => [ 0x8FB0 ],
			0x2FA1  => [ 0x8FB5 ],
			0x2FA2  => [ 0x9091 ],
			0x2FA3  => [ 0x9149 ],
			0x2FA4  => [ 0x91C6 ],
			0x2FA5  => [ 0x91CC ],
			0x2FA6  => [ 0x91D1 ],
			0x2FA7  => [ 0x9577 ],
			0x2FA8  => [ 0x9580 ],
			0x2FA9  => [ 0x961C ],
			0x2FAA  => [ 0x96B6 ],
			0x2FAB  => [ 0x96B9 ],
			0x2FAC  => [ 0x96E8 ],
			0x2FAD  => [ 0x9751 ],
			0x2FAE  => [ 0x975E ],
			0x2FAF  => [ 0x9762 ],
			0x2FB0  => [ 0x9769 ],
			0x2FB1  => [ 0x97CB ],
			0x2FB2  => [ 0x97ED ],
			0x2FB3  => [ 0x97F3 ],
			0x2FB4  => [ 0x9801 ],
			0x2FB5  => [ 0x98A8 ],
			0x2FB6  => [ 0x98DB ],
			0x2FB7  => [ 0x98DF ],
			0x2FB8  => [ 0x9996 ],
			0x2FB9  => [ 0x9999 ],
			0x2FBA  => [ 0x99AC ],
			0x2FBB  => [ 0x9AA8 ],
			0x2FBC  => [ 0x9AD8 ],
			0x2FBD  => [ 0x9ADF ],
			0x2FBE  => [ 0x9B25 ],
			0x2FBF  => [ 0x9B2F ],
			0x2FC0  => [ 0x9B32 ],
			0x2FC1  => [ 0x9B3C ],
			0x2FC2  => [ 0x9B5A ],
			0x2FC3  => [ 0x9CE5 ],
			0x2FC4  => [ 0x9E75 ],
			0x2FC5  => [ 0x9E7F ],
			0x2FC6  => [ 0x9EA5 ],
			0x2FC7  => [ 0x9EBB ],
			0x2FC8  => [ 0x9EC3 ],
			0x2FC9  => [ 0x9ECD ],
			0x2FCA  => [ 0x9ED1 ],
			0x2FCB  => [ 0x9EF9 ],
			0x2FCC  => [ 0x9EFD ],
			0x2FCD  => [ 0x9F0E ],
			0x2FCE  => [ 0x9F13 ],
			0x2FCF  => [ 0x9F20 ],
			0x2FD0  => [ 0x9F3B ],
			0x2FD1  => [ 0x9F4A ],
			0x2FD2  => [ 0x9F52 ],
			0x2FD3  => [ 0x9F8D ],
			0x2FD4  => [ 0x9F9C ],
			0x2FD5  => [ 0x9FA0 ],
			0x3002  => [ 0x2E ],
			0x3036  => [ 0x3012 ],
			0x3038  => [ 0x5341 ],
			0x3039  => [ 0x5344 ],
			0x303A  => [ 0x5345 ],
			0x309F  => [ 0x3088, 0x308A ],
			0x30FF  => [ 0x30B3, 0x30C8 ],
			0x3131  => [ 0x1100 ],
			0x3132  => [ 0x1101 ],
			0x3133  => [ 0x11AA ],
			0x3134  => [ 0x1102 ],
			0x3135  => [ 0x11AC ],
			0x3136  => [ 0x11AD ],
			0x3137  => [ 0x1103 ],
			0x3138  => [ 0x1104 ],
			0x3139  => [ 0x1105 ],
			0x313A  => [ 0x11B0 ],
			0x313B  => [ 0x11B1 ],
			0x313C  => [ 0x11B2 ],
			0x313D  => [ 0x11B3 ],
			0x313E  => [ 0x11B4 ],
			0x313F  => [ 0x11B5 ],
			0x3140  => [ 0x111A ],
			0x3141  => [ 0x1106 ],
			0x3142  => [ 0x1107 ],
			0x3143  => [ 0x1108 ],
			0x3144  => [ 0x1121 ],
			0x3145  => [ 0x1109 ],
			0x3146  => [ 0x110A ],
			0x3147  => [ 0x110B ],
			0x3148  => [ 0x110C ],
			0x3149  => [ 0x110D ],
			0x314A  => [ 0x110E ],
			0x314B  => [ 0x110F ],
			0x314C  => [ 0x1110 ],
			0x314D  => [ 0x1111 ],
			0x314E  => [ 0x1112 ],
			0x314F  => [ 0x1161 ],
			0x3150  => [ 0x1162 ],
			0x3151  => [ 0x1163 ],
			0x3152  => [ 0x1164 ],
			0x3153  => [ 0x1165 ],
			0x3154  => [ 0x1166 ],
			0x3155  => [ 0x1167 ],
			0x3156  => [ 0x1168 ],
			0x3157  => [ 0x1169 ],
			0x3158  => [ 0x116A ],
			0x3159  => [ 0x116B ],
			0x315A  => [ 0x116C ],
			0x315B  => [ 0x116D ],
			0x315C  => [ 0x116E ],
			0x315D  => [ 0x116F ],
			0x315E  => [ 0x1170 ],
			0x315F  => [ 0x1171 ],
			0x3160  => [ 0x1172 ],
			0x3161  => [ 0x1173 ],
			0x3162  => [ 0x1174 ],
			0x3163  => [ 0x1175 ],
			0x3165  => [ 0x1114 ],
			0x3166  => [ 0x1115 ],
			0x3167  => [ 0x11C7 ],
			0x3168  => [ 0x11C8 ],
			0x3169  => [ 0x11CC ],
			0x316A  => [ 0x11CE ],
			0x316B  => [ 0x11D3 ],
			0x316C  => [ 0x11D7 ],
			0x316D  => [ 0x11D9 ],
			0x316E  => [ 0x111C ],
			0x316F  => [ 0x11DD ],
			0x3170  => [ 0x11DF ],
			0x3171  => [ 0x111D ],
			0x3172  => [ 0x111E ],
			0x3173  => [ 0x1120 ],
			0x3174  => [ 0x1122 ],
			0x3175  => [ 0x1123 ],
			0x3176  => [ 0x1127 ],
			0x3177  => [ 0x1129 ],
			0x3178  => [ 0x112B ],
			0x3179  => [ 0x112C ],
			0x317A  => [ 0x112D ],
			0x317B  => [ 0x112E ],
			0x317C  => [ 0x112F ],
			0x317D  => [ 0x1132 ],
			0x317E  => [ 0x1136 ],
			0x317F  => [ 0x1140 ],
			0x3180  => [ 0x1147 ],
			0x3181  => [ 0x114C ],
			0x3182  => [ 0x11F1 ],
			0x3183  => [ 0x11F2 ],
			0x3184  => [ 0x1157 ],
			0x3185  => [ 0x1158 ],
			0x3186  => [ 0x1159 ],
			0x3187  => [ 0x1184 ],
			0x3188  => [ 0x1185 ],
			0x3189  => [ 0x1188 ],
			0x318A  => [ 0x1191 ],
			0x318B  => [ 0x1192 ],
			0x318C  => [ 0x1194 ],
			0x318D  => [ 0x119E ],
			0x318E  => [ 0x11A1 ],
			0x3192  => [ 0x4E00 ],
			0x3193  => [ 0x4E8C ],
			0x3194  => [ 0x4E09 ],
			0x3195  => [ 0x56DB ],
			0x3196  => [ 0x4E0A ],
			0x3197  => [ 0x4E2D ],
			0x3198  => [ 0x4E0B ],
			0x3199  => [ 0x7532 ],
			0x319A  => [ 0x4E59 ],
			0x319B  => [ 0x4E19 ],
			0x319C  => [ 0x4E01 ],
			0x319D  => [ 0x5929 ],
			0x319E  => [ 0x5730 ],
			0x319F  => [ 0x4EBA ],
			0x3244  => [ 0x554F ],
			0x3245  => [ 0x5E7C ],
			0x3246  => [ 0x6587 ],
			0x3247  => [ 0x7B8F ],
			0x3250  => [ 0x70, 0x74, 0x65 ],
			0x3251  => [ 0x32, 0x31 ],
			0x3252  => [ 0x32, 0x32 ],
			0x3253  => [ 0x32, 0x33 ],
			0x3254  => [ 0x32, 0x34 ],
			0x3255  => [ 0x32, 0x35 ],
			0x3256  => [ 0x32, 0x36 ],
			0x3257  => [ 0x32, 0x37 ],
			0x3258  => [ 0x32, 0x38 ],
			0x3259  => [ 0x32, 0x39 ],
			0x325A  => [ 0x33, 0x30 ],
			0x325B  => [ 0x33, 0x31 ],
			0x325C  => [ 0x33, 0x32 ],
			0x325D  => [ 0x33, 0x33 ],
			0x325E  => [ 0x33, 0x34 ],
			0x325F  => [ 0x33, 0x35 ],
			0x3260  => [ 0x1100 ],
			0x3261  => [ 0x1102 ],
			0x3262  => [ 0x1103 ],
			0x3263  => [ 0x1105 ],
			0x3264  => [ 0x1106 ],
			0x3265  => [ 0x1107 ],
			0x3266  => [ 0x1109 ],
			0x3267  => [ 0x110B ],
			0x3268  => [ 0x110C ],
			0x3269  => [ 0x110E ],
			0x326A  => [ 0x110F ],
			0x326B  => [ 0x1110 ],
			0x326C  => [ 0x1111 ],
			0x326D  => [ 0x1112 ],
			0x326E  => [ 0xAC00 ],
			0x326F  => [ 0xB098 ],
			0x3270  => [ 0xB2E4 ],
			0x3271  => [ 0xB77C ],
			0x3272  => [ 0xB9C8 ],
			0x3273  => [ 0xBC14 ],
			0x3274  => [ 0xC0AC ],
			0x3275  => [ 0xC544 ],
			0x3276  => [ 0xC790 ],
			0x3277  => [ 0xCC28 ],
			0x3278  => [ 0xCE74 ],
			0x3279  => [ 0xD0C0 ],
			0x327A  => [ 0xD30C ],
			0x327B  => [ 0xD558 ],
			0x327C  => [ 0xCC38, 0xACE0 ],
			0x327D  => [ 0xC8FC, 0xC758 ],
			0x327E  => [ 0xC6B0 ],
			0x3280  => [ 0x4E00 ],
			0x3281  => [ 0x4E8C ],
			0x3282  => [ 0x4E09 ],
			0x3283  => [ 0x56DB ],
			0x3284  => [ 0x4E94 ],
			0x3285  => [ 0x516D ],
			0x3286  => [ 0x4E03 ],
			0x3287  => [ 0x516B ],
			0x3288  => [ 0x4E5D ],
			0x3289  => [ 0x5341 ],
			0x328A  => [ 0x6708 ],
			0x328B  => [ 0x706B ],
			0x328C  => [ 0x6C34 ],
			0x328D  => [ 0x6728 ],
			0x328E  => [ 0x91D1 ],
			0x328F  => [ 0x571F ],
			0x3290  => [ 0x65E5 ],
			0x3291  => [ 0x682A ],
			0x3292  => [ 0x6709 ],
			0x3293  => [ 0x793E ],
			0x3294  => [ 0x540D ],
			0x3295  => [ 0x7279 ],
			0x3296  => [ 0x8CA1 ],
			0x3297  => [ 0x795D ],
			0x3298  => [ 0x52B4 ],
			0x3299  => [ 0x79D8 ],
			0x329A  => [ 0x7537 ],
			0x329B  => [ 0x5973 ],
			0x329C  => [ 0x9069 ],
			0x329D  => [ 0x512A ],
			0x329E  => [ 0x5370 ],
			0x329F  => [ 0x6CE8 ],
			0x32A0  => [ 0x9805 ],
			0x32A1  => [ 0x4F11 ],
			0x32A2  => [ 0x5199 ],
			0x32A3  => [ 0x6B63 ],
			0x32A4  => [ 0x4E0A ],
			0x32A5  => [ 0x4E2D ],
			0x32A6  => [ 0x4E0B ],
			0x32A7  => [ 0x5DE6 ],
			0x32A8  => [ 0x53F3 ],
			0x32A9  => [ 0x533B ],
			0x32AA  => [ 0x5B97 ],
			0x32AB  => [ 0x5B66 ],
			0x32AC  => [ 0x76E3 ],
			0x32AD  => [ 0x4F01 ],
			0x32AE  => [ 0x8CC7 ],
			0x32AF  => [ 0x5354 ],
			0x32B0  => [ 0x591C ],
			0x32B1  => [ 0x33, 0x36 ],
			0x32B2  => [ 0x33, 0x37 ],
			0x32B3  => [ 0x33, 0x38 ],
			0x32B4  => [ 0x33, 0x39 ],
			0x32B5  => [ 0x34, 0x30 ],
			0x32B6  => [ 0x34, 0x31 ],
			0x32B7  => [ 0x34, 0x32 ],
			0x32B8  => [ 0x34, 0x33 ],
			0x32B9  => [ 0x34, 0x34 ],
			0x32BA  => [ 0x34, 0x35 ],
			0x32BB  => [ 0x34, 0x36 ],
			0x32BC  => [ 0x34, 0x37 ],
			0x32BD  => [ 0x34, 0x38 ],
			0x32BE  => [ 0x34, 0x39 ],
			0x32BF  => [ 0x35, 0x30 ],
			0x32C0  => [ 0x31, 0x6708 ],
			0x32C1  => [ 0x32, 0x6708 ],
			0x32C2  => [ 0x33, 0x6708 ],
			0x32C3  => [ 0x34, 0x6708 ],
			0x32C4  => [ 0x35, 0x6708 ],
			0x32C5  => [ 0x36, 0x6708 ],
			0x32C6  => [ 0x37, 0x6708 ],
			0x32C7  => [ 0x38, 0x6708 ],
			0x32C8  => [ 0x39, 0x6708 ],
			0x32C9  => [ 0x31, 0x30, 0x6708 ],
			0x32CA  => [ 0x31, 0x31, 0x6708 ],
			0x32CB  => [ 0x31, 0x32, 0x6708 ],
			0x32CC  => [ 0x68, 0x67 ],
			0x32CD  => [ 0x65, 0x72, 0x67 ],
			0x32CE  => [ 0x65, 0x76 ],
			0x32CF  => [ 0x6C, 0x74, 0x64 ],
			0x32D0  => [ 0x30A2 ],
			0x32D1  => [ 0x30A4 ],
			0x32D2  => [ 0x30A6 ],
			0x32D3  => [ 0x30A8 ],
			0x32D4  => [ 0x30AA ],
			0x32D5  => [ 0x30AB ],
			0x32D6  => [ 0x30AD ],
			0x32D7  => [ 0x30AF ],
			0x32D8  => [ 0x30B1 ],
			0x32D9  => [ 0x30B3 ],
			0x32DA  => [ 0x30B5 ],
			0x32DB  => [ 0x30B7 ],
			0x32DC  => [ 0x30B9 ],
			0x32DD  => [ 0x30BB ],
			0x32DE  => [ 0x30BD ],
			0x32DF  => [ 0x30BF ],
			0x32E0  => [ 0x30C1 ],
			0x32E1  => [ 0x30C4 ],
			0x32E2  => [ 0x30C6 ],
			0x32E3  => [ 0x30C8 ],
			0x32E4  => [ 0x30CA ],
			0x32E5  => [ 0x30CB ],
			0x32E6  => [ 0x30CC ],
			0x32E7  => [ 0x30CD ],
			0x32E8  => [ 0x30CE ],
			0x32E9  => [ 0x30CF ],
			0x32EA  => [ 0x30D2 ],
			0x32EB  => [ 0x30D5 ],
			0x32EC  => [ 0x30D8 ],
			0x32ED  => [ 0x30DB ],
			0x32EE  => [ 0x30DE ],
			0x32EF  => [ 0x30DF ],
			0x32F0  => [ 0x30E0 ],
			0x32F1  => [ 0x30E1 ],
			0x32F2  => [ 0x30E2 ],
			0x32F3  => [ 0x30E4 ],
			0x32F4  => [ 0x30E6 ],
			0x32F5  => [ 0x30E8 ],
			0x32F6  => [ 0x30E9 ],
			0x32F7  => [ 0x30EA ],
			0x32F8  => [ 0x30EB ],
			0x32F9  => [ 0x30EC ],
			0x32FA  => [ 0x30ED ],
			0x32FB  => [ 0x30EF ],
			0x32FC  => [ 0x30F0 ],
			0x32FD  => [ 0x30F1 ],
			0x32FE  => [ 0x30F2 ],
			0x3300  => [ 0x30A2, 0x30D1, 0x30FC, 0x30C8 ],
			0x3301  => [ 0x30A2, 0x30EB, 0x30D5, 0x30A1 ],
			0x3302  => [ 0x30A2, 0x30F3, 0x30DA, 0x30A2 ],
			0x3303  => [ 0x30A2, 0x30FC, 0x30EB ],
			0x3304  => [ 0x30A4, 0x30CB, 0x30F3, 0x30B0 ],
			0x3305  => [ 0x30A4, 0x30F3, 0x30C1 ],
			0x3306  => [ 0x30A6, 0x30A9, 0x30F3 ],
			0x3307  => [ 0x30A8, 0x30B9, 0x30AF, 0x30FC, 0x30C9 ],
			0x3308  => [ 0x30A8, 0x30FC, 0x30AB, 0x30FC ],
			0x3309  => [ 0x30AA, 0x30F3, 0x30B9 ],
			0x330A  => [ 0x30AA, 0x30FC, 0x30E0 ],
			0x330B  => [ 0x30AB, 0x30A4, 0x30EA ],
			0x330C  => [ 0x30AB, 0x30E9, 0x30C3, 0x30C8 ],
			0x330D  => [ 0x30AB, 0x30ED, 0x30EA, 0x30FC ],
			0x330E  => [ 0x30AC, 0x30ED, 0x30F3 ],
			0x330F  => [ 0x30AC, 0x30F3, 0x30DE ],
			0x3310  => [ 0x30AE, 0x30AC ],
			0x3311  => [ 0x30AE, 0x30CB, 0x30FC ],
			0x3312  => [ 0x30AD, 0x30E5, 0x30EA, 0x30FC ],
			0x3313  => [ 0x30AE, 0x30EB, 0x30C0, 0x30FC ],
			0x3314  => [ 0x30AD, 0x30ED ],
			0x3315  => [ 0x30AD, 0x30ED, 0x30B0, 0x30E9, 0x30E0 ],
			0x3316  => [ 0x30AD, 0x30ED, 0x30E1, 0x30FC, 0x30C8, 0x30EB ],
			0x3317  => [ 0x30AD, 0x30ED, 0x30EF, 0x30C3, 0x30C8 ],
			0x3318  => [ 0x30B0, 0x30E9, 0x30E0 ],
			0x3319  => [ 0x30B0, 0x30E9, 0x30E0, 0x30C8, 0x30F3 ],
			0x331A  => [ 0x30AF, 0x30EB, 0x30BC, 0x30A4, 0x30ED ],
			0x331B  => [ 0x30AF, 0x30ED, 0x30FC, 0x30CD ],
			0x331C  => [ 0x30B1, 0x30FC, 0x30B9 ],
			0x331D  => [ 0x30B3, 0x30EB, 0x30CA ],
			0x331E  => [ 0x30B3, 0x30FC, 0x30DD ],
			0x331F  => [ 0x30B5, 0x30A4, 0x30AF, 0x30EB ],
			0x3320  => [ 0x30B5, 0x30F3, 0x30C1, 0x30FC, 0x30E0 ],
			0x3321  => [ 0x30B7, 0x30EA, 0x30F3, 0x30B0 ],
			0x3322  => [ 0x30BB, 0x30F3, 0x30C1 ],
			0x3323  => [ 0x30BB, 0x30F3, 0x30C8 ],
			0x3324  => [ 0x30C0, 0x30FC, 0x30B9 ],
			0x3325  => [ 0x30C7, 0x30B7 ],
			0x3326  => [ 0x30C9, 0x30EB ],
			0x3327  => [ 0x30C8, 0x30F3 ],
			0x3328  => [ 0x30CA, 0x30CE ],
			0x3329  => [ 0x30CE, 0x30C3, 0x30C8 ],
			0x332A  => [ 0x30CF, 0x30A4, 0x30C4 ],
			0x332B  => [ 0x30D1, 0x30FC, 0x30BB, 0x30F3, 0x30C8 ],
			0x332C  => [ 0x30D1, 0x30FC, 0x30C4 ],
			0x332D  => [ 0x30D0, 0x30FC, 0x30EC, 0x30EB ],
			0x332E  => [ 0x30D4, 0x30A2, 0x30B9, 0x30C8, 0x30EB ],
			0x332F  => [ 0x30D4, 0x30AF, 0x30EB ],
			0x3330  => [ 0x30D4, 0x30B3 ],
			0x3331  => [ 0x30D3, 0x30EB ],
			0x3332  => [ 0x30D5, 0x30A1, 0x30E9, 0x30C3, 0x30C9 ],
			0x3333  => [ 0x30D5, 0x30A3, 0x30FC, 0x30C8 ],
			0x3334  => [ 0x30D6, 0x30C3, 0x30B7, 0x30A7, 0x30EB ],
			0x3335  => [ 0x30D5, 0x30E9, 0x30F3 ],
			0x3336  => [ 0x30D8, 0x30AF, 0x30BF, 0x30FC, 0x30EB ],
			0x3337  => [ 0x30DA, 0x30BD ],
			0x3338  => [ 0x30DA, 0x30CB, 0x30D2 ],
			0x3339  => [ 0x30D8, 0x30EB, 0x30C4 ],
			0x333A  => [ 0x30DA, 0x30F3, 0x30B9 ],
			0x333B  => [ 0x30DA, 0x30FC, 0x30B8 ],
			0x333C  => [ 0x30D9, 0x30FC, 0x30BF ],
			0x333D  => [ 0x30DD, 0x30A4, 0x30F3, 0x30C8 ],
			0x333E  => [ 0x30DC, 0x30EB, 0x30C8 ],
			0x333F  => [ 0x30DB, 0x30F3 ],
			0x3340  => [ 0x30DD, 0x30F3, 0x30C9 ],
			0x3341  => [ 0x30DB, 0x30FC, 0x30EB ],
			0x3342  => [ 0x30DB, 0x30FC, 0x30F3 ],
			0x3343  => [ 0x30DE, 0x30A4, 0x30AF, 0x30ED ],
			0x3344  => [ 0x30DE, 0x30A4, 0x30EB ],
			0x3345  => [ 0x30DE, 0x30C3, 0x30CF ],
			0x3346  => [ 0x30DE, 0x30EB, 0x30AF ],
			0x3347  => [ 0x30DE, 0x30F3, 0x30B7, 0x30E7, 0x30F3 ],
			0x3348  => [ 0x30DF, 0x30AF, 0x30ED, 0x30F3 ],
			0x3349  => [ 0x30DF, 0x30EA ],
			0x334A  => [ 0x30DF, 0x30EA, 0x30D0, 0x30FC, 0x30EB ],
			0x334B  => [ 0x30E1, 0x30AC ],
			0x334C  => [ 0x30E1, 0x30AC, 0x30C8, 0x30F3 ],
			0x334D  => [ 0x30E1, 0x30FC, 0x30C8, 0x30EB ],
			0x334E  => [ 0x30E4, 0x30FC, 0x30C9 ],
			0x334F  => [ 0x30E4, 0x30FC, 0x30EB ],
			0x3350  => [ 0x30E6, 0x30A2, 0x30F3 ],
			0x3351  => [ 0x30EA, 0x30C3, 0x30C8, 0x30EB ],
			0x3352  => [ 0x30EA, 0x30E9 ],
			0x3353  => [ 0x30EB, 0x30D4, 0x30FC ],
			0x3354  => [ 0x30EB, 0x30FC, 0x30D6, 0x30EB ],
			0x3355  => [ 0x30EC, 0x30E0 ],
			0x3356  => [ 0x30EC, 0x30F3, 0x30C8, 0x30B2, 0x30F3 ],
			0x3357  => [ 0x30EF, 0x30C3, 0x30C8 ],
			0x3358  => [ 0x30, 0x70B9 ],
			0x3359  => [ 0x31, 0x70B9 ],
			0x335A  => [ 0x32, 0x70B9 ],
			0x335B  => [ 0x33, 0x70B9 ],
			0x335C  => [ 0x34, 0x70B9 ],
			0x335D  => [ 0x35, 0x70B9 ],
			0x335E  => [ 0x36, 0x70B9 ],
			0x335F  => [ 0x37, 0x70B9 ],
			0x3360  => [ 0x38, 0x70B9 ],
			0x3361  => [ 0x39, 0x70B9 ],
			0x3362  => [ 0x31, 0x30, 0x70B9 ],
			0x3363  => [ 0x31, 0x31, 0x70B9 ],
			0x3364  => [ 0x31, 0x32, 0x70B9 ],
			0x3365  => [ 0x31, 0x33, 0x70B9 ],
			0x3366  => [ 0x31, 0x34, 0x70B9 ],
			0x3367  => [ 0x31, 0x35, 0x70B9 ],
			0x3368  => [ 0x31, 0x36, 0x70B9 ],
			0x3369  => [ 0x31, 0x37, 0x70B9 ],
			0x336A  => [ 0x31, 0x38, 0x70B9 ],
			0x336B  => [ 0x31, 0x39, 0x70B9 ],
			0x336C  => [ 0x32, 0x30, 0x70B9 ],
			0x336D  => [ 0x32, 0x31, 0x70B9 ],
			0x336E  => [ 0x32, 0x32, 0x70B9 ],
			0x336F  => [ 0x32, 0x33, 0x70B9 ],
			0x3370  => [ 0x32, 0x34, 0x70B9 ],
			0x3371  => [ 0x68, 0x70, 0x61 ],
			0x3372  => [ 0x64, 0x61 ],
			0x3373  => [ 0x61, 0x75 ],
			0x3374  => [ 0x62, 0x61, 0x72 ],
			0x3375  => [ 0x6F, 0x76 ],
			0x3376  => [ 0x70, 0x63 ],
			0x3377  => [ 0x64, 0x6D ],
			0x3378  => [ 0x64, 0x6D, 0x32 ],
			0x3379  => [ 0x64, 0x6D, 0x33 ],
			0x337A  => [ 0x69, 0x75 ],
			0x337B  => [ 0x5E73, 0x6210 ],
			0x337C  => [ 0x662D, 0x548C ],
			0x337D  => [ 0x5927, 0x6B63 ],
			0x337E  => [ 0x660E, 0x6CBB ],
			0x337F  => [ 0x682A, 0x5F0F, 0x4F1A, 0x793E ],
			0x3380  => [ 0x70, 0x61 ],
			0x3381  => [ 0x6E, 0x61 ],
			0x3382  => [ 0x3BC, 0x61 ],
			0x3383  => [ 0x6D, 0x61 ],
			0x3384  => [ 0x6B, 0x61 ],
			0x3385  => [ 0x6B, 0x62 ],
			0x3386  => [ 0x6D, 0x62 ],
			0x3387  => [ 0x67, 0x62 ],
			0x3388  => [ 0x63, 0x61, 0x6C ],
			0x3389  => [ 0x6B, 0x63, 0x61, 0x6C ],
			0x338A  => [ 0x70, 0x66 ],
			0x338B  => [ 0x6E, 0x66 ],
			0x338C  => [ 0x3BC, 0x66 ],
			0x338D  => [ 0x3BC, 0x67 ],
			0x338E  => [ 0x6D, 0x67 ],
			0x338F  => [ 0x6B, 0x67 ],
			0x3390  => [ 0x68, 0x7A ],
			0x3391  => [ 0x6B, 0x68, 0x7A ],
			0x3392  => [ 0x6D, 0x68, 0x7A ],
			0x3393  => [ 0x67, 0x68, 0x7A ],
			0x3394  => [ 0x74, 0x68, 0x7A ],
			0x3395  => [ 0x3BC, 0x6C ],
			0x3396  => [ 0x6D, 0x6C ],
			0x3397  => [ 0x64, 0x6C ],
			0x3398  => [ 0x6B, 0x6C ],
			0x3399  => [ 0x66, 0x6D ],
			0x339A  => [ 0x6E, 0x6D ],
			0x339B  => [ 0x3BC, 0x6D ],
			0x339C  => [ 0x6D, 0x6D ],
			0x339D  => [ 0x63, 0x6D ],
			0x339E  => [ 0x6B, 0x6D ],
			0x339F  => [ 0x6D, 0x6D, 0x32 ],
			0x33A0  => [ 0x63, 0x6D, 0x32 ],
			0x33A1  => [ 0x6D, 0x32 ],
			0x33A2  => [ 0x6B, 0x6D, 0x32 ],
			0x33A3  => [ 0x6D, 0x6D, 0x33 ],
			0x33A4  => [ 0x63, 0x6D, 0x33 ],
			0x33A5  => [ 0x6D, 0x33 ],
			0x33A6  => [ 0x6B, 0x6D, 0x33 ],
			0x33A7  => [ 0x6D, 0x2215, 0x73 ],
			0x33A8  => [ 0x6D, 0x2215, 0x73, 0x32 ],
			0x33A9  => [ 0x70, 0x61 ],
			0x33AA  => [ 0x6B, 0x70, 0x61 ],
			0x33AB  => [ 0x6D, 0x70, 0x61 ],
			0x33AC  => [ 0x67, 0x70, 0x61 ],
			0x33AD  => [ 0x72, 0x61, 0x64 ],
			0x33AE  => [ 0x72, 0x61, 0x64, 0x2215, 0x73 ],
			0x33AF  => [ 0x72, 0x61, 0x64, 0x2215, 0x73, 0x32 ],
			0x33B0  => [ 0x70, 0x73 ],
			0x33B1  => [ 0x6E, 0x73 ],
			0x33B2  => [ 0x3BC, 0x73 ],
			0x33B3  => [ 0x6D, 0x73 ],
			0x33B4  => [ 0x70, 0x76 ],
			0x33B5  => [ 0x6E, 0x76 ],
			0x33B6  => [ 0x3BC, 0x76 ],
			0x33B7  => [ 0x6D, 0x76 ],
			0x33B8  => [ 0x6B, 0x76 ],
			0x33B9  => [ 0x6D, 0x76 ],
			0x33BA  => [ 0x70, 0x77 ],
			0x33BB  => [ 0x6E, 0x77 ],
			0x33BC  => [ 0x3BC, 0x77 ],
			0x33BD  => [ 0x6D, 0x77 ],
			0x33BE  => [ 0x6B, 0x77 ],
			0x33BF  => [ 0x6D, 0x77 ],
			0x33C0  => [ 0x6B, 0x3C9 ],
			0x33C1  => [ 0x6D, 0x3C9 ],
			0x33C3  => [ 0x62, 0x71 ],
			0x33C4  => [ 0x63, 0x63 ],
			0x33C5  => [ 0x63, 0x64 ],
			0x33C6  => [ 0x63, 0x2215, 0x6B, 0x67 ],
			0x33C8  => [ 0x64, 0x62 ],
			0x33C9  => [ 0x67, 0x79 ],
			0x33CA  => [ 0x68, 0x61 ],
			0x33CB  => [ 0x68, 0x70 ],
			0x33CC  => [ 0x69, 0x6E ],
			0x33CD  => [ 0x6B, 0x6B ],
			0x33CE  => [ 0x6B, 0x6D ],
			0x33CF  => [ 0x6B, 0x74 ],
			0x33D0  => [ 0x6C, 0x6D ],
			0x33D1  => [ 0x6C, 0x6E ],
			0x33D2  => [ 0x6C, 0x6F, 0x67 ],
			0x33D3  => [ 0x6C, 0x78 ],
			0x33D4  => [ 0x6D, 0x62 ],
			0x33D5  => [ 0x6D, 0x69, 0x6C ],
			0x33D6  => [ 0x6D, 0x6F, 0x6C ],
			0x33D7  => [ 0x70, 0x68 ],
			0x33D9  => [ 0x70, 0x70, 0x6D ],
			0x33DA  => [ 0x70, 0x72 ],
			0x33DB  => [ 0x73, 0x72 ],
			0x33DC  => [ 0x73, 0x76 ],
			0x33DD  => [ 0x77, 0x62 ],
			0x33DE  => [ 0x76, 0x2215, 0x6D ],
			0x33DF  => [ 0x61, 0x2215, 0x6D ],
			0x33E0  => [ 0x31, 0x65E5 ],
			0x33E1  => [ 0x32, 0x65E5 ],
			0x33E2  => [ 0x33, 0x65E5 ],
			0x33E3  => [ 0x34, 0x65E5 ],
			0x33E4  => [ 0x35, 0x65E5 ],
			0x33E5  => [ 0x36, 0x65E5 ],
			0x33E6  => [ 0x37, 0x65E5 ],
			0x33E7  => [ 0x38, 0x65E5 ],
			0x33E8  => [ 0x39, 0x65E5 ],
			0x33E9  => [ 0x31, 0x30, 0x65E5 ],
			0x33EA  => [ 0x31, 0x31, 0x65E5 ],
			0x33EB  => [ 0x31, 0x32, 0x65E5 ],
			0x33EC  => [ 0x31, 0x33, 0x65E5 ],
			0x33ED  => [ 0x31, 0x34, 0x65E5 ],
			0x33EE  => [ 0x31, 0x35, 0x65E5 ],
			0x33EF  => [ 0x31, 0x36, 0x65E5 ],
			0x33F0  => [ 0x31, 0x37, 0x65E5 ],
			0x33F1  => [ 0x31, 0x38, 0x65E5 ],
			0x33F2  => [ 0x31, 0x39, 0x65E5 ],
			0x33F3  => [ 0x32, 0x30, 0x65E5 ],
			0x33F4  => [ 0x32, 0x31, 0x65E5 ],
			0x33F5  => [ 0x32, 0x32, 0x65E5 ],
			0x33F6  => [ 0x32, 0x33, 0x65E5 ],
			0x33F7  => [ 0x32, 0x34, 0x65E5 ],
			0x33F8  => [ 0x32, 0x35, 0x65E5 ],
			0x33F9  => [ 0x32, 0x36, 0x65E5 ],
			0x33FA  => [ 0x32, 0x37, 0x65E5 ],
			0x33FB  => [ 0x32, 0x38, 0x65E5 ],
			0x33FC  => [ 0x32, 0x39, 0x65E5 ],
			0x33FD  => [ 0x33, 0x30, 0x65E5 ],
			0x33FE  => [ 0x33, 0x31, 0x65E5 ],
			0x33FF  => [ 0x67, 0x61, 0x6C ],
			0xA640  => [ 0xA641 ],
			0xA642  => [ 0xA643 ],
			0xA644  => [ 0xA645 ],
			0xA646  => [ 0xA647 ],
			0xA648  => [ 0xA649 ],
			0xA64A  => [ 0xA64B ],
			0xA64C  => [ 0xA64D ],
			0xA64E  => [ 0xA64F ],
			0xA650  => [ 0xA651 ],
			0xA652  => [ 0xA653 ],
			0xA654  => [ 0xA655 ],
			0xA656  => [ 0xA657 ],
			0xA658  => [ 0xA659 ],
			0xA65A  => [ 0xA65B ],
			0xA65C  => [ 0xA65D ],
			0xA65E  => [ 0xA65F ],
			0xA660  => [ 0xA661 ],
			0xA662  => [ 0xA663 ],
			0xA664  => [ 0xA665 ],
			0xA666  => [ 0xA667 ],
			0xA668  => [ 0xA669 ],
			0xA66A  => [ 0xA66B ],
			0xA66C  => [ 0xA66D ],
			0xA680  => [ 0xA681 ],
			0xA682  => [ 0xA683 ],
			0xA684  => [ 0xA685 ],
			0xA686  => [ 0xA687 ],
			0xA688  => [ 0xA689 ],
			0xA68A  => [ 0xA68B ],
			0xA68C  => [ 0xA68D ],
			0xA68E  => [ 0xA68F ],
			0xA690  => [ 0xA691 ],
			0xA692  => [ 0xA693 ],
			0xA694  => [ 0xA695 ],
			0xA696  => [ 0xA697 ],
			0xA698  => [ 0xA699 ],
			0xA69A  => [ 0xA69B ],
			0xA69C  => [ 0x44A ],
			0xA69D  => [ 0x44C ],
			0xA722  => [ 0xA723 ],
			0xA724  => [ 0xA725 ],
			0xA726  => [ 0xA727 ],
			0xA728  => [ 0xA729 ],
			0xA72A  => [ 0xA72B ],
			0xA72C  => [ 0xA72D ],
			0xA72E  => [ 0xA72F ],
			0xA732  => [ 0xA733 ],
			0xA734  => [ 0xA735 ],
			0xA736  => [ 0xA737 ],
			0xA738  => [ 0xA739 ],
			0xA73A  => [ 0xA73B ],
			0xA73C  => [ 0xA73D ],
			0xA73E  => [ 0xA73F ],
			0xA740  => [ 0xA741 ],
			0xA742  => [ 0xA743 ],
			0xA744  => [ 0xA745 ],
			0xA746  => [ 0xA747 ],
			0xA748  => [ 0xA749 ],
			0xA74A  => [ 0xA74B ],
			0xA74C  => [ 0xA74D ],
			0xA74E  => [ 0xA74F ],
			0xA750  => [ 0xA751 ],
			0xA752  => [ 0xA753 ],
			0xA754  => [ 0xA755 ],
			0xA756  => [ 0xA757 ],
			0xA758  => [ 0xA759 ],
			0xA75A  => [ 0xA75B ],
			0xA75C  => [ 0xA75D ],
			0xA75E  => [ 0xA75F ],
			0xA760  => [ 0xA761 ],
			0xA762  => [ 0xA763 ],
			0xA764  => [ 0xA765 ],
			0xA766  => [ 0xA767 ],
			0xA768  => [ 0xA769 ],
			0xA76A  => [ 0xA76B ],
			0xA76C  => [ 0xA76D ],
			0xA76E  => [ 0xA76F ],
			0xA770  => [ 0xA76F ],
			0xA779  => [ 0xA77A ],
			0xA77B  => [ 0xA77C ],
			0xA77D  => [ 0x1D79 ],
			0xA77E  => [ 0xA77F ],
			0xA780  => [ 0xA781 ],
			0xA782  => [ 0xA783 ],
			0xA784  => [ 0xA785 ],
			0xA786  => [ 0xA787 ],
			0xA78B  => [ 0xA78C ],
			0xA78D  => [ 0x265 ],
			0xA790  => [ 0xA791 ],
			0xA792  => [ 0xA793 ],
			0xA796  => [ 0xA797 ],
			0xA798  => [ 0xA799 ],
			0xA79A  => [ 0xA79B ],
			0xA79C  => [ 0xA79D ],
			0xA79E  => [ 0xA79F ],
			0xA7A0  => [ 0xA7A1 ],
			0xA7A2  => [ 0xA7A3 ],
			0xA7A4  => [ 0xA7A5 ],
			0xA7A6  => [ 0xA7A7 ],
			0xA7A8  => [ 0xA7A9 ],
			0xA7AA  => [ 0x266 ],
			0xA7AB  => [ 0x25C ],
			0xA7AC  => [ 0x261 ],
			0xA7AD  => [ 0x26C ],
			0xA7B0  => [ 0x29E ],
			0xA7B1  => [ 0x287 ],
			0xA7F8  => [ 0x127 ],
			0xA7F9  => [ 0x153 ],
			0xAB5C  => [ 0xA727 ],
			0xAB5D  => [ 0xAB37 ],
			0xAB5E  => [ 0x26B ],
			0xAB5F  => [ 0xAB52 ],
			0xF900  => [ 0x8C48 ],
			0xF901  => [ 0x66F4 ],
			0xF902  => [ 0x8ECA ],
			0xF903  => [ 0x8CC8 ],
			0xF904  => [ 0x6ED1 ],
			0xF905  => [ 0x4E32 ],
			0xF906  => [ 0x53E5 ],
			0xF907  => [ 0x9F9C ],
			0xF908  => [ 0x9F9C ],
			0xF909  => [ 0x5951 ],
			0xF90A  => [ 0x91D1 ],
			0xF90B  => [ 0x5587 ],
			0xF90C  => [ 0x5948 ],
			0xF90D  => [ 0x61F6 ],
			0xF90E  => [ 0x7669 ],
			0xF90F  => [ 0x7F85 ],
			0xF910  => [ 0x863F ],
			0xF911  => [ 0x87BA ],
			0xF912  => [ 0x88F8 ],
			0xF913  => [ 0x908F ],
			0xF914  => [ 0x6A02 ],
			0xF915  => [ 0x6D1B ],
			0xF916  => [ 0x70D9 ],
			0xF917  => [ 0x73DE ],
			0xF918  => [ 0x843D ],
			0xF919  => [ 0x916A ],
			0xF91A  => [ 0x99F1 ],
			0xF91B  => [ 0x4E82 ],
			0xF91C  => [ 0x5375 ],
			0xF91D  => [ 0x6B04 ],
			0xF91E  => [ 0x721B ],
			0xF91F  => [ 0x862D ],
			0xF920  => [ 0x9E1E ],
			0xF921  => [ 0x5D50 ],
			0xF922  => [ 0x6FEB ],
			0xF923  => [ 0x85CD ],
			0xF924  => [ 0x8964 ],
			0xF925  => [ 0x62C9 ],
			0xF926  => [ 0x81D8 ],
			0xF927  => [ 0x881F ],
			0xF928  => [ 0x5ECA ],
			0xF929  => [ 0x6717 ],
			0xF92A  => [ 0x6D6A ],
			0xF92B  => [ 0x72FC ],
			0xF92C  => [ 0x90CE ],
			0xF92D  => [ 0x4F86 ],
			0xF92E  => [ 0x51B7 ],
			0xF92F  => [ 0x52DE ],
			0xF930  => [ 0x64C4 ],
			0xF931  => [ 0x6AD3 ],
			0xF932  => [ 0x7210 ],
			0xF933  => [ 0x76E7 ],
			0xF934  => [ 0x8001 ],
			0xF935  => [ 0x8606 ],
			0xF936  => [ 0x865C ],
			0xF937  => [ 0x8DEF ],
			0xF938  => [ 0x9732 ],
			0xF939  => [ 0x9B6F ],
			0xF93A  => [ 0x9DFA ],
			0xF93B  => [ 0x788C ],
			0xF93C  => [ 0x797F ],
			0xF93D  => [ 0x7DA0 ],
			0xF93E  => [ 0x83C9 ],
			0xF93F  => [ 0x9304 ],
			0xF940  => [ 0x9E7F ],
			0xF941  => [ 0x8AD6 ],
			0xF942  => [ 0x58DF ],
			0xF943  => [ 0x5F04 ],
			0xF944  => [ 0x7C60 ],
			0xF945  => [ 0x807E ],
			0xF946  => [ 0x7262 ],
			0xF947  => [ 0x78CA ],
			0xF948  => [ 0x8CC2 ],
			0xF949  => [ 0x96F7 ],
			0xF94A  => [ 0x58D8 ],
			0xF94B  => [ 0x5C62 ],
			0xF94C  => [ 0x6A13 ],
			0xF94D  => [ 0x6DDA ],
			0xF94E  => [ 0x6F0F ],
			0xF94F  => [ 0x7D2F ],
			0xF950  => [ 0x7E37 ],
			0xF951  => [ 0x964B ],
			0xF952  => [ 0x52D2 ],
			0xF953  => [ 0x808B ],
			0xF954  => [ 0x51DC ],
			0xF955  => [ 0x51CC ],
			0xF956  => [ 0x7A1C ],
			0xF957  => [ 0x7DBE ],
			0xF958  => [ 0x83F1 ],
			0xF959  => [ 0x9675 ],
			0xF95A  => [ 0x8B80 ],
			0xF95B  => [ 0x62CF ],
			0xF95C  => [ 0x6A02 ],
			0xF95D  => [ 0x8AFE ],
			0xF95E  => [ 0x4E39 ],
			0xF95F  => [ 0x5BE7 ],
			0xF960  => [ 0x6012 ],
			0xF961  => [ 0x7387 ],
			0xF962  => [ 0x7570 ],
			0xF963  => [ 0x5317 ],
			0xF964  => [ 0x78FB ],
			0xF965  => [ 0x4FBF ],
			0xF966  => [ 0x5FA9 ],
			0xF967  => [ 0x4E0D ],
			0xF968  => [ 0x6CCC ],
			0xF969  => [ 0x6578 ],
			0xF96A  => [ 0x7D22 ],
			0xF96B  => [ 0x53C3 ],
			0xF96C  => [ 0x585E ],
			0xF96D  => [ 0x7701 ],
			0xF96E  => [ 0x8449 ],
			0xF96F  => [ 0x8AAA ],
			0xF970  => [ 0x6BBA ],
			0xF971  => [ 0x8FB0 ],
			0xF972  => [ 0x6C88 ],
			0xF973  => [ 0x62FE ],
			0xF974  => [ 0x82E5 ],
			0xF975  => [ 0x63A0 ],
			0xF976  => [ 0x7565 ],
			0xF977  => [ 0x4EAE ],
			0xF978  => [ 0x5169 ],
			0xF979  => [ 0x51C9 ],
			0xF97A  => [ 0x6881 ],
			0xF97B  => [ 0x7CE7 ],
			0xF97C  => [ 0x826F ],
			0xF97D  => [ 0x8AD2 ],
			0xF97E  => [ 0x91CF ],
			0xF97F  => [ 0x52F5 ],
			0xF980  => [ 0x5442 ],
			0xF981  => [ 0x5973 ],
			0xF982  => [ 0x5EEC ],
			0xF983  => [ 0x65C5 ],
			0xF984  => [ 0x6FFE ],
			0xF985  => [ 0x792A ],
			0xF986  => [ 0x95AD ],
			0xF987  => [ 0x9A6A ],
			0xF988  => [ 0x9E97 ],
			0xF989  => [ 0x9ECE ],
			0xF98A  => [ 0x529B ],
			0xF98B  => [ 0x66C6 ],
			0xF98C  => [ 0x6B77 ],
			0xF98D  => [ 0x8F62 ],
			0xF98E  => [ 0x5E74 ],
			0xF98F  => [ 0x6190 ],
			0xF990  => [ 0x6200 ],
			0xF991  => [ 0x649A ],
			0xF992  => [ 0x6F23 ],
			0xF993  => [ 0x7149 ],
			0xF994  => [ 0x7489 ],
			0xF995  => [ 0x79CA ],
			0xF996  => [ 0x7DF4 ],
			0xF997  => [ 0x806F ],
			0xF998  => [ 0x8F26 ],
			0xF999  => [ 0x84EE ],
			0xF99A  => [ 0x9023 ],
			0xF99B  => [ 0x934A ],
			0xF99C  => [ 0x5217 ],
			0xF99D  => [ 0x52A3 ],
			0xF99E  => [ 0x54BD ],
			0xF99F  => [ 0x70C8 ],
			0xF9A0  => [ 0x88C2 ],
			0xF9A1  => [ 0x8AAA ],
			0xF9A2  => [ 0x5EC9 ],
			0xF9A3  => [ 0x5FF5 ],
			0xF9A4  => [ 0x637B ],
			0xF9A5  => [ 0x6BAE ],
			0xF9A6  => [ 0x7C3E ],
			0xF9A7  => [ 0x7375 ],
			0xF9A8  => [ 0x4EE4 ],
			0xF9A9  => [ 0x56F9 ],
			0xF9AA  => [ 0x5BE7 ],
			0xF9AB  => [ 0x5DBA ],
			0xF9AC  => [ 0x601C ],
			0xF9AD  => [ 0x73B2 ],
			0xF9AE  => [ 0x7469 ],
			0xF9AF  => [ 0x7F9A ],
			0xF9B0  => [ 0x8046 ],
			0xF9B1  => [ 0x9234 ],
			0xF9B2  => [ 0x96F6 ],
			0xF9B3  => [ 0x9748 ],
			0xF9B4  => [ 0x9818 ],
			0xF9B5  => [ 0x4F8B ],
			0xF9B6  => [ 0x79AE ],
			0xF9B7  => [ 0x91B4 ],
			0xF9B8  => [ 0x96B8 ],
			0xF9B9  => [ 0x60E1 ],
			0xF9BA  => [ 0x4E86 ],
			0xF9BB  => [ 0x50DA ],
			0xF9BC  => [ 0x5BEE ],
			0xF9BD  => [ 0x5C3F ],
			0xF9BE  => [ 0x6599 ],
			0xF9BF  => [ 0x6A02 ],
			0xF9C0  => [ 0x71CE ],
			0xF9C1  => [ 0x7642 ],
			0xF9C2  => [ 0x84FC ],
			0xF9C3  => [ 0x907C ],
			0xF9C4  => [ 0x9F8D ],
			0xF9C5  => [ 0x6688 ],
			0xF9C6  => [ 0x962E ],
			0xF9C7  => [ 0x5289 ],
			0xF9C8  => [ 0x677B ],
			0xF9C9  => [ 0x67F3 ],
			0xF9CA  => [ 0x6D41 ],
			0xF9CB  => [ 0x6E9C ],
			0xF9CC  => [ 0x7409 ],
			0xF9CD  => [ 0x7559 ],
			0xF9CE  => [ 0x786B ],
			0xF9CF  => [ 0x7D10 ],
			0xF9D0  => [ 0x985E ],
			0xF9D1  => [ 0x516D ],
			0xF9D2  => [ 0x622E ],
			0xF9D3  => [ 0x9678 ],
			0xF9D4  => [ 0x502B ],
			0xF9D5  => [ 0x5D19 ],
			0xF9D6  => [ 0x6DEA ],
			0xF9D7  => [ 0x8F2A ],
			0xF9D8  => [ 0x5F8B ],
			0xF9D9  => [ 0x6144 ],
			0xF9DA  => [ 0x6817 ],
			0xF9DB  => [ 0x7387 ],
			0xF9DC  => [ 0x9686 ],
			0xF9DD  => [ 0x5229 ],
			0xF9DE  => [ 0x540F ],
			0xF9DF  => [ 0x5C65 ],
			0xF9E0  => [ 0x6613 ],
			0xF9E1  => [ 0x674E ],
			0xF9E2  => [ 0x68A8 ],
			0xF9E3  => [ 0x6CE5 ],
			0xF9E4  => [ 0x7406 ],
			0xF9E5  => [ 0x75E2 ],
			0xF9E6  => [ 0x7F79 ],
			0xF9E7  => [ 0x88CF ],
			0xF9E8  => [ 0x88E1 ],
			0xF9E9  => [ 0x91CC ],
			0xF9EA  => [ 0x96E2 ],
			0xF9EB  => [ 0x533F ],
			0xF9EC  => [ 0x6EBA ],
			0xF9ED  => [ 0x541D ],
			0xF9EE  => [ 0x71D0 ],
			0xF9EF  => [ 0x7498 ],
			0xF9F0  => [ 0x85FA ],
			0xF9F1  => [ 0x96A3 ],
			0xF9F2  => [ 0x9C57 ],
			0xF9F3  => [ 0x9E9F ],
			0xF9F4  => [ 0x6797 ],
			0xF9F5  => [ 0x6DCB ],
			0xF9F6  => [ 0x81E8 ],
			0xF9F7  => [ 0x7ACB ],
			0xF9F8  => [ 0x7B20 ],
			0xF9F9  => [ 0x7C92 ],
			0xF9FA  => [ 0x72C0 ],
			0xF9FB  => [ 0x7099 ],
			0xF9FC  => [ 0x8B58 ],
			0xF9FD  => [ 0x4EC0 ],
			0xF9FE  => [ 0x8336 ],
			0xF9FF  => [ 0x523A ],
			0xFA00  => [ 0x5207 ],
			0xFA01  => [ 0x5EA6 ],
			0xFA02  => [ 0x62D3 ],
			0xFA03  => [ 0x7CD6 ],
			0xFA04  => [ 0x5B85 ],
			0xFA05  => [ 0x6D1E ],
			0xFA06  => [ 0x66B4 ],
			0xFA07  => [ 0x8F3B ],
			0xFA08  => [ 0x884C ],
			0xFA09  => [ 0x964D ],
			0xFA0A  => [ 0x898B ],
			0xFA0B  => [ 0x5ED3 ],
			0xFA0C  => [ 0x5140 ],
			0xFA0D  => [ 0x55C0 ],
			0xFA10  => [ 0x585A ],
			0xFA12  => [ 0x6674 ],
			0xFA15  => [ 0x51DE ],
			0xFA16  => [ 0x732A ],
			0xFA17  => [ 0x76CA ],
			0xFA18  => [ 0x793C ],
			0xFA19  => [ 0x795E ],
			0xFA1A  => [ 0x7965 ],
			0xFA1B  => [ 0x798F ],
			0xFA1C  => [ 0x9756 ],
			0xFA1D  => [ 0x7CBE ],
			0xFA1E  => [ 0x7FBD ],
			0xFA20  => [ 0x8612 ],
			0xFA22  => [ 0x8AF8 ],
			0xFA25  => [ 0x9038 ],
			0xFA26  => [ 0x90FD ],
			0xFA2A  => [ 0x98EF ],
			0xFA2B  => [ 0x98FC ],
			0xFA2C  => [ 0x9928 ],
			0xFA2D  => [ 0x9DB4 ],
			0xFA2E  => [ 0x90DE ],
			0xFA2F  => [ 0x96B7 ],
			0xFA30  => [ 0x4FAE ],
			0xFA31  => [ 0x50E7 ],
			0xFA32  => [ 0x514D ],
			0xFA33  => [ 0x52C9 ],
			0xFA34  => [ 0x52E4 ],
			0xFA35  => [ 0x5351 ],
			0xFA36  => [ 0x559D ],
			0xFA37  => [ 0x5606 ],
			0xFA38  => [ 0x5668 ],
			0xFA39  => [ 0x5840 ],
			0xFA3A  => [ 0x58A8 ],
			0xFA3B  => [ 0x5C64 ],
			0xFA3C  => [ 0x5C6E ],
			0xFA3D  => [ 0x6094 ],
			0xFA3E  => [ 0x6168 ],
			0xFA3F  => [ 0x618E ],
			0xFA40  => [ 0x61F2 ],
			0xFA41  => [ 0x654F ],
			0xFA42  => [ 0x65E2 ],
			0xFA43  => [ 0x6691 ],
			0xFA44  => [ 0x6885 ],
			0xFA45  => [ 0x6D77 ],
			0xFA46  => [ 0x6E1A ],
			0xFA47  => [ 0x6F22 ],
			0xFA48  => [ 0x716E ],
			0xFA49  => [ 0x722B ],
			0xFA4A  => [ 0x7422 ],
			0xFA4B  => [ 0x7891 ],
			0xFA4C  => [ 0x793E ],
			0xFA4D  => [ 0x7949 ],
			0xFA4E  => [ 0x7948 ],
			0xFA4F  => [ 0x7950 ],
			0xFA50  => [ 0x7956 ],
			0xFA51  => [ 0x795D ],
			0xFA52  => [ 0x798D ],
			0xFA53  => [ 0x798E ],
			0xFA54  => [ 0x7A40 ],
			0xFA55  => [ 0x7A81 ],
			0xFA56  => [ 0x7BC0 ],
			0xFA57  => [ 0x7DF4 ],
			0xFA58  => [ 0x7E09 ],
			0xFA59  => [ 0x7E41 ],
			0xFA5A  => [ 0x7F72 ],
			0xFA5B  => [ 0x8005 ],
			0xFA5C  => [ 0x81ED ],
			0xFA5D  => [ 0x8279 ],
			0xFA5E  => [ 0x8279 ],
			0xFA5F  => [ 0x8457 ],
			0xFA60  => [ 0x8910 ],
			0xFA61  => [ 0x8996 ],
			0xFA62  => [ 0x8B01 ],
			0xFA63  => [ 0x8B39 ],
			0xFA64  => [ 0x8CD3 ],
			0xFA65  => [ 0x8D08 ],
			0xFA66  => [ 0x8FB6 ],
			0xFA67  => [ 0x9038 ],
			0xFA68  => [ 0x96E3 ],
			0xFA69  => [ 0x97FF ],
			0xFA6A  => [ 0x983B ],
			0xFA6B  => [ 0x6075 ],
			0xFA6C  => [ 0x242EE ],
			0xFA6D  => [ 0x8218 ],
			0xFA70  => [ 0x4E26 ],
			0xFA71  => [ 0x51B5 ],
			0xFA72  => [ 0x5168 ],
			0xFA73  => [ 0x4F80 ],
			0xFA74  => [ 0x5145 ],
			0xFA75  => [ 0x5180 ],
			0xFA76  => [ 0x52C7 ],
			0xFA77  => [ 0x52FA ],
			0xFA78  => [ 0x559D ],
			0xFA79  => [ 0x5555 ],
			0xFA7A  => [ 0x5599 ],
			0xFA7B  => [ 0x55E2 ],
			0xFA7C  => [ 0x585A ],
			0xFA7D  => [ 0x58B3 ],
			0xFA7E  => [ 0x5944 ],
			0xFA7F  => [ 0x5954 ],
			0xFA80  => [ 0x5A62 ],
			0xFA81  => [ 0x5B28 ],
			0xFA82  => [ 0x5ED2 ],
			0xFA83  => [ 0x5ED9 ],
			0xFA84  => [ 0x5F69 ],
			0xFA85  => [ 0x5FAD ],
			0xFA86  => [ 0x60D8 ],
			0xFA87  => [ 0x614E ],
			0xFA88  => [ 0x6108 ],
			0xFA89  => [ 0x618E ],
			0xFA8A  => [ 0x6160 ],
			0xFA8B  => [ 0x61F2 ],
			0xFA8C  => [ 0x6234 ],
			0xFA8D  => [ 0x63C4 ],
			0xFA8E  => [ 0x641C ],
			0xFA8F  => [ 0x6452 ],
			0xFA90  => [ 0x6556 ],
			0xFA91  => [ 0x6674 ],
			0xFA92  => [ 0x6717 ],
			0xFA93  => [ 0x671B ],
			0xFA94  => [ 0x6756 ],
			0xFA95  => [ 0x6B79 ],
			0xFA96  => [ 0x6BBA ],
			0xFA97  => [ 0x6D41 ],
			0xFA98  => [ 0x6EDB ],
			0xFA99  => [ 0x6ECB ],
			0xFA9A  => [ 0x6F22 ],
			0xFA9B  => [ 0x701E ],
			0xFA9C  => [ 0x716E ],
			0xFA9D  => [ 0x77A7 ],
			0xFA9E  => [ 0x7235 ],
			0xFA9F  => [ 0x72AF ],
			0xFAA0  => [ 0x732A ],
			0xFAA1  => [ 0x7471 ],
			0xFAA2  => [ 0x7506 ],
			0xFAA3  => [ 0x753B ],
			0xFAA4  => [ 0x761D ],
			0xFAA5  => [ 0x761F ],
			0xFAA6  => [ 0x76CA ],
			0xFAA7  => [ 0x76DB ],
			0xFAA8  => [ 0x76F4 ],
			0xFAA9  => [ 0x774A ],
			0xFAAA  => [ 0x7740 ],
			0xFAAB  => [ 0x78CC ],
			0xFAAC  => [ 0x7AB1 ],
			0xFAAD  => [ 0x7BC0 ],
			0xFAAE  => [ 0x7C7B ],
			0xFAAF  => [ 0x7D5B ],
			0xFAB0  => [ 0x7DF4 ],
			0xFAB1  => [ 0x7F3E ],
			0xFAB2  => [ 0x8005 ],
			0xFAB3  => [ 0x8352 ],
			0xFAB4  => [ 0x83EF ],
			0xFAB5  => [ 0x8779 ],
			0xFAB6  => [ 0x8941 ],
			0xFAB7  => [ 0x8986 ],
			0xFAB8  => [ 0x8996 ],
			0xFAB9  => [ 0x8ABF ],
			0xFABA  => [ 0x8AF8 ],
			0xFABB  => [ 0x8ACB ],
			0xFABC  => [ 0x8B01 ],
			0xFABD  => [ 0x8AFE ],
			0xFABE  => [ 0x8AED ],
			0xFABF  => [ 0x8B39 ],
			0xFAC0  => [ 0x8B8A ],
			0xFAC1  => [ 0x8D08 ],
			0xFAC2  => [ 0x8F38 ],
			0xFAC3  => [ 0x9072 ],
			0xFAC4  => [ 0x9199 ],
			0xFAC5  => [ 0x9276 ],
			0xFAC6  => [ 0x967C ],
			0xFAC7  => [ 0x96E3 ],
			0xFAC8  => [ 0x9756 ],
			0xFAC9  => [ 0x97DB ],
			0xFACA  => [ 0x97FF ],
			0xFACB  => [ 0x980B ],
			0xFACC  => [ 0x983B ],
			0xFACD  => [ 0x9B12 ],
			0xFACE  => [ 0x9F9C ],
			0xFACF  => [ 0x2284A ],
			0xFAD0  => [ 0x22844 ],
			0xFAD1  => [ 0x233D5 ],
			0xFAD2  => [ 0x3B9D ],
			0xFAD3  => [ 0x4018 ],
			0xFAD4  => [ 0x4039 ],
			0xFAD5  => [ 0x25249 ],
			0xFAD6  => [ 0x25CD0 ],
			0xFAD7  => [ 0x27ED3 ],
			0xFAD8  => [ 0x9F43 ],
			0xFAD9  => [ 0x9F8E ],
			0xFB00  => [ 0x66, 0x66 ],
			0xFB01  => [ 0x66, 0x69 ],
			0xFB02  => [ 0x66, 0x6C ],
			0xFB03  => [ 0x66, 0x66, 0x69 ],
			0xFB04  => [ 0x66, 0x66, 0x6C ],
			0xFB05  => [ 0x73, 0x74 ],
			0xFB06  => [ 0x73, 0x74 ],
			0xFB13  => [ 0x574, 0x576 ],
			0xFB14  => [ 0x574, 0x565 ],
			0xFB15  => [ 0x574, 0x56B ],
			0xFB16  => [ 0x57E, 0x576 ],
			0xFB17  => [ 0x574, 0x56D ],
			0xFB1D  => [ 0x5D9, 0x5B4 ],
			0xFB1F  => [ 0x5F2, 0x5B7 ],
			0xFB20  => [ 0x5E2 ],
			0xFB21  => [ 0x5D0 ],
			0xFB22  => [ 0x5D3 ],
			0xFB23  => [ 0x5D4 ],
			0xFB24  => [ 0x5DB ],
			0xFB25  => [ 0x5DC ],
			0xFB26  => [ 0x5DD ],
			0xFB27  => [ 0x5E8 ],
			0xFB28  => [ 0x5EA ],
			0xFB2A  => [ 0x5E9, 0x5C1 ],
			0xFB2B  => [ 0x5E9, 0x5C2 ],
			0xFB2C  => [ 0x5E9, 0x5BC, 0x5C1 ],
			0xFB2D  => [ 0x5E9, 0x5BC, 0x5C2 ],
			0xFB2E  => [ 0x5D0, 0x5B7 ],
			0xFB2F  => [ 0x5D0, 0x5B8 ],
			0xFB30  => [ 0x5D0, 0x5BC ],
			0xFB31  => [ 0x5D1, 0x5BC ],
			0xFB32  => [ 0x5D2, 0x5BC ],
			0xFB33  => [ 0x5D3, 0x5BC ],
			0xFB34  => [ 0x5D4, 0x5BC ],
			0xFB35  => [ 0x5D5, 0x5BC ],
			0xFB36  => [ 0x5D6, 0x5BC ],
			0xFB38  => [ 0x5D8, 0x5BC ],
			0xFB39  => [ 0x5D9, 0x5BC ],
			0xFB3A  => [ 0x5DA, 0x5BC ],
			0xFB3B  => [ 0x5DB, 0x5BC ],
			0xFB3C  => [ 0x5DC, 0x5BC ],
			0xFB3E  => [ 0x5DE, 0x5BC ],
			0xFB40  => [ 0x5E0, 0x5BC ],
			0xFB41  => [ 0x5E1, 0x5BC ],
			0xFB43  => [ 0x5E3, 0x5BC ],
			0xFB44  => [ 0x5E4, 0x5BC ],
			0xFB46  => [ 0x5E6, 0x5BC ],
			0xFB47  => [ 0x5E7, 0x5BC ],
			0xFB48  => [ 0x5E8, 0x5BC ],
			0xFB49  => [ 0x5E9, 0x5BC ],
			0xFB4A  => [ 0x5EA, 0x5BC ],
			0xFB4B  => [ 0x5D5, 0x5B9 ],
			0xFB4C  => [ 0x5D1, 0x5BF ],
			0xFB4D  => [ 0x5DB, 0x5BF ],
			0xFB4E  => [ 0x5E4, 0x5BF ],
			0xFB4F  => [ 0x5D0, 0x5DC ],
			0xFB50  => [ 0x671 ],
			0xFB51  => [ 0x671 ],
			0xFB52  => [ 0x67B ],
			0xFB53  => [ 0x67B ],
			0xFB54  => [ 0x67B ],
			0xFB55  => [ 0x67B ],
			0xFB56  => [ 0x67E ],
			0xFB57  => [ 0x67E ],
			0xFB58  => [ 0x67E ],
			0xFB59  => [ 0x67E ],
			0xFB5A  => [ 0x680 ],
			0xFB5B  => [ 0x680 ],
			0xFB5C  => [ 0x680 ],
			0xFB5D  => [ 0x680 ],
			0xFB5E  => [ 0x67A ],
			0xFB5F  => [ 0x67A ],
			0xFB60  => [ 0x67A ],
			0xFB61  => [ 0x67A ],
			0xFB62  => [ 0x67F ],
			0xFB63  => [ 0x67F ],
			0xFB64  => [ 0x67F ],
			0xFB65  => [ 0x67F ],
			0xFB66  => [ 0x679 ],
			0xFB67  => [ 0x679 ],
			0xFB68  => [ 0x679 ],
			0xFB69  => [ 0x679 ],
			0xFB6A  => [ 0x6A4 ],
			0xFB6B  => [ 0x6A4 ],
			0xFB6C  => [ 0x6A4 ],
			0xFB6D  => [ 0x6A4 ],
			0xFB6E  => [ 0x6A6 ],
			0xFB6F  => [ 0x6A6 ],
			0xFB70  => [ 0x6A6 ],
			0xFB71  => [ 0x6A6 ],
			0xFB72  => [ 0x684 ],
			0xFB73  => [ 0x684 ],
			0xFB74  => [ 0x684 ],
			0xFB75  => [ 0x684 ],
			0xFB76  => [ 0x683 ],
			0xFB77  => [ 0x683 ],
			0xFB78  => [ 0x683 ],
			0xFB79  => [ 0x683 ],
			0xFB7A  => [ 0x686 ],
			0xFB7B  => [ 0x686 ],
			0xFB7C  => [ 0x686 ],
			0xFB7D  => [ 0x686 ],
			0xFB7E  => [ 0x687 ],
			0xFB7F  => [ 0x687 ],
			0xFB80  => [ 0x687 ],
			0xFB81  => [ 0x687 ],
			0xFB82  => [ 0x68D ],
			0xFB83  => [ 0x68D ],
			0xFB84  => [ 0x68C ],
			0xFB85  => [ 0x68C ],
			0xFB86  => [ 0x68E ],
			0xFB87  => [ 0x68E ],
			0xFB88  => [ 0x688 ],
			0xFB89  => [ 0x688 ],
			0xFB8A  => [ 0x698 ],
			0xFB8B  => [ 0x698 ],
			0xFB8C  => [ 0x691 ],
			0xFB8D  => [ 0x691 ],
			0xFB8E  => [ 0x6A9 ],
			0xFB8F  => [ 0x6A9 ],
			0xFB90  => [ 0x6A9 ],
			0xFB91  => [ 0x6A9 ],
			0xFB92  => [ 0x6AF ],
			0xFB93  => [ 0x6AF ],
			0xFB94  => [ 0x6AF ],
			0xFB95  => [ 0x6AF ],
			0xFB96  => [ 0x6B3 ],
			0xFB97  => [ 0x6B3 ],
			0xFB98  => [ 0x6B3 ],
			0xFB99  => [ 0x6B3 ],
			0xFB9A  => [ 0x6B1 ],
			0xFB9B  => [ 0x6B1 ],
			0xFB9C  => [ 0x6B1 ],
			0xFB9D  => [ 0x6B1 ],
			0xFB9E  => [ 0x6BA ],
			0xFB9F  => [ 0x6BA ],
			0xFBA0  => [ 0x6BB ],
			0xFBA1  => [ 0x6BB ],
			0xFBA2  => [ 0x6BB ],
			0xFBA3  => [ 0x6BB ],
			0xFBA4  => [ 0x6C0 ],
			0xFBA5  => [ 0x6C0 ],
			0xFBA6  => [ 0x6C1 ],
			0xFBA7  => [ 0x6C1 ],
			0xFBA8  => [ 0x6C1 ],
			0xFBA9  => [ 0x6C1 ],
			0xFBAA  => [ 0x6BE ],
			0xFBAB  => [ 0x6BE ],
			0xFBAC  => [ 0x6BE ],
			0xFBAD  => [ 0x6BE ],
			0xFBAE  => [ 0x6D2 ],
			0xFBAF  => [ 0x6D2 ],
			0xFBB0  => [ 0x6D3 ],
			0xFBB1  => [ 0x6D3 ],
			0xFBD3  => [ 0x6AD ],
			0xFBD4  => [ 0x6AD ],
			0xFBD5  => [ 0x6AD ],
			0xFBD6  => [ 0x6AD ],
			0xFBD7  => [ 0x6C7 ],
			0xFBD8  => [ 0x6C7 ],
			0xFBD9  => [ 0x6C6 ],
			0xFBDA  => [ 0x6C6 ],
			0xFBDB  => [ 0x6C8 ],
			0xFBDC  => [ 0x6C8 ],
			0xFBDD  => [ 0x6C7, 0x674 ],
			0xFBDE  => [ 0x6CB ],
			0xFBDF  => [ 0x6CB ],
			0xFBE0  => [ 0x6C5 ],
			0xFBE1  => [ 0x6C5 ],
			0xFBE2  => [ 0x6C9 ],
			0xFBE3  => [ 0x6C9 ],
			0xFBE4  => [ 0x6D0 ],
			0xFBE5  => [ 0x6D0 ],
			0xFBE6  => [ 0x6D0 ],
			0xFBE7  => [ 0x6D0 ],
			0xFBE8  => [ 0x649 ],
			0xFBE9  => [ 0x649 ],
			0xFBEA  => [ 0x626, 0x627 ],
			0xFBEB  => [ 0x626, 0x627 ],
			0xFBEC  => [ 0x626, 0x6D5 ],
			0xFBED  => [ 0x626, 0x6D5 ],
			0xFBEE  => [ 0x626, 0x648 ],
			0xFBEF  => [ 0x626, 0x648 ],
			0xFBF0  => [ 0x626, 0x6C7 ],
			0xFBF1  => [ 0x626, 0x6C7 ],
			0xFBF2  => [ 0x626, 0x6C6 ],
			0xFBF3  => [ 0x626, 0x6C6 ],
			0xFBF4  => [ 0x626, 0x6C8 ],
			0xFBF5  => [ 0x626, 0x6C8 ],
			0xFBF6  => [ 0x626, 0x6D0 ],
			0xFBF7  => [ 0x626, 0x6D0 ],
			0xFBF8  => [ 0x626, 0x6D0 ],
			0xFBF9  => [ 0x626, 0x649 ],
			0xFBFA  => [ 0x626, 0x649 ],
			0xFBFB  => [ 0x626, 0x649 ],
			0xFBFC  => [ 0x6CC ],
			0xFBFD  => [ 0x6CC ],
			0xFBFE  => [ 0x6CC ],
			0xFBFF  => [ 0x6CC ],
			0xFC00  => [ 0x626, 0x62C ],
			0xFC01  => [ 0x626, 0x62D ],
			0xFC02  => [ 0x626, 0x645 ],
			0xFC03  => [ 0x626, 0x649 ],
			0xFC04  => [ 0x626, 0x64A ],
			0xFC05  => [ 0x628, 0x62C ],
			0xFC06  => [ 0x628, 0x62D ],
			0xFC07  => [ 0x628, 0x62E ],
			0xFC08  => [ 0x628, 0x645 ],
			0xFC09  => [ 0x628, 0x649 ],
			0xFC0A  => [ 0x628, 0x64A ],
			0xFC0B  => [ 0x62A, 0x62C ],
			0xFC0C  => [ 0x62A, 0x62D ],
			0xFC0D  => [ 0x62A, 0x62E ],
			0xFC0E  => [ 0x62A, 0x645 ],
			0xFC0F  => [ 0x62A, 0x649 ],
			0xFC10  => [ 0x62A, 0x64A ],
			0xFC11  => [ 0x62B, 0x62C ],
			0xFC12  => [ 0x62B, 0x645 ],
			0xFC13  => [ 0x62B, 0x649 ],
			0xFC14  => [ 0x62B, 0x64A ],
			0xFC15  => [ 0x62C, 0x62D ],
			0xFC16  => [ 0x62C, 0x645 ],
			0xFC17  => [ 0x62D, 0x62C ],
			0xFC18  => [ 0x62D, 0x645 ],
			0xFC19  => [ 0x62E, 0x62C ],
			0xFC1A  => [ 0x62E, 0x62D ],
			0xFC1B  => [ 0x62E, 0x645 ],
			0xFC1C  => [ 0x633, 0x62C ],
			0xFC1D  => [ 0x633, 0x62D ],
			0xFC1E  => [ 0x633, 0x62E ],
			0xFC1F  => [ 0x633, 0x645 ],
			0xFC20  => [ 0x635, 0x62D ],
			0xFC21  => [ 0x635, 0x645 ],
			0xFC22  => [ 0x636, 0x62C ],
			0xFC23  => [ 0x636, 0x62D ],
			0xFC24  => [ 0x636, 0x62E ],
			0xFC25  => [ 0x636, 0x645 ],
			0xFC26  => [ 0x637, 0x62D ],
			0xFC27  => [ 0x637, 0x645 ],
			0xFC28  => [ 0x638, 0x645 ],
			0xFC29  => [ 0x639, 0x62C ],
			0xFC2A  => [ 0x639, 0x645 ],
			0xFC2B  => [ 0x63A, 0x62C ],
			0xFC2C  => [ 0x63A, 0x645 ],
			0xFC2D  => [ 0x641, 0x62C ],
			0xFC2E  => [ 0x641, 0x62D ],
			0xFC2F  => [ 0x641, 0x62E ],
			0xFC30  => [ 0x641, 0x645 ],
			0xFC31  => [ 0x641, 0x649 ],
			0xFC32  => [ 0x641, 0x64A ],
			0xFC33  => [ 0x642, 0x62D ],
			0xFC34  => [ 0x642, 0x645 ],
			0xFC35  => [ 0x642, 0x649 ],
			0xFC36  => [ 0x642, 0x64A ],
			0xFC37  => [ 0x643, 0x627 ],
			0xFC38  => [ 0x643, 0x62C ],
			0xFC39  => [ 0x643, 0x62D ],
			0xFC3A  => [ 0x643, 0x62E ],
			0xFC3B  => [ 0x643, 0x644 ],
			0xFC3C  => [ 0x643, 0x645 ],
			0xFC3D  => [ 0x643, 0x649 ],
			0xFC3E  => [ 0x643, 0x64A ],
			0xFC3F  => [ 0x644, 0x62C ],
			0xFC40  => [ 0x644, 0x62D ],
			0xFC41  => [ 0x644, 0x62E ],
			0xFC42  => [ 0x644, 0x645 ],
			0xFC43  => [ 0x644, 0x649 ],
			0xFC44  => [ 0x644, 0x64A ],
			0xFC45  => [ 0x645, 0x62C ],
			0xFC46  => [ 0x645, 0x62D ],
			0xFC47  => [ 0x645, 0x62E ],
			0xFC48  => [ 0x645, 0x645 ],
			0xFC49  => [ 0x645, 0x649 ],
			0xFC4A  => [ 0x645, 0x64A ],
			0xFC4B  => [ 0x646, 0x62C ],
			0xFC4C  => [ 0x646, 0x62D ],
			0xFC4D  => [ 0x646, 0x62E ],
			0xFC4E  => [ 0x646, 0x645 ],
			0xFC4F  => [ 0x646, 0x649 ],
			0xFC50  => [ 0x646, 0x64A ],
			0xFC51  => [ 0x647, 0x62C ],
			0xFC52  => [ 0x647, 0x645 ],
			0xFC53  => [ 0x647, 0x649 ],
			0xFC54  => [ 0x647, 0x64A ],
			0xFC55  => [ 0x64A, 0x62C ],
			0xFC56  => [ 0x64A, 0x62D ],
			0xFC57  => [ 0x64A, 0x62E ],
			0xFC58  => [ 0x64A, 0x645 ],
			0xFC59  => [ 0x64A, 0x649 ],
			0xFC5A  => [ 0x64A, 0x64A ],
			0xFC5B  => [ 0x630, 0x670 ],
			0xFC5C  => [ 0x631, 0x670 ],
			0xFC5D  => [ 0x649, 0x670 ],
			0xFC64  => [ 0x626, 0x631 ],
			0xFC65  => [ 0x626, 0x632 ],
			0xFC66  => [ 0x626, 0x645 ],
			0xFC67  => [ 0x626, 0x646 ],
			0xFC68  => [ 0x626, 0x649 ],
			0xFC69  => [ 0x626, 0x64A ],
			0xFC6A  => [ 0x628, 0x631 ],
			0xFC6B  => [ 0x628, 0x632 ],
			0xFC6C  => [ 0x628, 0x645 ],
			0xFC6D  => [ 0x628, 0x646 ],
			0xFC6E  => [ 0x628, 0x649 ],
			0xFC6F  => [ 0x628, 0x64A ],
			0xFC70  => [ 0x62A, 0x631 ],
			0xFC71  => [ 0x62A, 0x632 ],
			0xFC72  => [ 0x62A, 0x645 ],
			0xFC73  => [ 0x62A, 0x646 ],
			0xFC74  => [ 0x62A, 0x649 ],
			0xFC75  => [ 0x62A, 0x64A ],
			0xFC76  => [ 0x62B, 0x631 ],
			0xFC77  => [ 0x62B, 0x632 ],
			0xFC78  => [ 0x62B, 0x645 ],
			0xFC79  => [ 0x62B, 0x646 ],
			0xFC7A  => [ 0x62B, 0x649 ],
			0xFC7B  => [ 0x62B, 0x64A ],
			0xFC7C  => [ 0x641, 0x649 ],
			0xFC7D  => [ 0x641, 0x64A ],
			0xFC7E  => [ 0x642, 0x649 ],
			0xFC7F  => [ 0x642, 0x64A ],
			0xFC80  => [ 0x643, 0x627 ],
			0xFC81  => [ 0x643, 0x644 ],
			0xFC82  => [ 0x643, 0x645 ],
			0xFC83  => [ 0x643, 0x649 ],
			0xFC84  => [ 0x643, 0x64A ],
			0xFC85  => [ 0x644, 0x645 ],
			0xFC86  => [ 0x644, 0x649 ],
			0xFC87  => [ 0x644, 0x64A ],
			0xFC88  => [ 0x645, 0x627 ],
			0xFC89  => [ 0x645, 0x645 ],
			0xFC8A  => [ 0x646, 0x631 ],
			0xFC8B  => [ 0x646, 0x632 ],
			0xFC8C  => [ 0x646, 0x645 ],
			0xFC8D  => [ 0x646, 0x646 ],
			0xFC8E  => [ 0x646, 0x649 ],
			0xFC8F  => [ 0x646, 0x64A ],
			0xFC90  => [ 0x649, 0x670 ],
			0xFC91  => [ 0x64A, 0x631 ],
			0xFC92  => [ 0x64A, 0x632 ],
			0xFC93  => [ 0x64A, 0x645 ],
			0xFC94  => [ 0x64A, 0x646 ],
			0xFC95  => [ 0x64A, 0x649 ],
			0xFC96  => [ 0x64A, 0x64A ],
			0xFC97  => [ 0x626, 0x62C ],
			0xFC98  => [ 0x626, 0x62D ],
			0xFC99  => [ 0x626, 0x62E ],
			0xFC9A  => [ 0x626, 0x645 ],
			0xFC9B  => [ 0x626, 0x647 ],
			0xFC9C  => [ 0x628, 0x62C ],
			0xFC9D  => [ 0x628, 0x62D ],
			0xFC9E  => [ 0x628, 0x62E ],
			0xFC9F  => [ 0x628, 0x645 ],
			0xFCA0  => [ 0x628, 0x647 ],
			0xFCA1  => [ 0x62A, 0x62C ],
			0xFCA2  => [ 0x62A, 0x62D ],
			0xFCA3  => [ 0x62A, 0x62E ],
			0xFCA4  => [ 0x62A, 0x645 ],
			0xFCA5  => [ 0x62A, 0x647 ],
			0xFCA6  => [ 0x62B, 0x645 ],
			0xFCA7  => [ 0x62C, 0x62D ],
			0xFCA8  => [ 0x62C, 0x645 ],
			0xFCA9  => [ 0x62D, 0x62C ],
			0xFCAA  => [ 0x62D, 0x645 ],
			0xFCAB  => [ 0x62E, 0x62C ],
			0xFCAC  => [ 0x62E, 0x645 ],
			0xFCAD  => [ 0x633, 0x62C ],
			0xFCAE  => [ 0x633, 0x62D ],
			0xFCAF  => [ 0x633, 0x62E ],
			0xFCB0  => [ 0x633, 0x645 ],
			0xFCB1  => [ 0x635, 0x62D ],
			0xFCB2  => [ 0x635, 0x62E ],
			0xFCB3  => [ 0x635, 0x645 ],
			0xFCB4  => [ 0x636, 0x62C ],
			0xFCB5  => [ 0x636, 0x62D ],
			0xFCB6  => [ 0x636, 0x62E ],
			0xFCB7  => [ 0x636, 0x645 ],
			0xFCB8  => [ 0x637, 0x62D ],
			0xFCB9  => [ 0x638, 0x645 ],
			0xFCBA  => [ 0x639, 0x62C ],
			0xFCBB  => [ 0x639, 0x645 ],
			0xFCBC  => [ 0x63A, 0x62C ],
			0xFCBD  => [ 0x63A, 0x645 ],
			0xFCBE  => [ 0x641, 0x62C ],
			0xFCBF  => [ 0x641, 0x62D ],
			0xFCC0  => [ 0x641, 0x62E ],
			0xFCC1  => [ 0x641, 0x645 ],
			0xFCC2  => [ 0x642, 0x62D ],
			0xFCC3  => [ 0x642, 0x645 ],
			0xFCC4  => [ 0x643, 0x62C ],
			0xFCC5  => [ 0x643, 0x62D ],
			0xFCC6  => [ 0x643, 0x62E ],
			0xFCC7  => [ 0x643, 0x644 ],
			0xFCC8  => [ 0x643, 0x645 ],
			0xFCC9  => [ 0x644, 0x62C ],
			0xFCCA  => [ 0x644, 0x62D ],
			0xFCCB  => [ 0x644, 0x62E ],
			0xFCCC  => [ 0x644, 0x645 ],
			0xFCCD  => [ 0x644, 0x647 ],
			0xFCCE  => [ 0x645, 0x62C ],
			0xFCCF  => [ 0x645, 0x62D ],
			0xFCD0  => [ 0x645, 0x62E ],
			0xFCD1  => [ 0x645, 0x645 ],
			0xFCD2  => [ 0x646, 0x62C ],
			0xFCD3  => [ 0x646, 0x62D ],
			0xFCD4  => [ 0x646, 0x62E ],
			0xFCD5  => [ 0x646, 0x645 ],
			0xFCD6  => [ 0x646, 0x647 ],
			0xFCD7  => [ 0x647, 0x62C ],
			0xFCD8  => [ 0x647, 0x645 ],
			0xFCD9  => [ 0x647, 0x670 ],
			0xFCDA  => [ 0x64A, 0x62C ],
			0xFCDB  => [ 0x64A, 0x62D ],
			0xFCDC  => [ 0x64A, 0x62E ],
			0xFCDD  => [ 0x64A, 0x645 ],
			0xFCDE  => [ 0x64A, 0x647 ],
			0xFCDF  => [ 0x626, 0x645 ],
			0xFCE0  => [ 0x626, 0x647 ],
			0xFCE1  => [ 0x628, 0x645 ],
			0xFCE2  => [ 0x628, 0x647 ],
			0xFCE3  => [ 0x62A, 0x645 ],
			0xFCE4  => [ 0x62A, 0x647 ],
			0xFCE5  => [ 0x62B, 0x645 ],
			0xFCE6  => [ 0x62B, 0x647 ],
			0xFCE7  => [ 0x633, 0x645 ],
			0xFCE8  => [ 0x633, 0x647 ],
			0xFCE9  => [ 0x634, 0x645 ],
			0xFCEA  => [ 0x634, 0x647 ],
			0xFCEB  => [ 0x643, 0x644 ],
			0xFCEC  => [ 0x643, 0x645 ],
			0xFCED  => [ 0x644, 0x645 ],
			0xFCEE  => [ 0x646, 0x645 ],
			0xFCEF  => [ 0x646, 0x647 ],
			0xFCF0  => [ 0x64A, 0x645 ],
			0xFCF1  => [ 0x64A, 0x647 ],
			0xFCF2  => [ 0x640, 0x64E, 0x651 ],
			0xFCF3  => [ 0x640, 0x64F, 0x651 ],
			0xFCF4  => [ 0x640, 0x650, 0x651 ],
			0xFCF5  => [ 0x637, 0x649 ],
			0xFCF6  => [ 0x637, 0x64A ],
			0xFCF7  => [ 0x639, 0x649 ],
			0xFCF8  => [ 0x639, 0x64A ],
			0xFCF9  => [ 0x63A, 0x649 ],
			0xFCFA  => [ 0x63A, 0x64A ],
			0xFCFB  => [ 0x633, 0x649 ],
			0xFCFC  => [ 0x633, 0x64A ],
			0xFCFD  => [ 0x634, 0x649 ],
			0xFCFE  => [ 0x634, 0x64A ],
			0xFCFF  => [ 0x62D, 0x649 ],
			0xFD00  => [ 0x62D, 0x64A ],
			0xFD01  => [ 0x62C, 0x649 ],
			0xFD02  => [ 0x62C, 0x64A ],
			0xFD03  => [ 0x62E, 0x649 ],
			0xFD04  => [ 0x62E, 0x64A ],
			0xFD05  => [ 0x635, 0x649 ],
			0xFD06  => [ 0x635, 0x64A ],
			0xFD07  => [ 0x636, 0x649 ],
			0xFD08  => [ 0x636, 0x64A ],
			0xFD09  => [ 0x634, 0x62C ],
			0xFD0A  => [ 0x634, 0x62D ],
			0xFD0B  => [ 0x634, 0x62E ],
			0xFD0C  => [ 0x634, 0x645 ],
			0xFD0D  => [ 0x634, 0x631 ],
			0xFD0E  => [ 0x633, 0x631 ],
			0xFD0F  => [ 0x635, 0x631 ],
			0xFD10  => [ 0x636, 0x631 ],
			0xFD11  => [ 0x637, 0x649 ],
			0xFD12  => [ 0x637, 0x64A ],
			0xFD13  => [ 0x639, 0x649 ],
			0xFD14  => [ 0x639, 0x64A ],
			0xFD15  => [ 0x63A, 0x649 ],
			0xFD16  => [ 0x63A, 0x64A ],
			0xFD17  => [ 0x633, 0x649 ],
			0xFD18  => [ 0x633, 0x64A ],
			0xFD19  => [ 0x634, 0x649 ],
			0xFD1A  => [ 0x634, 0x64A ],
			0xFD1B  => [ 0x62D, 0x649 ],
			0xFD1C  => [ 0x62D, 0x64A ],
			0xFD1D  => [ 0x62C, 0x649 ],
			0xFD1E  => [ 0x62C, 0x64A ],
			0xFD1F  => [ 0x62E, 0x649 ],
			0xFD20  => [ 0x62E, 0x64A ],
			0xFD21  => [ 0x635, 0x649 ],
			0xFD22  => [ 0x635, 0x64A ],
			0xFD23  => [ 0x636, 0x649 ],
			0xFD24  => [ 0x636, 0x64A ],
			0xFD25  => [ 0x634, 0x62C ],
			0xFD26  => [ 0x634, 0x62D ],
			0xFD27  => [ 0x634, 0x62E ],
			0xFD28  => [ 0x634, 0x645 ],
			0xFD29  => [ 0x634, 0x631 ],
			0xFD2A  => [ 0x633, 0x631 ],
			0xFD2B  => [ 0x635, 0x631 ],
			0xFD2C  => [ 0x636, 0x631 ],
			0xFD2D  => [ 0x634, 0x62C ],
			0xFD2E  => [ 0x634, 0x62D ],
			0xFD2F  => [ 0x634, 0x62E ],
			0xFD30  => [ 0x634, 0x645 ],
			0xFD31  => [ 0x633, 0x647 ],
			0xFD32  => [ 0x634, 0x647 ],
			0xFD33  => [ 0x637, 0x645 ],
			0xFD34  => [ 0x633, 0x62C ],
			0xFD35  => [ 0x633, 0x62D ],
			0xFD36  => [ 0x633, 0x62E ],
			0xFD37  => [ 0x634, 0x62C ],
			0xFD38  => [ 0x634, 0x62D ],
			0xFD39  => [ 0x634, 0x62E ],
			0xFD3A  => [ 0x637, 0x645 ],
			0xFD3B  => [ 0x638, 0x645 ],
			0xFD3C  => [ 0x627, 0x64B ],
			0xFD3D  => [ 0x627, 0x64B ],
			0xFD50  => [ 0x62A, 0x62C, 0x645 ],
			0xFD51  => [ 0x62A, 0x62D, 0x62C ],
			0xFD52  => [ 0x62A, 0x62D, 0x62C ],
			0xFD53  => [ 0x62A, 0x62D, 0x645 ],
			0xFD54  => [ 0x62A, 0x62E, 0x645 ],
			0xFD55  => [ 0x62A, 0x645, 0x62C ],
			0xFD56  => [ 0x62A, 0x645, 0x62D ],
			0xFD57  => [ 0x62A, 0x645, 0x62E ],
			0xFD58  => [ 0x62C, 0x645, 0x62D ],
			0xFD59  => [ 0x62C, 0x645, 0x62D ],
			0xFD5A  => [ 0x62D, 0x645, 0x64A ],
			0xFD5B  => [ 0x62D, 0x645, 0x649 ],
			0xFD5C  => [ 0x633, 0x62D, 0x62C ],
			0xFD5D  => [ 0x633, 0x62C, 0x62D ],
			0xFD5E  => [ 0x633, 0x62C, 0x649 ],
			0xFD5F  => [ 0x633, 0x645, 0x62D ],
			0xFD60  => [ 0x633, 0x645, 0x62D ],
			0xFD61  => [ 0x633, 0x645, 0x62C ],
			0xFD62  => [ 0x633, 0x645, 0x645 ],
			0xFD63  => [ 0x633, 0x645, 0x645 ],
			0xFD64  => [ 0x635, 0x62D, 0x62D ],
			0xFD65  => [ 0x635, 0x62D, 0x62D ],
			0xFD66  => [ 0x635, 0x645, 0x645 ],
			0xFD67  => [ 0x634, 0x62D, 0x645 ],
			0xFD68  => [ 0x634, 0x62D, 0x645 ],
			0xFD69  => [ 0x634, 0x62C, 0x64A ],
			0xFD6A  => [ 0x634, 0x645, 0x62E ],
			0xFD6B  => [ 0x634, 0x645, 0x62E ],
			0xFD6C  => [ 0x634, 0x645, 0x645 ],
			0xFD6D  => [ 0x634, 0x645, 0x645 ],
			0xFD6E  => [ 0x636, 0x62D, 0x649 ],
			0xFD6F  => [ 0x636, 0x62E, 0x645 ],
			0xFD70  => [ 0x636, 0x62E, 0x645 ],
			0xFD71  => [ 0x637, 0x645, 0x62D ],
			0xFD72  => [ 0x637, 0x645, 0x62D ],
			0xFD73  => [ 0x637, 0x645, 0x645 ],
			0xFD74  => [ 0x637, 0x645, 0x64A ],
			0xFD75  => [ 0x639, 0x62C, 0x645 ],
			0xFD76  => [ 0x639, 0x645, 0x645 ],
			0xFD77  => [ 0x639, 0x645, 0x645 ],
			0xFD78  => [ 0x639, 0x645, 0x649 ],
			0xFD79  => [ 0x63A, 0x645, 0x645 ],
			0xFD7A  => [ 0x63A, 0x645, 0x64A ],
			0xFD7B  => [ 0x63A, 0x645, 0x649 ],
			0xFD7C  => [ 0x641, 0x62E, 0x645 ],
			0xFD7D  => [ 0x641, 0x62E, 0x645 ],
			0xFD7E  => [ 0x642, 0x645, 0x62D ],
			0xFD7F  => [ 0x642, 0x645, 0x645 ],
			0xFD80  => [ 0x644, 0x62D, 0x645 ],
			0xFD81  => [ 0x644, 0x62D, 0x64A ],
			0xFD82  => [ 0x644, 0x62D, 0x649 ],
			0xFD83  => [ 0x644, 0x62C, 0x62C ],
			0xFD84  => [ 0x644, 0x62C, 0x62C ],
			0xFD85  => [ 0x644, 0x62E, 0x645 ],
			0xFD86  => [ 0x644, 0x62E, 0x645 ],
			0xFD87  => [ 0x644, 0x645, 0x62D ],
			0xFD88  => [ 0x644, 0x645, 0x62D ],
			0xFD89  => [ 0x645, 0x62D, 0x62C ],
			0xFD8A  => [ 0x645, 0x62D, 0x645 ],
			0xFD8B  => [ 0x645, 0x62D, 0x64A ],
			0xFD8C  => [ 0x645, 0x62C, 0x62D ],
			0xFD8D  => [ 0x645, 0x62C, 0x645 ],
			0xFD8E  => [ 0x645, 0x62E, 0x62C ],
			0xFD8F  => [ 0x645, 0x62E, 0x645 ],
			0xFD92  => [ 0x645, 0x62C, 0x62E ],
			0xFD93  => [ 0x647, 0x645, 0x62C ],
			0xFD94  => [ 0x647, 0x645, 0x645 ],
			0xFD95  => [ 0x646, 0x62D, 0x645 ],
			0xFD96  => [ 0x646, 0x62D, 0x649 ],
			0xFD97  => [ 0x646, 0x62C, 0x645 ],
			0xFD98  => [ 0x646, 0x62C, 0x645 ],
			0xFD99  => [ 0x646, 0x62C, 0x649 ],
			0xFD9A  => [ 0x646, 0x645, 0x64A ],
			0xFD9B  => [ 0x646, 0x645, 0x649 ],
			0xFD9C  => [ 0x64A, 0x645, 0x645 ],
			0xFD9D  => [ 0x64A, 0x645, 0x645 ],
			0xFD9E  => [ 0x628, 0x62E, 0x64A ],
			0xFD9F  => [ 0x62A, 0x62C, 0x64A ],
			0xFDA0  => [ 0x62A, 0x62C, 0x649 ],
			0xFDA1  => [ 0x62A, 0x62E, 0x64A ],
			0xFDA2  => [ 0x62A, 0x62E, 0x649 ],
			0xFDA3  => [ 0x62A, 0x645, 0x64A ],
			0xFDA4  => [ 0x62A, 0x645, 0x649 ],
			0xFDA5  => [ 0x62C, 0x645, 0x64A ],
			0xFDA6  => [ 0x62C, 0x62D, 0x649 ],
			0xFDA7  => [ 0x62C, 0x645, 0x649 ],
			0xFDA8  => [ 0x633, 0x62E, 0x649 ],
			0xFDA9  => [ 0x635, 0x62D, 0x64A ],
			0xFDAA  => [ 0x634, 0x62D, 0x64A ],
			0xFDAB  => [ 0x636, 0x62D, 0x64A ],
			0xFDAC  => [ 0x644, 0x62C, 0x64A ],
			0xFDAD  => [ 0x644, 0x645, 0x64A ],
			0xFDAE  => [ 0x64A, 0x62D, 0x64A ],
			0xFDAF  => [ 0x64A, 0x62C, 0x64A ],
			0xFDB0  => [ 0x64A, 0x645, 0x64A ],
			0xFDB1  => [ 0x645, 0x645, 0x64A ],
			0xFDB2  => [ 0x642, 0x645, 0x64A ],
			0xFDB3  => [ 0x646, 0x62D, 0x64A ],
			0xFDB4  => [ 0x642, 0x645, 0x62D ],
			0xFDB5  => [ 0x644, 0x62D, 0x645 ],
			0xFDB6  => [ 0x639, 0x645, 0x64A ],
			0xFDB7  => [ 0x643, 0x645, 0x64A ],
			0xFDB8  => [ 0x646, 0x62C, 0x62D ],
			0xFDB9  => [ 0x645, 0x62E, 0x64A ],
			0xFDBA  => [ 0x644, 0x62C, 0x645 ],
			0xFDBB  => [ 0x643, 0x645, 0x645 ],
			0xFDBC  => [ 0x644, 0x62C, 0x645 ],
			0xFDBD  => [ 0x646, 0x62C, 0x62D ],
			0xFDBE  => [ 0x62C, 0x62D, 0x64A ],
			0xFDBF  => [ 0x62D, 0x62C, 0x64A ],
			0xFDC0  => [ 0x645, 0x62C, 0x64A ],
			0xFDC1  => [ 0x641, 0x645, 0x64A ],
			0xFDC2  => [ 0x628, 0x62D, 0x64A ],
			0xFDC3  => [ 0x643, 0x645, 0x645 ],
			0xFDC4  => [ 0x639, 0x62C, 0x645 ],
			0xFDC5  => [ 0x635, 0x645, 0x645 ],
			0xFDC6  => [ 0x633, 0x62E, 0x64A ],
			0xFDC7  => [ 0x646, 0x62C, 0x64A ],
			0xFDF0  => [ 0x635, 0x644, 0x6D2 ],
			0xFDF1  => [ 0x642, 0x644, 0x6D2 ],
			0xFDF2  => [ 0x627, 0x644, 0x644, 0x647 ],
			0xFDF3  => [ 0x627, 0x643, 0x628, 0x631 ],
			0xFDF4  => [ 0x645, 0x62D, 0x645, 0x62F ],
			0xFDF5  => [ 0x635, 0x644, 0x639, 0x645 ],
			0xFDF6  => [ 0x631, 0x633, 0x648, 0x644 ],
			0xFDF7  => [ 0x639, 0x644, 0x64A, 0x647 ],
			0xFDF8  => [ 0x648, 0x633, 0x644, 0x645 ],
			0xFDF9  => [ 0x635, 0x644, 0x649 ],
			0xFDFC  => [ 0x631, 0x6CC, 0x627, 0x644 ],
			0xFE11  => [ 0x3001 ],
			0xFE17  => [ 0x3016 ],
			0xFE18  => [ 0x3017 ],
			0xFE31  => [ 0x2014 ],
			0xFE32  => [ 0x2013 ],
			0xFE39  => [ 0x3014 ],
			0xFE3A  => [ 0x3015 ],
			0xFE3B  => [ 0x3010 ],
			0xFE3C  => [ 0x3011 ],
			0xFE3D  => [ 0x300A ],
			0xFE3E  => [ 0x300B ],
			0xFE3F  => [ 0x3008 ],
			0xFE40  => [ 0x3009 ],
			0xFE41  => [ 0x300C ],
			0xFE42  => [ 0x300D ],
			0xFE43  => [ 0x300E ],
			0xFE44  => [ 0x300F ],
			0xFE51  => [ 0x3001 ],
			0xFE58  => [ 0x2014 ],
			0xFE5D  => [ 0x3014 ],
			0xFE5E  => [ 0x3015 ],
			0xFE63  => [ 0x2D ],
			0xFE71  => [ 0x640, 0x64B ],
			0xFE77  => [ 0x640, 0x64E ],
			0xFE79  => [ 0x640, 0x64F ],
			0xFE7B  => [ 0x640, 0x650 ],
			0xFE7D  => [ 0x640, 0x651 ],
			0xFE7F  => [ 0x640, 0x652 ],
			0xFE80  => [ 0x621 ],
			0xFE81  => [ 0x622 ],
			0xFE82  => [ 0x622 ],
			0xFE83  => [ 0x623 ],
			0xFE84  => [ 0x623 ],
			0xFE85  => [ 0x624 ],
			0xFE86  => [ 0x624 ],
			0xFE87  => [ 0x625 ],
			0xFE88  => [ 0x625 ],
			0xFE89  => [ 0x626 ],
			0xFE8A  => [ 0x626 ],
			0xFE8B  => [ 0x626 ],
			0xFE8C  => [ 0x626 ],
			0xFE8D  => [ 0x627 ],
			0xFE8E  => [ 0x627 ],
			0xFE8F  => [ 0x628 ],
			0xFE90  => [ 0x628 ],
			0xFE91  => [ 0x628 ],
			0xFE92  => [ 0x628 ],
			0xFE93  => [ 0x629 ],
			0xFE94  => [ 0x629 ],
			0xFE95  => [ 0x62A ],
			0xFE96  => [ 0x62A ],
			0xFE97  => [ 0x62A ],
			0xFE98  => [ 0x62A ],
			0xFE99  => [ 0x62B ],
			0xFE9A  => [ 0x62B ],
			0xFE9B  => [ 0x62B ],
			0xFE9C  => [ 0x62B ],
			0xFE9D  => [ 0x62C ],
			0xFE9E  => [ 0x62C ],
			0xFE9F  => [ 0x62C ],
			0xFEA0  => [ 0x62C ],
			0xFEA1  => [ 0x62D ],
			0xFEA2  => [ 0x62D ],
			0xFEA3  => [ 0x62D ],
			0xFEA4  => [ 0x62D ],
			0xFEA5  => [ 0x62E ],
			0xFEA6  => [ 0x62E ],
			0xFEA7  => [ 0x62E ],
			0xFEA8  => [ 0x62E ],
			0xFEA9  => [ 0x62F ],
			0xFEAA  => [ 0x62F ],
			0xFEAB  => [ 0x630 ],
			0xFEAC  => [ 0x630 ],
			0xFEAD  => [ 0x631 ],
			0xFEAE  => [ 0x631 ],
			0xFEAF  => [ 0x632 ],
			0xFEB0  => [ 0x632 ],
			0xFEB1  => [ 0x633 ],
			0xFEB2  => [ 0x633 ],
			0xFEB3  => [ 0x633 ],
			0xFEB4  => [ 0x633 ],
			0xFEB5  => [ 0x634 ],
			0xFEB6  => [ 0x634 ],
			0xFEB7  => [ 0x634 ],
			0xFEB8  => [ 0x634 ],
			0xFEB9  => [ 0x635 ],
			0xFEBA  => [ 0x635 ],
			0xFEBB  => [ 0x635 ],
			0xFEBC  => [ 0x635 ],
			0xFEBD  => [ 0x636 ],
			0xFEBE  => [ 0x636 ],
			0xFEBF  => [ 0x636 ],
			0xFEC0  => [ 0x636 ],
			0xFEC1  => [ 0x637 ],
			0xFEC2  => [ 0x637 ],
			0xFEC3  => [ 0x637 ],
			0xFEC4  => [ 0x637 ],
			0xFEC5  => [ 0x638 ],
			0xFEC6  => [ 0x638 ],
			0xFEC7  => [ 0x638 ],
			0xFEC8  => [ 0x638 ],
			0xFEC9  => [ 0x639 ],
			0xFECA  => [ 0x639 ],
			0xFECB  => [ 0x639 ],
			0xFECC  => [ 0x639 ],
			0xFECD  => [ 0x63A ],
			0xFECE  => [ 0x63A ],
			0xFECF  => [ 0x63A ],
			0xFED0  => [ 0x63A ],
			0xFED1  => [ 0x641 ],
			0xFED2  => [ 0x641 ],
			0xFED3  => [ 0x641 ],
			0xFED4  => [ 0x641 ],
			0xFED5  => [ 0x642 ],
			0xFED6  => [ 0x642 ],
			0xFED7  => [ 0x642 ],
			0xFED8  => [ 0x642 ],
			0xFED9  => [ 0x643 ],
			0xFEDA  => [ 0x643 ],
			0xFEDB  => [ 0x643 ],
			0xFEDC  => [ 0x643 ],
			0xFEDD  => [ 0x644 ],
			0xFEDE  => [ 0x644 ],
			0xFEDF  => [ 0x644 ],
			0xFEE0  => [ 0x644 ],
			0xFEE1  => [ 0x645 ],
			0xFEE2  => [ 0x645 ],
			0xFEE3  => [ 0x645 ],
			0xFEE4  => [ 0x645 ],
			0xFEE5  => [ 0x646 ],
			0xFEE6  => [ 0x646 ],
			0xFEE7  => [ 0x646 ],
			0xFEE8  => [ 0x646 ],
			0xFEE9  => [ 0x647 ],
			0xFEEA  => [ 0x647 ],
			0xFEEB  => [ 0x647 ],
			0xFEEC  => [ 0x647 ],
			0xFEED  => [ 0x648 ],
			0xFEEE  => [ 0x648 ],
			0xFEEF  => [ 0x649 ],
			0xFEF0  => [ 0x649 ],
			0xFEF1  => [ 0x64A ],
			0xFEF2  => [ 0x64A ],
			0xFEF3  => [ 0x64A ],
			0xFEF4  => [ 0x64A ],
			0xFEF5  => [ 0x644, 0x622 ],
			0xFEF6  => [ 0x644, 0x622 ],
			0xFEF7  => [ 0x644, 0x623 ],
			0xFEF8  => [ 0x644, 0x623 ],
			0xFEF9  => [ 0x644, 0x625 ],
			0xFEFA  => [ 0x644, 0x625 ],
			0xFEFB  => [ 0x644, 0x627 ],
			0xFEFC  => [ 0x644, 0x627 ],
			0xFF0D  => [ 0x2D ],
			0xFF0E  => [ 0x2E ],
			0xFF10  => [ 0x30 ],
			0xFF11  => [ 0x31 ],
			0xFF12  => [ 0x32 ],
			0xFF13  => [ 0x33 ],
			0xFF14  => [ 0x34 ],
			0xFF15  => [ 0x35 ],
			0xFF16  => [ 0x36 ],
			0xFF17  => [ 0x37 ],
			0xFF18  => [ 0x38 ],
			0xFF19  => [ 0x39 ],
			0xFF21  => [ 0x61 ],
			0xFF22  => [ 0x62 ],
			0xFF23  => [ 0x63 ],
			0xFF24  => [ 0x64 ],
			0xFF25  => [ 0x65 ],
			0xFF26  => [ 0x66 ],
			0xFF27  => [ 0x67 ],
			0xFF28  => [ 0x68 ],
			0xFF29  => [ 0x69 ],
			0xFF2A  => [ 0x6A ],
			0xFF2B  => [ 0x6B ],
			0xFF2C  => [ 0x6C ],
			0xFF2D  => [ 0x6D ],
			0xFF2E  => [ 0x6E ],
			0xFF2F  => [ 0x6F ],
			0xFF30  => [ 0x70 ],
			0xFF31  => [ 0x71 ],
			0xFF32  => [ 0x72 ],
			0xFF33  => [ 0x73 ],
			0xFF34  => [ 0x74 ],
			0xFF35  => [ 0x75 ],
			0xFF36  => [ 0x76 ],
			0xFF37  => [ 0x77 ],
			0xFF38  => [ 0x78 ],
			0xFF39  => [ 0x79 ],
			0xFF3A  => [ 0x7A ],
			0xFF41  => [ 0x61 ],
			0xFF42  => [ 0x62 ],
			0xFF43  => [ 0x63 ],
			0xFF44  => [ 0x64 ],
			0xFF45  => [ 0x65 ],
			0xFF46  => [ 0x66 ],
			0xFF47  => [ 0x67 ],
			0xFF48  => [ 0x68 ],
			0xFF49  => [ 0x69 ],
			0xFF4A  => [ 0x6A ],
			0xFF4B  => [ 0x6B ],
			0xFF4C  => [ 0x6C ],
			0xFF4D  => [ 0x6D ],
			0xFF4E  => [ 0x6E ],
			0xFF4F  => [ 0x6F ],
			0xFF50  => [ 0x70 ],
			0xFF51  => [ 0x71 ],
			0xFF52  => [ 0x72 ],
			0xFF53  => [ 0x73 ],
			0xFF54  => [ 0x74 ],
			0xFF55  => [ 0x75 ],
			0xFF56  => [ 0x76 ],
			0xFF57  => [ 0x77 ],
			0xFF58  => [ 0x78 ],
			0xFF59  => [ 0x79 ],
			0xFF5A  => [ 0x7A ],
			0xFF5F  => [ 0x2985 ],
			0xFF60  => [ 0x2986 ],
			0xFF61  => [ 0x2E ],
			0xFF62  => [ 0x300C ],
			0xFF63  => [ 0x300D ],
			0xFF64  => [ 0x3001 ],
			0xFF65  => [ 0x30FB ],
			0xFF66  => [ 0x30F2 ],
			0xFF67  => [ 0x30A1 ],
			0xFF68  => [ 0x30A3 ],
			0xFF69  => [ 0x30A5 ],
			0xFF6A  => [ 0x30A7 ],
			0xFF6B  => [ 0x30A9 ],
			0xFF6C  => [ 0x30E3 ],
			0xFF6D  => [ 0x30E5 ],
			0xFF6E  => [ 0x30E7 ],
			0xFF6F  => [ 0x30C3 ],
			0xFF70  => [ 0x30FC ],
			0xFF71  => [ 0x30A2 ],
			0xFF72  => [ 0x30A4 ],
			0xFF73  => [ 0x30A6 ],
			0xFF74  => [ 0x30A8 ],
			0xFF75  => [ 0x30AA ],
			0xFF76  => [ 0x30AB ],
			0xFF77  => [ 0x30AD ],
			0xFF78  => [ 0x30AF ],
			0xFF79  => [ 0x30B1 ],
			0xFF7A  => [ 0x30B3 ],
			0xFF7B  => [ 0x30B5 ],
			0xFF7C  => [ 0x30B7 ],
			0xFF7D  => [ 0x30B9 ],
			0xFF7E  => [ 0x30BB ],
			0xFF7F  => [ 0x30BD ],
			0xFF80  => [ 0x30BF ],
			0xFF81  => [ 0x30C1 ],
			0xFF82  => [ 0x30C4 ],
			0xFF83  => [ 0x30C6 ],
			0xFF84  => [ 0x30C8 ],
			0xFF85  => [ 0x30CA ],
			0xFF86  => [ 0x30CB ],
			0xFF87  => [ 0x30CC ],
			0xFF88  => [ 0x30CD ],
			0xFF89  => [ 0x30CE ],
			0xFF8A  => [ 0x30CF ],
			0xFF8B  => [ 0x30D2 ],
			0xFF8C  => [ 0x30D5 ],
			0xFF8D  => [ 0x30D8 ],
			0xFF8E  => [ 0x30DB ],
			0xFF8F  => [ 0x30DE ],
			0xFF90  => [ 0x30DF ],
			0xFF91  => [ 0x30E0 ],
			0xFF92  => [ 0x30E1 ],
			0xFF93  => [ 0x30E2 ],
			0xFF94  => [ 0x30E4 ],
			0xFF95  => [ 0x30E6 ],
			0xFF96  => [ 0x30E8 ],
			0xFF97  => [ 0x30E9 ],
			0xFF98  => [ 0x30EA ],
			0xFF99  => [ 0x30EB ],
			0xFF9A  => [ 0x30EC ],
			0xFF9B  => [ 0x30ED ],
			0xFF9C  => [ 0x30EF ],
			0xFF9D  => [ 0x30F3 ],
			0xFF9E  => [ 0x3099 ],
			0xFF9F  => [ 0x309A ],
			0xFFA1  => [ 0x1100 ],
			0xFFA2  => [ 0x1101 ],
			0xFFA3  => [ 0x11AA ],
			0xFFA4  => [ 0x1102 ],
			0xFFA5  => [ 0x11AC ],
			0xFFA6  => [ 0x11AD ],
			0xFFA7  => [ 0x1103 ],
			0xFFA8  => [ 0x1104 ],
			0xFFA9  => [ 0x1105 ],
			0xFFAA  => [ 0x11B0 ],
			0xFFAB  => [ 0x11B1 ],
			0xFFAC  => [ 0x11B2 ],
			0xFFAD  => [ 0x11B3 ],
			0xFFAE  => [ 0x11B4 ],
			0xFFAF  => [ 0x11B5 ],
			0xFFB0  => [ 0x111A ],
			0xFFB1  => [ 0x1106 ],
			0xFFB2  => [ 0x1107 ],
			0xFFB3  => [ 0x1108 ],
			0xFFB4  => [ 0x1121 ],
			0xFFB5  => [ 0x1109 ],
			0xFFB6  => [ 0x110A ],
			0xFFB7  => [ 0x110B ],
			0xFFB8  => [ 0x110C ],
			0xFFB9  => [ 0x110D ],
			0xFFBA  => [ 0x110E ],
			0xFFBB  => [ 0x110F ],
			0xFFBC  => [ 0x1110 ],
			0xFFBD  => [ 0x1111 ],
			0xFFBE  => [ 0x1112 ],
			0xFFC2  => [ 0x1161 ],
			0xFFC3  => [ 0x1162 ],
			0xFFC4  => [ 0x1163 ],
			0xFFC5  => [ 0x1164 ],
			0xFFC6  => [ 0x1165 ],
			0xFFC7  => [ 0x1166 ],
			0xFFCA  => [ 0x1167 ],
			0xFFCB  => [ 0x1168 ],
			0xFFCC  => [ 0x1169 ],
			0xFFCD  => [ 0x116A ],
			0xFFCE  => [ 0x116B ],
			0xFFCF  => [ 0x116C ],
			0xFFD2  => [ 0x116D ],
			0xFFD3  => [ 0x116E ],
			0xFFD4  => [ 0x116F ],
			0xFFD5  => [ 0x1170 ],
			0xFFD6  => [ 0x1171 ],
			0xFFD7  => [ 0x1172 ],
			0xFFDA  => [ 0x1173 ],
			0xFFDB  => [ 0x1174 ],
			0xFFDC  => [ 0x1175 ],
			0xFFE0  => [ 0xA2 ],
			0xFFE1  => [ 0xA3 ],
			0xFFE2  => [ 0xAC ],
			0xFFE4  => [ 0xA6 ],
			0xFFE5  => [ 0xA5 ],
			0xFFE6  => [ 0x20A9 ],
			0xFFE8  => [ 0x2502 ],
			0xFFE9  => [ 0x2190 ],
			0xFFEA  => [ 0x2191 ],
			0xFFEB  => [ 0x2192 ],
			0xFFEC  => [ 0x2193 ],
			0xFFED  => [ 0x25A0 ],
			0xFFEE  => [ 0x25CB ],
			0x10400 => [ 0x10428 ],
			0x10401 => [ 0x10429 ],
			0x10402 => [ 0x1042A ],
			0x10403 => [ 0x1042B ],
			0x10404 => [ 0x1042C ],
			0x10405 => [ 0x1042D ],
			0x10406 => [ 0x1042E ],
			0x10407 => [ 0x1042F ],
			0x10408 => [ 0x10430 ],
			0x10409 => [ 0x10431 ],
			0x1040A => [ 0x10432 ],
			0x1040B => [ 0x10433 ],
			0x1040C => [ 0x10434 ],
			0x1040D => [ 0x10435 ],
			0x1040E => [ 0x10436 ],
			0x1040F => [ 0x10437 ],
			0x10410 => [ 0x10438 ],
			0x10411 => [ 0x10439 ],
			0x10412 => [ 0x1043A ],
			0x10413 => [ 0x1043B ],
			0x10414 => [ 0x1043C ],
			0x10415 => [ 0x1043D ],
			0x10416 => [ 0x1043E ],
			0x10417 => [ 0x1043F ],
			0x10418 => [ 0x10440 ],
			0x10419 => [ 0x10441 ],
			0x1041A => [ 0x10442 ],
			0x1041B => [ 0x10443 ],
			0x1041C => [ 0x10444 ],
			0x1041D => [ 0x10445 ],
			0x1041E => [ 0x10446 ],
			0x1041F => [ 0x10447 ],
			0x10420 => [ 0x10448 ],
			0x10421 => [ 0x10449 ],
			0x10422 => [ 0x1044A ],
			0x10423 => [ 0x1044B ],
			0x10424 => [ 0x1044C ],
			0x10425 => [ 0x1044D ],
			0x10426 => [ 0x1044E ],
			0x10427 => [ 0x1044F ],
			0x118A0 => [ 0x118C0 ],
			0x118A1 => [ 0x118C1 ],
			0x118A2 => [ 0x118C2 ],
			0x118A3 => [ 0x118C3 ],
			0x118A4 => [ 0x118C4 ],
			0x118A5 => [ 0x118C5 ],
			0x118A6 => [ 0x118C6 ],
			0x118A7 => [ 0x118C7 ],
			0x118A8 => [ 0x118C8 ],
			0x118A9 => [ 0x118C9 ],
			0x118AA => [ 0x118CA ],
			0x118AB => [ 0x118CB ],
			0x118AC => [ 0x118CC ],
			0x118AD => [ 0x118CD ],
			0x118AE => [ 0x118CE ],
			0x118AF => [ 0x118CF ],
			0x118B0 => [ 0x118D0 ],
			0x118B1 => [ 0x118D1 ],
			0x118B2 => [ 0x118D2 ],
			0x118B3 => [ 0x118D3 ],
			0x118B4 => [ 0x118D4 ],
			0x118B5 => [ 0x118D5 ],
			0x118B6 => [ 0x118D6 ],
			0x118B7 => [ 0x118D7 ],
			0x118B8 => [ 0x118D8 ],
			0x118B9 => [ 0x118D9 ],
			0x118BA => [ 0x118DA ],
			0x118BB => [ 0x118DB ],
			0x118BC => [ 0x118DC ],
			0x118BD => [ 0x118DD ],
			0x118BE => [ 0x118DE ],
			0x118BF => [ 0x118DF ],
			0x1D15E => [ 0x1D157, 0x1D165 ],
			0x1D15F => [ 0x1D158, 0x1D165 ],
			0x1D160 => [ 0x1D158, 0x1D165, 0x1D16E ],
			0x1D161 => [ 0x1D158, 0x1D165, 0x1D16F ],
			0x1D162 => [ 0x1D158, 0x1D165, 0x1D170 ],
			0x1D163 => [ 0x1D158, 0x1D165, 0x1D171 ],
			0x1D164 => [ 0x1D158, 0x1D165, 0x1D172 ],
			0x1D1BB => [ 0x1D1B9, 0x1D165 ],
			0x1D1BC => [ 0x1D1BA, 0x1D165 ],
			0x1D1BD => [ 0x1D1B9, 0x1D165, 0x1D16E ],
			0x1D1BE => [ 0x1D1BA, 0x1D165, 0x1D16E ],
			0x1D1BF => [ 0x1D1B9, 0x1D165, 0x1D16F ],
			0x1D1C0 => [ 0x1D1BA, 0x1D165, 0x1D16F ],
			0x1D400 => [ 0x61 ],
			0x1D401 => [ 0x62 ],
			0x1D402 => [ 0x63 ],
			0x1D403 => [ 0x64 ],
			0x1D404 => [ 0x65 ],
			0x1D405 => [ 0x66 ],
			0x1D406 => [ 0x67 ],
			0x1D407 => [ 0x68 ],
			0x1D408 => [ 0x69 ],
			0x1D409 => [ 0x6A ],
			0x1D40A => [ 0x6B ],
			0x1D40B => [ 0x6C ],
			0x1D40C => [ 0x6D ],
			0x1D40D => [ 0x6E ],
			0x1D40E => [ 0x6F ],
			0x1D40F => [ 0x70 ],
			0x1D410 => [ 0x71 ],
			0x1D411 => [ 0x72 ],
			0x1D412 => [ 0x73 ],
			0x1D413 => [ 0x74 ],
			0x1D414 => [ 0x75 ],
			0x1D415 => [ 0x76 ],
			0x1D416 => [ 0x77 ],
			0x1D417 => [ 0x78 ],
			0x1D418 => [ 0x79 ],
			0x1D419 => [ 0x7A ],
			0x1D41A => [ 0x61 ],
			0x1D41B => [ 0x62 ],
			0x1D41C => [ 0x63 ],
			0x1D41D => [ 0x64 ],
			0x1D41E => [ 0x65 ],
			0x1D41F => [ 0x66 ],
			0x1D420 => [ 0x67 ],
			0x1D421 => [ 0x68 ],
			0x1D422 => [ 0x69 ],
			0x1D423 => [ 0x6A ],
			0x1D424 => [ 0x6B ],
			0x1D425 => [ 0x6C ],
			0x1D426 => [ 0x6D ],
			0x1D427 => [ 0x6E ],
			0x1D428 => [ 0x6F ],
			0x1D429 => [ 0x70 ],
			0x1D42A => [ 0x71 ],
			0x1D42B => [ 0x72 ],
			0x1D42C => [ 0x73 ],
			0x1D42D => [ 0x74 ],
			0x1D42E => [ 0x75 ],
			0x1D42F => [ 0x76 ],
			0x1D430 => [ 0x77 ],
			0x1D431 => [ 0x78 ],
			0x1D432 => [ 0x79 ],
			0x1D433 => [ 0x7A ],
			0x1D434 => [ 0x61 ],
			0x1D435 => [ 0x62 ],
			0x1D436 => [ 0x63 ],
			0x1D437 => [ 0x64 ],
			0x1D438 => [ 0x65 ],
			0x1D439 => [ 0x66 ],
			0x1D43A => [ 0x67 ],
			0x1D43B => [ 0x68 ],
			0x1D43C => [ 0x69 ],
			0x1D43D => [ 0x6A ],
			0x1D43E => [ 0x6B ],
			0x1D43F => [ 0x6C ],
			0x1D440 => [ 0x6D ],
			0x1D441 => [ 0x6E ],
			0x1D442 => [ 0x6F ],
			0x1D443 => [ 0x70 ],
			0x1D444 => [ 0x71 ],
			0x1D445 => [ 0x72 ],
			0x1D446 => [ 0x73 ],
			0x1D447 => [ 0x74 ],
			0x1D448 => [ 0x75 ],
			0x1D449 => [ 0x76 ],
			0x1D44A => [ 0x77 ],
			0x1D44B => [ 0x78 ],
			0x1D44C => [ 0x79 ],
			0x1D44D => [ 0x7A ],
			0x1D44E => [ 0x61 ],
			0x1D44F => [ 0x62 ],
			0x1D450 => [ 0x63 ],
			0x1D451 => [ 0x64 ],
			0x1D452 => [ 0x65 ],
			0x1D453 => [ 0x66 ],
			0x1D454 => [ 0x67 ],
			0x1D456 => [ 0x69 ],
			0x1D457 => [ 0x6A ],
			0x1D458 => [ 0x6B ],
			0x1D459 => [ 0x6C ],
			0x1D45A => [ 0x6D ],
			0x1D45B => [ 0x6E ],
			0x1D45C => [ 0x6F ],
			0x1D45D => [ 0x70 ],
			0x1D45E => [ 0x71 ],
			0x1D45F => [ 0x72 ],
			0x1D460 => [ 0x73 ],
			0x1D461 => [ 0x74 ],
			0x1D462 => [ 0x75 ],
			0x1D463 => [ 0x76 ],
			0x1D464 => [ 0x77 ],
			0x1D465 => [ 0x78 ],
			0x1D466 => [ 0x79 ],
			0x1D467 => [ 0x7A ],
			0x1D468 => [ 0x61 ],
			0x1D469 => [ 0x62 ],
			0x1D46A => [ 0x63 ],
			0x1D46B => [ 0x64 ],
			0x1D46C => [ 0x65 ],
			0x1D46D => [ 0x66 ],
			0x1D46E => [ 0x67 ],
			0x1D46F => [ 0x68 ],
			0x1D470 => [ 0x69 ],
			0x1D471 => [ 0x6A ],
			0x1D472 => [ 0x6B ],
			0x1D473 => [ 0x6C ],
			0x1D474 => [ 0x6D ],
			0x1D475 => [ 0x6E ],
			0x1D476 => [ 0x6F ],
			0x1D477 => [ 0x70 ],
			0x1D478 => [ 0x71 ],
			0x1D479 => [ 0x72 ],
			0x1D47A => [ 0x73 ],
			0x1D47B => [ 0x74 ],
			0x1D47C => [ 0x75 ],
			0x1D47D => [ 0x76 ],
			0x1D47E => [ 0x77 ],
			0x1D47F => [ 0x78 ],
			0x1D480 => [ 0x79 ],
			0x1D481 => [ 0x7A ],
			0x1D482 => [ 0x61 ],
			0x1D483 => [ 0x62 ],
			0x1D484 => [ 0x63 ],
			0x1D485 => [ 0x64 ],
			0x1D486 => [ 0x65 ],
			0x1D487 => [ 0x66 ],
			0x1D488 => [ 0x67 ],
			0x1D489 => [ 0x68 ],
			0x1D48A => [ 0x69 ],
			0x1D48B => [ 0x6A ],
			0x1D48C => [ 0x6B ],
			0x1D48D => [ 0x6C ],
			0x1D48E => [ 0x6D ],
			0x1D48F => [ 0x6E ],
			0x1D490 => [ 0x6F ],
			0x1D491 => [ 0x70 ],
			0x1D492 => [ 0x71 ],
			0x1D493 => [ 0x72 ],
			0x1D494 => [ 0x73 ],
			0x1D495 => [ 0x74 ],
			0x1D496 => [ 0x75 ],
			0x1D497 => [ 0x76 ],
			0x1D498 => [ 0x77 ],
			0x1D499 => [ 0x78 ],
			0x1D49A => [ 0x79 ],
			0x1D49B => [ 0x7A ],
			0x1D49C => [ 0x61 ],
			0x1D49E => [ 0x63 ],
			0x1D49F => [ 0x64 ],
			0x1D4A2 => [ 0x67 ],
			0x1D4A5 => [ 0x6A ],
			0x1D4A6 => [ 0x6B ],
			0x1D4A9 => [ 0x6E ],
			0x1D4AA => [ 0x6F ],
			0x1D4AB => [ 0x70 ],
			0x1D4AC => [ 0x71 ],
			0x1D4AE => [ 0x73 ],
			0x1D4AF => [ 0x74 ],
			0x1D4B0 => [ 0x75 ],
			0x1D4B1 => [ 0x76 ],
			0x1D4B2 => [ 0x77 ],
			0x1D4B3 => [ 0x78 ],
			0x1D4B4 => [ 0x79 ],
			0x1D4B5 => [ 0x7A ],
			0x1D4B6 => [ 0x61 ],
			0x1D4B7 => [ 0x62 ],
			0x1D4B8 => [ 0x63 ],
			0x1D4B9 => [ 0x64 ],
			0x1D4BB => [ 0x66 ],
			0x1D4BD => [ 0x68 ],
			0x1D4BE => [ 0x69 ],
			0x1D4BF => [ 0x6A ],
			0x1D4C0 => [ 0x6B ],
			0x1D4C1 => [ 0x6C ],
			0x1D4C2 => [ 0x6D ],
			0x1D4C3 => [ 0x6E ],
			0x1D4C5 => [ 0x70 ],
			0x1D4C6 => [ 0x71 ],
			0x1D4C7 => [ 0x72 ],
			0x1D4C8 => [ 0x73 ],
			0x1D4C9 => [ 0x74 ],
			0x1D4CA => [ 0x75 ],
			0x1D4CB => [ 0x76 ],
			0x1D4CC => [ 0x77 ],
			0x1D4CD => [ 0x78 ],
			0x1D4CE => [ 0x79 ],
			0x1D4CF => [ 0x7A ],
			0x1D4D0 => [ 0x61 ],
			0x1D4D1 => [ 0x62 ],
			0x1D4D2 => [ 0x63 ],
			0x1D4D3 => [ 0x64 ],
			0x1D4D4 => [ 0x65 ],
			0x1D4D5 => [ 0x66 ],
			0x1D4D6 => [ 0x67 ],
			0x1D4D7 => [ 0x68 ],
			0x1D4D8 => [ 0x69 ],
			0x1D4D9 => [ 0x6A ],
			0x1D4DA => [ 0x6B ],
			0x1D4DB => [ 0x6C ],
			0x1D4DC => [ 0x6D ],
			0x1D4DD => [ 0x6E ],
			0x1D4DE => [ 0x6F ],
			0x1D4DF => [ 0x70 ],
			0x1D4E0 => [ 0x71 ],
			0x1D4E1 => [ 0x72 ],
			0x1D4E2 => [ 0x73 ],
			0x1D4E3 => [ 0x74 ],
			0x1D4E4 => [ 0x75 ],
			0x1D4E5 => [ 0x76 ],
			0x1D4E6 => [ 0x77 ],
			0x1D4E7 => [ 0x78 ],
			0x1D4E8 => [ 0x79 ],
			0x1D4E9 => [ 0x7A ],
			0x1D4EA => [ 0x61 ],
			0x1D4EB => [ 0x62 ],
			0x1D4EC => [ 0x63 ],
			0x1D4ED => [ 0x64 ],
			0x1D4EE => [ 0x65 ],
			0x1D4EF => [ 0x66 ],
			0x1D4F0 => [ 0x67 ],
			0x1D4F1 => [ 0x68 ],
			0x1D4F2 => [ 0x69 ],
			0x1D4F3 => [ 0x6A ],
			0x1D4F4 => [ 0x6B ],
			0x1D4F5 => [ 0x6C ],
			0x1D4F6 => [ 0x6D ],
			0x1D4F7 => [ 0x6E ],
			0x1D4F8 => [ 0x6F ],
			0x1D4F9 => [ 0x70 ],
			0x1D4FA => [ 0x71 ],
			0x1D4FB => [ 0x72 ],
			0x1D4FC => [ 0x73 ],
			0x1D4FD => [ 0x74 ],
			0x1D4FE => [ 0x75 ],
			0x1D4FF => [ 0x76 ],
			0x1D500 => [ 0x77 ],
			0x1D501 => [ 0x78 ],
			0x1D502 => [ 0x79 ],
			0x1D503 => [ 0x7A ],
			0x1D504 => [ 0x61 ],
			0x1D505 => [ 0x62 ],
			0x1D507 => [ 0x64 ],
			0x1D508 => [ 0x65 ],
			0x1D509 => [ 0x66 ],
			0x1D50A => [ 0x67 ],
			0x1D50D => [ 0x6A ],
			0x1D50E => [ 0x6B ],
			0x1D50F => [ 0x6C ],
			0x1D510 => [ 0x6D ],
			0x1D511 => [ 0x6E ],
			0x1D512 => [ 0x6F ],
			0x1D513 => [ 0x70 ],
			0x1D514 => [ 0x71 ],
			0x1D516 => [ 0x73 ],
			0x1D517 => [ 0x74 ],
			0x1D518 => [ 0x75 ],
			0x1D519 => [ 0x76 ],
			0x1D51A => [ 0x77 ],
			0x1D51B => [ 0x78 ],
			0x1D51C => [ 0x79 ],
			0x1D51E => [ 0x61 ],
			0x1D51F => [ 0x62 ],
			0x1D520 => [ 0x63 ],
			0x1D521 => [ 0x64 ],
			0x1D522 => [ 0x65 ],
			0x1D523 => [ 0x66 ],
			0x1D524 => [ 0x67 ],
			0x1D525 => [ 0x68 ],
			0x1D526 => [ 0x69 ],
			0x1D527 => [ 0x6A ],
			0x1D528 => [ 0x6B ],
			0x1D529 => [ 0x6C ],
			0x1D52A => [ 0x6D ],
			0x1D52B => [ 0x6E ],
			0x1D52C => [ 0x6F ],
			0x1D52D => [ 0x70 ],
			0x1D52E => [ 0x71 ],
			0x1D52F => [ 0x72 ],
			0x1D530 => [ 0x73 ],
			0x1D531 => [ 0x74 ],
			0x1D532 => [ 0x75 ],
			0x1D533 => [ 0x76 ],
			0x1D534 => [ 0x77 ],
			0x1D535 => [ 0x78 ],
			0x1D536 => [ 0x79 ],
			0x1D537 => [ 0x7A ],
			0x1D538 => [ 0x61 ],
			0x1D539 => [ 0x62 ],
			0x1D53B => [ 0x64 ],
			0x1D53C => [ 0x65 ],
			0x1D53D => [ 0x66 ],
			0x1D53E => [ 0x67 ],
			0x1D540 => [ 0x69 ],
			0x1D541 => [ 0x6A ],
			0x1D542 => [ 0x6B ],
			0x1D543 => [ 0x6C ],
			0x1D544 => [ 0x6D ],
			0x1D546 => [ 0x6F ],
			0x1D54A => [ 0x73 ],
			0x1D54B => [ 0x74 ],
			0x1D54C => [ 0x75 ],
			0x1D54D => [ 0x76 ],
			0x1D54E => [ 0x77 ],
			0x1D54F => [ 0x78 ],
			0x1D550 => [ 0x79 ],
			0x1D552 => [ 0x61 ],
			0x1D553 => [ 0x62 ],
			0x1D554 => [ 0x63 ],
			0x1D555 => [ 0x64 ],
			0x1D556 => [ 0x65 ],
			0x1D557 => [ 0x66 ],
			0x1D558 => [ 0x67 ],
			0x1D559 => [ 0x68 ],
			0x1D55A => [ 0x69 ],
			0x1D55B => [ 0x6A ],
			0x1D55C => [ 0x6B ],
			0x1D55D => [ 0x6C ],
			0x1D55E => [ 0x6D ],
			0x1D55F => [ 0x6E ],
			0x1D560 => [ 0x6F ],
			0x1D561 => [ 0x70 ],
			0x1D562 => [ 0x71 ],
			0x1D563 => [ 0x72 ],
			0x1D564 => [ 0x73 ],
			0x1D565 => [ 0x74 ],
			0x1D566 => [ 0x75 ],
			0x1D567 => [ 0x76 ],
			0x1D568 => [ 0x77 ],
			0x1D569 => [ 0x78 ],
			0x1D56A => [ 0x79 ],
			0x1D56B => [ 0x7A ],
			0x1D56C => [ 0x61 ],
			0x1D56D => [ 0x62 ],
			0x1D56E => [ 0x63 ],
			0x1D56F => [ 0x64 ],
			0x1D570 => [ 0x65 ],
			0x1D571 => [ 0x66 ],
			0x1D572 => [ 0x67 ],
			0x1D573 => [ 0x68 ],
			0x1D574 => [ 0x69 ],
			0x1D575 => [ 0x6A ],
			0x1D576 => [ 0x6B ],
			0x1D577 => [ 0x6C ],
			0x1D578 => [ 0x6D ],
			0x1D579 => [ 0x6E ],
			0x1D57A => [ 0x6F ],
			0x1D57B => [ 0x70 ],
			0x1D57C => [ 0x71 ],
			0x1D57D => [ 0x72 ],
			0x1D57E => [ 0x73 ],
			0x1D57F => [ 0x74 ],
			0x1D580 => [ 0x75 ],
			0x1D581 => [ 0x76 ],
			0x1D582 => [ 0x77 ],
			0x1D583 => [ 0x78 ],
			0x1D584 => [ 0x79 ],
			0x1D585 => [ 0x7A ],
			0x1D586 => [ 0x61 ],
			0x1D587 => [ 0x62 ],
			0x1D588 => [ 0x63 ],
			0x1D589 => [ 0x64 ],
			0x1D58A => [ 0x65 ],
			0x1D58B => [ 0x66 ],
			0x1D58C => [ 0x67 ],
			0x1D58D => [ 0x68 ],
			0x1D58E => [ 0x69 ],
			0x1D58F => [ 0x6A ],
			0x1D590 => [ 0x6B ],
			0x1D591 => [ 0x6C ],
			0x1D592 => [ 0x6D ],
			0x1D593 => [ 0x6E ],
			0x1D594 => [ 0x6F ],
			0x1D595 => [ 0x70 ],
			0x1D596 => [ 0x71 ],
			0x1D597 => [ 0x72 ],
			0x1D598 => [ 0x73 ],
			0x1D599 => [ 0x74 ],
			0x1D59A => [ 0x75 ],
			0x1D59B => [ 0x76 ],
			0x1D59C => [ 0x77 ],
			0x1D59D => [ 0x78 ],
			0x1D59E => [ 0x79 ],
			0x1D59F => [ 0x7A ],
			0x1D5A0 => [ 0x61 ],
			0x1D5A1 => [ 0x62 ],
			0x1D5A2 => [ 0x63 ],
			0x1D5A3 => [ 0x64 ],
			0x1D5A4 => [ 0x65 ],
			0x1D5A5 => [ 0x66 ],
			0x1D5A6 => [ 0x67 ],
			0x1D5A7 => [ 0x68 ],
			0x1D5A8 => [ 0x69 ],
			0x1D5A9 => [ 0x6A ],
			0x1D5AA => [ 0x6B ],
			0x1D5AB => [ 0x6C ],
			0x1D5AC => [ 0x6D ],
			0x1D5AD => [ 0x6E ],
			0x1D5AE => [ 0x6F ],
			0x1D5AF => [ 0x70 ],
			0x1D5B0 => [ 0x71 ],
			0x1D5B1 => [ 0x72 ],
			0x1D5B2 => [ 0x73 ],
			0x1D5B3 => [ 0x74 ],
			0x1D5B4 => [ 0x75 ],
			0x1D5B5 => [ 0x76 ],
			0x1D5B6 => [ 0x77 ],
			0x1D5B7 => [ 0x78 ],
			0x1D5B8 => [ 0x79 ],
			0x1D5B9 => [ 0x7A ],
			0x1D5BA => [ 0x61 ],
			0x1D5BB => [ 0x62 ],
			0x1D5BC => [ 0x63 ],
			0x1D5BD => [ 0x64 ],
			0x1D5BE => [ 0x65 ],
			0x1D5BF => [ 0x66 ],
			0x1D5C0 => [ 0x67 ],
			0x1D5C1 => [ 0x68 ],
			0x1D5C2 => [ 0x69 ],
			0x1D5C3 => [ 0x6A ],
			0x1D5C4 => [ 0x6B ],
			0x1D5C5 => [ 0x6C ],
			0x1D5C6 => [ 0x6D ],
			0x1D5C7 => [ 0x6E ],
			0x1D5C8 => [ 0x6F ],
			0x1D5C9 => [ 0x70 ],
			0x1D5CA => [ 0x71 ],
			0x1D5CB => [ 0x72 ],
			0x1D5CC => [ 0x73 ],
			0x1D5CD => [ 0x74 ],
			0x1D5CE => [ 0x75 ],
			0x1D5CF => [ 0x76 ],
			0x1D5D0 => [ 0x77 ],
			0x1D5D1 => [ 0x78 ],
			0x1D5D2 => [ 0x79 ],
			0x1D5D3 => [ 0x7A ],
			0x1D5D4 => [ 0x61 ],
			0x1D5D5 => [ 0x62 ],
			0x1D5D6 => [ 0x63 ],
			0x1D5D7 => [ 0x64 ],
			0x1D5D8 => [ 0x65 ],
			0x1D5D9 => [ 0x66 ],
			0x1D5DA => [ 0x67 ],
			0x1D5DB => [ 0x68 ],
			0x1D5DC => [ 0x69 ],
			0x1D5DD => [ 0x6A ],
			0x1D5DE => [ 0x6B ],
			0x1D5DF => [ 0x6C ],
			0x1D5E0 => [ 0x6D ],
			0x1D5E1 => [ 0x6E ],
			0x1D5E2 => [ 0x6F ],
			0x1D5E3 => [ 0x70 ],
			0x1D5E4 => [ 0x71 ],
			0x1D5E5 => [ 0x72 ],
			0x1D5E6 => [ 0x73 ],
			0x1D5E7 => [ 0x74 ],
			0x1D5E8 => [ 0x75 ],
			0x1D5E9 => [ 0x76 ],
			0x1D5EA => [ 0x77 ],
			0x1D5EB => [ 0x78 ],
			0x1D5EC => [ 0x79 ],
			0x1D5ED => [ 0x7A ],
			0x1D5EE => [ 0x61 ],
			0x1D5EF => [ 0x62 ],
			0x1D5F0 => [ 0x63 ],
			0x1D5F1 => [ 0x64 ],
			0x1D5F2 => [ 0x65 ],
			0x1D5F3 => [ 0x66 ],
			0x1D5F4 => [ 0x67 ],
			0x1D5F5 => [ 0x68 ],
			0x1D5F6 => [ 0x69 ],
			0x1D5F7 => [ 0x6A ],
			0x1D5F8 => [ 0x6B ],
			0x1D5F9 => [ 0x6C ],
			0x1D5FA => [ 0x6D ],
			0x1D5FB => [ 0x6E ],
			0x1D5FC => [ 0x6F ],
			0x1D5FD => [ 0x70 ],
			0x1D5FE => [ 0x71 ],
			0x1D5FF => [ 0x72 ],
			0x1D600 => [ 0x73 ],
			0x1D601 => [ 0x74 ],
			0x1D602 => [ 0x75 ],
			0x1D603 => [ 0x76 ],
			0x1D604 => [ 0x77 ],
			0x1D605 => [ 0x78 ],
			0x1D606 => [ 0x79 ],
			0x1D607 => [ 0x7A ],
			0x1D608 => [ 0x61 ],
			0x1D609 => [ 0x62 ],
			0x1D60A => [ 0x63 ],
			0x1D60B => [ 0x64 ],
			0x1D60C => [ 0x65 ],
			0x1D60D => [ 0x66 ],
			0x1D60E => [ 0x67 ],
			0x1D60F => [ 0x68 ],
			0x1D610 => [ 0x69 ],
			0x1D611 => [ 0x6A ],
			0x1D612 => [ 0x6B ],
			0x1D613 => [ 0x6C ],
			0x1D614 => [ 0x6D ],
			0x1D615 => [ 0x6E ],
			0x1D616 => [ 0x6F ],
			0x1D617 => [ 0x70 ],
			0x1D618 => [ 0x71 ],
			0x1D619 => [ 0x72 ],
			0x1D61A => [ 0x73 ],
			0x1D61B => [ 0x74 ],
			0x1D61C => [ 0x75 ],
			0x1D61D => [ 0x76 ],
			0x1D61E => [ 0x77 ],
			0x1D61F => [ 0x78 ],
			0x1D620 => [ 0x79 ],
			0x1D621 => [ 0x7A ],
			0x1D622 => [ 0x61 ],
			0x1D623 => [ 0x62 ],
			0x1D624 => [ 0x63 ],
			0x1D625 => [ 0x64 ],
			0x1D626 => [ 0x65 ],
			0x1D627 => [ 0x66 ],
			0x1D628 => [ 0x67 ],
			0x1D629 => [ 0x68 ],
			0x1D62A => [ 0x69 ],
			0x1D62B => [ 0x6A ],
			0x1D62C => [ 0x6B ],
			0x1D62D => [ 0x6C ],
			0x1D62E => [ 0x6D ],
			0x1D62F => [ 0x6E ],
			0x1D630 => [ 0x6F ],
			0x1D631 => [ 0x70 ],
			0x1D632 => [ 0x71 ],
			0x1D633 => [ 0x72 ],
			0x1D634 => [ 0x73 ],
			0x1D635 => [ 0x74 ],
			0x1D636 => [ 0x75 ],
			0x1D637 => [ 0x76 ],
			0x1D638 => [ 0x77 ],
			0x1D639 => [ 0x78 ],
			0x1D63A => [ 0x79 ],
			0x1D63B => [ 0x7A ],
			0x1D63C => [ 0x61 ],
			0x1D63D => [ 0x62 ],
			0x1D63E => [ 0x63 ],
			0x1D63F => [ 0x64 ],
			0x1D640 => [ 0x65 ],
			0x1D641 => [ 0x66 ],
			0x1D642 => [ 0x67 ],
			0x1D643 => [ 0x68 ],
			0x1D644 => [ 0x69 ],
			0x1D645 => [ 0x6A ],
			0x1D646 => [ 0x6B ],
			0x1D647 => [ 0x6C ],
			0x1D648 => [ 0x6D ],
			0x1D649 => [ 0x6E ],
			0x1D64A => [ 0x6F ],
			0x1D64B => [ 0x70 ],
			0x1D64C => [ 0x71 ],
			0x1D64D => [ 0x72 ],
			0x1D64E => [ 0x73 ],
			0x1D64F => [ 0x74 ],
			0x1D650 => [ 0x75 ],
			0x1D651 => [ 0x76 ],
			0x1D652 => [ 0x77 ],
			0x1D653 => [ 0x78 ],
			0x1D654 => [ 0x79 ],
			0x1D655 => [ 0x7A ],
			0x1D656 => [ 0x61 ],
			0x1D657 => [ 0x62 ],
			0x1D658 => [ 0x63 ],
			0x1D659 => [ 0x64 ],
			0x1D65A => [ 0x65 ],
			0x1D65B => [ 0x66 ],
			0x1D65C => [ 0x67 ],
			0x1D65D => [ 0x68 ],
			0x1D65E => [ 0x69 ],
			0x1D65F => [ 0x6A ],
			0x1D660 => [ 0x6B ],
			0x1D661 => [ 0x6C ],
			0x1D662 => [ 0x6D ],
			0x1D663 => [ 0x6E ],
			0x1D664 => [ 0x6F ],
			0x1D665 => [ 0x70 ],
			0x1D666 => [ 0x71 ],
			0x1D667 => [ 0x72 ],
			0x1D668 => [ 0x73 ],
			0x1D669 => [ 0x74 ],
			0x1D66A => [ 0x75 ],
			0x1D66B => [ 0x76 ],
			0x1D66C => [ 0x77 ],
			0x1D66D => [ 0x78 ],
			0x1D66E => [ 0x79 ],
			0x1D66F => [ 0x7A ],
			0x1D670 => [ 0x61 ],
			0x1D671 => [ 0x62 ],
			0x1D672 => [ 0x63 ],
			0x1D673 => [ 0x64 ],
			0x1D674 => [ 0x65 ],
			0x1D675 => [ 0x66 ],
			0x1D676 => [ 0x67 ],
			0x1D677 => [ 0x68 ],
			0x1D678 => [ 0x69 ],
			0x1D679 => [ 0x6A ],
			0x1D67A => [ 0x6B ],
			0x1D67B => [ 0x6C ],
			0x1D67C => [ 0x6D ],
			0x1D67D => [ 0x6E ],
			0x1D67E => [ 0x6F ],
			0x1D67F => [ 0x70 ],
			0x1D680 => [ 0x71 ],
			0x1D681 => [ 0x72 ],
			0x1D682 => [ 0x73 ],
			0x1D683 => [ 0x74 ],
			0x1D684 => [ 0x75 ],
			0x1D685 => [ 0x76 ],
			0x1D686 => [ 0x77 ],
			0x1D687 => [ 0x78 ],
			0x1D688 => [ 0x79 ],
			0x1D689 => [ 0x7A ],
			0x1D68A => [ 0x61 ],
			0x1D68B => [ 0x62 ],
			0x1D68C => [ 0x63 ],
			0x1D68D => [ 0x64 ],
			0x1D68E => [ 0x65 ],
			0x1D68F => [ 0x66 ],
			0x1D690 => [ 0x67 ],
			0x1D691 => [ 0x68 ],
			0x1D692 => [ 0x69 ],
			0x1D693 => [ 0x6A ],
			0x1D694 => [ 0x6B ],
			0x1D695 => [ 0x6C ],
			0x1D696 => [ 0x6D ],
			0x1D697 => [ 0x6E ],
			0x1D698 => [ 0x6F ],
			0x1D699 => [ 0x70 ],
			0x1D69A => [ 0x71 ],
			0x1D69B => [ 0x72 ],
			0x1D69C => [ 0x73 ],
			0x1D69D => [ 0x74 ],
			0x1D69E => [ 0x75 ],
			0x1D69F => [ 0x76 ],
			0x1D6A0 => [ 0x77 ],
			0x1D6A1 => [ 0x78 ],
			0x1D6A2 => [ 0x79 ],
			0x1D6A3 => [ 0x7A ],
			0x1D6A4 => [ 0x131 ],
			0x1D6A5 => [ 0x237 ],
			0x1D6A8 => [ 0x3B1 ],
			0x1D6A9 => [ 0x3B2 ],
			0x1D6AA => [ 0x3B3 ],
			0x1D6AB => [ 0x3B4 ],
			0x1D6AC => [ 0x3B5 ],
			0x1D6AD => [ 0x3B6 ],
			0x1D6AE => [ 0x3B7 ],
			0x1D6AF => [ 0x3B8 ],
			0x1D6B0 => [ 0x3B9 ],
			0x1D6B1 => [ 0x3BA ],
			0x1D6B2 => [ 0x3BB ],
			0x1D6B3 => [ 0x3BC ],
			0x1D6B4 => [ 0x3BD ],
			0x1D6B5 => [ 0x3BE ],
			0x1D6B6 => [ 0x3BF ],
			0x1D6B7 => [ 0x3C0 ],
			0x1D6B8 => [ 0x3C1 ],
			0x1D6B9 => [ 0x3B8 ],
			0x1D6BA => [ 0x3C3 ],
			0x1D6BB => [ 0x3C4 ],
			0x1D6BC => [ 0x3C5 ],
			0x1D6BD => [ 0x3C6 ],
			0x1D6BE => [ 0x3C7 ],
			0x1D6BF => [ 0x3C8 ],
			0x1D6C0 => [ 0x3C9 ],
			0x1D6C1 => [ 0x2207 ],
			0x1D6C2 => [ 0x3B1 ],
			0x1D6C3 => [ 0x3B2 ],
			0x1D6C4 => [ 0x3B3 ],
			0x1D6C5 => [ 0x3B4 ],
			0x1D6C6 => [ 0x3B5 ],
			0x1D6C7 => [ 0x3B6 ],
			0x1D6C8 => [ 0x3B7 ],
			0x1D6C9 => [ 0x3B8 ],
			0x1D6CA => [ 0x3B9 ],
			0x1D6CB => [ 0x3BA ],
			0x1D6CC => [ 0x3BB ],
			0x1D6CD => [ 0x3BC ],
			0x1D6CE => [ 0x3BD ],
			0x1D6CF => [ 0x3BE ],
			0x1D6D0 => [ 0x3BF ],
			0x1D6D1 => [ 0x3C0 ],
			0x1D6D2 => [ 0x3C1 ],
			0x1D6D3 => [ 0x3C3 ],
			0x1D6D4 => [ 0x3C3 ],
			0x1D6D5 => [ 0x3C4 ],
			0x1D6D6 => [ 0x3C5 ],
			0x1D6D7 => [ 0x3C6 ],
			0x1D6D8 => [ 0x3C7 ],
			0x1D6D9 => [ 0x3C8 ],
			0x1D6DA => [ 0x3C9 ],
			0x1D6DB => [ 0x2202 ],
			0x1D6DC => [ 0x3B5 ],
			0x1D6DD => [ 0x3B8 ],
			0x1D6DE => [ 0x3BA ],
			0x1D6DF => [ 0x3C6 ],
			0x1D6E0 => [ 0x3C1 ],
			0x1D6E1 => [ 0x3C0 ],
			0x1D6E2 => [ 0x3B1 ],
			0x1D6E3 => [ 0x3B2 ],
			0x1D6E4 => [ 0x3B3 ],
			0x1D6E5 => [ 0x3B4 ],
			0x1D6E6 => [ 0x3B5 ],
			0x1D6E7 => [ 0x3B6 ],
			0x1D6E8 => [ 0x3B7 ],
			0x1D6E9 => [ 0x3B8 ],
			0x1D6EA => [ 0x3B9 ],
			0x1D6EB => [ 0x3BA ],
			0x1D6EC => [ 0x3BB ],
			0x1D6ED => [ 0x3BC ],
			0x1D6EE => [ 0x3BD ],
			0x1D6EF => [ 0x3BE ],
			0x1D6F0 => [ 0x3BF ],
			0x1D6F1 => [ 0x3C0 ],
			0x1D6F2 => [ 0x3C1 ],
			0x1D6F3 => [ 0x3B8 ],
			0x1D6F4 => [ 0x3C3 ],
			0x1D6F5 => [ 0x3C4 ],
			0x1D6F6 => [ 0x3C5 ],
			0x1D6F7 => [ 0x3C6 ],
			0x1D6F8 => [ 0x3C7 ],
			0x1D6F9 => [ 0x3C8 ],
			0x1D6FA => [ 0x3C9 ],
			0x1D6FB => [ 0x2207 ],
			0x1D6FC => [ 0x3B1 ],
			0x1D6FD => [ 0x3B2 ],
			0x1D6FE => [ 0x3B3 ],
			0x1D6FF => [ 0x3B4 ],
			0x1D700 => [ 0x3B5 ],
			0x1D701 => [ 0x3B6 ],
			0x1D702 => [ 0x3B7 ],
			0x1D703 => [ 0x3B8 ],
			0x1D704 => [ 0x3B9 ],
			0x1D705 => [ 0x3BA ],
			0x1D706 => [ 0x3BB ],
			0x1D707 => [ 0x3BC ],
			0x1D708 => [ 0x3BD ],
			0x1D709 => [ 0x3BE ],
			0x1D70A => [ 0x3BF ],
			0x1D70B => [ 0x3C0 ],
			0x1D70C => [ 0x3C1 ],
			0x1D70D => [ 0x3C3 ],
			0x1D70E => [ 0x3C3 ],
			0x1D70F => [ 0x3C4 ],
			0x1D710 => [ 0x3C5 ],
			0x1D711 => [ 0x3C6 ],
			0x1D712 => [ 0x3C7 ],
			0x1D713 => [ 0x3C8 ],
			0x1D714 => [ 0x3C9 ],
			0x1D715 => [ 0x2202 ],
			0x1D716 => [ 0x3B5 ],
			0x1D717 => [ 0x3B8 ],
			0x1D718 => [ 0x3BA ],
			0x1D719 => [ 0x3C6 ],
			0x1D71A => [ 0x3C1 ],
			0x1D71B => [ 0x3C0 ],
			0x1D71C => [ 0x3B1 ],
			0x1D71D => [ 0x3B2 ],
			0x1D71E => [ 0x3B3 ],
			0x1D71F => [ 0x3B4 ],
			0x1D720 => [ 0x3B5 ],
			0x1D721 => [ 0x3B6 ],
			0x1D722 => [ 0x3B7 ],
			0x1D723 => [ 0x3B8 ],
			0x1D724 => [ 0x3B9 ],
			0x1D725 => [ 0x3BA ],
			0x1D726 => [ 0x3BB ],
			0x1D727 => [ 0x3BC ],
			0x1D728 => [ 0x3BD ],
			0x1D729 => [ 0x3BE ],
			0x1D72A => [ 0x3BF ],
			0x1D72B => [ 0x3C0 ],
			0x1D72C => [ 0x3C1 ],
			0x1D72D => [ 0x3B8 ],
			0x1D72E => [ 0x3C3 ],
			0x1D72F => [ 0x3C4 ],
			0x1D730 => [ 0x3C5 ],
			0x1D731 => [ 0x3C6 ],
			0x1D732 => [ 0x3C7 ],
			0x1D733 => [ 0x3C8 ],
			0x1D734 => [ 0x3C9 ],
			0x1D735 => [ 0x2207 ],
			0x1D736 => [ 0x3B1 ],
			0x1D737 => [ 0x3B2 ],
			0x1D738 => [ 0x3B3 ],
			0x1D739 => [ 0x3B4 ],
			0x1D73A => [ 0x3B5 ],
			0x1D73B => [ 0x3B6 ],
			0x1D73C => [ 0x3B7 ],
			0x1D73D => [ 0x3B8 ],
			0x1D73E => [ 0x3B9 ],
			0x1D73F => [ 0x3BA ],
			0x1D740 => [ 0x3BB ],
			0x1D741 => [ 0x3BC ],
			0x1D742 => [ 0x3BD ],
			0x1D743 => [ 0x3BE ],
			0x1D744 => [ 0x3BF ],
			0x1D745 => [ 0x3C0 ],
			0x1D746 => [ 0x3C1 ],
			0x1D747 => [ 0x3C3 ],
			0x1D748 => [ 0x3C3 ],
			0x1D749 => [ 0x3C4 ],
			0x1D74A => [ 0x3C5 ],
			0x1D74B => [ 0x3C6 ],
			0x1D74C => [ 0x3C7 ],
			0x1D74D => [ 0x3C8 ],
			0x1D74E => [ 0x3C9 ],
			0x1D74F => [ 0x2202 ],
			0x1D750 => [ 0x3B5 ],
			0x1D751 => [ 0x3B8 ],
			0x1D752 => [ 0x3BA ],
			0x1D753 => [ 0x3C6 ],
			0x1D754 => [ 0x3C1 ],
			0x1D755 => [ 0x3C0 ],
			0x1D756 => [ 0x3B1 ],
			0x1D757 => [ 0x3B2 ],
			0x1D758 => [ 0x3B3 ],
			0x1D759 => [ 0x3B4 ],
			0x1D75A => [ 0x3B5 ],
			0x1D75B => [ 0x3B6 ],
			0x1D75C => [ 0x3B7 ],
			0x1D75D => [ 0x3B8 ],
			0x1D75E => [ 0x3B9 ],
			0x1D75F => [ 0x3BA ],
			0x1D760 => [ 0x3BB ],
			0x1D761 => [ 0x3BC ],
			0x1D762 => [ 0x3BD ],
			0x1D763 => [ 0x3BE ],
			0x1D764 => [ 0x3BF ],
			0x1D765 => [ 0x3C0 ],
			0x1D766 => [ 0x3C1 ],
			0x1D767 => [ 0x3B8 ],
			0x1D768 => [ 0x3C3 ],
			0x1D769 => [ 0x3C4 ],
			0x1D76A => [ 0x3C5 ],
			0x1D76B => [ 0x3C6 ],
			0x1D76C => [ 0x3C7 ],
			0x1D76D => [ 0x3C8 ],
			0x1D76E => [ 0x3C9 ],
			0x1D76F => [ 0x2207 ],
			0x1D770 => [ 0x3B1 ],
			0x1D771 => [ 0x3B2 ],
			0x1D772 => [ 0x3B3 ],
			0x1D773 => [ 0x3B4 ],
			0x1D774 => [ 0x3B5 ],
			0x1D775 => [ 0x3B6 ],
			0x1D776 => [ 0x3B7 ],
			0x1D777 => [ 0x3B8 ],
			0x1D778 => [ 0x3B9 ],
			0x1D779 => [ 0x3BA ],
			0x1D77A => [ 0x3BB ],
			0x1D77B => [ 0x3BC ],
			0x1D77C => [ 0x3BD ],
			0x1D77D => [ 0x3BE ],
			0x1D77E => [ 0x3BF ],
			0x1D77F => [ 0x3C0 ],
			0x1D780 => [ 0x3C1 ],
			0x1D781 => [ 0x3C3 ],
			0x1D782 => [ 0x3C3 ],
			0x1D783 => [ 0x3C4 ],
			0x1D784 => [ 0x3C5 ],
			0x1D785 => [ 0x3C6 ],
			0x1D786 => [ 0x3C7 ],
			0x1D787 => [ 0x3C8 ],
			0x1D788 => [ 0x3C9 ],
			0x1D789 => [ 0x2202 ],
			0x1D78A => [ 0x3B5 ],
			0x1D78B => [ 0x3B8 ],
			0x1D78C => [ 0x3BA ],
			0x1D78D => [ 0x3C6 ],
			0x1D78E => [ 0x3C1 ],
			0x1D78F => [ 0x3C0 ],
			0x1D790 => [ 0x3B1 ],
			0x1D791 => [ 0x3B2 ],
			0x1D792 => [ 0x3B3 ],
			0x1D793 => [ 0x3B4 ],
			0x1D794 => [ 0x3B5 ],
			0x1D795 => [ 0x3B6 ],
			0x1D796 => [ 0x3B7 ],
			0x1D797 => [ 0x3B8 ],
			0x1D798 => [ 0x3B9 ],
			0x1D799 => [ 0x3BA ],
			0x1D79A => [ 0x3BB ],
			0x1D79B => [ 0x3BC ],
			0x1D79C => [ 0x3BD ],
			0x1D79D => [ 0x3BE ],
			0x1D79E => [ 0x3BF ],
			0x1D79F => [ 0x3C0 ],
			0x1D7A0 => [ 0x3C1 ],
			0x1D7A1 => [ 0x3B8 ],
			0x1D7A2 => [ 0x3C3 ],
			0x1D7A3 => [ 0x3C4 ],
			0x1D7A4 => [ 0x3C5 ],
			0x1D7A5 => [ 0x3C6 ],
			0x1D7A6 => [ 0x3C7 ],
			0x1D7A7 => [ 0x3C8 ],
			0x1D7A8 => [ 0x3C9 ],
			0x1D7A9 => [ 0x2207 ],
			0x1D7AA => [ 0x3B1 ],
			0x1D7AB => [ 0x3B2 ],
			0x1D7AC => [ 0x3B3 ],
			0x1D7AD => [ 0x3B4 ],
			0x1D7AE => [ 0x3B5 ],
			0x1D7AF => [ 0x3B6 ],
			0x1D7B0 => [ 0x3B7 ],
			0x1D7B1 => [ 0x3B8 ],
			0x1D7B2 => [ 0x3B9 ],
			0x1D7B3 => [ 0x3BA ],
			0x1D7B4 => [ 0x3BB ],
			0x1D7B5 => [ 0x3BC ],
			0x1D7B6 => [ 0x3BD ],
			0x1D7B7 => [ 0x3BE ],
			0x1D7B8 => [ 0x3BF ],
			0x1D7B9 => [ 0x3C0 ],
			0x1D7BA => [ 0x3C1 ],
			0x1D7BB => [ 0x3C3 ],
			0x1D7BC => [ 0x3C3 ],
			0x1D7BD => [ 0x3C4 ],
			0x1D7BE => [ 0x3C5 ],
			0x1D7BF => [ 0x3C6 ],
			0x1D7C0 => [ 0x3C7 ],
			0x1D7C1 => [ 0x3C8 ],
			0x1D7C2 => [ 0x3C9 ],
			0x1D7C3 => [ 0x2202 ],
			0x1D7C4 => [ 0x3B5 ],
			0x1D7C5 => [ 0x3B8 ],
			0x1D7C6 => [ 0x3BA ],
			0x1D7C7 => [ 0x3C6 ],
			0x1D7C8 => [ 0x3C1 ],
			0x1D7C9 => [ 0x3C0 ],
			0x1D7CA => [ 0x3DD ],
			0x1D7CB => [ 0x3DD ],
			0x1D7CE => [ 0x30 ],
			0x1D7CF => [ 0x31 ],
			0x1D7D0 => [ 0x32 ],
			0x1D7D1 => [ 0x33 ],
			0x1D7D2 => [ 0x34 ],
			0x1D7D3 => [ 0x35 ],
			0x1D7D4 => [ 0x36 ],
			0x1D7D5 => [ 0x37 ],
			0x1D7D6 => [ 0x38 ],
			0x1D7D7 => [ 0x39 ],
			0x1D7D8 => [ 0x30 ],
			0x1D7D9 => [ 0x31 ],
			0x1D7DA => [ 0x32 ],
			0x1D7DB => [ 0x33 ],
			0x1D7DC => [ 0x34 ],
			0x1D7DD => [ 0x35 ],
			0x1D7DE => [ 0x36 ],
			0x1D7DF => [ 0x37 ],
			0x1D7E0 => [ 0x38 ],
			0x1D7E1 => [ 0x39 ],
			0x1D7E2 => [ 0x30 ],
			0x1D7E3 => [ 0x31 ],
			0x1D7E4 => [ 0x32 ],
			0x1D7E5 => [ 0x33 ],
			0x1D7E6 => [ 0x34 ],
			0x1D7E7 => [ 0x35 ],
			0x1D7E8 => [ 0x36 ],
			0x1D7E9 => [ 0x37 ],
			0x1D7EA => [ 0x38 ],
			0x1D7EB => [ 0x39 ],
			0x1D7EC => [ 0x30 ],
			0x1D7ED => [ 0x31 ],
			0x1D7EE => [ 0x32 ],
			0x1D7EF => [ 0x33 ],
			0x1D7F0 => [ 0x34 ],
			0x1D7F1 => [ 0x35 ],
			0x1D7F2 => [ 0x36 ],
			0x1D7F3 => [ 0x37 ],
			0x1D7F4 => [ 0x38 ],
			0x1D7F5 => [ 0x39 ],
			0x1D7F6 => [ 0x30 ],
			0x1D7F7 => [ 0x31 ],
			0x1D7F8 => [ 0x32 ],
			0x1D7F9 => [ 0x33 ],
			0x1D7FA => [ 0x34 ],
			0x1D7FB => [ 0x35 ],
			0x1D7FC => [ 0x36 ],
			0x1D7FD => [ 0x37 ],
			0x1D7FE => [ 0x38 ],
			0x1D7FF => [ 0x39 ],
			0x1EE00 => [ 0x627 ],
			0x1EE01 => [ 0x628 ],
			0x1EE02 => [ 0x62C ],
			0x1EE03 => [ 0x62F ],
			0x1EE05 => [ 0x648 ],
			0x1EE06 => [ 0x632 ],
			0x1EE07 => [ 0x62D ],
			0x1EE08 => [ 0x637 ],
			0x1EE09 => [ 0x64A ],
			0x1EE0A => [ 0x643 ],
			0x1EE0B => [ 0x644 ],
			0x1EE0C => [ 0x645 ],
			0x1EE0D => [ 0x646 ],
			0x1EE0E => [ 0x633 ],
			0x1EE0F => [ 0x639 ],
			0x1EE10 => [ 0x641 ],
			0x1EE11 => [ 0x635 ],
			0x1EE12 => [ 0x642 ],
			0x1EE13 => [ 0x631 ],
			0x1EE14 => [ 0x634 ],
			0x1EE15 => [ 0x62A ],
			0x1EE16 => [ 0x62B ],
			0x1EE17 => [ 0x62E ],
			0x1EE18 => [ 0x630 ],
			0x1EE19 => [ 0x636 ],
			0x1EE1A => [ 0x638 ],
			0x1EE1B => [ 0x63A ],
			0x1EE1C => [ 0x66E ],
			0x1EE1D => [ 0x6BA ],
			0x1EE1E => [ 0x6A1 ],
			0x1EE1F => [ 0x66F ],
			0x1EE21 => [ 0x628 ],
			0x1EE22 => [ 0x62C ],
			0x1EE24 => [ 0x647 ],
			0x1EE27 => [ 0x62D ],
			0x1EE29 => [ 0x64A ],
			0x1EE2A => [ 0x643 ],
			0x1EE2B => [ 0x644 ],
			0x1EE2C => [ 0x645 ],
			0x1EE2D => [ 0x646 ],
			0x1EE2E => [ 0x633 ],
			0x1EE2F => [ 0x639 ],
			0x1EE30 => [ 0x641 ],
			0x1EE31 => [ 0x635 ],
			0x1EE32 => [ 0x642 ],
			0x1EE34 => [ 0x634 ],
			0x1EE35 => [ 0x62A ],
			0x1EE36 => [ 0x62B ],
			0x1EE37 => [ 0x62E ],
			0x1EE39 => [ 0x636 ],
			0x1EE3B => [ 0x63A ],
			0x1EE42 => [ 0x62C ],
			0x1EE47 => [ 0x62D ],
			0x1EE49 => [ 0x64A ],
			0x1EE4B => [ 0x644 ],
			0x1EE4D => [ 0x646 ],
			0x1EE4E => [ 0x633 ],
			0x1EE4F => [ 0x639 ],
			0x1EE51 => [ 0x635 ],
			0x1EE52 => [ 0x642 ],
			0x1EE54 => [ 0x634 ],
			0x1EE57 => [ 0x62E ],
			0x1EE59 => [ 0x636 ],
			0x1EE5B => [ 0x63A ],
			0x1EE5D => [ 0x6BA ],
			0x1EE5F => [ 0x66F ],
			0x1EE61 => [ 0x628 ],
			0x1EE62 => [ 0x62C ],
			0x1EE64 => [ 0x647 ],
			0x1EE67 => [ 0x62D ],
			0x1EE68 => [ 0x637 ],
			0x1EE69 => [ 0x64A ],
			0x1EE6A => [ 0x643 ],
			0x1EE6C => [ 0x645 ],
			0x1EE6D => [ 0x646 ],
			0x1EE6E => [ 0x633 ],
			0x1EE6F => [ 0x639 ],
			0x1EE70 => [ 0x641 ],
			0x1EE71 => [ 0x635 ],
			0x1EE72 => [ 0x642 ],
			0x1EE74 => [ 0x634 ],
			0x1EE75 => [ 0x62A ],
			0x1EE76 => [ 0x62B ],
			0x1EE77 => [ 0x62E ],
			0x1EE79 => [ 0x636 ],
			0x1EE7A => [ 0x638 ],
			0x1EE7B => [ 0x63A ],
			0x1EE7C => [ 0x66E ],
			0x1EE7E => [ 0x6A1 ],
			0x1EE80 => [ 0x627 ],
			0x1EE81 => [ 0x628 ],
			0x1EE82 => [ 0x62C ],
			0x1EE83 => [ 0x62F ],
			0x1EE84 => [ 0x647 ],
			0x1EE85 => [ 0x648 ],
			0x1EE86 => [ 0x632 ],
			0x1EE87 => [ 0x62D ],
			0x1EE88 => [ 0x637 ],
			0x1EE89 => [ 0x64A ],
			0x1EE8B => [ 0x644 ],
			0x1EE8C => [ 0x645 ],
			0x1EE8D => [ 0x646 ],
			0x1EE8E => [ 0x633 ],
			0x1EE8F => [ 0x639 ],
			0x1EE90 => [ 0x641 ],
			0x1EE91 => [ 0x635 ],
			0x1EE92 => [ 0x642 ],
			0x1EE93 => [ 0x631 ],
			0x1EE94 => [ 0x634 ],
			0x1EE95 => [ 0x62A ],
			0x1EE96 => [ 0x62B ],
			0x1EE97 => [ 0x62E ],
			0x1EE98 => [ 0x630 ],
			0x1EE99 => [ 0x636 ],
			0x1EE9A => [ 0x638 ],
			0x1EE9B => [ 0x63A ],
			0x1EEA1 => [ 0x628 ],
			0x1EEA2 => [ 0x62C ],
			0x1EEA3 => [ 0x62F ],
			0x1EEA5 => [ 0x648 ],
			0x1EEA6 => [ 0x632 ],
			0x1EEA7 => [ 0x62D ],
			0x1EEA8 => [ 0x637 ],
			0x1EEA9 => [ 0x64A ],
			0x1EEAB => [ 0x644 ],
			0x1EEAC => [ 0x645 ],
			0x1EEAD => [ 0x646 ],
			0x1EEAE => [ 0x633 ],
			0x1EEAF => [ 0x639 ],
			0x1EEB0 => [ 0x641 ],
			0x1EEB1 => [ 0x635 ],
			0x1EEB2 => [ 0x642 ],
			0x1EEB3 => [ 0x631 ],
			0x1EEB4 => [ 0x634 ],
			0x1EEB5 => [ 0x62A ],
			0x1EEB6 => [ 0x62B ],
			0x1EEB7 => [ 0x62E ],
			0x1EEB8 => [ 0x630 ],
			0x1EEB9 => [ 0x636 ],
			0x1EEBA => [ 0x638 ],
			0x1EEBB => [ 0x63A ],
			0x1F12A => [ 0x3014, 0x73, 0x3015 ],
			0x1F12B => [ 0x63 ],
			0x1F12C => [ 0x72 ],
			0x1F12D => [ 0x63, 0x64 ],
			0x1F12E => [ 0x77, 0x7A ],
			0x1F130 => [ 0x61 ],
			0x1F131 => [ 0x62 ],
			0x1F132 => [ 0x63 ],
			0x1F133 => [ 0x64 ],
			0x1F134 => [ 0x65 ],
			0x1F135 => [ 0x66 ],
			0x1F136 => [ 0x67 ],
			0x1F137 => [ 0x68 ],
			0x1F138 => [ 0x69 ],
			0x1F139 => [ 0x6A ],
			0x1F13A => [ 0x6B ],
			0x1F13B => [ 0x6C ],
			0x1F13C => [ 0x6D ],
			0x1F13D => [ 0x6E ],
			0x1F13E => [ 0x6F ],
			0x1F13F => [ 0x70 ],
			0x1F140 => [ 0x71 ],
			0x1F141 => [ 0x72 ],
			0x1F142 => [ 0x73 ],
			0x1F143 => [ 0x74 ],
			0x1F144 => [ 0x75 ],
			0x1F145 => [ 0x76 ],
			0x1F146 => [ 0x77 ],
			0x1F147 => [ 0x78 ],
			0x1F148 => [ 0x79 ],
			0x1F149 => [ 0x7A ],
			0x1F14A => [ 0x68, 0x76 ],
			0x1F14B => [ 0x6D, 0x76 ],
			0x1F14C => [ 0x73, 0x64 ],
			0x1F14D => [ 0x73, 0x73 ],
			0x1F14E => [ 0x70, 0x70, 0x76 ],
			0x1F14F => [ 0x77, 0x63 ],
			0x1F16A => [ 0x6D, 0x63 ],
			0x1F16B => [ 0x6D, 0x64 ],
			0x1F190 => [ 0x64, 0x6A ],
			0x1F200 => [ 0x307B, 0x304B ],
			0x1F201 => [ 0x30B3, 0x30B3 ],
			0x1F202 => [ 0x30B5 ],
			0x1F210 => [ 0x624B ],
			0x1F211 => [ 0x5B57 ],
			0x1F212 => [ 0x53CC ],
			0x1F213 => [ 0x30C7 ],
			0x1F214 => [ 0x4E8C ],
			0x1F215 => [ 0x591A ],
			0x1F216 => [ 0x89E3 ],
			0x1F217 => [ 0x5929 ],
			0x1F218 => [ 0x4EA4 ],
			0x1F219 => [ 0x6620 ],
			0x1F21A => [ 0x7121 ],
			0x1F21B => [ 0x6599 ],
			0x1F21C => [ 0x524D ],
			0x1F21D => [ 0x5F8C ],
			0x1F21E => [ 0x518D ],
			0x1F21F => [ 0x65B0 ],
			0x1F220 => [ 0x521D ],
			0x1F221 => [ 0x7D42 ],
			0x1F222 => [ 0x751F ],
			0x1F223 => [ 0x8CA9 ],
			0x1F224 => [ 0x58F0 ],
			0x1F225 => [ 0x5439 ],
			0x1F226 => [ 0x6F14 ],
			0x1F227 => [ 0x6295 ],
			0x1F228 => [ 0x6355 ],
			0x1F229 => [ 0x4E00 ],
			0x1F22A => [ 0x4E09 ],
			0x1F22B => [ 0x904A ],
			0x1F22C => [ 0x5DE6 ],
			0x1F22D => [ 0x4E2D ],
			0x1F22E => [ 0x53F3 ],
			0x1F22F => [ 0x6307 ],
			0x1F230 => [ 0x8D70 ],
			0x1F231 => [ 0x6253 ],
			0x1F232 => [ 0x7981 ],
			0x1F233 => [ 0x7A7A ],
			0x1F234 => [ 0x5408 ],
			0x1F235 => [ 0x6E80 ],
			0x1F236 => [ 0x6709 ],
			0x1F237 => [ 0x6708 ],
			0x1F238 => [ 0x7533 ],
			0x1F239 => [ 0x5272 ],
			0x1F23A => [ 0x55B6 ],
			0x1F240 => [ 0x3014, 0x672C, 0x3015 ],
			0x1F241 => [ 0x3014, 0x4E09, 0x3015 ],
			0x1F242 => [ 0x3014, 0x4E8C, 0x3015 ],
			0x1F243 => [ 0x3014, 0x5B89, 0x3015 ],
			0x1F244 => [ 0x3014, 0x70B9, 0x3015 ],
			0x1F245 => [ 0x3014, 0x6253, 0x3015 ],
			0x1F246 => [ 0x3014, 0x76D7, 0x3015 ],
			0x1F247 => [ 0x3014, 0x52DD, 0x3015 ],
			0x1F248 => [ 0x3014, 0x6557, 0x3015 ],
			0x1F250 => [ 0x5F97 ],
			0x1F251 => [ 0x53EF ],
			0x2F800 => [ 0x4E3D ],
			0x2F801 => [ 0x4E38 ],
			0x2F802 => [ 0x4E41 ],
			0x2F803 => [ 0x20122 ],
			0x2F804 => [ 0x4F60 ],
			0x2F805 => [ 0x4FAE ],
			0x2F806 => [ 0x4FBB ],
			0x2F807 => [ 0x5002 ],
			0x2F808 => [ 0x507A ],
			0x2F809 => [ 0x5099 ],
			0x2F80A => [ 0x50E7 ],
			0x2F80B => [ 0x50CF ],
			0x2F80C => [ 0x349E ],
			0x2F80D => [ 0x2063A ],
			0x2F80E => [ 0x514D ],
			0x2F80F => [ 0x5154 ],
			0x2F810 => [ 0x5164 ],
			0x2F811 => [ 0x5177 ],
			0x2F812 => [ 0x2051C ],
			0x2F813 => [ 0x34B9 ],
			0x2F814 => [ 0x5167 ],
			0x2F815 => [ 0x518D ],
			0x2F816 => [ 0x2054B ],
			0x2F817 => [ 0x5197 ],
			0x2F818 => [ 0x51A4 ],
			0x2F819 => [ 0x4ECC ],
			0x2F81A => [ 0x51AC ],
			0x2F81B => [ 0x51B5 ],
			0x2F81C => [ 0x291DF ],
			0x2F81D => [ 0x51F5 ],
			0x2F81E => [ 0x5203 ],
			0x2F81F => [ 0x34DF ],
			0x2F820 => [ 0x523B ],
			0x2F821 => [ 0x5246 ],
			0x2F822 => [ 0x5272 ],
			0x2F823 => [ 0x5277 ],
			0x2F824 => [ 0x3515 ],
			0x2F825 => [ 0x52C7 ],
			0x2F826 => [ 0x52C9 ],
			0x2F827 => [ 0x52E4 ],
			0x2F828 => [ 0x52FA ],
			0x2F829 => [ 0x5305 ],
			0x2F82A => [ 0x5306 ],
			0x2F82B => [ 0x5317 ],
			0x2F82C => [ 0x5349 ],
			0x2F82D => [ 0x5351 ],
			0x2F82E => [ 0x535A ],
			0x2F82F => [ 0x5373 ],
			0x2F830 => [ 0x537D ],
			0x2F831 => [ 0x537F ],
			0x2F832 => [ 0x537F ],
			0x2F833 => [ 0x537F ],
			0x2F834 => [ 0x20A2C ],
			0x2F835 => [ 0x7070 ],
			0x2F836 => [ 0x53CA ],
			0x2F837 => [ 0x53DF ],
			0x2F838 => [ 0x20B63 ],
			0x2F839 => [ 0x53EB ],
			0x2F83A => [ 0x53F1 ],
			0x2F83B => [ 0x5406 ],
			0x2F83C => [ 0x549E ],
			0x2F83D => [ 0x5438 ],
			0x2F83E => [ 0x5448 ],
			0x2F83F => [ 0x5468 ],
			0x2F840 => [ 0x54A2 ],
			0x2F841 => [ 0x54F6 ],
			0x2F842 => [ 0x5510 ],
			0x2F843 => [ 0x5553 ],
			0x2F844 => [ 0x5563 ],
			0x2F845 => [ 0x5584 ],
			0x2F846 => [ 0x5584 ],
			0x2F847 => [ 0x5599 ],
			0x2F848 => [ 0x55AB ],
			0x2F849 => [ 0x55B3 ],
			0x2F84A => [ 0x55C2 ],
			0x2F84B => [ 0x5716 ],
			0x2F84C => [ 0x5606 ],
			0x2F84D => [ 0x5717 ],
			0x2F84E => [ 0x5651 ],
			0x2F84F => [ 0x5674 ],
			0x2F850 => [ 0x5207 ],
			0x2F851 => [ 0x58EE ],
			0x2F852 => [ 0x57CE ],
			0x2F853 => [ 0x57F4 ],
			0x2F854 => [ 0x580D ],
			0x2F855 => [ 0x578B ],
			0x2F856 => [ 0x5832 ],
			0x2F857 => [ 0x5831 ],
			0x2F858 => [ 0x58AC ],
			0x2F859 => [ 0x214E4 ],
			0x2F85A => [ 0x58F2 ],
			0x2F85B => [ 0x58F7 ],
			0x2F85C => [ 0x5906 ],
			0x2F85D => [ 0x591A ],
			0x2F85E => [ 0x5922 ],
			0x2F85F => [ 0x5962 ],
			0x2F860 => [ 0x216A8 ],
			0x2F861 => [ 0x216EA ],
			0x2F862 => [ 0x59EC ],
			0x2F863 => [ 0x5A1B ],
			0x2F864 => [ 0x5A27 ],
			0x2F865 => [ 0x59D8 ],
			0x2F866 => [ 0x5A66 ],
			0x2F867 => [ 0x36EE ],
			0x2F869 => [ 0x5B08 ],
			0x2F86A => [ 0x5B3E ],
			0x2F86B => [ 0x5B3E ],
			0x2F86C => [ 0x219C8 ],
			0x2F86D => [ 0x5BC3 ],
			0x2F86E => [ 0x5BD8 ],
			0x2F86F => [ 0x5BE7 ],
			0x2F870 => [ 0x5BF3 ],
			0x2F871 => [ 0x21B18 ],
			0x2F872 => [ 0x5BFF ],
			0x2F873 => [ 0x5C06 ],
			0x2F875 => [ 0x5C22 ],
			0x2F876 => [ 0x3781 ],
			0x2F877 => [ 0x5C60 ],
			0x2F878 => [ 0x5C6E ],
			0x2F879 => [ 0x5CC0 ],
			0x2F87A => [ 0x5C8D ],
			0x2F87B => [ 0x21DE4 ],
			0x2F87C => [ 0x5D43 ],
			0x2F87D => [ 0x21DE6 ],
			0x2F87E => [ 0x5D6E ],
			0x2F87F => [ 0x5D6B ],
			0x2F880 => [ 0x5D7C ],
			0x2F881 => [ 0x5DE1 ],
			0x2F882 => [ 0x5DE2 ],
			0x2F883 => [ 0x382F ],
			0x2F884 => [ 0x5DFD ],
			0x2F885 => [ 0x5E28 ],
			0x2F886 => [ 0x5E3D ],
			0x2F887 => [ 0x5E69 ],
			0x2F888 => [ 0x3862 ],
			0x2F889 => [ 0x22183 ],
			0x2F88A => [ 0x387C ],
			0x2F88B => [ 0x5EB0 ],
			0x2F88C => [ 0x5EB3 ],
			0x2F88D => [ 0x5EB6 ],
			0x2F88E => [ 0x5ECA ],
			0x2F88F => [ 0x2A392 ],
			0x2F890 => [ 0x5EFE ],
			0x2F891 => [ 0x22331 ],
			0x2F892 => [ 0x22331 ],
			0x2F893 => [ 0x8201 ],
			0x2F894 => [ 0x5F22 ],
			0x2F895 => [ 0x5F22 ],
			0x2F896 => [ 0x38C7 ],
			0x2F897 => [ 0x232B8 ],
			0x2F898 => [ 0x261DA ],
			0x2F899 => [ 0x5F62 ],
			0x2F89A => [ 0x5F6B ],
			0x2F89B => [ 0x38E3 ],
			0x2F89C => [ 0x5F9A ],
			0x2F89D => [ 0x5FCD ],
			0x2F89E => [ 0x5FD7 ],
			0x2F89F => [ 0x5FF9 ],
			0x2F8A0 => [ 0x6081 ],
			0x2F8A1 => [ 0x393A ],
			0x2F8A2 => [ 0x391C ],
			0x2F8A3 => [ 0x6094 ],
			0x2F8A4 => [ 0x226D4 ],
			0x2F8A5 => [ 0x60C7 ],
			0x2F8A6 => [ 0x6148 ],
			0x2F8A7 => [ 0x614C ],
			0x2F8A8 => [ 0x614E ],
			0x2F8A9 => [ 0x614C ],
			0x2F8AA => [ 0x617A ],
			0x2F8AB => [ 0x618E ],
			0x2F8AC => [ 0x61B2 ],
			0x2F8AD => [ 0x61A4 ],
			0x2F8AE => [ 0x61AF ],
			0x2F8AF => [ 0x61DE ],
			0x2F8B0 => [ 0x61F2 ],
			0x2F8B1 => [ 0x61F6 ],
			0x2F8B2 => [ 0x6210 ],
			0x2F8B3 => [ 0x621B ],
			0x2F8B4 => [ 0x625D ],
			0x2F8B5 => [ 0x62B1 ],
			0x2F8B6 => [ 0x62D4 ],
			0x2F8B7 => [ 0x6350 ],
			0x2F8B8 => [ 0x22B0C ],
			0x2F8B9 => [ 0x633D ],
			0x2F8BA => [ 0x62FC ],
			0x2F8BB => [ 0x6368 ],
			0x2F8BC => [ 0x6383 ],
			0x2F8BD => [ 0x63E4 ],
			0x2F8BE => [ 0x22BF1 ],
			0x2F8BF => [ 0x6422 ],
			0x2F8C0 => [ 0x63C5 ],
			0x2F8C1 => [ 0x63A9 ],
			0x2F8C2 => [ 0x3A2E ],
			0x2F8C3 => [ 0x6469 ],
			0x2F8C4 => [ 0x647E ],
			0x2F8C5 => [ 0x649D ],
			0x2F8C6 => [ 0x6477 ],
			0x2F8C7 => [ 0x3A6C ],
			0x2F8C8 => [ 0x654F ],
			0x2F8C9 => [ 0x656C ],
			0x2F8CA => [ 0x2300A ],
			0x2F8CB => [ 0x65E3 ],
			0x2F8CC => [ 0x66F8 ],
			0x2F8CD => [ 0x6649 ],
			0x2F8CE => [ 0x3B19 ],
			0x2F8CF => [ 0x6691 ],
			0x2F8D0 => [ 0x3B08 ],
			0x2F8D1 => [ 0x3AE4 ],
			0x2F8D2 => [ 0x5192 ],
			0x2F8D3 => [ 0x5195 ],
			0x2F8D4 => [ 0x6700 ],
			0x2F8D5 => [ 0x669C ],
			0x2F8D6 => [ 0x80AD ],
			0x2F8D7 => [ 0x43D9 ],
			0x2F8D8 => [ 0x6717 ],
			0x2F8D9 => [ 0x671B ],
			0x2F8DA => [ 0x6721 ],
			0x2F8DB => [ 0x675E ],
			0x2F8DC => [ 0x6753 ],
			0x2F8DD => [ 0x233C3 ],
			0x2F8DE => [ 0x3B49 ],
			0x2F8DF => [ 0x67FA ],
			0x2F8E0 => [ 0x6785 ],
			0x2F8E1 => [ 0x6852 ],
			0x2F8E2 => [ 0x6885 ],
			0x2F8E3 => [ 0x2346D ],
			0x2F8E4 => [ 0x688E ],
			0x2F8E5 => [ 0x681F ],
			0x2F8E6 => [ 0x6914 ],
			0x2F8E7 => [ 0x3B9D ],
			0x2F8E8 => [ 0x6942 ],
			0x2F8E9 => [ 0x69A3 ],
			0x2F8EA => [ 0x69EA ],
			0x2F8EB => [ 0x6AA8 ],
			0x2F8EC => [ 0x236A3 ],
			0x2F8ED => [ 0x6ADB ],
			0x2F8EE => [ 0x3C18 ],
			0x2F8EF => [ 0x6B21 ],
			0x2F8F0 => [ 0x238A7 ],
			0x2F8F1 => [ 0x6B54 ],
			0x2F8F2 => [ 0x3C4E ],
			0x2F8F3 => [ 0x6B72 ],
			0x2F8F4 => [ 0x6B9F ],
			0x2F8F5 => [ 0x6BBA ],
			0x2F8F6 => [ 0x6BBB ],
			0x2F8F7 => [ 0x23A8D ],
			0x2F8F8 => [ 0x21D0B ],
			0x2F8F9 => [ 0x23AFA ],
			0x2F8FA => [ 0x6C4E ],
			0x2F8FB => [ 0x23CBC ],
			0x2F8FC => [ 0x6CBF ],
			0x2F8FD => [ 0x6CCD ],
			0x2F8FE => [ 0x6C67 ],
			0x2F8FF => [ 0x6D16 ],
			0x2F900 => [ 0x6D3E ],
			0x2F901 => [ 0x6D77 ],
			0x2F902 => [ 0x6D41 ],
			0x2F903 => [ 0x6D69 ],
			0x2F904 => [ 0x6D78 ],
			0x2F905 => [ 0x6D85 ],
			0x2F906 => [ 0x23D1E ],
			0x2F907 => [ 0x6D34 ],
			0x2F908 => [ 0x6E2F ],
			0x2F909 => [ 0x6E6E ],
			0x2F90A => [ 0x3D33 ],
			0x2F90B => [ 0x6ECB ],
			0x2F90C => [ 0x6EC7 ],
			0x2F90D => [ 0x23ED1 ],
			0x2F90E => [ 0x6DF9 ],
			0x2F90F => [ 0x6F6E ],
			0x2F910 => [ 0x23F5E ],
			0x2F911 => [ 0x23F8E ],
			0x2F912 => [ 0x6FC6 ],
			0x2F913 => [ 0x7039 ],
			0x2F914 => [ 0x701E ],
			0x2F915 => [ 0x701B ],
			0x2F916 => [ 0x3D96 ],
			0x2F917 => [ 0x704A ],
			0x2F918 => [ 0x707D ],
			0x2F919 => [ 0x7077 ],
			0x2F91A => [ 0x70AD ],
			0x2F91B => [ 0x20525 ],
			0x2F91C => [ 0x7145 ],
			0x2F91D => [ 0x24263 ],
			0x2F91E => [ 0x719C ],
			0x2F920 => [ 0x7228 ],
			0x2F921 => [ 0x7235 ],
			0x2F922 => [ 0x7250 ],
			0x2F923 => [ 0x24608 ],
			0x2F924 => [ 0x7280 ],
			0x2F925 => [ 0x7295 ],
			0x2F926 => [ 0x24735 ],
			0x2F927 => [ 0x24814 ],
			0x2F928 => [ 0x737A ],
			0x2F929 => [ 0x738B ],
			0x2F92A => [ 0x3EAC ],
			0x2F92B => [ 0x73A5 ],
			0x2F92C => [ 0x3EB8 ],
			0x2F92D => [ 0x3EB8 ],
			0x2F92E => [ 0x7447 ],
			0x2F92F => [ 0x745C ],
			0x2F930 => [ 0x7471 ],
			0x2F931 => [ 0x7485 ],
			0x2F932 => [ 0x74CA ],
			0x2F933 => [ 0x3F1B ],
			0x2F934 => [ 0x7524 ],
			0x2F935 => [ 0x24C36 ],
			0x2F936 => [ 0x753E ],
			0x2F937 => [ 0x24C92 ],
			0x2F938 => [ 0x7570 ],
			0x2F939 => [ 0x2219F ],
			0x2F93A => [ 0x7610 ],
			0x2F93B => [ 0x24FA1 ],
			0x2F93C => [ 0x24FB8 ],
			0x2F93D => [ 0x25044 ],
			0x2F93E => [ 0x3FFC ],
			0x2F93F => [ 0x4008 ],
			0x2F940 => [ 0x76F4 ],
			0x2F941 => [ 0x250F3 ],
			0x2F942 => [ 0x250F2 ],
			0x2F943 => [ 0x25119 ],
			0x2F944 => [ 0x25133 ],
			0x2F945 => [ 0x771E ],
			0x2F946 => [ 0x771F ],
			0x2F947 => [ 0x771F ],
			0x2F948 => [ 0x774A ],
			0x2F949 => [ 0x4039 ],
			0x2F94A => [ 0x778B ],
			0x2F94B => [ 0x4046 ],
			0x2F94C => [ 0x4096 ],
			0x2F94D => [ 0x2541D ],
			0x2F94E => [ 0x784E ],
			0x2F94F => [ 0x788C ],
			0x2F950 => [ 0x78CC ],
			0x2F951 => [ 0x40E3 ],
			0x2F952 => [ 0x25626 ],
			0x2F953 => [ 0x7956 ],
			0x2F954 => [ 0x2569A ],
			0x2F955 => [ 0x256C5 ],
			0x2F956 => [ 0x798F ],
			0x2F957 => [ 0x79EB ],
			0x2F958 => [ 0x412F ],
			0x2F959 => [ 0x7A40 ],
			0x2F95A => [ 0x7A4A ],
			0x2F95B => [ 0x7A4F ],
			0x2F95C => [ 0x2597C ],
			0x2F95D => [ 0x25AA7 ],
			0x2F95E => [ 0x25AA7 ],
			0x2F960 => [ 0x4202 ],
			0x2F961 => [ 0x25BAB ],
			0x2F962 => [ 0x7BC6 ],
			0x2F963 => [ 0x7BC9 ],
			0x2F964 => [ 0x4227 ],
			0x2F965 => [ 0x25C80 ],
			0x2F966 => [ 0x7CD2 ],
			0x2F967 => [ 0x42A0 ],
			0x2F968 => [ 0x7CE8 ],
			0x2F969 => [ 0x7CE3 ],
			0x2F96A => [ 0x7D00 ],
			0x2F96B => [ 0x25F86 ],
			0x2F96C => [ 0x7D63 ],
			0x2F96D => [ 0x4301 ],
			0x2F96E => [ 0x7DC7 ],
			0x2F96F => [ 0x7E02 ],
			0x2F970 => [ 0x7E45 ],
			0x2F971 => [ 0x4334 ],
			0x2F972 => [ 0x26228 ],
			0x2F973 => [ 0x26247 ],
			0x2F974 => [ 0x4359 ],
			0x2F975 => [ 0x262D9 ],
			0x2F976 => [ 0x7F7A ],
			0x2F977 => [ 0x2633E ],
			0x2F978 => [ 0x7F95 ],
			0x2F979 => [ 0x7FFA ],
			0x2F97A => [ 0x8005 ],
			0x2F97B => [ 0x264DA ],
			0x2F97C => [ 0x26523 ],
			0x2F97D => [ 0x8060 ],
			0x2F97E => [ 0x265A8 ],
			0x2F97F => [ 0x8070 ],
			0x2F980 => [ 0x2335F ],
			0x2F981 => [ 0x43D5 ],
			0x2F982 => [ 0x80B2 ],
			0x2F983 => [ 0x8103 ],
			0x2F984 => [ 0x440B ],
			0x2F985 => [ 0x813E ],
			0x2F986 => [ 0x5AB5 ],
			0x2F987 => [ 0x267A7 ],
			0x2F988 => [ 0x267B5 ],
			0x2F989 => [ 0x23393 ],
			0x2F98A => [ 0x2339C ],
			0x2F98B => [ 0x8201 ],
			0x2F98C => [ 0x8204 ],
			0x2F98D => [ 0x8F9E ],
			0x2F98E => [ 0x446B ],
			0x2F98F => [ 0x8291 ],
			0x2F990 => [ 0x828B ],
			0x2F991 => [ 0x829D ],
			0x2F992 => [ 0x52B3 ],
			0x2F993 => [ 0x82B1 ],
			0x2F994 => [ 0x82B3 ],
			0x2F995 => [ 0x82BD ],
			0x2F996 => [ 0x82E6 ],
			0x2F997 => [ 0x26B3C ],
			0x2F998 => [ 0x82E5 ],
			0x2F999 => [ 0x831D ],
			0x2F99A => [ 0x8363 ],
			0x2F99B => [ 0x83AD ],
			0x2F99C => [ 0x8323 ],
			0x2F99D => [ 0x83BD ],
			0x2F99E => [ 0x83E7 ],
			0x2F99F => [ 0x8457 ],
			0x2F9A0 => [ 0x8353 ],
			0x2F9A1 => [ 0x83CA ],
			0x2F9A2 => [ 0x83CC ],
			0x2F9A3 => [ 0x83DC ],
			0x2F9A4 => [ 0x26C36 ],
			0x2F9A5 => [ 0x26D6B ],
			0x2F9A6 => [ 0x26CD5 ],
			0x2F9A7 => [ 0x452B ],
			0x2F9A8 => [ 0x84F1 ],
			0x2F9A9 => [ 0x84F3 ],
			0x2F9AA => [ 0x8516 ],
			0x2F9AB => [ 0x273CA ],
			0x2F9AC => [ 0x8564 ],
			0x2F9AD => [ 0x26F2C ],
			0x2F9AE => [ 0x455D ],
			0x2F9AF => [ 0x4561 ],
			0x2F9B0 => [ 0x26FB1 ],
			0x2F9B1 => [ 0x270D2 ],
			0x2F9B2 => [ 0x456B ],
			0x2F9B3 => [ 0x8650 ],
			0x2F9B4 => [ 0x865C ],
			0x2F9B5 => [ 0x8667 ],
			0x2F9B6 => [ 0x8669 ],
			0x2F9B7 => [ 0x86A9 ],
			0x2F9B8 => [ 0x8688 ],
			0x2F9B9 => [ 0x870E ],
			0x2F9BA => [ 0x86E2 ],
			0x2F9BB => [ 0x8779 ],
			0x2F9BC => [ 0x8728 ],
			0x2F9BD => [ 0x876B ],
			0x2F9BE => [ 0x8786 ],
			0x2F9C0 => [ 0x87E1 ],
			0x2F9C1 => [ 0x8801 ],
			0x2F9C2 => [ 0x45F9 ],
			0x2F9C3 => [ 0x8860 ],
			0x2F9C4 => [ 0x8863 ],
			0x2F9C5 => [ 0x27667 ],
			0x2F9C6 => [ 0x88D7 ],
			0x2F9C7 => [ 0x88DE ],
			0x2F9C8 => [ 0x4635 ],
			0x2F9C9 => [ 0x88FA ],
			0x2F9CA => [ 0x34BB ],
			0x2F9CB => [ 0x278AE ],
			0x2F9CC => [ 0x27966 ],
			0x2F9CD => [ 0x46BE ],
			0x2F9CE => [ 0x46C7 ],
			0x2F9CF => [ 0x8AA0 ],
			0x2F9D0 => [ 0x8AED ],
			0x2F9D1 => [ 0x8B8A ],
			0x2F9D2 => [ 0x8C55 ],
			0x2F9D3 => [ 0x27CA8 ],
			0x2F9D4 => [ 0x8CAB ],
			0x2F9D5 => [ 0x8CC1 ],
			0x2F9D6 => [ 0x8D1B ],
			0x2F9D7 => [ 0x8D77 ],
			0x2F9D8 => [ 0x27F2F ],
			0x2F9D9 => [ 0x20804 ],
			0x2F9DA => [ 0x8DCB ],
			0x2F9DB => [ 0x8DBC ],
			0x2F9DC => [ 0x8DF0 ],
			0x2F9DD => [ 0x208DE ],
			0x2F9DE => [ 0x8ED4 ],
			0x2F9DF => [ 0x8F38 ],
			0x2F9E0 => [ 0x285D2 ],
			0x2F9E1 => [ 0x285ED ],
			0x2F9E2 => [ 0x9094 ],
			0x2F9E3 => [ 0x90F1 ],
			0x2F9E4 => [ 0x9111 ],
			0x2F9E5 => [ 0x2872E ],
			0x2F9E6 => [ 0x911B ],
			0x2F9E7 => [ 0x9238 ],
			0x2F9E8 => [ 0x92D7 ],
			0x2F9E9 => [ 0x92D8 ],
			0x2F9EA => [ 0x927C ],
			0x2F9EB => [ 0x93F9 ],
			0x2F9EC => [ 0x9415 ],
			0x2F9ED => [ 0x28BFA ],
			0x2F9EE => [ 0x958B ],
			0x2F9EF => [ 0x4995 ],
			0x2F9F0 => [ 0x95B7 ],
			0x2F9F1 => [ 0x28D77 ],
			0x2F9F2 => [ 0x49E6 ],
			0x2F9F3 => [ 0x96C3 ],
			0x2F9F4 => [ 0x5DB2 ],
			0x2F9F5 => [ 0x9723 ],
			0x2F9F6 => [ 0x29145 ],
			0x2F9F7 => [ 0x2921A ],
			0x2F9F8 => [ 0x4A6E ],
			0x2F9F9 => [ 0x4A76 ],
			0x2F9FA => [ 0x97E0 ],
			0x2F9FB => [ 0x2940A ],
			0x2F9FC => [ 0x4AB2 ],
			0x2F9FD => [ 0x29496 ],
			0x2F9FE => [ 0x980B ],
			0x2F9FF => [ 0x980B ],
			0x2FA00 => [ 0x9829 ],
			0x2FA01 => [ 0x295B6 ],
			0x2FA02 => [ 0x98E2 ],
			0x2FA03 => [ 0x4B33 ],
			0x2FA04 => [ 0x9929 ],
			0x2FA05 => [ 0x99A7 ],
			0x2FA06 => [ 0x99C2 ],
			0x2FA07 => [ 0x99FE ],
			0x2FA08 => [ 0x4BCE ],
			0x2FA09 => [ 0x29B30 ],
			0x2FA0A => [ 0x9B12 ],
			0x2FA0B => [ 0x9C40 ],
			0x2FA0C => [ 0x9CFD ],
			0x2FA0D => [ 0x4CCE ],
			0x2FA0E => [ 0x4CED ],
			0x2FA0F => [ 0x9D67 ],
			0x2FA10 => [ 0x2A0CE ],
			0x2FA11 => [ 0x4CF8 ],
			0x2FA12 => [ 0x2A105 ],
			0x2FA13 => [ 0x2A20E ],
			0x2FA14 => [ 0x2A291 ],
			0x2FA15 => [ 0x9EBB ],
			0x2FA16 => [ 0x4D56 ],
			0x2FA17 => [ 0x9EF9 ],
			0x2FA18 => [ 0x9EFE ],
			0x2FA19 => [ 0x9F05 ],
			0x2FA1A => [ 0x9F0F ],
			0x2FA1B => [ 0x9F16 ],
			0x2FA1C => [ 0x9F3B ],
			0x2FA1D => [ 0x2A600 ]
		],
		'norm_combcls'       => [
			0x334   => 1,
			0x335   => 1,
			0x336   => 1,
			0x337   => 1,
			0x338   => 1,
			0x93C   => 7,
			0x9BC   => 7,
			0xA3C   => 7,
			0xABC   => 7,
			0xB3C   => 7,
			0xCBC   => 7,
			0x1037  => 7,
			0x3099  => 8,
			0x309A  => 8,
			0x94D   => 9,
			0x9CD   => 9,
			0xA4D   => 9,
			0xACD   => 9,
			0xB4D   => 9,
			0xBCD   => 9,
			0xC4D   => 9,
			0xCCD   => 9,
			0xD4D   => 9,
			0xDCA   => 9,
			0xE3A   => 9,
			0xF84   => 9,
			0x1039  => 9,
			0x1714  => 9,
			0x1734  => 9,
			0x17D2  => 9,
			0x5B0   => 10,
			0x5B1   => 11,
			0x5B2   => 12,
			0x5B3   => 13,
			0x5B4   => 14,
			0x5B5   => 15,
			0x5B6   => 16,
			0x5B7   => 17,
			0x5B8   => 18,
			0x5B9   => 19,
			0x5BB   => 20,
			0x5Bc   => 21,
			0x5BD   => 22,
			0x5BF   => 23,
			0x5C1   => 24,
			0x5C2   => 25,
			0xFB1E  => 26,
			0x64B   => 27,
			0x64C   => 28,
			0x64D   => 29,
			0x64E   => 30,
			0x64F   => 31,
			0x650   => 32,
			0x651   => 33,
			0x652   => 34,
			0x670   => 35,
			0x711   => 36,
			0xC55   => 84,
			0xC56   => 91,
			0xE38   => 103,
			0xE39   => 103,
			0xE48   => 107,
			0xE49   => 107,
			0xE4A   => 107,
			0xE4B   => 107,
			0xEB8   => 118,
			0xEB9   => 118,
			0xEC8   => 122,
			0xEC9   => 122,
			0xECA   => 122,
			0xECB   => 122,
			0xF71   => 129,
			0xF72   => 130,
			0xF7A   => 130,
			0xF7B   => 130,
			0xF7C   => 130,
			0xF7D   => 130,
			0xF80   => 130,
			0xF74   => 132,
			0x321   => 202,
			0x322   => 202,
			0x327   => 202,
			0x328   => 202,
			0x31B   => 216,
			0xF39   => 216,
			0x1D165 => 216,
			0x1D166 => 216,
			0x1D16E => 216,
			0x1D16F => 216,
			0x1D170 => 216,
			0x1D171 => 216,
			0x1D172 => 216,
			0x302A  => 218,
			0x316   => 220,
			0x317   => 220,
			0x318   => 220,
			0x319   => 220,
			0x31C   => 220,
			0x31D   => 220,
			0x31E   => 220,
			0x31F   => 220,
			0x320   => 220,
			0x323   => 220,
			0x324   => 220,
			0x325   => 220,
			0x326   => 220,
			0x329   => 220,
			0x32A   => 220,
			0x32B   => 220,
			0x32C   => 220,
			0x32D   => 220,
			0x32E   => 220,
			0x32F   => 220,
			0x330   => 220,
			0x331   => 220,
			0x332   => 220,
			0x333   => 220,
			0x339   => 220,
			0x33A   => 220,
			0x33B   => 220,
			0x33C   => 220,
			0x347   => 220,
			0x348   => 220,
			0x349   => 220,
			0x34D   => 220,
			0x34E   => 220,
			0x353   => 220,
			0x354   => 220,
			0x355   => 220,
			0x356   => 220,
			0x591   => 220,
			0x596   => 220,
			0x59B   => 220,
			0x5A3   => 220,
			0x5A4   => 220,
			0x5A5   => 220,
			0x5A6   => 220,
			0x5A7   => 220,
			0x5AA   => 220,
			0x655   => 220,
			0x656   => 220,
			0x6E3   => 220,
			0x6EA   => 220,
			0x6ED   => 220,
			0x731   => 220,
			0x734   => 220,
			0x737   => 220,
			0x738   => 220,
			0x739   => 220,
			0x73B   => 220,
			0x73C   => 220,
			0x73E   => 220,
			0x742   => 220,
			0x744   => 220,
			0x746   => 220,
			0x748   => 220,
			0x952   => 220,
			0xF18   => 220,
			0xF19   => 220,
			0xF35   => 220,
			0xF37   => 220,
			0xFC6   => 220,
			0x193B  => 220,
			0x20E8  => 220,
			0x1D17B => 220,
			0x1D17C => 220,
			0x1D17D => 220,
			0x1D17E => 220,
			0x1D17F => 220,
			0x1D180 => 220,
			0x1D181 => 220,
			0x1D182 => 220,
			0x1D18A => 220,
			0x1D18B => 220,
			0x59A   => 222,
			0x5AD   => 222,
			0x1929  => 222,
			0x302D  => 222,
			0x302E  => 224,
			0x302F  => 224,
			0x1D16D => 226,
			0x5AE   => 228,
			0x18A9  => 228,
			0x302B  => 228,
			0x300   => 230,
			0x301   => 230,
			0x302   => 230,
			0x303   => 230,
			0x304   => 230,
			0x305   => 230,
			0x306   => 230,
			0x307   => 230,
			0x308   => 230,
			0x309   => 230,
			0x30A   => 230,
			0x30B   => 230,
			0x30C   => 230,
			0x30D   => 230,
			0x30E   => 230,
			0x30F   => 230,
			0x310   => 230,
			0x311   => 230,
			0x312   => 230,
			0x313   => 230,
			0x314   => 230,
			0x33D   => 230,
			0x33E   => 230,
			0x33F   => 230,
			0x340   => 230,
			0x341   => 230,
			0x342   => 230,
			0x343   => 230,
			0x344   => 230,
			0x346   => 230,
			0x34A   => 230,
			0x34B   => 230,
			0x34C   => 230,
			0x350   => 230,
			0x351   => 230,
			0x352   => 230,
			0x357   => 230,
			0x363   => 230,
			0x364   => 230,
			0x365   => 230,
			0x366   => 230,
			0x367   => 230,
			0x368   => 230,
			0x369   => 230,
			0x36A   => 230,
			0x36B   => 230,
			0x36C   => 230,
			0x36D   => 230,
			0x36E   => 230,
			0x36F   => 230,
			0x483   => 230,
			0x484   => 230,
			0x485   => 230,
			0x486   => 230,
			0x592   => 230,
			0x593   => 230,
			0x594   => 230,
			0x595   => 230,
			0x597   => 230,
			0x598   => 230,
			0x599   => 230,
			0x59C   => 230,
			0x59D   => 230,
			0x59E   => 230,
			0x59F   => 230,
			0x5A0   => 230,
			0x5A1   => 230,
			0x5A8   => 230,
			0x5A9   => 230,
			0x5AB   => 230,
			0x5AC   => 230,
			0x5AF   => 230,
			0x5C4   => 230,
			0x610   => 230,
			0x611   => 230,
			0x612   => 230,
			0x613   => 230,
			0x614   => 230,
			0x615   => 230,
			0x653   => 230,
			0x654   => 230,
			0x657   => 230,
			0x658   => 230,
			0x6D6   => 230,
			0x6D7   => 230,
			0x6D8   => 230,
			0x6D9   => 230,
			0x6DA   => 230,
			0x6DB   => 230,
			0x6DC   => 230,
			0x6DF   => 230,
			0x6E0   => 230,
			0x6E1   => 230,
			0x6E2   => 230,
			0x6E4   => 230,
			0x6E7   => 230,
			0x6E8   => 230,
			0x6EB   => 230,
			0x6EC   => 230,
			0x730   => 230,
			0x732   => 230,
			0x733   => 230,
			0x735   => 230,
			0x736   => 230,
			0x73A   => 230,
			0x73D   => 230,
			0x73F   => 230,
			0x740   => 230,
			0x741   => 230,
			0x743   => 230,
			0x745   => 230,
			0x747   => 230,
			0x749   => 230,
			0x74A   => 230,
			0x951   => 230,
			0x953   => 230,
			0x954   => 230,
			0xF82   => 230,
			0xF83   => 230,
			0xF86   => 230,
			0xF87   => 230,
			0x170D  => 230,
			0x193A  => 230,
			0x20D0  => 230,
			0x20D1  => 230,
			0x20D4  => 230,
			0x20D5  => 230,
			0x20D6  => 230,
			0x20D7  => 230,
			0x20DB  => 230,
			0x20DC  => 230,
			0x20E1  => 230,
			0x20E7  => 230,
			0x20E9  => 230,
			0xFE20  => 230,
			0xFE21  => 230,
			0xFE22  => 230,
			0xFE23  => 230,
			0x1D185 => 230,
			0x1D186 => 230,
			0x1D187 => 230,
			0x1D189 => 230,
			0x1D188 => 230,
			0x1D1AA => 230,
			0x1D1AB => 230,
			0x1D1AC => 230,
			0x1D1AD => 230,
			0x315   => 232,
			0x31A   => 232,
			0x302C  => 232,
			0x35F   => 233,
			0x362   => 233,
			0x35D   => 234,
			0x35E   => 234,
			0x360   => 234,
			0x361   => 234,
			0x345   => 240
		]
	];
	
	// NP See below
	// Internal settings, do not mess with them
	protected $sub_version = 'main';
	protected $_punycode_prefix = 'xn--';
	protected $_invalid_ucs = 0x80000000;
	protected $_max_ucs = 0x10FFFF;
	protected $_base = 36;
	protected $_tmin = 1;
	protected $_tmax = 26;
	protected $_skew = 38;
	protected $_damp = 700;
	protected $_initial_bias = 72;
	protected $_initial_n = 0x80;
	protected $_sbase = 0xAC00;
	protected $_lbase = 0x1100;
	protected $_vbase = 0x1161;
	protected $_tbase = 0x11A7;
	protected $_lcount = 19;
	protected $_vcount = 21;
		protected $_tcount = 28;   // _vcount * _tcount
	protected $_ncount = 588; // _lcount * _tcount * _vcount
protected $_scount = 11172;
	protected $_error = false;
	// See {@link set_paramter()} for details of how to change the following
	// settings from within your script / application
	protected $_api_encoding = 'utf8';   // Default input charset is UTF-8
	protected $_allow_overlong = false;  // Overlong UTF-8 encodings are forbidden
	protected $_strict_mode = false;     // Behave strict or not
	protected $_idn_version = 2003;      // Can be either 2003 (old, default) or 2008
	private $version = '0.9.0';
	
	/**
	 * the constructor
	 *
	 * @param array $options
	 *
	 * @return boolean
	 * @since 0.5.2
	 */
	public function __construct( $options = false ) {
		$this->slast = $this->_sbase + $this->_lcount * $this->_vcount * $this->_tcount;
		// If parameters are given, pass these to the respective method
		if ( is_array( $options ) ) {
			$this->set_parameter( $options );
		}
		
		// populate mbstring overloading cache if not set
		if ( self::$_mb_string_overload === null ) {
			self::$_mb_string_overload = ( extension_loaded( 'mbstring' ) && ( ini_get( 'mbstring.func_overload' ) & 0x02 ) === 0x02 );
		}
	}
	
	/**
	 * Sets a new option value. Available options and values:
	 * [encoding - Use either UTF-8, UCS4 as array or UCS4 as string as input ('utf8' for UTF-8,
	 *         'ucs4_string' and 'ucs4_array' respectively for UCS4); The output is always UTF-8]
	 * [overlong - Unicode does not allow unnecessarily long encodings of chars,
	 *             to allow this, set this parameter to true, else to false;
	 *             default is false.]
	 * [strict - true: strict mode, good for registration purposes - Causes errors
	 *           on failures; false: loose mode, ideal for "wildlife" applications
	 *           by silently ignoring errors and returning the original input instead
	 *
	 * @param    mixed     Parameter to set (string: single parameter; array of Parameter => Value pairs)
	 * @param    string    Value to use (if parameter 1 is a string)
	 *
	 * @return   boolean   true on success, false otherwise
	 */
	public function set_parameter( $option, $value = false ) {
		if ( ! is_array( $option ) ) {
			$option = [ $option => $value ];
		}
		foreach ( $option as $k => $v ) {
			switch ( $k ) {
				case 'encoding':
					switch ( $v ) {
						case 'utf8':
						case 'ucs4_string':
						case 'ucs4_array':
							$this->_api_encoding = $v;
							break;
						default:
							$this->_error( 'Set Parameter: Unknown parameter ' . $v . ' for option ' . $k );
							
							return false;
					}
					break;
				case 'overlong':
					$this->_allow_overlong = ( $v ) ? true : false;
					break;
				case 'strict':
					$this->_strict_mode = ( $v ) ? true : false;
					break;
				case 'idn_version':
					if ( in_array( $v, [ '2003', '2008' ] ) ) {
						$this->_idn_version = $v;
					} else {
						$this->_error( 'Set Parameter: Unknown parameter ' . $v . ' for option ' . $k );
					}
					break;
				case 'encode_german_sz': // Deprecated
					if ( ! $v ) {
						self::$NP['replacemaps'][0xDF] = [ 0x73, 0x73 ];
					} else {
						unset( self::$NP['replacemaps'][0xDF] );
					}
					break;
				default:
					$this->_error( 'Set Parameter: Unknown option ' . $k );
					
					return false;
			}
		}
		
		return true;
	}
	
	/**
	 * Internal error handling method
	 *
	 * @param  string $error
	 */
	protected function _error( $error = '' ) {
		$this->_error = $error;
	}
	
	public function get_version() {
		return $this->version . '-' . $this->sub_version;
	}
	
	/**
	 * Decode a given ACE domain name
	 *
	 * @param    string   Domain name (ACE string)
	 * [@param    string   Desired output encoding, see {@link set_parameter}]
	 *
	 * @return   string   Decoded Domain name (UTF-8 or UCS-4)
	 */
	public function decode( $input, $one_time_encoding = false ) {
		// Optionally set
		if ( $one_time_encoding ) {
			switch ( $one_time_encoding ) {
				case 'utf8':
				case 'ucs4_string':
				case 'ucs4_array':
					break;
				default:
					$this->_error( 'Unknown encoding ' . $one_time_encoding );
					
					return false;
			}
		}
		// Make sure to drop any newline characters around
		$input = trim( $input );
		
		// Negotiate input and try to determine, whether it is a plain string,
		// an email address or something like a complete URL
		if ( strpos( $input, '@' ) ) { // Maybe it is an email address
			// No no in strict mode
			if ( $this->_strict_mode ) {
				$this->_error( 'Only simple domain name parts can be handled in strict mode' );
				
				return false;
			}
			list ( $email_pref, $input ) = explode( '@', $input, 2 );
			$arr = explode( '.', $input );
			foreach ( $arr as $k => $v ) {
				if ( preg_match( '!^' . preg_quote( $this->_punycode_prefix, '!' ) . '!', $v ) ) {
					$conv = $this->_decode( $v );
					if ( $conv ) {
						$arr[ $k ] = $conv;
					}
				}
			}
			$input = join( '.', $arr );
			$arr   = explode( '.', $email_pref );
			foreach ( $arr as $k => $v ) {
				if ( preg_match( '!^' . preg_quote( $this->_punycode_prefix, '!' ) . '!', $v ) ) {
					$conv = $this->_decode( $v );
					if ( $conv ) {
						$arr[ $k ] = $conv;
					}
				}
			}
			$email_pref = join( '.', $arr );
			$return     = $email_pref . '@' . $input;
		} elseif ( preg_match( '![:\./]!', $input ) ) { // Or a complete domain name (with or without paths / parameters)
			// No no in strict mode
			if ( $this->_strict_mode ) {
				$this->_error( 'Only simple domain name parts can be handled in strict mode' );
				
				return false;
			}
			$parsed = parse_url( $input );
			if ( isset( $parsed['host'] ) ) {
				$arr = explode( '.', $parsed['host'] );
				foreach ( $arr as $k => $v ) {
					$conv = $this->_decode( $v );
					if ( $conv ) {
						$arr[ $k ] = $conv;
					}
				}
				$parsed['host'] = join( '.', $arr );
				$return         = ( empty( $parsed['scheme'] ) ? '' : $parsed['scheme'] . ( strtolower( $parsed['scheme'] ) == 'mailto' ? ':' : '://' ) ) .
				                  ( empty( $parsed['user'] ) ? '' : $parsed['user'] . ( empty( $parsed['pass'] ) ? '' : ':' . $parsed['pass'] ) . '@' ) .
				                  $parsed['host'] .
				                  ( empty( $parsed['port'] ) ? '' : ':' . $parsed['port'] ) .
				                  ( empty( $parsed['path'] ) ? '' : $parsed['path'] ) .
				                  ( empty( $parsed['query'] ) ? '' : '?' . $parsed['query'] ) .
				                  ( empty( $parsed['fragment'] ) ? '' : '#' . $parsed['fragment'] );
			} else { // parse_url seems to have failed, try without it
				$arr = explode( '.', $input );
				foreach ( $arr as $k => $v ) {
					$conv      = $this->_decode( $v );
					$arr[ $k ] = ( $conv ) ? $conv : $v;
				}
				$return = join( '.', $arr );
			}
		} else { // Otherwise we consider it being a pure domain name string
			$return = $this->_decode( $input );
			if ( ! $return ) {
				$return = $input;
			}
		}
		// The output is UTF-8 by default, other output formats need conversion here
		// If one time encoding is given, use this, else the objects property
		switch ( ( $one_time_encoding ) ? $one_time_encoding : $this->_api_encoding ) {
			case 'utf8':
				return $return; // break;
			case 'ucs4_string':
				return $this->_ucs4_to_ucs4_string( $this->_utf8_to_ucs4( $return ) );  // break;
			case 'ucs4_array':
				return $this->_utf8_to_ucs4( $return ); // break;
			default:
				$this->_error( 'Unsupported output format' );
				
				return false;
		}
	}
	
	/**
	 * The actual decoding algorithm
	 *
	 * @param string
	 *
	 * @return mixed
	 */
	protected function _decode( $encoded ) {
		$decoded = [];
		// find the Punycode prefix
		if ( ! preg_match( '!^' . preg_quote( $this->_punycode_prefix, '!' ) . '!', $encoded ) ) {
			$this->_error( 'This is not a punycode string' );
			
			return false;
		}
		$encode_test = preg_replace( '!^' . preg_quote( $this->_punycode_prefix, '!' ) . '!', '', $encoded );
		// If nothing left after removing the prefix, it is hopeless
		if ( ! $encode_test ) {
			$this->_error( 'The given encoded string was empty' );
			
			return false;
		}
		// Find last occurence of the delimiter
		$delim_pos = strrpos( $encoded, '-' );
		if ( $delim_pos > self::byteLength( $this->_punycode_prefix ) ) {
			for ( $k = self::byteLength( $this->_punycode_prefix ); $k < $delim_pos; ++ $k ) {
				$decoded[] = ord( $encoded{$k} );
			}
		}
		$deco_len = count( $decoded );
		$enco_len = self::byteLength( $encoded );
		
		// Wandering through the strings; init
		$is_first = true;
		$bias     = $this->_initial_bias;
		$idx      = 0;
		$char     = $this->_initial_n;
		
		for ( $enco_idx = ( $delim_pos ) ? ( $delim_pos + 1 ) : 0; $enco_idx < $enco_len; ++ $deco_len ) {
			for ( $old_idx = $idx, $w = 1, $k = $this->_base; 1; $k += $this->_base ) {
				$digit = $this->_decode_digit( $encoded{$enco_idx ++} );
				$idx += $digit * $w;
				$t = ( $k <= $bias ) ? $this->_tmin :
					( ( $k >= $bias + $this->_tmax ) ? $this->_tmax : ( $k - $bias ) );
				if ( $digit < $t ) {
					break;
				}
				$w = (int) ( $w * ( $this->_base - $t ) );
			}
			$bias     = $this->_adapt( $idx - $old_idx, $deco_len + 1, $is_first );
			$is_first = false;
			$char += (int) ( $idx / ( $deco_len + 1 ) );
			$idx %= ( $deco_len + 1 );
			if ( $deco_len > 0 ) {
				// Make room for the decoded char
				for ( $i = $deco_len; $i > $idx; $i -- ) {
					$decoded[ $i ] = $decoded[ ( $i - 1 ) ];
				}
			}
			$decoded[ $idx ++ ] = $char;
		}
		
		return $this->_ucs4_to_utf8( $decoded );
	}
	
	/**
	 * Gets the length of a string in bytes even if mbstring function
	 * overloading is turned on
	 *
	 * @param string $string the string for which to get the length.
	 *
	 * @return integer the length of the string in bytes.
	 */
	protected static function byteLength( $string ) {
		if ( self::$_mb_string_overload ) {
			return mb_strlen( $string, '8bit' );
		}
		
		return strlen( (binary) $string );
	}
	
	/**
	 * Decode a certain digit
	 *
	 * @param    int $cp
	 *
	 * @return int
	 */
	protected function _decode_digit( $cp ) {
		$cp = ord( $cp );
		
		return ( $cp - 48 < 10 ) ? $cp - 22 : ( ( $cp - 65 < 26 ) ? $cp - 65 : ( ( $cp - 97 < 26 ) ? $cp - 97 : $this->_base ) );
	}
	
	/**
	 * Adapt the bias according to the current code point and position
	 *
	 * @param int $delta
	 * @param int $npoints
	 * @param int $is_first
	 *
	 * @return int
	 */
	protected function _adapt( $delta, $npoints, $is_first ) {
		$delta = intval( $is_first ? ( $delta / $this->_damp ) : ( $delta / 2 ) );
		$delta += intval( $delta / $npoints );
		for ( $k = 0; $delta > ( ( $this->_base - $this->_tmin ) * $this->_tmax ) / 2; $k += $this->_base ) {
			$delta = intval( $delta / ( $this->_base - $this->_tmin ) );
		}
		
		return intval( $k + ( $this->_base - $this->_tmin + 1 ) * $delta / ( $delta + $this->_skew ) );
	}
	
	/**
	 * Convert UCS-4 string into UTF-8 string
	 * See _utf8_to_ucs4() for details
	 *
	 * @param string $input
	 *
	 * @return string
	 */
	protected function _ucs4_to_utf8( $input ) {
		$output = '';
		foreach ( $input as $k => $v ) {
			if ( $v < 128 ) { // 7bit are transferred literally
				$output .= chr( $v );
			} elseif ( $v < ( 1 << 11 ) ) { // 2 bytes
				$output .= chr( 192 + ( $v >> 6 ) ) . chr( 128 + ( $v & 63 ) );
			} elseif ( $v < ( 1 << 16 ) ) { // 3 bytes
				$output .= chr( 224 + ( $v >> 12 ) ) . chr( 128 + ( ( $v >> 6 ) & 63 ) ) . chr( 128 + ( $v & 63 ) );
			} elseif ( $v < ( 1 << 21 ) ) { // 4 bytes
				$output .= chr( 240 + ( $v >> 18 ) ) . chr( 128 + ( ( $v >> 12 ) & 63 ) ) . chr( 128 + ( ( $v >> 6 ) & 63 ) ) . chr( 128 + ( $v & 63 ) );
			} else {
				$this->_error( 'Conversion from UCS-4 to UTF-8 failed: malformed input at byte ' . $k );
				
				return false;
			}
		}
		
		return $output;
	}
	
	/**
	 * Convert UCS-4 array into UCS-4 string
	 *
	 * @param array $input
	 *
	 * @return string
	 */
	protected function _ucs4_to_ucs4_string( $input ) {
		$output = '';
		// Take array values and split output to 4 bytes per value
		// The bit mask is 255, which reads &11111111
		foreach ( $input as $v ) {
			$output .= chr( ( $v >> 24 ) & 255 ) . chr( ( $v >> 16 ) & 255 ) . chr( ( $v >> 8 ) & 255 ) . chr( $v & 255 );
		}
		
		return $output;
	}
	
	/**
	 * This converts an UTF-8 encoded string to its UCS-4 representation
	 * By talking about UCS-4 "strings" we mean arrays of 32bit integers representing
	 * each of the "chars". This is due to PHP not being able to handle strings with
	 * bit depth different from 8. This apllies to the reverse method _ucs4_to_utf8(), too.
	 * The following UTF-8 encodings are supported:
	 * bytes bits  representation
	 * 1        7  0xxxxxxx
	 * 2       11  110xxxxx 10xxxxxx
	 * 3       16  1110xxxx 10xxxxxx 10xxxxxx
	 * 4       21  11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
	 * 5       26  111110xx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
	 * 6       31  1111110x 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
	 * Each x represents a bit that can be used to store character data.
	 * The five and six byte sequences are part of Annex D of ISO/IEC 10646-1:2000
	 *
	 * @param string $input
	 *
	 * @return string
	 */
	protected function _utf8_to_ucs4( $input ) {
		$output  = [];
		$out_len = 0;
		$inp_len = self::byteLength( $input );
		$mode    = 'next';
		$test    = 'none';
		for ( $k = 0; $k < $inp_len; ++ $k ) {
			$v = ord( $input{$k} ); // Extract byte from input string
			if ( $v < 128 ) { // We found an ASCII char - put into stirng as is
				$output[ $out_len ] = $v;
				++ $out_len;
				if ( 'add' == $mode ) {
					$this->_error( 'Conversion from UTF-8 to UCS-4 failed: malformed input at byte ' . $k );
					
					return false;
				}
				continue;
			}
			if ( 'next' == $mode ) { // Try to find the next start byte; determine the width of the Unicode char
				$start_byte = $v;
				$mode       = 'add';
				$test       = 'range';
				if ( $v >> 5 == 6 ) { // &110xxxxx 10xxxxx
					$next_byte = 0; // Tells, how many times subsequent bitmasks must rotate 6bits to the left
					$v         = ( $v - 192 ) << 6;
				} elseif ( $v >> 4 == 14 ) { // &1110xxxx 10xxxxxx 10xxxxxx
					$next_byte = 1;
					$v         = ( $v - 224 ) << 12;
				} elseif ( $v >> 3 == 30 ) { // &11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
					$next_byte = 2;
					$v         = ( $v - 240 ) << 18;
				} elseif ( $v >> 2 == 62 ) { // &111110xx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
					$next_byte = 3;
					$v         = ( $v - 248 ) << 24;
				} elseif ( $v >> 1 == 126 ) { // &1111110x 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
					$next_byte = 4;
					$v         = ( $v - 252 ) << 30;
				} else {
					$this->_error( 'This might be UTF-8, but I don\'t understand it at byte ' . $k );
					
					return false;
				}
				if ( 'add' == $mode ) {
					$output[ $out_len ] = (int) $v;
					++ $out_len;
					continue;
				}
			}
			if ( 'add' == $mode ) {
				if ( ! $this->_allow_overlong && $test == 'range' ) {
					$test = 'none';
					if ( ( $v < 0xA0 && $start_byte == 0xE0 ) || ( $v < 0x90 && $start_byte == 0xF0 ) || ( $v > 0x8F && $start_byte == 0xF4 ) ) {
						$this->_error( 'Bogus UTF-8 character detected (out of legal range) at byte ' . $k );
						
						return false;
					}
				}
				if ( $v >> 6 == 2 ) { // Bit mask must be 10xxxxxx
					$v = ( $v - 128 ) << ( $next_byte * 6 );
					$output[ ( $out_len - 1 ) ] += $v;
					-- $next_byte;
				} else {
					$this->_error( 'Conversion from UTF-8 to UCS-4 failed: malformed input at byte ' . $k );
					
					return false;
				}
				if ( $next_byte < 0 ) {
					$mode = 'next';
				}
			}
		} // for
		return $output;
	}
	
	/**
	 * Removes a weakness of encode(), which cannot properly handle URIs but instead encodes their
	 * path or query components, too.
	 *
	 * @param string $uri Expects the URI as a UTF-8 (or ASCII) string
	 *
	 * @return  string  The URI encoded to Punycode, everything but the host component is left alone
	 * @since 0.6.4
	 */
	public function encode_uri( $uri ) {
		$parsed = parse_url( $uri );
		if ( ! isset( $parsed['host'] ) ) {
			$this->_error( 'The given string does not look like a URI' );
			
			return false;
		}
		$arr = explode( '.', $parsed['host'] );
		foreach ( $arr as $k => $v ) {
			$conv = $this->encode( $v, 'utf8' );
			if ( $conv ) {
				$arr[ $k ] = $conv;
			}
		}
		$parsed['host'] = join( '.', $arr );
		$return         = ( empty( $parsed['scheme'] ) ? '' : $parsed['scheme'] . ( strtolower( $parsed['scheme'] ) == 'mailto' ? ':' : '://' ) ) .
		                  ( empty( $parsed['user'] ) ? '' : $parsed['user'] . ( empty( $parsed['pass'] ) ? '' : ':' . $parsed['pass'] ) . '@' ) .
		                  $parsed['host'] .
		                  ( empty( $parsed['port'] ) ? '' : ':' . $parsed['port'] ) .
		                  ( empty( $parsed['path'] ) ? '' : $parsed['path'] ) .
		                  ( empty( $parsed['query'] ) ? '' : '?' . $parsed['query'] ) .
		                  ( empty( $parsed['fragment'] ) ? '' : '#' . $parsed['fragment'] );
		
		return $return;
	}
	
	/**
	 * Encode a given UTF-8 domain name
	 *
	 * @param    string   Domain name (UTF-8 or UCS-4)
	 * [@param    string   Desired input encoding, see {@link set_parameter}]
	 *
	 * @return   string   Encoded Domain name (ACE string)
	 */
	public function encode( $decoded, $one_time_encoding = false ) {
		// Forcing conversion of input to UCS4 array
		// If one time encoding is given, use this, else the objects property
		switch ( $one_time_encoding ? $one_time_encoding : $this->_api_encoding ) {
			case 'utf8':
				$decoded = $this->_utf8_to_ucs4( $decoded );
				break;
			case 'ucs4_string':
				$decoded = $this->_ucs4_string_to_ucs4( $decoded );
			case 'ucs4_array':
				break;
			default:
				$this->_error( 'Unsupported input format: ' . ( $one_time_encoding ? $one_time_encoding : $this->_api_encoding ) );
				
				return false;
		}
		
		// No input, no output, what else did you expect?
		if ( empty( $decoded ) ) {
			return '';
		}
		
		// Anchors for iteration
		$last_begin = 0;
		// Output string
		$output = '';
		foreach ( $decoded as $k => $v ) {
			// Make sure to use just the plain dot
			switch ( $v ) {
				case 0x3002:
				case 0xFF0E:
				case 0xFF61:
					$decoded[ $k ] = 0x2E;
				// Right, no break here, the above are converted to dots anyway
				// Stumbling across an anchoring character
				case 0x2E:
				case 0x2F:
				case 0x3A:
				case 0x3F:
				case 0x40:
					// Neither email addresses nor URLs allowed in strict mode
					if ( $this->_strict_mode ) {
						$this->_error( 'Neither email addresses nor URLs are allowed in strict mode.' );
						
						return false;
					} else {
						// Skip first char
						if ( $k ) {
							$encoded = '';
							$encoded = $this->_encode( array_slice( $decoded, $last_begin, ( ( $k ) - $last_begin ) ) );
							if ( $encoded ) {
								$output .= $encoded;
							} else {
								$output .= $this->_ucs4_to_utf8( array_slice( $decoded, $last_begin, ( ( $k ) - $last_begin ) ) );
							}
							$output .= chr( $decoded[ $k ] );
						}
						$last_begin = $k + 1;
					}
			}
		}
		// Catch the rest of the string
		if ( $last_begin ) {
			$inp_len = sizeof( $decoded );
			$encoded = '';
			$encoded = $this->_encode( array_slice( $decoded, $last_begin, ( ( $inp_len ) - $last_begin ) ) );
			if ( $encoded ) {
				$output .= $encoded;
			} else {
				$output .= $this->_ucs4_to_utf8( array_slice( $decoded, $last_begin, ( ( $inp_len ) - $last_begin ) ) );
			}
			
			return $output;
		} else {
			if ( false !== ( $output = $this->_encode( $decoded ) ) ) {
				return $output;
			} else {
				return $this->_ucs4_to_utf8( $decoded );
			}
		}
	}
	
	/**
	 * Convert UCS-4 strin into UCS-4 garray
	 *
	 * @param  string $input
	 *
	 * @return array
	 */
	protected function _ucs4_string_to_ucs4( $input ) {
		$output  = [];
		$inp_len = self::byteLength( $input );
		// Input length must be dividable by 4
		if ( $inp_len % 4 ) {
			$this->_error( 'Input UCS4 string is broken' );
			
			return false;
		}
		// Empty input - return empty output
		if ( ! $inp_len ) {
			return $output;
		}
		for ( $i = 0, $out_len = - 1; $i < $inp_len; ++ $i ) {
			// Increment output position every 4 input bytes
			if ( ! ( $i % 4 ) ) {
				$out_len ++;
				$output[ $out_len ] = 0;
			}
			$output[ $out_len ] += ord( $input{$i} ) << ( 8 * ( 3 - ( $i % 4 ) ) );
		}
		
		return $output;
	}
	
	/**
	 * The actual encoding algorithm
	 *
	 * @param  string
	 *
	 * @return mixed
	 */
	protected function _encode( $decoded ) {
		// We cannot encode a domain name containing the Punycode prefix
		$extract    = self::byteLength( $this->_punycode_prefix );
		$check_pref = $this->_utf8_to_ucs4( $this->_punycode_prefix );
		$check_deco = array_slice( $decoded, 0, $extract );
		
		if ( $check_pref == $check_deco ) {
			$this->_error( 'This is already a punycode string' );
			
			return false;
		}
		// We will not try to encode strings consisting of basic code points only
		$encodable = false;
		foreach ( $decoded as $k => $v ) {
			if ( $v > 0x7a ) {
				$encodable = true;
				break;
			}
		}
		if ( ! $encodable ) {
			$this->_error( 'The given string does not contain encodable chars' );
			
			return false;
		}
		// Do NAMEPREP
		$decoded = $this->_nameprep( $decoded );
		if ( ! $decoded || ! is_array( $decoded ) ) {
			return false; // NAMEPREP failed
		}
		$deco_len = count( $decoded );
		if ( ! $deco_len ) {
			return false; // Empty array
		}
		$codecount = 0; // How many chars have been consumed
		$encoded   = '';
		// Copy all basic code points to output
		for ( $i = 0; $i < $deco_len; ++ $i ) {
			$test = $decoded[ $i ];
			// Will match [-0-9a-zA-Z]
			if ( ( 0x2F < $test && $test < 0x40 ) || ( 0x40 < $test && $test < 0x5B ) || ( 0x60 < $test && $test <= 0x7B ) || ( 0x2D == $test ) ) {
				$encoded .= chr( $decoded[ $i ] );
				$codecount ++;
			}
		}
		if ( $codecount == $deco_len ) {
			return $encoded; // All codepoints were basic ones
		}
		// Start with the prefix; copy it to output
		$encoded = $this->_punycode_prefix . $encoded;
		// If we have basic code points in output, add an hyphen to the end
		if ( $codecount ) {
			$encoded .= '-';
		}
		// Now find and encode all non-basic code points
		$is_first = true;
		$cur_code = $this->_initial_n;
		$bias     = $this->_initial_bias;
		$delta    = 0;
		while ( $codecount < $deco_len ) {
			// Find the smallest code point >= the current code point and
			// remember the last ouccrence of it in the input
			for ( $i = 0, $next_code = $this->_max_ucs; $i < $deco_len; $i ++ ) {
				if ( $decoded[ $i ] >= $cur_code && $decoded[ $i ] <= $next_code ) {
					$next_code = $decoded[ $i ];
				}
			}
			$delta += ( $next_code - $cur_code ) * ( $codecount + 1 );
			$cur_code = $next_code;
			
			// Scan input again and encode all characters whose code point is $cur_code
			for ( $i = 0; $i < $deco_len; $i ++ ) {
				if ( $decoded[ $i ] < $cur_code ) {
					$delta ++;
				} elseif ( $decoded[ $i ] == $cur_code ) {
					for ( $q = $delta, $k = $this->_base; 1; $k += $this->_base ) {
						$t = ( $k <= $bias ) ? $this->_tmin :
							( ( $k >= $bias + $this->_tmax ) ? $this->_tmax : $k - $bias );
						if ( $q < $t ) {
							break;
						}
						$encoded .= $this->_encode_digit( intval( $t + ( ( $q - $t ) % ( $this->_base - $t ) ) ) ); //v0.4.5 Changed from ceil() to intval()
						$q = (int) ( ( $q - $t ) / ( $this->_base - $t ) );
					}
					$encoded .= $this->_encode_digit( $q );
					$bias = $this->_adapt( $delta, $codecount + 1, $is_first );
					$codecount ++;
					$delta    = 0;
					$is_first = false;
				}
			}
			$delta ++;
			$cur_code ++;
		}
		
		return $encoded;
	}
	
	/**
	 * Do Nameprep according to RFC3491 and RFC3454
	 *
	 * @param    array    Unicode Characters
	 *
	 * @return   string   Unicode Characters, Nameprep'd
	 */
	protected function _nameprep( $input ) {
		$output = [];
		//
		// Mapping
		// Walking through the input array, performing the required steps on each of
		// the input chars and putting the result into the output array
		// While mapping required chars we apply the cannonical ordering
		foreach ( $input as $v ) {
			// Map to nothing == skip that code point
			if ( in_array( $v, self::$NP['map_nothing'] ) ) {
				continue;
			}
			// Try to find prohibited input
			if ( in_array( $v, self::$NP['prohibit'] ) || in_array( $v, self::$NP['general_prohibited'] ) ) {
				$this->_error( 'NAMEPREP: Prohibited input U+' . sprintf( '%08X', $v ) );
				
				return false;
			}
			foreach ( self::$NP['prohibit_ranges'] as $range ) {
				if ( $range[0] <= $v && $v <= $range[1] ) {
					$this->_error( 'NAMEPREP: Prohibited input U+' . sprintf( '%08X', $v ) );
					
					return false;
				}
			}
			
			if ( 0xAC00 <= $v && $v <= 0xD7AF ) {
				// Hangul syllable decomposition
				foreach ( $this->_hangul_decompose( $v ) as $out ) {
					$output[] = (int) $out;
				}
			} elseif ( ( $this->_idn_version == '2003' ) && isset( self::$NP['replacemaps_2003'][ $v ] ) ) {
				foreach ( $this->_apply_cannonical_ordering( self::$NP['replacemaps_2003'][ $v ] ) as $out ) {
					$output[] = (int) $out;
				}
			} elseif ( ( $this->_idn_version == '2008' ) && isset( self::$NP['replacemaps'][ $v ] ) ) {
				foreach ( $this->_apply_cannonical_ordering( self::$NP['replacemaps'][ $v ] ) as $out ) {
					$output[] = (int) $out;
				}
			} else {
				$output[] = (int) $v;
			}
		}
		// Before applying any Combining, try to rearrange any Hangul syllables
		$output = $this->_hangul_compose( $output );
		//
		// Combine code points
		//
		$last_class   = 0;
		$last_starter = 0;
		$out_len      = count( $output );
		for ( $i = 0; $i < $out_len; ++ $i ) {
			$class = $this->_get_combining_class( $output[ $i ] );
			if ( ( ! $last_class || $last_class > $class ) && $class ) {
				// Try to match
				$seq_len = $i - $last_starter;
				$out     = $this->_combine( array_slice( $output, $last_starter, $seq_len ) );
				// On match: Replace the last starter with the composed character and remove
				// the now redundant non-starter(s)
				if ( $out ) {
					$output[ $last_starter ] = $out;
					if ( count( $out ) != $seq_len ) {
						for ( $j = $i + 1; $j < $out_len; ++ $j ) {
							$output[ $j - 1 ] = $output[ $j ];
						}
						unset( $output[ $out_len ] );
					}
					// Rewind the for loop by one, since there can be more possible compositions
					$i --;
					$out_len --;
					$last_class = ( $i == $last_starter ) ? 0 : $this->_get_combining_class( $output[ $i - 1 ] );
					continue;
				}
			}
			// The current class is 0
			if ( ! $class ) {
				$last_starter = $i;
			}
			$last_class = $class;
		}
		
		return $output;
	}
	
	/**
	 * Decomposes a Hangul syllable
	 * (see http://www.unicode.org/unicode/reports/tr15/#Hangul
	 *
	 * @param    integer 32bit UCS4 code point
	 *
	 * @return   array    Either Hangul Syllable decomposed or original 32bit value as one value array
	 */
	protected function _hangul_decompose( $char ) {
		$sindex = (int) $char - $this->_sbase;
		if ( $sindex < 0 || $sindex >= $this->_scount ) {
			return [ $char ];
		}
		$result   = [];
		$result[] = (int) $this->_lbase + $sindex / $this->_ncount;
		$result[] = (int) $this->_vbase + ( $sindex % $this->_ncount ) / $this->_tcount;
		$T        = intval( $this->_tbase + $sindex % $this->_tcount );
		if ( $T != $this->_tbase ) {
			$result[] = $T;
		}
		
		return $result;
	}
	
	/**
	 * Applies the cannonical ordering of a decomposed UCS4 sequence
	 *
	 * @param    array      Decomposed UCS4 sequence
	 *
	 * @return   array      Ordered USC4 sequence
	 */
	protected function _apply_cannonical_ordering( $input ) {
		$swap = true;
		$size = count( $input );
		while ( $swap ) {
			$swap = false;
			$last = $this->_get_combining_class( intval( $input[0] ) );
			for ( $i = 0; $i < $size - 1; ++ $i ) {
				$next = $this->_get_combining_class( intval( $input[ $i + 1 ] ) );
				if ( $next != 0 && $last > $next ) {
					// Move item leftward until it fits
					for ( $j = $i + 1; $j > 0; -- $j ) {
						if ( $this->_get_combining_class( intval( $input[ $j - 1 ] ) ) <= $next ) {
							break;
						}
						$t               = intval( $input[ $j ] );
						$input[ $j ]     = intval( $input[ $j - 1 ] );
						$input[ $j - 1 ] = $t;
						$swap            = true;
					}
					// Reentering the loop looking at the old character again
					$next = $last;
				}
				$last = $next;
			}
		}
		
		return $input;
	}
	
	/**
	 * Returns the combining class of a certain wide char
	 *
	 * @param    integer    Wide char to check (32bit integer)
	 *
	 * @return   integer    Combining class if found, else 0
	 */
	protected function _get_combining_class( $char ) {
		return isset( self::$NP['norm_combcls'][ $char ] ) ? self::$NP['norm_combcls'][ $char ] : 0;
	}
	
	/**
	 * Ccomposes a Hangul syllable
	 * (see http://www.unicode.org/unicode/reports/tr15/#Hangul
	 *
	 * @param    array    Decomposed UCS4 sequence
	 *
	 * @return   array    UCS4 sequence with syllables composed
	 */
	protected function _hangul_compose( $input ) {
		$inp_len = count( $input );
		if ( ! $inp_len ) {
			return [];
		}
		$result   = [];
		$last     = (int) $input[0];
		$result[] = $last; // copy first char from input to output
		
		for ( $i = 1; $i < $inp_len; ++ $i ) {
			$char   = (int) $input[ $i ];
			$sindex = $last - $this->_sbase;
			$lindex = $last - $this->_lbase;
			$vindex = $char - $this->_vbase;
			$tindex = $char - $this->_tbase;
			// Find out, whether two current characters are LV and T
			if ( 0 <= $sindex && $sindex < $this->_scount && ( $sindex % $this->_tcount == 0 ) && 0 <= $tindex && $tindex <= $this->_tcount ) {
				// create syllable of form LVT
				$last += $tindex;
				$result[ ( count( $result ) - 1 ) ] = $last; // reset last
				continue; // discard char
			}
			// Find out, whether two current characters form L and V
			if ( 0 <= $lindex && $lindex < $this->_lcount && 0 <= $vindex && $vindex < $this->_vcount ) {
				// create syllable of form LV
				$last                               = (int) $this->_sbase + ( $lindex * $this->_vcount + $vindex ) * $this->_tcount;
				$result[ ( count( $result ) - 1 ) ] = $last; // reset last
				continue; // discard char
			}
			// if neither case was true, just add the character
			$last     = $char;
			$result[] = $char;
		}
		
		return $result;
	}
	
	/**
	 * Do composition of a sequence of starter and non-starter
	 *
	 * @param    array      UCS4 Decomposed sequence
	 *
	 * @return   array      Ordered USC4 sequence
	 */
	protected function _combine( $input ) {
		$inp_len = count( $input );
		if ( 0 == $inp_len ) {
			return false;
		}
		foreach ( self::$NP['replacemaps'] as $np_src => $np_target ) {
			if ( $np_target[0] != $input[0] ) {
				continue;
			}
			if ( count( $np_target ) != $inp_len ) {
				continue;
			}
			$hit = false;
			foreach ( $input as $k2 => $v2 ) {
				if ( $v2 == $np_target[ $k2 ] ) {
					$hit = true;
				} else {
					$hit = false;
					break;
				}
			}
			if ( $hit ) {
				return $np_src;
			}
		}
		
		return false;
	}
	
	/**
	 * Encoding a certain digit
	 *
	 * @param    int $d
	 *
	 * @return string
	 */
	protected function _encode_digit( $d ) {
		return chr( $d + 22 + 75 * ( $d < 26 ) );
	}
	
	/**
	 * Use this method to get the last error ocurred
	 *
	 * @param    void
	 *
	 * @return   string   The last error, that occured
	 */
	public function get_last_error() {
		return $this->_error;
	}
	
	/**
	 * Attempts to return a concrete IDNA instance for either php4 or php5,
	 * only creating a new instance if no IDNA instance with the same
	 * parameters currently exists.
	 *
	 * @param array $params Set of paramaters
	 *
	 * @return object idna_convert
	 * @access public
	 */
	public function singleton( $params = [] ) {
		static $instances = [];
		
		$signature = serialize( $params );
		if ( ! isset( $instances[ $signature ] ) ) {
			$instances[ $signature ] = idna_convert::getInstance( $params );
		}
		
		return $instances[ $signature ];
	}
	
	/**
	 * Attempts to return a concrete IDNA instance.
	 *
	 * @param array $params Set of paramaters
	 *
	 * @return idna_convert
	 * @access public
	 */
	public function getInstance( $params = [] ) {
		return new idna_convert( $params );
	}
}