<?php

namespace FDG\OnlineContract\Admin;

use FDG\OnlineContract\Setup\Setup;
use FDG\OnlineContract\Setup\Init;
use FDG\OnlineContract\Frontend\Frontend;
use FDG\OnlineContract\Setup\Messages;
use WP_Query;

class Admin
{
    /**
     * Initialize a new instance of the Contract class
     *
     */
    public function __construct()
    {
        $this->settings = Init::getSettings();
        $this->setup = (new Setup(Init::getSettings()));

        $currentPlugins = $this->get_plugin_info();
        foreach ($currentPlugins as $plugin) {
            if ($plugin['name'] == 'Online Contract PDF Print Integration' && $plugin['version'] < '1.0.2' && $plugin['active'] == 1) {
                (new Messages('Your Online Contract PDF Print Addon needs to be updated in order to work better with the newest version of the WP Online Contract plugin.', 'error'));
            }
            if ($plugin['name'] == 'Online Contract Multisite' && $plugin['version'] < '1.0.1' && $plugin['active'] == 1) {
                (new Messages('Your Online Contract Multisite Addon needs to be updated in order to work better with the newest version of the WP Online Contract plugin.', 'error'));
            }
            if ($plugin['name'] == 'Online Contract Stripe Payments' && $plugin['version'] < '2.1.1' && $plugin['active'] == 1) {
                (new Messages('Your Online Contract Stripe Payments Addon needs to be updated in order to work better with the newest version of the WP Online Contract plugin.', 'error'));
            }
        }
    }

    /**
     * Create hooks for admin calls
     *
     * @return void
     */
    public function registerHooks()
    {
        add_action('admin_enqueue_scripts', [$this, 'enqueueAssets']);
        add_filter('dashboard_glance_items', [$this, 'contractGlanceItems'], 10, 1);
        add_action('wp_dashboard_setup', [$this, 'addDashboardWidgets']);
    }

    /**
     * Add css and js for admin
     *
     * @return void
     */
    public function enqueueAssets()
    {
        $typeRocketThemeUrl = ONLINE_CONTRACT_URL . 'typerocket/wordpress/assets/templates';
        wp_enqueue_style('fdg_contract_admin_css', $typeRocketThemeUrl . '/css/admin.css');
        wp_enqueue_script('fdg_contract_admin_js', $typeRocketThemeUrl . '/js/admin.js');
        wp_enqueue_script('fdg_contract_admin', ONLINE_CONTRACT_URL . 'js/adminContract.js', array('jquery'));
    }

    /**
     * Convert shortcodes from the form / options to human-readable
     *
     * @param object $form
     * @param string $shortcode
     * @return string
     */
    public static function convertShortCodeAdmin($form, $shortcode)
    {
        $formret = '';
        $desc = '';
        switch ($shortcode['data_type']) {
            case 'text':
                $formret = $form->text($shortcode['label']);
                if (isset($shortcode['required_input_on_front'])) {
                    $formret->setAttribute('required', true);
                }
                if ($shortcode['descriptive_text']) {
                    $desc = '<small>' . $shortcode['descriptive_text'] . '</small><br><br>';
                }
                break;
            case 'textarea':
                $formret = $form->textarea($shortcode['label']);
                if (isset($shortcode['required_input_on_front'])) {
                    $formret->setAttribute('required', true);
                }
                if ($shortcode['descriptive_text']) {
                    $desc = '<small>' . $shortcode['descriptive_text'] . '</small><br><br>';
                }
                break;
            case 'date':
                $formret = $form->date($shortcode['label']);
                if (isset($shortcode['required_input_on_front'])) {
                    $formret->setAttribute('required', true);
                }
                if ($shortcode['descriptive_text']) {
                    $desc = '<small>' . $shortcode['descriptive_text'] . '</small><br><br>';
                }
                break;
            case 'checkbox':
                $formret = $form->checkbox($shortcode['label'])->setText(' ' . $shortcode['label'])->setRenderSetting('raw');
                if (isset($shortcode['required_input_on_front'])) {
                    $formret->setAttribute('required', true);
                }
                $desc = '<br><small>' . $shortcode['descriptive_text'] . '</small><br><br>';
                break;
            case 'radio':
                $options = array();
                $optionsarr = explode(',', str_replace(', ', ',', $shortcode['options']));
                foreach ($optionsarr as $option) {
                    $optionval = strtolower(str_replace(' ', '_', preg_replace("/[^A-Za-z0-9 ]/", '', $option)));
                    $options[$option] = $optionval;
                }
                $formret = $form->radio($shortcode['label'])->setOptions($options);
                if (isset($shortcode['required_input_on_front'])) {
                    $formret->setAttribute('required', true);
                }
                $desc = '<small>' . $shortcode['descriptive_text'] . '</small><br><br>';
                break;
            case 'select':
                $options = array();
                $optionsarr = explode(',', str_replace(', ', ',', $shortcode['options']));
                foreach ($optionsarr as $option) {
                    $optionval = strtolower(str_replace(' ', '_', preg_replace("/[^A-Za-z0-9 ]/", '', $option)));
                    $options[$option] = $optionval;
                }
                $formret = $form->select($shortcode['label'])->setOptions($options);
                if ($shortcode['required_input_on_front'] == 1) {
                    $formret->setAttribute('required', true);
                }
                if ($shortcode['descriptive_text']) {
                    $desc = '<small>' . $shortcode['descriptive_text'] . '</small><br><br>';
                }
                break;
            case 'file':
                $formret = $form->file($shortcode['label']);
                if (isset($shortcode['required_input_on_front'])) {
                    $formret->setAttribute('required', true);
                }
                if ($shortcode['descriptive_text']) {
                    $desc = '<small>' . $shortcode['descriptive_text'] . '</small><br><br>';
                }
                break;
            case 'editor':
                $formret = $form->editor($shortcode['label']);
                if (isset($shortcode['required_input_on_front'])) {
                    $formret->setAttribute('required', true);
                }
                if ($shortcode['descriptive_text']) {
                    $desc = '<small style="position: relative; top: -20px;">' . $shortcode['descriptive_text'] . '</small><br><br>';
                }
                break;
        }

        return $formret . $desc;
    }

    /**
     * Check to make sure contract plugin setup has been completed
     *
     * @return string
     */
    public static function fieldChecks()
    {
        $settings = Init::getSettings();
        $chk = '';

        if (tr_options_field('fdg_shortcodes.contract_shortcodes') == null) {
            $chk .= '<li>' . __('You have not created any shortcodes. Please add new shortcodes or the plugin will not work as expected.', 'onlinecontract') . '</li>';
        }

        if (tr_options_field('fdg_template_options.contract_templates') == null) {
            $chk .= '<li>' . __('You have not created any contract templates. Please add new contract template under Templates > Contract Templates in order to create a new contract.', 'onlinecontract') . '</li>';
        }

        if ($settings['companyName'] == null) {
            $chk .= '<li>' . __('Enter your company name into the Settings > Plugin Setup to ensure your company is used in the email notifications.', 'onlinecontract') . '</li>';
        }

        if ($settings['currencyType'] == null) {
            $chk .= '<li>' . __('Enter your your currency type into the Settings > Plugin Setup to ensure payments are setup to us the correct type.', 'onlinecontract') . '</li>';
        }

        if ($settings['adminEmail'] == null) {
            $chk .= '<li>' . __('Enter your administrative email into the Settings > Plugin Setup. This will be used to send emails from.', 'onlinecontract') . '</li>';
        }

        if ($settings['dateFormat'] == null) {
            $chk .= '<li>' . __('You should add a date format in the Settings > Plugin Setup. This will set the visible format for the dates throughout the plugin.', 'onlinecontract') . '</li>';
        }

        if ($settings['datePickerFormat'] == null) {
            $chk .= '<li>' . __('You should add a date picker format in the Settings > Plugin Setup. This will set the date format for the date pickers used throughout the plugin.', 'onlinecontract') . '</li>';
        }

        if ($settings['emailSubject'] == null) {
            $chk .= '<li>' . __('You should add a email subject in the Notification Email Template on the Templates > Other Templates screen. This will allow you to send emails with a custom subject.', 'onlinecontract') . '</li>';
        }

        if ($settings['emailContent'] == null) {
            $chk .= '<li>' . __('You should add contents of your notification email in the Notification Email Template on the Templates > Other Templates screen. This ensures that you notification email has content.', 'onlinecontract') . '</li>';
        }

        if ($chk != '') {
            $chk = '<h3>' . __('Before creating a contract, make sure you have finished your setup.', 'onlinecontract') . '</h3><ul class="fieldcheck">' . $chk . '</ul><p>' . __('Need help? We are here to assist you. Drop us a line by visiting our Help Desk and submitting a ticket or view our answers to common problems. You can view our help desk here:', 'onlinecontract') . ' <a href="https://support.futuredesigngroup.com" target="_blank">https://support.futuredesigngroup.com/</a>';
        }

        return $chk;
    }

    /**
     * Enqueue scripts for admin datepicker
     *
     * @return void
     */
    public function admin_enqueue_script()
    {
        global $post_type;
        $settings = Init::getSettings();
        wp_dequeue_script('jquery-ui-datepicker');
        if ($post_type == 'contract') {
            if ($settings['datePickerFormat'] != NULL) {
                wp_enqueue_script('fdg_contract_date_js', ONLINE_CONTRACT_URL . 'js/admin.js', array('jquery', 'jquery-ui-datepicker'));
                $data = array(
                    'dateformat' => $settings['datePickerFormat']
                );
                wp_localize_script('fdg_contract_date_js', '_fdg_contractadmin_send', $data);
            }
        }
    }

    /**
     * Add dashboard items
     *
     * @param array $items
     * @return array
     */
    public function contractGlanceItems($items = array())
    {
        echo '<style>';
        echo '.post-count.contract-count a:before, .post-count.contract-count span:before{
            font: 400 15px/1 typerocket-icons !important;
            content: \'\e905\' !important;
            speak: none !important;
            top: 2px;
            position: relative;
            -webkit-font-smoothing: antialiased;
        }';
        echo '</style>';
        $post_types = array('contract');
        foreach ($post_types as $type) {
            if (!post_type_exists($type)) continue;
            $num_posts = wp_count_posts($type);
            if ($num_posts) {
                $published = intval($num_posts->publish);
                $post_type = get_post_type_object($type);
                $text = _n('%s ' . $post_type->labels->singular_name, '%s ' . $post_type->labels->name, $published, 'onlinecontract');
                $text = sprintf($text, number_format_i18n($published));
                if (current_user_can($post_type->cap->edit_posts)) {
                    $output = '<a href="edit.php?post_type=' . $post_type->name . '">' . $text . '</a>';
                    echo '<li class="post-count ' . $post_type->name . '-count">' . $output . '</li>';
                } else {
                    $output = '<span>' . $text . '</span>';
                    echo '<li class="post-count ' . $post_type->name . '-count">' . $output . '</li>';
                }
            }
        }
        return $items;
    }

    /**
     * Add new dashboard widget
     *
     * @param object $post
     * @param object $callback_args
     * @return string
     */
    public function dashboardWidgetFunction($post, $callback_args)
    {
        $contracts = '';
        $current_user = wp_get_current_user();
        $email = $current_user->user_email;
        $emailshortcode = $this->settings['clientEmailShortcode'];

        $emailshortcode = Frontend::convertSingleShortcode('[' . $emailshortcode . ']');
        $args = array(
            'post_type' => 'contract',
            'meta_query' => array(
                array(
                    'key' => $emailshortcode,
                    'value' => $email,
                    'compare' => '=',
                )
            )
        );
        $query = new WP_Query($args);
        $i = 0;
        if ($query->have_posts()) :
            while ($query->have_posts()) : $query->the_post();
                if (tr_posts_field('remove_signature_box?') != 1) {
                    if (tr_posts_field('signature') == '-') {
                        $status = array('Unsigned', 'red');
                    } else {
                        $status = array('Signed', 'green');
                    }
                } else {
                    $status = array('Cannot Sign Online', 'gray');
                }
                $contracts .= '<a href="' . get_the_permalink() . '" target="_blank"><div style="float:right; background-color:' . $status[1] . '; color:#fff; font-weight:bold; padding:5px 10px; position:relative; top:-5px;">' . $status[0] . '</div>' . __('View') . ' ' . get_the_title() . '</a><div style="clear:both; float:none;"></div><br>';
            endwhile;
        else :
            $contracts = '<p>' . __('You do not have any open contracts.', 'onlinecontract') . '</p>';

        endif;

        echo wp_kses($contracts, 'post');
    }

    /**
     * Add dashboard widget function to dashboard
     *
     * @return void
     */
    public function addDashboardWidgets()
    {
        wp_add_dashboard_widget('dashboard_widget', __('Your Contracts', 'onlinecontract'), [$this, 'dashboardWidgetFunction']);
    }

    /**
     * Get a list of all custom and default shortcodes
     *
     * @return string
     */
    public static function getAllShortcodes()
    {
        $str = '<h3>' . __('Default Plugin Shortcodes', 'onlinecontract') . '</h3>';
        $str .= '<strong>[contract-link]</strong><br>' . __('This will display a link to the contract', 'onlinecontract') . ' <strong style="color:red;">(used only for email notifications)</strong><br><br>';
        $str .= '<strong>[businessname]</strong><br>' . __('This will allow you to add your business name on your contract', 'onlinecontract') . '<br><br>';
        $str .= '<strong>[contractcurrency]</strong><br>' . __('This will display your currency symbol on your contract', 'onlinecontract') . '<br><br>';
        $str .= '<strong>[postid]</strong><br>' . __('This will display the post ID of the contract', 'onlinecontract') . '<br><br>';
        $str .= '<strong>[currentdate]</strong><br>' . __('This will display the current date using the format you chose in the settings', 'onlinecontract');

        $str .= '<h3>' . __('Available Custom Shortcodes', 'onlinecontract') . '</h3>';
        foreach (tr_options_field('fdg_shortcodes.contract_shortcodes') as $shortcode) {
            $str .= '<strong>' . $shortcode['shortcode'] . '</strong><br>' . $shortcode['descriptive_text'] . '<br><br>';
        }
        return $str;
    }

    /**
     * Get information about contract plugin, used for updates and information
     *
     * @return array
     */
    public function get_plugin_info()
    {

        // Get all plugins
        include_once(ABSPATH . 'wp-admin/includes/plugin.php');
        $all_plugins = get_plugins();

        // Get active plugins
        $active_plugins = get_option('active_plugins');

        // Assemble array of name, version, and whether plugin is active (boolean)
        foreach ($all_plugins as $key => $value) {
            $is_active = (in_array($key, $active_plugins)) ? true : false;
            $plugins[$key] = array(
                'name'    => $value['Name'],
                'version' => $value['Version'],
                'active'  => $is_active,
            );
        }

        return $plugins;
    }
}
