<?php

namespace FDG\OnlineContract\Admin;

use FDG\OnlineContract\Frontend\Frontend;
use FDG\OnlineContract\Admin\Admin;
use FDG\OnlineContract\Setup\Setup;
use FDG\OnlineContract\Setup\Init;
use FDG\OnlineContract\Setup\Messages;

class Contract
{
    /**
     * Initialize a new instance of the Contract class
     *
     */
    public function __construct()
    {
        $this->settings = Init::getSettings();
        $this->setup = (new Setup(Init::getSettings()));
        $this->cpt = tr_post_type('contract', __('Contracts', 'onlinecontract'));
        $this->cpt->setIcon('pencil');
        if ($this->setup->validatePurchaseCode() == 1) {
            if ($this->settings['slug'] != NULL) {
                $this->cpt->setSlug($this->settings['slug']);
            } else {
                $this->cpt->setSlug('contract');
            }
            $this->buildCPTMeta();
            $this->buildCPTCategory();
        } else {
            new Messages(__('Purchase Code Missing or Invalid', 'onlinecontract'), 'notice-error');
            $this->cpt->setArgument('supports', ['']);
            $this->cptMeta = tr_meta_box(__(__('Purchase Code Missing or Invalid', 'onlinecontract'), 'onlinecontract'));
            $this->cptMeta->addScreen('contract');
            $this->cptMeta->setCallback(
                function () {
                    echo '<p>' . __('Your purchase is either invalid or has not been input. Please go to the Purchase Code page using the following link. Complete the installation of our plugin. Once this is done, your custom options will appear here.', 'onlinecontract') . '<br><a href="' . get_admin_url() . 'edit.php?post_type=contract&page=purchase">' . __('Purchase Code', 'onlinecontract') . '</a></p>
                    <a href="https://codecanyon.net/user/futuredesigngrp/portfolio?ref=futuredesigngrp" target="_blank">' . __('Purchase', 'onlinecontract') . ' WP Online Contract</a></p>';
                }
            );
        }
    }

    /**
     * Register all contract hooks
     *
     * @return void
     */
    public function registerHooks()
    {
        add_filter('manage_contract_posts_columns', [$this, 'createColumns']);
        add_action('manage_contract_posts_custom_column',  [$this, 'showColumns']);
        add_filter('manage_edit-contract_sortable_columns', [$this, 'sortableColumns']);
        add_action('pre_get_posts', [$this, 'customOrderBy']);
    }

    /**
     * Build the contract post type meta if fields checks is true
     *
     * @return void
     */
    public function buildCPTMeta()
    {
        $this->cptMeta = tr_meta_box(__('Default Options', 'onlinecontract'));
        $this->cptMeta->setLabel(__('Default Options', 'onlinecontract'));
        $this->cptMeta->addScreen('contract');
        $this->cptMeta->setPriority('high');
        $this->cptMeta->setCallback(
            function () {
                global $post;
                if (Admin::fieldChecks() == '') {
                    $options = array();
                    $shortcodespayment = array(__('Do not display payment options', 'onlinecontract') => 'no');
                    $shortcodes = array('' => '');
                    $shortcodessimple = array('' => '');
                    $form = tr_form();
                    if (tr_posts_field('signature') != '-' && tr_posts_field('signature') != NULL) {
                        echo '<img src="' . tr_posts_field('signature') . '" style="width:100%; height:auto;">';
                        echo '<p style="text-align:center;">' . __('This contract has already been signed. It cannot be modified until you remove the signature and have your client resign.', 'onlinecontract') . '</p>';
                        echo '<p style="text-align:center;">' . __('Signed');
                        if (tr_posts_field('printed_name') != '-' && tr_posts_field('printed_name') != NULL) {
                            echo ' by <strong>' . tr_posts_field('printed_name') . '</strong>';
                        }
                        echo ' on <strong>' . tr_posts_field('generated_sign_date') . '</strong>';
                        echo '</p>';
                        echo '<p style="text-align:center;"><span class="spinner_fdg"></span><a href="' . get_permalink() . 'removesignature/' . $post->ID . '" onclick="addSpinner(jQuery(\'spinner\'))" class="removesignature">' . __('Remove Signature', 'onlinecontract') . '</a></p>';
                    } else {
                        echo $form->hidden('Signature')->setDefault('-');
                        echo $form->hidden('Printed Name')->setDefault('-');
                        echo $form->hidden('IP Address')->setDefault('-');
                        echo $form->hidden('Generated Sign Date')->setDefault('-');
                        $options[''] = '';
                        foreach (tr_options_field('fdg_template_options.contract_templates') as $template) {
                            $options[$template['contract_name']] = $template['contract_name'];
                        }
                        foreach (tr_options_field('fdg_shortcodes.contract_shortcodes') as $shortcode) {
                            $shortcodespayment[$shortcode['label']] = str_replace(array('[', ']'), '', $shortcode['shortcode']);
                            $shortcodes[$shortcode['label']] = str_replace(array('[', ']'), '', $shortcode['shortcode']);
                            $shortcodessimple[$shortcode['label']] = strtolower(str_replace(array(' '), '_', $shortcode['label']));
                        }
                        $selectedTemplate = '';
                        if (isset($_GET['template'])) {
                            $selectedTemplate = $_GET['template'];
                        }
                        echo $form->row(
                            $form->select('Contract Template')->setOptions($options)->setLabel(__('Contract Template', 'onlinecontract'))->setDefault($selectedTemplate),
                            $form->text('Redirect after signing?')->setLabel(__('Redirect after signing?', 'onlinecontract'))
                        );
                        echo $form->row(
                            $form->select('Payment Option Shortcode')->setOptions($shortcodespayment)->setLabel(__('Payment Option Shortcode', 'onlinecontract')),
                            $form->select('Payment Item Name Shortcode')->setOptions($shortcodes)->setLabel(__('Payment Item Name Shortcode', 'onlinecontract'))
                        );
                        echo $form->row(
                            $form->select('Your Representative Shortcode')->setOptions($shortcodes)->setLabel(__('Your Representative Shortcode', 'onlinecontract')),
                            $form->select('Client Email Shortcode')->setOptions($shortcodessimple)->setLabel(__('Client Email Shortcode', 'onlinecontract'))
                        );
                        echo $form->checkbox('Can This Contract Be Repeatedly Signed?')->setLabel(__('Can This Contract Be Repeatedly Signed?', 'onlinecontract'));
                        echo $form->checkbox('Remove Signature Box?')->setLabel(__('Remove Signature Box?', 'onlinecontract'));
                        echo $form->checkbox('Force Use to Print Name?')->setLabel(__('Force User to Print Name?', 'onlinecontract'));
                        echo $form->checkbox('Force Use to Use Prepared Fields?')->setLabel(__('Force User to Use Prepared Fields?', 'onlinecontract'));
                        echo $form->checkbox('Send Email to Client on Update?')->setLabel(__('Send Email to Client on Update?', 'onlinecontract'));
                        echo '<small>' . __('Only check this box after you have saved the contract at least once. If not, the email may be sent with incomplete information.', 'onlinecontract') . '</small>';
                    }
                } else {
                    echo Admin::fieldChecks();
                }
            }
        );
        $this->cptMeta = tr_meta_box('Custom Options');
        if (tr_posts_field('signature') == '-' || tr_posts_field('signature') == NULL) {
            $this->cptMeta->setLabel(__('Contract Details', 'onlinecontract'));
        } else {
            $this->cptMeta->setLabel(__('Custom Options', 'onlinecontract'));
        }
        $this->cptMeta->addScreen('contract');
        $this->cptMeta->setPriority('core');
        $this->cptMeta->setCallback(
            function () {
                global $post;
                if (Admin::fieldChecks() == '') {
                    if (tr_posts_field('signature') == '-' || tr_posts_field('signature') == NULL) {
                        $form = tr_form();
                        $i = 0;
                        foreach (tr_options_field('fdg_shortcodes.contract_shortcodes') as $singleshortcode) {
                            if (isset($singleshortcode['template']) && $singleshortcode['template'] != '') {
                                $i++;
                            }
                        }
                        if ($i == 0) {
                            foreach (tr_options_field('fdg_shortcodes.contract_shortcodes') as $shortcode) {
                                if (!isset($shortcode['function_only'])) {
                                    echo Admin::convertShortcodeAdmin($form, $shortcode);
                                }
                            }
                        } else {
                            foreach (tr_options_field('fdg_shortcodes.contract_shortcodes') as $shortcode) {
                                if (!isset($shortcode['function_only'])) {
                                    $this->selectShortCodes($form, $shortcode, tr_posts_field('contract_template'), tr_options_field('fdg_plugin_options.default_contract_template'));
                                }
                            }
                        }
                    } else {
                        echo '<table style="width:100%;">';
                        echo '<tr><td colspan="2"><h3>' . __('Default Options', 'onlinecontract') . '</h3></td></tr>';
                        echo '<tr>
                            <td><strong>' . __('Contract Template', 'onlinecontract') . '</strong></td>
                            <td style="text-align:left; width:70%;">' . tr_posts_field('contract_template') . '</td>
                            </tr>
                            <tr><td colspan="2">&nbsp;</td></tr>';
                        if (tr_posts_field('payment_option_shortcode') != 'no') {
                            echo '<tr>
                            <td><strong>' . __('Accept Payment?', 'onlinecontract') . '</strong></td>
                            <td style="text-align:left; width:70%;">' . __('Yes', 'onlinecontract') . '</td>
                            </tr>
                            <tr><td colspan="2">&nbsp;</td></tr>';
                        } else {
                            echo '<tr>
                            <td><strong>' . __('Accept Payment?', 'onlinecontract') . '</strong></td>
                            <td style="text-align:left; width:70%;">' . __('No', 'onlinecontract') . '</td>
                            </tr>
                            <tr><td colspan="2">&nbsp;</td></tr>';
                        }
                        if (tr_posts_field('redirect_after_signing?') != NULL) {
                            echo '<tr>
                            <td><strong>' . __('Redirect After Signing?', 'onlinecontract') . '</strong></td>
                            <td style="text-align:left; width:70%;">' . tr_posts_field('redirect_after_signing?') . '</td>
                            </tr>
                            <tr><td colspan="2">&nbsp;</td></tr>';
                        } else {
                            echo '<tr>
                            <td><strong>' . __('Redirect After Signing?', 'onlinecontract') . '</strong></td>
                            <td style="text-align:left; width:70%;">' . __('No', 'onlinecontract') . '</td>
                            </tr>
                            <tr><td colspan="2">&nbsp;</td></tr>';
                        }
                        echo '<tr><td colspan="2"><hr></td></tr>';
                        echo '<tr><td colspan="2"><h3>' . __('Custom Options', 'onlinecontract') . '</h3></td></tr>';
                        foreach (tr_options_field('fdg_shortcodes.contract_shortcodes') as $shortcode) {
                            if (!isset($shortcode['function_only'])) {
                                echo '<tr><td><strong>' . $shortcode['label'] . '</strong></td><td style="text-align:left; width:70%;">' . Frontend::formatSingleShortcode($shortcode['shortcode']) . '</td></tr><tr><td colspan="2">&nbsp;</td></tr>';
                            }
                        }
                        if (is_plugin_active('wp-online-contract-woocommerce/online-contract-woocommerce.php')) {
                            echo '<tr><td colspan="2"><hr></td></tr>';
                            echo '<tr><td colspan="2"><h3>' . __('Woocommerce Options', 'onlinecontract') . '</h3></td></tr>';
                            echo '<tr><td><strong>' . __('Full Address', 'onlinecontract_woo') . '</strong></td><td style="text-align:left; width:70%;">' . tr_posts_field('woocommerce_mailing_address') . '</td></tr><tr><td colspan="2">&nbsp;</td></tr>';
                        }
                        echo '</table>';
                    }
                } else {
                    echo '<p>' . __('Please fix the items above before this section will open.', 'onlinecontract') . '</p>';
                }
            }
        );

        $this->cptMeta = tr_meta_box('Audit Log');
        $this->cptMeta->setPriority('low');
        $this->cptMeta->addScreen('contract');
        $this->cptMeta->setCallback(
            function () {
                global $post;
                $form = tr_form();
                echo $form->hidden('Audit Log');
                echo $form->textarea('Audit Log')->getValue();
            }
        );
    }

    /**
     * Undocumented function
     *
     * @return void
     */
    public function selectShortCodes($form, $shortcode, $tr_contract_template, $tr_contract_template_default)
    {
        if (isset($_GET['template'])) {
            $tr_contract_template =  $_GET['template'];
        }
        if ($tr_contract_template != $tr_contract_template_default) {
            if ($shortcode['template'] == $tr_contract_template) {
                echo Admin::convertShortcodeAdmin($form, $shortcode);
            }
        } else {
            if (!isset($shortcode['template']) || $shortcode['template'] == '') {
                echo Admin::convertShortcodeAdmin($form, $shortcode);
            }
        }
    }

    /**
     * Build contract post type category
     *
     * @return void
     */
    public function buildCPTCategory()
    {
        $this->category = tr_taxonomy('Contract Type');
        $this->category->apply($this->cpt);
        $this->category->setArgument(
            'labels',
            array(
                'add_new_item' => 'Add New ' . __('Contract Type', 'onlinecontract'),
                'add_or_remove_items' => 'Add or remove contract types',
                'all_items' => 'All ' . __('Contract Type', 'onlinecontract') . 's',
                'choose_from_most_used' => 'Choose from the most used contract types',
                'edit_item' => 'Edit ' . __('Contract Type', 'onlinecontract'),
                'name' => '' . __('Contract Type', 'onlinecontract') . 's',
                'menu_name' => '' . __('Contract Type', 'onlinecontract') . 's',
                'new_item_name' => 'New ' . __('Contract Type', 'onlinecontract') . ' Name',
                'not_found' => 'No contract types found.',
                'parent_item' => 'Parent ' . __('Contract Type', 'onlinecontract'),
                'parent_item_colon' => 'Parent ' . __('Contract Type', 'onlinecontract') . ':',
                'popular_items' => 'Popular ' . __('Contract Type', 'onlinecontract') . 's',
                'search_items' => 'Search ' . __('Contract Type', 'onlinecontract') . 's',
                'separate_items_with_commas' => 'Separate contract types with commas',
                'singular_name' => '' . __('Contract Type', 'onlinecontract'),
                'update_item' => 'Update ' . __('Contract Type', 'onlinecontract'),
                'view_item' => 'View ' . __('Contract Type', 'onlinecontract')
            )
        );
        $this->category->setHierarchical();
        if (Admin::fieldChecks() == '') {
            $this->cpt->setArgument('supports', ['title', 'revisions']);
        } else {
            $this->cpt->setArgument('supports', ['']);
        }
    }

    /**
     * Create columns for contract post type
     *
     * @param  array $columns
     * @return array
     */
    function createColumns($columns)
    {
        $columns = array('cb' => 'cb', 'title' => __('Title', 'onlinecontract'), 'signed' => __('Signed', 'onlinecontract'), 'printed_name' => __('Printed Name', 'onlinecontract'), 'redirected' => __('Redirected?', 'onlinecontract'), 'contract_type' => __('Contract Type', 'onlinecontract'), 'date' => __('Date', 'onlinecontract'));
        unset($columns['author']);
        unset($columns['comments']);
        return $columns;
    }

    /**
     * Show all columns for contract post type
     *
     * @param array $column_name
     * @return void
     */
    public function showColumns($column_name)
    {
        global $post;
        if ($column_name == 'signed') {
            $value = tr_posts_field('signature');
            if ($value != '-') {
                echo '<span style="color:#fff; background-color:green; padding:5px 10px; border-radius:5px;">' . __('Signed', 'onlinecontract') . '</span>';
            } else {
                echo '<span style="color:#fff; background-color:red; padding:5px 10px; border-radius:5px;">' . __('Unsigned', 'onlinecontract') . '</span>';
            }
        }
        if ($column_name == 'printed_name') {
            $value = tr_posts_field('printed_name');
            if ($value != '-') {
                echo wp_kses($value, 'post');
            } else {
                echo '<em>' . __('Not Set', 'onlinecontract') . '</em>';
            }
        }
        if ($column_name == 'redirected') {
            $value = tr_posts_field('redirect_after_signing?');
            if ($value != '') {
                echo wp_kses($value, 'post');
            } else {
                echo '<em>' . __('Not Set', 'onlinecontract') . '</em>';
            }
        }
        if ($column_name == 'contract_type') {
            $termsind = '';
            $terms = get_the_terms($post->ID, 'contract_type');
            if ($terms != '') {
                if (!isset($terms->errors)) {
                    foreach ($terms as $term) {
                        $termsind .= $term->name . ', ';
                    }
                    echo wp_kses(rtrim($termsind, ', '), 'post');
                }
            } else {
                echo '<em>' . __('Not Set', 'onlinecontract') . '</em>';
            }
        }
    }

    /**
     * Make contract post type columns sortable
     *
     * @param array $columns
     * @return array
     */
    public function sortableColumns($columns)
    {
        $columns['signed'] = 'signed';
        $columns['contract_type'] = 'contract_type';

        return $columns;
    }

    /**
     * Create custom order by based on meta key
     *
     * @param object $query
     * @return void
     */
    function customOrderBy($query)
    {
        if (!is_admin())
            return;

        $orderby = $query->get('orderby');

        if ('signed' == $orderby) {
            $query->set('meta_key', 'signature');
            $query->set('orderby', 'meta_value');
        }
    }
}
