<?php

namespace FDG\OnlineContract\Frontend;

use FDG\OnlineContract\Setup\Setup;
use FDG\OnlineContract\Setup\Init;
use FDG\OnlineContract\Admin\Admin;
use FDG\OnlineContract\Setup\Parser;
use WP_Query;

class Frontend
{
    /**
     * Initialize a new instance of the Frontend class
     *
     */
    public function __construct()
    {
        $this->settings = Init::getSettings();
        $this->setup = (new Setup(Init::getSettings()));
    }

    /**
     * Initialize a new instance of the Contract class
     *
     */
    public function registerHooks()
    {
        add_filter('template_include', [$this, 'includeTemplateFunction'], 1);
        add_filter('archive_template', [$this, 'archiveTemplate']);
        add_action('wp_enqueue_scripts', [$this, 'enqueueAssets']);
        add_action('pre_get_posts', [$this, 'disableFeed']);
        add_shortcode('clientadd', [$this, 'clientAdd']);
        add_shortcode('emaillist', [$this, 'emailList']);
    }

    /**
     * Setup assets for frontend
     *
     * @return void
     */
    public function enqueueAssets()
    {
        wp_enqueue_style('fdg_contract_css', ONLINE_CONTRACT_URL . 'css/style.css');
        if ($this->settings['fontAwesome'] == 1) {
            wp_enqueue_style('font-awesome', 'https://use.fontawesome.com/releases/v5.1.0/css/all.css');
            wp_enqueue_style('font-awesome', 'https://use.fontawesome.com/releases/v5.1.0/css/brands.css');
        }
        wp_enqueue_script('fdg_contract_signature_js', ONLINE_CONTRACT_URL . 'js/signature_pad.min.js', array('jquery'));
        wp_enqueue_script('fdg_contract_signatureoverwrite_js', ONLINE_CONTRACT_URL . 'js/html2canvas.min.js', array('jquery'));
        wp_enqueue_script('fdg_contract_js', ONLINE_CONTRACT_URL . 'js/init.js', array('jquery'));
        wp_localize_script('fdg_contract_js', '_fdg_contract_send', array(
            'ajax_url' => site_url() . '/' . $this->settings['slug'] . '/savesignature',
            'ajax_url_repeat' => site_url() . '/' . $this->settings['slug'] . '/duplicate',
            'ip_address' => Setup::get_client_ip_server()
        ));

        if (is_page() || is_single()) {
            wp_enqueue_style('datepicker-style', ONLINE_CONTRACT_URL . 'css/jquery-ui.min.css', array(), null);
            wp_enqueue_script('jquery-ui-datepicker');
            if ($this->settings['datePickerFormat'] != NULL) {
                wp_deregister_script('jquery-ui-datepicker');
                wp_register_script('jquery-ui-datepicker', ONLINE_CONTRACT_URL . 'js/jquery-ui.min.js', false, '1.12.1', 'true');
                wp_enqueue_script('jquery-ui-datepicker');
                wp_enqueue_script('fdg_contract_date_js', ONLINE_CONTRACT_URL . 'js/admin.js', array('jquery'));
                $data = array(
                    'dateformat' => $this->settings['datePickerFormat']
                );
                wp_localize_script('fdg_contract_date_js', '_fdg_contractadmin_send', $data);
            }
        }
    }

    /**
     * Include the correct front end template
     *
     * @param string $template_path
     * @return void
     */
    public function includeTemplateFunction($template_path)
    {

        if (get_post_type() == 'contract') {
            if (is_single()) {
                if ($theme_file = locate_template(array('single-contract.php'))) {
                    $template_path = $theme_file;
                } else {
                    $template_path = ONLINE_CONTRACT_DIR . 'theme/single-contract.php';
                }
            }
        }
        return $template_path;
    }

    /**
     * Remove access from archives with a blank template
     *
     * @param string $archive_template
     * @return string
     */
    public function archiveTemplate($archive_template)
    {
        global $post;
        if (is_post_type_archive('contract')) {
            $archive_template = ONLINE_CONTRACT_DIR . 'theme/partials/archive.php';
        }
        return $archive_template;
    }

    /**
     * Remove contracts from RSS
     *
     * @param object $query
     * @return void
     */
    public function disableFeed($query)
    {
        if ($query->is_feed() && in_array('contract', (array) $query->get('post_type'))) {
            die('Feed disabled');
        }
    }

    /**
     * Format the selected template and replace shortcodes with information
     *
     * @param string $template
     * @return string
     */
    public static function formatTemplate($template)
    {
        $settings = Init::getSettings();
        $formattedtemplate = $template;
        preg_match_all("^\[(.*?)\]^", $formattedtemplate, $extract, PREG_PATTERN_ORDER);
        array_pop($extract);
        foreach ($extract[0] as $extractval) {
            if (Frontend::getDefaultShortcodes($extractval) != NULL) {
                $formattedtemplate = str_replace($extractval, Frontend::getDefaultShortcodes($extractval), $formattedtemplate);
            }
            foreach (tr_options_field('fdg_shortcodes.contract_shortcodes') as $shortcode) {
                if ($extractval == $shortcode['shortcode']) {
                    $shortcodefunction = $shortcode['function'];
                    if ($shortcodefunction != '') {
                        preg_match_all("^\[(.*?)\]^", $shortcodefunction, $extractfunction, PREG_PATTERN_ORDER);
                        array_pop($extractfunction);
                        foreach ($extractfunction[0] as $extractfunctionval) {
                            foreach (tr_options_field('fdg_shortcodes.contract_shortcodes') as $shortcodefunctionarr) {
                                if ($extractfunctionval == $shortcodefunctionarr['shortcode']) {
                                    $shortcodeconvertedfunction = strtolower(str_replace(array(), '', str_replace(' ', '_', $shortcodefunctionarr['label'])));
                                    $shortcodefunction = str_replace($extractfunctionval, tr_posts_field($shortcodeconvertedfunction), $shortcodefunction);
                                }
                            }
                        }
                        $shortcodefunctionstring = round((new Parser())->Execute($shortcodefunction), $settings['decimalRound']);
                        $shortcodeconverted = strtolower(str_replace(array(), '', str_replace(' ', '_', $shortcode['label'])));
                        $formattedtemplate = str_replace($extractval, $shortcodefunctionstring, $formattedtemplate);
                    } else {
                        $shortcodeconverted = strtolower(str_replace(array(), '', str_replace(' ', '_', $shortcode['label'])));
                        $formattedtemplate = str_replace($extractval, tr_posts_field($shortcodeconverted), $formattedtemplate);
                    }
                }
            }
        }

        return $formattedtemplate;
    }

    /**
     * Contract a single shortcode as needed
     *
     * @param string $shortcode
     * @return string
     */
    public static function formatSingleShortcode($shortcode)
    {
        $formattedshortcode = $shortcode;
        preg_match_all("^\[(.*?)\]^", $formattedshortcode, $extract, PREG_PATTERN_ORDER);
        array_pop($extract);

        foreach ($extract[0] as $extractval) {
            if (Frontend::getDefaultShortcodes($extractval) != NULL) {
                $formattedshortcode = str_replace($extractval, Frontend::getDefaultShortcodes($extractval), $formattedshortcode);
            }
            foreach (tr_options_field('fdg_shortcodes.contract_shortcodes') as $shortcode) {
                if ($extractval == $shortcode['shortcode']) {
                    $shortcodeconverted = strtolower(str_replace(array(), '', str_replace(' ', '_', $shortcode['label'])));
                    $formattedshortcode = str_replace($extractval, tr_posts_field($shortcodeconverted), $formattedshortcode);
                }
            }
        }

        return $formattedshortcode;
    }

    /**
     * Convert shortcode to human-readable information
     *
     * @param string $shortcode
     * @return string
     */
    public static function convertSingleShortcode($shortcode)
    {
        $formattedshortcode = $shortcode;
        preg_match_all("^\[(.*?)\]^", $formattedshortcode, $extract, PREG_PATTERN_ORDER);
        array_pop($extract);

        foreach ($extract[0] as $extractval) {
            if (Frontend::getDefaultShortcodes($extractval) != NULL) {
                $formattedshortcode = str_replace($extractval, Frontend::getDefaultShortcodes($extractval), $formattedshortcode);
            }
            if (tr_options_field('fdg_shortcodes.contract_shortcodes') != NULL) {
                foreach (tr_options_field('fdg_shortcodes.contract_shortcodes') as $shortcode) {
                    if ($extractval == $shortcode['shortcode']) {
                        $shortcodeconverted = preg_replace('/[^\da-z ]/i', '', $shortcode['label']);
                        $shortcodeconverted = strtolower(str_replace(array(), '', str_replace(' ', '_', $shortcodeconverted)));
                        $formattedshortcode = str_replace($extractval, $shortcodeconverted, $formattedshortcode);
                    }
                }
            }
        }

        return $formattedshortcode;
    }

    /**
     * Get plugin defualt sortcodes
     *
     * @param string $extract
     * @return string
     */
    public static function getDefaultShortcodes($extract)
    {
        $settings = Init::getSettings();
        $val = '';
        switch ($extract) {
            case '[currentdate]':
                $val = date($settings['dateFormat']);
                break;
            case '[postid]':
                $val = get_the_ID();
                break;
            case '[businessname]':
                $val = $settings['companyName'];
                break;
            case '[contractcurrency]':
                $val = $settings['currencySymbol'];
                break;
            case '[onlinecontractpagebreak]':
                $val = '<div class="page-break"></div>';
                break;
        }

        return $val;
    }

    /**
     * Shortcode for client add form
     *
     * @param array $atts
     * @return string
     */
    public function clientAdd($atts)
    {
        $settings = Init::getSettings();
        $shortcodeout = '<div class="typerocket-container">';
        $form = tr_form()->useUrl('post', $settings['slug'] . '/createcontract');
        $shortcodeout .=  $form->open();
        $shortcodeout .=  $form->hidden('Signature')->setDefault('-');
        $shortcodeout .=  $form->hidden('Printed Name')->setDefault('-');
        $shortcodeout .=  $form->hidden('IP Address')->setDefault('-');
        $shortcodeout .=  $form->hidden('Generated Sign Date')->setDefault('-');
        $shortcodeout .=  $form->hidden('Can This Contract Be Repeatedly Signed?')->setDefault('');
        if (isset($atts['template']) && $atts['template'] != '') {
            $shortcodeout .=  $form->hidden('Contract Template')->setDefault($atts['template']);
        } else {
            $shortcodeout .=  $form->hidden('Contract Template')->setDefault('-');
        }
        foreach (tr_options_field('fdg_shortcodes.contract_shortcodes') as $shortcode) {
            if (!isset($shortcode['function_only']) && isset($shortcode['visible_on_front'])) {
                if (isset($atts['template']) && $atts['template'] != '') {
                    if ($shortcode['template'] == $atts['template']) {
                        $shortcodeout .=  Admin::convertShortCodeAdmin($form, $shortcode);
                    }
                } else {
                    if (!isset($shortcode['template']) || $shortcode['template'] == '') {
                        $shortcodeout .=  Admin::convertShortCodeAdmin($form, $shortcode);
                    }
                }
            }
        }
        if ($settings['enableRecaptcha'] == 1) {
            if ($settings['recaptchaSiteKey'] != '') {
                $shortcodeout .= '
                <script src="https://www.google.com/recaptcha/api.js" async defer></script>
                <br>
                <br>
                <div class="g-recaptcha" data-callback="recaptchaCallback" data-sitekey="' . $settings['recaptchaSiteKey'] . '"></div>
                <br>
                <br>
                ';
            } else {
                $shortcodeout .= '<br><p style="color:red; font-style:italic;">' . __('Recaptcha is not setup correctly. In order to submit this form, you must setup recaptcha. Please go to Contracts > Settings > Google Recaptcha to complete your setup.', 'onlinecontract') . '</p>';
            }
            $shortcodeout .=  $form->submit(__('Initiate Contract', 'onlinecontract'))->setAttribute('disabled', true)->setAttribute('id', 'clientadd');
        } else {
            $shortcodeout .=  $form->submit(__('Initiate Contract', 'onlinecontract'))->setAttribute('id', 'clientadd');
        }
        $shortcodeout .=  $form->close();
        $shortcodeout .=  '</div>';
        return $shortcodeout;
    }

    /**
     * Shortcode for email list display
     *
     * @param array $atts
     * @param string $content
     * @return string
     */
    function emaillist($atts, $content = null)
    {
        $settings = Init::getSettings();
        extract(shortcode_atts(array('type' => ''), $atts));
        $contracts = '';
        $current_user = wp_get_current_user();
        $email = $current_user->user_email;

        $emailshortcode = $settings['clientEmailShortcode'];

        $emailshortcode = $this->convertSingleShortcode('[' . $emailshortcode . ']');
        if (isset($type) && $type != '') {
            $args = array(
                'post_type' => 'contract',
                'tax_query' => array(
                    array(
                        'taxonomy' => 'contract_type',
                        'field'    => 'slug',
                        'terms'    =>  $type,
                    ),
                ),
                'meta_query' => array(
                    array(
                        'key' => $emailshortcode,
                        'value' => $email,
                        'compare' => '=',
                    )
                )
            );
        } else {
            $args = array(
                'post_type' => 'contract',
                'meta_query' => array(
                    array(
                        'key' => $emailshortcode,
                        'value' => $email,
                        'compare' => '=',
                    )
                )
            );
        }
        $query = new WP_Query($args);
        $i = 0;
        if ($query->have_posts()) :
            $contracts .= '<ul class="contractlistitem">';
            while ($query->have_posts()) : $query->the_post();
                $contracts .= '<li>' . get_the_title() . '
				<ul class="contractlistiteminner">';
                if (tr_posts_field('remove_signature_box?') != 1) {
                    if (tr_posts_field('signature') == '-') {
                        $contracts .= '<li>' . __('Status', 'onlinecontract') . ': ' . __('Unsigned', 'onlinecontract') . '</li>';
                    } else {
                        $contracts .= '<li>' . __('Status', 'onlinecontract') . ': ' . __('Signed', 'onlinecontract') . '</li>';
                    }
                } else {
                    $contracts .= '<li>' . __('Status', 'onlinecontract') . ': ' . __('Cannot Sign Online', 'onlinecontract') . '</li>';
                }
                if (tr_posts_field('signature') == '-') {
                    $contracts .= '<li><a href="' . get_the_permalink() . '" target="_blank">' . __('View Contract') . '</a></li>';
                } else {
                    $contracts .= '<li><a href="' . get_the_permalink() . '#signed" target="_blank">' . __('Print Contract') . '</a></li>';
                }
                $contracts .= '</ul>
			</li>';
            endwhile;
            $contracts .= '</ul>';
        else :
            $contracts = '<p>' . __('You do not have any open contracts.', 'onlinecontract') . '</p>';

        endif;
        wp_reset_query();
        return $contracts;
    }
}
