<?php

namespace FDG\OnlineContract\Setup;

class Exception extends \Exception
{
}

class Parser
{

	private $Pos;
	private $Text;

	public $Variables = [];
	public $OnVariable;
	public $Functions = [];
	public $OnFunction;

	/**
	 * Setup the parser and get all sent numbers
	 *
	 * @param float $Value
	 * @return boolean
	 */
	private function GetNumber(float &$Value = null): bool
	{
		if (!ctype_digit($Char = $this->Text[$this->Pos] ?? false))
			return false;
		$OldPos = $this->Pos;
		$WasDec = false;
		do {
			if ($Char == '.') {
				if ($WasDec) {
					$Result = false;
					break;
				}
				$WasDec = true;
			}
			$this->Pos++;
		} while (ctype_digit($Char = $this->Text[$this->Pos] ?? false) || ($Char == '.'));
		if (isset($Result) || !in_array($Char, [false, '+', '-', '/', '*', '^', ')'])) {
			$this->Pos = $OldPos;
			return false;
		}
		$Value = (float) substr($this->Text, $OldPos, $this->Pos - $OldPos);
		return true;
	}

	/**
	 * Get identity of string
	 *
	 * @param boolean $Kind
	 * @param string $Name
	 * @return boolean
	 */
	private function GetIdentity(bool &$Kind = null, string &$Name = null): bool
	{
		if (!(ctype_alpha($Char = $this->Text[$this->Pos] ?? false) || ($Char == '_')))
			return false;
		$OldPos = $this->Pos;
		do {
			$this->Pos++;
		} while (ctype_alnum($Char = $this->Text[$this->Pos] ?? false) || ($Char == '_'));
		if (!in_array($Char, [false, '+', '-', '/', '*', '^', '(', ')'])) {
			$this->Pos = $OldPos;
			return false;
		}
		$Kind = ($Char != '(');
		$Name = substr($this->Text, $OldPos, $this->Pos - $OldPos);
		return true;
	}

	/**
	 * Get math variables or display error message
	 *
	 * @param string $Name
	 * @param float $Value
	 * @return void
	 */
	private function ProVariable(string $Name, float &$Value = null): void
	{
		if (is_numeric($Value = $this->Variables[$Name] ?? false))
			return;
		if (isset($this->OnVariable))
			call_user_func_array($this->OnVariable, [$Name, &$Value]);
		if (!is_numeric($Value))
			throw new Exception(__('Unknown variable', ' onlinecontract') . ': ' . $Name . '. ' . __('Please verify your input. Remember, if a function uses math, it must have a number as the input.', 'onlinecontract'), 5);
		$this->Variables[$Name] = $Value;
	}

	/**
	 * Add argument if needed
	 *
	 * @param array $Arguments
	 * @param string $Argument
	 * @return void
	 */
	private function AddArgument(&$Arguments, $Argument)
	{
		if ($Argument == '')
			throw new Exception(__('Empty argument', 'onlinecontract') . ': ' . __('Please verify your input. You are missing a field used for a math calculation.', 'onlinecontract'), 4);
		$Arguments[] = $Argument;
	}

	/**
	 * Get all arguments
	 *
	 * @param array $Arguments
	 * @return boolean
	 */
	private function GetArguments(&$Arguments = []): bool
	{
		$B = 1;
		$this->Pos++;
		$Mark = $this->Pos;
		while ((($Char = $this->Text[$this->Pos] ?? false) !== false) && ($B > 0)) {
			if (($Char == ',') && ($B == 1)) {
				$this->AddArgument($Arguments, substr($this->Text, $Mark, $this->Pos - $Mark));
				$Mark = $this->Pos + 1;
			} elseif ($Char == ')') $B--;
			elseif ($Char == '(') $B++;
			$this->Pos++;
		}
		if (!in_array($Char, [false, '+', '-', '/', '*', '^', ')']))
			return false;
		$this->AddArgument($Arguments, substr($this->Text, $Mark, $this->Pos - $Mark - 1));
		return true;
	}

	/**
	 * Return results from an argument
	 *
	 * @param array $Arguments
	 * @return array
	 */
	private function ProArgument($Arguments)
	{
		$OPos = $this->Pos;
		$OText = $this->Text;
		$Result = [];
		foreach ($Arguments as $Argument)
			$Result[] = $this->Perform($Argument);
		$this->Pos = $OPos;
		$this->Text = $OText;
		return $Result;
	}

	/**
	 * Check if function, if so, do the math
	 *
	 * @param string $Name
	 * @param float $Value
	 * @return void
	 */
	private function ProFunction(string $Name, float &$Value = null): void
	{
		if (!isset($this->Functions[$Name])) {
			if (isset($this->OnFunction))
				call_user_func_array($this->OnFunction, [$Name, &$Function]);
			if (!isset($Function))
				throw new Exception(__('Unknown function', 'onlinecontract') . ': ' . $Name . ' ' . __('is not a valid function in your function shortcode.', 'onlinecontract'), 6);
			$this->Functions[$Name] = $Function;
		} else
			$Function = $this->Functions[$Name];
		if (!$this->GetArguments($Arguments))
			throw new Exception(__('Syntax error', 'onlinecontract') . ': ' . __('Please verify your input. You are missing a field used for a math calculation or entered text in a way the system cannot interpret.', 'onlinecontract'), 1);
		if (isset($Function['arc'])) {
			if ($Function['arc'] != count($Arguments))
				throw new Exception(__('Invalid argument count', 'onlinecontract') . ': ' . __('Please verify your input. You have too many variables in your input.', 'onlinecontract'), 3);
			$Value = call_user_func_array($Function['ref'], $this->ProArgument($Arguments));
		} else
			$Value = call_user_func($Function['ref'], $this->ProArgument($Arguments));
	}

	/**
	 * Calculate the math
	 *
	 * @return float
	 */
	private function Term(): float
	{
		if ($this->Text[$this->Pos] == '(') {
			$this->Pos++;
			$Value = $this->Calculate();
			$this->Pos++;
			if (!in_array($this->Text[$this->Pos] ?? false, [false, '+', '-', '/', '*', '^', ')']))
				throw new Exception(__('Syntax error', 'onlinecontract') . ': ' . __('Please verify your input. You are missing a field used for a math calculation or entered text in a way the system cannot interpret.', 'onlinecontract'), 1);

			return $Value;
		}
		if (!$this->GetNumber($Value))
			if ($this->GetIdentity($Kind, $Name)) {
				if ($Kind)
					$this->ProVariable($Name, $Value);
				else
					$this->ProFunction($Name, $Value);
			} else
				throw new Exception(__('Syntax error', 'onlinecontract') . ': ' . __('Please verify your input. You are missing a field used for a math calculation or entered text in a way the system cannot interpret.', 'onlinecontract'), 1);
		return $Value;
	}

	/**
	 * Complete the math at the inner level
	 *
	 * @return float
	 */
	private function SubTerm(): float
	{
		$Value = $this->Term();
		while (in_array($Char = $this->Text[$this->Pos] ?? false, ['*', '^', '/'])) {
			$this->Pos++;
			$Term = $this->Term();
			switch ($Char) {
				case '*':
					$Value = $Value * $Term;
					break;
				case '/':
					if ($Term == 0)
						throw new Exception('Division by zero', 7);
					$Value = $Value / $Term;
					break;
				case '^':
					$Value = pow($Value, $Term);
					break;
			}
		}
		return $Value;
	}

	/**
	 * Final calculation
	 *
	 * @return float
	 */
	private function Calculate(): float
	{
		$Value = $this->SubTerm();
		while (in_array($Char = $this->Text[$this->Pos] ?? false, ['+', '-'])) {
			$this->Pos++;
			$SubTerm = $this->SubTerm();
			if ($Char == '-')
				$SubTerm = -$SubTerm;
			$Value += $SubTerm;
		}
		return $Value;
	}

	/**
	 * Perform calculations and return results
	 *
	 * @param string $Formula
	 * @return float
	 */
	private function Perform(string $Formula): float
	{
		$this->Pos = 0;
		if (in_array($Formula[0], ['-', '+']))
			$Formula = '0' . $Formula;
		$this->Text = $Formula;
		return $this->Calculate();
	}

	/**
	 * Execute equations
	 *
	 * @param string $Formula
	 * @return float
	 */
	public function Execute(string $Formula): float
	{
		$B = 0;
		for ($I = 0; $I < strlen($Formula); $I++) {
			switch ($Formula[$I]) {
				case '(':
					$B++;
					break;
				case ')':
					$B--;
					break;
			}
		}
		if ($B != 0)
			throw new Exception('Unmatched brackets', 2);
		return $this->Perform(str_replace(' ', '', strtolower($Formula)));
	}
}
